#!/bin/bash

# HTTPS Setup Script for Titan Drilling Node.js Services
# This script automates the setup of Apache reverse proxy with HTTPS

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "🔒 Titan Drilling HTTPS Setup Script"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""

# Color codes
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Configuration
DOMAIN="titandrillingzm.com"
API_SUBDOMAIN="api.titandrillingzm.com"
XAMPP_PATH="/Applications/XAMPP/xamppfiles"
CURRENT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

echo -e "${YELLOW}This script will:${NC}"
echo "1. Check Apache modules"
echo "2. Copy reverse proxy configuration"
echo "3. Setup SSL certificates (self-signed for testing)"
echo "4. Configure Apache"
echo "5. Test and restart Apache"
echo ""

# Check if running as root
if [[ $EUID -ne 0 ]]; then
   echo -e "${RED}❌ This script must be run with sudo${NC}"
   echo "Usage: sudo ./setup-https.sh"
   exit 1
fi

# Step 1: Check DNS
echo -e "${YELLOW}Step 1: Checking DNS configuration...${NC}"
if ping -c 1 "$API_SUBDOMAIN" &> /dev/null; then
    echo -e "${GREEN}✅ DNS for $API_SUBDOMAIN is configured${NC}"
else
    echo -e "${YELLOW}⚠️  Warning: $API_SUBDOMAIN is not resolving${NC}"
    echo "Please add an A record or CNAME for 'api' pointing to your server IP"
    echo ""
    read -p "Continue anyway? (y/n) " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        exit 1
    fi
fi

# Step 2: Check Apache modules
echo ""
echo -e "${YELLOW}Step 2: Checking Apache modules...${NC}"

HTTPD_CONF="$XAMPP_PATH/etc/httpd.conf"
MODULES_TO_CHECK=(
    "mod_proxy.so"
    "mod_proxy_http.so"
    "mod_proxy_wstunnel.so"
    "mod_ssl.so"
    "mod_headers.so"
    "mod_rewrite.so"
    "mod_socache_shmcb.so"
)

MISSING_MODULES=()

for module in "${MODULES_TO_CHECK[@]}"; do
    if grep -q "^LoadModule.*$module" "$HTTPD_CONF"; then
        echo -e "${GREEN}✅ $module is enabled${NC}"
    else
        echo -e "${RED}❌ $module is NOT enabled${NC}"
        MISSING_MODULES+=("$module")
    fi
done

if [ ${#MISSING_MODULES[@]} -ne 0 ]; then
    echo ""
    echo -e "${YELLOW}⚠️  Some modules are missing. Do you want to enable them automatically?${NC}"
    read -p "Enable missing modules? (y/n) " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        # Backup httpd.conf
        cp "$HTTPD_CONF" "$HTTPD_CONF.backup.$(date +%Y%m%d_%H%M%S)"
        echo -e "${GREEN}✅ Backed up httpd.conf${NC}"
        
        # Enable modules
        for module in "${MISSING_MODULES[@]}"; do
            sed -i.bak "s/^#LoadModule.*$module/LoadModule ${module%.so}_module modules\/$module/" "$HTTPD_CONF"
            echo -e "${GREEN}✅ Enabled $module${NC}"
        done
    else
        echo -e "${RED}❌ Cannot proceed without required modules${NC}"
        echo "Please enable these modules manually in $HTTPD_CONF"
        exit 1
    fi
fi

# Step 3: Copy reverse proxy configuration
echo ""
echo -e "${YELLOW}Step 3: Installing reverse proxy configuration...${NC}"

EXTRA_DIR="$XAMPP_PATH/etc/extra"
PROXY_CONF="$EXTRA_DIR/apache-reverse-proxy.conf"

# Backup existing config if it exists
if [ -f "$PROXY_CONF" ]; then
    cp "$PROXY_CONF" "$PROXY_CONF.backup.$(date +%Y%m%d_%H%M%S)"
    echo -e "${GREEN}✅ Backed up existing configuration${NC}"
fi

# Copy new configuration
cp "$CURRENT_DIR/apache-reverse-proxy.conf" "$PROXY_CONF"
echo -e "${GREEN}✅ Installed reverse proxy configuration${NC}"

# Check if configuration is included in httpd.conf
if grep -q "Include etc/extra/apache-reverse-proxy.conf" "$HTTPD_CONF"; then
    echo -e "${GREEN}✅ Configuration already included in httpd.conf${NC}"
else
    echo ""
    echo "Include etc/extra/apache-reverse-proxy.conf" >> "$HTTPD_CONF"
    echo -e "${GREEN}✅ Added configuration to httpd.conf${NC}"
fi

# Step 4: Setup SSL Certificates
echo ""
echo -e "${YELLOW}Step 4: Setting up SSL certificates...${NC}"
echo "Choose SSL certificate option:"
echo "1) Self-signed certificate (for testing)"
echo "2) Let's Encrypt certificate (production - requires domain to be live)"
echo "3) Skip (I'll configure SSL manually later)"
read -p "Enter choice (1-3): " ssl_choice

case $ssl_choice in
    1)
        echo -e "${YELLOW}Creating self-signed certificate...${NC}"
        
        # Create SSL directories
        mkdir -p "$XAMPP_PATH/etc/ssl.crt"
        mkdir -p "$XAMPP_PATH/etc/ssl.key"
        
        # Generate self-signed certificate
        openssl req -x509 -nodes -days 365 -newkey rsa:2048 \
          -keyout "$XAMPP_PATH/etc/ssl.key/server.key" \
          -out "$XAMPP_PATH/etc/ssl.crt/server.crt" \
          -subj "/C=ZM/ST=Zambia/L=Lusaka/O=Titan Drilling/CN=$DOMAIN"
        
        echo -e "${GREEN}✅ Self-signed certificate created${NC}"
        echo -e "${YELLOW}⚠️  Note: Browsers will show security warnings for self-signed certificates${NC}"
        ;;
    2)
        echo -e "${YELLOW}Installing Let's Encrypt certificate...${NC}"
        
        # Check if certbot is installed
        if ! command -v certbot &> /dev/null; then
            echo -e "${YELLOW}Installing certbot...${NC}"
            brew install certbot
        fi
        
        # Stop Apache
        echo -e "${YELLOW}Stopping Apache temporarily...${NC}"
        "$XAMPP_PATH/bin/apachectl" stop
        
        # Generate certificate
        certbot certonly --standalone \
          -d "$DOMAIN" \
          -d "www.$DOMAIN" \
          -d "$API_SUBDOMAIN" \
          --non-interactive \
          --agree-tos \
          --email "admin@$DOMAIN"
        
        if [ $? -eq 0 ]; then
            echo -e "${GREEN}✅ Let's Encrypt certificate installed${NC}"
            
            # Update Apache config to use Let's Encrypt
            sed -i.bak "s|/Applications/XAMPP/xamppfiles/etc/ssl.crt/server.crt|/etc/letsencrypt/live/$DOMAIN/fullchain.pem|g" "$PROXY_CONF"
            sed -i.bak "s|/Applications/XAMPP/xamppfiles/etc/ssl.key/server.key|/etc/letsencrypt/live/$DOMAIN/privkey.pem|g" "$PROXY_CONF"
            echo -e "${GREEN}✅ Updated configuration to use Let's Encrypt${NC}"
        else
            echo -e "${RED}❌ Failed to install Let's Encrypt certificate${NC}"
            echo "Falling back to self-signed certificate..."
            
            mkdir -p "$XAMPP_PATH/etc/ssl.crt"
            mkdir -p "$XAMPP_PATH/etc/ssl.key"
            
            openssl req -x509 -nodes -days 365 -newkey rsa:2048 \
              -keyout "$XAMPP_PATH/etc/ssl.key/server.key" \
              -out "$XAMPP_PATH/etc/ssl.crt/server.crt" \
              -subj "/C=ZM/ST=Zambia/L=Lusaka/O=Titan Drilling/CN=$DOMAIN"
        fi
        ;;
    3)
        echo -e "${YELLOW}⚠️  Skipping SSL setup. You'll need to configure certificates manually.${NC}"
        ;;
    *)
        echo -e "${RED}❌ Invalid choice${NC}"
        exit 1
        ;;
esac

# Step 5: Test Apache configuration
echo ""
echo -e "${YELLOW}Step 5: Testing Apache configuration...${NC}"

"$XAMPP_PATH/bin/apachectl" configtest

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✅ Apache configuration is valid${NC}"
else
    echo -e "${RED}❌ Apache configuration has errors${NC}"
    echo "Please check the configuration and try again"
    exit 1
fi

# Step 6: Restart Apache
echo ""
echo -e "${YELLOW}Step 6: Restarting Apache...${NC}"

"$XAMPP_PATH/bin/apachectl" restart

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✅ Apache restarted successfully${NC}"
else
    echo -e "${RED}❌ Failed to restart Apache${NC}"
    echo "Check Apache error log: $XAMPP_PATH/logs/error_log"
    exit 1
fi

# Step 7: Check if Node.js services are running
echo ""
echo -e "${YELLOW}Step 7: Checking Node.js services...${NC}"

if pgrep -f "node.*server.js" > /dev/null; then
    echo -e "${GREEN}✅ Node.js services are running${NC}"
else
    echo -e "${YELLOW}⚠️  Node.js services are not running${NC}"
    echo "Start them with: cd $CURRENT_DIR && node server.js"
fi

# Summary
echo ""
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo -e "${GREEN}✅ Setup Complete!${NC}"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""
echo "Your API endpoints are now available at:"
echo -e "${GREEN}https://$API_SUBDOMAIN/equipment${NC}"
echo -e "${GREEN}https://$API_SUBDOMAIN/documents${NC}"
echo -e "${GREEN}https://$API_SUBDOMAIN/inspection${NC}"
echo "... and all other services"
echo ""
echo "Test with:"
echo "  curl https://$API_SUBDOMAIN/equipment/"
echo ""
echo "View logs:"
echo "  Apache: tail -f $XAMPP_PATH/logs/api-error_log"
echo "  Node.js: Check your Node.js console"
echo ""
echo "Next steps:"
echo "1. Update your React Native app to use new HTTPS URLs"
echo "2. Test all API endpoints"
echo "3. Update any documentation"
echo ""

if [ "$ssl_choice" == "2" ]; then
    echo "SSL Certificate auto-renewal:"
    echo "  Test: sudo certbot renew --dry-run"
    echo "  Setup cron: sudo crontab -e"
    echo "  Add line: 0 3 * * * certbot renew --post-hook '$XAMPP_PATH/bin/apachectl graceful'"
    echo ""
fi

echo -e "${YELLOW}📖 For more details, see HTTPS_SETUP_GUIDE.md${NC}"
echo ""

