const express = require('express');
const { mongoose } = require('./dbConnection'); // Use shared connection
const multer = require('multer');
const path = require('path');
const fs = require('fs');

const app = express();
const PORT = process.env.NOTICE_BOARD_PORT || 5015;

// Middleware
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// CORS
app.use((req, res, next) => {
  res.header('Access-Control-Allow-Origin', '*');
  res.header('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, OPTIONS');
  res.header('Access-Control-Allow-Headers', 'Origin, X-Requested-With, Content-Type, Accept, Authorization');
  if (req.method === 'OPTIONS') {
    return res.sendStatus(200);
  }
  next();
});

// MongoDB Connection
const MONGO_URI = 'mongodb://api.titandrillingzm.com:27017/titan_drilling';

// Using shared MongoDB connection from dbConnection.js
console.log('✅ Using shared MongoDB connection');

// Notice Schema
const noticeSchema = new mongoose.Schema({
  noticeId: { type: String, required: true, unique: true },
  title: { type: String, required: true },
  description: { type: String, required: true },
  imageUrl: { type: String, default: null },
  createdBy: { type: String, required: true }, // userId who created
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
  // Track which users have read this notice
  readBy: [{
    userId: String,
    readAt: Date
  }]
});

const Notice = mongoose.model('Notice', noticeSchema, 'globalnotices');

// Configure multer for image uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const userId = req.params.userId || 'default';
    const uploadDir = `/root/node-mongo-api/uploads/documents/notices/${userId}`;
    
    // Create directory if it doesn't exist
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now();
    const ext = path.extname(file.originalname);
    cb(null, `notice_${uniqueSuffix}${ext}`);
  }
});

const upload = multer({
  storage: storage,
  limits: { fileSize: 5 * 1024 * 1024 }, // 5MB limit
  fileFilter: (req, file, cb) => {
    const allowedTypes = /jpeg|jpg|png|gif/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);
    
    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('Only images are allowed (jpeg, jpg, png, gif)'));
    }
  }
});

// ==================== ROUTES ====================

// Health Check
app.get('/notices', (req, res) => {
  res.json({ 
    success: true, 
    message: 'Notice Board API is running',
    port: PORT 
  });
});

// Upload Image
app.post('/notices/upload-image/:userId', upload.single('image'), (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ success: false, error: 'No image file provided' });
    }

    const imageUrl = `https://api.titandrillingzm.com:${PORT}/notices/download-image/${req.params.userId}/${req.file.filename}`;
    
    console.log('✅ Image uploaded:', imageUrl);
    
    res.json({
      success: true,
      imageUrl: imageUrl,
      filename: req.file.filename
    });
  } catch (error) {
    console.error('❌ Error uploading image:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Download Image
app.get('/notices/download-image/:userId/:filename', (req, res) => {
  try {
    const { userId, filename } = req.params;
    const filePath = `/root/node-mongo-api/uploads/documents/notices/${userId}/${filename}`;
    
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({ success: false, error: 'Image not found' });
    }
    
    res.sendFile(filePath);
  } catch (error) {
    console.error('❌ Error downloading image:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Create Notice
app.post('/notices', async (req, res) => {
  try {
    const { noticeId, title, description, imageUrl, createdBy } = req.body;
    
    console.log('📝 Creating notice:', { noticeId, title, createdBy });
    
    if (!noticeId || !title || !description || !createdBy) {
      return res.status(400).json({ 
        success: false, 
        error: 'Missing required fields: noticeId, title, description, createdBy' 
      });
    }
    
    const notice = new Notice({
      noticeId,
      title,
      description,
      imageUrl: imageUrl || null,
      createdBy,
      createdAt: new Date(),
      updatedAt: new Date(),
      readBy: []
    });
    
    await notice.save();
    
    console.log('✅ Notice created successfully:', noticeId);
    
    res.json({
      success: true,
      data: notice,
      message: 'Notice created successfully'
    });
  } catch (error) {
    console.error('❌ Error creating notice:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get All Notices (with read status for a specific user)
app.get('/notices/all/:userId', async (req, res) => {
  try {
    const { userId } = req.params;
    
    console.log('📋 Fetching all notices for user:', userId);
    
    const notices = await Notice.find().sort({ createdAt: -1 });
    
    // Add read status for this user
    const noticesWithReadStatus = notices.map(notice => {
      const noticeObj = notice.toObject();
      const isRead = notice.readBy.some(read => read.userId === userId);
      
      return {
        ...noticeObj,
        isRead,
        unreadCount: notice.readBy.length
      };
    });
    
    console.log(`✅ Found ${notices.length} notices`);
    
    res.json({
      success: true,
      data: noticesWithReadStatus,
      total: notices.length
    });
  } catch (error) {
    console.error('❌ Error fetching notices:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get Unread Count for User
app.get('/notices/unread-count/:userId', async (req, res) => {
  try {
    const { userId } = req.params;
    
    console.log('📊 Getting unread count for user:', userId);
    
    const notices = await Notice.find();
    const unreadCount = notices.filter(notice => 
      !notice.readBy.some(read => read.userId === userId)
    ).length;
    
    console.log(`✅ Unread count: ${unreadCount}`);
    
    res.json({
      success: true,
      unreadCount
    });
  } catch (error) {
    console.error('❌ Error getting unread count:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Mark Notice as Read
app.post('/notices/mark-read/:noticeId/:userId', async (req, res) => {
  try {
    const { noticeId, userId } = req.params;
    
    console.log('👁️ Marking notice as read:', { noticeId, userId });
    
    const notice = await Notice.findOne({ noticeId });
    
    if (!notice) {
      return res.status(404).json({ success: false, error: 'Notice not found' });
    }
    
    // Check if already read
    const alreadyRead = notice.readBy.some(read => read.userId === userId);
    
    if (!alreadyRead) {
      notice.readBy.push({
        userId,
        readAt: new Date()
      });
      
      await notice.save();
      console.log('✅ Notice marked as read');
    } else {
      console.log('ℹ️ Notice already read by this user');
    }
    
    res.json({
      success: true,
      message: 'Notice marked as read'
    });
  } catch (error) {
    console.error('❌ Error marking notice as read:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get Single Notice
app.get('/notices/:noticeId', async (req, res) => {
  try {
    const { noticeId } = req.params;
    const { userId } = req.query;
    
    console.log('📄 Fetching notice:', noticeId);
    
    const notice = await Notice.findOne({ noticeId });
    
    if (!notice) {
      return res.status(404).json({ success: false, error: 'Notice not found' });
    }
    
    const noticeObj = notice.toObject();
    const isRead = userId ? notice.readBy.some(read => read.userId === userId) : false;
    
    res.json({
      success: true,
      data: {
        ...noticeObj,
        isRead
      }
    });
  } catch (error) {
    console.error('❌ Error fetching notice:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Delete Notice
app.delete('/notices/:noticeId/:userId', async (req, res) => {
  try {
    const { noticeId, userId } = req.params;
    
    console.log('🗑️ Deleting notice:', { noticeId, userId });
    
    const notice = await Notice.findOne({ noticeId });
    
    if (!notice) {
      return res.status(404).json({ success: false, error: 'Notice not found' });
    }
    
    // Only creator can delete
    if (notice.createdBy !== userId) {
      return res.status(403).json({ 
        success: false, 
        error: 'Only the creator can delete this notice' 
      });
    }
    
    // Delete image if exists
    if (notice.imageUrl) {
      try {
        const filename = notice.imageUrl.split('/').pop();
        const imagePath = `/root/node-mongo-api/uploads/documents/notices/${userId}/${filename}`;
        if (fs.existsSync(imagePath)) {
          fs.unlinkSync(imagePath);
          console.log('✅ Image deleted:', filename);
        }
      } catch (err) {
        console.warn('⚠️ Error deleting image:', err);
      }
    }
    
    await Notice.deleteOne({ noticeId });
    
    console.log('✅ Notice deleted successfully');
    
    res.json({
      success: true,
      message: 'Notice deleted successfully'
    });
  } catch (error) {
    console.error('❌ Error deleting notice:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Update Notice
app.put('/notices/:noticeId/:userId', async (req, res) => {
  try {
    const { noticeId, userId } = req.params;
    const { title, description, imageUrl } = req.body;
    
    console.log('✏️ Updating notice:', { noticeId, userId });
    
    const notice = await Notice.findOne({ noticeId });
    
    if (!notice) {
      return res.status(404).json({ success: false, error: 'Notice not found' });
    }
    
    // Only creator can update
    if (notice.createdBy !== userId) {
      return res.status(403).json({ 
        success: false, 
        error: 'Only the creator can update this notice' 
      });
    }
    
    if (title) notice.title = title;
    if (description) notice.description = description;
    if (imageUrl !== undefined) notice.imageUrl = imageUrl;
    notice.updatedAt = new Date();
    
    await notice.save();
    
    console.log('✅ Notice updated successfully');
    
    res.json({
      success: true,
      data: notice,
      message: 'Notice updated successfully'
    });
  } catch (error) {
    console.error('❌ Error updating notice:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Start Server
app.listen(PORT, '0.0.0.0', () => {
  console.log(`✅ Notice Board API running on port ${PORT}`);
  console.log(`📡 API URL: https://api.titandrillingzm.com:${PORT}`);
});

module.exports = app;

