@extends('welcome')

@section('title', 'Create DVIR Report - Step 1')

@section('content')
<div class="dvir-create-container">
    <!-- Header Section -->
    <div class="dvir-header">
        <div class="header-content">
            <div class="page-title">
                <h1>New Inspection</h1>
                <p>Create a new Driver Vehicle Inspection Report</p>
            </div>
        </div>
    </div>

    <!-- Main Form Container -->
    <div class="form-container" id="step1Container">
        <div class="form-card">
            <!-- Request Number Display -->
            <div class="request-number-display">
                <div class="request-number-row">
                    <span class="request-label">Request #</span>
                    <span class="request-value" id="requestNumber">Select country and project first</span>
                </div>
            </div>

            <!-- Country Selection -->
            <div class="form-group">
                <label class="form-label required-field">Country</label>
                <select class="form-control" id="country" onchange="updateProjects()">
                    <option value="">Select Country</option>
                    <option value="Zambia">Zambia</option>
                    <option value="DRC">DRC</option>
                </select>
            </div>

            <!-- Project Selection -->
            <div class="form-group">
                <label class="form-label required-field">Project</label>
                <select class="form-control" id="project" disabled onchange="generateRequestNumber()">
                    <option value="">Select Project</option>
                </select>
            </div>

            <!-- Inspector -->
            <div class="form-group">
                <label class="form-label required-field">Inspector</label>
                <input type="text" class="form-control" id="inspector" placeholder="Enter Inspector Name">
            </div>

            <!-- Equipment Selection -->
            <div class="equipment-selection-row">
                <span class="equipment-label">Equipment</span>
                <button class="equipment-select-btn" onclick="openEquipmentModal()">
                    <span id="equipmentSelectText">Select</span>
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M9 18l6-6-6-6"/>
                    </svg>
                </button>
            </div>

            <!-- Selected Equipment Details -->
            <div class="equipment-details-box" id="equipmentDetailsBox" style="display: none;">
                <div class="equipment-info-row">
                    <span class="equipment-info-text" id="equipmentInfoText"></span>
                </div>
                
                <!-- Meter Reading Section -->
                <div class="meter-reading-section">
                    <label class="meter-reading-label required-field">Meter Reading</label>
                    <div class="meter-reading-input-row">
                        <input type="number" class="meter-reading-input" id="meterReading" placeholder="Enter meter reading">
                        <span class="meter-unit" id="meterUnit">Miles</span>
                    </div>
                    <div class="meter-reading-error" id="meterReadingError" style="display: none;"></div>
                    <div class="current-meter-row">
                        <span class="current-meter-label">Current Meter Reading: </span>
                        <span class="current-meter-value" id="currentMeterValue">0 Miles</span>
                    </div>
                </div>
            </div>

            <!-- Inspection Type Selection -->
            <div class="inspection-type-section">
                <div class="checkbox-row">
                    <input type="checkbox" id="dvirCheckbox" class="inspection-checkbox" checked>
                    <label for="dvirCheckbox" class="checkbox-label">DVIR - Driver Vehicle Inspection Report</label>
                </div>
            </div>
        </div>

        <!-- Action Buttons -->
        <div class="action-buttons">
            <button class="btn-cancel" onclick="handleBackPress()">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M19 12H5"/>
                    <path d="M12 19l-7-7 7-7"/>
                </svg>
                <span>Cancel</span>
            </button>
            <button class="btn-next" id="nextButton" onclick="handleNextPress()" disabled>
                <span>Next</span>
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M5 12h14"/>
                    <path d="M12 5l7 7-7 7"/>
                </svg>
            </button>
        </div>
    </div>

    <!-- Equipment Selection Modal -->
    <div class="modal-overlay" id="equipmentModal" style="display: none;">
        <div class="modal-content equipment-modal">
            <div class="modal-header">
                <h3 class="modal-title">Select Equipment</h3>
                <div class="modal-header-actions">
                    <button class="refresh-equipment-btn" onclick="refreshEquipmentData()" title="Refresh from Firebase">
                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="23 4 23 10 17 10"/>
                            <polyline points="1 20 1 14 7 14"/>
                            <path d="M20.49 9A9 9 0 0 0 5.64 5.64L1 10m22 4l-4.64 4.36A9 9 0 0 1 3.51 15"/>
                        </svg>
                    </button>
                    <button class="modal-close" onclick="closeEquipmentModal()">
                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <line x1="18" y1="6" x2="6" y2="18"/>
                            <line x1="6" y1="6" x2="18" y2="18"/>
                        </svg>
                    </button>
                </div>
            </div>
            
            <!-- Equipment Search -->
            <div class="equipment-search-container">
                <div class="search-input-wrapper">
                    <input type="text" class="equipment-search-input" id="equipmentSearchInput" placeholder="Search equipment...">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="search-icon">
                        <circle cx="11" cy="11" r="8"/>
                        <path d="M21 21l-4.35-4.35"/>
                    </svg>
                </div>
            </div>

            <!-- Equipment Tabs -->
            <div class="equipment-tabs">
                <button class="tab-button active" onclick="filterEquipment('ALL')">ALL</button>
                <button class="tab-button" onclick="filterEquipment('LV')">Light Vehicles</button>
                <button class="tab-button" onclick="filterEquipment('HV')">Heavy Vehicles</button>
            </div>

            <!-- Equipment List -->
            <div class="equipment-list-container">
                <div class="equipment-list" id="equipmentList">
                    <!-- Equipment items will be loaded here -->
                </div>
            </div>

            <div class="modal-footer">
                <button class="btn-close" onclick="closeEquipmentModal()">Close</button>
            </div>
        </div>
    </div>

    <!-- Loading Modal -->
    <div class="modal-overlay" id="loadingModal" style="display: none;">
        <div class="loading-modal-content">
            <div class="loading-spinner"></div>
            <h3>Please Wait</h3>
            <p>Processing Equipment Data</p>
        </div>
    </div>

    <!-- Toast Container -->
    <div class="toast-container" id="toastContainer"></div>

    <!-- Step 2: Inspection Checklist -->
    <div class="step-container" id="step2Container" style="display: none;">
        <div class="step-header">
            <h1>Inspection Checklist</h1>
            <p>Complete the inspection checklist items</p>
        </div>

        <div class="step-content">
            <div class="instruction-text">
                <p>Tap on list to expand</p>
            </div>

            <div class="checklist-container">
                <!-- Requirements Section -->
                <div class="checklist-card" id="requirements-card">
                    <div class="card-header" onclick="toggleSection('requirements')">
                        <div class="header-left">
                            <span class="expand-icon" id="requirements-icon">▼</span>
                            <h3 class="section-title">Requirements</h3>
                        </div>
                    </div>
                    <div class="card-content" id="requirements-content" style="display: none;">
                        <div class="checklist-items">
                            <div class="checklist-item" data-item-id="req1">
                                <div class="item-content">
                                    <p class="item-text">I hold a valid driver's licence</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req1', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req1', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req1" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req1', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req1" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req1')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req1" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req1')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req1')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="req2">
                                <div class="item-content">
                                    <p class="item-text">I will obey speed limits and road signs</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req2', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req2', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req2" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req2', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req2" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req2')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req2" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req2')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req2')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="req3">
                                <div class="item-content">
                                    <p class="item-text">I will not overload the vehicle and ensure the load is secure</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req3', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req3', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req3" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req3', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req3" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req3')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req3" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req3')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req3')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="req4">
                                <div class="item-content">
                                    <p class="item-text">I have completed the vehicle pre check list and satisfied all is order for my trip</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req4', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req4', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req4" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req4', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req4" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req4')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req4" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req4')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req4')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="req5">
                                <div class="item-content">
                                    <p class="item-text">I am fit for work – (Not under the influence of alcohol, am well rested, am healthy)</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req5', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req5', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req5" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req5', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req5" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req5')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req5" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req5')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req5')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="req6">
                                <div class="item-content">
                                    <p class="item-text">I have read and understand the Titan driving procedure, National driving rules and cliental rules</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req6', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req6', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req6" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req6', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req6" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req6')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req6" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req6')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req6')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="req7">
                                <div class="item-content">
                                    <p class="item-text">I will ensure every passenger and myself wears a seat belt (No unauthorised passengers)</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req7', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req7', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req7" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req7', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req7" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req7')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req7" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req7')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req7')" title="Take Photo">📷</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Hazards Section -->
                <div class="checklist-card" id="hazards-card">
                    <div class="card-header" onclick="toggleSection('hazards')">
                        <div class="header-left">
                            <span class="expand-icon" id="hazards-icon">▼</span>
                            <h3 class="section-title">Hazards – Are you aware of:</h3>
                        </div>
                    </div>
                    <div class="card-content" id="hazards-content" style="display: none;">
                        <div class="checklist-items">
                            <div class="checklist-item" data-item-id="haz1">
                                <div class="item-content">
                                    <p class="item-text">Pedestrians crossing the road, school children and drunkards</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz1', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz1', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz1" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz1', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz1" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz1')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz1" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz1')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz1')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="haz2">
                                <div class="item-content">
                                    <p class="item-text">Motor bikes and taxis stopping suddenly or pulling in to the road suddenly</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz2', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz2', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz2" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz2', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz2" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz2')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz2" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz2')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz2')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="haz3">
                                <div class="item-content">
                                    <p class="item-text">Oncoming vehicles overtaking on blind rises or corners</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz3', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz3', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz3" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz3', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz3" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz3')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz3" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz3')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz3')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="haz4">
                                <div class="item-content">
                                    <p class="item-text">Heavy trucks or plant taking up width of the road</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz4', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz4', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz4" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz4', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz4" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz4')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz4" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz4')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz4')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="haz5">
                                <div class="item-content">
                                    <p class="item-text">Riots or unrest</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz5', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz5', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz5" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz5', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz5" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz5')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz5" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz5')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz5')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="haz6">
                                <div class="item-content">
                                    <p class="item-text">Gravel/Sandy roads can be slippery – Loss of control</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz6', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz6', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz6" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz6', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz6" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz6')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz6" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz6')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz6')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="haz7">
                                <div class="item-content">
                                    <p class="item-text">Tyre blowouts or punctures – Loss of control</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz7', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz7', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz7" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz7', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz7" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz7')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz7" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz7')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz7')" title="Take Photo">📷</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Step 2 Navigation Buttons -->
        <div class="nav-buttons">
            <button class="nav-btn back-btn" onclick="showStep1()">
                <span>← Back</span>
            </button>
            <button class="nav-btn next-btn" id="step2NextButton" onclick="handleStep2Next()" disabled>
                <span>Next →</span>
            </button>
        </div>
    </div>

    <!-- Step 3: Summary and Report Generation -->
    <div class="step-container" id="step3Container" style="display: none;">
        <div class="step-header">
            <h1>Summary & Report Generation</h1>
            <p>Review and generate your DVIR report</p>
        </div>

        <div class="step-content">
            <!-- Faulty Items Display -->
            <div class="summary-row">
                <span class="summary-label">Faulty Items</span>
                <span class="summary-value faulty" id="faultyItemsCount">0</span>
            </div>

            <!-- Overall Condition -->
            <div class="form-group">
                <label class="form-label required-field">Overall Condition</label>
                <div class="select-container">
                    <select class="form-control" id="overallCondition" onchange="validateStep3()">
                        <option value="">Select Overall Condition</option>
                        <option value="Proceed with journey">Proceed with journey</option>
                        <option value="Do not proceed with journey">Do not proceed with journey</option>
                        <option value="Report to Supervisor">Report to Supervisor</option>
                    </select>
                </div>
                <div class="error-message" id="overallConditionError" style="display: none;"></div>
            </div>

            <!-- Driver Status -->
            <div class="form-group">
                <label class="form-label required-field">Driver Status</label>
                <div class="radio-group">
                    <label class="radio-option">
                        <input type="radio" name="equipmentStatus" value="Fit to travel" onchange="handleDriverStatusChange()">
                        <span class="radio-label">Fit to travel</span>
                    </label>
                    <label class="radio-option">
                        <input type="radio" name="equipmentStatus" value="Unfit to travel" onchange="handleDriverStatusChange()">
                        <span class="radio-label">Unfit to travel</span>
                    </label>
                </div>
                <div class="error-message" id="equipmentStatusError" style="display: none;"></div>
            </div>

            <!-- Priority (only shown if unfit to travel) -->
            <div class="form-group" id="priorityGroup" style="display: none;">
                <label class="form-label">Priority</label>
                <div class="priority-container">
                    <div class="priority-option" onclick="selectPriority('Low')">
                        <div class="priority-icon">🟢</div>
                        <span>Low</span>
                    </div>
                    <div class="priority-option" onclick="selectPriority('Medium')">
                        <div class="priority-icon">🟡</div>
                        <span>Medium</span>
                    </div>
                    <div class="priority-option" onclick="selectPriority('High')">
                        <div class="priority-icon">🟠</div>
                        <span>High</span>
                    </div>
                    <div class="priority-option" onclick="selectPriority('Emergency')">
                        <div class="priority-icon">🔴</div>
                        <span>Emergency</span>
                    </div>
                </div>
            </div>

            <!-- Equipment Safe to Use -->
            <div class="form-group">
                <label class="form-label required-field">Equipment Safe to Use</label>
                <div class="radio-group">
                    <label class="radio-option">
                        <input type="radio" name="safeToUse" value="Yes" onchange="validateStep3()">
                        <span class="radio-label">Yes</span>
                    </label>
                    <label class="radio-option">
                        <input type="radio" name="safeToUse" value="No" onchange="validateStep3()">
                        <span class="radio-label">No</span>
                    </label>
                </div>
                <div class="error-message" id="safeToUseError" style="display: none;"></div>
            </div>

            <!-- Additional Note -->
            <div class="form-group">
                <label class="form-label">Additional Note</label>
                <div class="input-with-icon">
                    <textarea class="form-control textarea" id="additionalNote" rows="4" placeholder="Enter additional notes..."></textarea>
                    <button class="voice-btn" onclick="handleVoiceInput()" title="Voice Input">
                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M12 1a3 3 0 0 0-3 3v8a3 3 0 0 0 6 0V4a3 3 0 0 0-3-3z"/>
                            <path d="M19 10v2a7 7 0 0 1-14 0v-2"/>
                            <line x1="12" y1="19" x2="12" y2="23"/>
                            <line x1="8" y1="23" x2="16" y2="23"/>
                        </svg>
                    </button>
                </div>
            </div>

            <!-- Signature Section -->
            <div class="form-group">
                <label class="form-label required-field">Inspector Signature</label>
                <div class="signature-container">
                    <div class="signature-box" id="signatureBox" onclick="handleSignature()">
                        <div class="signature-placeholder" id="signaturePlaceholder">
                            <svg xmlns="http://www.w3.org/2000/svg" width="40" height="40" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
                            </svg>
                            <span>Click to add signature</span>
                        </div>
                        <div class="signature-display" id="signatureDisplay" style="display: none;">
                            <img id="signatureImage" src="" alt="Signature">
                            <button class="change-signature-btn" onclick="handleSignature()">Change Signature</button>
                        </div>
                    </div>
                </div>
                <div class="error-message" id="signatureError" style="display: none;"></div>
            </div>

            <!-- Acknowledgment -->
            <div class="acknowledgment-section">
                <div class="acknowledgment-header">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <polyline points="20,6 9,17 4,12"/>
                    </svg>
                    <h3>Operator's Acknowledgment</h3>
                </div>
                <p>I hereby certify that the inspection has been done accurately and completely according to compliance with all applicable state and Federal Motor Carrier Safety Regulations.</p>
            </div>
        </div>

        <!-- Step 3 Navigation Buttons -->
        <div class="nav-buttons">
            <button class="nav-btn back-btn" onclick="showStep2()">
                <span>← Back</span>
            </button>
            <button class="nav-btn next-btn" id="step3NextButton" onclick="handleStep3Next()" disabled>
                <span>Generate Report</span>
            </button>
        </div>
    </div>

    <!-- Signature Modal -->
    <div class="modal-overlay" id="signatureModal" style="display: none;">
        <div class="modal-content signature-modal">
            <div class="modal-header">
                <h3 class="modal-title">Add Signature</h3>
                <button class="modal-close" onclick="closeSignatureModal()">
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <line x1="18" y1="6" x2="6" y2="18"/>
                        <line x1="6" y1="6" x2="18" y2="18"/>
                    </svg>
                </button>
            </div>
            <div class="signature-pad-container">
                <canvas id="signaturePad" width="600" height="200"></canvas>
            </div>
            <div class="signature-actions">
                <button class="btn-clear" onclick="clearSignature()">Clear</button>
                <button class="btn-save" onclick="saveSignature()">Save Signature</button>
            </div>
        </div>
    </div>

    <!-- Success Modal -->
    <div class="modal-overlay" id="successModal" style="display: none;">
        <div class="modal-content success-modal">
            <div class="success-content">
                <div class="success-icon">✅</div>
                <h3>Report Generated Successfully!</h3>
                <p>Your DVIR report has been generated and saved.</p>
                <div class="success-actions">
                    <button class="btn-primary" onclick="viewPDF()">View PDF</button>
                    <button class="btn-secondary" onclick="closeSuccessModal()">Close</button>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Global variables
let selectedEquipment = null;
let equipmentData = [];
let filteredEquipment = [];
let currentTab = 'ALL';

// Step management
let currentStep = 1;
let step2Data = {};
let step3Data = {};
let signatureData = null;
let signatureStatus = false;

// Configuration
const CONFIG = {
    countries: {
        'Zambia': [
            { label: 'Kansanshi', value: 'Kansanshi' },
            { label: 'Kalumbila', value: 'Kalumbila' },
            { label: 'Kimteto', value: 'Kimteto' },
            { label: 'Kobold', value: 'Kobold' },
            { label: 'FQM Exploration', value: 'FQM Exploration' },
            { label: 'Mimosa', value: 'Mimosa' }
        ],
        'DRC': [
            { label: 'Lubumbashi', value: 'Lubumbashi' },
            { label: 'Musompo', value: 'Musompo' },
            { label: 'IME', value: 'IME' },
            { label: 'Kamoa', value: 'Kamoa' }
        ]
    }
};

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    loadEquipmentData();
    setupEventListeners();
    initializeActionButtons();
});

// Initialize action buttons as disabled
function initializeActionButtons() {
    const actionButtons = document.querySelectorAll('.action-btn');
    actionButtons.forEach(btn => {
        btn.disabled = true;
        btn.style.opacity = '0.5';
        btn.style.cursor = 'not-allowed';
    });
    console.log('Action buttons initialized as disabled');
}

// Setup event listeners
function setupEventListeners() {
    // Inspector input validation
    document.getElementById('inspector').addEventListener('input', validateForm);
    
    // Meter reading validation
    document.getElementById('meterReading').addEventListener('input', validateMeterReading);
    
    // Equipment search
    document.getElementById('equipmentSearchInput').addEventListener('input', filterEquipmentBySearch);
    
    // DVIR checkbox
    document.getElementById('dvirCheckbox').addEventListener('change', validateForm);
}

// Update projects based on country selection
function updateProjects() {
    const country = document.getElementById('country').value;
    const projectSelect = document.getElementById('project');
    
    // Clear existing options
    projectSelect.innerHTML = '<option value="">Select Project</option>';
    
    if (country && CONFIG.countries[country]) {
        projectSelect.disabled = false;
        CONFIG.countries[country].forEach(project => {
            const option = document.createElement('option');
            option.value = project.value;
            option.textContent = project.label;
            projectSelect.appendChild(option);
        });
    } else {
        projectSelect.disabled = true;
    }
    
    generateRequestNumber();
    validateForm();
}

// Generate request number
function generateRequestNumber() {
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    
    if (!country || !project) {
        document.getElementById('requestNumber').textContent = 'Select country and project first';
        return;
    }
    
    const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
    const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
    const timestamp = Date.now();
    const requestNumber = `${countryCode}-${projectCode}-DVIR-${String(timestamp).slice(-3)}`;
    
    document.getElementById('requestNumber').textContent = requestNumber;
    validateForm();
}

// Load equipment data from Firebase
async function loadEquipmentData() {
    showLoadingModal();
    
    try {
        // Fetch from Firebase database
        const response = await fetch('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/EquipmentsDataJSON/GlobalEquipmentJsonData.json');
        
        if (response.ok) {
            const firebaseData = await response.json();
            console.log('Firebase data received:', firebaseData);
            
            // Transform Firebase data to our format
            equipmentData = transformFirebaseData(firebaseData);
            console.log('Transformed equipment data:', equipmentData);
        } else {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        filteredEquipment = [...equipmentData];
        renderEquipmentList();
        showToast('success', 'Success', `Loaded ${equipmentData.length} equipment items from Firebase`);
    } catch (error) {
        console.error('Error loading equipment from Firebase:', error);
        // Fallback to mock data
        equipmentData = getMockEquipmentData();
        filteredEquipment = [...equipmentData];
        renderEquipmentList();
        showToast('error', 'Error', 'Failed to load equipment from Firebase. Using offline data.');
    } finally {
        hideLoadingModal();
    }
}

// Transform Firebase data to our equipment format
function transformFirebaseData(firebaseData) {
    const equipmentList = [];
    
    if (!firebaseData || typeof firebaseData !== 'object') {
        console.warn('Invalid Firebase data format');
        return equipmentList;
    }
    
    // Iterate through Firebase data
    Object.keys(firebaseData).forEach(equipmentId => {
        const equipment = firebaseData[equipmentId];
        
        if (equipment && typeof equipment === 'object') {
            // Extract equipment information with fallbacks
            const transformedEquipment = {
                id: equipmentId,
                equipmentName: equipment.equipmentName || equipment.name || equipment.equipmentName || 'Unknown Equipment',
                equipmentNumber: equipment.equipmentNumber || equipment.equipmentId || equipmentId,
                mainCategory: equipment.mainCategory || equipment.category || equipment.equipmentCategory || 'General',
                model: equipment.model || equipment.equipmentModel || equipment.equipmentModel || 'N/A',
                serialNumber: equipment.serialNumber || equipment.serial || equipment.equipmentSerial || 'N/A',
                meterReading: equipment.meterReading || equipment.currentMeter || equipment.meterReading || '0',
                meterUnit: equipment.meterUnit || equipment.unit || equipment.meterUnit || 'Hours',
                status: equipment.status || equipment.equipmentStatus || equipment.status || 'Available',
                location: equipment.location || equipment.equipmentLocation || 'N/A',
                manufacturer: equipment.manufacturer || equipment.equipmentManufacturer || 'N/A',
                year: equipment.year || equipment.equipmentYear || 'N/A',
                color: equipment.color || equipment.equipmentColor || 'N/A',
                notes: equipment.notes || equipment.equipmentNotes || '',
                lastInspection: equipment.lastInspection || equipment.lastInspectionDate || 'N/A',
                nextInspection: equipment.nextInspection || equipment.nextInspectionDate || 'N/A'
            };
            
            equipmentList.push(transformedEquipment);
        }
    });
    
    console.log(`Transformed ${equipmentList.length} equipment items`);
    return equipmentList;
}

// Mock equipment data for demonstration
function getMockEquipmentData() {
    return [
        {
            id: '1',
            equipmentName: 'Excavator CAT 320D',
            equipmentNumber: 'EQ-001',
            mainCategory: 'Heavy Vehicles TDC',
            model: 'CAT 320D',
            meterReading: 1250,
            meterUnit: 'Hours',
            status: 'Available'
        },
        {
            id: '2',
            equipmentName: 'Pickup Truck Ford F-150',
            equipmentNumber: 'EQ-002',
            mainCategory: 'Light Vehicles TDC',
            model: 'Ford F-150',
            meterReading: 45000,
            meterUnit: 'Miles',
            status: 'Available'
        },
        {
            id: '3',
            equipmentName: 'Bulldozer CAT D6T',
            equipmentNumber: 'EQ-003',
            mainCategory: 'Heavy Vehicles TDC',
            model: 'CAT D6T',
            meterReading: 2100,
            meterUnit: 'Hours',
            status: 'Maintenance Required'
        },
        {
            id: '4',
            equipmentName: 'SUV Toyota Land Cruiser',
            equipmentNumber: 'EQ-004',
            mainCategory: 'Light Vehicles TDC',
            model: 'Toyota Land Cruiser',
            meterReading: 32000,
            meterUnit: 'Miles',
            status: 'Available'
        }
    ];
}

// Open equipment modal
function openEquipmentModal() {
    document.getElementById('equipmentModal').style.display = 'flex';
    renderEquipmentList();
}

// Refresh equipment data from Firebase
async function refreshEquipmentData() {
    showToast('info', 'Refreshing', 'Loading equipment data from Firebase...');
    await loadEquipmentData();
}

// Close equipment modal
function closeEquipmentModal() {
    document.getElementById('equipmentModal').style.display = 'none';
}

// Filter equipment by tab
function filterEquipment(tab) {
    currentTab = tab;
    
    // Update tab buttons
    document.querySelectorAll('.tab-button').forEach(btn => btn.classList.remove('active'));
    event.target.classList.add('active');
    
    // Filter equipment
    if (tab === 'ALL') {
        filteredEquipment = [...equipmentData];
    } else if (tab === 'LV') {
        filteredEquipment = equipmentData.filter(item => item.mainCategory === 'Light Vehicles TDC');
    } else if (tab === 'HV') {
        filteredEquipment = equipmentData.filter(item => item.mainCategory === 'Heavy Vehicles TDC');
    }
    
    renderEquipmentList();
}

// Filter equipment by search
function filterEquipmentBySearch() {
    const searchQuery = document.getElementById('equipmentSearchInput').value.toLowerCase();
    
    if (currentTab === 'ALL') {
        filteredEquipment = equipmentData.filter(item => 
            item.equipmentName.toLowerCase().includes(searchQuery) ||
            item.equipmentNumber.toLowerCase().includes(searchQuery) ||
            item.model.toLowerCase().includes(searchQuery)
        );
    } else {
        const baseFilter = currentTab === 'LV' ? 'Light Vehicles TDC' : 'Heavy Vehicles TDC';
        filteredEquipment = equipmentData.filter(item => 
            item.mainCategory === baseFilter && (
                item.equipmentName.toLowerCase().includes(searchQuery) ||
                item.equipmentNumber.toLowerCase().includes(searchQuery) ||
                item.model.toLowerCase().includes(searchQuery)
            )
        );
    }
    
    renderEquipmentList();
}

// Render equipment list
function renderEquipmentList() {
    const equipmentList = document.getElementById('equipmentList');
    equipmentList.innerHTML = '';
    
    if (filteredEquipment.length === 0) {
        equipmentList.innerHTML = '<div class="no-equipment">No equipment found</div>';
        return;
    }
    
    filteredEquipment.forEach(equipment => {
        const equipmentItem = createEquipmentItem(equipment);
        equipmentList.appendChild(equipmentItem);
    });
}

// Create equipment item element
function createEquipmentItem(equipment) {
    const item = document.createElement('div');
    item.className = 'equipment-item';
    
    const statusColor = getStatusColor(equipment.status);
    const isSelected = selectedEquipment && selectedEquipment.id === equipment.id;
    
    item.innerHTML = `
        <div class="equipment-info" onclick="selectEquipment('${equipment.id}')">
            <div class="equipment-header">
                <span class="equipment-name">${equipment.equipmentName || 'Equipment'}</span>
                <span class="equipment-number">${equipment.equipmentNumber || 'N/A'}</span>
            </div>
            <div class="equipment-details">
                <div class="detail-row">
                    <span class="detail-label">Category</span>
                    <span class="detail-value">${equipment.mainCategory || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Model</span>
                    <span class="detail-value">${equipment.model || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Serial Number</span>
                    <span class="detail-value">${equipment.serialNumber || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Meter Reading</span>
                    <span class="detail-value">${equipment.meterReading || '0'} ${equipment.meterUnit || 'Hours'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Location</span>
                    <span class="detail-value">${equipment.location || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Manufacturer</span>
                    <span class="detail-value">${equipment.manufacturer || 'N/A'}</span>
                </div>
                <div class="detail-row no-border">
                    <span class="detail-label">Status</span>
                    <span class="detail-value status" style="color: ${statusColor}">${equipment.status || 'Available'}</span>
                </div>
            </div>
        </div>
        <button class="select-equipment-btn ${isSelected ? 'selected' : ''}" onclick="selectEquipment('${equipment.id}')">
            ${isSelected ? 'Selected' : 'Select'}
        </button>
    `;
    
    return item;
}

// Get status color
function getStatusColor(status) {
    switch (status) {
        case 'Available': return '#2ecc71';
        case 'Maintenance Required': return '#f1c40f';
        case 'In Maintenance': return '#e67e22';
        case 'Breakdown': return '#e74c3c';
        case 'Deploy': return '#3498db';
        default: return '#666';
    }
}

// Select equipment
function selectEquipment(equipmentId) {
    console.log('=== EQUIPMENT SELECTION ===');
    console.log('Equipment ID:', equipmentId);
    
    selectedEquipment = equipmentData.find(item => item.id === equipmentId);
    
    if (selectedEquipment) {
        console.log('✅ Equipment found and selected');
        console.log('');
        
        console.log('📋 EQUIPMENT DETAILS:');
        console.log('  Equipment ID:', selectedEquipment.id);
        console.log('  Equipment Name:', selectedEquipment.equipmentName);
        console.log('  Equipment Number:', selectedEquipment.equipmentNumber);
        console.log('  Main Category:', selectedEquipment.mainCategory);
        console.log('  Model:', selectedEquipment.model);
        console.log('  Serial Number:', selectedEquipment.serialNumber);
        console.log('  Equipment Status:', selectedEquipment.status);
        console.log('  Location:', selectedEquipment.location);
        console.log('  Manufacturer:', selectedEquipment.manufacturer);
        console.log('  Year:', selectedEquipment.year);
        console.log('  Color:', selectedEquipment.color);
        console.log('  Notes:', selectedEquipment.notes);
        console.log('  Last Inspection:', selectedEquipment.lastInspection);
        console.log('  Next Inspection:', selectedEquipment.nextInspection);
        console.log('');
        
        console.log('📊 METER READING DETAILS:');
        console.log('  Current Meter Reading:', selectedEquipment.meterReading);
        console.log('  Meter Unit:', selectedEquipment.meterUnit);
        console.log('');
        
        // Update UI
        document.getElementById('equipmentSelectText').textContent = 'Change equipment';
        document.getElementById('equipmentDetailsBox').style.display = 'block';
        
        // Update equipment info with more details
        const equipmentInfo = `Equipment: ${selectedEquipment.equipmentName || 'N/A'} | Number: ${selectedEquipment.equipmentNumber || 'N/A'} | Category: ${selectedEquipment.mainCategory || 'N/A'} | Model: ${selectedEquipment.model || 'N/A'}`;
        document.getElementById('equipmentInfoText').textContent = equipmentInfo;
        
        // Update meter reading
        document.getElementById('meterReading').value = selectedEquipment.meterReading || '';
        document.getElementById('meterUnit').textContent = selectedEquipment.meterUnit || 'Hours';
        document.getElementById('currentMeterValue').textContent = `${selectedEquipment.meterReading || '0'} ${selectedEquipment.meterUnit || 'Hours'}`;
        
        console.log('🔄 UI UPDATES:');
        console.log('  Equipment Info Text:', equipmentInfo);
        console.log('  Meter Reading Input Value:', selectedEquipment.meterReading || '');
        console.log('  Meter Unit Display:', selectedEquipment.meterUnit || 'Hours');
        console.log('  Current Meter Value Display:', `${selectedEquipment.meterReading || '0'} ${selectedEquipment.meterUnit || 'Hours'}`);
        console.log('');
        
        // Update all equipment items to show selection state
        updateEquipmentSelectionUI();
        
        // Close modal
        closeEquipmentModal();
        
        // Validate form
        validateForm();
        validateMeterReading();
        
        console.log('✅ Equipment selection completed successfully');
        console.log('=== END EQUIPMENT SELECTION ===');
        
        showToast('success', 'Equipment Selected', `${selectedEquipment.equipmentName} has been selected`);
    } else {
        console.log('❌ Equipment not found with ID:', equipmentId);
        console.log('Available equipment IDs:', equipmentData.map(eq => eq.id));
    }
}

// Update equipment selection UI
function updateEquipmentSelectionUI() {
    // Re-render the equipment list to show updated selection states
    renderEquipmentList();
}

// Validate meter reading
function validateMeterReading() {
    console.log('=== METER READING VALIDATION ===');
    
    const meterReadingInput = document.getElementById('meterReading').value;
    const meterReading = parseFloat(meterReadingInput);
    const errorDiv = document.getElementById('meterReadingError');
    
    console.log('  Meter Reading Input (raw):', meterReadingInput);
    console.log('  Meter Reading Input (parsed):', meterReading);
    console.log('  Selected Equipment:', selectedEquipment ? 'Yes' : 'No');
    
    if (selectedEquipment) {
        console.log('  Equipment Current Meter Reading:', selectedEquipment.meterReading);
        console.log('  Equipment Meter Unit:', selectedEquipment.meterUnit);
        console.log('  Equipment Meter Reading (parsed):', parseFloat(selectedEquipment.meterReading));
    }
    
    if (selectedEquipment && meterReading && meterReading <= parseFloat(selectedEquipment.meterReading)) {
        const errorMessage = `Meter reading must be greater than ${selectedEquipment.meterReading || '0'}.`;
        console.log('❌ Meter Reading Validation Failed:', errorMessage);
        errorDiv.textContent = errorMessage;
        errorDiv.style.display = 'block';
        console.log('=== END METER READING VALIDATION (FAILED) ===');
        return false;
    } else {
        console.log('✅ Meter Reading Validation Passed');
        errorDiv.style.display = 'none';
        console.log('=== END METER READING VALIDATION (PASSED) ===');
        return true;
    }
}

// Validate form
function validateForm() {
    console.log('=== FORM VALIDATION ===');
    
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    const inspector = document.getElementById('inspector').value.trim();
    const dvirChecked = document.getElementById('dvirCheckbox').checked;
    
    console.log('  Country:', country, country ? '✅' : '❌');
    console.log('  Project:', project, project ? '✅' : '❌');
    console.log('  Inspector:', inspector, inspector ? '✅' : '❌');
    console.log('  Selected Equipment:', selectedEquipment ? '✅' : '❌');
    console.log('  DVIR Checked:', dvirChecked, dvirChecked ? '✅' : '❌');
    
    const meterReadingValid = validateMeterReading();
    console.log('  Meter Reading Valid:', meterReadingValid, meterReadingValid ? '✅' : '❌');
    
    const isValid = country && project && inspector && selectedEquipment && dvirChecked && meterReadingValid;
    
    console.log('  Overall Form Valid:', isValid, isValid ? '✅' : '❌');
    console.log('  Next Button Disabled:', !isValid);
    console.log('=== END FORM VALIDATION ===');
    
    document.getElementById('nextButton').disabled = !isValid;
    
    return isValid;
}

// Handle next button press
function handleNextPress() {
    console.log('=== DVIR STEP 1 - NEXT BUTTON CLICKED ===');
    console.log('Timestamp:', new Date().toISOString());
    console.log('');
    
    if (!validateForm()) {
        console.log('❌ Form validation failed - cannot proceed');
        showToast('error', 'Error', 'Please fill all required fields and ensure meter reading is valid.');
        return;
    }
    
    console.log('✅ Form validation passed - proceeding with data collection');
    console.log('');
    
    // Collect all form data
    const requestNumber = document.getElementById('requestNumber').textContent;
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    const inspector = document.getElementById('inspector').value;
    const meterReading = document.getElementById('meterReading').value;
    const meterUnit = document.getElementById('meterUnit').textContent;
    const currentMeterValue = document.getElementById('currentMeterValue').textContent;
    const dvirChecked = document.getElementById('dvirCheckbox').checked;
    
    console.log('📋 REQUEST INFORMATION:');
    console.log('  Request Number:', requestNumber);
    console.log('');
    
    console.log('🌍 LOCATION INFORMATION:');
    console.log('  Country:', country);
    console.log('  Project:', project);
    console.log('');
    
    console.log('👤 INSPECTOR INFORMATION:');
    console.log('  Inspector Name:', inspector);
    console.log('');
    
    console.log('🚛 EQUIPMENT INFORMATION:');
    if (selectedEquipment) {
        console.log('  Equipment ID:', selectedEquipment.id);
        console.log('  Equipment Name:', selectedEquipment.equipmentName);
        console.log('  Equipment Number:', selectedEquipment.equipmentNumber);
        console.log('  Main Category:', selectedEquipment.mainCategory);
        console.log('  Model:', selectedEquipment.model);
        console.log('  Serial Number:', selectedEquipment.serialNumber);
        console.log('  Equipment Status:', selectedEquipment.status);
        console.log('  Location:', selectedEquipment.location);
        console.log('  Manufacturer:', selectedEquipment.manufacturer);
        console.log('  Year:', selectedEquipment.year);
        console.log('  Color:', selectedEquipment.color);
        console.log('  Notes:', selectedEquipment.notes);
        console.log('  Last Inspection:', selectedEquipment.lastInspection);
        console.log('  Next Inspection:', selectedEquipment.nextInspection);
    } else {
        console.log('  ❌ No equipment selected');
    }
    console.log('');
    
    console.log('📊 METER READING INFORMATION:');
    console.log('  Current Meter Reading (from equipment):', selectedEquipment ? selectedEquipment.meterReading : 'N/A');
    console.log('  Meter Unit (from equipment):', selectedEquipment ? selectedEquipment.meterUnit : 'N/A');
    console.log('  New Meter Reading (user input):', meterReading);
    console.log('  Meter Unit (display):', meterUnit);
    console.log('  Current Meter Value (display):', currentMeterValue);
    console.log('  Meter Reading Validation:', validateMeterReading() ? '✅ Valid' : '❌ Invalid');
    console.log('');
    
    console.log('✅ INSPECTION TYPES:');
    console.log('  DVIR Checked:', dvirChecked);
    console.log('');
    
    // Save step 1 data
    const step1Data = {
        requestNumber: requestNumber,
        country: country,
        project: project,
        inspector: inspector,
        selectedEquipment: selectedEquipment,
        meterReading: meterReading,
        meterUnit: meterUnit,
        currentMeterValue: currentMeterValue,
        inspectionTypes: {
            DVIR: dvirChecked
        },
        timestamp: new Date().getTime(),
        dateTime: new Date().toISOString()
    };
    
    console.log('📦 COMPLETE STEP 1 DATA OBJECT:');
    console.log(JSON.stringify(step1Data, null, 2));
    console.log('');
    
    console.log('💾 SAVING TO SESSION STORAGE...');
    // Store in session storage
    sessionStorage.setItem('dvirStep1Data', JSON.stringify(step1Data));
    console.log('✅ Data saved to session storage successfully');
    console.log('');
    
    console.log('🔄 REDIRECTING TO STEP 2...');
    console.log('=== END OF STEP 1 DATA LOGGING ===');
    
    // Show step 2 instead of redirecting
    showStep2();
}

// Handle back button press
function handleBackPress() {
    if (confirm('Do you want to exit this step? Any unsaved data will be lost.')) {
        window.history.back();
    }
}

// Show loading modal
function showLoadingModal() {
    document.getElementById('loadingModal').style.display = 'flex';
}

// Hide loading modal
function hideLoadingModal() {
    document.getElementById('loadingModal').style.display = 'none';
}

// Show toast notification
function showToast(type, title, message) {
    const toast = document.createElement('div');
    toast.className = `toast toast-${type}`;
    toast.innerHTML = `
        <div class="toast-content">
            <h4>${title}</h4>
            <p>${message}</p>
        </div>
        <button class="toast-close" onclick="this.parentElement.remove()">×</button>
    `;
    
    document.getElementById('toastContainer').appendChild(toast);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (toast.parentElement) {
            toast.remove();
        }
    }, 5000);
}

// ==================== STEP MANAGEMENT FUNCTIONS ====================

// Show Step 1
function showStep1() {
    currentStep = 1;
    document.getElementById('step1Container').style.display = 'block';
    document.getElementById('step2Container').style.display = 'none';
    document.getElementById('step3Container').style.display = 'none';
    console.log('Switched to Step 1');
}

// Show Step 2
function showStep2() {
    currentStep = 2;
    document.getElementById('step1Container').style.display = 'none';
    document.getElementById('step2Container').style.display = 'block';
    document.getElementById('step3Container').style.display = 'none';
    console.log('Switched to Step 2');
    loadStep2Data();
    validateStep2();
}

// Load existing Step 2 data
function loadStep2Data() {
    // Restore button states
    Object.entries(step2Data).forEach(([key, value]) => {
        if (key.includes('_Status')) {
            const [section, itemId, field] = key.split('_');
            const itemContainer = document.querySelector(`[data-item-id="${itemId}"]`);
            if (itemContainer) {
                const buttons = itemContainer.querySelectorAll('.status-btn');
                buttons.forEach(btn => {
                    btn.classList.remove('selected');
                    if (btn.textContent.trim() === value.status) {
                        btn.classList.add('selected');
                    }
                });
                
                // Restore note if exists
                if (value.note) {
                    const noteInput = itemContainer.querySelector('.note-input');
                    if (noteInput) {
                        noteInput.value = value.note;
                    }
                }
            }
        }
        
        // Restore photo indicators and previews
        if (key.startsWith('photo_')) {
            const itemId = key.replace('photo_', '');
            updatePhotoIndicator(itemId, true);
            showImagePreview(itemId, value.data);
        }
    });
    
    console.log('Step 2 data loaded:', step2Data);
}

// Show Step 3
function showStep3() {
    currentStep = 3;
    document.getElementById('step1Container').style.display = 'none';
    document.getElementById('step2Container').style.display = 'none';
    document.getElementById('step3Container').style.display = 'block';
    console.log('Switched to Step 3');
    loadFaultyItems();
    validateStep3();
}

// ==================== STEP 2 FUNCTIONS ====================

// Toggle section expansion
function toggleSection(sectionName) {
    const content = document.getElementById(`${sectionName}-content`);
    const icon = document.getElementById(`${sectionName}-icon`);
    
    if (content.style.display === 'none') {
        content.style.display = 'block';
        icon.textContent = '▲';
    } else {
        content.style.display = 'none';
        icon.textContent = '▼';
    }
    console.log(`Toggled ${sectionName} section`);
}

// Set status for checklist item
function setStatus(section, itemId, status) {
    const key = `${section}_${itemId}_Status`;
    step2Data[key] = { status: status, note: step2Data[key]?.note || '' };
    
    // Update button states for this specific item
    const itemContainer = document.querySelector(`[data-item-id="${itemId}"]`);
    if (itemContainer) {
        const buttons = itemContainer.querySelectorAll('.status-btn');
        buttons.forEach(btn => {
            btn.classList.remove('selected');
            if (btn.textContent.trim() === status) {
                btn.classList.add('selected');
            }
        });
        
        // Enable action buttons after selection
        const actionButtons = itemContainer.querySelectorAll('.action-btn');
        actionButtons.forEach(btn => {
            btn.disabled = false;
            btn.style.opacity = '1';
            btn.style.cursor = 'pointer';
        });
    }
    
    console.log(`Set status for ${key}: ${status}`);
    console.log('Current step2Data:', step2Data);
    validateStep2();
}

// Toggle note input for an item
function toggleNote(itemId) {
    const noteContainer = document.getElementById(`note-${itemId}`);
    if (noteContainer) {
        const isVisible = noteContainer.style.display !== 'none';
        noteContainer.style.display = isVisible ? 'none' : 'block';
        
        if (!isVisible) {
            // Focus on the input when showing
            const input = noteContainer.querySelector('.note-input');
            if (input) {
                input.focus();
            }
        }
        
        console.log(`Toggled note for item: ${itemId}, visible: ${!isVisible}`);
    }
}

// Update note for an item
function updateNote(section, itemId, note) {
    const key = `${section}_${itemId}_Status`;
    if (step2Data[key]) {
        step2Data[key].note = note;
    } else {
        step2Data[key] = { status: '', note: note };
    }
    
    console.log(`Updated note for ${key}: ${note}`);
    console.log('Current step2Data:', step2Data);
    validateStep2();
}

// Take photo (enhanced with file input)
function takePhoto(itemId) {
    // Create a file input element
    const fileInput = document.createElement('input');
    fileInput.type = 'file';
    fileInput.accept = 'image/*';
    fileInput.capture = 'environment'; // Use back camera on mobile
    
    fileInput.onchange = function(event) {
        const file = event.target.files[0];
        if (file) {
            // Validate file size (max 5MB)
            if (file.size > 5 * 1024 * 1024) {
                showToast('error', 'Error', 'Image file is too large. Please select an image smaller than 5MB.');
                return;
            }
            
            // Validate file type
            if (!file.type.startsWith('image/')) {
                showToast('error', 'Error', 'Please select a valid image file.');
                return;
            }
            
            // Process the image
            const reader = new FileReader();
            reader.onload = function(e) {
                const imageData = e.target.result;
                
                // Store image data
                const key = `photo_${itemId}`;
                step2Data[key] = {
                    data: imageData,
                    filename: file.name,
                    timestamp: new Date().getTime()
                };
                
                // Show success message
                showToast('success', 'Photo Added', `Photo captured for item ${itemId}`);
                console.log(`Photo captured for item: ${itemId}`, {
                    filename: file.name,
                    size: file.size,
                    type: file.type
                });
                
                // Update UI to show photo was taken and display preview
                updatePhotoIndicator(itemId, true);
                showImagePreview(itemId, imageData);
            };
            
            reader.readAsDataURL(file);
        }
    };
    
    // Trigger file input
    fileInput.click();
    console.log(`Photo capture initiated for item: ${itemId}`);
}

// Update photo indicator
function updatePhotoIndicator(itemId, hasPhoto) {
    const itemContainer = document.querySelector(`[data-item-id="${itemId}"]`);
    if (itemContainer) {
        const photoButton = itemContainer.querySelector('[onclick*="takePhoto"]');
        if (photoButton) {
            if (hasPhoto) {
                photoButton.style.backgroundColor = '#2ecc71';
                photoButton.title = 'Photo Added - Click to change';
            } else {
                photoButton.style.backgroundColor = '';
                photoButton.title = 'Take Photo';
            }
        }
    }
}

// Show image preview
function showImagePreview(itemId, imageData) {
    const previewContainer = document.getElementById(`image-preview-${itemId}`);
    const previewImage = document.getElementById(`preview-img-${itemId}`);
    
    if (previewContainer && previewImage) {
        previewImage.src = imageData;
        previewContainer.style.display = 'block';
        console.log(`Image preview shown for item: ${itemId}`);
    }
}

// Remove image
function removeImage(itemId) {
    const key = `photo_${itemId}`;
    delete step2Data[key];
    
    // Hide preview container
    const previewContainer = document.getElementById(`image-preview-${itemId}`);
    if (previewContainer) {
        previewContainer.style.display = 'none';
    }
    
    // Update photo indicator
    updatePhotoIndicator(itemId, false);
    
    // Show success message
    showToast('info', 'Photo Removed', `Photo removed for item ${itemId}`);
    console.log(`Image removed for item: ${itemId}`);
    
    // Validate step 2
    validateStep2();
}

// Copy item (enhanced functionality)
function copyItem(itemId) {
    // Find the item text
    const itemContainer = document.querySelector(`[data-item-id="${itemId}"]`);
    if (itemContainer) {
        const itemText = itemContainer.querySelector('.item-text').textContent;
        
        // Copy to clipboard
        if (navigator.clipboard) {
            navigator.clipboard.writeText(itemText).then(() => {
                showToast('success', 'Copied', 'Item text copied to clipboard');
                console.log(`Copied item text: ${itemText}`);
            }).catch(err => {
                console.error('Failed to copy text: ', err);
                showToast('error', 'Error', 'Failed to copy text');
            });
        } else {
            // Fallback for older browsers
            const textArea = document.createElement('textarea');
            textArea.value = itemText;
            document.body.appendChild(textArea);
            textArea.select();
            try {
                document.execCommand('copy');
                showToast('success', 'Copied', 'Item text copied to clipboard');
                console.log(`Copied item text: ${itemText}`);
            } catch (err) {
                console.error('Failed to copy text: ', err);
                showToast('error', 'Error', 'Failed to copy text');
            }
            document.body.removeChild(textArea);
        }
    }
}

// Validate Step 2
function validateStep2() {
    const hasSelections = Object.keys(step2Data).length > 0;
    const nextButton = document.getElementById('step2NextButton');
    
    nextButton.disabled = !hasSelections;
    console.log(`Step 2 validation: ${hasSelections ? 'Valid' : 'Invalid'}`);
    
    return hasSelections;
}

// Handle Step 2 Next
function handleStep2Next() {
    if (!validateStep2()) {
        showToast('error', 'Error', 'Please complete at least one checklist item.');
        return;
    }
    
    console.log('Step 2 data:', step2Data);
    showToast('success', 'Step 2 Complete', 'Checklist data saved successfully');
    showStep3();
}

// ==================== STEP 3 FUNCTIONS ====================

// Load faulty items count
function loadFaultyItems() {
    let faultyCount = 0;
    Object.values(step2Data).forEach(item => {
        if (item.status === 'No' || item.status === 'Fail') {
            faultyCount++;
        }
    });
    
    document.getElementById('faultyItemsCount').textContent = faultyCount;
    console.log(`Faulty items count: ${faultyCount}`);
}

// Handle driver status change
function handleDriverStatusChange() {
    const status = document.querySelector('input[name="equipmentStatus"]:checked');
    const priorityGroup = document.getElementById('priorityGroup');
    
    if (status && status.value === 'Unfit to travel') {
        priorityGroup.style.display = 'block';
    } else {
        priorityGroup.style.display = 'none';
        step3Data.priority = null;
    }
    
    validateStep3();
    console.log('Driver status changed:', status ? status.value : 'None');
}

// Select priority
function selectPriority(priority) {
    step3Data.priority = priority;
    
    // Update UI
    document.querySelectorAll('.priority-option').forEach(option => {
        option.classList.remove('selected');
        if (option.textContent.includes(priority)) {
            option.classList.add('selected');
        }
    });
    
    console.log('Priority selected:', priority);
    validateStep3();
}

// Handle voice input (placeholder)
function handleVoiceInput() {
    console.log('Voice input triggered');
    showToast('info', 'Voice Input', 'Voice input functionality would be implemented here');
}

// Handle signature
function handleSignature() {
    document.getElementById('signatureModal').style.display = 'flex';
    console.log('Signature modal opened');
}

// Close signature modal
function closeSignatureModal() {
    document.getElementById('signatureModal').style.display = 'none';
    console.log('Signature modal closed');
}

// Clear signature
function clearSignature() {
    const canvas = document.getElementById('signaturePad');
    const ctx = canvas.getContext('2d');
    ctx.clearRect(0, 0, canvas.width, canvas.height);
    console.log('Signature cleared');
}

// Save signature
function saveSignature() {
    const canvas = document.getElementById('signaturePad');
    const dataURL = canvas.toDataURL('image/png');
    
    signatureData = dataURL;
    signatureStatus = true;
    
    // Update UI
    document.getElementById('signaturePlaceholder').style.display = 'none';
    document.getElementById('signatureDisplay').style.display = 'block';
    document.getElementById('signatureImage').src = dataURL;
    
    closeSignatureModal();
    validateStep3();
    console.log('Signature saved');
}

// Validate Step 3
function validateStep3() {
    const overallCondition = document.getElementById('overallCondition').value;
    const equipmentStatus = document.querySelector('input[name="equipmentStatus"]:checked');
    const safeToUse = document.querySelector('input[name="safeToUse"]:checked');
    
    let isValid = overallCondition && equipmentStatus && safeToUse && signatureStatus;
    
    // Clear previous errors
    document.getElementById('overallConditionError').style.display = 'none';
    document.getElementById('equipmentStatusError').style.display = 'none';
    document.getElementById('safeToUseError').style.display = 'none';
    document.getElementById('signatureError').style.display = 'none';
    
    // Show specific errors
    if (!overallCondition) {
        document.getElementById('overallConditionError').textContent = 'Overall Condition is required';
        document.getElementById('overallConditionError').style.display = 'block';
    }
    
    if (!equipmentStatus) {
        document.getElementById('equipmentStatusError').textContent = 'Driver Status is required';
        document.getElementById('equipmentStatusError').style.display = 'block';
    }
    
    if (!safeToUse) {
        document.getElementById('safeToUseError').textContent = 'Equipment Safe to Use is required';
        document.getElementById('safeToUseError').style.display = 'block';
    }
    
    if (!signatureStatus) {
        document.getElementById('signatureError').textContent = 'Inspector Signature is required';
        document.getElementById('signatureError').style.display = 'block';
    }
    
    const nextButton = document.getElementById('step3NextButton');
    nextButton.disabled = !isValid;
    
    console.log(`Step 3 validation: ${isValid ? 'Valid' : 'Invalid'}`);
    return isValid;
}

// Handle Step 3 Next (Generate Report)
function handleStep3Next() {
    if (!validateStep3()) {
        showToast('error', 'Error', 'Please complete all required fields.');
        return;
    }
    
    // Collect Step 3 data
    step3Data = {
        overallCondition: document.getElementById('overallCondition').value,
        equipmentStatus: document.querySelector('input[name="equipmentStatus"]:checked').value,
        priority: step3Data.priority || null,
        safeToUse: document.querySelector('input[name="safeToUse"]:checked').value,
        additionalNote: document.getElementById('additionalNote').value,
        signatureStatus: signatureStatus,
        signatureData: signatureData,
        faultyItems: parseInt(document.getElementById('faultyItemsCount').textContent)
    };
    
    console.log('Step 3 data:', step3Data);
    generateReport();
}

// Generate PDF Report
async function generateReport() {
    try {
        showLoadingModal();
        
        // Get Step 1 data
        const step1Data = JSON.parse(sessionStorage.getItem('dvirStep1Data') || '{}');
        
        // Create report data
        const reportData = {
            id: Date.now().toString(),
            dvirNumber: `DVIR-${String(Date.now()).slice(-6)}`,
            step1: step1Data,
            step2: step2Data,
            step3: step3Data,
            createdAt: new Date().toISOString()
        };
        
        console.log('Generating report with data:', reportData);
        
        // Generate PDF HTML
        const pdfHtml = generatePDFHTML(reportData);
        
        // Create blob and download
        const blob = new Blob([pdfHtml], { type: 'text/html' });
        const url = URL.createObjectURL(blob);
        
        // Create download link
        const a = document.createElement('a');
        a.href = url;
        a.download = `DVIR_Report_${reportData.dvirNumber}.html`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
        
        hideLoadingModal();
        showSuccessModal();
        
        console.log('Report generated successfully');
        
    } catch (error) {
        console.error('Error generating report:', error);
        hideLoadingModal();
        showToast('error', 'Error', 'Failed to generate report. Please try again.');
    }
}

// Generate PDF HTML
function generatePDFHTML(reportData) {
    const { step1, step2, step3 } = reportData;
    
    // Build checklist rows
    let checklistRows = '';
    Object.entries(step2).forEach(([key, value]) => {
        if (key.startsWith('photo_')) return; // Skip photo entries
        
        const [section, itemId, field] = key.split('_');
        const isYes = value.status === 'Yes';
        const isNo = value.status === 'No';
        
        // Get photo data if exists
        const photoKey = `photo_${itemId}`;
        const photoData = step2[photoKey];
        const photoHtml = photoData ? `<br><img src="${photoData.data}" style="max-width: 200px; max-height: 150px; border: 1px solid #ccc; margin-top: 5px;" alt="Photo">` : '';
        
        checklistRows += `
            <tr>
                <td>${getItemText(section, itemId)}</td>
                <td>A</td>
                <td>${isYes ? '✓' : ''}</td>
                <td>${isNo ? '✓' : ''}</td>
                <td>${isYes ? '✓' : ''}</td>
                <td>${isNo ? '✓' : ''}</td>
                <td>${value.note || ''}${photoHtml}</td>
            </tr>
        `;
    });
    
    return `
        <!DOCTYPE html>
        <html>
        <head>
            <title>DVIR Report ${reportData.dvirNumber}</title>
            <style>
                body { font-family: Arial, sans-serif; margin: 20px; }
                .header { text-align: center; margin-bottom: 30px; }
                .section { margin-bottom: 20px; border: 1px solid #ccc; padding: 15px; }
                table { width: 100%; border-collapse: collapse; margin: 10px 0; }
                th, td { border: 1px solid #ccc; padding: 8px; text-align: left; }
                th { background-color: #f0f0f0; }
                .signature { margin-top: 30px; text-align: center; }
                .signature img { max-width: 300px; max-height: 150px; border: 1px solid #ccc; }
            </style>
        </head>
        <body>
            <div class="header">
                <h1>DAILY DD RIG PRE-START CHECKLIST</h1>
                <p><strong>Report #:</strong> ${reportData.dvirNumber}</p>
                <p><strong>Date:</strong> ${new Date().toLocaleDateString()}</p>
            </div>
            
            <div class="section">
                <h3>Equipment Information</h3>
                <p><strong>Equipment:</strong> ${step1.selectedEquipment?.equipmentName || 'N/A'}</p>
                <p><strong>Equipment Number:</strong> ${step1.selectedEquipment?.equipmentNumber || 'N/A'}</p>
                <p><strong>Meter Reading:</strong> ${step1.meterReading || 'N/A'} ${step1.selectedEquipment?.meterUnit || 'Hours'}</p>
                <p><strong>Inspector:</strong> ${step1.inspector || 'N/A'}</p>
                <p><strong>Country:</strong> ${step1.country || 'N/A'}</p>
                <p><strong>Project:</strong> ${step1.project || 'N/A'}</p>
            </div>
            
            <div class="section">
                <h3>Inspection Checklist</h3>
                <table>
                    <tr>
                        <th>Description</th>
                        <th>Class</th>
                        <th>In Order (Yes)</th>
                        <th>In Order (No)</th>
                        <th>Defective (Yes)</th>
                        <th>Defective (No)</th>
                        <th>Comment</th>
                    </tr>
                    ${checklistRows}
                </table>
            </div>
            
            <div class="section">
                <h3>Summary</h3>
                <p><strong>Faulty Items:</strong> ${step3.faultyItems}</p>
                <p><strong>Overall Condition:</strong> ${step3.overallCondition}</p>
                <p><strong>Driver Status:</strong> ${step3.equipmentStatus}</p>
                ${step3.priority ? `<p><strong>Priority:</strong> ${step3.priority}</p>` : ''}
                <p><strong>Safe to Use:</strong> ${step3.safeToUse}</p>
                <p><strong>Additional Note:</strong> ${step3.additionalNote || 'N/A'}</p>
            </div>
            
            <div class="signature">
                <h3>Inspector Signature</h3>
                ${step3.signatureData ? `<img src="${step3.signatureData}" alt="Signature">` : '<p>No signature provided</p>'}
            </div>
        </body>
        </html>
    `;
}

// Get item text by section and item ID
function getItemText(section, itemId) {
    const items = {
        requirements: {
            req1: "I hold a valid driver's licence",
            req2: "I will obey speed limits and road signs",
            req3: "I will not overload the vehicle and ensure the load is secure",
            req4: "I have completed the vehicle pre check list and satisfied all is order for my trip",
            req5: "I am fit for work – (Not under the influence of alcohol, am well rested, am healthy)",
            req6: "I have read and understand the Titan driving procedure, National driving rules and cliental rules",
            req7: "I will ensure every passenger and myself wears a seat belt (No unauthorised passengers)"
        },
        hazards: {
            haz1: "Pedestrians crossing the road, school children and drunkards",
            haz2: "Motor bikes and taxis stopping suddenly or pulling in to the road suddenly",
            haz3: "Oncoming vehicles overtaking on blind rises or corners",
            haz4: "Heavy trucks or plant taking up width of the road",
            haz5: "Riots or unrest",
            haz6: "Gravel/Sandy roads can be slippery – Loss of control",
            haz7: "Tyre blowouts or punctures – Loss of control"
        }
    };
    
    return items[section]?.[itemId] || 'Unknown item';
}

// Show success modal
function showSuccessModal() {
    document.getElementById('successModal').style.display = 'flex';
}

// Close success modal
function closeSuccessModal() {
    document.getElementById('successModal').style.display = 'none';
    // Reset form
    resetForm();
}

// View PDF (placeholder)
function viewPDF() {
    console.log('View PDF clicked');
    showToast('info', 'PDF Viewer', 'PDF viewer functionality would be implemented here');
    closeSuccessModal();
}

// Reset form
function resetForm() {
    // Reset all steps
    currentStep = 1;
    step2Data = {};
    step3Data = {};
    signatureData = null;
    signatureStatus = false;
    
    // Reset Step 1
    document.getElementById('country').value = '';
    document.getElementById('project').value = '';
    document.getElementById('inspector').value = '';
    document.getElementById('dvirCheckbox').checked = true;
    selectedEquipment = null;
    document.getElementById('equipmentDetailsBox').style.display = 'none';
    document.getElementById('equipmentSelectText').textContent = 'Select';
    
    // Reset Step 2
    Object.keys(step2Data).forEach(key => {
        const [section, itemId, field] = key.split('_');
        const buttons = document.querySelectorAll(`[onclick*="setStatus('${section}', '${itemId}'"]`);
        buttons.forEach(btn => btn.classList.remove('selected'));
    });
    
    // Reset Step 3
    document.getElementById('overallCondition').value = '';
    document.querySelectorAll('input[name="equipmentStatus"]').forEach(radio => radio.checked = false);
    document.querySelectorAll('input[name="safeToUse"]').forEach(radio => radio.checked = false);
    document.getElementById('additionalNote').value = '';
    document.getElementById('priorityGroup').style.display = 'none';
    document.getElementById('signaturePlaceholder').style.display = 'block';
    document.getElementById('signatureDisplay').style.display = 'none';
    
    // Show Step 1
    showStep1();
    
    console.log('Form reset successfully');
}

// Initialize signature pad
function initSignaturePad() {
    const canvas = document.getElementById('signaturePad');
    const ctx = canvas.getContext('2d');
    let isDrawing = false;
    
    canvas.addEventListener('mousedown', (e) => {
        isDrawing = true;
        const rect = canvas.getBoundingClientRect();
        ctx.beginPath();
        ctx.moveTo(e.clientX - rect.left, e.clientY - rect.top);
    });
    
    canvas.addEventListener('mousemove', (e) => {
        if (!isDrawing) return;
        const rect = canvas.getBoundingClientRect();
        ctx.lineTo(e.clientX - rect.left, e.clientY - rect.top);
        ctx.stroke();
    });
    
    canvas.addEventListener('mouseup', () => {
        isDrawing = false;
    });
    
    // Touch events for mobile
    canvas.addEventListener('touchstart', (e) => {
        e.preventDefault();
        isDrawing = true;
        const rect = canvas.getBoundingClientRect();
        const touch = e.touches[0];
        ctx.beginPath();
        ctx.moveTo(touch.clientX - rect.left, touch.clientY - rect.top);
    });
    
    canvas.addEventListener('touchmove', (e) => {
        e.preventDefault();
        if (!isDrawing) return;
        const rect = canvas.getBoundingClientRect();
        const touch = e.touches[0];
        ctx.lineTo(touch.clientX - rect.left, touch.clientY - rect.top);
        ctx.stroke();
    });
    
    canvas.addEventListener('touchend', (e) => {
        e.preventDefault();
        isDrawing = false;
    });
    
    // Set up drawing style
    ctx.strokeStyle = '#000';
    ctx.lineWidth = 2;
    ctx.lineCap = 'round';
    ctx.lineJoin = 'round';
}

// Initialize signature pad when page loads
document.addEventListener('DOMContentLoaded', function() {
    initSignaturePad();
});
</script>

<style>
/* CSS Variables */
:root {
    --primary-color: #0055a5;
    --primary-hover: #3b82f6;
    --text-primary: #0f172a;
    --text-secondary: #374151;
    --text-muted: #64748b;
    --bg-primary: #f8fafc;
    --bg-white: #ffffff;
    --bg-hover: #f8fafc;
    --border-color: #e2e8f0;
    --border-light: #d1d5db;
    --shadow-sm: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    --shadow-lg: 0 10px 25px rgba(0, 0, 0, 0.15);
    --radius-sm: 4px;
    --radius-md: 8px;
    --radius-lg: 12px;
    --transition-base: all 0.2s ease;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Import Google Fonts */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* Main Container */
.dvir-create-container {
    min-height: 100vh;
    background: var(--bg-primary);
    font-family: var(--font-family);
}

/* Header Section */
.dvir-header {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
}

.header-content {
    padding: 32px;
}

.page-title h1 {
    color: var(--text-primary);
    font-weight: 700;
    font-size: 32px;
    line-height: 1.2;
    letter-spacing: -0.025em;
    margin: 0 0 8px 0;
}

.page-title p {
    color: var(--text-muted);
    font-weight: 400;
    font-size: 16px;
    line-height: 1.5;
    margin: 0;
}

/* Form Container */
.form-container {
    max-width: 800px;
    margin: 0 auto;
    padding: 0 16px;
}

.form-card {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-md);
    padding: 32px;
    margin-bottom: 24px;
}

/* Request Number Display */
.request-number-display {
    background: #f0f0f0;
    border-radius: var(--radius-sm);
    padding: 16px;
    margin-bottom: 24px;
}

.request-number-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.request-label {
    font-size: 16px;
    color: var(--primary-color);
    font-weight: 500;
}

.request-value {
    font-size: 16px;
    color: var(--primary-color);
    font-weight: 600;
}

/* Form Groups */
.form-group {
    margin-bottom: 24px;
}

.form-label {
    display: block;
    font-size: 16px;
    color: var(--primary-color);
    margin-bottom: 8px;
    font-weight: 500;
}

.required-field::after {
    content: ' *';
    color: #e74c3c;
    font-weight: 600;
}

.form-control {
    width: 100%;
    height: 48px;
    border: 1px solid var(--primary-color);
    border-radius: var(--radius-sm);
    padding: 0 16px;
    font-size: 16px;
    color: var(--text-primary);
    background: var(--bg-white);
    transition: var(--transition-base);
}

.form-control:focus {
    outline: none;
    border-color: var(--primary-hover);
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.form-control:disabled {
    background: #e0e0e0;
    cursor: not-allowed;
}

/* Equipment Selection */
.equipment-selection-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: #f0f0f0;
    padding: 16px;
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
}

.equipment-label {
    font-size: 16px;
    color: var(--primary-color);
    font-weight: 600;
}

.equipment-select-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    background: none;
    border: none;
    color: var(--primary-color);
    font-size: 16px;
    cursor: pointer;
    transition: var(--transition-base);
}

.equipment-select-btn:hover {
    color: var(--primary-hover);
}

/* Equipment Details Box */
.equipment-details-box {
    background: #f0f0f0;
    border-radius: var(--radius-sm);
    padding: 20px;
    margin-bottom: 24px;
}

.equipment-info-row {
    margin-bottom: 16px;
}

.equipment-info-text {
    font-size: 16px;
    color: var(--text-primary);
    line-height: 1.5;
}

/* Meter Reading Section */
.meter-reading-section {
    border-top: 1px solid #ddd;
    padding-top: 16px;
}

.meter-reading-label {
    display: block;
    font-size: 16px;
    color: var(--text-primary);
    margin-bottom: 8px;
    font-weight: 600;
}

.meter-reading-input-row {
    display: flex;
    align-items: center;
    background: var(--bg-white);
    border: 1px solid #ccc;
    border-radius: var(--radius-sm);
    padding: 0 16px;
    margin-bottom: 8px;
}

.meter-reading-input {
    flex: 1;
    border: none;
    outline: none;
    font-size: 16px;
    color: var(--text-primary);
    padding: 12px 0;
}

.meter-unit {
    color: var(--text-muted);
    font-size: 14px;
    margin-left: 8px;
}

.meter-reading-error {
    color: #e74c3c;
    font-size: 12px;
    margin-top: 4px;
}

.current-meter-row {
    display: flex;
    align-items: center;
    margin-top: 8px;
}

.current-meter-label {
    font-size: 13px;
    color: var(--text-muted);
}

.current-meter-value {
    font-size: 13px;
    color: var(--text-primary);
    font-weight: 500;
    margin-left: 4px;
}

/* Inspection Type Section */
.inspection-type-section {
    border-top: 1px solid #e0e0e0;
    margin-top: 16px;
    padding-top: 16px;
}

.checkbox-row {
    display: flex;
    align-items: center;
    padding: 8px 0;
}

.inspection-checkbox {
    width: 20px;
    height: 20px;
    margin-right: 12px;
    accent-color: var(--primary-color);
}

.checkbox-label {
    font-size: 16px;
    color: var(--primary-color);
    font-weight: 500;
    cursor: pointer;
}

/* Action Buttons */
.action-buttons {
    display: flex;
    justify-content: space-between;
    padding: 16px 0;
    background: #f0f0f0;
    border-top: 1px solid #ddd;
    position: sticky;
    bottom: 0;
    left: 0;
    right: 0;
    margin: 0 -16px;
    padding: 16px;
}

.btn-cancel {
    display: flex;
    align-items: center;
    gap: 8px;
    background: none;
    border: none;
    color: var(--primary-color);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    padding: 12px 20px;
    transition: var(--transition-base);
}

.btn-cancel:hover {
    color: var(--primary-hover);
}

.btn-next {
    display: flex;
    align-items: center;
    gap: 8px;
    background: var(--primary-color);
    border: none;
    color: var(--bg-white);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    padding: 12px 20px;
    border-radius: var(--radius-sm);
    transition: var(--transition-base);
}

.btn-next:hover:not(:disabled) {
    background: var(--primary-hover);
}

.btn-next:disabled {
    opacity: 0.6;
    cursor: not-allowed;
}

/* Modal Styles */
.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    justify-content: center;
    align-items: center;
    z-index: 1000;
}

.modal-content {
    background: var(--bg-white);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-lg);
    max-width: 90%;
    max-height: 90%;
    overflow: hidden;
}

.equipment-modal {
    width: 800px;
    max-height: 600px;
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 24px;
    border-bottom: 1px solid var(--border-color);
}

.modal-header-actions {
    display: flex;
    align-items: center;
    gap: 12px;
}

.modal-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.modal-close {
    background: none;
    border: none;
    cursor: pointer;
    padding: 8px;
    border-radius: var(--radius-sm);
    transition: var(--transition-base);
}

.modal-close:hover {
    background: var(--bg-hover);
}

.refresh-equipment-btn {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 48px;
    height: 48px;
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    border-radius: var(--radius-sm);
    cursor: pointer;
    transition: var(--transition-base);
}

.refresh-equipment-btn:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

/* Equipment Search */
.equipment-search-container {
    padding: 16px 24px;
    border-bottom: 1px solid var(--border-color);
}

.search-input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
}

.equipment-search-input {
    width: 100%;
    height: 48px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    padding: 0 16px 0 48px;
    font-size: 16px;
    color: var(--text-primary);
    background: var(--bg-white);
    transition: var(--transition-base);
}

.equipment-search-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.search-icon {
    position: absolute;
    left: 16px;
    color: var(--text-muted);
    pointer-events: none;
}

/* Equipment Tabs */
.equipment-tabs {
    display: flex;
    padding: 0 24px;
    border-bottom: 1px solid var(--border-color);
}

.tab-button {
    flex: 1;
    padding: 16px;
    background: none;
    border: none;
    border-bottom: 2px solid transparent;
    font-size: 16px;
    font-weight: 600;
    color: var(--text-muted);
    cursor: pointer;
    transition: var(--transition-base);
}

.tab-button.active {
    color: var(--primary-color);
    border-bottom-color: var(--primary-color);
}

.tab-button:hover {
    color: var(--primary-color);
}

/* Equipment List */
.equipment-list-container {
    max-height: 300px;
    overflow-y: auto;
    padding: 16px 24px;
}

.equipment-list {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.equipment-item {
    display: flex;
    background: #F2F2F2;
    border: 1px solid #eee;
    border-radius: var(--radius-sm);
    padding: 16px;
    transition: var(--transition-base);
}

.equipment-item:hover {
    box-shadow: var(--shadow-sm);
}

.equipment-info {
    flex: 1;
    cursor: pointer;
}

.equipment-header {
    margin-bottom: 12px;
}

.equipment-name {
    font-size: 18px;
    color: var(--primary-color);
    font-weight: 600;
}

.equipment-details {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.detail-row {
    display: flex;
    justify-content: space-between;
    padding: 4px 0;
    border-bottom: 1px solid #ddd;
}

.detail-row.no-border {
    border-bottom: none;
}

.detail-label {
    font-size: 14px;
    color: var(--text-muted);
    width: 40%;
}

.detail-value {
    font-size: 14px;
    color: var(--text-primary);
    width: 60%;
    text-align: right;
    font-weight: 500;
}

.detail-value.status {
    font-weight: 600;
}

.select-equipment-btn {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 8px 16px;
    border-radius: var(--radius-sm);
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    align-self: flex-end;
    margin-top: auto;
}

.select-equipment-btn:hover {
    background: var(--primary-hover);
}

.select-equipment-btn.selected {
    background: #2ecc71;
    color: white;
}

.select-equipment-btn.selected:hover {
    background: #27ae60;
}

.no-equipment {
    text-align: center;
    color: var(--text-muted);
    font-size: 16px;
    padding: 40px 0;
}

/* Modal Footer */
.modal-footer {
    display: flex;
    justify-content: flex-end;
    padding: 16px 24px;
    border-top: 1px solid var(--border-color);
}

.btn-close {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-sm);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
}

.btn-close:hover {
    background: var(--primary-hover);
}

/* Loading Modal */
.loading-modal-content {
    background: var(--bg-white);
    padding: 40px;
    border-radius: var(--radius-lg);
    text-align: center;
    box-shadow: var(--shadow-lg);
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid var(--border-color);
    border-top: 4px solid var(--primary-color);
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin: 0 auto 20px;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-modal-content h3 {
    font-size: 18px;
    font-weight: 600;
    color: var(--primary-color);
    margin: 0 0 8px 0;
}

.loading-modal-content p {
    font-size: 16px;
    color: var(--text-muted);
    margin: 0;
}

/* Toast Notifications */
.toast-container {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 9999;
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.toast {
    display: flex;
    align-items: center;
    justify-content: space-between;
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    box-shadow: var(--shadow-md);
    padding: 16px;
    min-width: 300px;
    animation: slideIn 0.3s ease;
}

@keyframes slideIn {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

.toast-success {
    border-left: 4px solid #2ecc71;
}

.toast-error {
    border-left: 4px solid #e74c3c;
}

.toast-info {
    border-left: 4px solid #3498db;
}

.toast-content h4 {
    font-size: 14px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0 0 4px 0;
}

.toast-content p {
    font-size: 13px;
    color: var(--text-muted);
    margin: 0;
}

.toast-close {
    background: none;
    border: none;
    font-size: 18px;
    color: var(--text-muted);
    cursor: pointer;
    padding: 4px;
    margin-left: 12px;
}

.toast-close:hover {
    color: var(--text-primary);
}

/* Responsive Design */
@media (max-width: 768px) {
    .form-container {
        padding: 0 8px;
    }
    
    .form-card {
        padding: 20px;
    }
    
    .header-content {
        padding: 20px;
    }
    
    .page-title h1 {
        font-size: 24px;
    }
    
    .equipment-modal {
        width: 95%;
        max-height: 80%;
    }
    
    .equipment-item {
        flex-direction: column;
        gap: 12px;
    }
    
    .select-equipment-btn {
        align-self: stretch;
        text-align: center;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 12px;
    }
    
    .btn-cancel,
    .btn-next {
        justify-content: center;
    }
}

@media (max-width: 480px) {
    .form-card {
        padding: 16px;
    }
    
    .header-content {
        padding: 16px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .equipment-tabs {
        flex-direction: column;
    }
    
    .tab-button {
        border-bottom: 1px solid var(--border-color);
        border-right: none;
    }
    
    .tab-button.active {
        border-bottom-color: var(--primary-color);
        background: var(--bg-hover);
    }
}

/* ==================== STEP CONTAINER STYLES ==================== */
.step-container {
    min-height: 100vh;
    background: var(--bg-primary);
    font-family: var(--font-family);
}

.step-header {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
    padding: 32px;
}

.step-header h1 {
    color: var(--text-primary);
    font-weight: 700;
    font-size: 32px;
    line-height: 1.2;
    letter-spacing: -0.025em;
    margin: 0 0 8px 0;
}

.step-header p {
    color: var(--text-muted);
    font-weight: 400;
    font-size: 16px;
    line-height: 1.5;
    margin: 0;
}

.step-content {
    max-width: 800px;
    margin: 0 auto;
    padding: 0 16px;
}

/* ==================== STEP 2 STYLES ==================== */
.instruction-text {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    padding: 16px;
    margin-bottom: 24px;
    text-align: center;
}

.instruction-text p {
    color: var(--primary-color);
    font-weight: 500;
    margin: 0;
}

.checklist-container {
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.checklist-card {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    overflow: hidden;
    box-shadow: var(--shadow-sm);
}

.card-header {
    background: #f0f0f0;
    padding: 16px;
    cursor: pointer;
    transition: var(--transition-base);
    border-bottom: 1px solid var(--border-color);
}

.card-header:hover {
    background: #e9e9e9;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 12px;
}

.expand-icon {
    font-size: 18px;
    color: var(--primary-color);
    transition: var(--transition-base);
}

.section-title {
    font-size: 18px;
    font-weight: 600;
    color: var(--primary-color);
    margin: 0;
}

.card-content {
    padding: 16px;
}

.checklist-items {
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.checklist-item {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    padding: 16px;
    background: #f8f9fa;
    border-radius: var(--radius-sm);
    border: 1px solid #e9ecef;
}

.item-content {
    flex: 1;
    margin-right: 16px;
}

.item-text {
    font-size: 14px;
    color: var(--text-primary);
    margin: 0 0 12px 0;
    line-height: 1.4;
}

.status-buttons {
    display: flex;
    gap: 8px;
}

.status-btn {
    padding: 6px 12px;
    border: 1px solid var(--border-color);
    background: var(--bg-white);
    color: var(--text-primary);
    border-radius: var(--radius-sm);
    font-size: 12px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.status-btn:hover {
    background: var(--bg-hover);
}

.status-btn.selected {
    background: var(--primary-color);
    color: var(--bg-white);
    border-color: var(--primary-color);
}

.item-actions {
    display: flex;
    gap: 8px;
    align-items: center;
}

.action-btn {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 8px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    font-size: 14px;
    transition: var(--transition-base);
    opacity: 0.5;
    cursor: not-allowed;
}

.action-btn:hover:not(:disabled) {
    background: var(--primary-hover);
}

.action-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

.note-input-container {
    margin-top: 12px;
    padding: 8px;
    background: #f8f9fa;
    border-radius: var(--radius-sm);
    border: 1px solid #e9ecef;
}

.note-input {
    width: 100%;
    padding: 8px 12px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    font-size: 14px;
    color: var(--text-primary);
    background: var(--bg-white);
    transition: var(--transition-base);
}

.note-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 2px rgba(0, 85, 165, 0.1);
}

.note-input::placeholder {
    color: var(--text-muted);
    font-style: italic;
}

/* Image Preview Styles */
.image-preview-container {
    margin-top: 12px;
    padding: 12px;
    background: #f8f9fa;
    border-radius: var(--radius-sm);
    border: 1px solid #e9ecef;
}

.image-preview-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 8px;
}

.image-preview-label {
    font-size: 12px;
    font-weight: 600;
    color: var(--primary-color);
}

.remove-image-btn {
    background: #e74c3c;
    color: white;
    border: none;
    border-radius: 50%;
    width: 20px;
    height: 20px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    font-size: 14px;
    font-weight: bold;
    transition: var(--transition-base);
}

.remove-image-btn:hover {
    background: #c0392b;
    transform: scale(1.1);
}

.image-preview {
    max-width: 100%;
    max-height: 200px;
    border-radius: var(--radius-sm);
    border: 1px solid #ddd;
    object-fit: contain;
    display: block;
    margin: 0 auto;
}

/* ==================== STEP 3 STYLES ==================== */
.summary-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: #f0f0f0;
    padding: 16px;
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
}

.summary-label {
    font-size: 16px;
    font-weight: 500;
    color: var(--primary-color);
}

.summary-value {
    font-size: 16px;
    font-weight: 600;
}

.summary-value.faulty {
    color: #e74c3c;
}

.select-container {
    position: relative;
}

.radio-group {
    display: flex;
    gap: 16px;
    margin-top: 8px;
}

.radio-option {
    display: flex;
    align-items: center;
    gap: 8px;
    cursor: pointer;
}

.radio-option input[type="radio"] {
    margin: 0;
    accent-color: var(--primary-color);
}

.radio-label {
    font-size: 14px;
    color: var(--text-primary);
    cursor: pointer;
}

.priority-container {
    display: flex;
    gap: 16px;
    margin-top: 8px;
    flex-wrap: wrap;
}

.priority-option {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 8px;
    padding: 12px;
    border: 2px solid var(--border-color);
    border-radius: var(--radius-sm);
    cursor: pointer;
    transition: var(--transition-base);
    min-width: 80px;
}

.priority-option:hover {
    border-color: var(--primary-color);
}

.priority-option.selected {
    border-color: var(--primary-color);
    background: var(--bg-hover);
}

.priority-icon {
    font-size: 24px;
}

.input-with-icon {
    position: relative;
    display: flex;
    align-items: flex-end;
}

.textarea {
    resize: vertical;
    min-height: 100px;
}

.voice-btn {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 12px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    margin-left: 8px;
    transition: var(--transition-base);
}

.voice-btn:hover {
    background: var(--primary-hover);
}

.signature-container {
    margin-top: 16px;
}

.signature-box {
    border: 2px dashed var(--border-color);
    border-radius: var(--radius-sm);
    padding: 20px;
    text-align: center;
    cursor: pointer;
    transition: var(--transition-base);
    min-height: 120px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.signature-box:hover {
    border-color: var(--primary-color);
    background: var(--bg-hover);
}

.signature-placeholder {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 8px;
    color: var(--text-muted);
}

.signature-display {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 12px;
}

.signature-display img {
    max-width: 300px;
    max-height: 150px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
}

.change-signature-btn {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 8px 16px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    font-size: 12px;
}

.acknowledgment-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    padding: 20px;
    margin-top: 24px;
}

.acknowledgment-header {
    display: flex;
    align-items: center;
    gap: 8px;
    margin-bottom: 12px;
}

.acknowledgment-header h3 {
    font-size: 16px;
    font-weight: 600;
    color: var(--primary-color);
    margin: 0;
}

.acknowledgment-section p {
    font-size: 13px;
    color: var(--text-primary);
    line-height: 1.5;
    margin: 0;
}

/* ==================== NAVIGATION BUTTONS ==================== */
.nav-buttons {
    display: flex;
    justify-content: space-between;
    padding: 16px 0;
    background: #f0f0f0;
    border-top: 1px solid #ddd;
    position: sticky;
    bottom: 0;
    left: 0;
    right: 0;
    margin: 0 -16px;
    padding: 16px;
}

.nav-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border-radius: var(--radius-sm);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    border: none;
}

.back-btn {
    background: none;
    color: var(--primary-color);
}

.back-btn:hover {
    color: var(--primary-hover);
}

.next-btn {
    background: var(--primary-color);
    color: var(--bg-white);
}

.next-btn:hover:not(:disabled) {
    background: var(--primary-hover);
}

.next-btn:disabled {
    opacity: 0.6;
    cursor: not-allowed;
}

/* ==================== SIGNATURE MODAL STYLES ==================== */
.signature-modal {
    width: 90%;
    max-width: 700px;
    max-height: 80vh;
}

.signature-pad-container {
    padding: 20px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    margin: 16px 0;
    background: var(--bg-white);
}

#signaturePad {
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    cursor: crosshair;
}

.signature-actions {
    display: flex;
    justify-content: space-between;
    gap: 12px;
    margin-top: 16px;
}

.btn-clear {
    background: #6c757d;
    color: var(--bg-white);
    border: none;
    padding: 10px 20px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    font-weight: 500;
}

.btn-save {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 10px 20px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    font-weight: 500;
}

/* ==================== SUCCESS MODAL STYLES ==================== */
.success-modal {
    width: 90%;
    max-width: 400px;
}

.success-content {
    text-align: center;
    padding: 20px;
}

.success-icon {
    font-size: 48px;
    margin-bottom: 16px;
}

.success-content h3 {
    font-size: 20px;
    font-weight: 600;
    color: var(--primary-color);
    margin: 0 0 8px 0;
}

.success-content p {
    font-size: 14px;
    color: var(--text-muted);
    margin: 0 0 20px 0;
}

.success-actions {
    display: flex;
    gap: 12px;
    justify-content: center;
}

.btn-primary {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 10px 20px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    font-weight: 500;
}

.btn-secondary {
    background: #6c757d;
    color: var(--bg-white);
    border: none;
    padding: 10px 20px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    font-weight: 500;
}

/* ==================== ERROR MESSAGE STYLES ==================== */
.error-message {
    color: #e74c3c;
    font-size: 12px;
    margin-top: 4px;
    display: none;
}

/* ==================== RESPONSIVE DESIGN ==================== */
@media (max-width: 768px) {
    .step-content {
        padding: 0 8px;
    }
    
    .step-header {
        padding: 20px;
    }
    
    .step-header h1 {
        font-size: 24px;
    }
    
    .checklist-item {
        flex-direction: column;
        gap: 12px;
    }
    
    .item-content {
        margin-right: 0;
    }
    
    .item-actions {
        align-self: stretch;
        justify-content: center;
    }
    
    .nav-buttons {
        flex-direction: column;
        gap: 12px;
    }
    
    .nav-btn {
        justify-content: center;
    }
    
    .priority-container {
        justify-content: center;
    }
    
    .radio-group {
        flex-direction: column;
        gap: 8px;
    }
}

@media (max-width: 480px) {
    .step-header {
        padding: 16px;
    }
    
    .step-header h1 {
        font-size: 20px;
    }
    
    .signature-modal {
        width: 95%;
    }
    
    .signature-pad-container {
        padding: 10px;
    }
    
    #signaturePad {
        width: 100%;
        height: 150px;
    }
}
</style>
@endsection
