import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  StatusBar,
  Modal,
  ScrollView,
  Platform,
  Alert,
  Dimensions,
  Animated,
  KeyboardAvoidingView,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation, useRoute } from '@react-navigation/native';
import DropDownPicker from 'react-native-dropdown-picker';
import Toast from 'react-native-toast-message';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import NetInfo from '@react-native-community/netinfo';
import DatePicker from 'react-native-date-picker';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.25;

// Custom Input Field Component
const CustomInputField = ({
  label,
  value,
  onChangeText,
  isRequired = false,
  showError = false,
  errorMessage = '',
  icon,
  onIconPress,
  editable = true,
  onPress,
  inputRef,
  containerRef,
  ...props
}) => {
  return (
    <View ref={containerRef} style={inputStyles.container}>
      <Text style={inputStyles.label}>
        {label}
        {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
      </Text>
      <TouchableOpacity
        style={[inputStyles.inputContainer, showError ? inputStyles.inputError : null]}
        onPress={onPress}
        disabled={!onPress}
      >
        <TextInput
          ref={inputRef}
          style={[inputStyles.input, { flex: 1 }]}
          value={value}
          onChangeText={onChangeText}
          placeholderTextColor="#888"
          editable={editable}
          pointerEvents={onPress ? 'none' : 'auto'}
          autoCapitalize="none"
          autoCorrect={false}
          {...props}
        />
        {icon && (
          <TouchableOpacity style={inputStyles.iconButton} onPress={onIconPress}>
            {icon}
          </TouchableOpacity>
        )}
      </TouchableOpacity>
      {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
    </View>
  );
};

const Add_equipment_update = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { equipment } = route.params || {}; // Get equipment data if in update mode

  const [mainCategoryModalVisible, setMainCategoryModalVisible] = useState(false);
  const [loadingModalVisible, setLoadingModalVisible] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });
  const slideAnim = useRef(new Animated.Value(300)).current;
  const scrollViewRef = useRef(null);
  const lastServiceMeterContainerRef = useRef(null);
  const equipmentNameContainerRef = useRef(null);

  const dynamicField1Ref = useRef(null);
  const dynamicField2Ref = useRef(null);
  const dynamicField3Ref = useRef(null);
  const equipmentNumberRef = useRef(null);
  const equipmentNameRef = useRef(null);
  const meterReadingRef = useRef(null);
  const modelRef = useRef(null);
  const lastServiceMeterRef = useRef(null);
  const lastServiceDateRef = useRef(null);
  const vinRef = useRef(null);
  const fleetValueRef = useRef(null);
  const purchaseDateRef = useRef(null);
  const warrantyExpDateRef = useRef(null);
  const operatorNameRef = useRef(null);

  const [countryOpen, setCountryOpen] = useState(false);
  const [selectedCountry, setSelectedCountry] = useState(null);
  const [isCountrySelectable, setIsCountrySelectable] = useState(false);
  const [countryItems, setCountryItems] = useState([
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ]);

  const [mainCategoryOpen, setMainCategoryOpen] = useState(false);
  const [selectedMainCategory, setSelectedMainCategory] = useState(null);
  const [mainCategoryItems, setMainCategoryItems] = useState([
    { label: 'Light Vehicles', value: 'Light Vehicles' },
    { label: 'Heavy Vehicles', value: 'Heavy Vehicles' },
    { label: 'Rigs', value: 'Rigs' },
    { label: 'Heavy Plant Equipment', value: 'Heavy Plant Equipment' },
    { label: 'Auxiliary Equipment', value: 'Auxiliary Equipment' },
    { label: 'Trailers', value: 'Trailers' },
    { label: 'Lathes', value: 'Lathes' },
  ]);

  const [dynamicField1, setDynamicField1] = useState('');
  const [dynamicField2, setDynamicField2] = useState('');
  const [dynamicField3, setDynamicField3] = useState('');

  const [rigTypeOpen, setRigTypeOpen] = useState(false);
  const [rigType, setRigType] = useState(null);
  const [rigTypeItems, setRigTypeItems] = useState([
    { label: 'Track Mounted', value: 'Track Mounted' },
    { label: 'Trailer Mounted', value: 'Trailer Mounted' },
    { label: 'Truck Mounted', value: 'Truck Mounted' },
  ]);

  const [meterOpen, setMeterOpen] = useState(false);
  const [meterUnit, setMeterUnit] = useState('Miles');
  const [meterItems, setMeterItems] = useState([
    { label: 'Miles', value: 'Miles' },
    { label: 'KM', value: 'Kilometers' },
    { label: 'Hours', value: 'Hours' },
  ]);

  const [priorityOpen, setPriorityOpen] = useState(false);
  const [maintenancePriority, setMaintenancePriority] = useState('1');
  const [priorityItems, setPriorityItems] = useState([
    { label: '1', value: '1' },
    { label: '2', value: '2' },
    { label: '3', value: '3' },
  ]);

  const [ownershipOpen, setOwnershipOpen] = useState(false);
  const [ownershipMode, setOwnershipMode] = useState('Owned Asset');
  const [ownershipItems, setOwnershipItems] = useState([
    { label: 'Owned Asset', value: 'Owned Asset' },
    { label: 'Leased', value: 'Leased' },
    { label: 'Rented', value: 'Rented' },
  ]);

  const [optionalFieldsVisible, setOptionalFieldsVisible] = useState(false);
  const [errors, setErrors] = useState({});
  const [formData, setFormData] = useState({
    equipmentNumber: '',
    equipmentName: '',
    meterReading: '',
    model: '',
    lastServiceMeter: '',
    lastServiceDate: '',
    vin: '',
    fleetValue: '',
    purchaseDate: '',
    warrantyExpDate: '',
    operatorName: '',
  });

  // Date picker states
  const [lastServiceDate, setLastServiceDate] = useState(null);
  const [purchaseDate, setPurchaseDate] = useState(null);
  const [warrantyExpDate, setWarrantyExpDate] = useState(null);
  const [tempDate, setTempDate] = useState(new Date());
  const [datePickerVisible, setDatePickerVisible] = useState(false);
  const [currentDateField, setCurrentDateField] = useState('');

  const uniqueEquipmentNumber = equipment ? equipment.id : `EQ-${Math.floor(Date.now() / 1000)}`;
  const isUpdateMode = !!equipment;

  // Date parsing function
  const parseDate = (dateStr) => {
    if (!dateStr) return null;
    const parts = dateStr.split('-');
    if (parts.length !== 3) return null;
    const day = parseInt(parts[0], 10);
    const month = ['jan', 'feb', 'mar', 'apr', 'may', 'jun', 'jul', 'aug', 'sep', 'oct', 'nov', 'dec'].indexOf(parts[1].toLowerCase());
    const year = parseInt(parts[2], 10);
    if (isNaN(day) || month === -1 || isNaN(year)) return null;
    return new Date(year, month, day);
  };

  // Date formatting function
  const formatDate = (date) => {
    if (!date || isNaN(date.getTime())) return 'Select Date';
    return date.toLocaleDateString('en-GB', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
    }).replace(/ /g, '-');
  };

  // Open date picker for specific field
  const openDatePicker = (field) => {
    setCurrentDateField(field);
    let initialDate;
    switch (field) {
      case 'lastServiceDate':
        initialDate = lastServiceDate || new Date();
        break;
      case 'purchaseDate':
        initialDate = purchaseDate || new Date();
        break;
      case 'warrantyExpDate':
        initialDate = warrantyExpDate || new Date();
        break;
      default:
        initialDate = new Date();
    }
    setTempDate(initialDate);
    setDatePickerVisible(true);
  };

  // Handle date confirmation
  const handleDateConfirm = () => {
    switch (currentDateField) {
      case 'lastServiceDate':
        setLastServiceDate(tempDate);
        setFormData({ ...formData, lastServiceDate: formatDate(tempDate) });
        break;
      case 'purchaseDate':
        setPurchaseDate(tempDate);
        setFormData({ ...formData, purchaseDate: formatDate(tempDate) });
        break;
      case 'warrantyExpDate':
        setWarrantyExpDate(tempDate);
        setFormData({ ...formData, warrantyExpDate: formatDate(tempDate) });
        break;
    }
    setDatePickerVisible(false);
  };

  // Handle date cancellation
  const handleDateCancel = () => {
    setDatePickerVisible(false);
  };

  const getFieldMappings = (category) => {
    switch (category) {
      case 'Light Vehicles':
      case 'Heavy Vehicles':
        return {
          field1Key: 'vehicleNumber',
          field2Key: 'registrationNumber',
          field3Key: 'vinNumber',
          field1Label: 'Vehicle Number',
          field2Label: 'Registration Number',
          field3Label: 'VIN Number',
        };
      case 'Rigs':
        return {
          field1Key: 'rigType',
          field2Key: 'chassisNumber',
          field3Key: 'serialNumber',
          field1Label: 'Rig Type',
          field2Label: 'Chassis Number',
          field3Label: 'Serial Number',
        };
      case 'Heavy Plant Equipment':
        return {
          field1Key: 'equipmentNumberDynamic',
          field2Key: 'equipmentNameDynamic',
          field3Key: 'vinNumber',
          field1Label: 'Equipment Number',
          field2Label: 'Equipment Name',
          field3Label: 'VIN Number',
        };
      case 'Auxiliary Equipment':
        return {
          field1Key: 'type',
          field2Key: 'modelDynamic',
          field3Key: 'serialNumber',
          field1Label: 'Type',
          field2Label: 'Model',
          field3Label: 'Serial Number',
        };
      case 'Trailers':
        return {
          field1Key: 'type',
          field2Key: 'chassisNumber',
          field3Key: 'chassisNumberDuplicate',
          field1Label: 'Type',
          field2Label: 'Chassis Number',
          field3Label: 'Chassis Number',
        };
      case 'Lathes':
        return {
          field1Key: 'type',
          field2Key: 'modelDynamic',
          field3Key: 'serialNumber',
          field1Label: 'Type',
          field2Label: 'Model',
          field3Label: 'Serial Number',
        };
      default:
        return {
          field1Key: 'field1',
          field2Key: 'field2',
          field3Key: 'field3',
          field1Label: 'Field 1',
          field2Label: 'Field 2',
          field3Label: 'Field 3',
        };
    }
  };

  // Pre-populate fields if in update mode
  useEffect(() => {
    const initializeData = async () => {
      try {
        const storedCountry = await AsyncStorage.getItem('selectedCountry');
        if (storedCountry && ['Zambia', 'DRC', 'Both Countries'].includes(storedCountry)) {
          setSelectedCountry(storedCountry === 'Both Countries' ? 'Zambia' : storedCountry);
          setIsCountrySelectable(storedCountry === 'Both Countries');
        } else {
          setIsCountrySelectable(true);
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Please select a country.',
            position: 'bottom',
          });
        }

        if (equipment) {
          setSelectedCountry(equipment.country || null);
          setSelectedMainCategory(equipment.mainCategory || null);
          const mappings = getFieldMappings(equipment.mainCategory);
          if (equipment.mainCategory === 'Rigs') {
            setRigType(equipment[mappings.field1Key] || null);
          } else {
            setDynamicField1(equipment[mappings.field1Key] || '');
          }
          setDynamicField2(equipment[mappings.field2Key] || '');
          setDynamicField3(equipment[mappings.field3Key] || '');
          setMeterUnit(equipment.meterUnit || 'Miles');
          setMaintenancePriority(equipment.maintenancePriority || '1');
          setOwnershipMode(equipment.ownershipMode || 'Owned Asset');
          setFormData({
            equipmentNumber: equipment.equipmentNumber || '',
            equipmentName: equipment.equipmentName || '',
            meterReading: equipment.meterReading || '',
            model: equipment.model || '',
            lastServiceMeter: equipment.lastServiceMeter || '',
            lastServiceDate: equipment.lastServiceDate || '',
            vin: equipment.vin || '',
            fleetValue: equipment.fleetValue || '',
            purchaseDate: equipment.purchaseDate || '',
            warrantyExpDate: equipment.warrantyExpDate || '',
            operatorName: equipment.operatorName || '',
          });
          setLastServiceDate(parseDate(equipment.lastServiceDate));
          setPurchaseDate(parseDate(equipment.purchaseDate));
          setWarrantyExpDate(parseDate(equipment.warrantyExpDate));
          setOptionalFieldsVisible(true);
          Animated.timing(slideAnim, {
            toValue: 0,
            duration: 0,
            useNativeDriver: true,
          }).start();
        }
      } catch (error) {
        console.error('Error initializing data:', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to initialize data.',
          position: 'bottom',
        });
      }
    };
    initializeData();
  }, [equipment]);

  useEffect(() => {
    if (selectedMainCategory && !isUpdateMode) {
      setDynamicField1('');
      setDynamicField2('');
      setDynamicField3('');
      if (selectedMainCategory === 'Rigs') {
        setRigType(null);
      }
    }
  }, [selectedMainCategory, isUpdateMode]);

  const checkNetworkLoader = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const showLoadingModal = () => {
    setLoadingModalVisible(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();
  };

  const hideLoadingModal = () => {
    setLoadingModalVisible(false);
    rotateAnim.setValue(0);
  };

  const handleInputChange = (name, value) => {
    setFormData({ ...formData, [name]: value });
    setErrors({ ...errors, [name]: '' });
  };

  const showEquipmentInfoAlert = () => {
    Alert.alert(
      'Information',
      'E.g., Plate Number, Serial Number, any Company Unique number, etc.',
      [{ text: 'OK', style: 'cancel' }],
      { cancelable: true }
    );
  };

  const handleSave = async () => {
    const requiredFields = {
      country: 'Country',
      mainCategory: 'Main Category',
      dynamicField1: 'Field 1',
      dynamicField2: 'Field 2',
    };

    const nonMandatoryCategories = ['Light Vehicles', 'Heavy Vehicles', 'Rigs'];

    if (!nonMandatoryCategories.includes(selectedMainCategory)) {
      requiredFields.equipmentNumber = 'Equipment Number';
      requiredFields.equipmentName = 'Equipment Name';
    }

    const newErrors = {};
    if (!selectedCountry) {
      newErrors.country = isCountrySelectable
        ? 'Please select a country'
        : 'Please select a country in Home screen';
    }
    if (!selectedMainCategory) {
      newErrors.mainCategory = 'Please select a main category';
    }
    if (!dynamicField1 && selectedMainCategory !== 'Rigs') {
      newErrors.dynamicField1 = `Please enter ${getFieldMappings(selectedMainCategory).field1Label}`;
    }
    if (!rigType && selectedMainCategory === 'Rigs') {
      newErrors.dynamicField1 = `Please select ${getFieldMappings(selectedMainCategory).field1Label}`;
    }
    if (!dynamicField2) {
      newErrors.dynamicField2 = `Please enter ${getFieldMappings(selectedMainCategory).field2Label}`;
    }

    Object.entries(requiredFields).forEach(([key, label]) => {
      if (
        key !== 'country' &&
        key !== 'mainCategory' &&
        key !== 'dynamicField1' &&
        key !== 'dynamicField2' &&
        !formData[key]
      ) {
        newErrors[key] = `Please fill "${label}"`;
      }
    });

    if (Object.keys(newErrors).length > 0) {
      setErrors(newErrors);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Please fill all required fields.',
        position: 'bottom',
      });
      return;
    }

    const fieldMappings = getFieldMappings(selectedMainCategory);
    const equipmentDataToSave = {
      ...formData,
      country: selectedCountry,
      meterUnit,
      id: uniqueEquipmentNumber,
      maintenancePriority,
      ownershipMode,
      mainCategory: selectedMainCategory,
      [fieldMappings.field1Key]: selectedMainCategory === 'Rigs' ? rigType : dynamicField1,
      [fieldMappings.field2Key]: dynamicField2,
      [fieldMappings.field3Key]: dynamicField3 || 'N/A',
      createdAt: equipment?.createdAt || new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      status: equipment?.status || 'Available',
      equipmentCategory: selectedMainCategory,
      firebaseKey: uniqueEquipmentNumber,
    };

    showLoadingModal();

    try {
      const user = auth().currentUser;
      if (!user) {
        hideLoadingModal();
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'User not authenticated. Please log in.',
          position: 'bottom',
        });
        navigation.replace('AdminLogin');
        return;
      }

      const userId = user.uid;
      const isOnline = await checkNetworkLoader();

      if (isOnline) {
        const userEquipmentRef = database().ref(`/equipment/${userId}/${uniqueEquipmentNumber}`);
        await userEquipmentRef.set(equipmentDataToSave);

        const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
        const globalSnapshot = await globalRef.once('value');
        let globalData = globalSnapshot.val() || [];
        globalData = Array.isArray(globalData) ? globalData : [];

        if (isUpdateMode) {
          const index = globalData.findIndex((item) => item.id === equipment.id);
          if (index !== -1) {
            globalData[index] = equipmentDataToSave;
          }
        } else {
          globalData.push(equipmentDataToSave);
        }
        await globalRef.set(globalData);

        const existing = await AsyncStorage.getItem('equipment');
        let equipmentList = existing ? JSON.parse(existing) : [];
        if (isUpdateMode) {
          const index = equipmentList.findIndex((item) => item.id === equipment.id);
          if (index !== -1) {
            equipmentList[index] = equipmentDataToSave;
          }
        } else {
          equipmentList.push(equipmentDataToSave);
        }
        await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));

        hideLoadingModal();
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: `Equipment ${isUpdateMode ? 'updated' : 'saved'} successfully to database.`,
          position: 'bottom',
        });
        navigation.goBack();
      } else {
        const existing = await AsyncStorage.getItem('equipment');
        let equipmentList = existing ? JSON.parse(existing) : [];
        if (isUpdateMode) {
          const index = equipmentList.findIndex((item) => item.id === equipment.id);
          if (index !== -1) {
            equipmentList[index] = { ...equipmentDataToSave, pendingSync: true };
          }
        } else {
          equipmentList.push({ ...equipmentDataToSave, pendingSync: true });
        }
        await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));

        hideLoadingModal();
        Toast.show({
          type: 'info',
          text1: 'Offline',
          text2: `Equipment ${isUpdateMode ? 'updated' : 'saved'} locally. Will sync when online.`,
          position: 'bottom',
        });
        navigation.goBack();
      }
    } catch (error) {
      console.error(`Error ${isUpdateMode ? 'updating' : 'saving'} equipment:`, error);
      let errorMessage = `Failed to ${isUpdateMode ? 'update' : 'save'} equipment to database.`;
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Check Firebase security rules.';
      } else if (error.code === 'NETWORK_ERROR') {
        const existing = await AsyncStorage.getItem('equipment');
        let equipmentList = existing ? JSON.parse(existing) : [];
        if (isUpdateMode) {
          const index = equipmentList.findIndex((item) => item.id === equipment.id);
          if (index !== -1) {
            equipmentList[index] = { ...equipmentDataToSave, pendingSync: true };
          }
        } else {
          equipmentList.push({ ...equipmentDataToSave, pendingSync: true });
        }
        await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));
        errorMessage = `Offline: Equipment ${isUpdateMode ? 'updated' : 'saved'} locally, will sync when online.`;
        Toast.show({
          type: 'info',
          text1: 'Offline',
          text2: errorMessage,
          position: 'bottom',
        });
        hideLoadingModal();
        navigation.goBack();
        return;
      }
      hideLoadingModal();
      setErrors({ general: errorMessage });
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
        position: 'bottom',
      });
    }
  };

  useEffect(() => {
    const syncPendingEquipment = async () => {
      const isOnline = await checkNetworkLoader();
      if (!isOnline) return;

      const user = auth().currentUser;
      if (!user) return;

      const userId = user.uid;
      try {
        const existing = await AsyncStorage.getItem('equipment');
        const equipmentList = existing ? JSON.parse(existing) : [];
        const pending = equipmentList.filter((item) => item.pendingSync);

        if (pending.length > 0) {
          showLoadingModal();
          const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
          const globalSnapshot = await globalRef.once('value');
          let globalData = globalSnapshot.val() || [];
          globalData = Array.isArray(globalData) ? globalData : [];

          for (const item of pending) {
            const userEquipmentRef = database().ref(`/equipment/${userId}/${item.id}`);
            await userEquipmentRef.set({ ...item, pendingSync: false });

            const existingGlobalIndex = globalData.findIndex((g) => g.id === item.id);
            if (existingGlobalIndex === -1) {
              globalData.push({ ...item, pendingSync: false });
            } else {
              globalData[existingGlobalIndex] = { ...item, pendingSync: false };
            }
          }

          await globalRef.set(globalData);

          const updatedEquipment = equipmentList.map((item) =>
            item.pendingSync ? { ...item, pendingSync: false } : item
          );
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipment));

          hideLoadingModal();
          Toast.show({
            type: 'success',
            text1: 'Success',
            text2: 'Pending equipment synced to database.',
            position: 'bottom',
          });
        }
      } catch (error) {
        console.error('Error syncing pending equipment:', error);
        hideLoadingModal();
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to sync pending equipment.',
          position: 'bottom',
        });
      }
    };

    const unsubscribe = NetInfo.addEventListener((state) => {
      if (state.isConnected) {
        syncPendingEquipment();
      }
    });

    return () => unsubscribe();
  }, []);

  const handleRefresh = () => {
    setFormData({
      equipmentNumber: '',
      equipmentName: '',
      meterReading: '',
      model: '',
      lastServiceMeter: '',
      lastServiceDate: '',
      vin: '',
      fleetValue: '',
      purchaseDate: '',
      warrantyExpDate: '',
      operatorName: '',
    });
    setLastServiceDate(null);
    setPurchaseDate(null);
    setWarrantyExpDate(null);
    setMeterUnit('Miles');
    setMaintenancePriority('1');
    setOwnershipMode('Owned Asset');
    setSelectedMainCategory(null);
    setDynamicField1('');
    setDynamicField2('');
    setDynamicField3('');
    setRigType(null);
    setErrors({});
    setOptionalFieldsVisible(false);
    slideAnim.setValue(300);
    if (isCountrySelectable) {
      setSelectedCountry(null);
    }
  };

  const selectMainCategory = (category) => {
    setSelectedMainCategory(category);
    setMainCategoryModalVisible(false);
    setErrors({ ...errors, mainCategory: '' });
  };

  const handleMainCategoryPlusClick = () => {
    if (!selectedCountry) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: isCountrySelectable
          ? 'Please select a country'
          : 'Please select a country in Home screen',
        position: 'bottom',
      });
    } else {
      setMainCategoryModalVisible(true);
    }
  };

  const toggleOptionalFields = () => {
    const willBeVisible = !optionalFieldsVisible;
    setOptionalFieldsVisible(willBeVisible);

    if (lastServiceMeterRef.current) {
      lastServiceMeterRef.current.blur();
    }
    Animated.timing(slideAnim, {
      toValue: willBeVisible ? 0 : 300,
      duration: 400,
      useNativeDriver: true,
    }).start();

    setTimeout(() => {
      if (!scrollViewRef.current) return;

      if (willBeVisible) {
        if (lastServiceMeterContainerRef.current) {
          lastServiceMeterContainerRef.current.measureLayout(
            scrollViewRef.current.getScrollResponder(),
            (x, y) => scrollViewRef.current.scrollTo({ y, animated: true }),
            (error) => console.error('measureLayout error:', error)
          );
        }
      } else {
        if (equipmentNameContainerRef.current) {
          equipmentNameContainerRef.current.measureLayout(
            scrollViewRef.current.getScrollResponder(),
            (x, y) => scrollViewRef.current.scrollTo({ y, animated: true }),
            (error) => console.error('measureLayout error:', error)
          );
        }
      }
    }, 400);
  };

  const { field1Label, field2Label, field3Label } = selectedMainCategory
    ? getFieldMappings(selectedMainCategory)
    : { field1Label: 'Field 1', field2Label: 'Field 2', field3Label: 'Field 3' };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>{isUpdateMode ? 'Update Equipment' : 'Add Equipment'}</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity onPress={handleRefresh} style={styles.iconButton}>
              <MaterialIcons name="refresh" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity onPress={handleSave}>
              <View style={styles.savebtn01}>
                <Text style={styles.saveText}>{isUpdateMode ? 'Update' : 'Save'}</Text>
              </View>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        style={styles.keyboardAvoidingView}
      >
        <ScrollView
          ref={scrollViewRef}
          style={styles.mainContent}
          keyboardShouldPersistTaps="always"
          showsVerticalScrollIndicator={false}
        >
          <View style={styles.equipmentIdRow}>
            <Text style={styles.equipmentIdLabel}>Equipment#</Text>
            <Text style={styles.equipmentIdValue}>{uniqueEquipmentNumber}</Text>
          </View>

          {errors.general && (
            <Text style={styles.generalErrorText}>{errors.general}</Text>
          )}

          <View style={styles.inputContainer}>
            <View style={inputStyles.container}>
              <Text style={inputStyles.label}>
                Country <Text style={inputStyles.requiredError}>*</Text>
              </Text>
              <DropDownPicker
                open={countryOpen}
                value={selectedCountry}
                items={countryItems}
                setOpen={setCountryOpen}
                setValue={setSelectedCountry}
                setItems={setCountryItems}
                placeholder="Select Country"
                style={[styles.dropdown, !isCountrySelectable && styles.disabledDropdown]}
                dropDownContainerStyle={styles.dropdownContainer}
                textStyle={styles.dropdownText}
                placeholderStyle={styles.placeholderStyle}
                disabled={!isCountrySelectable}
                disabledStyle={{ opacity: 0.6 }}
                zIndex={5000}
                dropDownDirection="AUTO"
                onChangeValue={() => setErrors({ ...errors, country: '' })}
              />
              {errors.country && (
                <Text style={inputStyles.errorText}>{errors.country}</Text>
              )}
            </View>

            <CustomInputField
              label="Main Category"
              value={selectedMainCategory || ''}
              onChangeText={() => {}}
              isRequired
              showError={!!errors.mainCategory}
              errorMessage={errors.mainCategory}
              editable={false}
              placeholder="Select Main Category"
              icon={<Ionicons name="add" size={width * 0.05} color="#015185" />}
              onIconPress={handleMainCategoryPlusClick}
              onPress={handleMainCategoryPlusClick}
            />

            {selectedMainCategory === 'Rigs' ? (
              <View style={inputStyles.container}>
                <Text style={inputStyles.label}>
                  {field1Label} <Text style={inputStyles.requiredError}>*</Text>
                </Text>
                <DropDownPicker
                  open={rigTypeOpen}
                  value={rigType}
                  items={rigTypeItems}
                  setOpen={setRigTypeOpen}
                  setValue={setRigType}
                  setItems={setRigTypeItems}
                  placeholder={`Select ${field1Label}`}
                  style={[styles.dropdown, errors.dynamicField1 ? inputStyles.inputError : null]}
                  dropDownContainerStyle={styles.dropdownContainer}
                  textStyle={styles.dropdownText}
                  placeholderStyle={styles.placeholderStyle}
                  zIndex={4000}
                  dropDownDirection="AUTO"
                  onChangeValue={() => setErrors({ ...errors, dynamicField1: '' })}
                />
                {errors.dynamicField1 && (
                  <Text style={inputStyles.errorText}>{errors.dynamicField1}</Text>
                )}
              </View>
            ) : (
              <CustomInputField
                label={field1Label}
                value={dynamicField1}
                onChangeText={(text) => {
                  setDynamicField1(text);
                  setErrors({ ...errors, dynamicField1: '' });
                }}
                isRequired
                showError={!!errors.dynamicField1}
                errorMessage={errors.dynamicField1}
                placeholder={`Enter ${field1Label}`}
                inputRef={dynamicField1Ref}
                returnKeyType="next"
                onSubmitEditing={() => dynamicField2Ref.current?.focus()}
              />
            )}

            <CustomInputField
              label={field2Label}
              value={dynamicField2}
              onChangeText={(text) => {
                setDynamicField2(text);
                setErrors({ ...errors, dynamicField2: '' });
              }}
              isRequired
              showError={!!errors.dynamicField2}
              errorMessage={errors.dynamicField2}
              placeholder={`Enter ${field2Label}`}
              inputRef={dynamicField2Ref}
              returnKeyType="next"
              onSubmitEditing={() => dynamicField3Ref.current?.focus()}
            />

            <CustomInputField
              label={`${field3Label} (Optional)`}
              value={dynamicField3}
              onChangeText={(text) => setDynamicField3(text)}
              placeholder={`Enter ${field3Label}`}
              inputRef={dynamicField3Ref}
              returnKeyType="next"
              onSubmitEditing={() => equipmentNumberRef.current?.focus()}
            />

            <CustomInputField
              label="Equipment #"
              value={formData.equipmentNumber}
              onChangeText={(text) => handleInputChange('equipmentNumber', text)}
              isRequired={
                !['Light Vehicles', 'Heavy Vehicles', 'Rigs'].includes(selectedMainCategory)
              }
              showError={!!errors.equipmentNumber}
              errorMessage={errors.equipmentNumber}
              icon={<MaterialIcons name="help-outline" size={width * 0.05} color="#015185" />}
              onIconPress={showEquipmentInfoAlert}
              placeholder="Enter Equipment Number"
              inputRef={equipmentNumberRef}
              returnKeyType="next"
              onSubmitEditing={() => equipmentNameRef.current?.focus()}
            />

            <CustomInputField
              label="Equipment Name"
              value={formData.equipmentName}
              onChangeText={(text) => handleInputChange('equipmentName', text)}
              isRequired={
                !['Light Vehicles', 'Heavy Vehicles', 'Rigs'].includes(selectedMainCategory)
              }
              showError={!!errors.equipmentName}
              errorMessage={errors.equipmentName}
              placeholder="Equipment Name"
              inputRef={equipmentNameRef}
              containerRef={equipmentNameContainerRef}
              returnKeyType="next"
              onSubmitEditing={() => meterReadingRef.current?.focus()}
            />

            <View style={inputStyles.container}>
              <Text style={inputStyles.label}>Meter Reading</Text>
              <View style={quantityStyles.row}>
                <View style={[inputStyles.inputContainer, errors.meterReading ? inputStyles.inputError : null, quantityStyles.inputWrapper]}>
                  <TextInput
                    ref={meterReadingRef}
                    style={[inputStyles.input, { flex: 1 }]}
                    placeholder="Enter Meter Reading"
                    keyboardType="numeric"
                    value={formData.meterReading}
                    onChangeText={(text) => handleInputChange('meterReading', text)}
                    placeholderTextColor="#888"
                    autoCapitalize="none"
                    autoCorrect={false}
                    returnKeyType="next"
                    onSubmitEditing={() => modelRef.current?.focus()}
                  />
                </View>
                <View style={quantityStyles.unitBox}>
                  <DropDownPicker
                    open={meterOpen}
                    value={meterUnit}
                    items={meterItems}
                    setOpen={setMeterOpen}
                    setValue={setMeterUnit}
                    setItems={setMeterItems}
                    style={[styles.dropdown, quantityStyles.unitDropdown]}
                    dropDownContainerStyle={styles.dropdownContainer}
                    textStyle={styles.dropdownText}
                    placeholderStyle={styles.placeholderStyle}
                    zIndex={2000}
                    dropDownDirection="AUTO"
                  />
                </View>
              </View>
              {errors.meterReading && (
                <Text style={inputStyles.errorText}>{errors.meterReading}</Text>
              )}
            </View>

            <CustomInputField
              label="Model"
              value={formData.model}
              onChangeText={(text) => handleInputChange('model', text)}
              showError={!!errors.model}
              errorMessage={errors.model}
              placeholder="Enter Model Number"
              inputRef={modelRef}
              returnKeyType="next"
              onSubmitEditing={() => lastServiceMeterRef.current?.focus()}
            />
            <View style={styles.optionalRow}>
              <TouchableOpacity onPress={toggleOptionalFields} activeOpacity={0.7}>
                <Text style={styles.optionalText}>Optional Fields</Text>
              </TouchableOpacity>
              <TouchableOpacity onPress={toggleOptionalFields} activeOpacity={0.7}>
                <MaterialIcons
                  name={optionalFieldsVisible ? 'arrow-drop-up' : 'arrow-drop-down'}
                  size={width * 0.05}
                  color="#888"
                />
              </TouchableOpacity>
            </View>

            {optionalFieldsVisible && (
              <Animated.View
                style={[
                  styles.optionalFieldsContainer,
                  {
                    transform: [{ translateY: slideAnim }],
                    opacity: slideAnim.interpolate({
                      inputRange: [0, 300],
                      outputRange: [1, 0],
                    }),
                  },
                ]}
              >
                <CustomInputField
                  label="Last Service Meter Reading"
                  value={formData.lastServiceMeter}
                  onChangeText={(text) => handleInputChange('lastServiceMeter', text)}
                  keyboardType="numeric"
                  placeholder="Last Service Meter Reading"
                  inputRef={lastServiceMeterRef}
                  containerRef={lastServiceMeterContainerRef}
                  returnKeyType="next"
                  onSubmitEditing={() => lastServiceDateRef.current?.focus()}
                />

                <CustomInputField
                  label="Last Service Date"
                  value={formatDate(lastServiceDate)}
                  onChangeText={() => {}}
                  placeholder="Select Last Service Date"
                  editable={false}
                  onPress={() => openDatePicker('lastServiceDate')}
                  inputRef={lastServiceDateRef}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#015185" />}
                  onIconPress={() => openDatePicker('lastServiceDate')}
                  returnKeyType="next"
                  onSubmitEditing={() => vinRef.current?.focus()}
                />

                <CustomInputField
                  label="VIN #"
                  value={formData.vin}
                  onChangeText={(text) => handleInputChange('vin', text)}
                  placeholder="VIN #"
                  inputRef={vinRef}
                  returnKeyType="next"
                  onSubmitEditing={() => fleetValueRef.current?.focus()}
                />

                <CustomInputField
                  label="Fleet Value"
                  value={formData.fleetValue}
                  onChangeText={(text) => handleInputChange('fleetValue', text)}
                  keyboardType="numeric"
                  placeholder="Fleet Value"
                  inputRef={fleetValueRef}
                  returnKeyType="next"
                  onSubmitEditing={() => purchaseDateRef.current?.focus()}
                />

                <CustomInputField
                  label="Purchase Date"
                  value={formatDate(purchaseDate)}
                  onChangeText={() => {}}
                  placeholder="Select Purchase Date"
                  editable={false}
                  onPress={() => openDatePicker('purchaseDate')}
                  inputRef={purchaseDateRef}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#015185" />}
                  onIconPress={() => openDatePicker('purchaseDate')}
                  returnKeyType="next"
                  onSubmitEditing={() => warrantyExpDateRef.current?.focus()}
                />

                <CustomInputField
                  label="Warranty Exp Date"
                  value={formatDate(warrantyExpDate)}
                  onChangeText={() => {}}
                  placeholder="Select Warranty Exp Date"
                  editable={false}
                  onPress={() => openDatePicker('warrantyExpDate')}
                  inputRef={warrantyExpDateRef}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#015185" />}
                  onIconPress={() => openDatePicker('warrantyExpDate')}
                  returnKeyType="next"
                  onSubmitEditing={() => operatorNameRef.current?.focus()}
                />

                <CustomInputField
                  label="Operator Name"
                  value={formData.operatorName}
                  onChangeText={(text) => handleInputChange('operatorName', text)}
                  placeholder="Operator Name"
                  inputRef={operatorNameRef}
                  returnKeyType="done"
                />

                <View style={[inputStyles.container, { flexDirection: 'row', alignItems: 'center' }]}>
                  <Text style={[inputStyles.label, { flex: 1 }]}>Maintenance Priority</Text>
                  <View style={styles.dropdownWrapper}>
                    <DropDownPicker
                      open={priorityOpen}
                      value={maintenancePriority}
                      items={priorityItems}
                      setOpen={setPriorityOpen}
                      setValue={setMaintenancePriority}
                      setItems={setPriorityItems}
                      style={[styles.dropdownButton, { width: width * 0.2 }]}
                      dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: height * 0.15 }]}
                      textStyle={styles.dropdownText}
                      placeholderStyle={styles.placeholderStyle}
                      zIndex={1000}
                      arrowIconStyle={{ tintColor: '#888' }}
                      dropDownDirection="TOP"
                    />
                  </View>
                </View>

                <View style={[inputStyles.container, { flexDirection:'row', alignItems: 'center' }]}>
                  <Text style={[inputStyles.label, { flex: 1 }]}>Ownership Mode</Text>
                  <View style={styles.dropdownWrapper}>
                    <DropDownPicker
                      open={ownershipOpen}
                      value={ownershipMode}
                      items={ownershipItems}
                      setOpen={setOwnershipOpen}
                      setValue={setOwnershipMode}
                      setItems={setOwnershipItems}
                      style={[styles.dropdownButton, { width: width * 0.3 }]}
                      dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: height * 0.15 }]}
                      textStyle={styles.dropdownText}
                      placeholderStyle={styles.placeholderStyle}
                      zIndex={900}
                      arrowIconStyle={{ tintColor: '#888' }}
                      dropDownDirection="TOP"
                    />
                  </View>
                </View>
              </Animated.View>
            )}
            <View style={{ height: width * 0.3 }} />
          </View>
        </ScrollView>
      </KeyboardAvoidingView>

      <Modal
        visible={mainCategoryModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setMainCategoryModalVisible(false)}
      >
        <View style={styles.modalContainer}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Main Category</Text>
            {mainCategoryItems.map((item) => (
              <TouchableOpacity
                key={item.value}
                style={styles.modalItem}
                onPress={() => selectMainCategory(item.value)}
              >
                <Text style={styles.modalItemText}>{item.label}</Text>
              </TouchableOpacity>
            ))}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setMainCategoryModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={loadingModalVisible}
        transparent={true}
        animationType="fade"
      >
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Processing Equipment Data</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={datePickerVisible}
        transparent={true}
        animationType="fade"
        onRequestClose={handleDateCancel}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.datePickerContainer}>
            <DatePicker
              date={tempDate}
              onDateChange={setTempDate}
              mode="date"
              androidVariant="nativeAndroid"
            />
            <View style={styles.datePickerButtons}>
              <TouchableOpacity
                style={[styles.datePickerButton, styles.cancelButton]}
                onPress={handleDateCancel}
              >
                <Text style={styles.datePickerButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.datePickerButton, styles.doneButton]}
                onPress={handleDateConfirm}
              >
                <Text style={styles.datePickerButtonText}>Done</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      <Toast />
    </View>
  );
};

const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.04,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: width * 0.01,
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  inputContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: width * 0.012,
    backgroundColor: '#fff',
  },
  input: {
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.035,
    fontSize: width * 0.04,
    color: '#000',
  },
  inputError: {
    borderColor: 'red',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: width * 0.005,
    marginLeft: width * 0.005,
  },
  iconButton: {
    padding: width * 0.025,
  },
});

const quantityStyles = StyleSheet.create({
  row: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  inputWrapper: {
    flex: 1,
    marginRight: width * 0.02,
  },
  unitBox: {
    width: width * 0.25,
  },
  unitDropdown: {
    height: width * 0.12,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: width * 0.012,
    backgroundColor: '#fff',
    paddingHorizontal: width * 0.02,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  keyboardAvoidingView: {
    flex: 1,
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.11,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: '400',
    marginLeft: width * 0.025,
  },
  saveText: {
    color: '#0055A5',
    fontSize: width * 0.045,
    fontWeight: '600',
  },
  savebtn01: {
    backgroundColor: 'white',
    paddingHorizontal: width * 0.035,
    paddingVertical: width * 0.007,
    borderRadius: width * 0.007,
  },
  mainContent: {
    flex: 1,
    padding: width * 0.04,
  },
  equipmentIdRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F2F2F2',
    borderRadius: width * 0.015,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.035,
    marginBottom: width * 0.04,
  },
  equipmentIdLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
  equipmentIdValue: {
    fontSize: width * 0.038,
    color: '#025383',
    fontWeight: 'bold',
  },
  generalErrorText: {
    color: '#d32f2f',
    fontSize: width * 0.035,
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  inputContainer: {
    width: '100%',
  },
  dropdown: {
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#015185',
    height: width * 0.12,
    paddingHorizontal: width * 0.03,
  },
  disabledDropdown: {
    backgroundColor: '#e0e0e0',
  },
  dropdownContainer: {
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#015185',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 2,
    maxHeight: height * 0.25,
  },
  dropdownButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#fff',
    height: width * 0.12,
    borderRadius: width * 0.012,
    paddingHorizontal: width * 0.03,
    borderWidth: 1,
    borderColor: '#015185',
  },
  dropdownText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  placeholderStyle: {
    fontSize: width * 0.04,
    color: '#888',
  },
  iconButton: {
    padding: width * 0.012,
  },
  optionalRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginTop: width * 0.025,
    paddingVertical: width * 0.025,
    borderTopWidth: 1,
    borderTopColor: '#eee',
  },
  optionalText: {
    fontSize: width * 0.04,
    color: '#888',
  },
  optionalFieldsContainer: {
    marginTop: width * 0.025,
    borderTopWidth: 1,
    borderTopColor: '#eee',
    paddingTop: width * 0.025,
  },
  modalContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  modalContent: {
    width: width * 0.9,
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    maxHeight: height * 0.8,
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    marginBottom: width * 0.037,
  },
  modalItem: {
    padding: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  modalItemText: {
    fontSize: width * 0.04,
  },
  modalCloseButton: {
    marginTop: width * 0.037,
    alignItems: 'center',
  },
  modalCloseText: {
    color: '#3481BC',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  dropdownWrapper: {
    position: 'relative',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: width * 0.7,
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
  },
  syncIcon: {
    width: width * 0.075,
    height: width * 0.075,
    marginRight: width * 0.025,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  datePickerContainer: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  datePickerButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
    marginTop: 10,
  },
  datePickerButton: {
    flex: 1,
    borderRadius: 5,
    paddingVertical: 10,
    alignItems: 'center',
    marginHorizontal: 5,
  },
  cancelButton: {
    backgroundColor: '#A0A0A0',
  },
  doneButton: {
    backgroundColor: '#015185',
  },
  datePickerButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
});

export default Add_equipment_update;