const admin = require('firebase-admin');

// Function to send notice board notifications to all admin users
async function sendNoticeBoardNotificationToAllAdmins(noticeData) {
  try {
    console.log('🚀 Starting notice board notification to all admins...');
    
    // Get all admin users from the database
    const adminsSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    
    if (!adminsSnapshot.exists()) {
      console.log('⚠️ No admin users found in database');
      return { success: false, error: 'No admin users found' };
    }

    const admins = adminsSnapshot.val();
    const results = [];
    let totalAdmins = 0;
    let successCount = 0;
    let failureCount = 0;
    let noDevicesCount = 0;
    let invalidTokenCount = 0;

    // Process each admin user
    for (const [adminId, adminData] of Object.entries(admins)) {
      totalAdmins++;
      
      try {
        console.log(`🔍 Checking admin ${adminId} (${adminData.name || 'Unknown'}) for FCM tokens...`);
        
        let devices = {};
        
        // First check if there are devices in the new multi-device structure
        if (adminData.devices && Object.keys(adminData.devices).length > 0) {
          devices = adminData.devices;
          console.log(`📱 Admin ${adminId} has ${Object.keys(devices).length} devices in multi-device structure:`, Object.keys(devices));
          
          // Log each device's status
          Object.entries(devices).forEach(([deviceKey, deviceData]) => {
            console.log(`📱 Device ${deviceKey}: token=${deviceData.token ? 'YES' : 'NO'}, status=${deviceData.fcmStatus || 'unknown'}`);
          });
        }
        // Fallback to the old single FCM token structure
        else if (adminData.NotificationFCMtoken) {
          devices = {
            FCM: {
              token: adminData.NotificationFCMtoken,
              fcmStatus: 'active',
              lastTokenUpdate: adminData.lastTokenUpdate || new Date().toISOString(),
              deviceInfo: {
                platform: 'legacy',
                deviceId: 'legacy',
                timestamp: new Date().toISOString()
              }
            }
          };
          console.log(`📱 Admin ${adminId} has single FCM token in legacy structure:`, adminData.NotificationFCMtoken.substring(0, 20) + '...');
        }
        
        if (Object.keys(devices).length === 0) {
          console.log(`⚠️ Admin ${adminId} has no FCM devices or tokens`);
          noDevicesCount++;
          results.push({ 
            adminId, 
            adminName: adminData.name || 'Unknown', 
            status: 'no_devices', 
            message: 'No FCM devices or tokens found' 
          });
          continue;
        }

        // Send notification to all devices of this admin
        const adminResult = await sendNoticeNotificationToAdmin(adminId, adminData, noticeData, devices);
        results.push(adminResult);
        
        if (adminResult.status === 'success') {
          successCount++;
        } else {
          failureCount++;
        }
        
        // Count invalid tokens
        if (adminResult.invalidTokens) {
          invalidTokenCount += adminResult.invalidTokens;
        }
        
      } catch (error) {
        console.error(`❌ Error processing admin ${adminId}:`, error);
        results.push({ 
          adminId, 
          adminName: adminData.name || 'Unknown', 
          status: 'error', 
          error: error.message 
        });
        failureCount++;
      }
    }

    console.log(`✅ Notice board notification completed!`);
    console.log(`📊 Results: ${successCount} successful, ${failureCount} failed, ${noDevicesCount} no devices, ${invalidTokenCount} invalid tokens, ${totalAdmins} total admins`);

    return {
      success: successCount > 0,
      message: `Notice board notification sent to ${successCount} admins, ${failureCount} failed, ${noDevicesCount} no devices, ${invalidTokenCount} invalid tokens`,
      totalAdmins,
      successfulAdmins: successCount,
      failedAdmins: failureCount,
      noDevicesAdmins: noDevicesCount,
      invalidTokens: invalidTokenCount,
      results: results
    };

  } catch (error) {
    console.error('❌ Error in sendNoticeBoardNotificationToAllAdmins:', error);
    return { success: false, error: error.message };
  }
}

// Function to send notice notification to a specific admin
async function sendNoticeNotificationToAdmin(adminId, adminData, noticeData, devices) {
  try {
    console.log(`📱 Sending notice notification to admin: ${adminData.name || adminId}`);
    
    // Save notification to admin's database
    const saveResult = await saveNoticeNotificationToAdmin(adminId, noticeData);
    if (!saveResult.success) {
      console.warn(`⚠️ Failed to save notice notification to admin ${adminId} database`);
    }

    // Send FCM notification to all devices
    const deviceResults = [];
    let deviceSuccessCount = 0;
    let deviceFailureCount = 0;
    let invalidTokenCount = 0;

    for (const [deviceKey, deviceData] of Object.entries(devices)) {
      if (deviceData.token && deviceData.fcmStatus === 'active') {
        try {
          const fcmResult = await sendNoticeFCMNotification(
            deviceData.token, 
            noticeData, 
            adminData, 
            deviceKey
          );
          
          if (fcmResult.success) {
            deviceSuccessCount++;
            deviceResults.push({ 
              deviceKey, 
              status: 'success', 
              messageId: fcmResult.messageId 
            });
          } else {
            deviceFailureCount++;
            deviceResults.push({ 
              deviceKey, 
              status: 'failed', 
              error: fcmResult.error 
            });
            
            // If token is invalid, mark device as inactive
            if (fcmResult.code === 'messaging/invalid-registration-token' || 
                fcmResult.code === 'messaging/registration-token-not-registered') {
              await markDeviceAsInactive(adminId, deviceKey);
              invalidTokenCount++;
            }
          }
        } catch (error) {
          deviceFailureCount++;
          deviceResults.push({ 
            deviceKey, 
            status: 'error', 
            error: error.message 
          });
        }
      } else {
        console.log(`⚠️ Skipping device ${deviceKey}: token=${!!deviceData.token}, status=${deviceData.fcmStatus}`);
      }
    }

    return {
      adminId,
      adminName: adminData.name || 'Unknown',
      status: deviceSuccessCount > 0 ? 'success' : 'failed',
      message: `Sent to ${deviceSuccessCount} devices, ${deviceFailureCount} failed, ${invalidTokenCount} invalid tokens`,
      totalDevices: Object.keys(devices).length,
      successfulDevices: deviceSuccessCount,
      failedDevices: deviceFailureCount,
      invalidTokens: invalidTokenCount,
      notificationSaved: saveResult.success,
      deviceResults: deviceResults
    };

  } catch (error) {
    console.error(`❌ Error sending notice notification to admin ${adminId}:`, error);
    return {
      adminId,
      adminName: adminData.name || 'Unknown',
      status: 'error',
      error: error.message
    };
  }
}

// Function to mark a device as inactive due to invalid token
async function markDeviceAsInactive(userId, deviceKey) {
  try {
    await admin.database().ref(`/Globalusers/admins/${userId}/devices/${deviceKey}/fcmStatus`).set('inactive');
    await admin.database().ref(`/Globalusers/admins/${userId}/devices/${deviceKey}/lastTokenError`).set({
      error: 'Invalid FCM token',
      timestamp: new Date().toISOString()
    });
    console.log(`✅ Marked device ${deviceKey} as inactive for user ${userId}`);
  } catch (error) {
    console.error(`❌ Error marking device ${deviceKey} as inactive:`, error);
  }
}

// Function to save notice notification to admin's database
async function saveNoticeNotificationToAdmin(adminId, noticeData) {
  try {
    const notificationRef = admin.database().ref(`/Globalusers/admins/${adminId}/notifications`);
    const newNotificationRef = notificationRef.push();
    
    const notificationData = {
      title: `📢 New Notice: ${noticeData.title}`,
      body: noticeData.description.substring(0, 100) + (noticeData.description.length > 100 ? '...' : ''),
      data: {
        type: 'notice_board',
        noticeId: noticeData.noticeId,
        title: noticeData.title,
        description: noticeData.description,
        imageUrl: noticeData.imageUrl || null,
        createdAt: noticeData.createdAt,
        source: 'notice_board',
        priority: 'normal'
      },
      timestamp: Date.now(),
      isRead: false,
      createdAt: new Date().toISOString(),
    };
    
    await newNotificationRef.set(notificationData);
    
    console.log(`✅ Notice notification saved to admin ${adminId} database`);
    return { success: true, notificationId: newNotificationRef.key };
  } catch (error) {
    console.error(`❌ Error saving notice notification to admin ${adminId}:`, error);
    return { success: false, error: error.message };
  }
}

// Function to send FCM notification for notice
async function sendNoticeFCMNotification(fcmToken, noticeData, adminData, deviceKey) {
  try {
    const message = {
      token: fcmToken,
      notification: {
        title: `📢 New Notice: ${noticeData.title}`,
        body: noticeData.description.substring(0, 100) + (noticeData.description.length > 100 ? '...' : ''),
      },
      data: {
        type: 'notice_board',
        noticeId: noticeData.noticeId || 'unknown',
        title: noticeData.title,
        description: noticeData.description,
        imageUrl: noticeData.imageUrl || '',
        createdAt: noticeData.createdAt || Date.now().toString(),
        source: 'notice_board',
        priority: 'normal',
        deviceKey: deviceKey,
        adminName: adminData.name || 'Admin',
        adminRole: adminData.role || 'Admin'
      },
      android: {
        notification: {
          channelId: 'notice_board',
          importance: 'high',
          priority: 'high',

          icon: 'ic_notification',
          color: '#3481BC',
          sound: 'default',
          vibrateTimingsMillis: [0, 250, 250, 250],
          lightSettings: {
            color: '#3481BC',
            lightOnDurationMillis: 1000,
            lightOffDurationMillis: 1000,
          },
        },
        priority: 'high',
      },
      apns: {
        payload: {
          aps: {
            'content-available': 1,
            'mutable-content': 1,
            sound: 'default',
            badge: 1,
            alert: {
              title: `📢 New Notice: ${noticeData.title}`,
              body: noticeData.description.substring(0, 100) + (noticeData.description.length > 100 ? '...' : ''),
            },
            category: 'NOTICE_BOARD',
          },
        },
        headers: {
          'apns-priority': '10',
          'apns-push-type': 'alert',
        },
      },
    };

    console.log(`📱 Sending notice FCM notification to device ${deviceKey} with token:`, fcmToken.substring(0, 20) + '...');
    const response = await admin.messaging().send(message);
    
    console.log(`✅ FCM notice notification sent successfully to device ${deviceKey}`);
    return { success: true, messageId: response };
    
  } catch (error) {
    console.error(`❌ Error sending FCM notice notification to device ${deviceKey}:`, error);
    
    // Handle specific FCM errors
    if (error.code === 'messaging/invalid-registration-token' || 
        error.code === 'messaging/registration-token-not-registered') {
      console.log(`🔄 FCM token is invalid or expired for device ${deviceKey}, marking as inactive`);
      return { success: false, error: 'Invalid or expired FCM token', code: error.code };
    }
    
    return { success: false, error: error.message, code: error.code };
  }
}

// Function to create notice board notification channel (for Android)
async function createNoticeBoardNotificationChannel() {
  try {
    // This function can be called from the client side to create the channel
    // For now, we'll just log that it should be created
    console.log('📱 Notice board notification channel should be created on client side');
    return { success: true, message: 'Channel creation handled on client side' };
  } catch (error) {
    console.error('❌ Error creating notice board notification channel:', error);
    return { success: false, error: error.message };
  }
}

// Function to send test notice board notification
async function sendTestNoticeBoardNotification() {
  try {
    const testNoticeData = {
      noticeId: 'test_' + Date.now(),
      title: '🧪 Test Notice',
      description: 'This is a test notice to verify the notice board notification system is working correctly.',
      imageUrl: null,
      createdAt: new Date().toISOString()
    };

    console.log('🧪 Sending test notice board notification...');
    const result = await sendNoticeBoardNotificationToAllAdmins(testNoticeData);
    
    return result;
  } catch (error) {
    console.error('❌ Error sending test notice board notification:', error);
    return { success: false, error: error.message };
  }
}

// Function to get notice board notification statistics
async function getNoticeBoardNotificationStats() {
  try {
    const adminsSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    
    if (!adminsSnapshot.exists()) {
      return { success: false, error: 'No admin users found' };
    }

    const admins = adminsSnapshot.val();
    const stats = {
      totalAdmins: 0,
      adminsWithDevices: 0,
      adminsWithLegacyTokens: 0,
      adminsWithNoTokens: 0,
      totalDevices: 0,
      totalLegacyTokens: 0,
      adminDetails: []
    };

    for (const [adminId, adminData] of Object.entries(admins)) {
      stats.totalAdmins++;
      
      const adminDetail = {
        adminId,
        adminName: adminData.name || 'Unknown',
        hasDevices: false,
        hasLegacyToken: false,
        deviceCount: 0,
        legacyToken: null
      };

      // Check multi-device structure
      if (adminData.devices && Object.keys(adminData.devices).length > 0) {
        adminDetail.hasDevices = true;
        adminDetail.deviceCount = Object.keys(adminData.devices).length;
        stats.adminsWithDevices++;
        stats.totalDevices += adminDetail.deviceCount;
      }

      // Check legacy token structure
      if (adminData.NotificationFCMtoken) {
        adminDetail.hasLegacyToken = true;
        adminDetail.legacyToken = adminData.NotificationFCMtoken.substring(0, 20) + '...';
        stats.adminsWithLegacyTokens++;
        stats.totalLegacyTokens++;
      }

      if (!adminDetail.hasDevices && !adminDetail.hasLegacyToken) {
        stats.adminsWithNoTokens++;
      }

      stats.adminDetails.push(adminDetail);
    }

    return {
      success: true,
      stats: stats
    };

  } catch (error) {
    console.error('❌ Error getting notice board notification stats:', error);
    return { success: false, error: error.message };
  }
}

// Function to manually check FCM tokens for a specific user
async function checkUserFCMTokens(userId) {
  try {
    const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userId}`).once('value');
    
    if (!userSnapshot.exists()) {
      return { success: false, error: 'User not found' };
    }

    const userData = userSnapshot.val();
    const result = {
      userId,
      userName: userData.name || 'Unknown',
      hasDevices: false,
      hasLegacyToken: false,
      devices: {},
      legacyToken: null,
      deviceCount: 0,
      activeDeviceCount: 0
    };

    // Check multi-device structure
    if (userData.devices && Object.keys(userData.devices).length > 0) {
      result.hasDevices = true;
      result.devices = userData.devices;
      result.deviceCount = Object.keys(userData.devices).length;
      
      // Count active devices
      Object.entries(userData.devices).forEach(([deviceKey, deviceData]) => {
        if (deviceData.fcmStatus === 'active' && deviceData.token) {
          result.activeDeviceCount++;
        }
      });
    }

    // Check legacy token structure
    if (userData.NotificationFCMtoken) {
      result.hasLegacyToken = true;
      result.legacyToken = userData.NotificationFCMtoken;
    }

    return {
      success: true,
      result: result
    };

  } catch (error) {
    console.error('❌ Error checking user FCM tokens:', error);
    return { success: false, error: error.message };
  }
}

// Function to send profile update notification to a specific user
async function sendProfileUpdateNotification(userId, updateData) {
  try {
    console.log(`🚀 Sending profile update notification to user ${userId}...`);
    
    const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userId}`).once('value');
    
    if (!userSnapshot.exists()) {
      console.log(`⚠️ User ${userId} not found in database`);
      return { success: false, error: 'User not found' };
    }

    const userData = userSnapshot.val();
    const results = [];
    let successCount = 0;
    let failureCount = 0;
    let noDevicesCount = 0;
    let invalidTokenCount = 0;

    // Prepare notification message
    const { adminName, changedFields, emailChanged, previousEmail, newEmail } = updateData;
    const baseMsg = `Your profile has been updated by ${adminName}.`;
    const details = changedFields && changedFields.length > 0 ? ` Changes: ${changedFields.join(', ')}` : '';
    const emailChangeMsg = emailChanged ? ` Email changed from ${previousEmail || 'N/A'} to ${newEmail}.` : '';
    const notificationMessage = `${baseMsg}${details}${emailChangeMsg}`.trim();

    // Check multi-device structure first
    if (userData.devices && Object.keys(userData.devices).length > 0) {
      console.log(`📱 User ${userId} has ${Object.keys(userData.devices).length} devices in multi-device structure`);
      
      for (const [deviceKey, deviceData] of Object.entries(userData.devices)) {
        if (deviceData.fcmStatus === 'active' && deviceData.token) {
          try {
            const result = await sendNoticeFCMNotification(
              deviceData.token,
              'Profile Updated',
              notificationMessage,
              {
                type: 'profile_updated',
                updatedBy: adminName,
                emailChanged: emailChanged,
                previousEmail: previousEmail,
                newEmail: newEmail,
                changedFields: changedFields,
                timestamp: new Date().toISOString()
              }
            );
            
            if (result.success) {
              successCount++;
              results.push({ deviceKey, status: 'success', message: 'Notification sent successfully' });
            } else {
              failureCount++;
              results.push({ deviceKey, status: 'failed', error: result.error });
              
              // Check if token is invalid
              if (result.error && (result.error.includes('invalid-registration-token') || result.error.includes('registration-token-not-registered'))) {
                invalidTokenCount++;
              }
            }
          } catch (error) {
            failureCount++;
            results.push({ deviceKey, status: 'error', error: error.message });
          }
        } else {
          noDevicesCount++;
          results.push({ deviceKey, status: 'inactive', message: 'Device inactive or no token' });
        }
      }
    }
    // Fallback to legacy token structure
    else if (userData.NotificationFCMtoken) {
      console.log(`📱 User ${userId} has legacy FCM token`);
      
      try {
        const result = await sendNoticeFCMNotification(
          userData.NotificationFCMtoken,
          'Profile Updated',
          notificationMessage,
          {
            type: 'profile_updated',
            updatedBy: adminName,
            emailChanged: emailChanged,
            previousEmail: previousEmail,
            newEmail: newEmail,
            changedFields: changedFields,
            timestamp: new Date().toISOString()
          }
        );
        
        if (result.success) {
          successCount++;
          results.push({ deviceKey: 'legacy', status: 'success', message: 'Notification sent successfully' });
        } else {
          failureCount++;
          results.push({ deviceKey: 'legacy', status: 'failed', error: result.error });
          
          // Check if token is invalid
          if (result.error && (result.error.includes('invalid-registration-token') || result.error.includes('registration-token-not-registered'))) {
            invalidTokenCount++;
          }
        }
      } catch (error) {
        failureCount++;
        results.push({ deviceKey: 'legacy', status: 'error', error: error.message });
      }
    } else {
      noDevicesCount++;
      results.push({ deviceKey: 'none', status: 'no_devices', message: 'No FCM devices or tokens found' });
    }

    console.log(`✅ Profile update notification completed for user ${userId}!`);
    console.log(`📊 Results: ${successCount} successful, ${failureCount} failed, ${noDevicesCount} no devices, ${invalidTokenCount} invalid tokens`);

    return {
      success: true,
      userId,
      userName: userData.name || 'Unknown',
      results: results,
      summary: {
        totalDevices: results.length,
        successCount,
        failureCount,
        noDevicesCount,
        invalidTokenCount
      }
    };

  } catch (error) {
    console.error(`❌ Error sending profile update notification to user ${userId}:`, error);
    return { success: false, error: error.message };
  }
}

module.exports = {
  sendNoticeBoardNotificationToAllAdmins,
  sendNoticeNotificationToAdmin,
  saveNoticeNotificationToAdmin,
  sendNoticeFCMNotification,
  createNoticeBoardNotificationChannel,
  sendTestNoticeBoardNotification,
  getNoticeBoardNotificationStats,
  checkUserFCMTokens,
  sendProfileUpdateNotification
};
