const admin = require('firebase-admin');

// Initialize Firebase Admin if not already initialized
if (!admin.apps.length) {
  admin.initializeApp();
}

// Function to send push notification to a specific FCM token
async function sendPushNotificationToToken(token, title, body, data = {}) {
  try {
    // Convert all data values to strings as required by FCM
    const stringData = {};
    Object.keys(data).forEach(key => {
      stringData[key] = String(data[key]);
    });

    const message = {
      token: token,
      notification: {
        title: title,
        body: body,
      },
      data: {
        ...stringData,
        title: title,
        body: body,
      },
      android: {
        priority: 'high',
        notification: {
          title: title,
          body: body,
          sound: 'default',
          channel_id: 'default',
          icon: 'ic_notification',
          color: '#015185',
        },
        data: {
          ...stringData,
          title: title,
          body: body,
        },
      },
      apns: {
        payload: {
          aps: {
            alert: {
              title: title,
              body: body,
            },
            sound: 'default',
            badge: 1,
            'content-available': 1,
            'mutable-content': 1,
            category: 'GENERAL',
          },
        },
        headers: {
          'apns-priority': '10',
          'apns-push-type': 'alert',
        },
      },
    };

    console.log('📱 Sending FCM message to token:', token.substring(0, 20) + '...');
    const response = await admin.messaging().send(message);
    console.log('✅ Successfully sent message:', response);
    return { success: true, messageId: response };
  } catch (error) {
    console.error('❌ Error sending message:', error);
    
    // Handle specific FCM errors
    if (error.code === 'messaging/invalid-registration-token' || 
        error.code === 'messaging/registration-token-not-registered') {
      console.log('🔄 FCM token is invalid or expired, marking device as inactive');
      return { success: false, error: 'Invalid or expired FCM token', code: error.code };
    }
    
    return { success: false, error: error.message, code: error.code };
  }
}

// Function to send push notification by email (requires getting FCM token first)
async function sendPushNotificationByEmail(email, title, body, data = {}) {
  try {
    console.log('📧 Sending push notification by email:', email);
    
    // Get user by email from Firebase Auth
    const userRecord = await admin.auth().getUserByEmail(email);
    
    if (!userRecord) {
      console.log('❌ User not found with email:', email);
      return { success: false, error: 'User not found with this email' };
    }

    console.log('✅ User found:', userRecord.uid);

    // Get FCM tokens from Realtime Database (matching AdminLoginScreen storage)
    const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userRecord.uid}`).once('value');
    
    if (!userSnapshot.exists()) {
      console.log('❌ User not found in admin database:', userRecord.uid);
      return { success: false, error: 'User not found in admin database' };
    }

    const userData = userSnapshot.val();
    console.log('📋 User data retrieved:', Object.keys(userData));
    
    // Check for devices structure (new storage method from AdminLoginScreen)
    const devices = userData.devices || {};
    console.log('📱 Devices found:', Object.keys(devices));
    
    if (Object.keys(devices).length === 0) {
      // Fallback: check for old NotificationFCMtoken field
      if (userData.NotificationFCMtoken) {
        console.log('📱 Using old FCM token format');
        const result = await sendPushNotificationToToken(userData.NotificationFCMtoken, title, body, {
          ...data,
          deviceKey: 'legacy',
          deviceId: 'legacy',
          targetDevice: 'legacy',
        });
        
        if (result.success) {
          // Save notification to user's database
          const saveResult = await saveNotificationToUser(userRecord.uid, title, body, data);
          
          return { 
            success: true, 
            message: 'Sent to 1 legacy device',
            results: [{ deviceKey: 'legacy', status: 'success', messageId: result.messageId }],
            totalDevices: 1,
            successfulDevices: 1,
            failedDevices: 0,
            notificationSaved: saveResult.success
          };
        } else {
          // If token is invalid, mark it as inactive
          if (result.code === 'messaging/invalid-registration-token' || 
              result.code === 'messaging/registration-token-not-registered') {
            await markLegacyTokenAsInactive(userRecord.uid);
          }
          return { success: false, error: result.error };
        }
      }
      
      console.log('❌ No FCM tokens found for user');
      return { success: false, error: 'No FCM tokens found for this user' };
    }

    // Save notification to user's database first
    const saveResult = await saveNotificationToUser(userRecord.uid, title, body, data);
    if (!saveResult.success) {
      console.warn('⚠️ Failed to save notification to database, but continuing with FCM send');
    }

    // Send notification to all active devices
    const results = [];
    let successCount = 0;
    let failureCount = 0;
    let invalidTokenCount = 0;

    for (const [deviceKey, deviceData] of Object.entries(devices)) {
      if (deviceData.token && deviceData.fcmStatus === 'active') {
        console.log(`📱 Sending to device: ${deviceKey}`);
        try {
          const result = await sendPushNotificationToToken(deviceData.token, title, body, {
            ...data,
            deviceKey: deviceKey,
            deviceId: deviceData.deviceInfo?.deviceId || 'unknown',
            targetDevice: deviceKey,
            userName: userData.name,
            userRole: userData.role,
          });
          
          if (result.success) {
            successCount++;
            results.push({ deviceKey, status: 'success', messageId: result.messageId });
            console.log(`✅ Successfully sent to device: ${deviceKey}`);
          } else {
            failureCount++;
            results.push({ deviceKey, status: 'failed', error: result.error });
            console.log(`❌ Failed to send to device: ${deviceKey}:`, result.error);
            
            // If token is invalid, mark device as inactive
            if (result.code === 'messaging/invalid-registration-token' || 
                result.code === 'messaging/registration-token-not-registered') {
              await markDeviceAsInactive(userRecord.uid, deviceKey);
              invalidTokenCount++;
            }
          }
        } catch (error) {
          failureCount++;
          results.push({ deviceKey, status: 'error', error: error.message });
          console.log(`❌ Error sending to device: ${deviceKey}:`, error.message);
        }
      } else {
        console.log(`⚠️ Skipping device ${deviceKey}: token=${!!deviceData.token}, status=${deviceData.fcmStatus}`);
      }
    }

    console.log(`📊 Notification results: ${successCount} success, ${failureCount} failed, ${invalidTokenCount} invalid tokens`);

    return { 
      success: successCount > 0, 
      message: `Sent to ${successCount} devices, ${failureCount} failed, ${invalidTokenCount} invalid tokens`,
      results: results,
      totalDevices: Object.keys(devices).length,
      successfulDevices: successCount,
      failedDevices: failureCount,
      invalidTokens: invalidTokenCount,
      notificationSaved: saveResult.success
    };
  } catch (error) {
    console.error('❌ Error sending push notification by email:', error);
    return { success: false, error: error.message };
  }
}

// Function to mark a device as inactive due to invalid token
async function markDeviceAsInactive(userId, deviceKey) {
  try {
    await admin.database().ref(`/Globalusers/admins/${userId}/devices/${deviceKey}/fcmStatus`).set('inactive');
    await admin.database().ref(`/Globalusers/admins/${userId}/devices/${deviceKey}/lastTokenError`).set({
      error: 'Invalid FCM token',
      timestamp: new Date().toISOString()
    });
    console.log(`✅ Marked device ${deviceKey} as inactive for user ${userId}`);
  } catch (error) {
    console.error(`❌ Error marking device ${deviceKey} as inactive:`, error);
  }
}

// Function to mark legacy token as inactive
async function markLegacyTokenAsInactive(userId) {
  try {
    await admin.database().ref(`/Globalusers/admins/${userId}/fcmStatus`).set('inactive');
    await admin.database().ref(`/Globalusers/admins/${userId}/lastTokenError`).set({
      error: 'Invalid legacy FCM token',
      timestamp: new Date().toISOString()
    });
    console.log(`✅ Marked legacy FCM token as inactive for user ${userId}`);
  } catch (error) {
    console.error(`❌ Error marking legacy FCM token as inactive:`, error);
  }
}

// Function to get FCM token by email
async function getFCMTokenByEmail(email) {
  try {
    console.log('🔍 Getting FCM token for email:', email);
    
    // Get user by email from Firebase Auth
    const userRecord = await admin.auth().getUserByEmail(email);
    
    if (!userRecord) {
      console.log('❌ User not found in Firebase Auth:', email);
      return { success: false, error: 'User not found with this email' };
    }

    console.log('✅ User found in Firebase Auth:', userRecord.uid);

    // Get FCM tokens from Realtime Database (matching AdminLoginScreen storage)
    const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userRecord.uid}`).once('value');
    
    if (!userSnapshot.exists()) {
      console.log('❌ User not found in admin database:', userRecord.uid);
      return { success: false, error: 'User not found in admin database' };
    }

    const userData = userSnapshot.val();
    console.log('📋 User data retrieved:', Object.keys(userData));
    
    // Check for devices structure (new storage method from AdminLoginScreen)
    const devices = userData.devices || {};
    console.log('📱 Devices found:', Object.keys(devices));
    
    if (Object.keys(devices).length === 0) {
      // Fallback: check for old NotificationFCMtoken field
      if (userData.NotificationFCMtoken) {
        console.log('📱 Found old FCM token format');
        return { 
          success: true, 
          userId: userRecord.uid,
          fcmToken: userData.NotificationFCMtoken,
          totalDevices: 1,
          userData: {
            name: userData.name,
            email: userData.email,
            role: userData.role,
            employeeNumber: userData.employeeNumber
          }
        };
      }
      
      console.log('❌ No FCM tokens found for user');
      return { success: false, error: 'No FCM tokens found for this user' };
    }

    // Get the first active FCM token from devices
    let activeToken = null;
    let deviceKey = null;
    let deviceInfo = null;
    
    for (const [key, device] of Object.entries(devices)) {
      if (device.token && device.fcmStatus === 'active') {
        activeToken = device.token;
        deviceKey = key;
        deviceInfo = device;
        break;
      }
    }
    
    if (!activeToken) {
      console.log('❌ No active FCM tokens found in devices');
      return { success: false, error: 'No active FCM tokens found for this user' };
    }

    console.log('✅ Active FCM token found:', deviceKey);

    // Return all devices with their tokens
    const deviceTokens = [];
    for (const [deviceKey, deviceData] of Object.entries(devices)) {
      if (deviceData.token && deviceData.fcmStatus === 'active') {
        deviceTokens.push({
          deviceKey: deviceKey,
          fcmToken: deviceData.token,
          deviceId: deviceData.deviceInfo?.deviceId || 'unknown',
          platform: deviceData.deviceInfo?.platform || 'unknown',
          lastTokenUpdate: deviceData.lastTokenUpdate || 'unknown',
          isCurrentDevice: deviceData.isCurrentDevice || false,
          fcmStatus: deviceData.fcmStatus || 'unknown'
        });
      }
    }

    return { 
      success: true, 
      userId: userRecord.uid,
      fcmToken: activeToken, // Return the first active token for backward compatibility
      deviceKey: deviceKey,
      deviceInfo: deviceInfo,
      userData: {
        name: userData.name,
        email: userData.email,
        role: userData.role,
        employeeNumber: userData.employeeNumber
      },
      devices: deviceTokens,
      totalDevices: deviceTokens.length
    };
  } catch (error) {
    console.error('❌ Error getting FCM token by email:', error);
    return { success: false, error: error.message };
  }
}

// Function to save notification to user's database
async function saveNotificationToUser(userId, title, body, data = {}) {
  try {
    const notificationRef = admin.database().ref(`/Globalusers/admins/${userId}/notifications`);
    const newNotificationRef = notificationRef.push();
    
    const notificationData = {
      title: title,
      body: body,
      data: data,
      timestamp: Date.now(),
      isRead: false,
      createdAt: new Date().toISOString(),
    };
    
    await newNotificationRef.set(notificationData);
    
    console.log(`✅ Notification saved to user ${userId} database`);
    return { success: true, notificationId: newNotificationRef.key };
  } catch (error) {
    console.error('Error saving notification to user database:', error);
    return { success: false, error: error.message };
  }
}

// Function to send notification to all devices of a user
async function sendNotificationToAllUserDevices(userId, title, body, data = {}) {
  try {
    console.log('📱 Sending notification to all devices for user:', userId);
    
    const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userId}`).once('value');
    
    if (!userSnapshot.exists()) {
      console.log('❌ User not found in admin database:', userId);
      return { success: false, error: 'User not found in admin database' };
    }

    const userData = userSnapshot.val();
    console.log('📋 User data retrieved:', Object.keys(userData));
    
    // Check for devices structure (new storage method from AdminLoginScreen)
    const devices = userData.devices || {};
    console.log('📱 Devices found:', Object.keys(devices));
    
    if (Object.keys(devices).length === 0) {
      // Fallback: check for old NotificationFCMtoken field
      if (userData.NotificationFCMtoken) {
        console.log('📱 Using old FCM token format');
        const result = await sendPushNotificationToToken(userData.NotificationFCMtoken, title, body, {
          ...data,
          deviceKey: 'legacy',
          deviceId: 'legacy',
          targetDevice: 'legacy',
          userName: userData.name,
          userRole: userData.role,
        });
        
        if (result.success) {
          // Save notification to user's database
          const saveResult = await saveNotificationToUser(userId, title, body, data);
          
          return { 
            success: true, 
            message: 'Sent to 1 legacy device',
            results: [{ deviceKey: 'legacy', status: 'success', messageId: result.messageId }],
            totalDevices: 1,
            successfulDevices: 1,
            failedDevices: 0,
            notificationSaved: saveResult.success
          };
        } else {
          // If token is invalid, mark it as inactive
          if (result.code === 'messaging/invalid-registration-token' || 
              result.code === 'messaging/registration-token-not-registered') {
            await markLegacyTokenAsInactive(userId);
          }
          return { success: false, error: result.error };
        }
      }
      
      console.log('❌ No FCM tokens found for user');
      return { success: false, error: 'No FCM tokens found for this user' };
    }

    // Save notification to user's database first
    const saveResult = await saveNotificationToUser(userId, title, body, data);
    if (!saveResult.success) {
      console.warn('⚠️ Failed to save notification to database, but continuing with FCM send');
    }

    // Send notification to all active devices
    const results = [];
    let successCount = 0;
    let failureCount = 0;
    let invalidTokenCount = 0;

    for (const [deviceKey, deviceData] of Object.entries(devices)) {
      if (deviceData.token && deviceData.fcmStatus === 'active') {
        console.log(`📱 Sending to device: ${deviceKey}`);
        try {
          const result = await sendPushNotificationToToken(deviceData.token, title, body, {
            ...data,
            deviceKey: deviceKey,
            deviceId: deviceData.deviceInfo?.deviceId || 'unknown',
            targetDevice: deviceKey,
            userName: userData.name,
            userRole: userData.role,
          });
          
          if (result.success) {
            successCount++;
            results.push({ deviceKey, status: 'success', messageId: result.messageId });
            console.log(`✅ Successfully sent to device: ${deviceKey}`);
          } else {
            failureCount++;
            results.push({ deviceKey, status: 'failed', error: result.error });
            console.log(`❌ Failed to send to device: ${deviceKey}:`, result.error);
            
            // If token is invalid, mark device as inactive
            if (result.code === 'messaging/invalid-registration-token' || 
                result.code === 'messaging/registration-token-not-registered') {
              await markDeviceAsInactive(userId, deviceKey);
              invalidTokenCount++;
            }
          }
        } catch (error) {
          failureCount++;
          results.push({ deviceKey, status: 'error', error: error.message });
          console.log(`❌ Error sending to device: ${deviceKey}:`, error.message);
        }
      } else {
        console.log(`⚠️ Skipping device ${deviceKey}: token=${!!deviceData.token}, status=${deviceData.fcmStatus}`);
      }
    }

    console.log(`📊 Notification results: ${successCount} success, ${failureCount} failed, ${invalidTokenCount} invalid tokens`);

    return { 
      success: successCount > 0, 
      message: `Sent to ${successCount} devices, ${failureCount} failed, ${invalidTokenCount} invalid tokens`,
      results: results,
      totalDevices: Object.keys(devices).length,
      successfulDevices: successCount,
      failedDevices: failureCount,
      invalidTokens: invalidTokenCount,
      notificationSaved: saveResult.success
    };
  } catch (error) {
    console.error('❌ Error sending notification to all user devices:', error);
    return { success: false, error: error.message };
  }
}

// Function to get all users with FCM tokens from Realtime Database
async function getAllUsersWithFCMTokens() {
  try {
    const adminsSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    
    if (!adminsSnapshot.exists()) {
      return { success: false, error: 'No admin users found' };
    }

    const admins = adminsSnapshot.val();
    const usersWithTokens = [];

    for (const [userId, userData] of Object.entries(admins)) {
      const devices = userData.devices || {};
      const activeDevices = Object.entries(devices).filter(([_, device]) => 
        device.token && device.fcmStatus === 'active'
      );
      
      if (activeDevices.length > 0) {
        usersWithTokens.push({
          userId,
          name: userData.name || 'Unknown',
          email: userData.email || 'No email',
          role: userData.role || 'Unknown',
          totalDevices: activeDevices.length,
          devices: activeDevices.map(([deviceKey, device]) => ({
            deviceKey,
            fcmToken: device.token,
            deviceId: device.deviceInfo?.deviceId || 'unknown',
            platform: device.deviceInfo?.platform || 'unknown',
            lastTokenUpdate: device.lastTokenUpdate || 'Unknown',
            isCurrentDevice: device.isCurrentDevice || false
          }))
        });
      }
    }

    return { 
      success: true, 
      users: usersWithTokens,
      totalUsers: usersWithTokens.length
    };
  } catch (error) {
    console.error('Error getting all users with FCM tokens:', error);
    return { success: false, error: error.message };
  }
}

// Function to send high-priority push notification (for important messages)
async function sendHighPriorityNotification(token, title, body, data = {}) {
  try {
    // Convert all data values to strings as required by FCM
    const stringData = {};
    Object.keys(data).forEach(key => {
      stringData[key] = String(data[key]);
    });

    const message = {
      token: token,
      notification: {
        title: title,
        body: body,
      },
      data: {
        ...stringData,
        title: title,
        body: body,
        priority: 'high',
        importance: 'high',

      },
      android: {
        priority: 'high',
        notification: {
          title: title,
          body: body,
          sound: 'default',
          channel_id: 'high_priority',
  
          icon: 'ic_notification',
          color: '#015185',
        },
        data: {
          ...stringData,
          priority: 'high',
          importance: 'high',
  
        },
      },
      apns: {
        payload: {
          aps: {
            alert: {
              title: title,
              body: body,
            },
            sound: 'default',
            badge: 1,
            'content-available': 1,
            'mutable-content': 1,
            category: 'HIGH_PRIORITY',
          },
        },
        headers: {
          'apns-priority': '10',
          'apns-push-type': 'alert',
        },
      },
    };

    console.log('📱 Sending high-priority FCM message to token:', token.substring(0, 20) + '...');
    const response = await admin.messaging().send(message);
    console.log('✅ Successfully sent high-priority message:', response);
    return { success: true, messageId: response };
  } catch (error) {
    console.error('❌ Error sending high-priority message:', error);
    return { success: false, error: error.message };
  }
}

// Function to validate FCM token before sending
async function validateFCMToken(token) {
  try {
    // Send a silent message to validate the token
    const message = {
      token: token,
      data: {
        type: 'token_validation',
        timestamp: Date.now().toString(),
      },
      android: {
        priority: 'normal',
        data: {
          type: 'token_validation',
          timestamp: Date.now().toString(),
        },
      },
      apns: {
        payload: {
          aps: {
            'content-available': 1,
            'mutable-content': 1,
          },
        },
      },
    };

    const response = await admin.messaging().send(message);
    return { valid: true, messageId: response };
  } catch (error) {
    if (error.code === 'messaging/invalid-registration-token' || 
        error.code === 'messaging/registration-token-not-registered') {
      return { valid: false, error: 'Invalid or expired FCM token' };
    }
    return { valid: false, error: error.message };
  }
}

// Enhanced function to send push notification with validation
async function sendPushNotificationWithValidation(token, title, body, data = {}) {
  try {
    // First validate the token
    const validation = await validateFCMToken(token);
    if (!validation.valid) {
      return { success: false, error: validation.error };
    }

    // Then send the actual notification
    return await sendPushNotificationToToken(token, title, body, data);
  } catch (error) {
    console.error('Error in sendPushNotificationWithValidation:', error);
    return { success: false, error: error.message };
  }
}

// Function to store FCM token for a user in Realtime Database
async function storeFCMToken(userId, fcmToken) {
  try {
    await admin.database().ref(`/Globalusers/admins/${userId}`).update({
      NotificationFCMtoken: fcmToken,
      lastTokenUpdate: new Date().toISOString(),
      deviceInfo: {
        platform: 'web', // or 'mobile' depending on context
        timestamp: new Date().toISOString(),
      },
    });
    
    return { success: true, message: 'FCM token stored successfully in Realtime Database' };
  } catch (error) {
    console.error('Error storing FCM token:', error);
    return { success: false, error: error.message };
  }
}

// Function to send test push notification with enhanced payload
async function sendTestPushNotification(token) {
  const testData = {
    type: 'test',
    timestamp: Date.now().toString(),
    message: 'This is a test push notification from Titan Drilling App',
    action: 'open_app',
    screen: 'home',
  };
  
  return await sendPushNotificationToToken(
    token, 
    '🧪 Test Notification', 
    'This is a test push notification from Titan Drilling App! Tap to open.', 
    testData
  );
}

// Function to debug FCM token storage for a user
async function debugFCMTokenStorage(userId) {
  try {
    console.log('🔍 Debugging FCM token storage for user:', userId);
    
    const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userId}`).once('value');
    
    if (!userSnapshot.exists()) {
      return { success: false, error: 'User not found in admin database' };
    }

    const userData = userSnapshot.val();
    const debugInfo = {
      userId: userId,
      hasDevices: false,
      devicesCount: 0,
      activeDevicesCount: 0,
      hasLegacyToken: false,
      legacyToken: null,
      devices: {},
      deviceTokens: []
    };

    // Check for legacy token
    if (userData.NotificationFCMtoken) {
      debugInfo.hasLegacyToken = true;
      debugInfo.legacyToken = userData.NotificationFCMtoken;
    }

    // Check for new devices structure
    const devices = userData.devices || {};
    if (Object.keys(devices).length > 0) {
      debugInfo.hasDevices = true;
      debugInfo.devicesCount = Object.keys(devices).length;
      
      for (const [deviceKey, deviceData] of Object.entries(devices)) {
        const deviceInfo = {
          deviceKey: deviceKey,
          hasToken: !!deviceData.token,
          tokenLength: deviceData.token ? deviceData.token.length : 0,
          fcmStatus: deviceData.fcmStatus || 'unknown',
          isActive: deviceData.fcmStatus === 'active',
          isCurrentDevice: deviceData.isCurrentDevice || false,
          platform: deviceData.deviceInfo?.platform || 'unknown',
          deviceId: deviceData.deviceInfo?.deviceId || 'unknown',
          lastTokenUpdate: deviceData.lastTokenUpdate || 'unknown',
          lastLogin: deviceData.lastLogin || 'unknown'
        };
        
        debugInfo.devices[deviceKey] = deviceInfo;
        
        if (deviceData.token && deviceData.fcmStatus === 'active') {
          debugInfo.activeDevicesCount++;
          debugInfo.deviceTokens.push({
            deviceKey: deviceKey,
            token: deviceData.token.substring(0, 20) + '...',
            fullToken: deviceData.token
          });
        }
      }
    }

    return { success: true, debugInfo: debugInfo };
  } catch (error) {
    console.error('❌ Error debugging FCM token storage:', error);
    return { success: false, error: error.message };
  }
}

// Function to refresh and validate FCM tokens for a user
async function refreshAndValidateFCMTokens(userId) {
  try {
    console.log('🔄 Refreshing and validating FCM tokens for user:', userId);
    
    const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userId}`).once('value');
    
    if (!userSnapshot.exists()) {
      return { success: false, error: 'User not found in admin database' };
    }

    const userData = userSnapshot.val();
    const devices = userData.devices || {};
    const results = [];
    let validTokens = 0;
    let invalidTokens = 0;
    let updatedTokens = 0;

    // Validate each device's FCM token
    for (const [deviceKey, deviceData] of Object.entries(devices)) {
      if (deviceData.token && deviceData.fcmStatus === 'active') {
        try {
          const validation = await validateFCMToken(deviceData.token);
          
          if (validation.valid) {
            validTokens++;
            results.push({ deviceKey, status: 'valid', messageId: validation.messageId });
          } else {
            invalidTokens++;
            results.push({ deviceKey, status: 'invalid', error: validation.error });
            
            // Mark device as inactive
            await markDeviceAsInactive(userId, deviceKey);
          }
        } catch (error) {
          invalidTokens++;
          results.push({ deviceKey, status: 'error', error: error.message });
          
          // Mark device as inactive
          await markDeviceAsInactive(userId, deviceKey);
        }
      }
    }

    // Check legacy token if exists
    if (userData.NotificationFCMtoken) {
      try {
        const validation = await validateFCMToken(userData.NotificationFCMtoken);
        
        if (validation.valid) {
          validTokens++;
          results.push({ deviceKey: 'legacy', status: 'valid', messageId: validation.messageId });
        } else {
          invalidTokens++;
          results.push({ deviceKey: 'legacy', status: 'invalid', error: validation.error });
          
          // Mark legacy token as inactive
          await markLegacyTokenAsInactive(userId);
        }
      } catch (error) {
        invalidTokens++;
        results.push({ deviceKey: 'legacy', status: 'error', error: error.message });
        
        // Mark legacy token as inactive
        await markLegacyTokenAsInactive(userId);
      }
    }

    return {
      success: true,
      message: `Token validation completed: ${validTokens} valid, ${invalidTokens} invalid`,
      validTokens,
      invalidTokens,
      updatedTokens,
      results
    };

  } catch (error) {
    console.error('❌ Error refreshing and validating FCM tokens:', error);
    return { success: false, error: error.message };
  }
}

// Function to clean up invalid FCM tokens across all users
async function cleanupInvalidFCMTokens() {
  try {
    console.log('🧹 Starting cleanup of invalid FCM tokens...');
    
    const adminsSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    
    if (!adminsSnapshot.exists()) {
      return { success: false, error: 'No admin users found' };
    }

    const admins = adminsSnapshot.val();
    const cleanupResults = [];
    let totalUsersProcessed = 0;
    let totalDevicesCleaned = 0;
    let totalLegacyTokensCleaned = 0;

    for (const [userId, userData] of Object.entries(admins)) {
      try {
        totalUsersProcessed++;
        const userResult = await refreshAndValidateFCMTokens(userId);
        
        if (userResult.success) {
          const devices = userData.devices || {};
          const deviceCount = Object.keys(devices).length;
          const legacyToken = userData.NotificationFCMtoken ? 1 : 0;
          
          cleanupResults.push({
            userId,
            userName: userData.name || 'Unknown',
            status: 'success',
            validTokens: userResult.validTokens,
            invalidTokens: userResult.invalidTokens,
            deviceCount,
            legacyToken
          });
          
          totalDevicesCleaned += userResult.invalidTokens;
          if (userData.NotificationFCMtoken && userResult.results.find(r => r.deviceKey === 'legacy' && r.status === 'invalid')) {
            totalLegacyTokensCleaned++;
          }
        } else {
          cleanupResults.push({
            userId,
            userName: userData.name || 'Unknown',
            status: 'failed',
            error: userResult.error
          });
        }
      } catch (error) {
        cleanupResults.push({
          userId,
          userName: userData.name || 'Unknown',
          status: 'error',
          error: error.message
        });
      }
    }

    return {
      success: true,
      message: `Cleanup completed: ${totalUsersProcessed} users processed, ${totalDevicesCleaned} invalid devices cleaned, ${totalLegacyTokensCleaned} legacy tokens cleaned`,
      totalUsersProcessed,
      totalDevicesCleaned,
      totalLegacyTokensCleaned,
      results: cleanupResults
    };

  } catch (error) {
    console.error('❌ Error cleaning up invalid FCM tokens:', error);
    return { success: false, error: error.message };
  }
}

// Function to get FCM token statistics for all users
async function getFCMTokenStatistics() {
  try {
    console.log('📊 Getting FCM token statistics for all users...');
    
    const adminsSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    
    if (!adminsSnapshot.exists()) {
      return { success: false, error: 'No admin users found' };
    }

    const admins = adminsSnapshot.val();
    const stats = {
      totalUsers: 0,
      usersWithActiveTokens: 0,
      usersWithInactiveTokens: 0,
      usersWithNoTokens: 0,
      totalDevices: 0,
      activeDevices: 0,
      inactiveDevices: 0,
      totalLegacyTokens: 0,
      activeLegacyTokens: 0,
      inactiveLegacyTokens: 0,
      platformBreakdown: {
        ios: 0,
        android: 0,
        legacy: 0,
        unknown: 0
      },
      userDetails: []
    };

    for (const [userId, userData] of Object.entries(admins)) {
      stats.totalUsers++;
      
      const userDetail = {
        userId,
        userName: userData.name || 'Unknown',
        hasDevices: false,
        hasLegacyToken: false,
        deviceCount: 0,
        activeDeviceCount: 0,
        inactiveDeviceCount: 0,
        legacyTokenStatus: 'none'
      };

      // Check multi-device structure
      const devices = userData.devices || {};
      if (Object.keys(devices).length > 0) {
        userDetail.hasDevices = true;
        userDetail.deviceCount = Object.keys(devices).length;
        stats.totalDevices += userDetail.deviceCount;
        
        for (const [deviceKey, deviceData] of Object.entries(devices)) {
          if (deviceData.fcmStatus === 'active') {
            userDetail.activeDeviceCount++;
            stats.activeDevices++;
            
            // Platform breakdown
            const platform = deviceData.deviceInfo?.platform || 'unknown';
            if (platform === 'ios') stats.platformBreakdown.ios++;
            else if (platform === 'android') stats.platformBreakdown.android++;
            else stats.platformBreakdown.unknown++;
          } else {
            userDetail.inactiveDeviceCount++;
            stats.inactiveDevices++;
          }
        }
        
        if (userDetail.activeDeviceCount > 0) {
          stats.usersWithActiveTokens++;
        } else {
          stats.usersWithInactiveTokens++;
        }
      }

      // Check legacy token structure
      if (userData.NotificationFCMtoken) {
        userDetail.hasLegacyToken = true;
        stats.totalLegacyTokens++;
        
        // Try to determine if legacy token is still active
        try {
          const validation = await validateFCMToken(userData.NotificationFCMtoken);
          if (validation.valid) {
            userDetail.legacyTokenStatus = 'active';
            stats.activeLegacyTokens++;
            stats.platformBreakdown.legacy++;
          } else {
            userDetail.legacyTokenStatus = 'inactive';
            stats.inactiveLegacyTokens++;
          }
        } catch (error) {
          userDetail.legacyTokenStatus = 'error';
          stats.inactiveLegacyTokens++;
        }
      }

      if (!userDetail.hasDevices && !userDetail.hasLegacyToken) {
        stats.usersWithNoTokens++;
      }

      stats.userDetails.push(userDetail);
    }

    return {
      success: true,
      stats: stats
    };

  } catch (error) {
    console.error('❌ Error getting FCM token statistics:', error);
    return { success: false, error: error.message };
  }
}

module.exports = {
  sendPushNotificationToToken,
  sendPushNotificationByEmail,
  getFCMTokenByEmail,
  storeFCMToken,
  sendTestPushNotification,
  sendNotificationToAllUserDevices,
  getAllUsersWithFCMTokens,
  sendHighPriorityNotification,
  validateFCMToken,
  sendPushNotificationWithValidation,
  debugFCMTokenStorage,
  refreshAndValidateFCMTokens,
  cleanupInvalidFCMTokens,
  getFCMTokenStatistics
};

