const functions = require('firebase-functions/v1');
const admin = require('firebase-admin');
const nodemailer = require('nodemailer');

// Initialize Firebase Admin if not already initialized
if (!admin.apps.length) {
  admin.initializeApp();
}

const db = admin.database();

// Email configuration - Using Gmail SMTP server
const emailConfig = {
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'mandeep.owninfotech@gmail.com',
    pass: 'aeqa purb mxtl hkhh',
  },
};

// Create transporter
const transporter = nodemailer.createTransport(emailConfig);

/**
 * HR Application Approval/Rejection Notification Function
 * Handles both email and push notifications when HR applications are approved/rejected
 * Triggers when HR application status is updated at /GlobalHrApplications/admins/{applicationType}/{userId}/{applicationId}
 */
exports.ApproveApplicationSentNotify = functions.database
  .ref('/GlobalHrApplications/admins/{applicationType}/{userId}/{applicationId}')
  .onUpdate(async (change, context) => {
    try {
      console.log('🏢 HR APPLICATION APPROVAL/REJECTION NOTIFICATION TRIGGERED! 🏢');
      console.log('Path:', context.params);
      console.log('Function version: v1.0 - Approval/Rejection notifications');
      
      const beforeData = change.before.val();
      const afterData = change.after.val();
      const { applicationType, userId, applicationId } = context.params;

      console.log('HR Application updated:', { applicationType, userId, applicationId });

      if (!afterData) {
        console.log('No HR application data found after update');
        return null;
      }

      // Check if this is an approval/rejection update
      const isApprovalUpdate = checkIfApprovalUpdate(beforeData, afterData);
      
      if (!isApprovalUpdate.isUpdate) {
        console.log('Not an approval/rejection update, skipping notification');
        return null;
      }

      console.log('Approval/Rejection detected:', isApprovalUpdate);

      // Get the HR application details
      const {
        fullName,
        employeeNo,
        country,
        project,
        department,
        nrcNo,
        submittedAt,
        status,
        userUid,
        // Application type specific fields
        leaveStartDate,
        leaveEndDate,
        reasonOfLeave,
        salaryAdvanceAmount,
        deceasedName,
        appealGrounds,
        grievanceDetails,
        selectedClinics
      } = afterData;
      
      if (!fullName || !employeeNo) {
        console.log('Missing required HR application data');
        return null;
      }

      // Get the application creator's information
      const applicationCreator = await getApplicationCreator(userId);
      
      if (!applicationCreator) {
        console.log('Application creator not found, skipping notification');
        return null;
      }

      // Prepare email content for the application creator
      const creatorEmailContent = prepareApprovalRejectionEmailContent({
        applicationData: afterData,
        applicationType,
        fullName,
        employeeNo,
        country,
        project,
        department,
        nrcNo,
        submittedAt,
        status,
        applicationCreator,
        approvalUpdate: isApprovalUpdate
      });

      // Send email to application creator
      console.log(`📧 Sending ${isApprovalUpdate.action} notification email to application creator: ${applicationCreator.email}`);
      const creatorResult = await sendBulkEmails(
        [applicationCreator], 
        creatorEmailContent, 
        `HR Application ${isApprovalUpdate.action} - ${formatApplicationType(applicationType)}`
      );
      console.log('📧 Application creator email result:', creatorResult);

      // Special handling for salary advance applications - notify Finance and HR Managers after supervisor approval
      if (applicationType === 'salary_advance_application' && 
          isApprovalUpdate.approverRole === 'Supervisor' && 
          isApprovalUpdate.action === 'Approved') {
        
        console.log('💰 Salary Advance - Supervisor approved, notifying Finance and HR Managers...');
        
        // Get Finance Managers matching the application's country and project
        const financeManagersToNotify = await getFinanceManagersByCountryProject(country, project);
        
        // Get HR Managers matching the application's country and project
        const hrManagersToNotify = await getHrManagersByCountryProject(country, project);
        
        // Prepare email content for Finance and HR Managers
        const managerEmailContent = prepareManagerNotificationEmailContent({
          applicationData: afterData,
          applicationType,
          fullName,
          employeeNo,
          country,
          project,
          department,
          nrcNo,
          submittedAt,
          status,
          applicationCreator,
          approvalUpdate: isApprovalUpdate
        });

        // Send emails to Finance Managers
        if (financeManagersToNotify.length > 0) {
          console.log(`📧 Sending notification to ${financeManagersToNotify.length} Finance Managers...`);
          const financeResult = await sendBulkEmails(
            financeManagersToNotify, 
            managerEmailContent, 
            `Salary Advance - Supervisor Approved: ${formatApplicationType(applicationType)}`
          );
          console.log('📧 Finance Manager email result:', financeResult);

          // Send FCM push notifications to Finance Managers
          console.log('🔔 Starting FCM push notifications to Finance Managers...');
          await sendFCMNotificationsToManagers(afterData, financeManagersToNotify, 'FinanceManager', {
            applicationType,
            fullName,
            employeeNo,
            country,
            project,
            department,
            status,
            approvalUpdate: isApprovalUpdate
          }, applicationId);

          // Save notifications to Finance Managers
          console.log('📱 Saving notifications to Finance Managers...');
          await saveNotificationsToManagers(afterData, financeManagersToNotify, 'FinanceManager', {
            applicationType,
            fullName,
            employeeNo,
            country,
            project,
            department,
            status,
            approvalUpdate: isApprovalUpdate
          }, applicationId);
        } else {
          console.log('⚠️ No Finance Managers found matching the application country + project');
        }

        // Send emails to HR Managers
        if (hrManagersToNotify.length > 0) {
          console.log(`📧 Sending notification to ${hrManagersToNotify.length} HR Managers...`);
          const hrResult = await sendBulkEmails(
            hrManagersToNotify, 
            managerEmailContent, 
            `Salary Advance - Supervisor Approved: ${formatApplicationType(applicationType)}`
          );
          console.log('📧 HR Manager email result:', hrResult);

          // Send FCM push notifications to HR Managers
          console.log('🔔 Starting FCM push notifications to HR Managers...');
          await sendFCMNotificationsToManagers(afterData, hrManagersToNotify, 'HrManager', {
            applicationType,
            fullName,
            employeeNo,
            country,
            project,
            department,
            status,
            approvalUpdate: isApprovalUpdate
          }, applicationId);

          // Save notifications to HR Managers
          console.log('📱 Saving notifications to HR Managers...');
          await saveNotificationsToManagers(afterData, hrManagersToNotify, 'HrManager', {
            applicationType,
            fullName,
            employeeNo,
            country,
            project,
            department,
            status,
            approvalUpdate: isApprovalUpdate
          }, applicationId);
        } else {
          console.log('⚠️ No HR Managers found matching the application country + project');
        }
      }

      // 🔔 SEND FCM PUSH NOTIFICATION TO APPLICATION CREATOR
      console.log('🔔 Starting FCM push notification to application creator...');
      await sendFCMNotificationToCreator(afterData, applicationCreator, {
        applicationType,
        fullName,
        employeeNo,
        country,
        project,
        department,
        status,
        approvalUpdate: isApprovalUpdate
      }, applicationId);

      // 📱 SAVE NOTIFICATION TO APPLICATION CREATOR
      console.log('📱 Saving notification to application creator...');
      await saveNotificationToCreator(afterData, applicationCreator, {
        applicationType,
        fullName,
        employeeNo,
        country,
        project,
        department,
        status,
        approvalUpdate: isApprovalUpdate
      }, applicationId);

      console.log('✅ HR Application approval/rejection notifications sent successfully');
      return null;

    } catch (error) {
      console.error('Error in ApproveApplicationSentNotify:', error);
      return null;
    }
  });

/**
 * Check if the update is an approval/rejection
 */
function checkIfApprovalUpdate(beforeData, afterData) {
  if (!beforeData || !afterData) {
    return { isUpdate: false };
  }

  // Check Supervisor status change
  if (beforeData.Supervisor?.status !== afterData.Supervisor?.status) {
    const newStatus = afterData.Supervisor?.status;
    if (newStatus === 'Approved' || newStatus === 'Rejected') {
      return {
        isUpdate: true,
        action: newStatus === 'Approved' ? 'Approved' : 'Rejected',
        approverRole: 'Supervisor',
        approverName: afterData.Supervisor?.name || 'Supervisor',
        reason: afterData.Supervisor?.reason || null,
        timestamp: afterData.Supervisor?.timestamp || new Date().toISOString()
      };
    }
  }

  // Check HR Manager status change
  if (beforeData.HrManager?.status !== afterData.HrManager?.status) {
    const newStatus = afterData.HrManager?.status;
    if (newStatus === 'Approved' || newStatus === 'Rejected') {
      return {
        isUpdate: true,
        action: newStatus === 'Approved' ? 'Approved' : 'Rejected',
        approverRole: 'HR Manager',
        approverName: afterData.HrManager?.name || 'HR Manager',
        reason: afterData.HrManager?.reason || null,
        timestamp: afterData.HrManager?.timestamp || new Date().toISOString()
      };
    }
  }

  // Check Finance Manager status change
  if (beforeData.FinanceManager?.status !== afterData.FinanceManager?.status) {
    const newStatus = afterData.FinanceManager?.status;
    if (newStatus === 'Approved' || newStatus === 'Rejected') {
      return {
        isUpdate: true,
        action: newStatus === 'Approved' ? 'Approved' : 'Rejected',
        approverRole: 'Finance Manager',
        approverName: afterData.FinanceManager?.name || 'Finance Manager',
        reason: afterData.FinanceManager?.reason || null,
        timestamp: afterData.FinanceManager?.timestamp || new Date().toISOString()
      };
    }
  }

  // Check overall status change
  if (beforeData.status !== afterData.status && 
      (afterData.status === 'Approved' || afterData.status === 'Rejected')) {
    return {
      isUpdate: true,
      action: afterData.status,
      approverRole: 'System',
      approverName: 'System',
      reason: null,
      timestamp: new Date().toISOString()
    };
  }

  return { isUpdate: false };
}

/**
 * Get HR application creator's information
 */
async function getApplicationCreator(userId) {
  try {
    console.log('Getting HR application creator for userId:', userId);
    
    const userSnapshot = await db.ref(`/Globalusers/admins/${userId}`).once('value');
    const userData = userSnapshot.val();
    
    if (!userData || !userData.email) {
      console.log('No user data or email found for userId:', userId);
      return null;
    }
    
    const applicationCreator = {
      email: userData.email,
      name: userData.name || userData.fullName || 'Application Creator',
      role: userData.role || 'User',
      userId: userId
    };
    
    console.log('Found HR application creator:', applicationCreator);
    return applicationCreator;
    
  } catch (error) {
    console.error('Error getting HR application creator:', error);
    return null;
  }
}

/**
 * Prepare approval/rejection email content
 */
function prepareApprovalRejectionEmailContent(data) {
  const {
    applicationData,
    applicationType,
    fullName,
    employeeNo,
    country,
    project,
    department,
    nrcNo,
    submittedAt,
    status,
    applicationCreator,
    approvalUpdate
  } = data;

  // Format application type for display
  const applicationTypeDisplay = formatApplicationType(applicationType);
  
  const emailSubject = `HR Application ${approvalUpdate.action}: ${applicationTypeDisplay}`;
  
  const isApproved = approvalUpdate.action === 'Approved';
  const statusColor = isApproved ? '#28a745' : '#dc3545';
  const statusBgColor = isApproved ? '#d4edda' : '#f8d7da';
  const statusBorderColor = isApproved ? '#c3e6cb' : '#f5c6cb';
  
  const emailBody = `
    <html>
      <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333;">
        <div style="max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="background-color: #015185; color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0;">
            <h1 style="margin: 0; font-size: 24px;">HR Application ${approvalUpdate.action}</h1>
            <p style="margin: 10px 0 0 0; font-size: 16px;">${applicationTypeDisplay}</p>
          </div>
          
          <div style="background-color: #f8f9fa; padding: 20px; border: 1px solid #dee2e6;">
            <div style="background-color: ${statusBgColor}; padding: 15px; border-left: 4px solid ${statusColor}; margin-bottom: 20px;">
              <h2 style="color: ${statusColor}; margin-top: 0;">
                ${isApproved ? '✅ Application Approved' : '❌ Application Rejected'}
              </h2>
              <p style="margin: 0;">
                <strong>Status:</strong> Your ${applicationTypeDisplay} has been ${approvalUpdate.action.toLowerCase()} by ${approvalUpdate.approverName}.
              </p>
              ${approvalUpdate.reason ? `<p style="margin: 10px 0 0 0;"><strong>Reason:</strong> ${approvalUpdate.reason}</p>` : ''}
            </div>
            
            <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold; width: 40%;">Application Type:</td>
                <td style="padding: 8px;">${applicationTypeDisplay}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Employee Name:</td>
                <td style="padding: 8px;">${fullName}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Employee Number:</td>
                <td style="padding: 8px;">${employeeNo}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Country:</td>
                <td style="padding: 8px;">${country || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Project:</td>
                <td style="padding: 8px;">${project || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Department:</td>
                <td style="padding: 8px;">${department || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">NRC Number:</td>
                <td style="padding: 8px;">${nrcNo || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Status:</td>
                <td style="padding: 8px;">
                  <span style="background-color: ${statusColor}; color: white; padding: 2px 8px; border-radius: 4px;">
                    ${status}
                  </span>
                </td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Submitted Date:</td>
                <td style="padding: 8px;">${new Date(submittedAt).toLocaleDateString('en-GB')} ${new Date(submittedAt).toLocaleTimeString()}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">${approvalUpdate.action} by:</td>
                <td style="padding: 8px;">${approvalUpdate.approverName}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">${approvalUpdate.action} Date:</td>
                <td style="padding: 8px;">${new Date(approvalUpdate.timestamp).toLocaleDateString('en-GB')} ${new Date(approvalUpdate.timestamp).toLocaleTimeString()}</td>
              </tr>
            </table>
            
            ${getApplicationSpecificDetails(applicationType, applicationData)}
            
            <div style="text-align: center; margin: 20px 0;">
              <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px;">
                <p style="margin: 0; font-size: 14px; color: #6c757d;">
                  <strong>${isApproved ? 'Next Steps:' : 'What This Means:'}</strong><br>
                  ${isApproved 
                    ? 'Your application has been approved and will be processed accordingly. Please contact HR if you have any questions.'
                    : 'Your application has been rejected. Please review the reason provided and contact your supervisor or HR if you need clarification or wish to appeal.'}
                </p>
              </div>
            </div>
            
            <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin-top: 20px;">
              <p style="margin: 0; font-size: 14px; color: #6c757d; text-align: center;">
                This is an automated notification from the Titan Drilling HR Management System.<br>
                Please do not reply to this email.
              </p>
            </div>
          </div>
        </div>
      </body>
    </html>
  `;

  return {
    subject: emailSubject,
    html: emailBody,
    text: convertHtmlToText(emailBody)
  };
}

/**
 * Format application type for display
 */
function formatApplicationType(applicationType) {
  const typeMap = {
    'leave_application': 'Leave Application',
    'funeral_grant_form': 'Funeral Grant Form',
    'salary_advance_application': 'Salary Advance Application',
    'application_for_appeal': 'Application for Appeal',
    'grievance_record': 'Grievance Record',
    'hospital_attendance_form': 'Hospital Attendance Form'
  };
  
  return typeMap[applicationType] || applicationType.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
}

/**
 * Get application specific details for email
 */
function getApplicationSpecificDetails(applicationType, applicationData) {
  switch (applicationType) {
    case 'leave_application':
      return `
        <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h3 style="color: #015185; margin-top: 0;">Leave Details</h3>
          <p><strong>Start Date:</strong> ${applicationData.leaveStartDate || 'N/A'}</p>
          <p><strong>End Date:</strong> ${applicationData.leaveEndDate || 'N/A'}</p>
          <p><strong>Report Date:</strong> ${applicationData.reportDate || 'N/A'}</p>
          <p><strong>Reason:</strong> ${applicationData.reasonOfLeave || 'N/A'}</p>
        </div>
      `;
    case 'salary_advance_application':
      return `
        <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h3 style="color: #015185; margin-top: 0;">Salary Advance Details</h3>
          <p><strong>Amount:</strong> K ${applicationData.salaryAdvanceAmount || 'N/A'}</p>
          <p><strong>Recovery Period:</strong> ${formatRecoveryPeriod(applicationData.selectedRecoveryPeriod) || 'N/A'}</p>
          <p><strong>Sites:</strong> ${formatSites(applicationData.selectedSites) || 'N/A'}</p>
          <p><strong>Reasons:</strong> ${formatReasons(applicationData.selectedReasons) || 'N/A'}</p>
        </div>
      `;
    case 'funeral_grant_form':
      return `
        <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h3 style="color: #015185; margin-top: 0;">Funeral Grant Details</h3>
          <p><strong>Deceased Name:</strong> ${applicationData.deceasedName || 'N/A'}</p>
          <p><strong>Relationship:</strong> ${applicationData.deceasedRelationship || 'N/A'}</p>
          <p><strong>Date of Death:</strong> ${applicationData.deceasedDate || 'N/A'}</p>
          <p><strong>From Date:</strong> ${applicationData.funeralFromDate || 'N/A'}</p>
          <p><strong>To Date:</strong> ${applicationData.funeralToDate || 'N/A'}</p>
        </div>
      `;
    case 'application_for_appeal':
      return `
        <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h3 style="color: #015185; margin-top: 0;">Appeal Details</h3>
          <p><strong>Sanction:</strong> ${applicationData.appealSanction || 'N/A'}</p>
          <p><strong>Offence:</strong> ${applicationData.appealOffence || 'N/A'}</p>
          <p><strong>Offence Date:</strong> ${applicationData.appealOffenceDate || 'N/A'}</p>
          <p><strong>Grounds for Appeal:</strong> ${applicationData.appealGrounds || 'N/A'}</p>
        </div>
      `;
    case 'grievance_record':
      return `
        <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h3 style="color: #015185; margin-top: 0;">Grievance Details</h3>
          <p><strong>Date:</strong> ${applicationData.grievanceDate || 'N/A'}</p>
          <p><strong>Place of Work:</strong> ${applicationData.grievancePlaceOfWork || 'N/A'}</p>
          <p><strong>Supervisor:</strong> ${applicationData.grievanceSupervisorName || 'N/A'}</p>
          <p><strong>Details:</strong> ${applicationData.grievanceDetails || 'N/A'}</p>
        </div>
      `;
    case 'hospital_attendance_form':
      return `
        <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h3 style="color: #015185; margin-top: 0;">Hospital Attendance Details</h3>
          <p><strong>Clinics:</strong> ${formatClinics(applicationData.selectedClinics) || 'N/A'}</p>
          <p><strong>Shift:</strong> ${applicationData.selectedShift || 'N/A'}</p>
          ${applicationData.otherClinic ? `<p><strong>Other Clinic:</strong> ${applicationData.otherClinic}</p>` : ''}
        </div>
      `;
    default:
      return '';
  }
}

/**
 * Helper functions for formatting data
 */
function formatRecoveryPeriod(period) {
  const periodMap = {
    '1_instalment': '1 Instalment',
    '2_instalments': '2 Instalments',
    '3_instalments': '3 Instalments'
  };
  return periodMap[period] || period;
}

function formatSites(sites) {
  if (!Array.isArray(sites)) return '';
  return sites.map(site => site.replace(/_/g, ' ').toUpperCase()).join(', ');
}

function formatReasons(reasons) {
  if (!Array.isArray(reasons)) return '';
  return reasons.map(reason => reason.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())).join(', ');
}

function formatClinics(clinics) {
  if (!Array.isArray(clinics)) return '';
  return clinics.map(clinic => clinic.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())).join(', ');
}

/**
 * Convert HTML to plain text
 */
function convertHtmlToText(html) {
  return html
    .replace(/<[^>]*>/g, '')
    .replace(/&nbsp;/g, ' ')
    .replace(/&amp;/g, '&')
    .replace(/&lt;/g, '<')
    .replace(/&gt;/g, '>')
    .replace(/&quot;/g, '"')
    .replace(/\s+/g, ' ')
    .trim();
}

/**
 * Send bulk emails
 */
async function sendBulkEmails(recipients, emailContent, emailType) {
  try {
    console.log(`🚀 Starting email queue for ${emailType}`);
    console.log(`📧 Total recipients: ${recipients.length}`);
    
    if (recipients.length === 0) {
      console.log('⚠️ No recipients to send emails to');
      return { success: true, sent: 0, failed: 0, results: [] };
    }
    
    const results = [];
    let successCount = 0;
    let failureCount = 0;
    
    // Process emails in batches to avoid overwhelming the SMTP server
    const batchSize = 3; // Send 3 emails at a time
    const batches = [];
    
    for (let i = 0; i < recipients.length; i += batchSize) {
      batches.push(recipients.slice(i, i + batchSize));
    }
    
    console.log(`📦 Processing ${batches.length} batches of ${batchSize} emails each`);
    
    for (let batchIndex = 0; batchIndex < batches.length; batchIndex++) {
      const batch = batches[batchIndex];
      console.log(`\n📦 Processing batch ${batchIndex + 1}/${batches.length} (${batch.length} emails)`);
      
      // Process batch with delay to avoid rate limiting
      const batchPromises = batch.map(async (recipient, index) => {
        try {
          console.log(`\n📧 Sending email ${batchIndex * batchSize + index + 1}/${recipients.length} to: ${recipient.email}`);
          
          const mailOptions = {
            from: "Titan Drilling HR Management System <support@titandrillingzm.com>",
            to: recipient.email,
            subject: emailContent.subject,
            html: emailContent.html,
            text: emailContent.text
          };

          // Retry mechanism for email sending
          let info;
          let retryCount = 0;
          const maxRetries = 3;
          
          while (retryCount < maxRetries) {
            try {
              info = await transporter.sendMail(mailOptions);
              break; // Success, exit retry loop
            } catch (retryError) {
              retryCount++;
              console.log(`⚠️ Email attempt ${retryCount}/${maxRetries} failed for ${recipient.email}: ${retryError.message}`);
              
              if (retryCount < maxRetries) {
                // Wait before retry (exponential backoff)
                const delay = Math.pow(2, retryCount) * 1000; // 2s, 4s, 8s
                console.log(`⏳ Retrying in ${delay}ms...`);
                await new Promise(resolve => setTimeout(resolve, delay));
              } else {
                throw retryError; // Final attempt failed
              }
            }
          }
          
          const result = {
            email: recipient.email,
            name: recipient.name,
            success: true,
            messageId: info.messageId,
            batch: batchIndex + 1,
            emailNumber: batchIndex * batchSize + index + 1,
            retryCount: retryCount
          };
          
          successCount++;
          console.log(`✅ Email sent successfully to ${recipient.email} (Message ID: ${info.messageId})${retryCount > 0 ? ` after ${retryCount} retries` : ''}`);
          
          // Log successful email
          await logEmailSent(recipient, emailType, info.messageId);
          
          return result;
          
        } catch (error) {
          console.error(`❌ Failed to send email to ${recipient.email}:`, error);
          
          const result = {
            email: recipient.email,
            name: recipient.name,
            success: false,
            error: error.message,
            batch: batchIndex + 1,
            emailNumber: batchIndex * batchSize + index + 1
          };
          
          failureCount++;
          return result;
        }
      });
      
      // Wait for all emails in this batch to complete
      const batchResults = await Promise.all(batchPromises);
      results.push(...batchResults);
      
      // Add delay between batches to avoid overwhelming SMTP server
      if (batchIndex < batches.length - 1) {
        console.log(`⏳ Waiting 3 seconds before next batch...`);
        await new Promise(resolve => setTimeout(resolve, 3000));
      }
    }
    
    console.log(`\n🎉 EMAIL QUEUE COMPLETED!`);
    console.log(`✅ Successful: ${successCount}`);
    console.log(`❌ Failed: ${failureCount}`);
    console.log(`📊 Total: ${recipients.length}`);
    
    return {
      success: true,
      sent: successCount,
      failed: failureCount,
      total: recipients.length,
      results: results
    };
    
  } catch (error) {
    console.error('❌ Critical error in email queue:', error);
    return {
      success: false,
      sent: 0,
      failed: recipients.length,
      total: recipients.length,
      error: error.message,
      results: []
    };
  }
}

/**
 * Log email sent in Firebase
 */
async function logEmailSent(recipient, emailType, messageId) {
  try {
    const emailLog = {
      recipient: recipient.email,
      recipientName: recipient.name,
      recipientRole: recipient.role,
      emailType,
      messageId,
      sentAt: admin.database.ServerValue.TIMESTAMP,
      status: 'sent'
    };

    await db.ref('/emailLogs/hrApplicationApprovals').push(emailLog);
  } catch (error) {
    console.error('Error logging email:', error);
  }
}

/**
 * Send FCM push notification to application creator
 */
async function sendFCMNotificationToCreator(applicationData, applicationCreator, extractedData, applicationId) {
  try {
    console.log('=== SENDING FCM PUSH NOTIFICATION TO CREATOR ===');
    
    const { applicationType, fullName, employeeNo, country, project, department, status, approvalUpdate } = extractedData;
    
    if (!applicationCreator) {
      console.log('⚠️ No application creator to send notification to');
      return { success: false, error: 'No application creator found' };
    }
    
    // Get application creator's user data for FCM tokens
    const creatorSnapshot = await db.ref(`/Globalusers/admins/${applicationCreator.userId}`).once('value');
    const creatorData = creatorSnapshot.val();
    
    if (!creatorData) {
      console.log('⚠️ Application creator data not found');
      return { success: false, error: 'Creator data not found' };
    }
    
    let devices = {};
    
    // Check for devices in the multi-device structure
    if (creatorData.devices && Object.keys(creatorData.devices).length > 0) {
      devices = creatorData.devices;
      console.log(`📱 Creator has ${Object.keys(devices).length} devices in multi-device structure`);
    }
    // Fallback to the old single FCM token structure
    else if (creatorData.NotificationFCMtoken) {
      devices = {
        FCM: {
          token: creatorData.NotificationFCMtoken,
          fcmStatus: 'active',
          lastTokenUpdate: creatorData.lastTokenUpdate || new Date().toISOString(),
          deviceInfo: {
            platform: 'legacy',
            deviceId: 'legacy',
            timestamp: new Date().toISOString()
          }
        }
      };
      console.log(`📱 Creator has legacy FCM token structure`);
    }
    
    if (Object.keys(devices).length === 0) {
      console.log('⚠️ Creator has no FCM tokens');
      return { success: false, error: 'No FCM tokens found' };
    }
    
    // Prepare push notification data
    const pushData = {
      type: 'hr_application_approval_rejection',
      applicationId: applicationId,
      applicationType: applicationType,
      country: country,
      project: project,
      employeeName: fullName,
      employeeNo: employeeNo,
      department: department,
      status: status,
      action: approvalUpdate.action,
      approverRole: approvalUpdate.approverRole,
      approverName: approvalUpdate.approverName,
      reason: approvalUpdate.reason || null,
      source: 'hr_application_approval',
      priority: 'high'
    };
    
    const pushTitle = `🏢 HR Application ${approvalUpdate.action}: ${formatApplicationType(applicationType)}`;
    const pushBody = `Your ${formatApplicationType(applicationType)} has been ${approvalUpdate.action.toLowerCase()} by ${approvalUpdate.approverName}`;
    
    // Send notification to each device
    const deviceResults = [];
    let deviceSuccessCount = 0;
    let deviceFailureCount = 0;
    let invalidTokenCount = 0;
    
    for (const [deviceKey, deviceData] of Object.entries(devices)) {
      if (deviceData.token && deviceData.fcmStatus === 'active') {
        try {
          const fcmResult = await sendFcmToToken(
            deviceData.token, 
            pushTitle,
            pushBody,
            pushData,
            deviceKey
          );
          
          if (fcmResult.success) {
            deviceSuccessCount++;
            deviceResults.push({ 
              deviceKey, 
              status: 'success', 
              messageId: fcmResult.messageId 
            });
          } else {
            deviceFailureCount++;
            deviceResults.push({ 
              deviceKey, 
              status: 'failed', 
              error: fcmResult.error 
            });
            
            // If token is invalid, mark device as inactive
            if (fcmResult.code === 'messaging/invalid-registration-token' || 
                fcmResult.code === 'messaging/registration-token-not-registered') {
              await markDeviceAsInactive(applicationCreator.userId, deviceKey);
              invalidTokenCount++;
            }
          }
        } catch (error) {
          deviceFailureCount++;
          deviceResults.push({ 
            deviceKey, 
            status: 'error', 
            error: error.message 
          });
        }
      }
    }
    
    console.log(`📊 Push notification summary: ${deviceSuccessCount} devices successful, ${deviceFailureCount} failed, ${invalidTokenCount} invalid tokens`);
    
    // Store push notification record
    const pushRecord = {
      applicationId: applicationId,
      type: 'hr_application_approval_rejection',
      title: pushTitle,
      body: pushBody,
      data: pushData,
      sentAt: new Date().toISOString(),
      creatorId: applicationCreator.userId,
      deviceSuccessCount,
      deviceFailureCount,
      invalidTokenCount,
      deviceResults,
      status: 'completed'
    };
    
    await db.ref(`/GlobalPushNotifications/hr_application_approvals/${applicationId}`).set(pushRecord);
    
    return {
      success: deviceSuccessCount > 0,
      deviceSuccessCount,
      deviceFailureCount,
      invalidTokenCount,
      message: 'Push notification sent to creator'
    };
    
  } catch (error) {
    console.error('Error sending FCM push notification to creator:', error);
    return {
      success: false,
      error: error.message,
      message: 'Failed to send push notification to creator'
    };
  }
}

/**
 * Send FCM to token helper function
 */
async function sendFcmToToken(token, title, body, data = {}, deviceKey = 'unknown') {
  try {
    const message = {
      token: token,
      notification: {
        title: title,
        body: body,
      },
      data: {
        type: data.type || 'hr_application_approval_rejection',
        applicationId: data.applicationId || 'unknown',
        applicationType: data.applicationType || '',
        country: data.country || '',
        project: data.project || '',
        employeeName: data.employeeName || '',
        employeeNo: data.employeeNo || '',
        department: data.department || '',
        status: data.status || '',
        action: data.action || '',
        approverRole: data.approverRole || '',
        approverName: data.approverName || '',
        reason: data.reason || '',
        source: data.source || 'hr_application_approval',
        priority: data.priority || 'high',
        deviceKey: deviceKey,
        timestamp: new Date().toISOString()
      },
      android: {
        notification: {
          channelId: 'hr_application_approvals',
          importance: 'high',
          priority: 'high',
          icon: 'ic_notification',
          color: data.action === 'Approved' ? '#28a745' : '#dc3545',
          sound: 'default',
          vibrateTimingsMillis: [0, 250, 250, 250],
          lightSettings: {
            color: data.action === 'Approved' ? '#28a745' : '#dc3545',
            lightOnDurationMillis: 1000,
            lightOffDurationMillis: 1000,
          },
        },
        priority: 'high',
      },
      apns: {
        payload: {
          aps: {
            'content-available': 1,
            'mutable-content': 1,
            sound: 'default',
            badge: 1,
            alert: {
              title: title,
              body: body,
            },
            category: 'HR_APPLICATION_APPROVAL',
          },
        },
        headers: {
          'apns-priority': '10',
          'apns-push-type': 'alert',
        },
      },
    };

    console.log(`📱 Sending HR application approval FCM notification to device ${deviceKey} with token:`, token.substring(0, 20) + '...');
    const response = await admin.messaging().send(message);
    
    console.log(`✅ FCM HR application approval notification sent successfully to device ${deviceKey}`);
    return { success: true, messageId: response };
    
  } catch (error) {
    console.error(`❌ Error sending FCM HR application approval notification to device ${deviceKey}:`, error);
    
    // Handle specific FCM errors
    if (error.code === 'messaging/invalid-registration-token' || 
        error.code === 'messaging/registration-token-not-registered') {
      console.log(`🔄 FCM token is invalid or expired for device ${deviceKey}, marking as inactive`);
      return { success: false, error: 'Invalid or expired FCM token', code: error.code };
    }
    
    return { success: false, error: error.message, code: error.code };
  }
}

/**
 * Mark device as inactive when FCM token is invalid
 */
async function markDeviceAsInactive(userId, deviceKey) {
  try {
    const deviceRef = db.ref(`/Globalusers/admins/${userId}/devices/${deviceKey}`);
    await deviceRef.update({
      fcmStatus: 'inactive',
      lastError: 'Invalid FCM token',
      lastErrorAt: new Date().toISOString()
    });
    console.log(`🔄 Marked device ${deviceKey} as inactive for user ${userId}`);
  } catch (error) {
    console.error(`❌ Error marking device ${deviceKey} as inactive for user ${userId}:`, error);
  }
}

/**
 * Save notification to application creator
 */
async function saveNotificationToCreator(applicationData, applicationCreator, extractedData, applicationId) {
  try {
    console.log('=== SAVING NOTIFICATION TO CREATOR ===');
    
    const { applicationType, fullName, employeeNo, country, project, department, status, approvalUpdate } = extractedData;
    
    // Prepare notification data
    const notificationData = {
      type: 'hr_application_approval_rejection',
      title: `🏢 HR Application ${approvalUpdate.action}: ${formatApplicationType(applicationType)}`,
      body: `Your ${formatApplicationType(applicationType)} has been ${approvalUpdate.action.toLowerCase()} by ${approvalUpdate.approverName}`,
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'hr_application_approval_rejection',
        applicationId: applicationId,
        applicationType: applicationType,
        employeeName: fullName,
        employeeNo: employeeNo,
        country: country,
        project: project,
        department: department,
        status: status,
        action: approvalUpdate.action,
        approverRole: approvalUpdate.approverRole,
        approverName: approvalUpdate.approverName,
        reason: approvalUpdate.reason || null,
        source: 'hr_application_approval'
      }
    };
    
    try {
      console.log(`📱 Saving notification to creator ${applicationCreator.userId} (${applicationCreator.name})`);
      
      // Create notification record
      const notificationRef = db.ref(`/Globalusers/admins/${applicationCreator.userId}/notifications`).push();
      const notificationRecord = {
        id: notificationRef.key,
        ...notificationData,
        userId: applicationCreator.userId,
        recipientName: applicationCreator.name,
        recipientEmail: applicationCreator.email
      };
      
      await notificationRef.set(notificationRecord);
      
      console.log(`✅ Notification saved to creator ${applicationCreator.userId} (${applicationCreator.name})`);
      
      return {
        success: true,
        notificationId: notificationRef.key,
        message: 'Notification saved to creator successfully'
      };
      
    } catch (error) {
      console.error(`❌ Error saving notification to creator ${applicationCreator.userId}:`, error);
      return {
        success: false,
        error: error.message,
        message: 'Failed to save notification to creator'
      };
    }
    
  } catch (error) {
    console.error('Error saving notification to creator:', error);
    return {
      success: false,
      error: error.message,
      message: 'Failed to save notification to creator'
    };
  }
}

/**
 * Get Finance Managers that match the application's country and project
 */
async function getFinanceManagersByCountryProject(appCountry, appProject) {
  try {
    console.log('🔍 Getting Finance Managers matching country and project:', { appCountry, appProject });
    
    const usersSnapshot = await db.ref('/Globalusers/admins').once('value');
    const users = usersSnapshot.val() || {};
    
    console.log(`Total users found in database: ${Object.keys(users).length}`);
    
    const matchingFinanceManagers = [];
    const skippedUsers = [];

    for (const [userId, userData] of Object.entries(users)) {
      if (!userData) {
        console.log(`Skipping user ${userId}: No user data`);
        continue;
      }

      const { role, email, name, countries, projects } = userData;
      
      // Skip users without email
      if (!email) {
        console.log(`Skipping user ${userId}: No email address`);
        skippedUsers.push({ userId, reason: 'No email address' });
        continue;
      }

      // Check if user is a Finance Manager
      const userRole = role || '';
      const isFinanceManager = checkIfFinanceManager(userRole);
      
      if (!isFinanceManager) {
        skippedUsers.push({ userId, name, role: userRole, reason: 'Not a Finance Manager role' });
        continue;
      }

      // Check if Finance Manager's countries and projects match the application
      const userCountries = Array.isArray(countries) ? countries : [];
      const userProjects = Array.isArray(projects) ? projects : [];
      
      const countryMatch = userCountries.includes(appCountry);
      const projectMatch = userProjects.includes(appProject);
      
      if (countryMatch && projectMatch) {
        matchingFinanceManagers.push({
          email,
          name: name || 'Unknown Finance Manager',
          role: userRole,
          userId,
          countries: userCountries,
          projects: userProjects,
          reason: 'Finance Manager with matching country and project'
        });
        console.log(`   ✅ ADDED Finance Manager: ${name} (${email}) - Countries: [${userCountries.join(', ')}], Projects: [${userProjects.join(', ')}]`);
      } else {
        skippedUsers.push({ 
          userId, 
          name, 
          role: userRole, 
          reason: `Finance Manager but no match - Country: ${countryMatch}, Project: ${projectMatch}`,
          userCountries,
          userProjects
        });
      }
    }
    
    console.log(`\n📊 FINANCE MANAGER MATCHING RESULTS:`);
    console.log(`Total users in database: ${Object.keys(users).length}`);
    console.log(`Matching Finance Managers: ${matchingFinanceManagers.length}`);
    console.log(`Users skipped: ${skippedUsers.length}`);
    console.log(`Application Country: ${appCountry}, Project: ${appProject}`);
    
    return matchingFinanceManagers;

  } catch (error) {
    console.error('❌ Error getting Finance Managers by country and project:', error);
    return [];
  }
}

/**
 * Get HR Managers that match the application's country and project
 */
async function getHrManagersByCountryProject(appCountry, appProject) {
  try {
    console.log('🔍 Getting HR Managers matching country and project:', { appCountry, appProject });
    
    const usersSnapshot = await db.ref('/Globalusers/admins').once('value');
    const users = usersSnapshot.val() || {};
    
    console.log(`Total users found in database: ${Object.keys(users).length}`);
    
    const matchingHrManagers = [];
    const skippedUsers = [];

    for (const [userId, userData] of Object.entries(users)) {
      if (!userData) {
        console.log(`Skipping user ${userId}: No user data`);
        continue;
      }

      const { role, email, name, countries, projects } = userData;
      
      // Skip users without email
      if (!email) {
        console.log(`Skipping user ${userId}: No email address`);
        skippedUsers.push({ userId, reason: 'No email address' });
        continue;
      }

      // Check if user is an HR Manager
      const userRole = role || '';
      const isHrManager = checkIfHrManager(userRole);
      
      if (!isHrManager) {
        skippedUsers.push({ userId, name, role: userRole, reason: 'Not an HR Manager role' });
        continue;
      }

      // Check if HR Manager's countries and projects match the application
      const userCountries = Array.isArray(countries) ? countries : [];
      const userProjects = Array.isArray(projects) ? projects : [];
      
      const countryMatch = userCountries.includes(appCountry);
      const projectMatch = userProjects.includes(appProject);
      
      if (countryMatch && projectMatch) {
        matchingHrManagers.push({
          email,
          name: name || 'Unknown HR Manager',
          role: userRole,
          userId,
          countries: userCountries,
          projects: userProjects,
          reason: 'HR Manager with matching country and project'
        });
        console.log(`   ✅ ADDED HR Manager: ${name} (${email}) - Countries: [${userCountries.join(', ')}], Projects: [${userProjects.join(', ')}]`);
      } else {
        skippedUsers.push({ 
          userId, 
          name, 
          role: userRole, 
          reason: `HR Manager but no match - Country: ${countryMatch}, Project: ${projectMatch}`,
          userCountries,
          userProjects
        });
      }
    }
    
    console.log(`\n📊 HR MANAGER MATCHING RESULTS:`);
    console.log(`Total users in database: ${Object.keys(users).length}`);
    console.log(`Matching HR Managers: ${matchingHrManagers.length}`);
    console.log(`Users skipped: ${skippedUsers.length}`);
    console.log(`Application Country: ${appCountry}, Project: ${appProject}`);
    
    return matchingHrManagers;

  } catch (error) {
    console.error('❌ Error getting HR Managers by country and project:', error);
    return [];
  }
}

/**
 * Check if user role is a Finance Manager
 */
function checkIfFinanceManager(role) {
  if (!role || typeof role !== 'string') return false;
  
  const roleLower = role.toLowerCase();
  
  // Check for Finance Manager roles
  return roleLower.includes('financemanager') || 
         roleLower.includes('finance manager') || 
         roleLower.includes('finance') ||
         roleLower.includes('financial');
}

/**
 * Check if user role is an HR Manager
 */
function checkIfHrManager(role) {
  if (!role || typeof role !== 'string') return false;
  
  const roleLower = role.toLowerCase();
  
  // Check for HR Manager roles
  return roleLower.includes('hrmanager') || 
         roleLower.includes('hr manager') || 
         roleLower.includes('hr') ||
         roleLower.includes('human resources');
}

/**
 * Prepare manager notification email content
 */
function prepareManagerNotificationEmailContent(data) {
  const {
    applicationData,
    applicationType,
    fullName,
    employeeNo,
    country,
    project,
    department,
    nrcNo,
    submittedAt,
    status,
    applicationCreator,
    approvalUpdate
  } = data;

  // Format application type for display
  const applicationTypeDisplay = formatApplicationType(applicationType);
  
  const emailSubject = `Supervisor Approved - Please Review: ${applicationTypeDisplay}`;
  
  const emailBody = `
    <html>
      <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333;">
        <div style="max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="background-color: #015185; color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0;">
            <h1 style="margin: 0; font-size: 24px;">Supervisor Approved - Please Review</h1>
            <p style="margin: 10px 0 0 0; font-size: 16px;">${applicationTypeDisplay}</p>
          </div>
          
          <div style="background-color: #f8f9fa; padding: 20px; border: 1px solid #dee2e6;">
            <div style="background-color: #d1ecf1; padding: 15px; border-left: 4px solid #0c5460; margin-bottom: 20px;">
              <h2 style="color: #0c5460; margin-top: 0;">
                📋 Supervisor Approved - Please Review
              </h2>
              <p style="margin: 0;">
                <strong>Status:</strong> Supervisor approved, please review this application.
              </p>
            </div>
            
            <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold; width: 40%;">Application Type:</td>
                <td style="padding: 8px;">${applicationTypeDisplay}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Employee Name:</td>
                <td style="padding: 8px;">${fullName}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Employee Number:</td>
                <td style="padding: 8px;">${employeeNo}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Country:</td>
                <td style="padding: 8px;">${country || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Project:</td>
                <td style="padding: 8px;">${project || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Department:</td>
                <td style="padding: 8px;">${department || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">NRC Number:</td>
                <td style="padding: 8px;">${nrcNo || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Status:</td>
                <td style="padding: 8px;">
                  <span style="background-color: #ffc107; color: #212529; padding: 2px 8px; border-radius: 4px;">Pending Review</span>
                </td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Submitted Date:</td>
                <td style="padding: 8px;">${new Date(submittedAt).toLocaleDateString('en-GB')} ${new Date(submittedAt).toLocaleTimeString()}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Approved by Supervisor:</td>
                <td style="padding: 8px;">${approvalUpdate.approverName}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Supervisor Approval Date:</td>
                <td style="padding: 8px;">${new Date(approvalUpdate.timestamp).toLocaleDateString('en-GB')} ${new Date(approvalUpdate.timestamp).toLocaleTimeString()}</td>
              </tr>
            </table>
            
            ${getApplicationSpecificDetails(applicationType, applicationData)}
            
            <div style="text-align: center; margin: 20px 0;">
              <p style="margin-bottom: 15px; font-weight: bold;">Please log in to the Titan Drilling HR system to review this application.</p>
              <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px;">
                <p style="margin: 0; font-size: 14px; color: #6c757d;">
                  <strong>Next Steps:</strong><br>
                  1. Review the application details<br>
                  2. Approve or reject the application<br>
                  3. Add any necessary comments
                </p>
              </div>
            </div>
            
            <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin-top: 20px;">
              <p style="margin: 0; font-size: 14px; color: #6c757d; text-align: center;">
                This is an automated notification from the Titan Drilling HR Management System.<br>
                Please do not reply to this email.
              </p>
            </div>
          </div>
        </div>
      </body>
    </html>
  `;

  return {
    subject: emailSubject,
    html: emailBody,
    text: convertHtmlToText(emailBody)
  };
}

/**
 * Send FCM push notifications to managers
 */
async function sendFCMNotificationsToManagers(applicationData, managersToNotify, managerType, extractedData, applicationId) {
  try {
    console.log(`=== SENDING FCM PUSH NOTIFICATIONS TO ${managerType.toUpperCase()}S ===`);
    
    const { applicationType, fullName, employeeNo, country, project, department, status, approvalUpdate } = extractedData;
    
    if (managersToNotify.length === 0) {
      console.log(`⚠️ No ${managerType}s to send notifications to`);
      return { success: false, error: `No ${managerType}s found` };
    }
    
    const results = [];
    let totalManagers = managersToNotify.length;
    let successCount = 0;
    let failureCount = 0;
    let noDevicesCount = 0;
    let invalidTokenCount = 0;
    
    // Prepare push notification data
    const pushData = {
      type: 'hr_application_supervisor_approved',
      applicationId: applicationId,
      applicationType: applicationType,
      country: country,
      project: project,
      employeeName: fullName,
      employeeNo: employeeNo,
      department: department,
      status: status,
      approverRole: approvalUpdate.approverRole,
      approverName: approvalUpdate.approverName,
      source: 'hr_application_approval',
      priority: 'high'
    };
    
    const pushTitle = `📋 Supervisor Approved - Please Review: ${formatApplicationType(applicationType)}`;
    const pushBody = `${fullName} (${employeeNo}) - Supervisor approved, please review this application`;
    
    // Process each manager
    for (const manager of managersToNotify) {
      try {
        console.log(`🔍 Checking ${managerType} ${manager.userId} (${manager.name}) for FCM tokens...`);
        
        // Get manager's user data for FCM tokens
        const managerSnapshot = await db.ref(`/Globalusers/admins/${manager.userId}`).once('value');
        const managerData = managerSnapshot.val();
        
        if (!managerData) {
          noDevicesCount++;
          console.log(`⚠️ ${managerType} ${manager.userId} data not found`);
          results.push({ managerId: manager.userId, status: 'no_data', message: `${managerType} data not found` });
          continue;
        }
        
        let devices = {};
        
        // Check for devices in the multi-device structure
        if (managerData.devices && Object.keys(managerData.devices).length > 0) {
          devices = managerData.devices;
          console.log(`📱 ${managerType} ${manager.userId} has ${Object.keys(devices).length} devices in multi-device structure`);
        }
        // Fallback to the old single FCM token structure
        else if (managerData.NotificationFCMtoken) {
          devices = {
            FCM: {
              token: managerData.NotificationFCMtoken,
              fcmStatus: 'active',
              lastTokenUpdate: managerData.lastTokenUpdate || new Date().toISOString(),
              deviceInfo: {
                platform: 'legacy',
                deviceId: 'legacy',
                timestamp: new Date().toISOString()
              }
            }
          };
          console.log(`📱 ${managerType} ${manager.userId} has legacy FCM token structure`);
        }
        
        if (Object.keys(devices).length === 0) {
          noDevicesCount++;
          console.log(`⚠️ ${managerType} ${manager.userId} has no FCM tokens`);
          results.push({ managerId: manager.userId, status: 'no_devices', message: 'No FCM tokens found' });
          continue;
        }
        
        // Send notification to each device
        const deviceResults = [];
        let deviceSuccessCount = 0;
        let deviceFailureCount = 0;
        
        for (const [deviceKey, deviceData] of Object.entries(devices)) {
          if (deviceData.token && deviceData.fcmStatus === 'active') {
            try {
              const fcmResult = await sendFcmToToken(
                deviceData.token, 
                pushTitle,
                pushBody,
                pushData,
                deviceKey
              );
              
              if (fcmResult.success) {
                deviceSuccessCount++;
                deviceResults.push({ 
                  deviceKey, 
                  status: 'success', 
                  messageId: fcmResult.messageId 
                });
              } else {
                deviceFailureCount++;
                deviceResults.push({ 
                  deviceKey, 
                  status: 'failed', 
                  error: fcmResult.error 
                });
                
                // If token is invalid, mark device as inactive
                if (fcmResult.code === 'messaging/invalid-registration-token' || 
                    fcmResult.code === 'messaging/registration-token-not-registered') {
                  await markDeviceAsInactive(manager.userId, deviceKey);
                  invalidTokenCount++;
                }
              }
            } catch (error) {
              deviceFailureCount++;
              deviceResults.push({ 
                deviceKey, 
                status: 'error', 
                error: error.message 
              });
            }
          }
        }
        
        if (deviceSuccessCount > 0) {
          successCount++;
          results.push({ 
            managerId: manager.userId, 
            status: 'success', 
            devices: deviceResults,
            successCount: deviceSuccessCount,
            failureCount: deviceFailureCount
          });
        } else {
          failureCount++;
          results.push({ 
            managerId: manager.userId, 
            status: 'failed', 
            devices: deviceResults,
            error: 'All device notifications failed'
          });
        }
        
      } catch (error) {
        failureCount++;
        console.error(`❌ Error processing ${managerType} ${manager.userId}:`, error);
        results.push({ managerId: manager.userId, status: 'error', error: error.message });
      }
    }
    
    console.log(`📊 Push notification summary: ${successCount}/${totalManagers} ${managerType}s successful, ${failureCount} failed, ${noDevicesCount} no devices, ${invalidTokenCount} invalid tokens`);
    
    // Store push notification record
    const pushRecord = {
      applicationId: applicationId,
      type: 'hr_application_supervisor_approved',
      title: pushTitle,
      body: pushBody,
      data: pushData,
      sentAt: new Date().toISOString(),
      totalManagers,
      successCount,
      failureCount,
      noDevicesCount,
      invalidTokenCount,
      results,
      status: 'completed'
    };
    
    await db.ref(`/GlobalPushNotifications/hr_application_${managerType.toLowerCase()}_notifications/${applicationId}`).set(pushRecord);
    
    return {
      success: true,
      totalManagers,
      successCount,
      failureCount,
      noDevicesCount,
      invalidTokenCount,
      message: `Push notifications sent successfully to ${managerType}s`
    };
    
  } catch (error) {
    console.error(`Error sending FCM push notifications to ${managerType}s:`, error);
    return {
      success: false,
      error: error.message,
      message: `Failed to send push notifications to ${managerType}s`
    };
  }
}

/**
 * Save notifications to managers
 */
async function saveNotificationsToManagers(applicationData, managersToNotify, managerType, extractedData, applicationId) {
  try {
    console.log(`=== SAVING NOTIFICATIONS TO ${managerType.toUpperCase()}S ===`);
    
    const { applicationType, fullName, employeeNo, country, project, department, status, approvalUpdate } = extractedData;
    
    // Prepare notification data
    const notificationData = {
      type: 'hr_application_supervisor_approved',
      title: `📋 Supervisor Approved - Please Review: ${formatApplicationType(applicationType)}`,
      body: `${fullName} (${employeeNo}) - Supervisor approved, please review this application`,
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'hr_application_supervisor_approved',
        applicationId: applicationId,
        applicationType: applicationType,
        employeeName: fullName,
        employeeNo: employeeNo,
        country: country,
        project: project,
        department: department,
        status: status,
        approverRole: approvalUpdate.approverRole,
        approverName: approvalUpdate.approverName,
        source: 'hr_application_approval'
      }
    };
    
    const results = [];
    let successCount = 0;
    let failureCount = 0;
    
    // Save notification to each manager
    for (const manager of managersToNotify) {
      try {
        console.log(`📱 Saving notification to ${managerType} ${manager.userId} (${manager.name})`);
        
        // Create notification record
        const notificationRef = db.ref(`/Globalusers/admins/${manager.userId}/notifications`).push();
        const notificationRecord = {
          id: notificationRef.key,
          ...notificationData,
          userId: manager.userId,
          recipientName: manager.name,
          recipientEmail: manager.email
        };
        
        await notificationRef.set(notificationRecord);
        
        successCount++;
        results.push({ 
          managerId: manager.userId, 
          status: 'success', 
          notificationId: notificationRef.key 
        });
        
        console.log(`✅ Notification saved to ${managerType} ${manager.userId} (${manager.name})`);
        
      } catch (error) {
        failureCount++;
        console.error(`❌ Error saving notification to ${managerType} ${manager.userId}:`, error);
        results.push({ 
          managerId: manager.userId, 
          status: 'failed', 
          error: error.message 
        });
      }
    }
    
    console.log(`📊 Notification saving summary: ${successCount} successful, ${failureCount} failed`);
    
    return {
      success: true,
      totalManagers: managersToNotify.length,
      successCount,
      failureCount,
      message: `Notifications saved to ${managerType} lists successfully`
    };
    
  } catch (error) {
    console.error(`Error saving notifications to ${managerType}s:`, error);
    return {
      success: false,
      error: error.message,
      message: `Failed to save notifications to ${managerType}s`
    };
  }
}

// Function is already exported above as exports.ApproveApplicationSentNotify
