const functions = require('firebase-functions/v1');
const admin = require('firebase-admin');
const nodemailer = require('nodemailer');

// Initialize Firebase Admin if not already initialized
if (!admin.apps.length) {
  admin.initializeApp();
}

const db = admin.database();

// Email configuration - Using Gmail SMTP server
const emailConfig = {
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'mandeep.owninfotech@gmail.com',
    pass: 'aeqa purb mxtl hkhh',
  },
};

// Create transporter
const transporter = nodemailer.createTransport(emailConfig);

/**
 * HR Application Notification Function
 * Handles both email and push notifications for HR applications
 * Triggers when HR application is created at /GlobalHrApplications/admins/{applicationType}/{userId}/{applicationId}
 */
exports.AdminHrAppliApplyNotify = functions.database
  .ref('/GlobalHrApplications/admins/{applicationType}/{userId}/{applicationId}')
  .onCreate(async (snapshot, context) => {
    try {
      console.log('🏢 HR APPLICATION NOTIFICATION TRIGGERED! 🏢');
      console.log('Path:', context.params);
      console.log('Snapshot exists:', !!snapshot);
      console.log('Function version: v1.2 - Enhanced debugging');
      
      const applicationData = snapshot.val();
      const { applicationType, userId, applicationId } = context.params;

      console.log('HR Application created:', { applicationType, userId, applicationId, applicationData });

      if (!applicationData) {
        console.log('No HR application data found');
        return null;
      }

      // Get the HR application details
      const {
        fullName,
        employeeNo,
        country,
        project,
        department,
        nrcNo,
        submittedAt,
        status = 'Pending',
        // Application type specific fields
        leaveStartDate,
        leaveEndDate,
        reasonOfLeave,
        salaryAdvanceAmount,
        deceasedName,
        appealGrounds,
        grievanceDetails,
        selectedClinics
      } = applicationData;
      
      if (!fullName || !employeeNo) {
        console.log('Missing required HR application data');
        console.log('Available fields:', Object.keys(applicationData));
        return null;
      }
      
      // Debug logging
      console.log('=== HR APPLICATION EMAIL NOTIFICATION DEBUG ===');
      console.log('Extracted data:', {
        applicationType,
        fullName,
        employeeNo,
        country,
        project,
        department,
        status
      });

      // Get supervisors matching the application's country and project
      const supervisorsToNotify = await getSupervisorsByCountryProject(country, project);
      
      // Get HR application creator's email
      const applicationCreator = await getApplicationCreator(userId);

      // Prepare email content for supervisors
      const supervisorEmailContent = prepareHrApplicationEmailContent({
        applicationData,
        applicationType,
        fullName,
        employeeNo,
        country,
        project,
        department,
        nrcNo,
        submittedAt,
        status,
        applicationCreator,
        isForSupervisor: true
      });

      // Prepare email content for application creator
      const creatorEmailContent = prepareHrApplicationEmailContent({
        applicationData,
        applicationType,
        fullName,
        employeeNo,
        country,
        project,
        department,
        nrcNo,
        submittedAt,
        status,
        applicationCreator,
        isForSupervisor: false
      });

      // Log all recipients for debugging
      console.log('=== EMAIL RECIPIENTS SUMMARY ===');
      console.log('Supervisors (country + project match):', supervisorsToNotify.length);
      console.log('Application creator:', applicationCreator ? 'Yes' : 'No');
      console.log('Total recipients:', supervisorsToNotify.length + (applicationCreator ? 1 : 0));
      console.log('================================');

      // Send emails to matching supervisors
      if (supervisorsToNotify.length > 0) {
        console.log(`📧 Sending emails to ${supervisorsToNotify.length} supervisors with matching country + project...`);
        
        const supervisorEmailResult = await sendBulkEmails(
          supervisorsToNotify, 
          supervisorEmailContent, 
          'HR Application - Supervisor Review Required'
        );
        console.log('📧 Supervisor email sending result:', supervisorEmailResult);
      } else {
        console.log('⚠️ No supervisors found matching the application country + project');
      }

      // Send confirmation email to application creator
      if (applicationCreator) {
        console.log(`📧 Sending confirmation email to application creator: ${applicationCreator.email}`);
        const creatorResult = await sendBulkEmails(
          [applicationCreator], 
          creatorEmailContent, 
          'HR Application - Submission Confirmation'
        );
        console.log('📧 Application creator email result:', creatorResult);
      }

      // 🔔 SEND FCM PUSH NOTIFICATIONS TO MATCHING SUPERVISORS
      console.log('🔔 Starting FCM push notifications to supervisors...');
      await sendFCMNotificationsToSupervisors(applicationData, supervisorsToNotify, applicationCreator, {
        applicationType,
        fullName,
        employeeNo,
        country,
        project,
        department,
        status
      }, applicationId);

      // 📱 SAVE NOTIFICATIONS TO SUPERVISOR NOTIFICATION LISTS
      console.log('📱 Saving notifications to supervisor notification lists...');
      await saveNotificationsToSupervisors(applicationData, supervisorsToNotify, applicationCreator, {
        applicationType,
        fullName,
        employeeNo,
        country,
        project,
        department,
        status
      }, applicationId);

      // 📱 SAVE CONFIRMATION NOTIFICATION TO APPLICATION CREATOR
      if (applicationCreator) {
        console.log('📱 Saving confirmation notification to application creator...');
        await saveConfirmationNotificationToCreator(applicationData, applicationCreator, {
          applicationType,
          fullName,
          employeeNo,
          country,
          project,
          department,
          status
        }, applicationId);
      }

      console.log('✅ HR Application notifications sent successfully');
      return null;

    } catch (error) {
      console.error('Error in AdminHrAppliApplyNotify:', error);
      return null;
    }
  });

/**
 * Get supervisors that match the application's country and project
 */
async function getSupervisorsByCountryProject(appCountry, appProject) {
  try {
    console.log('🔍 Getting supervisors matching country and project:', { appCountry, appProject });
    
    const usersSnapshot = await db.ref('/Globalusers/admins').once('value');
    const users = usersSnapshot.val() || {};
    
    console.log(`Total users found in database: ${Object.keys(users).length}`);
    
    const matchingSupervisors = [];
    const skippedUsers = [];

    for (const [userId, userData] of Object.entries(users)) {
      if (!userData) {
        console.log(`Skipping user ${userId}: No user data`);
        continue;
      }

      const { role, email, name, countries, projects } = userData;
      
      // Skip users without email
      if (!email) {
        console.log(`Skipping user ${userId}: No email address`);
        skippedUsers.push({ userId, reason: 'No email address' });
        continue;
      }

      // Check if user is a supervisor
      const userRole = role || '';
      const isSupervisor = checkIfSupervisor(userRole);
      
      if (!isSupervisor) {
        skippedUsers.push({ userId, name, role: userRole, reason: 'Not a supervisor role' });
        continue;
      }

      // Check if supervisor's countries and projects match the application
      const userCountries = Array.isArray(countries) ? countries : [];
      const userProjects = Array.isArray(projects) ? projects : [];
      
      const countryMatch = userCountries.includes(appCountry);
      const projectMatch = userProjects.includes(appProject);
      
      if (countryMatch && projectMatch) {
        matchingSupervisors.push({
          email,
          name: name || 'Unknown Supervisor',
          role: userRole,
          userId,
          countries: userCountries,
          projects: userProjects,
          reason: 'Supervisor with matching country and project'
        });
        console.log(`   ✅ ADDED Supervisor: ${name} (${email}) - Countries: [${userCountries.join(', ')}], Projects: [${userProjects.join(', ')}]`);
      } else {
        skippedUsers.push({ 
          userId, 
          name, 
          role: userRole, 
          reason: `Supervisor but no match - Country: ${countryMatch}, Project: ${projectMatch}`,
          userCountries,
          userProjects
        });
      }
    }
    
    console.log(`\n📊 SUPERVISOR MATCHING RESULTS:`);
    console.log(`Total users in database: ${Object.keys(users).length}`);
    console.log(`Matching supervisors: ${matchingSupervisors.length}`);
    console.log(`Users skipped: ${skippedUsers.length}`);
    console.log(`Application Country: ${appCountry}, Project: ${appProject}`);
    
    // Log detailed skipped reasons
    const skipReasons = {};
    skippedUsers.forEach(user => {
      skipReasons[user.reason] = (skipReasons[user.reason] || 0) + 1;
    });
    console.log('Skip reasons:', skipReasons);
    
    return matchingSupervisors;

  } catch (error) {
    console.error('❌ Error getting supervisors by country and project:', error);
    return [];
  }
}

/**
 * Check if user role is a supervisor
 */
function checkIfSupervisor(role) {
  if (!role || typeof role !== 'string') return false;
  
  const roleLower = role.toLowerCase();
  
  // Check for supervisor roles
  return roleLower.includes('supervisor') || 
         roleLower.includes('manager') || 
         roleLower.includes('lead') ||
         roleLower.includes('foreman') ||
         roleLower.includes('superintendent');
}

/**
 * Get HR application creator's information
 */
async function getApplicationCreator(userId) {
  try {
    console.log('Getting HR application creator for userId:', userId);
    
    const userSnapshot = await db.ref(`/Globalusers/admins/${userId}`).once('value');
    const userData = userSnapshot.val();
    
    if (!userData || !userData.email) {
      console.log('No user data or email found for userId:', userId);
      return null;
    }
    
    const applicationCreator = {
      email: userData.email,
      name: userData.name || userData.fullName || 'Application Creator',
      role: userData.role || 'User',
      userId: userId
    };
    
    console.log('Found HR application creator:', applicationCreator);
    return applicationCreator;
    
  } catch (error) {
    console.error('Error getting HR application creator:', error);
    return null;
  }
}

/**
 * Prepare HR application email content
 */
function prepareHrApplicationEmailContent(data) {
  const {
    applicationData,
    applicationType,
    fullName,
    employeeNo,
    country,
    project,
    department,
    nrcNo,
    submittedAt,
    status,
    applicationCreator,
    isForSupervisor
  } = data;

  // Format application type for display
  const applicationTypeDisplay = formatApplicationType(applicationType);
  
  const emailSubject = isForSupervisor 
    ? `HR Application: ${applicationTypeDisplay} - Review Required`
    : `HR Application: ${applicationTypeDisplay} - Submission Confirmed`;
  
  const emailBody = `
    <html>
      <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333;">
        <div style="max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="background-color: #015185; color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0;">
            <h1 style="margin: 0; font-size: 24px;">HR Application ${isForSupervisor ? 'Review' : 'Confirmation'}</h1>
            <p style="margin: 10px 0 0 0; font-size: 16px;">${applicationTypeDisplay}</p>
          </div>
          
          <div style="background-color: #f8f9fa; padding: 20px; border: 1px solid #dee2e6;">
            ${isForSupervisor ? `
              <h2 style="color: #015185; margin-top: 0;">New Application Requires Your Review</h2>
              <p style="background-color: #fff3cd; padding: 10px; border-left: 4px solid #ffc107; margin-bottom: 20px;">
                <strong>Action Required:</strong> Please review this HR application that matches your assigned country and project.
              </p>
            ` : `
              <h2 style="color: #015185; margin-top: 0;">Application Submitted Successfully</h2>
              <p style="background-color: #d4edda; padding: 10px; border-left: 4px solid #28a745; margin-bottom: 20px;">
                <strong>Confirmation:</strong> Your HR application has been submitted and is now under review.
              </p>
            `}
            
            <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold; width: 40%;">Application Type:</td>
                <td style="padding: 8px;">${applicationTypeDisplay}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Employee Name:</td>
                <td style="padding: 8px;">${fullName}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Employee Number:</td>
                <td style="padding: 8px;">${employeeNo}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Country:</td>
                <td style="padding: 8px;">${country || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Project:</td>
                <td style="padding: 8px;">${project || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Department:</td>
                <td style="padding: 8px;">${department || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">NRC Number:</td>
                <td style="padding: 8px;">${nrcNo || 'N/A'}</td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Status:</td>
                <td style="padding: 8px;"><span style="background-color: #ffc107; color: #212529; padding: 2px 8px; border-radius: 4px;">${status}</span></td>
              </tr>
              <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 8px; font-weight: bold;">Submitted Date:</td>
                <td style="padding: 8px;">${new Date(submittedAt).toLocaleDateString('en-GB')} ${new Date(submittedAt).toLocaleTimeString()}</td>
              </tr>
            </table>
            
            ${getApplicationSpecificDetails(applicationType, applicationData)}
            
            ${isForSupervisor ? `
              <div style="text-align: center; margin: 20px 0;">
                <p style="margin-bottom: 15px; font-weight: bold;">Please log in to the Titan Drilling HR system to review this application.</p>
                <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px;">
                  <p style="margin: 0; font-size: 14px; color: #6c757d;">
                    <strong>Next Steps:</strong><br>
                    1. Review the application details<br>
                    2. Approve or reject the application<br>
                    3. Add any necessary comments
                  </p>
                </div>
              </div>
            ` : `
              <div style="text-align: center; margin: 20px 0;">
                <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px;">
                  <p style="margin: 0; font-size: 14px; color: #6c757d;">
                    <strong>What's Next:</strong><br>
                    Your application is now being reviewed by the appropriate supervisors.<br>
                    You will be notified once a decision has been made.
                  </p>
                </div>
              </div>
            `}
            
            <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin-top: 20px;">
              <p style="margin: 0; font-size: 14px; color: #6c757d; text-align: center;">
                This is an automated notification from the Titan Drilling HR Management System.<br>
                Please do not reply to this email.
              </p>
            </div>
          </div>
        </div>
      </body>
    </html>
  `;

  return {
    subject: emailSubject,
    html: emailBody,
    text: convertHtmlToText(emailBody)
  };
}

/**
 * Format application type for display
 */
function formatApplicationType(applicationType) {
  const typeMap = {
    'leave_application': 'Leave Application',
    'funeral_grant_form': 'Funeral Grant Form',
    'salary_advance_application': 'Salary Advance Application',
    'application_for_appeal': 'Application for Appeal',
    'grievance_record': 'Grievance Record',
    'hospital_attendance_form': 'Hospital Attendance Form'
  };
  
  return typeMap[applicationType] || applicationType.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
}

/**
 * Get application specific details for email
 */
function getApplicationSpecificDetails(applicationType, applicationData) {
  switch (applicationType) {
    case 'leave_application':
      return `
        <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h3 style="color: #015185; margin-top: 0;">Leave Details</h3>
          <p><strong>Start Date:</strong> ${applicationData.leaveStartDate || 'N/A'}</p>
          <p><strong>End Date:</strong> ${applicationData.leaveEndDate || 'N/A'}</p>
          <p><strong>Report Date:</strong> ${applicationData.reportDate || 'N/A'}</p>
          <p><strong>Reason:</strong> ${applicationData.reasonOfLeave || 'N/A'}</p>
        </div>
      `;
    case 'salary_advance_application':
      return `
        <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h3 style="color: #015185; margin-top: 0;">Salary Advance Details</h3>
          <p><strong>Amount:</strong> K ${applicationData.salaryAdvanceAmount || 'N/A'}</p>
          <p><strong>Recovery Period:</strong> ${formatRecoveryPeriod(applicationData.selectedRecoveryPeriod) || 'N/A'}</p>
          <p><strong>Sites:</strong> ${formatSites(applicationData.selectedSites) || 'N/A'}</p>
          <p><strong>Reasons:</strong> ${formatReasons(applicationData.selectedReasons) || 'N/A'}</p>
        </div>
      `;
    case 'funeral_grant_form':
      return `
        <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h3 style="color: #015185; margin-top: 0;">Funeral Grant Details</h3>
          <p><strong>Deceased Name:</strong> ${applicationData.deceasedName || 'N/A'}</p>
          <p><strong>Relationship:</strong> ${applicationData.deceasedRelationship || 'N/A'}</p>
          <p><strong>Date of Death:</strong> ${applicationData.deceasedDate || 'N/A'}</p>
          <p><strong>From Date:</strong> ${applicationData.funeralFromDate || 'N/A'}</p>
          <p><strong>To Date:</strong> ${applicationData.funeralToDate || 'N/A'}</p>
        </div>
      `;
    case 'application_for_appeal':
      return `
        <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h3 style="color: #015185; margin-top: 0;">Appeal Details</h3>
          <p><strong>Sanction:</strong> ${applicationData.appealSanction || 'N/A'}</p>
          <p><strong>Offence:</strong> ${applicationData.appealOffence || 'N/A'}</p>
          <p><strong>Offence Date:</strong> ${applicationData.appealOffenceDate || 'N/A'}</p>
          <p><strong>Grounds for Appeal:</strong> ${applicationData.appealGrounds || 'N/A'}</p>
        </div>
      `;
    case 'grievance_record':
      return `
        <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h3 style="color: #015185; margin-top: 0;">Grievance Details</h3>
          <p><strong>Date:</strong> ${applicationData.grievanceDate || 'N/A'}</p>
          <p><strong>Place of Work:</strong> ${applicationData.grievancePlaceOfWork || 'N/A'}</p>
          <p><strong>Supervisor:</strong> ${applicationData.grievanceSupervisorName || 'N/A'}</p>
          <p><strong>Details:</strong> ${applicationData.grievanceDetails || 'N/A'}</p>
        </div>
      `;
    case 'hospital_attendance_form':
      return `
        <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h3 style="color: #015185; margin-top: 0;">Hospital Attendance Details</h3>
          <p><strong>Clinics:</strong> ${formatClinics(applicationData.selectedClinics) || 'N/A'}</p>
          <p><strong>Shift:</strong> ${applicationData.selectedShift || 'N/A'}</p>
          ${applicationData.otherClinic ? `<p><strong>Other Clinic:</strong> ${applicationData.otherClinic}</p>` : ''}
        </div>
      `;
    default:
      return '';
  }
}

/**
 * Helper functions for formatting data
 */
function formatRecoveryPeriod(period) {
  const periodMap = {
    '1_instalment': '1 Instalment',
    '2_instalments': '2 Instalments',
    '3_instalments': '3 Instalments'
  };
  return periodMap[period] || period;
}

function formatSites(sites) {
  if (!Array.isArray(sites)) return '';
  return sites.map(site => site.replace(/_/g, ' ').toUpperCase()).join(', ');
}

function formatReasons(reasons) {
  if (!Array.isArray(reasons)) return '';
  return reasons.map(reason => reason.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())).join(', ');
}

function formatClinics(clinics) {
  if (!Array.isArray(clinics)) return '';
  return clinics.map(clinic => clinic.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())).join(', ');
}

/**
 * Convert HTML to plain text
 */
function convertHtmlToText(html) {
  return html
    .replace(/<[^>]*>/g, '')
    .replace(/&nbsp;/g, ' ')
    .replace(/&amp;/g, '&')
    .replace(/&lt;/g, '<')
    .replace(/&gt;/g, '>')
    .replace(/&quot;/g, '"')
    .replace(/\s+/g, ' ')
    .trim();
}

/**
 * Send bulk emails
 */
async function sendBulkEmails(recipients, emailContent, emailType) {
  try {
    console.log(`🚀 Starting email queue for ${emailType}`);
    console.log(`📧 Total recipients: ${recipients.length}`);
    
    if (recipients.length === 0) {
      console.log('⚠️ No recipients to send emails to');
      return { success: true, sent: 0, failed: 0, results: [] };
    }
    
    const results = [];
    let successCount = 0;
    let failureCount = 0;
    
    // Process emails in batches to avoid overwhelming the SMTP server
    const batchSize = 3; // Send 3 emails at a time
    const batches = [];
    
    for (let i = 0; i < recipients.length; i += batchSize) {
      batches.push(recipients.slice(i, i + batchSize));
    }
    
    console.log(`📦 Processing ${batches.length} batches of ${batchSize} emails each`);
    
    for (let batchIndex = 0; batchIndex < batches.length; batchIndex++) {
      const batch = batches[batchIndex];
      console.log(`\n📦 Processing batch ${batchIndex + 1}/${batches.length} (${batch.length} emails)`);
      
      // Process batch with delay to avoid rate limiting
      const batchPromises = batch.map(async (recipient, index) => {
        try {
          console.log(`\n📧 Sending email ${batchIndex * batchSize + index + 1}/${recipients.length} to: ${recipient.email}`);
          
          const mailOptions = {
            from: "Titan Drilling HR Management System <support@titandrillingzm.com>",
            to: recipient.email,
            subject: emailContent.subject,
            html: emailContent.html,
            text: emailContent.text
          };

          // Retry mechanism for email sending
          let info;
          let retryCount = 0;
          const maxRetries = 3;
          
          while (retryCount < maxRetries) {
            try {
              info = await transporter.sendMail(mailOptions);
              break; // Success, exit retry loop
            } catch (retryError) {
              retryCount++;
              console.log(`⚠️ Email attempt ${retryCount}/${maxRetries} failed for ${recipient.email}: ${retryError.message}`);
              
              if (retryCount < maxRetries) {
                // Wait before retry (exponential backoff)
                const delay = Math.pow(2, retryCount) * 1000; // 2s, 4s, 8s
                console.log(`⏳ Retrying in ${delay}ms...`);
                await new Promise(resolve => setTimeout(resolve, delay));
              } else {
                throw retryError; // Final attempt failed
              }
            }
          }
          
          const result = {
            email: recipient.email,
            name: recipient.name,
            success: true,
            messageId: info.messageId,
            batch: batchIndex + 1,
            emailNumber: batchIndex * batchSize + index + 1,
            retryCount: retryCount
          };
          
          successCount++;
          console.log(`✅ Email sent successfully to ${recipient.email} (Message ID: ${info.messageId})${retryCount > 0 ? ` after ${retryCount} retries` : ''}`);
          
          // Log successful email
          await logEmailSent(recipient, emailType, info.messageId);
          
          return result;
          
        } catch (error) {
          console.error(`❌ Failed to send email to ${recipient.email}:`, error);
          
          const result = {
            email: recipient.email,
            name: recipient.name,
            success: false,
            error: error.message,
            batch: batchIndex + 1,
            emailNumber: batchIndex * batchSize + index + 1
          };
          
          failureCount++;
          return result;
        }
      });
      
      // Wait for all emails in this batch to complete
      const batchResults = await Promise.all(batchPromises);
      results.push(...batchResults);
      
      // Add delay between batches to avoid overwhelming SMTP server
      if (batchIndex < batches.length - 1) {
        console.log(`⏳ Waiting 3 seconds before next batch...`);
        await new Promise(resolve => setTimeout(resolve, 3000));
      }
    }
    
    console.log(`\n🎉 EMAIL QUEUE COMPLETED!`);
    console.log(`✅ Successful: ${successCount}`);
    console.log(`❌ Failed: ${failureCount}`);
    console.log(`📊 Total: ${recipients.length}`);
    
    return {
      success: true,
      sent: successCount,
      failed: failureCount,
      total: recipients.length,
      results: results
    };
    
  } catch (error) {
    console.error('❌ Critical error in email queue:', error);
    return {
      success: false,
      sent: 0,
      failed: recipients.length,
      total: recipients.length,
      error: error.message,
      results: []
    };
  }
}

/**
 * Log email sent in Firebase
 */
async function logEmailSent(recipient, emailType, messageId) {
  try {
    const emailLog = {
      recipient: recipient.email,
      recipientName: recipient.name,
      recipientRole: recipient.role,
      emailType,
      messageId,
      sentAt: admin.database.ServerValue.TIMESTAMP,
      status: 'sent'
    };

    await db.ref('/emailLogs/hrApplications').push(emailLog);
  } catch (error) {
    console.error('Error logging email:', error);
  }
}

/**
 * Send FCM push notifications to matching supervisors
 */
async function sendFCMNotificationsToSupervisors(applicationData, supervisorsToNotify, applicationCreator, extractedData, applicationId) {
  try {
    console.log('=== SENDING FCM PUSH NOTIFICATIONS TO SUPERVISORS ===');
    
    const { applicationType, fullName, employeeNo, country, project, department, status } = extractedData;
    
    if (supervisorsToNotify.length === 0) {
      console.log('⚠️ No supervisors to send notifications to');
      return { success: false, error: 'No supervisors found' };
    }
    
    const results = [];
    let totalSupervisors = supervisorsToNotify.length;
    let successCount = 0;
    let failureCount = 0;
    let noDevicesCount = 0;
    let invalidTokenCount = 0;
    
    // Prepare push notification data
    const pushData = {
      type: 'hr_application_created',
      applicationId: applicationId,
      applicationType: applicationType,
      country: country,
      project: project,
      employeeName: fullName,
      employeeNo: employeeNo,
      department: department,
      status: status,
      creatorEmail: applicationCreator?.email || 'Unknown',
      creatorName: applicationCreator?.name || 'Unknown',
      source: 'hr_application',
      priority: 'high'
    };
    
    const pushTitle = `🏢 New HR Application: ${formatApplicationType(applicationType)}`;
    const pushBody = `${fullName} (${employeeNo}) submitted ${formatApplicationType(applicationType)} for ${project}`;
    
    // Process each supervisor
    for (const supervisor of supervisorsToNotify) {
      try {
        console.log(`🔍 Checking supervisor ${supervisor.userId} (${supervisor.name}) for FCM tokens...`);
        
        // Get supervisor's user data for FCM tokens
        const supervisorSnapshot = await db.ref(`/Globalusers/admins/${supervisor.userId}`).once('value');
        const supervisorData = supervisorSnapshot.val();
        
        if (!supervisorData) {
          noDevicesCount++;
          console.log(`⚠️ Supervisor ${supervisor.userId} data not found`);
          results.push({ supervisorId: supervisor.userId, status: 'no_data', message: 'Supervisor data not found' });
          continue;
        }
        
        let devices = {};
        
        // Check for devices in the multi-device structure
        if (supervisorData.devices && Object.keys(supervisorData.devices).length > 0) {
          devices = supervisorData.devices;
          console.log(`📱 Supervisor ${supervisor.userId} has ${Object.keys(devices).length} devices in multi-device structure`);
        }
        // Fallback to the old single FCM token structure
        else if (supervisorData.NotificationFCMtoken) {
          devices = {
            FCM: {
              token: supervisorData.NotificationFCMtoken,
              fcmStatus: 'active',
              lastTokenUpdate: supervisorData.lastTokenUpdate || new Date().toISOString(),
              deviceInfo: {
                platform: 'legacy',
                deviceId: 'legacy',
                timestamp: new Date().toISOString()
              }
            }
          };
          console.log(`📱 Supervisor ${supervisor.userId} has legacy FCM token structure`);
        }
        
        if (Object.keys(devices).length === 0) {
          noDevicesCount++;
          console.log(`⚠️ Supervisor ${supervisor.userId} has no FCM tokens`);
          results.push({ supervisorId: supervisor.userId, status: 'no_devices', message: 'No FCM tokens found' });
          continue;
        }
        
        // Send notification to each device
        const deviceResults = [];
        let deviceSuccessCount = 0;
        let deviceFailureCount = 0;
        
        for (const [deviceKey, deviceData] of Object.entries(devices)) {
          if (deviceData.token && deviceData.fcmStatus === 'active') {
            try {
              const fcmResult = await sendFcmToToken(
                deviceData.token, 
                pushTitle,
                pushBody,
                pushData,
                deviceKey
              );
              
              if (fcmResult.success) {
                deviceSuccessCount++;
                deviceResults.push({ 
                  deviceKey, 
                  status: 'success', 
                  messageId: fcmResult.messageId 
                });
              } else {
                deviceFailureCount++;
                deviceResults.push({ 
                  deviceKey, 
                  status: 'failed', 
                  error: fcmResult.error 
                });
                
                // If token is invalid, mark device as inactive
                if (fcmResult.code === 'messaging/invalid-registration-token' || 
                    fcmResult.code === 'messaging/registration-token-not-registered') {
                  await markDeviceAsInactive(supervisor.userId, deviceKey);
                  invalidTokenCount++;
                }
              }
            } catch (error) {
              deviceFailureCount++;
              deviceResults.push({ 
                deviceKey, 
                status: 'error', 
                error: error.message 
              });
            }
          }
        }
        
        if (deviceSuccessCount > 0) {
          successCount++;
          results.push({ 
            supervisorId: supervisor.userId, 
            status: 'success', 
            devices: deviceResults,
            successCount: deviceSuccessCount,
            failureCount: deviceFailureCount
          });
        } else {
          failureCount++;
          results.push({ 
            supervisorId: supervisor.userId, 
            status: 'failed', 
            devices: deviceResults,
            error: 'All device notifications failed'
          });
        }
        
      } catch (error) {
        failureCount++;
        console.error(`❌ Error processing supervisor ${supervisor.userId}:`, error);
        results.push({ supervisorId: supervisor.userId, status: 'error', error: error.message });
      }
    }
    
    console.log(`📊 Push notification summary: ${successCount}/${totalSupervisors} supervisors successful, ${failureCount} failed, ${noDevicesCount} no devices, ${invalidTokenCount} invalid tokens`);
    
    // Store push notification record
    const pushRecord = {
      applicationId: applicationId,
      type: 'hr_application_supervisor_notification',
      title: pushTitle,
      body: pushBody,
      data: pushData,
      sentAt: new Date().toISOString(),
      totalSupervisors,
      successCount,
      failureCount,
      noDevicesCount,
      invalidTokenCount,
      results,
      status: 'completed'
    };
    
    await db.ref(`/GlobalPushNotifications/hr_applications/${applicationId}`).set(pushRecord);
    
    return {
      success: true,
      totalSupervisors,
      successCount,
      failureCount,
      noDevicesCount,
      invalidTokenCount,
      message: 'Push notifications sent successfully to supervisors'
    };
    
  } catch (error) {
    console.error('Error sending FCM push notifications to supervisors:', error);
    return {
      success: false,
      error: error.message,
      message: 'Failed to send push notifications to supervisors'
    };
  }
}

/**
 * Send FCM to token helper function
 */
async function sendFcmToToken(token, title, body, data = {}, deviceKey = 'unknown') {
  try {
    const message = {
      token: token,
      notification: {
        title: title,
        body: body,
      },
      data: {
        type: data.type || 'hr_application_created',
        applicationId: data.applicationId || 'unknown',
        applicationType: data.applicationType || '',
        country: data.country || '',
        project: data.project || '',
        employeeName: data.employeeName || '',
        employeeNo: data.employeeNo || '',
        department: data.department || '',
        status: data.status || '',
        creatorEmail: data.creatorEmail || '',
        creatorName: data.creatorName || '',
        source: data.source || 'hr_application',
        priority: data.priority || 'high',
        deviceKey: deviceKey,
        timestamp: new Date().toISOString()
      },
      android: {
        notification: {
          channelId: 'hr_applications',
          importance: 'high',
          priority: 'high',
          icon: 'ic_notification',
          color: '#015185',
          sound: 'default',
          vibrateTimingsMillis: [0, 250, 250, 250],
          lightSettings: {
            color: '#015185',
            lightOnDurationMillis: 1000,
            lightOffDurationMillis: 1000,
          },
        },
        priority: 'high',
      },
      apns: {
        payload: {
          aps: {
            'content-available': 1,
            'mutable-content': 1,
            sound: 'default',
            badge: 1,
            alert: {
              title: title,
              body: body,
            },
            category: 'HR_APPLICATION',
          },
        },
        headers: {
          'apns-priority': '10',
          'apns-push-type': 'alert',
        },
      },
    };

    console.log(`📱 Sending HR application FCM notification to device ${deviceKey} with token:`, token.substring(0, 20) + '...');
    const response = await admin.messaging().send(message);
    
    console.log(`✅ FCM HR application notification sent successfully to device ${deviceKey}`);
    return { success: true, messageId: response };
    
  } catch (error) {
    console.error(`❌ Error sending FCM HR application notification to device ${deviceKey}:`, error);
    
    // Handle specific FCM errors
    if (error.code === 'messaging/invalid-registration-token' || 
        error.code === 'messaging/registration-token-not-registered') {
      console.log(`🔄 FCM token is invalid or expired for device ${deviceKey}, marking as inactive`);
      return { success: false, error: 'Invalid or expired FCM token', code: error.code };
    }
    
    return { success: false, error: error.message, code: error.code };
  }
}

/**
 * Mark device as inactive when FCM token is invalid
 */
async function markDeviceAsInactive(userId, deviceKey) {
  try {
    const deviceRef = db.ref(`/Globalusers/admins/${userId}/devices/${deviceKey}`);
    await deviceRef.update({
      fcmStatus: 'inactive',
      lastError: 'Invalid FCM token',
      lastErrorAt: new Date().toISOString()
    });
    console.log(`🔄 Marked device ${deviceKey} as inactive for user ${userId}`);
  } catch (error) {
    console.error(`❌ Error marking device ${deviceKey} as inactive for user ${userId}:`, error);
  }
}

/**
 * Save notifications to supervisor notification lists
 */
async function saveNotificationsToSupervisors(applicationData, supervisorsToNotify, applicationCreator, extractedData, applicationId) {
  try {
    console.log('=== SAVING NOTIFICATIONS TO SUPERVISOR LISTS ===');
    
    const { applicationType, fullName, employeeNo, country, project, department, status } = extractedData;
    
    // Prepare notification data
    const notificationData = {
      type: 'hr_application_created',
      title: `🏢 New HR Application: ${formatApplicationType(applicationType)}`,
      body: `${fullName} (${employeeNo}) submitted ${formatApplicationType(applicationType)} for ${project}`,
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'hr_application_created',
        applicationId: applicationId,
        applicationType: applicationType,
        employeeName: fullName,
        employeeNo: employeeNo,
        country: country,
        project: project,
        department: department,
        status: status,
        creatorEmail: applicationCreator?.email || 'Unknown',
        creatorName: applicationCreator?.name || 'Unknown',
        source: 'hr_application'
      }
    };
    
    const results = [];
    let successCount = 0;
    let failureCount = 0;
    
    // Save notification to each supervisor
    for (const supervisor of supervisorsToNotify) {
      try {
        console.log(`📱 Saving notification to supervisor ${supervisor.userId} (${supervisor.name})`);
        
        // Create notification record
        const notificationRef = db.ref(`/Globalusers/admins/${supervisor.userId}/notifications`).push();
        const notificationRecord = {
          id: notificationRef.key,
          ...notificationData,
          userId: supervisor.userId,
          recipientName: supervisor.name,
          recipientEmail: supervisor.email
        };
        
        await notificationRef.set(notificationRecord);
        
        successCount++;
        results.push({ 
          supervisorId: supervisor.userId, 
          status: 'success', 
          notificationId: notificationRef.key 
        });
        
        console.log(`✅ Notification saved to supervisor ${supervisor.userId} (${supervisor.name})`);
        
      } catch (error) {
        failureCount++;
        console.error(`❌ Error saving notification to supervisor ${supervisor.userId}:`, error);
        results.push({ 
          supervisorId: supervisor.userId, 
          status: 'failed', 
          error: error.message 
        });
      }
    }
    
    console.log(`📊 Notification saving summary: ${successCount} successful, ${failureCount} failed`);
    
    return {
      success: true,
      totalSupervisors: supervisorsToNotify.length,
      successCount,
      failureCount,
      message: 'Notifications saved to supervisor lists successfully'
    };
    
  } catch (error) {
    console.error('Error saving notifications to supervisors:', error);
    return {
      success: false,
      error: error.message,
      message: 'Failed to save notifications to supervisors'
    };
  }
}

/**
 * Save confirmation notification to application creator
 */
async function saveConfirmationNotificationToCreator(applicationData, applicationCreator, extractedData, applicationId) {
  try {
    console.log('=== SAVING CONFIRMATION NOTIFICATION TO CREATOR ===');
    
    const { applicationType, fullName, employeeNo, country, project, department, status } = extractedData;
    
    // Prepare notification data
    const notificationData = {
      type: 'hr_application_confirmation',
      title: `✅ HR Application Submitted: ${formatApplicationType(applicationType)}`,
      body: `Your ${formatApplicationType(applicationType)} has been submitted and is under review`,
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'hr_application_confirmation',
        applicationId: applicationId,
        applicationType: applicationType,
        employeeName: fullName,
        employeeNo: employeeNo,
        country: country,
        project: project,
        department: department,
        status: status,
        source: 'hr_application_confirmation'
      }
    };
    
    try {
      console.log(`📱 Saving confirmation notification to creator ${applicationCreator.userId} (${applicationCreator.name})`);
      
      // Create notification record
      const notificationRef = db.ref(`/Globalusers/admins/${applicationCreator.userId}/notifications`).push();
      const notificationRecord = {
        id: notificationRef.key,
        ...notificationData,
        userId: applicationCreator.userId,
        recipientName: applicationCreator.name,
        recipientEmail: applicationCreator.email
      };
      
      await notificationRef.set(notificationRecord);
      
      console.log(`✅ Confirmation notification saved to creator ${applicationCreator.userId} (${applicationCreator.name})`);
      
      return {
        success: true,
        notificationId: notificationRef.key,
        message: 'Confirmation notification saved to creator successfully'
      };
      
    } catch (error) {
      console.error(`❌ Error saving confirmation notification to creator ${applicationCreator.userId}:`, error);
      return {
        success: false,
        error: error.message,
        message: 'Failed to save confirmation notification to creator'
      };
    }
    
  } catch (error) {
    console.error('Error saving confirmation notification to creator:', error);
    return {
      success: false,
      error: error.message,
      message: 'Failed to save confirmation notification to creator'
    };
  }
}

// Function is already exported above as exports.AdminHrAppliApplyNotify
