import { NativeModules, Platform } from 'react-native';

const { IOSNotificationBridge } = NativeModules;

class IOSNotificationManager {
  constructor() {
    this.isIOS = Platform.OS === 'ios';
    this.bridge = this.isIOS ? IOSNotificationBridge : null;
  }

  // Check if iOS notification bridge is available
  isAvailable() {
    return this.isIOS && this.bridge !== null;
  }

  // Request notification permissions
  async requestPermissions() {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.requestNotificationPermissions();
      console.log('✅ iOS: Notification permissions result:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to request notification permissions:', error);
      throw error;
    }
  }

  // Get current notification permissions
  async getPermissions() {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.getNotificationPermissions();
      console.log('✅ iOS: Current notification permissions:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to get notification permissions:', error);
      throw error;
    }
  }

  // Get FCM token
  async getFCMToken() {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.getFCMToken();
      console.log('✅ iOS: FCM token retrieved:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to get FCM token:', error);
      throw error;
    }
  }

  // Subscribe to topic
  async subscribeToTopic(topic) {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.subscribeToTopic(topic);
      console.log('✅ iOS: Subscribed to topic:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to subscribe to topic:', error);
      throw error;
    }
  }

  // Unsubscribe from topic
  async unsubscribeFromTopic(topic) {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.unsubscribeFromTopic(topic);
      console.log('✅ iOS: Unsubscribed from topic:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to unsubscribe from topic:', error);
      throw error;
    }
  }

  // Display local notification
  async displayLocalNotification(notificationData) {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.displayLocalNotification(notificationData);
      console.log('✅ iOS: Local notification displayed:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to display local notification:', error);
      throw error;
    }
  }

  // Cancel all notifications
  async cancelAllNotifications() {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.cancelAllNotifications();
      console.log('✅ iOS: All notifications cancelled:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to cancel all notifications:', error);
      throw error;
    }
  }

  // Cancel specific notification
  async cancelNotification(identifier) {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.cancelNotification(identifier);
      console.log('✅ iOS: Notification cancelled:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to cancel notification:', error);
      throw error;
    }
  }

  // Get delivered notifications
  async getDeliveredNotifications() {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.getDeliveredNotifications();
      console.log('✅ iOS: Delivered notifications retrieved:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to get delivered notifications:', error);
      throw error;
    }
  }

  // Get pending notifications
  async getPendingNotifications() {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.getPendingNotifications();
      console.log('✅ iOS: Pending notifications retrieved:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to get pending notifications:', error);
      throw error;
    }
  }

  // Set badge count
  async setBadgeCount(count) {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.setBadgeCount(count);
      console.log('✅ iOS: Badge count set:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to set badge count:', error);
      throw error;
    }
  }

  // Get current badge count
  async getBadgeCount() {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.getBadgeCount();
      console.log('✅ iOS: Badge count retrieved:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to get badge count:', error);
      throw error;
    }
  }

  // Clear badge
  async clearBadge() {
    if (!this.isAvailable()) {
      throw new Error('iOS notification bridge not available');
    }

    try {
      const result = await this.bridge.clearBadge();
      console.log('✅ iOS: Badge cleared:', result);
      return result;
    } catch (error) {
      console.error('❌ iOS: Failed to clear badge:', error);
      throw error;
    }
  }

  // Helper method to check if user has granted permissions
  async hasPermissions() {
    try {
      const permissions = await this.getPermissions();
      return permissions.authorizationStatus === 2; // UNAuthorizationStatus.authorized
    } catch (error) {
      console.error('❌ iOS: Failed to check permissions:', error);
      return false;
    }
  }

  // Helper method to request permissions if not granted
  async ensurePermissions() {
    try {
      const hasGranted = await this.hasPermissions();
      if (!hasGranted) {
        console.log('📱 iOS: Requesting notification permissions...');
        return await this.requestPermissions();
      } else {
        console.log('✅ iOS: Notification permissions already granted');
        return { granted: true, alreadyGranted: true };
      }
    } catch (error) {
      console.error('❌ iOS: Failed to ensure permissions:', error);
      throw error;
    }
  }

  // Helper method to create and display a welcome notification
  async displayWelcomeNotification(userName, userRole) {
    try {
      const notificationData = {
        title: '🎉 Welcome to Titan Drilling!',
        body: `Hello ${userName || 'Admin'}! You are logged in as ${userRole || 'User'}`,
        category: 'welcome',
        sound: 'default',
        badge: 1,
        userInfo: {
          type: 'welcome',
          userName: userName || 'Admin',
          userRole: userRole || 'User',
          timestamp: new Date().toISOString()
        }
      };

      return await this.displayLocalNotification(notificationData);
    } catch (error) {
      console.error('❌ iOS: Failed to display welcome notification:', error);
      throw error;
    }
  }

  // Helper method to create and display a profile update notification
  async displayProfileUpdateNotification(adminName, changedFields) {
    try {
      const notificationData = {
        title: 'Profile Updated',
        body: `Your profile has been updated by ${adminName}. Changes: ${changedFields.join(', ')}`,
        category: 'profile_updated',
        sound: 'default',
        badge: 1,
        userInfo: {
          type: 'profile_updated',
          updatedBy: adminName,
          changedFields: changedFields,
          timestamp: new Date().toISOString()
        }
      };

      return await this.displayLocalNotification(notificationData);
    } catch (error) {
      console.error('❌ iOS: Failed to display profile update notification:', error);
      throw error;
    }
  }
}

// Create singleton instance
const iosNotificationManager = new IOSNotificationManager();

export default iosNotificationManager;
