import {
  View,
  Text,
  Image,
  StyleSheet,
  SafeAreaView,
  TouchableOpacity,
  Animated,
  ActivityIndicator,
  Platform,
  ToastAndroid,
  Modal,
  ScrollView,
  KeyboardAvoidingView,
  PermissionsAndroid,
} from 'react-native';
import React, { useState, useRef, useEffect } from 'react';
import { useNavigation } from '@react-navigation/native';
import LinearGradient from 'react-native-linear-gradient';
import { TextInput } from 'react-native-paper';
import Toast from 'react-native-toast-message';
import auth from '@react-native-firebase/auth';
import database from '@react-native-firebase/database';
import messaging from '@react-native-firebase/messaging';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import notifee, { AndroidImportance } from '@notifee/react-native';

// Detect if running on emulator
let _isEmulatorCache = null;
const isEmulator = () => {
  if (_isEmulatorCache !== null) {
    return _isEmulatorCache;
  }
  
  try {
    if (Platform.OS === 'android') {
      const DeviceInfo = require('react-native-device-info');
      _isEmulatorCache = DeviceInfo.isEmulatorSync();
      return _isEmulatorCache;
    }
    _isEmulatorCache = false;
    return false;
  } catch (error) {
    console.warn('Could not detect emulator, assuming real device:', error);
    _isEmulatorCache = false;
    return false;
  }
};

// Request notification permission
const requestNotificationPermission = async () => {
  try {
    console.log('🔐 Requesting notification permissions...');
    
    if (isEmulator()) {
      console.log('ℹ️ Emulator detected - skipping notification permission request');
      return false;
    }
    
    if (Platform.OS === 'android' && Platform.Version >= 33) {
      console.log('📱 Android 13+ detected, requesting POST_NOTIFICATIONS permission...');
      try {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.POST_NOTIFICATIONS
        );
        console.log('📱 POST_NOTIFICATIONS permission result:', granted);
        if (granted !== PermissionsAndroid.RESULTS.GRANTED) {
          console.log('❌ POST_NOTIFICATIONS permission denied');
          return false;
        }
      } catch (permError) {
        console.warn('⚠️ Permission request failed:', permError.message);
        return false;
      }
    }
    
    console.log('📱 Requesting FCM permission...');
    try {
      const authStatus = await messaging().requestPermission();
      console.log('📱 FCM permission result:', authStatus);
      
      const enabled =
        authStatus === messaging.AuthorizationStatus.AUTHORIZED ||
        authStatus === messaging.AuthorizationStatus.PROVISIONAL;
      
      console.log('📱 FCM permission enabled:', enabled);
      return enabled;
    } catch (fcmError) {
      console.warn('⚠️ FCM permission request failed:', fcmError.message);
      return false;
    }
  } catch (error) {
    console.error('❌ Notification permission error:', error);
    return false;
  }
};

// Function to generate unique device identifier
const getDeviceIdentifier = async () => {
  try {
    let deviceId = await AsyncStorage.getItem('deviceId');
    
    if (!deviceId) {
      deviceId = `device_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
      await AsyncStorage.setItem('deviceId', deviceId);
      console.log('🆔 Generated new device ID:', deviceId);
    }
    
    return deviceId;
  } catch (error) {
    console.error('❌ Error getting device identifier:', error);
    return `fallback_${Date.now()}`;
  }
};

// Get FCM token and save with multi-device support
const ensureFcmTokenAndSave = async (userId) => {
  try {
    console.log('🔐 Starting FCM token setup for user:', userId);
    
    const runningOnEmulator = isEmulator();
    if (runningOnEmulator) {
      console.log('⚠️ Running on emulator - FCM token setup may be skipped or fail gracefully');
    }
    
    const permission = await requestNotificationPermission();
    if (!permission) {
      console.log('❌ Notification permission denied');
      if (runningOnEmulator) {
        console.log('ℹ️ This is normal on emulator - continuing without FCM token');
        return { token: null, isEmulator: true };
      }
      return null;
    }
    console.log('✅ Notification permission granted');
    
    console.log('📱 Getting FCM token...');
    let token = null;
    try {
      token = await messaging().getToken();
    } catch (tokenError) {
      console.warn('⚠️ Failed to get FCM token:', tokenError.message);
      if (runningOnEmulator) {
        console.log('ℹ️ FCM token unavailable on emulator - continuing without it');
        return { token: null, isEmulator: true };
      }
      throw tokenError;
    }
    
    if (!token) {
      console.log('❌ Failed to get FCM token');
      if (runningOnEmulator) {
        console.log('ℹ️ This is normal on emulator - continuing without FCM token');
        return { token: null, isEmulator: true };
      }
      return null;
    }
    console.log('✅ FCM token obtained:', token.substring(0, 20) + '...');
    
    const deviceId = await getDeviceIdentifier();
    console.log('📱 Device ID:', deviceId);
    
    const storedDeviceKey = await AsyncStorage.getItem('deviceKey');
    const storedDeviceId = await AsyncStorage.getItem('deviceId');
    const storedToken = await AsyncStorage.getItem('fcmToken');
    
    console.log('📱 Stored device info:', {
      deviceKey: storedDeviceKey,
      deviceId: storedDeviceId,
      hasToken: !!storedToken
    });
    
    const userRef = database().ref(`/Globalusers/admins/${userId}`);
    
    try {
      console.log('📊 Fetching existing user data...');
      const userSnapshot = await userRef.once('value');
      const userData = userSnapshot.val() || {};
      
      const existingDevices = userData.devices || {};
      let deviceKey = 'FCM';
      let isNewDevice = false;
      let isTokenChanged = false;
      
      if (storedDeviceKey && storedDeviceId && storedToken) {
        const existingDevice = existingDevices[storedDeviceKey];
        if (existingDevice && 
            existingDevice.deviceInfo?.deviceId === storedDeviceId &&
            existingDevice.token === storedToken) {
          console.log('✅ Device already has stored FCM token, no update needed');
          deviceKey = storedDeviceKey;
          
          await userRef.child('devices').child(deviceKey).child('lastLogin').set(new Date().toISOString());
          console.log('✅ Updated lastLogin timestamp for existing device');
          
          return { 
            token: storedToken, 
            deviceKey: storedDeviceKey, 
            deviceId: storedDeviceId,
            isExistingDevice: true 
          };
        } else if (existingDevice && 
                   existingDevice.deviceInfo?.deviceId === storedDeviceId &&
                   existingDevice.token !== storedToken) {
          console.log('🔄 Same device, token changed - updating token');
          deviceKey = storedDeviceKey;
          isTokenChanged = true;
        } else {
          console.log('🔄 Device info mismatch - treating as new device');
          isNewDevice = true;
        }
      } else {
        console.log('🆕 No stored device info - this is a new device');
        isNewDevice = true;
      }
      
      if (isNewDevice) {
        let counter = 1;
        while (existingDevices[`FCM${counter}`]) {
          counter++;
        }
        deviceKey = counter === 1 ? 'FCM' : `FCM${counter}`;
        console.log('🆕 Adding new device:', deviceKey);
      }
      
      const deviceData = {
        token: token,
        lastTokenUpdate: new Date().toISOString(),
        deviceInfo: {
          platform: Platform.OS,
          version: Platform.Version,
          deviceId: deviceId,
          timestamp: new Date().toISOString(),
          iosVersion: Platform.OS === 'ios' ? Platform.Version : null,
          deviceModel: Platform.OS === 'ios' ? 'iOS Device' : null,
          androidVersion: Platform.OS === 'android' ? Platform.Version : null,
          sdkVersion: Platform.OS === 'android' ? Platform.Version : null,
        },
        fcmStatus: 'active',
        lastLogin: new Date().toISOString(),
        isCurrentDevice: true,
        appVersion: '1.0.0',
        buildNumber: '1',
        lastTokenValidation: new Date().toISOString(),
      };
      
      console.log(`🔄 Saving device data to path: /Globalusers/admins/${userId}/devices/${deviceKey}`);
      const deviceRef = userRef.child('devices').child(deviceKey);
      await deviceRef.set(deviceData);
      console.log('✅ Device data saved successfully!');
      
      if (isNewDevice) {
        const activeDevicesCount = Object.keys(existingDevices).length + 1;
        console.log(`📊 Active devices count: ${activeDevicesCount}`);
        
        await userRef.update({
          lastLogin: new Date().toISOString(),
          activeDevices: activeDevicesCount,
          NotificationFCMtoken: token,
          lastTokenUpdate: new Date().toISOString(),
        });
        console.log('✅ User profile updated successfully!');
      } else {
        await userRef.update({
          lastLogin: new Date().toISOString(),
          NotificationFCMtoken: token,
          lastTokenUpdate: new Date().toISOString(),
        });
        console.log('✅ User profile updated for existing device');
      }
      
      try {
        for (const key of Object.keys(existingDevices)) {
          if (key !== deviceKey) {
            await userRef.child('devices').child(key).child('isCurrentDevice').set(false);
          }
        }
        console.log('✅ Other devices marked as inactive');
      } catch (markError) {
        console.warn('⚠️ Warning: Could not mark other devices as inactive:', markError);
      }
      
      console.log(`✅ FCM token ${isNewDevice ? 'saved' : 'updated'} to database for device: ${deviceKey}`);
      
      try {
        await AsyncStorage.setItem('fcmToken', token);
        await AsyncStorage.setItem('fcmTokenTimestamp', new Date().toISOString());
        await AsyncStorage.setItem('deviceKey', deviceKey);
        await AsyncStorage.setItem('deviceId', deviceId);
        console.log('✅ FCM token stored locally');
      } catch (storageError) {
        console.warn('⚠️ Failed to store FCM token locally:', storageError);
      }
      
      return { 
        token, 
        deviceKey, 
        deviceId, 
        isExistingDevice: !isNewDevice,
        isTokenChanged 
      };
      
    } catch (dbError) {
      console.error('❌ Database operation failed:', dbError);
      console.error('❌ Error code:', dbError.code);
      console.error('❌ Error message:', dbError.message);
      
      if (dbError.code === 'PERMISSION_DENIED') {
        console.error('🚨 PERMISSION DENIED: Firebase database rules are blocking writes');
      }
      
      throw dbError;
    }
    
  } catch (error) {
    console.error('❌ Error ensuring FCM token:', error);
    return null;
  }
};

// Reusable Gradient Button Component
const GradientButton = ({ onPress, title, style, isLoading }) => {
  return (
    <LinearGradient colors={['#015185', '#3481BC']} style={[styles.gradientButton, style]}>
      <TouchableOpacity style={styles.buttonTouchable} onPress={onPress} activeOpacity={0.8} disabled={isLoading}>
        {isLoading ? (
          <ActivityIndicator size="small" color="#fff" />
        ) : (
          <Text style={styles.googleButtonText}>{title}</Text>
        )}
      </TouchableOpacity>
    </LinearGradient>
  );
};

// Custom Animated Input Component
const AnimatedInput = ({
  label,
  value,
  onChangeText,
  secureTextEntry,
  keyboardType,
  autoCapitalize,
  leftIcon,
  rightIcon,
  onRightIconPress,
  hasError,
  onBlur,
}) => {
  const [isFocused, setIsFocused] = useState(false);
  const animatedLabel = useRef(new Animated.Value(value ? 1 : 0)).current;
  const animatedBorderWidth = useRef(new Animated.Value(1)).current;
  const animatedBorderColor = useRef(new Animated.Value(0)).current;

  React.useEffect(() => {
    Animated.timing(animatedLabel, {
      toValue: isFocused || value ? 1 : 0,
      duration: 200,
      useNativeDriver: false,
    }).start();

    Animated.timing(animatedBorderWidth, {
      toValue: isFocused ? 2 : 1,
      duration: 200,
      useNativeDriver: false,
    }).start();

    Animated.timing(animatedBorderColor, {
      toValue: hasError ? 2 : isFocused || value ? 1 : 0,
      duration: 200,
      useNativeDriver: false,
    }).start();
  }, [isFocused, value, hasError]);

  const labelStyle = {
    position: 'absolute',
    left: 45,
    top: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [18, -8],
    }),
    fontSize: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [16, 12],
    }),
    color: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: ['#888', '#015185'],
    }),
    backgroundColor: '#fff',
    paddingHorizontal: 4,
    zIndex: 2,
  };

  const borderColor = animatedBorderColor.interpolate({
    inputRange: [0, 1, 2],
    outputRange: ['#ddd', '#015185', '#ff0000'],
  });

  return (
    <View style={styles.inputContainer}>
      <Animated.View
        style={[
          styles.inputGradient,
          {
            borderWidth: animatedBorderWidth,
            borderColor: borderColor,
          },
        ]}
      >
        <Animated.Text style={labelStyle}>{label}</Animated.Text>
        <TextInput
          mode="outlined"
          value={value}
          onChangeText={onChangeText}
          secureTextEntry={secureTextEntry}
          keyboardType={keyboardType}
          autoCapitalize={autoCapitalize}
          style={styles.input}
          onFocus={() => setIsFocused(true)}
          onBlur={() => {
            setIsFocused(false);
            onBlur && onBlur();
          }}
          left={
            leftIcon && (
              <TextInput.Icon
                icon={() => <Image source={leftIcon} style={styles.inputIcon} />}
              />
            )
          }
          right={
            rightIcon && (
              <TextInput.Icon
                icon={() => (
                  <TouchableOpacity onPress={onRightIconPress}>
                    <Image source={rightIcon} style={styles.eyeIcon} />
                  </TouchableOpacity>
                )}
              />
            )
          }
          outlineStyle={{
            borderWidth: 0,
            borderRadius: 8,
          }}
          theme={{
            colors: {
              primary: 'transparent',
              text: '#333',
              placeholder: '#888',
            },
          }}
        />
      </Animated.View>
    </View>
  );
};

// SyncModal Component
const SyncModal = ({ visible, onClose }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  React.useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[
                styles.syncIcon,
                { transform: [{ rotate: rotateInterpolate }] },
              ]}
            />
            <Text style={styles.syncModalTextadd}>Processing Login</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function to show toast messages based on platform
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.show(message, ToastAndroid.LONG, ToastAndroid.BOTTOM);
  } else {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

// Helper function to trigger welcome notification
const triggerWelcomeNotification = async (userName) => {
  try {
    await notifee.createChannel({
      id: 'default',
      name: 'Default Channel',
      importance: AndroidImportance.HIGH,
    });

    await notifee.displayNotification({
      title: 'Login Success',
      body: `Welcome to Titan Drilling, ${userName || 'Team Member'}!`,
      android: {
        channelId: 'default',
        pressAction: { id: 'default' },
      },
    });
  } catch (error) {
    console.error('Error creating welcome notification:', error);
  }
};

// Helper function to generate a local session identifier (for navigation only)
const generateLocalSessionId = () => {
  return `${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
};

export default function TeamMemberLogin() {
  const navigation = useNavigation();
  const [identifier, setIdentifier] = useState('');
  const [password, setPassword] = useState('');
  const [showPassword, setShowPassword] = useState(false);
  const [identifierError, setIdentifierError] = useState(false);
  const [identifierErrorMessage, setIdentifierErrorMessage] = useState('');
  const [passwordError, setPasswordError] = useState(false);
  const [passwordErrorMessage, setPasswordErrorMessage] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);

  // Validate identifier (email or employee ID)
  const validateIdentifier = (identifier) => {
    const isEmpty = !identifier.trim();
    const isEmail = /\S+@\S+\.\S+/.test(identifier);
    const isEmployeeId = identifier.length >= 1; // Any text or number with at least 1 character
    const isTooLong = identifier.length > 100; // Increased limit for flexibility

    setIdentifierError(isEmpty || (!isEmail && !isEmployeeId) || isTooLong);
    if (isEmpty) {
      setIdentifierErrorMessage('Email/Employee ID is required');
    } else if (!isEmail && !isEmployeeId) {
      setIdentifierErrorMessage('Enter a valid email address or employee ID');
    } else if (isTooLong) {
      setIdentifierErrorMessage('Input cannot exceed 100 characters');
    } else {
      setIdentifierErrorMessage('');
    }
    return !(isEmpty || (!isEmail && !isEmployeeId) || isTooLong);
  };

  // Validate password
  const validatePassword = (password) => {
    const isEmpty = !password.trim();
    setPasswordError(isEmpty);
    if (isEmpty) {
      setPasswordErrorMessage('Password is required');
    } else {
      setPasswordErrorMessage('');
    }
    return !isEmpty;
  };

  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        showToast('error', 'Network Error', 'Internet connection not available');
        return false;
      }
      return true;
    } catch (error) {
      showToast('error', 'Network Error', 'Failed to check network status');
      return false;
    }
  };

// Find user by email or employee ID (optimized using indexed queries)
const findUserByEmailOrEmployeeId = async (input) => {
  try {
    const adminsRef = database().ref('/Globalusers/admins');

    // First try indexed query by email
    try {
      const emailQuery = adminsRef.orderByChild('email').equalTo(input).limitToFirst(1);
      const emailSnapshot = await emailQuery.once('value');
      if (emailSnapshot.exists()) {
        const userId = Object.keys(emailSnapshot.val())[0];
        const userData = emailSnapshot.val()[userId];
        return { userId, userData };
      }
    } catch (emailErr) {
      console.warn('Email query failed, falling back to full-scan:', emailErr?.message || emailErr);
    }

    // Then try indexed query by employeeNumber
    try {
      const empQuery = adminsRef.orderByChild('employeeNumber').equalTo(input).limitToFirst(1);
      const empSnapshot = await empQuery.once('value');
      if (empSnapshot.exists()) {
        const userId = Object.keys(empSnapshot.val())[0];
        const userData = empSnapshot.val()[userId];
        return { userId, userData };
      }
    } catch (empErr) {
      console.warn('Employee number query failed, falling back to full-scan:', empErr?.message || empErr);
    }

    // Fallback: full scan (rare, slower)
    const snapshot = await adminsRef.once('value');
    const admins = snapshot.val();
    if (!admins) return null;
    for (const [userId, userData] of Object.entries(admins)) {
      if (userData.email === input || userData.employeeNumber === input) {
        return { userId, userData };
      }
    }

    return null;
  } catch (error) {
    console.error('Error finding user:', error);
    return null;
  }
};

  // Handle Login
  const handleLogin = async (input, pwd) => {
    const isIdentifierValid = validateIdentifier(input);
    const isPasswordValid = validatePassword(pwd);
    if (!isIdentifierValid || !isPasswordValid) {
      showToast('error', 'Validation Error', 'Please correct the errors in the form');
      return;
    }
    setIsLoading(true);
    setSyncModalVisible(true);
    try {
      if (!(await checkNetworkStatus())) {
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      // First, try to find user by email or employee ID
      const userInfo = await findUserByEmailOrEmployeeId(input);
      
      if (!userInfo) {
        showToast('error', 'Login Error', 'User not found with provided email/employee ID');
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      const { userId, userData } = userInfo;

      // Try to authenticate with Firebase using the found email
      const userCredential = await auth().signInWithEmailAndPassword(userData.email, pwd);
      
      if (userCredential.user.uid !== userId) {
        throw new Error('User authentication mismatch');
      }

      if (userData.role === 'ProAdmin') {
        await auth().signOut();
        showToast('error', 'Access Denied', 'ProAdmin cannot login here');
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      // Store session data immediately for fast navigation
      await Promise.all([
        AsyncStorage.setItem('userUid', userId),
        AsyncStorage.setItem('userEmail', userData.email),
        AsyncStorage.setItem('Loginsuccess-01', 'true'),
        AsyncStorage.setItem('TeamMemberLogined', 'true'),
      ]);

      // Remove any admin session flags
      await AsyncStorage.removeItem('ProAdminLogined').catch(() => {});

      // Navigate immediately for a faster perceived login
      showToast('success', 'Success', 'Logged in successfully');
      setIsLoading(false);
      setSyncModalVisible(false);
      navigation.replace('MainApp');

      // Run slower/background tasks after navigation (fire-and-forget)
      setTimeout(() => {
        (async () => {
          try {
            // Update login status in DB
            const userRef = database().ref(`/Globalusers/admins/${userId}`);
            await userRef.update({
              loginStatus: true,
              lastLogin: new Date().toISOString(),
              firstLoginCompleted: true,
            }).catch(() => {});

            // Update FCM token in background
            try {
              const fcmResult = await ensureFcmTokenAndSave(userId);
              if (fcmResult && fcmResult.token) {
                console.log(`✅ FCM token ${fcmResult.isExistingDevice ? 'verified' : 'saved'} for device: ${fcmResult.deviceKey}`);
              }
            } catch (e) {
              console.warn('Background FCM update failed:', e?.message || e);
            }

            // Subscribe to per-user topic (skip on emulator)
            if (!isEmulator()) {
              try { await messaging().subscribeToTopic(`uid_${userId}`); } catch (e) { console.warn('subscribeToTopic background error:', e?.message || e); }
            }

            // Fire-and-forget welcome notification
            triggerWelcomeNotification(userData.name).catch((err) => console.warn('Welcome notification failed:', err));
          } catch (bgError) {
            console.warn('Background task error:', bgError);
          }
        })();
      }, 100);
    } catch (error) {
      const errorMessages = {
        'auth/invalid-email': 'Invalid email address.',
        'auth/wrong-password': 'Incorrect password.',
        'auth/user-not-found': 'No user found with this email.',
        'auth/too-many-requests': 'Too many attempts. Please try again later.',
        'auth/invalid-credential': 'Invalid credentials provided.',
        default: error.message || 'An unexpected error occurred.',
      };
      showToast('error', 'Login Error', errorMessages[error.code] || errorMessages.default);
      setIsLoading(false);
      setSyncModalVisible(false);
    }
  };

  const toggleShowPassword = () => {
    setShowPassword(!showPassword);
  };

  return (
    <SafeAreaView style={styles.container}>
      <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} />
      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        style={styles.keyboardAvoidingView}
      >
        <ScrollView contentContainerStyle={styles.scrollContent}>
          <View style={styles.contentContainer}>
            {/* Logo and Login Text */}
            <View style={styles.logoContainer}>
              <Image
                source={require('../../Images/adminlogin/logo.png')}
                style={styles.logo}
              />
              <Text style={styles.loginText}>Let’s Get You Started With Titan Drilling</Text>
            </View>
            {/* Form */}
            <View style={styles.formContainer}>
              {/* Animated Identifier Input */}
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Email/Employee ID"
                  value={identifier}
                  onChangeText={(text) => {
                    setIdentifier(text);
                    validateIdentifier(text);
                  }}
                  keyboardType="email-address"
                  autoCapitalize="none"
                  leftIcon={require('../../Images/adminlogin/email.png')}
                  hasError={identifierError}
                  onBlur={() => validateIdentifier(identifier)}
                />
                {identifierError && (
                  <Text style={styles.errorText}>{identifierErrorMessage}</Text>
                )}
              </View>
              {/* Animated Password Input */}
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Password"
                  value={password}
                  onChangeText={(text) => {
                    setPassword(text);
                    validatePassword(text);
                  }}
                  secureTextEntry={!showPassword}
                  autoCapitalize="none"
                  leftIcon={require('../../Images/adminlogin/key.png')}
                  rightIcon={
                    showPassword
                      ? require('../../Images/adminlogin/mdi_eye.png')
                      : require('../../Images/adminlogin/mdi_hide.png')
                  }
                  onRightIconPress={toggleShowPassword}
                  hasError={passwordError}
                  onBlur={() => validatePassword(password)}
                />
                {passwordError && (
                  <Text style={styles.errorText}>{passwordErrorMessage}</Text>
                )}
              </View>
              {/* Login Button */}
              <GradientButton
                title="Login"
                onPress={() => handleLogin(identifier, password)}
                isLoading={isLoading}
              />
            </View>
          </View>
        </ScrollView>
      </KeyboardAvoidingView>
      <Toast />
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
    paddingHorizontal: 10,
  },
  keyboardAvoidingView: {
    flex: 1,
  },
  scrollContent: {
    flexGrow: 1,
    justifyContent: 'center',
  },
  contentContainer: {
    flex: 1,
    justifyContent: 'center',
  },
  logoContainer: {
    alignItems: 'center',
    marginBottom: 20,
  },
  logo: {
    width: '80%',
    height: 100,
    resizeMode: 'contain',
  },
  loginText: {
    fontSize: 16,
    color: '#015185',
    marginTop: 10,
    textAlign: 'center',
  },
  formContainer: {
    width: '100%',
  },
  inputContainer: {
    marginBottom: 30,
    marginHorizontal: 10,
  },
  inputGradient: {
    borderRadius: 8,
    backgroundColor: '#fff',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  input: {
    backgroundColor: '#fff',
    height: 56,
    borderRadius: 8,
    fontSize: 16,
  },
  inputIcon: {
    width: 24,
    height: 24,
    tintColor: '#015185',
  },
  eyeIcon: {
    width: 24,
    height: 24,
    tintColor: '#015185',
  },
  gradientButton: {
    borderRadius: 6,
    marginTop: 10,
    marginBottom: 10,
    width: '95%',
    justifyContent: 'center',
    alignSelf: 'center',
  },
  buttonTouchable: {
    alignItems: 'center',
    justifyContent: 'center',
    height: 48,
    borderRadius: 6,
    width: '100%',
  },
  googleButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  networkErrorBanner: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    backgroundColor: '#ff0000',
    paddingVertical: 10,
    alignItems: 'center',
    zIndex: 1000,
  },
  networkErrorText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  errorText: {
    color: 'red',
    fontSize: 14,
    marginLeft: 10,
    marginTop: 4,
    marginBottom: 8,
    position: 'absolute',
    bottom: 0,
  },
  mainemailpass: {
    position: 'relative',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});