import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, StatusBar, Dimensions, Modal, ActivityIndicator, ScrollView, Alert } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import DateTimePicker from '@react-native-community/datetimepicker';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.32;

const EquipmentstatusKpiHome = () => {
  const navigation = useNavigation();
  const [inspections, setInspections] = useState([]);
  const [loading, setLoading] = useState(false);
  const [modalVisible, setModalVisible] = useState(false);
  const [userInfo, setUserInfo] = useState({ country: '', project: '' });
  // Tabs: 'processing', 'initialized', 'onhold'
  const [activeTab, setActiveTab] = useState('processing');
  // Add tab state: 'withJobCard' or 'withoutJobCard'
  const [openTab, setOpenTab] = useState('withJobCard'); // default to with job card
  // Date range state
  const today = new Date();
  const oneWeekAgo = new Date(today);
  oneWeekAgo.setDate(today.getDate() - 7);
  const [fromDate, setFromDate] = useState(oneWeekAgo);
  const [toDate, setToDate] = useState(today);
  const [showFromPicker, setShowFromPicker] = useState(false);
  const [showToPicker, setShowToPicker] = useState(false);

  useEffect(() => {
    const fetchInspections = async () => {
      setLoading(true);
      try {
        // Fetch all inspections globally (from all admins)
        const snap = await database().ref(`/GlobalInspectionReport/admins`).once('value');
        const adminsData = snap.val() || {};
        // Fetch all job cards globally (from all admins)
        const jobSnap = await database().ref(`/GlobalWorkOrders/admins`).once('value');
        const jobAdminsData = jobSnap.val() || {};
        // Build a map from inspectionId to job card info (global)
        const inspectionToJobCard = {};
        Object.entries(jobAdminsData).forEach(([adminId, jobCards]) => {
          if (jobCards && typeof jobCards === 'object') {
            Object.entries(jobCards).forEach(([jobCardId, jc]) => {
              const linkId = jc.inspectionId || jc.linkedInspectionId || jc.inspectionRef || jc.inspection_id || (jc.linkInspection && jc.linkInspection.id);
              if (linkId) {
                inspectionToJobCard[linkId] = {
                  jobCardNumber: jc.jobCardNumber || jc.requestNumber || '-',
                  jobCardId,
                  pdfDownloadUrl: jc.pdfDownloadUrl || null,
                  jobCardStatus: jc.status || '-',
                  jobCardStartDate: jc.startDate || '-',
                  jobCardDueDate: jc.dueDate || jc.due || '-',
                };
              }
            });
          }
        });
        // Flatten all inspections (include those without job cards)
        let allInspections = [];
        Object.entries(adminsData).forEach(([adminId, reportsObj]) => {
          if (reportsObj && typeof reportsObj === 'object') {
            Object.entries(reportsObj).forEach(([id, r]) => {
              const jobCard = inspectionToJobCard[id];
              allInspections.push({
                id,
                adminId,
                date: r?.step3?.createdAt || '',
                equipment: r?.step1?.selectedEquipment?.equipmentName || r?.step1?.selectedEquipment?.equipmentNumber || '-',
                status: r?.step3?.status || 'Closed',
                jobCardNumber: jobCard ? jobCard.jobCardNumber : 'No Job Card',
                jobCardId: jobCard ? jobCard.jobCardId : null,
                jobCardPdf: jobCard ? jobCard.pdfDownloadUrl : null,
                jobCardStatus: jobCard ? jobCard.jobCardStatus : '-',
                jobCardStartDate: jobCard ? jobCard.jobCardStartDate : '-',
                jobCardDueDate: jobCard ? jobCard.jobCardDueDate : '-',
                project: r?.step1?.project || '-',
                country: r?.step1?.country || '-',
                details: r,
              });
            });
          }
        });
        allInspections = allInspections.sort((a, b) => new Date(b.date) - new Date(a.date));
        setInspections(allInspections);
      } catch (e) {
        setInspections([]);
      }
      setLoading(false);
    };
    fetchInspections();
  }, []);

  const handleLockPress = async () => {
    setModalVisible(true);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        setUserInfo({ country: 'N/A', project: 'N/A' });
        return;
      }
      const snap = await database().ref(`/Globalusers/admins/${userUid}`).once('value');
      const data = snap.val();
      setUserInfo({
        country: Array.isArray(data?.countries) ? data.countries.join(', ') : (data?.country || 'N/A'),
        project: Array.isArray(data?.projects) ? data.projects.join(', ') : (data?.project || 'N/A'),
      });
    } catch (e) {
      setUserInfo({ country: 'N/A', project: 'N/A' });
    }
  };

  // Filter out job cards with status 'Complete', 'Success', or 'Rejected'
  const filteredInspections = inspections.filter(item => {
    const date = item.date ? new Date(item.date) : null;
    if (!date) return false;
    if (date < fromDate || date > toDate) return false;
    const status = (item.jobCardStatus || '').toLowerCase();
    if (status === 'complete' || status === 'success' || status === 'rejected') return false;
    return true;
  });

  // Tab filtering
  const inspectionsToShow = openTab === 'withJobCard'
    ? filteredInspections.filter(item => {
        if (!item.jobCardId) return false;
        // If job card due date is present and valid, only show if today or future
        if (item.jobCardDueDate && item.jobCardDueDate !== '-') {
          const due = new Date(item.jobCardDueDate);
          const now = new Date();
          // Remove time part for comparison
          due.setHours(0,0,0,0);
          now.setHours(0,0,0,0);
          if (due < now) return false;
        }
        return true;
      })
    : filteredInspections.filter(item => {
        if (item.jobCardId) return false;
        // No job card, always show
        return true;
      });

  // Collect all unique equipment objects used in the filtered inspections/job cards
  const equipmentMap = {};
  inspectionsToShow.forEach(item => {
    const eq = item.details?.step1?.selectedEquipment;
    if (eq && eq.equipmentNumber) {
      equipmentMap[eq.equipmentNumber] = eq;
    }
  });
  const uniqueEquipments = Object.values(equipmentMap);

  // Remove status summary and tabs

  const handleViewJobCard = (jobCardId) => {
    if (!jobCardId) return;
    Alert.alert('Job Card', `Open Job Card ID: ${jobCardId}`);
    // navigation.navigate('JobCardDetails', { jobCardId });
  };

  return (
    <View style={{ flex: 1, backgroundColor: '#fff' }}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()} style={styles.headerLeftIcon}>
            <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Equipments Used</Text>
          <TouchableOpacity onPress={handleLockPress} style={styles.headerLeftIcon}>
            <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>
      <View style={{ flex: 1, padding: 16 }}>
        {/* Date Range Picker */}
        <View style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'center', marginBottom: 12 }}>
          <TouchableOpacity onPress={() => setShowFromPicker(true)} style={{ marginRight: 12, borderWidth: 1, borderColor: '#015185', borderRadius: 6, padding: 8 }}>
            <Text style={{ color: '#015185', fontWeight: '700' }}>From: {fromDate.toLocaleDateString()}</Text>
          </TouchableOpacity>
          <TouchableOpacity onPress={() => setShowToPicker(true)} style={{ borderWidth: 1, borderColor: '#015185', borderRadius: 6, padding: 8 }}>
            <Text style={{ color: '#015185', fontWeight: '700' }}>To: {toDate.toLocaleDateString()}</Text>
          </TouchableOpacity>
        </View>
        {showFromPicker && (
          <DateTimePicker
            value={fromDate}
            mode="date"
            display="default"
            onChange={(event, selectedDate) => {
              setShowFromPicker(false);
              if (selectedDate) setFromDate(selectedDate);
            }}
            maximumDate={toDate}
          />
        )}
        {showToPicker && (
          <DateTimePicker
            value={toDate}
            mode="date"
            display="default"
            onChange={(event, selectedDate) => {
              setShowToPicker(false);
              if (selectedDate) setToDate(selectedDate);
            }}
            minimumDate={fromDate}
            maximumDate={today}
          />
        )}
        {/* Equipments Used Section */}
        <View style={{ marginBottom: 16 }}>
          <Text style={{ color: '#015185', fontWeight: 'bold', fontSize: 17, marginBottom: 8 }}>Equipments Used</Text>
          {uniqueEquipments.length === 0 ? (
            <Text style={{ color: '#888', textAlign: 'center' }}>No equipment used in this period.</Text>
          ) : (
            uniqueEquipments.map((eq, idx) => (
              <TouchableOpacity
                key={eq.equipmentNumber || idx}
                onPress={() => navigation.navigate('TrackEquipmentStatus', { equipment: eq })}
                activeOpacity={0.8}
              >
                <View style={{
                  backgroundColor: '#F5F7FA',
                  borderRadius: 8,
                  padding: 12,
                  marginBottom: 10,
                  borderWidth: 1,
                  borderColor: '#E0E0E0'
                }}>
                  <Text style={{ color: '#015185', fontWeight: '700', fontSize: 16 }}>{eq.equipmentName || '-'}</Text>
                  <Text style={{ color: '#2267A2', fontSize: 15 }}>Number: {eq.equipmentNumber || '-'}</Text>
                  <Text style={{ color: '#2267A2', fontSize: 15 }}>Category: {eq.mainCategory || '-'}</Text>
                  <Text style={{ color: '#2267A2', fontSize: 15 }}>Model: {eq.model || '-'}</Text>
                  <Text style={{ color: '#2267A2', fontSize: 15 }}>Meter Reading: {eq.meterReading || '-'} {eq.meterUnit || ''}</Text>
                  <Text style={{ color: '#2267A2', fontSize: 15 }}>Status: {eq.status || '-'}</Text>
                  {/* Add more fields as needed */}
                </View>
              </TouchableOpacity>
            ))
          )}
        </View>
      </View>
      <Modal visible={modalVisible} transparent animationType="fade" onRequestClose={() => setModalVisible(false)}>
        <View style={styles.modalContainer}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>User Info</Text>
            <Text style={styles.modalLabel}>Country:</Text>
            <Text style={styles.modalValue}>{userInfo.country}</Text>
            <Text style={styles.modalLabel}>Project:</Text>
            <Text style={styles.modalValue}>{userInfo.project}</Text>
            <TouchableOpacity style={styles.closeButton} onPress={() => setModalVisible(false)}>
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  header: {
    width: '100%',
    justifyContent: 'flex-end',
    paddingBottom: width * 0.04,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.05,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeftIcon: {
    paddingVertical: width * 0.02,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.055,
    fontWeight: '700',
    flex: 1,
    marginLeft: width * 0.03,
    textAlign: 'center',
  },
  headerRightIcons: {
    width: width * 0.06 + width * 0.02, // To balance the left icon
  },
  inspectionCard: {
    backgroundColor: '#F5F7FA',
    borderRadius: 8,
    padding: 12,
    marginBottom: 12,
    borderWidth: 1,
    borderColor: '#E0E0E0',
  },
  inspectionText: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 2,
  },
  inspectionLabel: {
    fontWeight: '700',
    color: '#2267A2',
  },
  modalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '75%',
    backgroundColor: '#FFF',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    maxWidth: 400,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '700',
    color: '#333',
    marginBottom: 16,
  },
  modalLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginBottom: 4,
  },
  modalValue: {
    fontSize: 14,
    color: '#333',
    marginBottom: 8,
    textAlign: 'center',
  },
  closeButton: {
    backgroundColor: '#3481BC',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 8,
    marginTop: 16,
  },
  closeButtonText: {
    fontSize: 16,
    fontWeight: '700',
    color: '#FFF',
  },
});

export default EquipmentstatusKpiHome;