import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, StatusBar, Dimensions, Modal, ActivityIndicator, ScrollView, Alert } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import DateTimePicker from '@react-native-community/datetimepicker';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.32;

const InspectionsClosed = () => {
  const navigation = useNavigation();
  const [inspections, setInspections] = useState([]);
  const [loading, setLoading] = useState(false);
  const [modalVisible, setModalVisible] = useState(false);
  const [userInfo, setUserInfo] = useState({ country: '', project: '' });
  const [activeTab, setActiveTab] = useState('complete'); // 'complete' or 'rejected'
  // Date range state
  const today = new Date();
  const oneWeekAgo = new Date(today);
  oneWeekAgo.setDate(today.getDate() - 7);
  const [fromDate, setFromDate] = useState(oneWeekAgo);
  const [toDate, setToDate] = useState(today);
  const [showFromPicker, setShowFromPicker] = useState(false);
  const [showToPicker, setShowToPicker] = useState(false);

  useEffect(() => {
    const fetchInspections = async () => {
      setLoading(true);
      try {
        // Fetch all inspections globally (from all admins)
        const snap = await database().ref(`/GlobalInspectionReport/admins`).once('value');
        const adminsData = snap.val() || {};
        // Fetch all job cards globally (from all admins)
        const jobSnap = await database().ref(`/GlobalWorkOrders/admins`).once('value');
        const jobAdminsData = jobSnap.val() || {};
        // Build a map from inspectionId to job card info (global)
        const inspectionToJobCard = {};
        Object.entries(jobAdminsData).forEach(([adminId, jobCards]) => {
          if (jobCards && typeof jobCards === 'object') {
            Object.entries(jobCards).forEach(([jobCardId, jc]) => {
              const linkId = jc.inspectionId || jc.linkedInspectionId || jc.inspectionRef || jc.inspection_id || (jc.linkInspection && jc.linkInspection.id);
              if (linkId) {
                inspectionToJobCard[linkId] = {
                  jobCardNumber: jc.jobCardNumber || jc.requestNumber || '-',
                  jobCardId,
                  pdfDownloadUrl: jc.pdfDownloadUrl || null,
                  jobCardStatus: jc.status || '-',
                  jobCardStartDate: jc.startDate || '-',
                  jobCardDueDate: jc.dueDate || jc.due || '-',
                };
              }
            });
          }
        });
        // Flatten all inspections
        let allInspections = [];
        Object.entries(adminsData).forEach(([adminId, reportsObj]) => {
          if (reportsObj && typeof reportsObj === 'object') {
            Object.entries(reportsObj).forEach(([id, r]) => {
              const jobCard = inspectionToJobCard[id];
              if (jobCard) {
                allInspections.push({
                  id,
                  adminId,
                  date: r?.step3?.createdAt || '',
                  equipment: r?.step1?.selectedEquipment?.equipmentName || r?.step1?.selectedEquipment?.equipmentNumber || '-',
                  status: r?.step3?.status || 'Closed',
                  jobCardNumber: jobCard.jobCardNumber,
                  jobCardId: jobCard.jobCardId,
                  jobCardPdf: jobCard.pdfDownloadUrl,
                  jobCardStatus: jobCard.jobCardStatus,
                  jobCardStartDate: jobCard.jobCardStartDate,
                  jobCardDueDate: jobCard.jobCardDueDate,
                  project: r?.step1?.project || '-',
                  country: r?.step1?.country || '-',
                  details: r,
                });
              }
            });
          }
        });
        allInspections = allInspections.sort((a, b) => new Date(b.date) - new Date(a.date));
        setInspections(allInspections);
      } catch (e) {
        setInspections([]);
      }
      setLoading(false);
    };
    fetchInspections();
  }, []);

  const handleLockPress = async () => {
    setModalVisible(true);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        setUserInfo({ country: 'N/A', project: 'N/A' });
        return;
      }
      const snap = await database().ref(`/Globalusers/admins/${userUid}`).once('value');
      const data = snap.val();
      setUserInfo({
        country: Array.isArray(data?.countries) ? data.countries.join(', ') : (data?.country || 'N/A'),
        project: Array.isArray(data?.projects) ? data.projects.join(', ') : (data?.project || 'N/A'),
      });
    } catch (e) {
      setUserInfo({ country: 'N/A', project: 'N/A' });
    }
  };

  // Tab filtering logic + date range filter
  const filteredInspections = inspections.filter(item => {
    const date = item.date ? new Date(item.date) : null;
    if (!date) return false;
    if (date < fromDate || date > toDate) return false;
    if (activeTab === 'complete') return (item.jobCardStatus || '').toLowerCase() === 'complete';
    if (activeTab === 'rejected') return (item.jobCardStatus || '').toLowerCase() === 'rejected';
    return true;
  });

  const handleViewJobCard = (jobCardId) => {
    if (!jobCardId) return;
    Alert.alert('Job Card', `Open Job Card ID: ${jobCardId}`);
    // navigation.navigate('JobCardDetails', { jobCardId });
  };

  return (
    <View style={{ flex: 1, backgroundColor: '#fff' }}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()} style={styles.headerLeftIcon}>
            <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Inspections with Job Cards</Text>
          <TouchableOpacity onPress={handleLockPress} style={styles.headerLeftIcon}>
            <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>
      <View style={{ flex: 1, padding: 16 }}>
        {/* Date Range Picker */}
        <View style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'center', marginBottom: 12 }}>
          <TouchableOpacity onPress={() => setShowFromPicker(true)} style={{ marginRight: 12, borderWidth: 1, borderColor: '#015185', borderRadius: 6, padding: 8 }}>
            <Text style={{ color: '#015185', fontWeight: '700' }}>From: {fromDate.toLocaleDateString()}</Text>
          </TouchableOpacity>
          <TouchableOpacity onPress={() => setShowToPicker(true)} style={{ borderWidth: 1, borderColor: '#015185', borderRadius: 6, padding: 8 }}>
            <Text style={{ color: '#015185', fontWeight: '700' }}>To: {toDate.toLocaleDateString()}</Text>
          </TouchableOpacity>
        </View>
        {showFromPicker && (
          <DateTimePicker
            value={fromDate}
            mode="date"
            display="default"
            onChange={(event, selectedDate) => {
              setShowFromPicker(false);
              if (selectedDate) setFromDate(selectedDate);
            }}
            maximumDate={toDate}
          />
        )}
        {showToPicker && (
          <DateTimePicker
            value={toDate}
            mode="date"
            display="default"
            onChange={(event, selectedDate) => {
              setShowToPicker(false);
              if (selectedDate) setToDate(selectedDate);
            }}
            minimumDate={fromDate}
            maximumDate={today}
          />
        )}
        {/* Tabs */}
        <View style={{ flexDirection: 'row', justifyContent: 'center', marginBottom: 12 }}>
          <TouchableOpacity onPress={() => setActiveTab('complete')} style={{ marginRight: 16 }}>
            <Text style={{ color: activeTab === 'complete' ? '#015185' : '#888', fontWeight: '700' }}>Complete</Text>
          </TouchableOpacity>
          <TouchableOpacity onPress={() => setActiveTab('rejected')}>
            <Text style={{ color: activeTab === 'rejected' ? '#015185' : '#888', fontWeight: '700' }}>Rejected</Text>
          </TouchableOpacity>
        </View>
        {loading ? (
          <ActivityIndicator size="large" color="#015185" style={{ marginTop: 32 }} />
        ) : (
          <ScrollView>
            {filteredInspections.length === 0 ? (
              <Text style={{ color: '#888', textAlign: 'center', marginTop: 32 }}>No inspections found.</Text>
            ) : (
              filteredInspections.map((item, idx) => (
                <View key={idx} style={styles.inspectionCard}>
                  <Text style={styles.inspectionText}><Text style={styles.inspectionLabel}>Date:</Text> {item.date ? new Date(item.date).toLocaleString() : '-'}</Text>
                  <Text style={styles.inspectionText}><Text style={styles.inspectionLabel}>Equipment:</Text> {item.equipment}</Text>
                  <Text style={styles.inspectionText}><Text style={styles.inspectionLabel}>Project:</Text> {item.project}</Text>
                  <Text style={styles.inspectionText}><Text style={styles.inspectionLabel}>Country:</Text> {item.country}</Text>
                  <Text style={styles.inspectionText}><Text style={styles.inspectionLabel}>Job Card Number:</Text> {item.jobCardNumber}</Text>
                  <Text style={styles.inspectionText}><Text style={styles.inspectionLabel}>Job Card Status:</Text> {item.jobCardStatus}</Text>
                  <Text style={styles.inspectionText}><Text style={styles.inspectionLabel}>Start Date:</Text> {item.jobCardStartDate && item.jobCardStartDate !== '-' ? new Date(item.jobCardStartDate).toLocaleString() : '-'}</Text>
                  <Text style={styles.inspectionText}><Text style={styles.inspectionLabel}>Due Date:</Text> {item.jobCardDueDate && item.jobCardDueDate !== '-' ? new Date(item.jobCardDueDate).toLocaleString() : '-'}</Text>
                  {item.jobCardId && (
                    <TouchableOpacity onPress={() => handleViewJobCard(item.jobCardId)} style={{ marginTop: 6, alignSelf: 'flex-start', backgroundColor: '#015185', borderRadius: 6, paddingVertical: 4, paddingHorizontal: 12 }}>
                      <Text style={{ color: '#fff', fontWeight: '700' }}>View Job Card</Text>
                    </TouchableOpacity>
                  )}
                </View>
              ))
            )}
          </ScrollView>
        )}
      </View>
      <Modal visible={modalVisible} transparent animationType="fade" onRequestClose={() => setModalVisible(false)}>
        <View style={styles.modalContainer}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>User Info</Text>
            <Text style={styles.modalLabel}>Country:</Text>
            <Text style={styles.modalValue}>{userInfo.country}</Text>
            <Text style={styles.modalLabel}>Project:</Text>
            <Text style={styles.modalValue}>{userInfo.project}</Text>
            <TouchableOpacity style={styles.closeButton} onPress={() => setModalVisible(false)}>
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  header: {
    width: '100%',
    justifyContent: 'flex-end',
    paddingBottom: width * 0.04,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.05,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeftIcon: {
    paddingVertical: width * 0.02,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.055,
    fontWeight: '700',
    flex: 1,
    marginLeft: width * 0.03,
    textAlign: 'center',
  },
  headerRightIcons: {
    width: width * 0.06 + width * 0.02, // To balance the left icon
  },
  inspectionCard: {
    backgroundColor: '#F5F7FA',
    borderRadius: 8,
    padding: 12,
    marginBottom: 12,
    borderWidth: 1,
    borderColor: '#E0E0E0',
  },
  inspectionText: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 2,
  },
  inspectionLabel: {
    fontWeight: '700',
    color: '#2267A2',
  },
  modalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '75%',
    backgroundColor: '#FFF',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    maxWidth: 400,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '700',
    color: '#333',
    marginBottom: 16,
  },
  modalLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginBottom: 4,
  },
  modalValue: {
    fontSize: 14,
    color: '#333',
    marginBottom: 8,
    textAlign: 'center',
  },
  closeButton: {
    backgroundColor: '#3481BC',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 8,
    marginTop: 16,
  },
  closeButtonText: {
    fontSize: 16,
    fontWeight: '700',
    color: '#FFF',
  },
});

export default InspectionsClosed;