import React, { useEffect, useState } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, StatusBar, Dimensions, ScrollView, ActivityIndicator } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import DateTimePickerModal from 'react-native-modal-datetime-picker';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.32;

const EmployeesPaticipated = () => {
  const navigation = useNavigation();
  const [loading, setLoading] = useState(true);
  const [userTaskMap, setUserTaskMap] = useState([]); // [{user, jobs: [{jobCardNumber, inspectionNumber, tasks: []}]}]
  const [dateRange, setDateRange] = useState({ from: null, to: null });
  const [datePickerVisible, setDatePickerVisible] = useState(false);
  const [datePickerMode, setDatePickerMode] = useState('from');

  useEffect(() => {
    const fetchData = async () => {
      setLoading(true);
      try {
        // 1. Fetch all job cards globally
        const jobCardsSnap = await database().ref('/GlobalWorkOrders/admins').once('value');
        const adminsData = jobCardsSnap.val() || {};
        const allJobCards = [];
        Object.entries(adminsData).forEach(([adminUid, jobCardsObj]) => {
          if (jobCardsObj && typeof jobCardsObj === 'object') {
            Object.entries(jobCardsObj).forEach(([jobId, job]) => {
              if (jobId === 'lastRequestNumber') return;
              allJobCards.push({ ...job, adminUid, jobId });
            });
          }
        });
        // 2. Build user-task map
        const userMap = {};
        for (const job of allJobCards) {
          // Date filtering
          const createdAt = job.createdAt ? new Date(job.createdAt) : null;
          if (dateRange.from && (!createdAt || createdAt < dateRange.from)) continue;
          if (dateRange.to && (!createdAt || createdAt > dateRange.to)) continue;
          const jobCardNumber = job.requestNumber || job.title || job.jobCardNumber || job.jobId;
          const inspectionNumber = job.linkInspection?.name || job.linkInspection?.id || '-';
          const jobCardStatus = job.status || job.jobStatus || job.state || '-';
          const items = Array.isArray(job.items) ? job.items : [];
          for (const task of items) {
            if (Array.isArray(task.assignTo)) {
              for (const user of task.assignTo) {
                if (!user?.id) continue;
                if (!userMap[user.id]) {
                  userMap[user.id] = {
                    user,
                    jobs: [],
                  };
                }
                // Find if this job is already added for this user
                let jobEntry = userMap[user.id].jobs.find(j => j.jobCardNumber === jobCardNumber);
                if (!jobEntry) {
                  jobEntry = {
                    jobCardNumber,
                    inspectionNumber,
                    adminUid: job.adminUid,
                    jobId: job.jobId,
                    jobCardStatus, // Store status here
                    createdAt,
                    tasks: [],
                  };
                  userMap[user.id].jobs.push(jobEntry);
                }
                jobEntry.tasks.push({
                  taskId: task.id,
                  text: task.text,
                  note: task.note,
                  status: task.status,
                });
              }
            }
          }
        }
        // 3. Fetch user details for all user IDs
        const userIds = Object.keys(userMap);
        const usersSnap = await database().ref('/Globalusers/admins').once('value');
        const allUsers = usersSnap.val() || {};
        userIds.forEach(uid => {
          const userDetails = allUsers[uid];
          if (userDetails) {
            userMap[uid].user = {
              ...userMap[uid].user,
              fullName: userDetails.fullName || userDetails.name || userMap[uid].user.fullName || userMap[uid].user.name,
              employeeNumber: userDetails.employeeNumber || userMap[uid].user.employeeNumber,
              mobile: userDetails.mobile || userDetails.phone || '-',
            };
          }
        });
        setUserTaskMap(Object.values(userMap));
      } catch (e) {
        setUserTaskMap([]);
      }
      setLoading(false);
    };
    fetchData();
  }, [dateRange]);

  return (
    <View style={{ flex: 1, backgroundColor: '#fff' }}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()} style={styles.headerLeftIcon}>
            <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Employees Who Participated in Jobs in a Given Period</Text>
          <View style={styles.headerRightIcons} />
        </View>
      </LinearGradient>
      {/* Date Range Picker UI */}
      <View style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'center', marginTop: 10, marginBottom: 8 }}>
        <TouchableOpacity
          style={{ backgroundColor: '#F0F0F0', borderRadius: 8, padding: 10, marginRight: 8 }}
          onPress={() => { setDatePickerMode('from'); setDatePickerVisible(true); }}
        >
          <Text style={{ color: '#015185', fontWeight: '600' }}>{dateRange.from ? `From: ${dateRange.from.toLocaleDateString()}` : 'Select From Date'}</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={{ backgroundColor: '#F0F0F0', borderRadius: 8, padding: 10, marginRight: 8 }}
          onPress={() => { setDatePickerMode('to'); setDatePickerVisible(true); }}
        >
          <Text style={{ color: '#015185', fontWeight: '600' }}>{dateRange.to ? `To: ${dateRange.to.toLocaleDateString()}` : 'Select To Date'}</Text>
        </TouchableOpacity>
        {(dateRange.from || dateRange.to) && (
          <TouchableOpacity style={{ backgroundColor: '#D81B60', borderRadius: 8, padding: 10 }} onPress={() => setDateRange({ from: null, to: null })}>
            <Text style={{ color: '#fff', fontWeight: '600' }}>Clear</Text>
          </TouchableOpacity>
        )}
      </View>
      <ScrollView contentContainerStyle={{ padding: 16 }}>
        {loading ? (
          <ActivityIndicator size="large" color="#015185" style={{ marginTop: 32 }} />
        ) : userTaskMap.length === 0 ? (
          <Text style={{ textAlign: 'center', color: '#888', marginTop: 32 }}>No participation data found.</Text>
        ) : (
          userTaskMap.map((entry, idx) => {
            // Determine the most recent job card status for this user (from their jobs)
            let mainStatus = '-';
            let statusColor = '#888';
            if (entry.jobs.length > 0) {
              // Use the status of the most recent job (last in array)
              mainStatus = entry.jobs[entry.jobs.length - 1].jobCardStatus || '-';
              if (typeof mainStatus === 'string') {
                const s = mainStatus.toLowerCase();
                if (s === 'completed' || s === 'complete' || s === 'success') statusColor = '#2ecc40';
                else if (s === 'rejected') statusColor = '#d81b60';
                else if (s === 'processing' || s === 'in progress') statusColor = '#ff9800';
                else if (s === 'initialized' || s === 'on hold') statusColor = '#fbc02d';
              }
            }
            return (
              <View key={entry.user.id || idx} style={styles.userCard}>
                <View style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between' }}>
                  <Text style={styles.userName}>{entry.user.fullName || entry.user.name}</Text>
                  <Text style={[styles.statusText, { color: statusColor }]}>{mainStatus}</Text>
                </View>
                <Text style={styles.userInfo}>Emp#: {entry.user.employeeNumber || '-'} | Mobile: {entry.user.mobile || '-'}</Text>
                {entry.jobs.map((job, jdx) => {
                  // Determine status color
                  let status = job.status || job.jobStatus || job.state || '';
                  let statusColor = '#888';
                  if (typeof status === 'string') {
                    const s = status.toLowerCase();
                    if (s === 'completed' || s === 'complete' || s === 'success') statusColor = '#2ecc40';
                    else if (s === 'rejected') statusColor = '#d81b60';
                    else if (s === 'processing' || s === 'in progress') statusColor = '#ff9800';
                    else if (s === 'initialized' || s === 'on hold') statusColor = '#fbc02d';
                  }
                  return (
                    <View key={job.jobCardNumber + '-' + jdx} style={styles.jobCard}>
                      <Text style={styles.jobInfo}>Job Card: {job.jobCardNumber || '-'}</Text>
                      <Text style={styles.jobInfo}>Inspection Report: {job.inspectionNumber || '-'}</Text>
                      <View style={{ flexDirection: 'row', alignItems: 'center', marginBottom: 2 }}>
                        <Text style={styles.jobInfo}>Job Card Status: </Text>
                        <Text style={[styles.statusText, { color: statusColor }]}>{status || '-'}</Text>
                      </View>
                      <Text style={styles.jobInfo}>Tasks:</Text>
                      {job.tasks.map((task, tidx) => (
                        <Text key={task.taskId || tidx} style={styles.taskText}>- {task.text} ({task.status})</Text>
                      ))}
                    </View>
                  );
                })}
              </View>
            );
          })
        )}
      </ScrollView>
      <DateTimePickerModal
        isVisible={datePickerVisible}
        mode="date"
        onConfirm={date => {
          if (datePickerMode === 'from') {
            setDateRange(r => ({ ...r, from: date, to: null }));
          } else {
            setDateRange(r => ({ ...r, to: date }));
          }
          setDatePickerVisible(false);
        }}
        onCancel={() => setDatePickerVisible(false)}
      />
    </View>
  );
};

const styles = StyleSheet.create({
  header: {
    width: '100%',
    justifyContent: 'flex-end',
    paddingBottom: width * 0.04,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.05,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeftIcon: {
    paddingVertical: width * 0.02,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.055,
    fontWeight: '700',
    flex: 1,
    marginLeft: width * 0.03,
    textAlign: 'center',
  },
  headerRightIcons: {
    width: width * 0.06 + width * 0.02, // To balance the left icon
  },
  userCard: {
    backgroundColor: '#F5F7FA',
    borderRadius: 10,
    padding: 14,
    marginBottom: 18,
    borderWidth: 1,
    borderColor: '#E0E0E0',
  },
  userName: {
    fontSize: 17,
    fontWeight: '700',
    color: '#015185',
    marginBottom: 2,
  },
  userInfo: {
    fontSize: 14,
    color: '#2267A2',
    marginBottom: 6,
  },
  jobCard: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 10,
    marginBottom: 8,
    borderWidth: 1,
    borderColor: '#E0E0E0',
  },
  jobInfo: {
    fontSize: 14,
    color: '#015185',
    marginBottom: 2,
  },
  statusText: {
    fontSize: 14,
    fontWeight: '600',
  },
  taskText: {
    fontSize: 13,
    color: '#444',
    marginLeft: 8,
  },
});

export default EmployeesPaticipated;