import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Modal,
  Animated,
  FlatList,
  ScrollView,
  Clipboard,
  Alert,
  Share,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const TeamMemberHome = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [passwordModalVisible, setPasswordModalVisible] = useState(false);
  const [selectedMember, setSelectedMember] = useState(null);
  const [teamMembers, setTeamMembers] = useState([]);
  const [dropdownVisible, setDropdownVisible] = useState(false);
  const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0 });
  const [isOnline, setIsOnline] = useState(true);
  const [hasShownOfflineToast, setHasShownOfflineToast] = useState(false);
  const [hasShownOnlineToast, setHasShownOnlineToast] = useState(false);
  const [pendingActions, setPendingActions] = useState([]);
  const [selectedCountry, setSelectedCountry] = useState('All');
  const [selectedProject, setSelectedProject] = useState('All');
  const [selectedRole, setSelectedRole] = useState('All');
  const [countryDropdownVisible, setCountryDropdownVisible] = useState(false);
  const [projectDropdownVisible, setProjectDropdownVisible] = useState(false);
  const [roleDropdownVisible, setRoleDropdownVisible] = useState(false);
  const [countryDropdownPosition, setCountryDropdownPosition] = useState({ x: 0, y: 0, width: 0 });
  const [projectDropdownPosition, setProjectDropdownPosition] = useState({ x: 0, y: 0, width: 0 });
  const [roleDropdownPosition, setRoleDropdownPosition] = useState({ x: 0, y: 0, width: 0 });
  const [isLoading, setIsLoading] = useState(true);
  const [adminProfile, setAdminProfile] = useState(null);
  const [pwQueueStatusMap, setPwQueueStatusMap] = useState({}); // userId -> 'queued' | 'done' | 'error'
  const [userCountry, setUserCountry] = useState([]);
  const [userProject, setUserProject] = useState([]);
  const [userPermissions, setUserPermissions] = useState([]);
  const [permissionModalVisible, setPermissionModalVisible] = useState(false);
  const [profileModalVisible, setProfileModalVisible] = useState(false);
  const [currentPage, setCurrentPage] = useState(0);
  const [itemsPerPage] = useState(10);
  const [isLoadingMore, setIsLoadingMore] = useState(false);
  const [hasMoreData, setHasMoreData] = useState(true);
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const isMounted = useRef(true);
  const countryButtonRef = useRef(null);
  const projectButtonRef = useRef(null);
  const roleButtonRef = useRef(null);

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const showToast = (type, title, message) => {
    if (!isMounted.current) return;
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  const filterTeamMembersByPermissions = (members, userCountry, userProject, userPermissions, adminUid) => {
    let filteredMembers = [...members];
    if (userPermissions.includes('OnlyMineView')) {
      filteredMembers = filteredMembers.filter(member => member.id === adminUid);
    } else {
      filteredMembers = filteredMembers.filter(member => {
        const countryMatch = userCountry.length === 0 || (member.countries && member.countries.some(c => userCountry.includes(c)));
        const projectMatch = userProject.length === 0 || (member.projects && member.projects.some(p => userProject.includes(p)));
        return countryMatch && projectMatch;
      });
    }
    return filteredMembers;
  };

  const getFilterOptions = () => {
    const allCountries = Array.from(new Set(teamMembers.flatMap(member => member.countries || [])));
    const allProjects = Array.from(new Set(teamMembers.flatMap(member => member.projects || [])));
    const allowedCountries = userCountry.length > 0 ? allCountries.filter(c => userCountry.includes(c)) : allCountries;
    const allowedProjects = userProject.length > 0 ? allProjects.filter(p => userProject.includes(p)) : allProjects;
    const countries = ['All', ...allowedCountries];
    const projects = ['All', ...allowedProjects];
    const roles = ['All', ...new Set(teamMembers.map(member => member.role || 'Operator'))];
    const countryCounts = countries.reduce((acc, country) => {
      if (country === 'All') {
        acc[country] = teamMembers.length;
      } else {
        acc[country] = teamMembers.filter(member => member.countries?.includes(country)).length;
      }
      return acc;
    }, {});
    const projectCounts = projects.reduce((acc, project) => {
      if (project === 'All') {
        acc[project] = teamMembers.length;
      } else {
        acc[project] = teamMembers.filter(member => member.projects?.includes(project)).length;
      }
      return acc;
    }, {});
    const roleCounts = roles.reduce((acc, role) => {
      if (role === 'All') {
        acc[role] = teamMembers.length;
      } else {
        acc[role] = teamMembers.filter(member => member.role === role).length;
      }
      return acc;
    }, {});
    return { countries, projects, roles, countryCounts, projectCounts, roleCounts };
  };

  useEffect(() => {
    const initialize = async () => {
      setIsLoading(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();

      const myUid = await AsyncStorage.getItem('userUid');
      if (!myUid) {
        showToast('error', 'Error', 'Admin not authenticated.');
        navigation.navigate('AdminLoginScreen');
        return;
      }
      const adminRef = database().ref(`/Globalusers/admins/${myUid}`);
      const adminSnap = await adminRef.once('value');
      if (!adminSnap.exists()) {
        showToast('error', 'Error', 'User not found in admin database.');
        navigation.navigate('AdminLoginScreen');
        return;
      }
      const userData = adminSnap.val();
      setAdminProfile({
        ...userData,
        id: myUid,
        fullName: userData.name || 'N/A',
        employeeNumber: userData.employeeNumber || 'N/A',
        projects: userData.projects || [],
        countries: userData.countries || [],
        role: userData.role || 'Operator',
        isBlocked: userData.isBlocked || false,
        password: userData.password || 'N/A',
        profilePic: userData.profilePic || null,
      });
      setUserCountry(userData.countries || []);
      setUserProject(userData.projects || []);
      const teamEmployModule = userData.modules?.find(m => m.module === 'teamEmploy');
      const permissions = teamEmployModule?.permissions || [];
      setUserPermissions(permissions);

      const teamMembersRef = database().ref(`/Globalusers/admins`);
      const snapshot = await teamMembersRef.once('value');
      const data = snapshot.val();
      let membersArray = [];
      if (data) {
        membersArray = Object.keys(data).map((key) => ({
          ...data[key],
          id: key,
          fullName: data[key].name || 'N/A',
          employeeNumber: data[key].employeeNumber || 'N/A',
          projects: data[key].projects || [],
          countries: data[key].countries || [],
          role: data[key].role || 'Operator',
          isBlocked: data[key].isBlocked || false,
          password: data[key].password || 'N/A',
          profilePic: data[key].profilePic || null,
        }));
      }
      let filteredMembers = membersArray.filter(member => {
        const countryMatch = userData.countries.length === 0 || (member.countries && member.countries.some(c => userData.countries.includes(c)));
        const projectMatch = userData.projects.length === 0 || (member.projects && member.projects.some(p => userData.projects.includes(p)));
        return countryMatch && projectMatch;
      });
      filteredMembers = filteredMembers.filter(member => member.id !== myUid);
      setTeamMembers(filteredMembers);
      setIsLoading(false);
      rotateAnim.setValue(0);

      await loadPendingActions();

      // Listen for password update queue/status
      const queueRef = database().ref('/AdminTasks/authPasswordUpdates');
      queueRef.on('value', (snap) => {
        const val = snap.val();
        const map = {};
        if (val && typeof val === 'object') {
          Object.values(val).forEach((item) => {
            if (item && item.userId && item.status) {
              map[item.userId] = item.status; // queued | done | error
            }
          });
        }
        setPwQueueStatusMap(map);
      });

      const unsubscribeNetInfo = NetInfo.addEventListener(async (state) => {
        if (!isMounted.current) return;
        const newOnlineStatus = state.isConnected;
        setIsOnline(newOnlineStatus);
        if (!newOnlineStatus && !hasShownOfflineToast) {
          showToast('info', 'Offline Mode', 'Please connect to internet');
          setHasShownOfflineToast(true);
        } else if (newOnlineStatus && !hasShownOnlineToast) {
          showToast('success', 'Online', 'Now online, data synced');
          setHasShownOnlineToast(true);
          setHasShownOfflineToast(false);
          if (pendingActions.length > 0) {
            await syncPendingActions();
          }
        }
      });

      return () => {
        isMounted.current = false;
        unsubscribeNetInfo();
        try { queueRef.off(); } catch (e) {}
      };
    };

    initialize();
  }, []);

  // Auto-clear done/error banners after a short delay
  useEffect(() => {
    const timers = [];
    Object.entries(pwQueueStatusMap).forEach(([userId, status]) => {
      if (status === 'done' || status === 'error') {
        const t = setTimeout(() => {
          setPwQueueStatusMap((prev) => {
            const next = { ...prev };
            delete next[userId];
            return next;
          });
        }, 6000);
        timers.push(t);
      }
    });
    return () => timers.forEach(clearTimeout);
  }, [pwQueueStatusMap]);

  const loadPendingActions = async () => {
    try {
      const storedPending = await AsyncStorage.getItem('pendingActionsTeamMemberHome');
      if (storedPending) {
        const actions = JSON.parse(storedPending);
        setPendingActions(actions);
        console.log('Loaded pending actions:', actions);
      }
    } catch (error) {
      console.error('Error loading pending actions:', error);
      showToast('error', 'Error', 'Failed to load pending actions.');
    }
  };

  const loadTeamMembers = async (adminUid) => {
    try {
      if (!adminUid) {
        showToast('error', 'Error', 'Admin not authenticated.');
        navigation.navigate('AdminLoginScreen');
        return;
      }

      const online = await checkNetworkStatus();
      setIsOnline(online);

      if (!online) {
        const cachedMembers = await AsyncStorage.getItem('teamMembers');
        if (cachedMembers) {
          const members = JSON.parse(cachedMembers);
          const filtered = filterTeamMembersByPermissions(members, userCountry, userProject, userPermissions, adminUid);
          setTeamMembers(filtered);
          showToast('info', 'Offline Mode', 'Loaded cached team members.');
        }
        setIsLoading(false);
        rotateAnim.setValue(0);
        return;
      }

      const teamMembersRef = database().ref(`/Globalusers/admins`);
      teamMembersRef.on('value', (snapshot) => {
        if (!isMounted.current) return;
        const data = snapshot.val();
        if (data) {
          const membersArray = Object.keys(data).map((key) => ({
            ...data[key],
            id: key,
            fullName: data[key].name || 'N/A',
            employeeNumber: data[key].employeeNumber || 'N/A',
            projects: data[key].projects || [],
            countries: data[key].countries || [],
            role: data[key].role || 'Operator',
            isBlocked: data[key].isBlocked || false,
            password: data[key].password || 'N/A',
            profilePic: data[key].profilePic || null,
            driverLicenseNumber: data[key].driverLicenseNumber || '',
            driverLicenseCode: data[key].driverLicenseCode || '',
            driverLicenseExpiry: data[key].driverLicenseExpiry || '',
            dateCompletedFirstAid: data[key].dateCompletedFirstAid || '',
            expiryDateFirstAid: data[key].expiryDateFirstAid || '',
            fireFightingDateAttended: data[key].fireFightingDateAttended || '',
            blastingLicenseDateIssued: data[key].blastingLicenseDateIssued || '',
          }));
          const filtered = filterTeamMembersByPermissions(membersArray, userCountry, userProject, userPermissions, adminUid);
          setTeamMembers(filtered);
          AsyncStorage.setItem('teamMembers', JSON.stringify(filtered));
        } else {
          setTeamMembers([]);
          AsyncStorage.setItem('teamMembers', JSON.stringify([]));
        }
        setIsLoading(false);
        rotateAnim.setValue(0);
      });

      return () => teamMembersRef.off();
    } catch (error) {
      console.error('Error loading team members:', error);
      showToast('error', 'Error', 'Failed to load team members.');
      setIsLoading(false);
      rotateAnim.setValue(0);
    }
  };

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const syncPendingActions = async () => {
    if (!isOnline) return;
    try {
      const storedPending = await AsyncStorage.getItem('pendingActionsTeamMemberHome');
      const actions = storedPending ? JSON.parse(storedPending) : [];
      if (actions.length === 0) return;
      setSyncModalVisible(true);
      const adminUid = await AsyncStorage.getItem('userUid');
      for (const action of actions) {
        if (action.type === 'addTeamMember') {
          await handleAddTeamMemberAction(action.teamMember, adminUid);
        } else if (action.type === 'deleteTeamMember') {
          await handleDeleteTeamMemberAction(adminUid, action.memberId);
        }
      }
      setPendingActions([]);
      await AsyncStorage.setItem('pendingActionsTeamMemberHome', JSON.stringify([]));
      await loadTeamMembers(adminUid);
    } catch (error) {
      console.error('Error syncing pending actions:', error);
      showToast('error', 'Error', 'Failed to sync pending actions');
    } finally {
      setSyncModalVisible(false);
    }
  };

  const handleAddTeamMemberAction = async (teamMember, adminUid) => {
    try {
      const teamMemberId = database().ref(`/Globalusers/admins`).push().key;
      const globalTeamMemberData = { ...teamMember, adminUid, id: teamMemberId };
      await database().ref(`/Globalusers/admins/${teamMemberId}`).set({
        ...teamMember,
        id: teamMemberId,
        driverLicenseNumber: teamMember.driverLicenseNumber || '',
        driverLicenseCode: teamMember.driverLicenseCode || '',
        driverLicenseExpiry: teamMember.driverLicenseExpiry || '',
        dateCompletedFirstAid: teamMember.dateCompletedFirstAid || '',
        expiryDateFirstAid: teamMember.expiryDateFirstAid || '',
        fireFightingDateAttended: teamMember.fireFightingDateAttended || '',
        blastingLicenseDateIssued: teamMember.blastingLicenseDateIssued || '',
      });
      await database().ref(`/GlobalTeamMemberInformation/${teamMemberId}`).set(globalTeamMemberData);
      const storedTeam = await AsyncStorage.getItem('teamMembers');
      let teamMembers = storedTeam ? JSON.parse(storedTeam) : [];
      teamMembers.push({ ...teamMember, id: teamMemberId });
      const filtered = filterTeamMembersByPermissions(teamMembers, userCountry, userProject, userPermissions, adminUid);
      setTeamMembers(filtered);
      await AsyncStorage.setItem('teamMembers', JSON.stringify(filtered));
      console.log('Synced add team member action for:', teamMember.fullName);
    } catch (error) {
      console.error('Error syncing add team member action:', error);
      throw error;
    }
  };

  const handleDeleteTeamMemberAction = async (adminUid, memberId) => {
    try {
      await database().ref(`/Globalusers/admins/${memberId}`).remove();
      await database().ref(`/GlobalTeamMemberInformation/${memberId}`).remove();
      const storedTeam = await AsyncStorage.getItem('teamMembers');
      let teamMembers = storedTeam ? JSON.parse(storedTeam) : [];
      teamMembers = teamMembers.filter((member) => member.id !== memberId);
      const filtered = filterTeamMembersByPermissions(teamMembers, userCountry, userProject, userPermissions, adminUid);
      setTeamMembers(filtered);
      await AsyncStorage.setItem('teamMembers', JSON.stringify(filtered));
      console.log('Synced delete team member action for ID:', memberId);
    } catch (error) {
      console.error('Error syncing delete team member action:', error);
      throw error;
    }
  };

  const handleCloudSync = async () => {
    setSyncModalVisible(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();
    try {
      const online = await checkNetworkStatus();
      if (!online) {
        showToast('info', 'Offline Mode', 'Cannot sync while offline');
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        return;
      }
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        throw new Error('Admin not authenticated.');
      }
      const teamMembersRef = database().ref(`/Globalusers/admins`);
      const snapshot = await teamMembersRef.once('value');
      const data = snapshot.val();
      const membersArray = data
        ? Object.keys(data).map((key) => ({
            ...data[key],
            id: key,
            fullName: data[key].name || 'N/A',
            employeeNumber: data[key].employeeNumber || 'N/A',
            projects: data[key].projects || [],
            countries: data[key].countries || [],
            role: data[key].role || 'Operator',
            isBlocked: data[key].isBlocked || false,
            password: data[key].password || 'N/A',
            profilePic: data[key].profilePic || null,
            driverLicenseNumber: data[key].driverLicenseNumber || '',
            driverLicenseCode: data[key].driverLicenseCode || '',
            driverLicenseExpiry: data[key].driverLicenseExpiry || '',
            dateCompletedFirstAid: data[key].dateCompletedFirstAid || '',
            expiryDateFirstAid: data[key].expiryDateFirstAid || '',
            fireFightingDateAttended: data[key].fireFightingDateAttended || '',
            blastingLicenseDateIssued: data[key].blastingLicenseDateIssued || '',
          }))
        : [];
      const filtered = filterTeamMembersByPermissions(membersArray, userCountry, userProject, userPermissions, adminUid);
      await AsyncStorage.setItem('teamMembers', JSON.stringify(filtered));
      setTeamMembers(filtered);
      showToast('success', 'Success', 'Data synced with cloud successfully.');
    } catch (error) {
      console.error('Error syncing data:', error);
      showToast('error', 'Error', 'Failed to sync data with cloud.');
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }
  };

  const handleMemberPress = (item, event) => {
    setSelectedMember(item);
    setDropdownVisible(true);
    const screenWidth = Dimensions.get('window').width;
    const screenHeight = Dimensions.get('window').height;
    const dropdownWidth = 200;
    const dropdownHeight = 300;
    let x = event.nativeEvent.pageX - dropdownWidth / 2;
    let y = event.nativeEvent.pageY - 50;
    if (x < 10) x = 10;
    else if (x + dropdownWidth > screenWidth - 10) x = screenWidth - dropdownWidth - 10;
    if (y < 10) y = event.nativeEvent.pageY + 30;
    else if (y + dropdownHeight > screenHeight - 10) y = screenHeight - dropdownHeight - 10;
    setDropdownPosition({ x, y });
  };

  const handleDropdownAction = async (action) => {
    if (action === 'view-profile') {
      setProfileModalVisible(true);
    } else if (action === 'view-password') {
      setPasswordModalVisible(true);
    } else if (action === 'edit') {
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Editing requires internet connection.');
        setDropdownVisible(false);
        return;
      }
      navigation.navigate('Edit_team_member', { teamData: selectedMember });
    } else if (action === 'share') {
      if (!selectedMember) return;
      
      try {
        const name = selectedMember.fullName || 'N/A';
        const username = selectedMember.email || selectedMember.employeeNumber || 'N/A';
        const password = selectedMember.password || 'N/A';
        
        // Create an attractive formatted message
        const shareText = `🔐 **User Credentials** 🔐\n\n` +
          `👤 **Name:** ${name}\n` +
          `📧 **Username:** ${username}\n` +
          `🔑 **Password:** ${password}\n\n` +
          `📱 *Shared from Titan Drilling App*\n` +
          `⏰ *${new Date().toLocaleDateString()} ${new Date().toLocaleTimeString()}*`;
        
        // Use React Native's Share API
        Share.share({
          message: shareText,
          title: `User Credentials - ${name}`,
        }, {
          dialogTitle: `Share ${name}'s Credentials`,
          excludedActivityTypes: [],
          tintColor: '#3481BC',
        });
      } catch (error) {
        console.error('Error sharing user credentials:', error);
        showToast('error', 'Error', 'Failed to share user credentials.');
      }
    } else if (action === 'block') {
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Blocking requires internet connection.');
        setDropdownVisible(false);
        return;
      }
      try {
        const adminUid = await AsyncStorage.getItem('userUid');
        const newBlockedStatus = !selectedMember.isBlocked;
        await database()
          .ref(`/Globalusers/admins/${selectedMember.id}`)
          .update({ isBlocked: newBlockedStatus });
        await database()
          .ref(`/GlobalTeamMemberInformation/${selectedMember.id}`)
          .update({ isBlocked: newBlockedStatus });
        showToast('success', 'Success', `Team member ${newBlockedStatus ? 'blocked' : 'unblocked'} successfully.`);
        const updatedMembers = teamMembers.map((member) =>
          member.id === selectedMember.id ? { ...member, isBlocked: newBlockedStatus } : member
        );
        const filtered = filterTeamMembersByPermissions(updatedMembers, userCountry, userProject, userPermissions, adminUid);
        setTeamMembers(filtered);
        await AsyncStorage.setItem('teamMembers', JSON.stringify(filtered));
      } catch (error) {
        console.error('Error updating block status:', error);
        showToast('error', 'Error', 'Failed to update team member status.');
      }
    } else if (action === 'delete') {
      try {
        const confirmDelete = await new Promise((resolve) => {
          Alert.alert(
            'Confirm Delete',
            `Are you sure you want to delete ${selectedMember?.fullName || 'this user'}?`,
            [
              { text: 'Cancel', style: 'cancel', onPress: () => resolve(false) },
              { text: 'Delete', style: 'destructive', onPress: () => resolve(true) },
            ],
            { cancelable: true }
          );
        });
        if (!confirmDelete) return;

        const adminUid = await AsyncStorage.getItem('userUid');
        if (!isOnline) {
          const newPendingActions = [...pendingActions, { type: 'deleteTeamMember', adminUid, memberId: selectedMember.id }];
          setPendingActions(newPendingActions);
          await AsyncStorage.setItem('pendingActionsTeamMemberHome', JSON.stringify(newPendingActions));
          const updatedMembers = teamMembers.filter((member) => member.id !== selectedMember.id);
          const filtered = filterTeamMembersByPermissions(updatedMembers, userCountry, userProject, userPermissions, adminUid);
          setTeamMembers(filtered);
          await AsyncStorage.setItem('teamMembers', JSON.stringify(filtered));
          showToast('info', 'Offline Mode', 'Deletion will sync when online.');
        } else {
          await database().ref(`/Globalusers/admins/${selectedMember.id}`).remove();
          await database().ref(`/GlobalTeamMemberInformation/${selectedMember.id}`).remove();
          const updatedMembers = teamMembers.filter((member) => member.id !== selectedMember.id);
          const filtered = filterTeamMembersByPermissions(updatedMembers, userCountry, userProject, userPermissions, adminUid);
          setTeamMembers(filtered);
          await AsyncStorage.setItem('teamMembers', JSON.stringify(filtered));
          showToast('success', 'Success', 'Team member deleted successfully.');
        }
      } catch (error) {
        console.error('Error deleting team member:', error);
        showToast('error', 'Error', 'Failed to delete team member.');
      }
    } else if (action === 'give-permission') {
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Assigning permissions requires internet connection.');
        setDropdownVisible(false);
        return;
      }
      navigation.navigate('TEpremissionR', { member: selectedMember });
    }
    setDropdownVisible(false);
  };

  const handleResetToDefault = async () => {
    if (!selectedMember) return;
    
    try {
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Password reset requires internet connection.');
        return;
      }

      const newPassword = selectedMember.employeeNumber || 'default123';
      
      // Update password in Firebase
      await database()
        .ref(`/Globalusers/admins/${selectedMember.id}`)
        .update({ password: newPassword });
      
      await database()
        .ref(`/GlobalTeamMemberInformation/${selectedMember.id}`)
        .update({ password: newPassword });

      // Update local state
      const updatedMembers = teamMembers.map((member) =>
        member.id === selectedMember.id ? { ...member, password: newPassword } : member
      );
      setTeamMembers(updatedMembers);
      await AsyncStorage.setItem('teamMembers', JSON.stringify(updatedMembers));

      // Update selected member for modal display
      setSelectedMember({ ...selectedMember, password: newPassword });

      showToast('success', 'Success', 'Password reset to default successfully.');
    } catch (error) {
      console.error('Error resetting password:', error);
      showToast('error', 'Error', 'Failed to reset password.');
    }
  };

  const handleGenerateComplex = async () => {
    if (!selectedMember) return;
    try {
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Password generation requires internet connection.');
        return;
      }
      // Simple 8+ char generator: letters and numbers only
      const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
      let newPassword = '';
      for (let i = 0; i < 8; i++) {
        newPassword += chars.charAt(Math.floor(Math.random() * chars.length));
      }
      await database().ref(`/Globalusers/admins/${selectedMember.id}`).update({ password: newPassword });
      await database().ref(`/GlobalTeamMemberInformation/${selectedMember.id}`).update({ password: newPassword });
      const updatedMembers = teamMembers.map((member) => member.id === selectedMember.id ? { ...member, password: newPassword } : member);
      setTeamMembers(updatedMembers);
      await AsyncStorage.setItem('teamMembers', JSON.stringify(updatedMembers));
      setSelectedMember({ ...selectedMember, password: newPassword });
      showToast('success', 'Success', 'Password generated.');
    } catch (error) {
      console.error('Error generating password:', error);
      showToast('error', 'Error', 'Failed to generate password.');
    }
  };

  const handleCopyToClipboard = async () => {
    if (!selectedMember) return;
    
    try {
      const employeeId = selectedMember.employeeNumber || 'N/A';
      const email = selectedMember.email || 'N/A';
      const password = selectedMember.password || 'N/A';
      
      const clipboardText = `Employee ID: ${employeeId}\nEmail: ${email}\nPassword: ${password}`;
      
      await Clipboard.setString(clipboardText);
      showToast('success', 'Copied!', 'User information copied to clipboard.');
    } catch (error) {
      console.error('Error copying to clipboard:', error);
      showToast('error', 'Error', 'Failed to copy to clipboard.');
    }
  };

  const filteredTeamMembers = teamMembers.filter((member) => {
    const matchesSearch =
      (member.fullName?.toLowerCase().includes(searchQuery.toLowerCase()) ||
       member.employeeNumber?.toLowerCase().includes(searchQuery.toLowerCase()) ||
       member.role?.toLowerCase().includes(searchQuery.toLowerCase())) ?? false;
    const matchesCountry =
      selectedCountry === 'All' || (member.countries && member.countries.includes(selectedCountry));
    const matchesProject =
      selectedProject === 'All' || (member.projects && member.projects.includes(selectedProject));
    const matchesRole =
      selectedRole === 'All' || member.role === selectedRole;
    return matchesSearch && matchesCountry && matchesProject && matchesRole;
  });

  // Combine admin profile and team members for FlatList
  const hasOnlyMineView = userPermissions.some(p => p.toLowerCase() === 'onlymineview');
  const allCombinedMembers =
    hasOnlyMineView
      ? [adminProfile ? { ...adminProfile, isSelf: true } : null].filter(Boolean)
      : [
          adminProfile ? { ...adminProfile, isSelf: true } : null,
          ...filteredTeamMembers.map(m => ({ ...m, isSelf: false })),
        ].filter(Boolean);

  // Paginate the data
  const paginatedMembers = allCombinedMembers.slice(0, (currentPage + 1) * itemsPerPage);
  
  // Check if there's more data to load
  const totalItems = allCombinedMembers.length;
  const totalPages = Math.ceil(totalItems / itemsPerPage);
  const hasMore = currentPage + 1 < totalPages;

  // Load more data when scrolling reaches the end
  const loadMoreData = () => {
    if (hasMore && !isLoadingMore) {
      setIsLoadingMore(true);
      setTimeout(() => {
        setCurrentPage(prevPage => prevPage + 1);
        setIsLoadingMore(false);
      }, 500); // Small delay to show loading state
    }
  };

  // Reset pagination when filters change
  useEffect(() => {
    setCurrentPage(0);
    setIsLoadingMore(false);
  }, [searchQuery, selectedCountry, selectedProject, selectedRole]);

  // Render footer for loading more
  const renderFooter = () => {
    if (isLoadingMore) {
      return (
        <View style={styles.loadingFooter}>
          <Animated.Image
            source={require('../../../Images/adminlogin/sync.png')}
            style={[styles.loadingFooterIcon, { transform: [{ rotate: rotateInterpolate }] }]}
          />
          <Text style={styles.loadingFooterText}>Loading more...</Text>
        </View>
      );
    }
    
    if (!hasMore && totalItems > 0) {
      return (
        <View style={styles.paginationInfo}>
          <Text style={styles.paginationText}>
            Showing {totalItems} of {totalItems} team members
          </Text>
        </View>
      );
    }
    
    return null;
  };

  // Render a team member (admin or not)
  const renderTeamMember = ({ item }) => {
    const isSelf = item.isSelf;
    return (
      <View style={[
        styles.teamMemberContainer,
        isSelf ? { borderWidth: 1, borderColor: '#4CAF50', marginBottom: 8 } : {},
      ]}>
        <View style={styles.teamMemberLeft}>
          {item.profilePic && (
            <Image
              source={{ uri: item.profilePic.url }}
              style={styles.profileImage}
            />
          )}
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Employee Number: </Text>
            {item.employeeNumber || 'N/A'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Name: </Text>
            {item.fullName || 'N/A'}
            {isSelf && <Text style={{ color: '#4CAF50', fontWeight: 'bold' }}> (My Profile)</Text>}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Role: </Text>
            {item.role || 'Operator'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Projects: </Text>
            {item.projects?.length > 0 ? item.projects.join(', ') : 'None'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Country: </Text>
            {item.countries?.length > 0 ? item.countries.join(', ') : 'None'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Status: </Text>
            {item.isBlocked ? 'Blocked' : 'Active'}
          </Text>
          {pwQueueStatusMap[item.id] === 'queued' && (
            <Text style={[styles.teamMemberText, { color: '#ff9800', fontWeight: '600' }]}>Password update queued…</Text>
          )}
          {pwQueueStatusMap[item.id] === 'done' && (
            <Text style={[styles.teamMemberText, { color: '#2e7d32', fontWeight: '600' }]}>Password updated</Text>
          )}
          {pwQueueStatusMap[item.id] === 'error' && (
            <Text style={[styles.teamMemberText, { color: '#d32f2f', fontWeight: '600' }]}>Password update failed</Text>
          )}
        </View>
        <View style={styles.teamMemberRight}>
          <View style={styles.roleBadge}>
            <Text style={styles.teamMemberRole}>{item.role || 'Operator'}</Text>
          </View>
          <TouchableOpacity 
            style={styles.menuButton}
            onPress={(event) => handleMemberPress(item, event)}
          >
            <MaterialIcons name="more-vert" size={width * 0.06} color="#666" />
          </TouchableOpacity>
        </View>
      </View>
    );
  };

  // Dropdown logic: show only allowed options for self, all for others
  const renderDropdown = () => (
    <Modal visible={dropdownVisible} transparent animationType="none">
      <TouchableOpacity
        style={styles.dropdownOverlay}
        onPress={() => setDropdownVisible(false)}
      >
        <View
          style={[
            styles.dropdownContainer,
            { 
              top: dropdownPosition.y, 
              left: dropdownPosition.x,
              position: 'absolute',
            },
          ]}
        >
          {selectedMember?.isSelf ? (
            // My profile: show Edit (if allowed), View Password, Share, and Give Permission
            <>
              <TouchableOpacity
                style={styles.dropdownItem}
                onPress={() => handleDropdownAction('view-profile')}
              >
                <MaterialIcons name="person" size={width * 0.05} color="#333" />
                <Text style={styles.dropdownText}>View Profile</Text>
              </TouchableOpacity>
              {userPermissions.includes('edit') && (
                <TouchableOpacity
                  style={styles.dropdownItem}
                  onPress={() => handleDropdownAction('edit')}
                >
                  <MaterialIcons name="edit" size={width * 0.05} color="#333" />
                  <Text style={styles.dropdownText}>Edit</Text>
                </TouchableOpacity>
              )}
              <TouchableOpacity
                style={styles.dropdownItem}
                onPress={() => handleDropdownAction('share')}
              >
                <MaterialIcons name="share" size={width * 0.05} color="#333" />
                <Text style={styles.dropdownText}>Share</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.dropdownItem}
                onPress={() => handleDropdownAction('view-password')}
              >
                <MaterialIcons name="lock" size={width * 0.05} color="#333" />
                <Text style={styles.dropdownText}>View Password</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.dropdownItem}
                onPress={() => handleDropdownAction('give-permission')}
              >
                <MaterialIcons name="security" size={width * 0.05} color="#333" />
                <Text style={styles.dropdownText}>Give Permission</Text>
              </TouchableOpacity>
            </>
          ) : (
            // Other profiles: show all options as per permissions
            [
              { icon: 'person', text: 'View Profile', action: 'view-profile' },
              { icon: 'edit', text: 'Edit', action: 'edit', permission: 'edit' },
              { icon: 'share', text: 'Share', action: 'share' },
              { icon: 'lock', text: 'View Password', action: 'view-password' },
              { icon: 'security', text: 'Give Permission', action: 'give-permission' },
              { icon: 'block', text: selectedMember?.isBlocked ? 'Unblock Team Member' : 'Block Team Member', action: 'block', permission: 'edit' },
              { icon: 'delete', text: 'Delete', action: 'delete' },
            ].map((option) => {
              if (option.permission && !userPermissions.includes(option.permission)) {
                return null;
              }
              return (
                <TouchableOpacity
                  key={option.action}
                  style={styles.dropdownItem}
                  onPress={() => handleDropdownAction(option.action)}
                >
                  <MaterialIcons name={option.icon} size={width * 0.05} color="#333" />
                  <Text style={styles.dropdownText}>{option.text}</Text>
                </TouchableOpacity>
              );
            })
          )}
        </View>
      </TouchableOpacity>
    </Modal>
  );

  const handleFilterPress = (type) => {
    let ref;
    if (type === 'country') ref = countryButtonRef;
    else if (type === 'project') ref = projectButtonRef;
    else if (type === 'role') ref = roleButtonRef;

    if (ref && ref.current) {
      ref.current.measureInWindow((x, y, width, height) => {
        const left = x;
        const top = y + height + 4;
        if (type === 'country') {
          setCountryDropdownPosition({ x: left, y: top, width });
          setCountryDropdownVisible(true);
          setProjectDropdownVisible(false);
          setRoleDropdownVisible(false);
        } else if (type === 'project') {
          setProjectDropdownPosition({ x: left, y: top, width });
          setProjectDropdownVisible(true);
          setCountryDropdownVisible(false);
          setRoleDropdownVisible(false);
        } else if (type === 'role') {
          setRoleDropdownPosition({ x: left, y: top, width });
          setRoleDropdownVisible(true);
          setCountryDropdownVisible(false);
          setProjectDropdownVisible(false);
        }
      });
    }
  };

  const renderFilterDropdown = (type, options, counts, selected, setSelected, visible, setVisible) => {
    let position = { x: 0, y: 0, width: 0 };
    if (type === 'country') position = countryDropdownPosition;
    else if (type === 'project') position = projectDropdownPosition;
    else if (type === 'role') position = roleDropdownPosition;

    return (
      <Modal visible={visible} transparent animationType="none">
        <TouchableOpacity
          style={styles.dropdownOverlay}
          onPress={() => setVisible(false)}
          activeOpacity={1}
        >
          <View
            style={[
              styles.dropdownContainer,
              {
                top: position.y,
                left: position.x,
                width: position.width,
                minWidth: 80,
                maxWidth: 300,
              },
            ]}
          >
            <ScrollView
              nestedScrollEnabled={true}
              showsVerticalScrollIndicator={true}
              style={{ maxHeight: 200 }}
            >
              {options.map((option) => (
                <TouchableOpacity
                  key={option}
                  style={[
                    styles.dropdownItem,
                    selected === option && styles.dropdownItemActive,
                  ]}
                  onPress={() => {
                    setSelected(option);
                    setVisible(false);
                  }}
                >
                  <MaterialIcons
                    name={type === 'country' ? 'public' : type === 'project' ? 'work' : 'person'}
                    size={width * 0.05}
                    color={selected === option ? '#3481BC' : '#333'}
                  />
                  <Text
                    style={[
                      styles.dropdownText,
                      selected === option && styles.dropdownTextActive,
                    ]}
                  >
                    {option} ({counts[option] || 0})
                  </Text>
                </TouchableOpacity>
              ))}
            </ScrollView>
          </View>
        </TouchableOpacity>
      </Modal>
    );
  };

  const renderFilterBar = () => {
    const { countries, projects, roles, countryCounts, projectCounts, roleCounts } = getFilterOptions();
    return (
      <View style={{ backgroundColor: '#f5f5f5' }}>
        <ScrollView
          horizontal
          showsHorizontalScrollIndicator={false}
          contentContainerStyle={styles.filterBar}
        >
          <TouchableOpacity
            ref={countryButtonRef}
            style={styles.filterButton}
            onPress={() => handleFilterPress('country')}
          >
            <MaterialIcons name="public" size={width * 0.05} color="#333" style={styles.filterIcon} />
            <Text style={styles.filterButtonText}>
              Country: {selectedCountry} ({countryCounts[selectedCountry] || 0})
            </Text>
            <MaterialIcons name="arrow-drop-down" size={width * 0.05} color="#333" />
          </TouchableOpacity>
          <TouchableOpacity
            ref={projectButtonRef}
            style={styles.filterButton}
            onPress={() => handleFilterPress('project')}
          >
            <MaterialIcons name="work" size={width * 0.05} color="#333" style={styles.filterIcon} />
            <Text style={styles.filterButtonText}>
              Project: {selectedProject} ({projectCounts[selectedProject] || 0})
            </Text>
            <MaterialIcons name="arrow-drop-down" size={width * 0.05} color="#333" />
          </TouchableOpacity>
          <TouchableOpacity
            ref={roleButtonRef}
            style={styles.filterButton}
            onPress={() => handleFilterPress('role')}
          >
            <MaterialIcons name="person" size={width * 0.05} color="#333" style={styles.filterIcon} />
            <Text style={styles.filterButtonText}>
              Role: {selectedRole} ({roleCounts[selectedRole] || 0})
            </Text>
            <MaterialIcons name="arrow-drop-down" size={width * 0.05} color="#333" />
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.filterButton}
            onPress={() => setPermissionModalVisible(true)}
          >
            <MaterialIcons name="verified-user" size={width * 0.05} color="#333" style={styles.filterIcon} />
            <Text style={styles.filterButtonText}>Permissions</Text>
          </TouchableOpacity>
        </ScrollView>
        {renderFilterDropdown(
          'country',
          countries,
          countryCounts,
          selectedCountry,
          setSelectedCountry,
          countryDropdownVisible,
          setCountryDropdownVisible
        )}
        {renderFilterDropdown(
          'project',
          projects,
          projectCounts,
          selectedProject,
          setSelectedProject,
          projectDropdownVisible,
          setProjectDropdownVisible
        )}
        {renderFilterDropdown(
          'role',
          roles,
          roleCounts,
          selectedRole,
          setSelectedRole,
          roleDropdownVisible,
          setRoleDropdownVisible
        )}
      </View>
    );
  };

  const renderAdminProfile = () => {
    if (!adminProfile) return null;
    return (
      <View style={[styles.teamMemberContainer, { borderWidth: 1, borderColor: '#4CAF50', marginBottom: 8 }]}> 
        <View style={styles.teamMemberLeft}>
          {adminProfile.profilePic && (
            <Image
              source={{ uri: adminProfile.profilePic.url }}
              style={styles.profileImage}
            />
          )}
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Employee Number: </Text>
            {adminProfile.employeeNumber || 'N/A'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Name: </Text>
            {adminProfile.fullName || 'N/A'} <Text style={{ color: '#4CAF50', fontWeight: 'bold' }}>(My Profile)</Text>
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Role: </Text>
            {adminProfile.role || 'Operator'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Projects: </Text>
            {adminProfile.projects?.length > 0 ? adminProfile.projects.join(', ') : 'None'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Country: </Text>
            {adminProfile.countries?.length > 0 ? adminProfile.countries.join(', ') : 'None'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Status: </Text>
            {adminProfile.isBlocked ? 'Blocked' : 'Active'}
          </Text>
          {pwQueueStatusMap[adminProfile.id] === 'queued' && (
            <Text style={[styles.teamMemberText, { color: '#ff9800', fontWeight: '600' }]}>Password update queued…</Text>
          )}
          {pwQueueStatusMap[adminProfile.id] === 'done' && (
            <Text style={[styles.teamMemberText, { color: '#2e7d32', fontWeight: '600' }]}>Password updated</Text>
          )}
          {pwQueueStatusMap[adminProfile.id] === 'error' && (
            <Text style={[styles.teamMemberText, { color: '#d32f2f', fontWeight: '600' }]}>Password update failed</Text>
          )}
        </View>
        <View style={styles.teamMemberRight}>
          <View style={styles.roleBadge}>
            <Text style={styles.teamMemberRole}>{adminProfile.role || 'Operator'}</Text>
          </View>
          <TouchableOpacity 
            style={styles.menuButton}
            onPress={(event) => handleMemberPress(adminProfile, event)}
          >
            <MaterialIcons name="more-vert" size={width * 0.06} color="#666" />
          </TouchableOpacity>
        </View>
      </View>
    );
  };

  const renderPermissionModal = () => (
    <Modal visible={permissionModalVisible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={[styles.syncModalContent, { maxWidth: '90%', minWidth: '70%' }]}>
          <Text style={styles.syncModalText}>User Permissions</Text>
          <View style={styles.syncIcondiv}>
            <View style={{ flex: 1 }}>
              <Text style={styles.permissionLabel}>Country:</Text>
              <Text style={styles.permissionValue}>{userCountry.length > 0 ? userCountry.join(', ') : 'None'}</Text>
              <Text style={styles.permissionLabel}>Project:</Text>
              <Text style={styles.permissionValue}>{userProject.length > 0 ? userProject.join(', ') : 'None'}</Text>
              <Text style={styles.permissionLabel}>Permissions:</Text>
              <Text style={styles.permissionValue}>{userPermissions.length > 0 ? userPermissions.join(', ') : 'None'}</Text>
            </View>
          </View>
          <TouchableOpacity
            style={styles.closeButton}
            onPress={() => setPermissionModalVisible(false)}
          >
            <Text style={styles.closeButtonText}>Close</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );

  const renderProfileModal = () => (
    <Modal visible={profileModalVisible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={[styles.syncModalContent, { maxWidth: '95%', minWidth: '80%', maxHeight: '80%' }]}>
          <Text style={styles.syncModalText}>Team Member Profile</Text>
          <ScrollView style={{ width: '100%' }} showsVerticalScrollIndicator={false}>
            <View style={styles.syncIcondiv}>
              <View style={{ width: '100%' }}>
                {/* Profile Picture */}
                {selectedMember?.profilePic && (
                  <View style={styles.profileImageContainer}>
                    <Image
                      source={{ uri: selectedMember.profilePic.url }}
                      style={styles.profileModalImage}
                    />
                  </View>
                )}
                
                {/* Basic Information */}
                <View style={styles.profileSection}>
                  <Text style={styles.profileSectionTitle}>Basic Information</Text>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="badge" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Employee ID:</Text>
                    <Text style={styles.profileInfoValue}>{selectedMember?.employeeNumber || 'N/A'}</Text>
                  </View>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="person" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Full Name:</Text>
                    <Text style={styles.profileInfoValue}>{selectedMember?.fullName || 'N/A'}</Text>
                  </View>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="email" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Email:</Text>
                    <Text style={styles.profileInfoValue}>{selectedMember?.email || 'N/A'}</Text>
                  </View>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="work" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Role:</Text>
                    <Text style={styles.profileInfoValue}>{selectedMember?.role || 'N/A'}</Text>
                  </View>
                </View>

                {/* Projects & Countries */}
                <View style={styles.profileSection}>
                  <Text style={styles.profileSectionTitle}>Projects & Countries</Text>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="business" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Projects:</Text>
                    <Text style={styles.profileInfoValue}>
                      {selectedMember?.projects?.length > 0 ? selectedMember.projects.join(', ') : 'None'}
                    </Text>
                  </View>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="public" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Countries:</Text>
                    <Text style={styles.profileInfoValue}>
                      {selectedMember?.countries?.length > 0 ? selectedMember.countries.join(', ') : 'None'}
                    </Text>
                  </View>
                </View>

                {/* Status & Security */}
                <View style={styles.profileSection}>
                  <Text style={styles.profileSectionTitle}>Status & Security</Text>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="security" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Status:</Text>
                    <Text style={[styles.profileInfoValue, { 
                      color: selectedMember?.isBlocked ? '#d32f2f' : '#4CAF50',
                      fontWeight: 'bold'
                    }]}>
                      {selectedMember?.isBlocked ? 'Blocked' : 'Active'}
                    </Text>
                  </View>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="lock" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Password:</Text>
                    <Text style={styles.profileInfoValue}>{selectedMember?.password || 'N/A'}</Text>
                  </View>
                </View>

                {/* Additional Information */}
                {selectedMember?.driverLicenseNumber && (
                  <View style={styles.profileSection}>
                    <Text style={styles.profileSectionTitle}>Additional Information</Text>
                    <View style={styles.profileInfoRow}>
                      <MaterialIcons name="drive-eta" size={width * 0.05} color="#3481BC" />
                      <Text style={styles.profileInfoLabel}>Driver License:</Text>
                      <Text style={styles.profileInfoValue}>{selectedMember.driverLicenseNumber}</Text>
                    </View>
                    {selectedMember.driverLicenseCode && (
                      <View style={styles.profileInfoRow}>
                        <MaterialIcons name="code" size={width * 0.05} color="#3481BC" />
                        <Text style={styles.profileInfoLabel}>License Code:</Text>
                        <Text style={styles.profileInfoValue}>{selectedMember.driverLicenseCode}</Text>
                      </View>
                    )}
                    {selectedMember.driverLicenseExpiry && (
                      <View style={styles.profileInfoRow}>
                        <MaterialIcons name="event" size={width * 0.05} color="#3481BC" />
                        <Text style={styles.profileInfoLabel}>License Expiry:</Text>
                        <Text style={styles.profileInfoValue}>{selectedMember.driverLicenseExpiry}</Text>
                      </View>
                    )}
                  </View>
                )}

                {/* Training & Certifications */}
                {(selectedMember?.dateCompletedFirstAid || selectedMember?.fireFightingDateAttended || selectedMember?.blastingLicenseDateIssued) && (
                  <View style={styles.profileSection}>
                    <Text style={styles.profileSectionTitle}>Training & Certifications</Text>
                    {selectedMember.dateCompletedFirstAid && (
                      <View style={styles.profileInfoRow}>
                        <MaterialIcons name="healing" size={width * 0.05} color="#3481BC" />
                        <Text style={styles.profileInfoLabel}>First Aid Completed:</Text>
                        <Text style={styles.profileInfoValue}>{selectedMember.dateCompletedFirstAid}</Text>
                      </View>
                    )}
                    {selectedMember.expiryDateFirstAid && (
                      <View style={styles.profileInfoRow}>
                        <MaterialIcons name="schedule" size={width * 0.05} color="#3481BC" />
                        <Text style={styles.profileInfoLabel}>First Aid Expiry:</Text>
                        <Text style={styles.profileInfoValue}>{selectedMember.expiryDateFirstAid}</Text>
                      </View>
                    )}
                    {selectedMember.fireFightingDateAttended && (
                      <View style={styles.profileInfoRow}>
                        <MaterialIcons name="local-fire-department" size={width * 0.05} color="#3481BC" />
                        <Text style={styles.profileInfoLabel}>Fire Fighting Training:</Text>
                        <Text style={styles.profileInfoValue}>{selectedMember.fireFightingDateAttended}</Text>
                      </View>
                    )}
                    {selectedMember.blastingLicenseDateIssued && (
                      <View style={styles.profileInfoRow}>
                        <MaterialIcons name="explosion" size={width * 0.05} color="#3481BC" />
                        <Text style={styles.profileInfoLabel}>Blasting License:</Text>
                        <Text style={styles.profileInfoValue}>{selectedMember.blastingLicenseDateIssued}</Text>
                      </View>
                    )}
                  </View>
                )}
              </View>
            </View>
          </ScrollView>
          <TouchableOpacity
            style={styles.closeButton}
            onPress={() => setProfileModalVisible(false)}
          >
            <Text style={styles.closeButtonText}>Close</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );

  return (
    <View style={styles.container}>
      {isLoading && (
        <Modal visible={isLoading} transparent animationType="fade">
          <View style={styles.loadingModalContainer}>
            <View style={styles.loadingModalContent}>
              <Text style={styles.loadingModalText}>Please Wait</Text>
              <View style={styles.loadingIcondiv}>
                <Animated.Image
                  source={require('../../../Images/adminlogin/sync.png')}
                  style={[styles.loadingIcon, { transform: [{ rotate: rotateInterpolate }] }]}
                />
                <Text style={styles.loadingModalTextadd}>Processing data syncing</Text>
              </View>
            </View>
          </View>
        </Modal>
      )}
      
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Team and Operator Management</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity onPress={handleCloudSync} style={styles.refreshButton}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud.png')}
                style={styles.refreshIcon}
              />
            </TouchableOpacity>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by name, employee number, or role..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={setSearchQuery}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity
            style={styles.filterButton}
            onPress={() => setFilterModalVisible(true)}
          >
            <MaterialIcons name="filter-list" size={width * 0.06} color="black" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {renderFilterBar()}

      <FlatList
        data={paginatedMembers}
        renderItem={renderTeamMember}
        keyExtractor={(item) => item.id + (item.isSelf ? '_self' : '')}
        style={styles.teamList}
        onEndReached={loadMoreData}
        onEndReachedThreshold={0.1}
        ListFooterComponent={renderFooter}
        ListEmptyComponent={<Text style={styles.emptyText}>No team members found.</Text>}
        removeClippedSubviews={true}
        maxToRenderPerBatch={10}
        updateCellsBatchingPeriod={100}
        initialNumToRender={10}
        windowSize={10}
      />

      <View style={styles.buttonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
          {userPermissions.includes('create') && (
            <TouchableOpacity
              style={styles.startButton}
              onPress={() => navigation.navigate('Team_memberHeader')}
            >
              <Ionicons name="add" size={width * 0.06} color="#fff" />
              <Text style={styles.buttonText}>Add Team Member</Text>
            </TouchableOpacity>
          )}
        </LinearGradient>
      </View>

      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal visible={passwordModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>User Information</Text>
            <View style={styles.syncIcondiv}>
              <View style={{ width: '100%' }}>
                <View style={styles.infoRow}>
                  <MaterialIcons name="badge" size={width * 0.05} color="#3481BC" style={{ marginRight: width * 0.02 }} />
                  <Text style={styles.infoLabel}>Employee ID:</Text>
                  <Text style={styles.infoValue}>{selectedMember?.employeeNumber || 'N/A'}</Text>
                </View>
                <View style={styles.infoRow}>
                  <MaterialIcons name="email" size={width * 0.05} color="#3481BC" style={{ marginRight: width * 0.02 }} />
                  <Text style={styles.infoLabel}>Email:</Text>
                  <Text style={styles.infoValue}>{selectedMember?.email || 'N/A'}</Text>
                </View>
                <View style={styles.infoRow}>
                  <MaterialIcons name="lock" size={width * 0.05} color="#3481BC" style={{ marginRight: width * 0.02 }} />
                  <Text style={styles.infoLabel}>Password:</Text>
                  <Text style={styles.infoValue}>{selectedMember?.password || 'Password not available'}</Text>
                </View>
                {/* <View style={styles.passwordButtonsContainer}>
                  <TouchableOpacity
                    style={styles.passwordButton}
                    onPress={() => handleResetToDefault()}
                  >
                    <MaterialIcons name="refresh" size={width * 0.04} color="#fff" />
                    <Text style={styles.passwordButtonText}>Reset to Default</Text>
                  </TouchableOpacity>
                  <TouchableOpacity
                    style={styles.passwordButton}
                    onPress={() => handleGenerateComplex()}
                  >
                    <MaterialIcons name="security" size={width * 0.04} color="#fff" />
                    <Text style={styles.passwordButtonText}>Generate Complex</Text>
                  </TouchableOpacity>
                </View> */}
                <TouchableOpacity
                  style={styles.copyButton}
                  onPress={() => handleCopyToClipboard()}
                >
                  <MaterialIcons name="content-copy" size={width * 0.04} color="#fff" />
                  <Text style={styles.copyButtonText}>Copy All to Clipboard</Text>
                </TouchableOpacity>
              </View>
            </View>
            <TouchableOpacity
              style={styles.closeButton}
              onPress={() => setPasswordModalVisible(false)}
            >
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal visible={filterModalVisible} transparent animationType="slide">
        <View style={styles.filterModalContainer}>
          <View style={styles.filterModalContent}>
            <Text style={styles.filterModalTitle}>Filter Team Members</Text>
            <TouchableOpacity
              style={styles.closeButton}
              onPress={() => setFilterModalVisible(false)}
            >
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      {renderDropdown()}
      {renderPermissionModal()}
      {renderProfileModal()}
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.02,
    marginHorizontal: width * 0.01,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  filterButtonText: {
    fontSize: width * 0.035,
    color: '#333',
    marginLeft: width * 0.015,
    marginRight: width * 0.01,
  },
  filterIcon: {
    marginRight: width * 0.01,
  },
  filterBar: {
    flexDirection: 'row',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    maxWidth: 400,
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    justifyContent: 'center',
    width: '100%',
    marginBottom: 10,
  },
  permissionLabel: {
    fontWeight: 'bold',
    color: '#333',
    fontSize: width * 0.037,
    marginTop: 6,
    marginBottom: 2,
  },
  permissionValue: {
    color: '#333',
    fontSize: width * 0.037,
    marginBottom: 4,
    flexWrap: 'wrap',
    width: '100%',
    lineHeight: width * 0.045,
  },
  teamList: {
    flex: 1,
    marginTop: width * 0.02,
  },
  teamMemberContainer: {
    flexDirection: 'row',
    padding: width * 0.04,
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
    backgroundColor: '#fff',
    marginHorizontal: width * 0.02,
    marginVertical: width * 0.01,
    borderRadius: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  teamMemberLeft: {
    flex: 2,
    paddingRight: width * 0.02,
  },
  teamMemberRight: {
    flex: 1,
    alignItems: 'flex-end',
    justifyContent: 'space-between',
    flexDirection: 'column',
  },
  teamMemberText: {
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.015,
    lineHeight: width * 0.05,
  },
  labelText: {
    fontWeight: 'bold',
    color: '#333',
  },
  teamMemberRole: {
    fontSize: width * 0.035,
    fontWeight: 'bold',
    color: '#fff',
  },
  roleBadge: {
    backgroundColor: '#4CAF50',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: 12,
    alignItems: 'center',
    marginBottom: width * 0.02,
  },
  menuButton: {
    padding: width * 0.01,
    borderRadius: width * 0.02,
    backgroundColor: '#f5f5f5',
    borderWidth: 1,
    borderColor: '#ddd',
  },
  emptyText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'transparent',
  },
  dropdownContainer: {
    position: 'absolute',
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 5,
    width: 200,
    maxWidth: width * 0.8,
    minWidth: 150,
    zIndex: 1000,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  dropdownItemActive: {
    backgroundColor: '#e6f0ff',
  },
  dropdownText: {
    marginLeft: width * 0.025,
    fontSize: width * 0.04,
    color: '#333',
  },
  dropdownTextActive: {
    color: '#3481BC',
    fontWeight: 'bold',
  },
  closeButton: {
    backgroundColor: '#3481BC',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
    borderRadius: width * 0.012,
    marginTop: 10,
  },
  closeButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  filterModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  filterModalContent: {
    width: '80%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
  },
  filterModalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.025,
  },
  profileImage: {
    width: width * 0.1,
    height: width * 0.1,
    borderRadius: width * 0.05,
    marginBottom: width * 0.015,
  },
  loadingModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  loadingModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  loadingIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  loadingModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  loadingIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  infoRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: width * 0.025,
    paddingVertical: width * 0.01,
  },
  infoLabel: {
    fontSize: width * 0.035,
    fontWeight: 'bold',
    color: '#333',
    marginRight: width * 0.02,
    minWidth: 80,
  },
  infoValue: {
    fontSize: width * 0.035,
    color: '#666',
    flex: 1,
  },
  passwordButtonsContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginTop: width * 0.02,
    gap: width * 0.02,
  },
  passwordButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#3481BC',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.015,
    flex: 1,
    justifyContent: 'center',
  },
  passwordButtonText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: 'bold',
    marginLeft: width * 0.01,
  },
  copyButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#4CAF50',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.015,
    justifyContent: 'center',
    marginTop: width * 0.015,
  },
  copyButtonText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: 'bold',
    marginLeft: width * 0.01,
  },
  // Profile Modal Styles
  profileImageContainer: {
    alignItems: 'center',
    marginBottom: width * 0.03,
  },
  profileModalImage: {
    width: width * 0.15,
    height: width * 0.15,
    borderRadius: width * 0.075,
    borderWidth: 3,
    borderColor: '#3481BC',
  },
  profileSection: {
    marginBottom: width * 0.04,
    paddingHorizontal: width * 0.02,
  },
  profileSectionTitle: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#3481BC',
    marginBottom: width * 0.02,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    paddingBottom: width * 0.01,
  },
  profileInfoRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: width * 0.015,
    paddingVertical: width * 0.01,
  },
  profileInfoLabel: {
    fontSize: width * 0.035,
    fontWeight: 'bold',
    color: '#333',
    marginLeft: width * 0.02,
    marginRight: width * 0.02,
    minWidth: 120,
  },
  profileInfoValue: {
    fontSize: width * 0.035,
    color: '#666',
    flex: 1,
    lineHeight: width * 0.045,
  },
  // Pagination Loading Styles
  loadingFooter: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: width * 0.03,
    paddingHorizontal: width * 0.04,
  },
  loadingFooterIcon: {
    width: width * 0.05,
    height: width * 0.05,
    marginRight: width * 0.02,
  },
  loadingFooterText: {
    fontSize: width * 0.035,
    color: '#666',
    fontStyle: 'italic',
  },
  // Pagination Info Styles
  paginationInfo: {
    alignItems: 'center',
    paddingVertical: width * 0.03,
    paddingHorizontal: width * 0.04,
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
    backgroundColor: '#f8f9fa',
  },
  paginationText: {
    fontSize: width * 0.035,
    color: '#666',
    fontWeight: '500',
  },
});

export default TeamMemberHome;