
import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  Image,
  ScrollView,
  RefreshControl,
  Modal,
  Animated,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');

const Sync_Data = () => {
  const navigation = useNavigation();
  const [counts, setCounts] = useState({
    inspection: 0,
    workOrder: 0,
    dailyReporting: 0,
    incidentReport: 0,
  });
  const [refreshing, setRefreshing] = useState(false);
  const [lastUpdated, setLastUpdated] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [syncMessage, setSyncMessage] = useState('Loading Data...');

  // Animation for sync icon
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    fetchCounts();
  }, []);

  useFocusEffect(
    React.useCallback(() => {
      console.log('Sync_Data screen focused, navigation object:', navigation);
      if (navigation) {
        console.log('Navigation is available');
      } else {
        console.error('Navigation is not available on focus');
      }
    }, [navigation])
  );

  const fetchCounts = async () => {
    try {
      setSyncModalVisible(true);
      setSyncMessage('Fetching Data...');
      
      // Start rotation animation
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
      
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        Toast.show({
          type: 'error',
          text1: 'Authentication Error',
          text2: 'No user is logged in.',
          position: 'bottom',
        });
        setLoading(false);
        return;
      }

      // Fetch inspection reports count
      const inspectionRef = database().ref(`/GlobalInspectionReport/admins/${userId}`);
      const inspectionSnapshot = await inspectionRef.once('value');
      const inspectionData = inspectionSnapshot.val() || {};
      const inspectionCount = Object.keys(inspectionData).filter(key => key !== 'lastRequestNumber').length;

      // Fetch work orders count
      const workOrderRef = database().ref(`/GlobalWorkOrders/admins/${userId}`);
      const workOrderSnapshot = await workOrderRef.once('value');
      const workOrderData = workOrderSnapshot.val() || {};
      const workOrderCount = Object.keys(workOrderData).filter(key => key !== 'lastRequestNumber').length;

      // Fetch daily reports count
      const dailyReportRef = database().ref(`/GlobalDailyreportdata/admins/${userId}`);
      const dailyReportSnapshot = await dailyReportRef.once('value');
      const dailyReportData = dailyReportSnapshot.val() || {};
      const dailyReportCount = Object.keys(dailyReportData).length;

      // Fetch incident reports count
      const incidentRef = database().ref(`/incidentReports`);
      const incidentSnapshot = await incidentRef.once('value');
      const incidentData = incidentSnapshot.val() || {};
      const incidentCount = Object.keys(incidentData).filter(key => key !== 'lastIncidentNumber').length;

      setCounts({
        inspection: inspectionCount,
        workOrder: workOrderCount,
        dailyReporting: dailyReportCount,
        incidentReport: incidentCount,
      });
      
      setLastUpdated(new Date());
      
      Toast.show({
        type: 'success',
        text1: 'Data Updated',
        text2: 'All counts have been refreshed successfully.',
        position: 'bottom',
      });
    } catch (error) {
      console.error('Error fetching counts:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to fetch data counts. Please try again.',
        position: 'bottom',
      });
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }
  };

  const onRefresh = async () => {
    setRefreshing(true);
    setSyncModalVisible(true);
    setSyncMessage('Refreshing Data...');
    
    // Start rotation animation
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();
    
    await fetchCounts();
    setRefreshing(false);
    setSyncModalVisible(false);
    rotateAnim.setValue(0);
  };

    const handleModulePress = (moduleName) => {
    if (!navigation) {
      console.error('Navigation object is not available');
      Toast.show({
        type: 'error',
        text1: 'Navigation Error',
        text2: 'Navigation is not available.',
        position: 'bottom',
      });
      return;
    }
    
    try {
      console.log('Attempting to navigate to:', moduleName);
      console.log('Navigation object:', navigation);
      switch (moduleName) {
        case 'inspection':
          console.log('Navigating to Inspection_Report_Home');
          navigation.navigate('Inspection_Report_Home');
          break;
        case 'workOrder':
          console.log('Navigating to WorkOrdertab');
          navigation.navigate('WorkOrdertab');
          break;
        case 'dailyReporting':
          console.log('Navigating to DailyIns_report_home');
          navigation.navigate('DailyIns_report_home');
          break;
        case 'incidentReport':
          console.log('Navigating to IncodentReport');
          try {
            navigation.navigate('IncodentReport');
            console.log('Navigation successful');
          } catch (navError) {
            console.error('Specific navigation error for incident report:', navError);
            Toast.show({
              type: 'error',
              text1: 'Navigation Error',
              text2: 'Failed to navigate to Incident Report. Please try again.',
              position: 'bottom',
            });
          }
          break;
        default:
          console.log('Unknown module:', moduleName);
          Toast.show({
            type: 'error',
            text1: 'Navigation Error',
            text2: 'Module not found.',
            position: 'bottom',
          });
          break;
      }
    } catch (error) {
      console.error('Navigation error:', error);
      Toast.show({
        type: 'error',
        text1: 'Navigation Error',
        text2: 'Failed to navigate to the selected module.',
        position: 'bottom',
      });
    }
  };

  const renderModuleCard = (title, count, icon, moduleName, color) => (
    <TouchableOpacity
      style={styles.moduleCard}
      onPress={() => handleModulePress(moduleName)}
      activeOpacity={0.8}
    >
      <View style={styles.cardHeader}>
        <View style={[styles.iconContainer, { backgroundColor: color }]}>
          <MaterialIcons name={icon} size={28} color="#fff" />
        </View>
        <View style={styles.countContainer}>
          <Text style={styles.countText}>{count}</Text>
        </View>
      </View>
      <View style={styles.cardContent}>
        <Text style={styles.moduleTitle}>{title}</Text>
        <View style={styles.arrowContainer}>
          <MaterialIcons name="arrow-forward-ios" size={20} color="#015185" />
        </View>
      </View>
    </TouchableOpacity>
  );

  return (
    <View style={styles.container}>
      <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
      
      {/* Header */}
     

      {/* Main Content */}
      <ScrollView 
        style={styles.mainContent}
        refreshControl={
          <RefreshControl
            refreshing={refreshing}
            onRefresh={onRefresh}
            colors={['#015185']}
            tintColor="#015185"
          />
        }
        showsVerticalScrollIndicator={false}
      >
        <View style={styles.sectionHeader}>
          <View>
            <Text style={styles.sectionTitle}>Module Overview</Text>
            <Text style={styles.sectionSubtitle}>Click on any module to view details</Text>
        </View>
          <TouchableOpacity onPress={fetchCounts} style={styles.mainRefreshButton}>
            <Image 
              source={require('../../Images/adminhome/equp/cloud.png')} 
              style={styles.cloudIcon} 
            />
          </TouchableOpacity>
        </View>
        
        <View style={styles.modulesGrid}>
          {renderModuleCard(
            'Inspection',
            counts.inspection,
            'assignment',
            'inspection',
            '#2196F3'
          )}
          {renderModuleCard(
            'Work Order',
            counts.workOrder,
            'work',
            'workOrder',
            '#4CAF50'
          )}
          {renderModuleCard(
            'Daily Reporting',
            counts.dailyReporting,
            'today',
            'dailyReporting',
            '#FF9800'
          )}
          {renderModuleCard(
            'Incident Report',
            counts.incidentReport,
            'report-problem',
            'incidentReport',
            '#F44336'
          )}
                </View>
      </ScrollView>

      {/* Info Section at Bottom */}
        <View style={styles.infoContainer}>
        <MaterialIcons name="info" size={20} color="#015185" />
        <View style={styles.infoTextContainer}>
            <Text style={styles.infoText}>
            All data is automatically synced with the cloud. Pull down to refresh or tap the refresh button.
          </Text>
          {lastUpdated && (
            <Text style={styles.lastUpdatedText}>
              Last updated: {lastUpdated.toLocaleTimeString()}
            </Text>
          )}
        </View>
      </View>

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>{syncMessage}</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    paddingTop: StatusBar.currentHeight || 44,
    paddingBottom: 20,
    backgroundColor: 'transparent',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    marginTop: 10,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  backButton: {
    padding: 8,
  },
  headerTitle: {
    color: '#fff',
    fontSize: 24,
    fontWeight: 'bold',
    marginLeft: 16,
  },
  refreshButton: {
    padding: 8,
    backgroundColor: 'rgba(255, 255, 255, 0.2)',
    borderRadius: 20,
  },
  refreshIcon: {
    width: 24,
    height: 24,
    resizeMode: 'contain',
  },
  mainContent: {
    flex: 1,
    padding: 20,
    paddingBottom: 120,
  },
  sectionTitle: {
    fontSize: 22,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 8,
  },
  sectionSubtitle: {
    fontSize: 16,
    color: '#666',
    marginBottom: 0,
  },
  sectionHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'flex-start',
    marginBottom: 30,
  },
  mainRefreshButton: {
    padding: 8,
    backgroundColor: '#f0f8ff',
    borderRadius: 20,
    borderWidth: 1,
    borderColor: '#e3f2fd',
  },
  cloudIcon: {
    width: 20,
    height: 20,
    resizeMode: 'contain',
    tintColor: '#015185',
  },
  modulesGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'space-between',
    marginBottom: 30,
  },
  moduleCard: {
    width: '48%',
    backgroundColor: '#fff',
    borderRadius: 2,
    padding: 20,
    marginBottom: 16,
    elevation: 4,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.15,
    shadowRadius: 8,
    borderWidth: 0,
  },
  cardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 12,
  },
  iconContainer: {
    width: 48,
    height: 48,
    borderRadius: 24,
    justifyContent: 'center',
    alignItems: 'center',
  },
  countContainer: {
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 14,
    paddingVertical: 8,
    borderRadius: 20,
    minWidth: 44,
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#e3f2fd',
  },
  countText: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
  },
  cardContent: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  moduleTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    flex: 1,
  },
  arrowContainer: {
    marginLeft: 8,
  },

  infoContainer: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    backgroundColor: '#e3f2fd',
    padding: 20,
    borderRadius: 12,
    borderWidth: 1,
    borderColor: '#bbdefb',
    position: 'absolute',
    bottom: 20,
    left: 20,
    right: 20,
    zIndex: 1,
  },
  infoText: {
    fontSize: 14,
    color: '#1976d2',
    marginLeft: 12,
    flex: 1,
    lineHeight: 20,
  },
  infoTextContainer: {
    flex: 1,
    marginLeft: 12,
  },
  lastUpdatedText: {
    fontSize: 12,
    color: '#666',
    marginTop: 4,
    fontStyle: 'italic',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default Sync_Data;