import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Dimensions,
  StatusBar,
  Modal,
  FlatList,
  TouchableWithoutFeedback,
  Animated,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import DateTimePicker from '@react-native-community/datetimepicker';
import Toast from 'react-native-toast-message';
import auth from '@react-native-firebase/auth';
import database from '@react-native-firebase/database';
import NetInfo from '@react-native-community/netinfo';
import { Dropdown } from 'react-native-element-dropdown';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.3;

const CustomInputField = ({ label, value, onChangeText, isRequired = false, showError = false, errorMessage = '', style, ...props }) => (
  <View style={[inputStyles.container, style]}>
    <Text style={inputStyles.label}>
      {label}
      {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
    </Text>
    <TextInput
      style={[inputStyles.input, showError ? inputStyles.inputError : null]}
      value={value}
      onChangeText={onChangeText}
      placeholderTextColor="#888"
      {...props}
    />
    {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
  </View>
);

const UNIT_OPTIONS = ['Miles', 'Kilometers', 'Hours'];
const TIME_UNIT_OPTIONS = ['Day', 'Week', 'Month', 'Year'];

const AddSchedule = () => {
  const navigation = useNavigation();
  const [repeatType, setRepeatType] = useState('Repeat');
  const [intervals, setIntervals] = useState([
    { checked: true, value: '', unit: 'Miles' },
    { checked: false, value: '', unit: 'Miles' },
  ]);
  const [intervalUnit, setIntervalUnit] = useState('Day');
  const [notifyBefore, setNotifyBefore] = useState({ value: '', unit: 'Miles', time: 'Day' });
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [scheduleTitle, setScheduleTitle] = useState('');
  const [note, setNote] = useState('');
  const [lastServiceDate, setLastServiceDate] = useState(new Date());
  const [tempDate, setTempDate] = useState(lastServiceDate);
  const [datePickerVisible, setDatePickerVisible] = useState(false);
  const [scheduleId, setScheduleId] = useState('');
  const [showErrors, setShowErrors] = useState(false);
  const [meterReading, setMeterReading] = useState('');
  const [meterUnit, setMeterUnit] = useState('Miles');
  const [tasks, setTasks] = useState([]);
  const [taskModalVisible, setTaskModalVisible] = useState(false);
  const [newTask, setNewTask] = useState('');
  const [intervalDropdownIndex, setIntervalDropdownIndex] = useState(null);
  const [meterUnitDropdownVisible, setMeterUnitDropdownVisible] = useState(false);

  const [isLoading, setIsLoading] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const animationRef = useRef(null);

  // New state for country and project dropdowns
  const [selectedCountry, setSelectedCountry] = useState('');
  const [selectedProject, setSelectedProject] = useState('');
  const [userCountries, setUserCountries] = useState([]);
  const [userProjects, setUserProjects] = useState([]);

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Project mapping for all possible countries
  const projectMap = {
    Zambia: [
      { label: 'Kansanshi', value: 'Kansanshi' },
      { label: 'Kalumbila', value: 'Kalumbila' },
      { label: 'Kimteto', value: 'Kimteto' },
      { label: 'Kobold', value: 'Kobold' },
      { label: 'FQM Exploration', value: 'FQM Exploration' },
      { label: 'Mimosa', value: 'Mimosa' },
    ],
    DRC: [
      { label: 'Lubumbashi', value: 'Lubumbashi' },
      { label: 'Musompo', value: 'Musompo' },
      { label: 'IME', value: 'IME' },
      { label: 'Kamoa', value: 'Kamoa' },
    ],
  };

  // Generate country options from user's countries
  const getCountryOptions = () => {
    return userCountries.map(country => ({
      label: country,
      value: country
    }));
  };

  const getProjectOptions = () => {
    if (!selectedCountry) return [];
    
    // Filter projects based on user's projects for the selected country
    const allProjectsForCountry = projectMap[selectedCountry] || [];
    const userProjectsForCountry = allProjectsForCountry.filter(project => 
      userProjects.includes(project.value)
    );
    
    return userProjectsForCountry;
  };

  // Fetch user data and generate schedule ID
  useEffect(() => {
    const unsubscribeAuth = auth().onAuthStateChanged((user) => {
      if (!user) {
        showToast('error', 'Error', 'User not authenticated.');
        navigation.navigate('AdminLoginScreen');
        return;
      }
      fetchUserData(user.uid);
    });

    const fetchUserData = async (uid) => {
      try {
        const userRef = database().ref(`/Globalusers/admins/${uid}`);
        const snapshot = await userRef.once('value');
        if (snapshot.exists()) {
          const userData = snapshot.val();
          
          // Normalize countries and projects to arrays
          const countries = Array.isArray(userData.countries) ? userData.countries : 
                          (userData.countries ? Object.values(userData.countries) : []);
          const projects = Array.isArray(userData.projects) ? userData.projects : 
                          (userData.projects ? Object.values(userData.projects) : []);
          
          console.log('User countries from Firebase:', countries);
          console.log('User projects from Firebase:', projects);
          
          setUserCountries(countries);
          setUserProjects(projects);
          
          // Set default country and project if user has only one
          if (countries.length === 1) {
            setSelectedCountry(countries[0]);
            console.log('Auto-selected country:', countries[0]);
          }
          if (projects.length === 1) {
            setSelectedProject(projects[0]);
            console.log('Auto-selected project:', projects[0]);
          }
        } else {
          console.log('No user data found for UID:', uid);
        }
      } catch (error) {
        console.error('Error fetching user data:', error);
        showToast('error', 'Error', 'Failed to load user data');
      }
    };

    const unsubscribeNetInfo = NetInfo.addEventListener((state) => {
      if (state.isConnected && state.isInternetReachable) {
        syncOfflineSchedules();
      }
    });

    return () => {
      unsubscribeAuth();
      unsubscribeNetInfo();
    };
  }, []);

  // Generate schedule ID preview when country and project are selected (without incrementing counter)
  useEffect(() => {
    if (selectedCountry && selectedProject) {
      generateScheduleIdPreview();
    }
  }, [selectedCountry, selectedProject]);

  // Helper functions for country and project codes
  const getCountryCode = (country) => {
    const countryCodes = {
      'Zambia': 'ZMB',
      'DRC': 'DRC',
    };
    return countryCodes[country] || country.substring(0, 3).toUpperCase();
  };

  const getProjectCode = (project) => {
    // Take first 3 characters of the project name and convert to uppercase
    if (!project) return 'XXX';
    return project.substring(0, 3).toUpperCase();
  };

  // Generate schedule ID preview (without incrementing counter)
  const generateScheduleIdPreview = async () => {
    try {
      const user = auth().currentUser;
      if (!user) return;

      // Get current counter from Firebase (without incrementing)
      const counterRef = database().ref('/GlobalServiceScheduled/counter');
      const snapshot = await counterRef.once('value');
      const currentCounter = snapshot.val() || 0;
      const nextCounter = currentCounter + 1; // Show next number but don't save it

      // Generate schedule ID preview
      const countryCode = getCountryCode(selectedCountry);
      const projectCode = getProjectCode(selectedProject);
      const formattedCounter = nextCounter.toString().padStart(3, '0');
      const previewScheduleId = `${countryCode}-${projectCode}-SS-${formattedCounter}`;
      
      console.log('Schedule ID Preview:', {
        country: selectedCountry,
        countryCode,
        project: selectedProject,
        projectCode,
        nextCounter,
        scheduleId: previewScheduleId
      });
      
      setScheduleId(previewScheduleId);
    } catch (error) {
      console.error('Error generating schedule ID preview:', error);
      // Fallback preview ID if there's an error
      const countryCode = selectedCountry ? selectedCountry.substring(0, 3).toUpperCase() : 'XXX';
      const projectCode = selectedProject ? selectedProject.substring(0, 3).toUpperCase() : 'XXX';
      setScheduleId(`${countryCode}-${projectCode}-SS-XXX`);
    }
  };

  // Generate actual schedule ID (increments counter) - called only when creating schedule
  const generateScheduleId = async () => {
    try {
      const user = auth().currentUser;
      if (!user) return;

      // Get global counter from Firebase
      const counterRef = database().ref('/GlobalServiceScheduled/counter');
      const snapshot = await counterRef.once('value');
      const currentCounter = snapshot.val() || 0;
      const newCounter = currentCounter + 1;

      // Update the global counter
      await counterRef.set(newCounter);

      // Generate actual schedule ID
      const countryCode = getCountryCode(selectedCountry);
      const projectCode = getProjectCode(selectedProject);
      const formattedCounter = newCounter.toString().padStart(3, '0');
      const newScheduleId = `${countryCode}-${projectCode}-SS-${formattedCounter}`;
      
      console.log('Generated Actual Schedule ID:', {
        country: selectedCountry,
        countryCode,
        project: selectedProject,
        projectCode,
        counter: newCounter,
        scheduleId: newScheduleId
      });
      
      return newScheduleId;
    } catch (error) {
      console.error('Error generating actual schedule ID:', error);
      // Fallback ID if there's an error
      const countryCode = selectedCountry ? selectedCountry.substring(0, 3).toUpperCase() : 'XXX';
      const projectCode = selectedProject ? selectedProject.substring(0, 3).toUpperCase() : 'XXX';
      return `${countryCode}-${projectCode}-SS-${Date.now()}`;
    }
  };

  const handleEquipmentSelect = (equipmentData) => {
    // Ensure equipmentNumber is always set if possible
    let equipmentNumber = equipmentData.equipmentNumber;
    if (!equipmentNumber && equipmentData.entryNumber) {
      equipmentNumber = equipmentData.entryNumber;
    }
    setSelectedEquipment({ ...equipmentData, equipmentNumber });
    setMeterReading(equipmentData?.meterReading?.toString() || '0');
    setMeterUnit(equipmentData?.meterUnit || 'Miles');
    setNotifyBefore({ ...notifyBefore, unit: equipmentData?.meterUnit || 'Miles' });
    intervals.forEach((_, idx) => {
      handleUnitSelect(equipmentData?.meterUnit || 'Miles', idx);
    });
  };

  const formatDate = (date) => {
    if (!date || isNaN(date.getTime())) return 'Select Date';
    return date.toLocaleDateString('en-GB', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
    }).replace(/ /g, '-');
  };

  const openDatePicker = () => {
    setTempDate(lastServiceDate || new Date());
    setDatePickerVisible(true);
  };

  const handleDateCancel = () => {
    setDatePickerVisible(false);
  };

  const openTaskModal = () => {
    setNewTask('');
    setTaskModalVisible(true);
  };

  const handleTaskConfirm = () => {
    if (newTask.trim()) {
      setTasks([...tasks, newTask.trim()]);
    }
    setTaskModalVisible(false);
    setNewTask('');
  };

  const handleTaskCancel = () => {
    setTaskModalVisible(false);
    setNewTask('');
  };

  const handleUnitSelect = (unit, index) => {
    setIntervals(intervals.map((item, i) => (i === index ? { ...item, unit } : item)));
    setIntervalDropdownIndex(null);
  };



  const handleMeterUnitSelect = (unit) => {
    setMeterUnit(unit);
    setNotifyBefore({ ...notifyBefore, unit });
    intervals.forEach((_, idx) => {
      handleUnitSelect(unit, idx);
    });
    setMeterUnitDropdownVisible(false);
  };

  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  // Function to send service schedule notifications (both email and push)
  const sendServiceScheduleNotifications = async (scheduleData) => {
    try {
      const {
        scheduleId,
        scheduleNumber,
        title,
        country,
        project,
        equipment,
        repeatType,
        tasks,
        creatorId
      } = scheduleData;

      // Get creator information
      const creatorSnapshot = await database().ref(`/Globalusers/admins/${creatorId}`).once('value');
      const creatorData = creatorSnapshot.val();
      const creatorName = creatorData?.name || 'Unknown User';

      console.log('📧 Service schedule emails will be sent automatically by Firebase function');

      // SEND IN-APP NOTIFICATIONS
      const adminsSnapshot = await database().ref('/Globalusers/admins').once('value');
      if (!adminsSnapshot.exists()) {
        console.log('No admin users found for notifications');
        return;
      }

      const allAdmins = adminsSnapshot.val();
      const notificationData = {
        type: 'service_schedule_created',
        title: `📅 New Service Schedule: ${scheduleNumber}`,
        body: `${creatorName} created service schedule for ${project} - ${equipment?.equipmentName || 'Equipment'}`,
        timestamp: Date.now(),
        isRead: false,
        scheduleNumber: scheduleNumber,
        country: country,
        project: project,
        equipment: equipment?.equipmentName || 'N/A',
        repeatType: repeatType,
        tasks: tasks ? tasks.length : 0,
        creatorName: creatorName,
        data: {
          type: 'service_schedule_created',
          scheduleId: scheduleId,
          scheduleNumber: scheduleNumber,
          title: title,
          country: country,
          project: project,
          equipment: equipment?.equipmentName || 'N/A',
          repeatType: repeatType,
          tasks: tasks ? tasks.length : 0,
          creatorName: creatorName,
          action: 'view_service_schedule'
        }
      };

      // Send notifications to all admins with proper roles
      let notificationsSent = 0;
      for (const [adminId, adminData] of Object.entries(allAdmins)) {
        // Check if user should receive service schedule notifications
        const userRole = adminData.role || '';
        const isEligibleRole = ['admin', 'ProAdmin', 'Supervisor', 'CEO', 'Manager'].some(role => 
          userRole.toLowerCase().includes(role.toLowerCase())
        ) || userRole.toLowerCase().includes('manager');

        if (isEligibleRole) {
          try {
            const notificationRef = database().ref(`/Globalusers/admins/${adminId}/notifications`).push();
            await notificationRef.set(notificationData);
            notificationsSent++;
            console.log(`📱 In-app notification sent to: ${adminData.name || 'Unknown'} (${userRole})`);
          } catch (error) {
            console.error(`❌ Failed to send notification to ${adminId}:`, error);
          }
        }
      }

      console.log(`✅ Service schedule in-app notifications sent to ${notificationsSent} users`);

      // SEND PUSH NOTIFICATIONS (using the proper FCM system)
      try {
        console.log('🔔 Sending FCM push notifications...');
        
        // Send push notifications to all eligible users with FCM tokens
        let pushNotificationsSent = 0;
        for (const [adminId, adminData] of Object.entries(allAdmins)) {
          const userRole = adminData.role || '';
          const isEligibleRole = ['admin', 'ProAdmin', 'Supervisor', 'CEO', 'Manager'].some(role => 
            userRole.toLowerCase().includes(role.toLowerCase())
          ) || userRole.toLowerCase().includes('manager');

          if (isEligibleRole && adminData.devices) {
            // Send to all active devices for this user
            for (const [deviceKey, deviceData] of Object.entries(adminData.devices)) {
              if (deviceData.token && deviceData.fcmStatus === 'active') {
                try {
                  const response = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/sendNotificationToAllUserDevices', {
                    method: 'POST',
                    headers: {
                      'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                      userId: adminId,
                      title: notificationData.title,
                      body: notificationData.body,
                      data: notificationData.data
                    }),
                  });

                  if (response.ok) {
                    const result = await response.json();
                    if (result.success) {
                      pushNotificationsSent += result.successfulDevices || 0;
                      console.log(`🔔 Push notification sent to: ${adminData.name || 'Unknown'} - ${result.successfulDevices || 0} devices`);
                    }
                  }
                  break; // Only send once per user (the function handles all devices)
                } catch (pushError) {
                  console.error(`❌ Error sending push to ${adminData.name}:`, pushError);
                }
              }
            }
          }
        }

        console.log(`✅ FCM push notifications sent to ${pushNotificationsSent} devices`);

      } catch (pushError) {
        console.error('❌ Error sending push notifications:', pushError);
      }

    } catch (error) {
      console.error('❌ Error in sendServiceScheduleNotifications:', error);
      throw error;
    }
  };

  const storeScheduleOffline = async (scheduleData) => {
    try {
      const offlineSchedules = await AsyncStorage.getItem('offlineSchedules');
      const schedules = offlineSchedules ? JSON.parse(offlineSchedules) : [];
      schedules.push(scheduleData);
      await AsyncStorage.setItem('offlineSchedules', JSON.stringify(schedules));
      showToast('success', 'Success', 'Schedule stored offline');
      return true;
    } catch (error) {
      console.error('Error storing schedule offline:', error);
      showToast('error', 'Error', 'Failed to store schedule offline');
      return false;
    }
  };

  const syncOfflineSchedules = async () => {
    try {
      const offlineSchedules = await AsyncStorage.getItem('offlineSchedules');
      if (!offlineSchedules) return;
      const schedules = JSON.parse(offlineSchedules);
      if (schedules.length === 0) return;

      const user = auth().currentUser;
      if (!user) {
        console.error('User not authenticated during sync');
        return;
      }
      const uid = user.uid;

      for (const schedule of schedules) {
        try {
          const scheduleRef = database().ref(`/GlobalServiceScheduled/admins/${uid}/serverschedule/${schedule.id}`);
          await scheduleRef.set(schedule);
        } catch (error) {
          console.error('Error syncing schedule:', error);
        }
      }

      await AsyncStorage.setItem('offlineSchedules', JSON.stringify([]));
      showToast('success', 'Success', 'Offline schedules synced to database');
    } catch (error) {
      console.error('Error syncing offline schedules:', error);
      showToast('error', 'Error', 'Failed to sync offline schedules');
    }
  };

  const handleCreateSchedule = async () => {
    setShowErrors(true);

    // Fix: Validate intervals for repeating schedules
    const validIntervals = intervals.filter((interval) => interval.checked && interval.value);
    if (
      !selectedCountry ||
      !selectedProject ||
      !scheduleTitle ||
      !note ||
      !selectedEquipment ||
      !meterReading ||
      isNaN(parseFloat(meterReading)) ||
      !lastServiceDate ||
      isNaN(lastServiceDate.getTime()) ||
      !tasks.length ||
      !repeatType ||
      (repeatType === 'Repeat' && validIntervals.length === 0)
    ) {
      showToast('error', 'Validation Error', 'Please fill all required fields, including Country, Project, and at least one interval for repeating schedules.');
      setIsLoading(false);
      return;
    }

    setIsLoading(true);
    animationRef.current = Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    );
    animationRef.current.start();

    // Generate actual schedule ID (increments counter)
    const actualScheduleId = await generateScheduleId();
    
    const scheduleData = {
      id: Date.now().toString(),
      scheduleId: actualScheduleId,
      country: selectedCountry,
      project: selectedProject,
      title: scheduleTitle,
      equipment: {
        entryNumber: selectedEquipment.entryNumber || '',
        equipmentName: selectedEquipment.equipmentName || 'TYRE - TYRE',
        equipmentNumber: selectedEquipment.equipmentNumber || '', // Ensure equipmentNumber is always set
        meterReading: parseFloat(meterReading) || 0,
        meterUnit: meterUnit,
      },
      note,
      lastServiceDate: formatDate(lastServiceDate),
      meterReading: parseFloat(meterReading) || 0,
      meterUnit,
      repeatType,
      intervals: validIntervals,
      intervalUnit,
      notifyBefore,
      tasks,
      status: 'upcoming',
      createdAt: new Date().toISOString(),
      dueHours: '12',
      hoursToGo: '10',
      daysToGo: '69',
    };

    try {
      const state = await NetInfo.fetch();
      if (state.isConnected && state.isInternetReachable) {
        const user = auth().currentUser;
        if (!user) {
          showToast('error', 'Error', 'User not authenticated. Please log in again.');
          navigation.navigate('AdminLoginScreen');
          return;
        }
        const uid = user.uid;

        const scheduleRef = database().ref(`/GlobalServiceScheduled/admins/${uid}/serverschedule/${scheduleData.id}`);
        await scheduleRef.set(scheduleData);

        // Send service schedule notifications
        try {
          console.log('📱 Sending service schedule notifications...');
          await sendServiceScheduleNotifications({
            scheduleId: scheduleData.id,
            scheduleNumber: actualScheduleId,
            title: scheduleTitle,
            country: selectedCountry,
            project: selectedProject,
            equipment: selectedEquipment,
            repeatType: repeatType,
            tasks: tasks,
            creatorId: uid
          });
          console.log('✅ Service schedule notifications sent successfully');
        } catch (notificationError) {
          console.error('❌ Error sending service schedule notifications:', notificationError);
          // Don't fail the schedule creation if notifications fail
        }

        showToast('success', 'Success', 'Schedule created successfully.');
        navigation.replace('ServiceScheduleHome');
      } else {
        const success = await storeScheduleOffline(scheduleData);
        if (success) {
          showToast('info', 'Offline', 'Schedule stored offline. Will sync when online.');
          navigation.replace('ServiceScheduleHome');
        }
      }
    } catch (error) {
      console.error('Error saving schedule:', error);
      const success = await storeScheduleOffline(scheduleData);
      if (success) {
        showToast('info', 'Offline', 'Schedule stored offline due to error. Will sync when online.');
        navigation.replace('ServiceScheduleHome');
      } else {
        showToast('error', 'Error', 'Failed to create schedule.');
      }
    } finally {
      setIsLoading(false);
      if (animationRef.current) {
        animationRef.current.stop();
        animationRef.current = null;
      }
      rotateAnim.setValue(0);
    }
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Add Schedule</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={handleCreateSchedule}>
              <Text style={styles.createButtonText}>Create</Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <ScrollView style={styles.mainContent} keyboardShouldPersistTaps="handled">
        <View style={styles.scheduleIdRow}>
          <Text style={styles.scheduleIdLabel}>Schedule ID</Text>
          <Text style={styles.scheduleIdValue}>{scheduleId}</Text>
        </View>

        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>
            Country
            <Text style={inputStyles.requiredError}> *</Text>
          </Text>
          <Dropdown
            style={[inputStyles.input, showErrors && !selectedCountry ? inputStyles.inputError : null]}
            placeholderStyle={inputStyles.placeholderStyle}
            selectedTextStyle={inputStyles.selectedTextStyle}
            data={getCountryOptions()}
            maxHeight={200}
            labelField="label"
            valueField="value"
            placeholder={userCountries.length > 0 ? "Select Country" : "Loading countries..."}
            value={selectedCountry}
            onChange={(item) => {
              setSelectedCountry(item.value);
              setSelectedProject(''); // Reset project when country changes
              setScheduleId(''); // Reset schedule ID preview
            }}
            disable={userCountries.length === 0}
            renderLeftIcon={() => (
              <MaterialIcons 
                style={inputStyles.iconStyle} 
                color="#015185" 
                name="public" 
                size={20} 
              />
            )}
            renderRightIcon={() => (
              <MaterialIcons 
                style={inputStyles.iconStyle} 
                color="#015185" 
                name="arrow-drop-down" 
                size={20} 
              />
            )}
          />
          {showErrors && !selectedCountry && (
            <Text style={inputStyles.errorText}>Country is required</Text>
          )}
        </View>

        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>
            Project
            <Text style={inputStyles.requiredError}> *</Text>
          </Text>
          <Dropdown
            style={[inputStyles.input, showErrors && !selectedProject ? inputStyles.inputError : null]}
            placeholderStyle={inputStyles.placeholderStyle}
            selectedTextStyle={inputStyles.selectedTextStyle}
            data={getProjectOptions()}
            maxHeight={200}
            labelField="label"
            valueField="value"
            placeholder={selectedCountry ? (getProjectOptions().length > 0 ? "Select Project" : "No projects available") : "Select Country first"}
            value={selectedProject}
            onChange={(item) => {
              setSelectedProject(item.value);
            }}
            disable={!selectedCountry || getProjectOptions().length === 0}
            renderLeftIcon={() => (
              <MaterialIcons 
                style={inputStyles.iconStyle} 
                color="#015185" 
                name="business" 
                size={20} 
              />
            )}
            renderRightIcon={() => (
              <MaterialIcons 
                style={inputStyles.iconStyle} 
                color="#015185" 
                name="arrow-drop-down" 
                size={20} 
              />
            )}
          />
          {showErrors && !selectedProject && (
            <Text style={inputStyles.errorText}>Project is required</Text>
          )}
        </View>

        <CustomInputField
          label="Schedule Title"
          value={scheduleTitle}
          onChangeText={setScheduleTitle}
          isRequired
          showError={showErrors && !scheduleTitle}
          errorMessage="Schedule Title is required"
        />

        <CustomInputField
          label={<Text>Note<Text style={inputStyles.requiredError}> *</Text></Text>}
          value={note}
          onChangeText={setNote}
          isRequired
          showError={showErrors && !note}
          errorMessage="Note is required"
        />

        <View style={inputStyles.container}>
          <View style={styles.rowBetween}>
            <Text style={inputStyles.label}>
              Equipments
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <TouchableOpacity
              onPress={() => navigation.navigate('Service_select_equipment_SS', { source: 'AddSchedule', onSelect: handleEquipmentSelect })}
            >
              <Text style={styles.selectText}>Select</Text>
            </TouchableOpacity>
          </View>
          {showErrors && !selectedEquipment && (
            <Text style={inputStyles.errorText}>Equipment is required</Text>
          )}
          {selectedEquipment ? (
            <View>
              <Text style={styles.selectedEquipmentText}>{selectedEquipment.equipmentName}</Text>
              <Text style={styles.selectedEquipmentDetail}>Entry Number: {selectedEquipment.entryNumber || 'N/A'}</Text>
              <Text style={styles.selectedEquipmentDetail}>Equipment Number: {selectedEquipment.equipmentNumber || 'N/A'}</Text>
              <Text style={styles.selectedEquipmentDetail}>Meter: {selectedEquipment.meterReading} {selectedEquipment.meterUnit}</Text>
            </View>
          ) : (
            <Text style={styles.placeholderText}>No equipment selected</Text>
          )}
          <View style={quantityStyles.row}>
            <CustomInputField
              label={<Text>Last Service Meter Reading<Text style={inputStyles.requiredError}> *</Text></Text>}
              value={meterReading}
              onChangeText={(text) => {
                // Fix: Allow only one decimal point and up to 2 decimal places
                const sanitizedText = text.replace(/[^0-9.]|^(.*\..*)\./g, '$1').replace(/(\.\d{2})\d+/, '$1');
                setMeterReading(sanitizedText);
              }}
              keyboardType="numeric"
              isRequired
              showError={showErrors && (!meterReading || isNaN(parseFloat(meterReading)))}
              errorMessage="Valid meter reading is required"
              style={quantityStyles.input}
            />
            <View style={quantityStyles.dropdownWrapper}>
              <Dropdown
                style={quantityStyles.unitBox}
                placeholderStyle={quantityStyles.unitText}
                selectedTextStyle={quantityStyles.unitText}
                data={UNIT_OPTIONS.map(unit => ({ label: unit, value: unit }))}
                maxHeight={150}
                labelField="label"
                valueField="value"
                placeholder="Select Unit"
                value={meterUnit}
                onChange={(item) => handleMeterUnitSelect(item.value)}
                renderRightIcon={() => (
                  <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
                )}
              />
            </View>
          </View>
        </View>

        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>
            Last Service Date
            <Text style={inputStyles.requiredError}> *</Text>
          </Text>
          <View style={quantityStyles.row}>
            <TouchableOpacity
              style={[quantityStyles.input1, { justifyContent: 'center' }]}
              onPress={openDatePicker}
            >
              <Text style={{ color: lastServiceDate && !isNaN(lastServiceDate.getTime()) ? '#222' : '#888', fontSize: width * 0.04 }}>
                {formatDate(lastServiceDate)}
              </Text>
            </TouchableOpacity>
            <TouchableOpacity onPress={openDatePicker}>
              <MaterialIcons name="calendar-today" size={22} color="#015185" style={{ marginLeft: 8 }} />
            </TouchableOpacity>
          </View>
          {showErrors && (!lastServiceDate || isNaN(lastServiceDate.getTime())) && (
            <Text style={inputStyles.errorText}>Last Service Date is required</Text>
          )}
        </View>

        {datePickerVisible && (
          <DateTimePicker
            value={tempDate || new Date()}
            mode="date"
            display="default"
            onChange={(event, selectedDate) => {
              if (event.type === 'set' && selectedDate) {
                setTempDate(selectedDate);
                setLastServiceDate(selectedDate);
                setDatePickerVisible(false);
              } else if (event.type === 'dismissed') {
                setDatePickerVisible(false);
              }
            }}
          />
        )}

        <View style={inputStyles.container}>
          <View style={styles.taskHeader}>
            <Text style={inputStyles.label}>
              Task
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <TouchableOpacity style={styles.addTaskButton} onPress={openTaskModal}>
              <Text style={styles.addTaskText}>+ Add Task</Text>
            </TouchableOpacity>
          </View>
          {showErrors && !tasks.length && (
            <Text style={inputStyles.errorText}>Task is required</Text>
          )}
          {tasks.length > 0 ? (
            tasks.map((task, idx) => (
              <Text key={idx} style={styles.taskItem}>- {task}</Text>
            ))
          ) : (
            <Text style={styles.placeholderText}>No tasks added</Text>
          )}
        </View>

        <Modal visible={taskModalVisible} transparent animationType="fade" onRequestClose={handleTaskCancel}>
          <View style={styles.modalOverlay}>
            <View style={styles.taskModalContainer}>
              <Text style={styles.modalTitle}>Add New Task</Text>
              <CustomInputField
                label="Task Name"
                value={newTask}
                onChangeText={setNewTask}
                style={{ width: '100%' }}
              />
              <View style={styles.modalButtons}>
                <TouchableOpacity style={[styles.modalButton, styles.cancelButton]} onPress={handleTaskCancel}>
                  <Text style={styles.modalButtonText}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity style={[styles.modalButton, styles.doneButton]} onPress={handleTaskConfirm}>
                  <Text style={styles.modalButtonText}>Done</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>

        <View style={inputStyles.container}>
          <View style={[styles.rowBetween, { marginBottom: 0 }]}>
            <Text style={inputStyles.label}>
              Repeat Type
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <View style={styles.repeatTypeRow}>
              <TouchableOpacity
                style={[styles.pillButton, repeatType === 'Repeat' && styles.pillButtonActive]}
                onPress={() => setRepeatType('Repeat')}
              >
                <Text style={[styles.pillButtonText, repeatType === 'Repeat' && styles.pillButtonTextActive]}>Repeat</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.pillButton, repeatType === 'One time' && styles.pillButtonActive]}
                onPress={() => setRepeatType('One time')}
              >
                <Text style={[styles.pillButtonText, repeatType === 'One time' && styles.pillButtonTextActive]}>One time</Text>
              </TouchableOpacity>
            </View>
          </View>
          {showErrors && !repeatType && (
            <Text style={inputStyles.errorText}>Repeat Type is required</Text>
          )}
        </View>

        <View style={inputStyles.container}>
          <Text style={inputStyles.labelSmall}>Intervals (whichever comes first)</Text>
          {intervals.map((interval, idx) => (
            <View key={idx} style={quantityStyles.row}>
              <TouchableOpacity
                style={styles.checkbox}
                onPress={() => {
                  setIntervals(intervals.map((item, i) => (i === idx ? { ...item, checked: !item.checked } : item)));
                }}
              >
                <MaterialIcons
                  name={interval.checked ? 'check-box' : 'check-box-outline-blank'}
                  size={22}
                  color={interval.checked ? '#015185' : '#A0A0A0'}
                />
              </TouchableOpacity>
              <Text style={styles.everyText}>Every</Text>
              <CustomInputField
                label=""
                value={interval.value}
                onChangeText={(text) => {
                  const numericText = text.replace(/[^0-9]/g, '');
                  if (numericText.length <= 10) {
                    setIntervals(intervals.map((item, i) => (i === idx ? { ...item, value: numericText } : item)));
                  }
                }}
                keyboardType="numeric"
                placeholder="0"
                editable={interval.checked}
                maxLength={10}
                style={[quantityStyles.input, { width: 70 }, !interval.checked && inputStyles.inputDisabled]}
              />
              <View style={quantityStyles.dropdownWrapper}>
                <Dropdown
                  style={quantityStyles.unitBox}
                  placeholderStyle={quantityStyles.unitText}
                  selectedTextStyle={quantityStyles.unitText}
                  data={UNIT_OPTIONS.map(unit => ({ label: unit, value: unit }))}
                  maxHeight={150}
                  labelField="label"
                  valueField="value"
                  placeholder="Select Unit"
                  value={interval.unit}
                  onChange={(item) => handleUnitSelect(item.value, idx)}
                  renderRightIcon={() => (
                    <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
                  )}
                />
              </View>
            </View>
          ))}
          {showErrors && repeatType === 'Repeat' && !intervals.some((interval) => interval.checked && interval.value) && (
            <Text style={inputStyles.errorText}>At least one interval is required for repeating schedules</Text>
          )}
          <View style={styles.pillButtonGroup}>
            {['Day', 'Week', 'Month', 'Year'].map((unit) => (
              <TouchableOpacity
                key={unit}
                style={[styles.pillButton, intervalUnit === unit && styles.pillButtonActive]}
                onPress={() => setIntervalUnit(unit)}
              >
                <Text style={[styles.pillButtonText, intervalUnit === unit && styles.pillButtonTextActive]}>{unit}</Text>
              </TouchableOpacity>
            ))}
          </View>
        </View>



        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>Notify Before</Text>
          <View style={quantityStyles.row}>
            <CustomInputField
              label=""
              value={notifyBefore.value}
              onChangeText={(text) => setNotifyBefore({ ...notifyBefore, value: text.replace(/[^0-9]/g, '') })}
              keyboardType="numeric"
              placeholder="0"
              style={[quantityStyles.input, { width: 70 }]}
            />
            <View style={quantityStyles.dropdownWrapper}>
              <Dropdown
                style={quantityStyles.unitBox}
                placeholderStyle={quantityStyles.unitText}
                selectedTextStyle={quantityStyles.unitText}
                data={UNIT_OPTIONS.map(unit => ({ label: unit, value: unit }))}
                maxHeight={150}
                labelField="label"
                valueField="value"
                placeholder="Select Unit"
                value={notifyBefore.unit}
                onChange={(item) => setNotifyBefore({ ...notifyBefore, unit: item.value })}
                renderRightIcon={() => (
                  <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
                )}
              />
            </View>
          </View>
          <View style={styles.pillButtonGroup}>
            {['Day', 'Week', 'Month'].map((unit) => (
              <TouchableOpacity
                key={unit}
                style={[styles.pillButton, notifyBefore.time === unit && styles.pillButtonActive]}
                onPress={() => setNotifyBefore({ ...notifyBefore, time: unit })}
              >
                <Text style={[styles.pillButtonText, notifyBefore.time === unit && styles.pillButtonTextActive]}>{unit}</Text>
              </TouchableOpacity>
            ))}
          </View>
          <Text style={styles.infoText}>
            The schedule will be marked as 'Due soon' once any of the above threshold is crossed.
          </Text>
        </View>



        <Modal visible={isLoading} transparent animationType="fade">
          <View style={styles.syncModalContainer}>
            <View style={styles.syncModalContent}>
              <Text style={styles.syncModalText}>Please Wait</Text>
              <View style={styles.syncIconDiv}>
                {/* <Animated.Image
                  source={require('../../assets/sync.png')} // Fix: Update path to match your project
                  style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
                /> */}
                <Text style={styles.syncModalTextAdd}>Creating Schedule</Text>
              </View>
            </View>
          </View>
        </Modal>

        <Toast />
      </ScrollView>
    </View>
  );
};

const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.04,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  inputError: {
    borderColor: 'red',
  },
  inputDisabled: {
    color: '#A0A0A0',
    borderColor: '#B0C4DE',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
  placeholderStyle: {
    fontSize: width * 0.04,
    color: '#888',
  },
  selectedTextStyle: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  iconStyle: {
    marginRight: 5,
  },
});

const quantityStyles = StyleSheet.create({
  row: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  input: {
    flex: 1,
    marginRight: 8,
  },
  input1: {
    flex: 1,
    marginRight: 8,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  unitBox: {
    flexDirection: 'row',
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    backgroundColor: '#fff',
    width: 100,
    justifyContent: 'space-between',
  },
  unitText: {
    color: '#015185',
    fontSize: width * 0.038,
    fontWeight: '500',
    marginRight: 2,
  },
  dropdownWrapper: {
    position: 'relative',
    zIndex: 1,
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.2)',
  },
  dropdownContainer: {
    position: 'absolute',
    top: width * 0.5,
    right: width * 0.12,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#015185',
    width: 120,
    zIndex: 1000,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
  },
  dropdownItem: {
    paddingVertical: 12,
    paddingHorizontal: 16,
  },
  dropdownText: {
    color: '#015185',
    fontSize: width * 0.038,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.019,
    backgroundColor: '#fff',
    borderRadius: width * 0.01,
  },
  createButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '500',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  rowBetween: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 12,
  },
  row: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 12,
  },
  labelSmall: {
    color: '#015185',
    marginTop: 10,
    marginBottom: 6,
    fontSize: width * 0.034,
  },
  selectText: {
    fontSize: width * 0.034,
    color: '#025383',
  },
  everyText: {
    color: '#015185',
    fontSize: 15,
    marginHorizontal: 8,
  },
  checkbox: {
    marginRight: 4,
  },
  taskHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 6,
  },
  addTaskButton: {
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 18,
    paddingVertical: 8,
  },
  addTaskText: {
    color: '#fff',
    fontSize: width * 0.034,
  },
  repeatTypeRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    width: 180,
  },
  pillButtonGroup: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    alignItems: 'center',
    marginTop: 4,
    marginBottom: 8,
  },
  pillButton: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 18,
    paddingVertical: 7,
    marginRight: 8,
    backgroundColor: '#fff',
  },
  pillButtonActive: {
    backgroundColor: '#015185',
  },
  pillButtonText: {
    color: '#015185',
    fontSize: width * 0.034,
    fontWeight: '500',
  },
  pillButtonTextActive: {
    color: '#fff',
  },
  infoText: {
    color: '#015185',
    fontSize: 13,
    marginTop: 10,
  },
  selectedEquipmentText: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: 'bold',
  },
  selectedEquipmentDetail: {
    fontSize: 14,
    color: '#015185',
    marginBottom: 4,
  },
  placeholderText: {
    fontSize: 15,
    color: '#A0A0A0',
    marginBottom: 12,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  datePickerContainer: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  datePickerButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
    marginTop: 10,
  },
  datePickerButton: {
    flex: 1,
    borderRadius: 5,
    paddingVertical: 10,
    alignItems: 'center',
    marginHorizontal: 5,
  },
  taskModalContainer: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#025383',
    marginBottom: 15,
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    flex: 1,
    borderRadius: 5,
    paddingVertical: 10,
    alignItems: 'center',
    marginHorizontal: 5,
  },
  cancelButton: {
    backgroundColor: '#A0A0A0',
  },
  doneButton: {
    backgroundColor: '#015185',
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  taskItem: {
    fontSize: 15,
    color: '#333',
    marginBottom: 4,
  },
  scheduleIdRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F2F2F2',
    borderRadius: 6,
    paddingHorizontal: 16,
    paddingVertical: 14,
    marginBottom: width * 0.04,
  },
  scheduleIdLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
  scheduleIdValue: {
    fontSize: width * 0.038,
    color: '#025383',
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)', // Fix: Corrected syntax
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextAdd: {
    fontSize: 14,
    color: '#333',
  },
  syncIconDiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default AddSchedule;