import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  StatusBar,
  ScrollView,
  Modal,
  Animated,
  Image,
  ToastAndroid,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation, useRoute } from '@react-navigation/native';
import DropDownPicker from 'react-native-dropdown-picker';
import Toast from 'react-native-toast-message';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import NetInfo from '@react-native-community/netinfo';
import equipmentData from '../EquipmentScreens/EquipmentCategory.json';

const TOTAL_HEADER_HEIGHT = 180;
const ITEMS_PER_PAGE = 10;

const Select_Equipment = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const [searchQuery, setSearchQuery] = useState('');
  const [firebaseEquipment, setFirebaseEquipment] = useState([]);
  const [jsonEquipment, setJsonEquipment] = useState([]);
  const [filteredEquipment, setFilteredEquipment] = useState([]);
  const [displayedEquipment, setDisplayedEquipment] = useState([]);
  const [page, setPage] = useState(1);
  const [statusModalVisible, setStatusModalVisible] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [showDeleteConfirm, setShowDeleteConfirm] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [selectedFilter, setSelectedFilter] = useState(null);
  const [selectedCountry, setSelectedCountry] = useState(null);

  const [statusOpen, setStatusOpen] = useState(false);
  const [statusItems, setStatusItems] = useState([
    { label: 'Available', value: 'Available' },
    { label: 'Maintenance Required', value: 'Maintenance Required' },
    { label: 'In Maintenance', value: 'In Maintenance' },
    { label: 'Breakdown', value: 'Breakdown' },
    { label: 'Deploy', value: 'Deploy' },
  ]);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  // Show toast notification
  const showToast = (type, text1, text2) => {
    if (Platform.OS === 'android') {
      ToastAndroid.show(`${text1}: ${text2}`, ToastAndroid.LONG);
    } else {
      Toast.show({
        type,
        text1,
        text2,
        position: 'bottom',
      });
    }
  };

  const normalizeEquipmentData = (data) => {
    const normalized = [];
    const currentDate = new Date();
    let nameCounter = 1;

    const randomDate = () => new Date(currentDate - Math.random() * 30 * 24 * 60 * 60 * 1000).toISOString();

    data["Light Vehicles TDC"]?.forEach((item, index) => {
      normalized.push({
        id: `lv-${index}`,
        equipmentNumber: item["Vehicle Number"] || `LV-${index + 1}`,
        equipmentName: `Vehicle ${nameCounter++}`,
        equipmentCategory: "Light Vehicle",
        mainCategory: "Light Vehicles TDC",
        model: item["Vehicle Type"] || "N/A",
        meterReading: Math.floor(Math.random() * 10000),
        meterUnit: "Miles",
        status: "Available",
        createdAt: randomDate(),
        country: item["Vehicle Number"]?.includes("ZAMBIA") ? "Zambia" : "Namibia",
        vin: item["VIN Number"] || "N/A",
        regNum: item["Reg Num"] || "N/A",
        dynamicField1: item["Vehicle Number"] || "N/A",
        dynamicField2: item["Reg Num"] || "N/A",
        dynamicField3: item["VIN Number"] || "N/A",
      });
    });

    data["Heavy Vehicles TDC"]?.forEach((item, index) => {
      normalized.push({
        id: `hv-${index}`,
        equipmentNumber: item["Vehicle Number"] || `HV-${index + 1}`,
        equipmentName: `Vehicle ${nameCounter++}`,
        equipmentCategory: "Heavy Vehicle",
        mainCategory: "Heavy Vehicles TDC",
        model: item["Vehicle Type"] || "N/A",
        meterReading: Math.floor(Math.random() * 10000),
        meterUnit: "Miles",
        status: "Available",
        createdAt: randomDate(),
        country: item["Vehicle Number"]?.includes("ZAMBIA") ? "Zambia" : "Namibia",
        vin: item["VIN Number/Chas#"] || "N/A",
        regNum: item["Reg Num"] || "N/A",
        dynamicField1: item["Vehicle Number"] || "N/A",
        dynamicField2: item["Reg Num"] || "N/A",
        dynamicField3: item["VIN Number/Chas#"] || "N/A",
      });
    });

    data["Containers"]?.forEach((item, index) => {
      normalized.push({
        id: `cont-${index}`,
        equipmentNumber: `CONT-${index + 1}`,
        equipmentName: `Container ${nameCounter++}`,
        equipmentCategory: "Container",
        mainCategory: "Containers",
        model: item["Container Type"] || "N/A",
        meterReading: 0,
        meterUnit: "N/A",
        status: "Available",
        createdAt: randomDate(),
        country: "Namibia",
        purpose: item["Purpose"] || "N/A",
        dynamicField1: item["Purpose"] || "N/A",
        dynamicField2: item["Container Type"] || "N/A",
        dynamicField3: item["Container Type"] || "N/A",
      });
    });

    data["TDC Rigs"]?.forEach((item, index) => {
      normalized.push({
        id: `rig-${index}`,
        equipmentNumber: item["Rig Name"] || `RIG-${index + 1}`,
        equipmentName: `Rig ${nameCounter++}`,
        equipmentCategory: "Rig",
        mainCategory: "TDC Rigs",
        model: item["Rig Type"] || "N/A",
        meterReading: Math.floor(Math.random() * 5000),
        meterUnit: "Hours",
        status: "Available",
        createdAt: randomDate(),
        country: "Namibia",
        truckType: item["Truck Type"] || "N/A",
        chassisNumber: item["Chassis Number"] || "N/A",
        dynamicField1: item["Rig Name"] || "N/A",
        dynamicField2: item["Truck Type"] || "N/A",
        dynamicField3: item["Chassis Number"] || "N/A",
      });
    });

    data["Machinery"]?.forEach((item, index) => {
      normalized.push({
        id: `mach-${index}`,
        equipmentNumber: item["Equipment Num"] || `MACH-${index + 1}`,
        equipmentName: `Machinery ${nameCounter++}`,
        equipmentCategory: "Machinery",
        mainCategory: "Machinery",
        model: item["Equipment Type"] || "N/A",
        meterReading: Math.floor(Math.random() * 5000),
        meterUnit: "Hours",
        status: "Available",
        createdAt: randomDate(),
        country: "Namibia",
        vin: item["VIN Number/Chas#"] || "N/A",
        dynamicField1: item["Equipment Num"] || "N/A",
        dynamicField2: item["Equipment Name"] || "N/A",
        dynamicField3: item["VIN Number/Chas#"] || "N/A",
      });
    });

    data["Mud Mixer Trailers"]?.forEach((item, index) => {
      normalized.push({
        id: `trailer-${index}`,
        equipmentNumber: `TRAILER-${item["Number"] || index + 1}`,
        equipmentName: `Trailer ${nameCounter++}`,
        equipmentCategory: "Mud Mixer Trailer",
        mainCategory: "Mud Mixer Trailers",
        model: item["Type"] || "N/A",
        meterReading: 0,
        meterUnit: "N/A",
        status: "Available",
        createdAt: randomDate(),
        country: "Namibia",
        serialNumber: item["Serial Number"] || "N/A",
        dynamicField1: item["Number"] || "N/A",
        dynamicField2: item["Serial Number"] || "N/A",
        dynamicField3: item["Type"] || "N/A",
      });
    });

    data["Lightning Shelters"]?.forEach((item, index) => {
      normalized.push({
        id: `shelter-${index}`,
        equipmentNumber: `SHELTER-${index + 1}`,
        equipmentName: `Shelter ${nameCounter++}`,
        equipmentCategory: "Lightning Shelter",
        mainCategory: "Lightning Shelters",
        model: "Shelter",
        meterReading: 0,
        meterUnit: "N/A",
        status: "Available",
        createdAt: randomDate(),
        country: "Namibia",
        shelterName: item["Shelter"] || "N/A",
        dynamicField1: item["Shelter"] || "N/A",
        dynamicField2: item["Shelter"] || "N/A",
        dynamicField3: item["Shelter"] || "N/A",
      });
    });

    data["Trailers"]?.forEach((item, index) => {
      normalized.push({
        id: `rod-trailer-${index}`,
        equipmentNumber: `ROD-TRAILER-${index + 1}`,
        equipmentName: `Trailer ${nameCounter++}`,
        equipmentCategory: "Trailer",
        mainCategory: "Trailers",
        model: item["Type"] || "N/A",
        meterReading: 0,
        meterUnit: "N/A",
        status: "Available",
        createdAt: randomDate(),
        country: "Namibia",
        chassisNumber: item["Chassis#"] || "N/A",
        dynamicField1: item["Type"] || "N/A",
        dynamicField2: item["Chassis#"] || "N/A",
        dynamicField3: item["Chassis#"] || "N/A",
      });
    });

    data["Appliances"]?.forEach((item, index) => {
      normalized.push({
        id: `appliance-${index}`,
        equipmentNumber: `APP-${index + 1}`,
        equipmentName: `Appliance ${nameCounter++}`,
        equipmentCategory: "Appliance",
        mainCategory: "Appliances",
        model: item["Model"] || "N/A",
        meterReading: 0,
        meterUnit: "N/A",
        status: "Available",
        createdAt: randomDate(),
        country: "Namibia",
        serialNumber: item["Model/Serial Num"] || "N/A",
        applianceType: item["Type"] || "N/A",
        dynamicField1: item["Type"] || "N/A",
        dynamicField2: item["Model"] || "N/A",
        dynamicField3: item["Model/Serial Num"] || "N/A",
      });
    });

    data["Generators"]?.forEach((item, index) => {
      normalized.push({
        id: `gen-${index}`,
        equipmentNumber: `GEN-${index + 1}`,
        equipmentName: `Generator ${nameCounter++}`,
        equipmentCategory: "Generator",
        mainCategory: "Generators",
        model: item["Model"] || "N/A",
        meterReading: Math.floor(Math.random() * 2000),
        meterUnit: "Hours",
        status: "Available",
        createdAt: randomDate(),
        country: "Namibia",
        serialNumber: item["Model/Serial Num"] || "N/A",
        generatorType: item["Type"] || "N/A",
        dynamicField1: item["Type"] || "N/A",
        dynamicField2: item["Model"] || "N/A",
        dynamicField3: item["Model/Serial Num"] || "N/A",
      });
    });

    data["Workshop Machinery"]?.forEach((item, index) => {
      normalized.push({
        id: `wmach-${index}`,
        equipmentNumber: `WMACH-${index + 1}`,
        equipmentName: `Machinery ${nameCounter++}`,
        equipmentCategory: "Workshop Machinery",
        mainCategory: "Workshop Machinery",
        model: item["Model"] || "N/A",
        meterReading: Math.floor(Math.random() * 1000),
        meterUnit: "Hours",
        status: "Available",
        createdAt: randomDate(),
        country: "Namibia",
        serialNumber: item["Serial Num"] || "N/A",
        machineryType: item["Type"] || "N/A",
        dynamicField1: item["Type"] || "N/A",
        dynamicField2: item["Model"] || "N/A",
        dynamicField3: item["Serial Num"] || "N/A",
      });
    });

    data["Pumps"]?.forEach((item, index) => {
      normalized.push({
        id: `pump-${index}`,
        equipmentNumber: `PUMP-${index + 1}`,
        equipmentName: `Pump ${nameCounter++}`,
        equipmentCategory: "Pump",
        mainCategory: "Pumps",
        model: item["Model"] || "N/A",
        meterReading: Math.floor(Math.random() * 1000),
        meterUnit: "Hours",
        status: "Available",
        createdAt: randomDate(),
        country: "Namibia",
        serialNumber: item["Serial Num"] || "N/A",
        pumpType: item["Type"] || "N/A",
        dynamicField1: item["Type"] || "N/A",
        dynamicField2: item["Model"] || "N/A",
        dynamicField3: item["Serial Num"] || "N/A",
      });
    });

    data["Tents"]?.forEach((item, index) => {
      normalized.push({
        id: `tent-${index}`,
        equipmentNumber: `TENT-${index + 1}`,
        equipmentName: `Tent ${nameCounter++}`,
        equipmentCategory: "Tent",
        mainCategory: "Tents",
        model: item["Model"] || "N/A",
        meterReading: 0,
        meterUnit: "N/A",
        status: "Available",
        createdAt: randomDate(),
        country: "Namibia",
        serialNumber: item["Serial Num"] || "N/A",
        tentType: item["Type"] || "N/A",
        dynamicField1: item["Type"] || "N/A",
        dynamicField2: item["Model"] || "N/A",
        dynamicField3: item["Serial Num"] || "N/A",
      });
    });

    return normalized.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
  };

  useEffect(() => {
    const loadData = async () => {
      setSyncModalVisible(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();

      try {
        const user = auth().currentUser;
        if (!user) {
          setSyncModalVisible(false);
          rotateAnim.setValue(0);
          showToast('error', 'Error', 'User not authenticated. Please log in.');
          navigation.replace('AdminLogin');
          return;
        }

        const userId = user.uid;
        const isOnline = await checkNetworkStatus();

        // Load and normalize JSON data
        const normalizedJsonData = normalizeEquipmentData(equipmentData);
        setJsonEquipment(normalizedJsonData);

        if (isOnline) {
          // Set up real-time listener for Firebase
          const equipmentRef = database().ref(`/equipment/${userId}`);
          equipmentRef.on('value', (snapshot) => {
            const data = snapshot.val();
            const parsedFirebaseData = data
              ? Object.values(data).sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt))
              : [];
            setFirebaseEquipment(parsedFirebaseData);
            AsyncStorage.setItem('equipment', JSON.stringify(parsedFirebaseData)).catch((error) =>
              console.error('Error saving to AsyncStorage:', error)
            );
            // Combine Firebase and JSON data, prioritizing Firebase
            const combinedData = [
              ...parsedFirebaseData,
              ...normalizedJsonData.filter(
                (jsonItem) => !parsedFirebaseData.some((fbItem) => fbItem.id === jsonItem.id)
              ),
            ];
            setFilteredEquipment(combinedData);
            setDisplayedEquipment(combinedData.slice(0, ITEMS_PER_PAGE));
          });

          return () => equipmentRef.off();
        } else {
          // Offline: Load from AsyncStorage
          const savedData = await AsyncStorage.getItem('equipment');
          const parsedFirebaseData = savedData ? JSON.parse(savedData) : [];
          setFirebaseEquipment(parsedFirebaseData);
          // Combine Firebase (offline) and JSON data
          const combinedData = [
            ...parsedFirebaseData,
            ...normalizedJsonData.filter(
              (jsonItem) => !parsedFirebaseData.some((fbItem) => fbItem.id === jsonItem.id)
            ),
          ];
          setFilteredEquipment(combinedData);
          setDisplayedEquipment(combinedData.slice(0, ITEMS_PER_PAGE));
          showToast('info', 'Offline', 'Equipment loaded from local storage.');
        }

        const storedCountry = await AsyncStorage.getItem('selectedCountry');
        setSelectedCountry(storedCountry || 'Both Countries');
      } catch (error) {
        console.error('Error loading equipment:', error.code, error.message);
        let errorMessage = 'Failed to load equipment.';
        if (error.code === 'PERMISSION_DENIED') {
          errorMessage = 'Permission denied. Check Firebase security rules.';
        } else if (error.code === 'NETWORK_ERROR') {
          errorMessage = 'Offline: Loading from local storage.';
          const savedData = await AsyncStorage.getItem('equipment');
          const parsedFirebaseData = savedData ? JSON.parse(savedData) : [];
          setFirebaseEquipment(parsedFirebaseData);
          const normalizedJsonData = normalizeEquipmentData(equipmentData);
          setJsonEquipment(normalizedJsonData);
          const combinedData = [
            ...parsedFirebaseData,
            ...normalizedJsonData.filter(
              (jsonItem) => !parsedFirebaseData.some((fbItem) => fbItem.id === jsonItem.id)
            ),
          ];
          setFilteredEquipment(combinedData);
          setDisplayedEquipment(combinedData.slice(0, ITEMS_PER_PAGE));
        }
        showToast(
          error.code === 'NETWORK_ERROR' ? 'info' : 'error',
          error.code === 'NETWORK_ERROR' ? 'Offline' : 'Error',
          errorMessage
        );
      } finally {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
      }
    };
    loadData();
  }, []);

  // Sync pending changes when online
  useEffect(() => {
    const syncPendingChanges = async () => {
      const isOnline = await checkNetworkStatus();
      if (!isOnline) return;

      const user = auth().currentUser;
      if (!user) return;

      const userId = user.uid;
      try {
        const existing = await AsyncStorage.getItem('equipment');
        const equipment = existing ? JSON.parse(existing) : [];
        const pending = equipment.filter((item) => item.pendingSync);

        if (pending.length > 0) {
          setSyncModalVisible(true);
          Animated.loop(
            Animated.timing(rotateAnim, {
              toValue: 1,
              duration: 1000,
              useNativeDriver: true,
            })
          ).start();

          for (const item of pending) {
            const equipmentRef = database().ref(`/equipment/${userId}/${item.id}`);
            await equipmentRef.set({ ...item, pendingSync: false });
          }

          const updatedEquipment = equipment.map((item) =>
            item.pendingSync ? { ...item, pendingSync: false } : item
          );
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipment));

          setSyncModalVisible(false);
          rotateAnim.setValue(0);
          showToast('success', 'Success', 'Pending changes synced to database.');
        }
      } catch (error) {
        console.error('Error syncing pending changes:', error);
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        showToast('error', 'Error', 'Failed to sync pending changes.');
      }
    };

    const unsubscribe = NetInfo.addEventListener((state) => {
      if (state.isConnected) {
        syncPendingChanges();
      }
    });

    return () => unsubscribe();
  }, []);

  const handleSearch = (query) => {
    setSearchQuery(query);
    setPage(1);
    filterEquipment(query, selectedFilter);
  };

  const filterEquipment = (query, filterType) => {
    const combinedData = [
      ...firebaseEquipment,
      ...jsonEquipment.filter(
        (jsonItem) => !firebaseEquipment.some((fbItem) => fbItem.id === jsonItem.id)
      ),
    ];
    let filtered = [...combinedData];

    if (selectedCountry && selectedCountry !== 'Both Countries') {
      filtered = filtered.filter((item) => item.country === selectedCountry);
    }

    if (query) {
      filtered = filtered.filter((item) => {
        switch (filterType) {
          case 'Created Date':
            return new Date(item.createdAt)
              .toLocaleDateString()
              .toLowerCase()
              .includes(query.toLowerCase());
          case 'Equipment Number':
            return (
              (item.equipmentNumber?.toLowerCase() || '').includes(
                query.toLowerCase()
              )
            );
          case 'Equipment Name':
            return (
              (item.equipmentName?.toLowerCase() || '').includes(
                query.toLowerCase()
              )
            );
          case 'Equipment Categories':
            return (
              (item.mainCategory?.toLowerCase() || '').includes(
                query.toLowerCase()
              )
            );
          case 'Meter Reading':
            return `${item.meterReading}`
              .toLowerCase()
              .includes(query.toLowerCase());
          default:
            return (
              (item.equipmentName?.toLowerCase() || '').includes(
                query.toLowerCase()
              ) ||
              (item.equipmentNumber?.toLowerCase() || '').includes(
                query.toLowerCase()
              ) ||
              (item.mainCategory?.toLowerCase() || '').includes(
                query.toLowerCase()
              )
            );
        }
      });
    }
    setFilteredEquipment(filtered);
    setDisplayedEquipment(filtered.slice(0, ITEMS_PER_PAGE));
  };

  const handleFilterSelect = (filter) => {
    setSelectedFilter(filter);
    setFilterModalVisible(false);
    filterEquipment(searchQuery, filter);
  };

  const handleRefresh = async () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const user = auth().currentUser;
      if (!user) {
        setIsRefreshing(false);
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        showToast('error', 'Error', 'User not authenticated. Please log in.');
        navigation.replace('AdminLogin');
        return;
      }

      const userId = user.uid;
      const isOnline = await checkNetworkStatus();

      // Reload JSON data
      const normalizedJsonData = normalizeEquipmentData(equipmentData);
      setJsonEquipment(normalizedJsonData);

      if (isOnline) {
        // Refresh Firebase data
        const equipmentRef = database().ref(`/equipment/${userId}`);
        const snapshot = await equipmentRef.once('value');
        const data = snapshot.val();
        const parsedFirebaseData = data
          ? Object.values(data).sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt))
          : [];
        setFirebaseEquipment(parsedFirebaseData);
        await AsyncStorage.setItem('equipment', JSON.stringify(parsedFirebaseData));
        // Combine data
        const combinedData = [
          ...parsedFirebaseData,
          ...normalizedJsonData.filter(
            (jsonItem) => !parsedFirebaseData.some((fbItem) => fbItem.id === jsonItem.id)
          ),
        ];
        setFilteredEquipment(combinedData);
        setDisplayedEquipment(combinedData.slice(0, ITEMS_PER_PAGE));
        showToast('success', 'Success', 'Equipment refreshed from database and JSON.');
      } else {
        // Offline: Load from AsyncStorage
        const savedData = await AsyncStorage.getItem('equipment');
        const parsedFirebaseData = savedData ? JSON.parse(savedData) : [];
        setFirebaseEquipment(parsedFirebaseData);
        // Combine data
        const combinedData = [
          ...parsedFirebaseData,
          ...normalizedJsonData.filter(
            (jsonItem) => !parsedFirebaseData.some((fbItem) => fbItem.id === jsonItem.id)
          ),
        ];
        setFilteredEquipment(combinedData);
        setDisplayedEquipment(combinedData.slice(0, ITEMS_PER_PAGE));
        showToast('info', 'Offline', 'Equipment refreshed from local storage and JSON.');
      }

      setSearchQuery('');
      setSelectedFilter(null);
      setPage(1);
    } catch (error) {
      console.error('Error refreshing equipment:', error.code, error.message);
      let errorMessage = 'Failed to refresh equipment.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Check Firebase security rules.';
      } else if (error.code === 'NETWORK_ERROR') {
        errorMessage = 'Offline: Refreshed from local storage and JSON.';
        const savedData = await AsyncStorage.getItem('equipment');
        const parsedFirebaseData = savedData ? JSON.parse(savedData) : [];
        setFirebaseEquipment(parsedFirebaseData);
        const normalizedJsonData = normalizeEquipmentData(equipmentData);
        setJsonEquipment(normalizedJsonData);
        const combinedData = [
          ...parsedFirebaseData,
          ...normalizedJsonData.filter(
            (jsonItem) => !parsedFirebaseData.some((fbItem) => fbItem.id === jsonItem.id)
          ),
        ];
        setFilteredEquipment(combinedData);
        setDisplayedEquipment(combinedData.slice(0, ITEMS_PER_PAGE));
      }
      showToast(
        error.code === 'NETWORK_ERROR' ? 'info' : 'error',
        error.code === 'NETWORK_ERROR' ? 'Offline' : 'Error',
        errorMessage
      );
    } finally {
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }
  };

  const { onSelect, source } = route.params || {};

  const handleScroll = ({ nativeEvent }) => {
    const { layoutMeasurement, contentOffset, contentSize } = nativeEvent;
    const isBottom =
      layoutMeasurement.height + contentOffset.y >= contentSize.height - 20;
    if (isBottom && filteredEquipment.length > displayedEquipment.length) {
      setPage((prev) => prev + 1);
    }
  };

  useEffect(() => {
    setDisplayedEquipment(filteredEquipment.slice(0, page * ITEMS_PER_PAGE));
  }, [page, filteredEquipment]);

  const handleSelectEquipment = (item) => {
    const equipmentData = {
      entryNumber: item.equipmentNumber || '',
      equipmentName: item.equipmentName || 'Tyre - Tyre',
      meterReading: item.meterReading || 0,
      meterUnit: item.meterUnit || 'Miles',
    };

    if (source === 'RequestMaintenance' && onSelect) {
      // Pass the entire equipment object to the parent component
      onSelect(equipmentData.equipmentName); // Pass only equipmentName to match CreateRequestMaintenance
      navigation.goBack();
    } else {
      showToast('error', 'Error', 'No valid callback provided for equipment selection.');
      navigation.goBack();
    }
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Available':
        return '#2ecc71';
      case 'Maintenance Required':
        return '#f1c40f';
      case 'In Maintenance':
        return '#e67e22';
      case 'Breakdown':
        return '#e74c3c';
      case 'Deploy':
        return '#3498db';
      default:
        return '#666';
    }
  };

  const getDynamicFieldLabels = (mainCategory) => {
    switch (mainCategory) {
      case 'Light Vehicles TDC':
      case 'Heavy Vehicles TDC':
        return { field1: 'Vehicle Number', field2: 'Registration', field3: 'VIN Number' };
      case 'TDC Rigs':
        return { field1: 'Rig Name', field2: 'Truck Type', field3: 'Chassis Number' };
      case 'Machinery':
        return { field1: 'Equipment Number', field2: 'Equipment Name', field3: 'VIN Number' };
      case 'Mud Mixer Trailers':
        return { field1: 'Number', field2: 'Serial Number', field3: 'Type' };
      case 'Trailers':
        return { field1: 'Type', field2: 'Chassis Number', field3: 'Chassis Number' };
      case 'Appliances':
      case 'Generators':
        return { field1: 'Type', field2: 'Model', field3: 'Serial Number' };
      case 'Workshop Machinery':
      case 'Pumps':
      case 'Tents':
        return { field1: 'Type', field2: 'Model', field3: 'Serial Number' };
      case 'Containers':
        return { field1: 'Purpose', field2: 'Container Type', field3: 'Container Type' };
      case 'Lightning Shelters':
        return { field1: 'Shelter Name', field2: 'Shelter Name', field3: 'Shelter Name' };
      default:
        return { field1: 'Field 1', field2: 'Field 2', field3: 'Field 3' };
    }
  };

  const renderAdditionalFields = (item) => {
    const fields = [];
    const { field1, field2, field3 } = getDynamicFieldLabels(item.mainCategory);
    if (item.dynamicField1 && item.dynamicField1 !== 'N/A')
      fields.push({ label: field1, value: item.dynamicField1 });
    if (item.dynamicField2 && item.dynamicField2 !== 'N/A')
      fields.push({ label: field2, value: item.dynamicField2 });
    if (item.dynamicField3 && item.dynamicField3 !== 'N/A')
      fields.push({ label: field3, value: item.dynamicField3 });

    return fields.map((field, index) => (
      <View key={index} style={styles.equipmentDetailItem}>
        <Text style={styles.detailLabel}>{field.label}</Text>
        <Text style={styles.detailValue}>{field.value}</Text>
      </View>
    ));
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar
          translucent
          backgroundColor="transparent"
          barStyle="light-content"
        />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={24} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Select Equipment</Text>
          </View>
          <TouchableOpacity onPress={handleRefresh}>
            <Image
              source={require('../../../Images/adminhome/equp/cloud_1.png')}
              style={styles.icons12}
            />
          </TouchableOpacity>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder={`Search${selectedFilter ? ' by ' + selectedFilter : '...'}`}
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons
              name="search"
              size={20}
              color="#888"
              style={styles.bladeIcon}
            />
          </View>
          <TouchableOpacity
            style={styles.filterButton}
            onPress={() => setFilterModalVisible(true)}
          >
            <MaterialIcons name="filter-list" size={24} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      <ScrollView
        style={styles.mainContent}
        onScroll={handleScroll}
        scrollEventThrottle={16}
      >
        {displayedEquipment.length > 0 ? (
          displayedEquipment.map((item) => (
            <View key={item.id} style={styles.equipmentRow}>
              <View style={styles.equipmentInfo}>
                <View style={styles.equipmentDetailItemheader}>
                  <Text style={styles.detailLabelheader}>
                    {item.equipmentName || 'Tyre - Tyre'}
                  </Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Category</Text>
                  <Text style={styles.detailValue}>
                    {item.mainCategory || 'Sample Category'}
                  </Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Model</Text>
                  <Text style={styles.detailValue}>{item.model || 'N/A'}</Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Meter Reading</Text>
                  <Text style={styles.detailValue}>
                    {item.meterReading} {item.meterUnit || 'Miles'}
                  </Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Created Date</Text>
                  <Text style={styles.detailValue}>
                    {new Date(item.createdAt).toLocaleDateString()}
                  </Text>
                </View>
                {renderAdditionalFields(item)}
                <View style={styles.equipmentDetailItemNoBorder}>
                  <Text
                    style={[
                      styles.detailLabelstatuss,
                      { color: getStatusColor(item.status || 'Available') },
                    ]}
                  >
                    {item.status || 'Available'}
                  </Text>
                  <TouchableOpacity
                    style={styles.selectButton}
                    onPress={() => handleSelectEquipment(item)}
                  >
                    <Text style={styles.selectButtonText}>Select</Text>
                  </TouchableOpacity>
                </View>
              </View>
            </View>
          ))
        ) : (
          <Text style={styles.placeholderText}>
            {searchQuery
              ? 'No matching equipment found'
              : selectedCountry && selectedCountry !== 'Both Countries'
              ? `No equipment found for ${selectedCountry}`
              : 'No equipment added yet'}
          </Text>
        )}
      </ScrollView>

      <View style={styles.buttonContainer}>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={styles.mapButtonGradient}
        >
          <TouchableOpacity style={styles.mapButton}>
            <Image
              source={require('../../../Images/adminhome/equp/mapview_1.png')}
              style={styles.mapIcon}
            />
            <Text style={styles.addButtonTextmap}>Map View</Text>
          </TouchableOpacity>
        </LinearGradient>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={styles.addButtonGradient}
        >
          <TouchableOpacity
            style={styles.addButton}
            onPress={() => navigation.navigate('Add_equipment')}
          >
            <Ionicons name="add" size={24} color="#fff" />
            <Text style={styles.addButtonText}>Add Equipments</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      <Modal
        visible={statusModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setStatusModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.statusModalContent}>
            <LinearGradient
              colors={['#3481BC', '#025383']}
              style={styles.statusModalHeader}
            >
              <Text style={styles.statusModalTitle}>Change Status</Text>
            </LinearGradient>
            <View style={styles.statusDropdownWrapper}>
              <DropDownPicker
                open={statusOpen}
                value={null}
                items={statusItems}
                setOpen={setStatusOpen}
                setValue={() => {}}
                setItems={setStatusItems}
                style={styles.dropdown}
                dropDownContainerStyle={styles.dropdownContainer}
                placeholder="Select Status"
                zIndex={1000}
                maxHeight={150}
              />
            </View>
            <TouchableOpacity
              style={styles.applyButton}
              onPress={() => setStatusModalVisible(false)}
            >
              <Text style={styles.applyButtonText}>Apply</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={filterModalVisible}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setFilterModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.filterModalContent}>
            <Text style={styles.filterModalTitle}>Filter By</Text>
            {[
              'Created Date',
              'Equipment Number',
              'Equipment Name',
              'Equipment Categories',
              'Meter Reading',
            ].map((filter) => (
              <TouchableOpacity
                key={filter}
                style={styles.filterItem}
                onPress={() => handleFilterSelect(filter)}
              >
                <Text style={styles.filterItemText}>{filter}</Text>
              </TouchableOpacity>
            ))}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setFilterModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={syncModalVisible}
        transparent
        animationType="fade"
      >
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[
                  styles.syncIcon,
                  { transform: [{ rotate: rotateInterpolate }] },
                ]}
              />
              <Text style={styles.syncModalTextadd}>
                Cloud Data Sync in Progress
              </Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={showDeleteConfirm}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setShowDeleteConfirm(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.deleteConfirmModalContent}>
            <Text style={styles.deleteConfirmTitle}>Confirm Delete</Text>
            <Text style={styles.deleteConfirmText}>
              Are you sure you want to delete this equipment?
            </Text>
            <View style={styles.deleteConfirmButtonContainer}>
              <TouchableOpacity
                style={[styles.deleteConfirmButton, { backgroundColor: '#666' }]}
                onPress={() => setShowDeleteConfirm(false)}
              >
                <Text style={styles.deleteConfirmButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.deleteConfirmButton, { backgroundColor: '#3481BC' }]}
                onPress={() => setShowDeleteConfirm(false)}
              >
                <Text style={styles.deleteConfirmButtonText}>OK</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 15,
    paddingTop: StatusBar.currentHeight || 44,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: 20,
    fontWeight: 'bold',
    marginLeft: 10,
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 15,
    marginTop: 10,
  },
  searchWrapper: {
    flexDirection: 'row',
    flex: 1,
    backgroundColor: '#f0f0f0',
    borderRadius: 35,
    paddingHorizontal: 10,
    alignItems: 'center',
  },
  searchInput: {
    flex: 1,
    height: 40,
    fontSize: 16,
    color: '#000',
    paddingVertical: 0,
  },
  bladeIcon: {
    marginLeft: 5,
  },
  filterButton: {
    marginLeft: 10,
    padding: 5,
  },
  mainContent: {
    flex: 1,
  },
  placeholderText: {
    fontSize: 18,
    color: '#888',
    textAlign: 'center',
    marginTop: 20,
  },
  equipmentRow: {
    marginHorizontal: 15,
    marginVertical: '3%',
    backgroundColor: '#F2F2F2',
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#eee',
  },
  equipmentInfo: {
    flex: 1,
  },
  equipmentDetailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingHorizontal: '5%',
    paddingRight: '10%',
  },
  equipmentDetailItemNoBorder: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingVertical: 5,
    paddingHorizontal: '5%',
  },
  equipmentDetailItemheader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: '5%',
  },
  detailLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: 3,
  },
  detailLabelstatuss: {
    fontSize: 14,
    fontWeight: '400',
    marginVertical: 3,
  },
  detailLabelheader: {
    fontSize: 18,
    color: '#015185',
    fontWeight: '600',
    width: '100%',
  },
  detailValue: {
    fontSize: 16,
    color: '#015185',
    width: '60%',
    textAlign: 'right',
    fontWeight: '500',
  },
  selectButton: {
    backgroundColor: '#3481BC',
    paddingHorizontal: 15,
    paddingVertical: 5,
    borderRadius: 5,
  },
  selectButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  mapButtonGradient: {
    borderRadius: 5,
    overflow: 'hidden',
    marginBottom: 10,
  },
  mapButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 20,
  },
  mapIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  addButtonGradient: {
    borderRadius: 5,
    overflow: 'hidden',
  },
  addButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 20,
  },
  addButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
    marginLeft: 5,
  },
  addButtonTextmap: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  statusModalContent: {
    width: '90%',
    backgroundColor: '#fff',
    borderRadius: 10,
    overflow: 'hidden',
  },
  statusModalHeader: {
    padding: 15,
  },
  statusModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#fff',
    height: 50,
  },
  statusDropdownWrapper: {
    padding: 20,
    zIndex: 1000,
  },
  dropdown: {
    backgroundColor: '#f0f0f0',
    borderRadius: 5,
    borderWidth: 0,
  },
  dropdownContainer: {
    backgroundColor: '#fff',
    borderRadius: 5,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 2,
  },
  applyButton: {
    backgroundColor: '#3481BC',
    padding: 10,
    borderRadius: 5,
    alignItems: 'center',
    margin: 20,
  },
  applyButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  filterModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
  },
  filterModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    marginBottom: 15,
    color: '#333',
  },
  filterItem: {
    padding: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  filterItemText: {
    fontSize: 16,
    color: '#333',
  },
  modalCloseButton: {
    marginTop: 15,
    alignItems: 'center',
  },
  modalCloseText: {
    color: '#3481BC',
    fontSize: 16,
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  icons12: {
    width: 24,
    height: 24,
  },
  deleteConfirmModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  deleteConfirmTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    marginBottom: 15,
    color: '#333',
  },
  deleteConfirmText: {
    fontSize: 16,
    color: '#666',
    marginBottom: 20,
    textAlign: 'center',
  },
  deleteConfirmButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  deleteConfirmButton: {
    flex: 1,
    padding: 10,
    borderRadius: 5,
    marginHorizontal: 5,
    alignItems: 'center',
  },
  deleteConfirmButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
});

export default Select_Equipment;