import React, { useState, useEffect, useRef, useCallback, useMemo } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
  ScrollView,
  Easing, // <-- Add this import
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const DROPDOWN_OPTIONS = [
  {
    label: 'Approve',
    value: 'approve',
    icon: <MaterialIcons name="check-circle" size={22} color="#3481BC" />,
  },
  {
    label: 'Edit',
    value: 'edit',
    icon: <MaterialIcons name="edit" size={22} color="#3481BC" />,
  },
  {
    label: 'View',
    value: 'view',
    icon: <MaterialIcons name="visibility" size={22} color="#4ECB71" />,
  },
  {
    label: 'Delete',
    value: 'delete',
    icon: <MaterialIcons name="delete" size={22} color="#C2185B" />,
  },
];

// Define priority icons and colors
const priorityIcons = {
  Low: { icon: 'arrow-downward', color: '#4ECB71' }, // Green
  Medium: { icon: 'priority-high', color: '#FFC107' }, // Yellow
  High: { icon: 'warning', color: '#FF5722' }, // Orange
  Emergency: { icon: 'error', color: '#C2185B' }, // Red
};

// Define status badge colors
const statusColors = {
  Requested: '#3481BC', // Blue
  Approved: '#4ECB71',  // Green
  Rejected: '#C2185B',  // Red
};

// SyncModal Component
const SyncModal = ({ visible, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    let animation;
    if (visible) {
      animation = Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 900,
          useNativeDriver: true,
          easing: Easing.linear, // <-- Use Easing.linear
        })
      );
      animation.start();
    } else {
      rotateAnim.setValue(0);
      if (animation) animation.stop();
    }
    return () => {
      if (animation) animation.stop && animation.stop();
    };
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const RequestmaintenanceHome = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [maintenanceRequests, setMaintenanceRequests] = useState([]);
  const [filteredRequests, setFilteredRequests] = useState([]);
  const [userId, setUserId] = useState(null);
  const [selectedRequest, setSelectedRequest] = useState(null);
  const [deleteConfirmVisible, setDeleteConfirmVisible] = useState(false);
  const [permissionsModalVisible, setPermissionsModalVisible] = useState(false);
  const [userData, setUserData] = useState({
    countries: [],
    projects: [],
    requestMaintenancePermissions: [],
    name: '',
    email: '',
  });

  // Add state for dropdown menu
  const [dropdownVisible, setDropdownVisible] = useState(false);
  const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0 });
  const [dropdownRequest, setDropdownRequest] = useState(null);

  // Memoized permission checks to prevent re-renders
  const hasOnlyMineView = useMemo(() => {
    return userData.requestMaintenancePermissions.some(p => 
      p && typeof p === 'string' && p.toLowerCase() === 'onlymineview'
    );
  }, [userData.requestMaintenancePermissions]);

  const hasPermission = useCallback((permission) => {
    return userData.requestMaintenancePermissions.includes(permission);
  }, [userData.requestMaintenancePermissions]);

  const hasCreatePermission = useMemo(() => hasPermission('create'), [hasPermission]);
  const hasEditPermission = useMemo(() => hasPermission('edit'), [hasPermission]);
  const hasViewPermission = useMemo(() => hasPermission('view'), [hasPermission]);
  const hasDeletePermission = useMemo(() => hasPermission('delete'), [hasPermission]);
  const hasApprovePermission = useMemo(() => hasPermission('approve'), [hasPermission]);

  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  // Show toast notification
  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  // Fetch user UID and data from AsyncStorage and Firebase
  useEffect(() => {
    const loadUserData = async () => {
      try {
        const uid = await AsyncStorage.getItem('userUid');
        const userEmail = await AsyncStorage.getItem('userEmail');
        if (!uid || !userEmail) {
          showToast('error', 'Error', 'User not logged in. Please log in again.');
          navigation.replace('AdminLogin');
          return;
        }

        setUserId(uid);

        // Fetch user data from Firebase
        const userSnapshot = await database()
          .ref('/Globalusers/admins')
          .orderByChild('email')
          .equalTo(userEmail)
          .once('value');
        const userDataRaw = userSnapshot.val();
        if (!userDataRaw) {
          showToast('error', 'Error', 'User not found in admin database.');
          navigation.replace('AdminLogin');
          return;
        }

        // Extract user data (first user found)
        const userIdKey = Object.keys(userDataRaw)[0];
        const data = userDataRaw[userIdKey];

        // Extract requestMaintenance permissions (case sensitive)
        const requestMaintenanceModule = data.modules?.find(mod => mod.module === 'requestMaintenance');
        const requestMaintenancePermissions = requestMaintenanceModule?.permissions || [];

        setUserData({
          countries: data.countries || [],
          projects: data.projects || [],
          requestMaintenancePermissions: requestMaintenancePermissions,
          name: data.name || data.fullName || 'Admin',
          email: data.email || userEmail,
        });
      } catch (error) {
        console.error('Error fetching user data:', error);
        showToast('error', 'Error', 'Failed to load user data.');
      }
    };

    loadUserData();
  }, [navigation]);

  // Memoized filter function
  const filterRequestsByPermissions = useCallback((requests) => {
    if (!requests || requests.length === 0) return [];

    let filteredRequests = requests;

    // If user has onlyMineView permission, show only their own requests
    if (hasOnlyMineView) {
      filteredRequests = requests.filter(request => request.userId === userId);
    } else {
      // Filter by user's assigned countries and projects
      if (userData.countries.length > 0 || userData.projects.length > 0) {
        filteredRequests = requests.filter(request => {
          const matchesCountry = userData.countries.length === 0 || 
            (request.country && userData.countries.includes(request.country));
          const matchesProject = userData.projects.length === 0 || 
            (request.project && userData.projects.includes(request.project));
          return matchesCountry && matchesProject;
        });
      }
    }

    return filteredRequests;
  }, [userData.countries, userData.projects, userId, hasOnlyMineView]);

  // Fetch maintenance requests
  const fetchMaintenanceRequests = useCallback(async () => {
    if (!userId) return;
    setSyncModalVisible(true);
    try {
      const isOnline = await checkNetworkStatus();
      let requests = [];

      if (isOnline) {
        // If user has onlyMineView permission, fetch only their requests
        if (hasOnlyMineView) {
          const snapshot = await database()
            .ref(`GlobalRequestMaintenancereport/admins/${userId}`)
            .once('value');
          const data = snapshot.val();
          requests = data ? Object.values(data) : [];
        } else {
          // Fetch all requests and filter by permissions
          const snapshot = await database()
            .ref(`GlobalRequestMaintenancereport/admins`)
            .once('value');
          const data = snapshot.val();
          if (data) {
            // Flatten all requests from all admins
            requests = Object.values(data).flatMap(adminRequests => 
              Object.values(adminRequests || {})
            );
          }
        }
        
        await AsyncStorage.setItem('globalMaintenanceRequests', JSON.stringify(requests));
        showToast('success', 'Success', 'Maintenance requests synced successfully.');
      } else {
        const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
        requests = storedRequests ? JSON.parse(storedRequests) : [];
        showToast('info', 'Offline', 'Using offline data. Will sync when online.');
      }

      // Sort requests by requestNumber descending (latest first)
      requests = requests.slice().sort((a, b) => {
        // Extract the numeric part from requestNumber (e.g., ZAM-KAL-RM-002 => 2)
        const getNum = (req) => {
          const match = (req.requestNumber || '').match(/(\d+)$/);
          return match ? parseInt(match[1], 10) : 0;
        };
        return getNum(b) - getNum(a);
      });

      // Apply permission-based filtering
      const filteredRequests = filterRequestsByPermissions(requests);
      setMaintenanceRequests(filteredRequests);
      setFilteredRequests(filteredRequests);
    } catch (error) {
      console.error('Error fetching maintenance requests:', error);
      let errorMessage = 'Failed to fetch maintenance requests.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Please contact support.';
      } else if (error.code === 'NETWORK_ERROR' || !(await checkNetworkStatus())) {
        errorMessage = 'Offline mode. Showing locally stored data.';
        const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
        const requests = storedRequests ? JSON.parse(storedRequests) : [];
        const filteredRequests = filterRequestsByPermissions(requests);
        setMaintenanceRequests(filteredRequests);
        setFilteredRequests(filteredRequests);
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      setSyncModalVisible(false);
    }
  }, [userId, filterRequestsByPermissions, hasOnlyMineView]);

  // Auto-refresh on screen focus (hard refresh every time)
  useFocusEffect(
    useCallback(() => {
      setSyncModalVisible(true);
      fetchMaintenanceRequests().finally(() => {
        setSyncModalVisible(false);
      });
    }, [fetchMaintenanceRequests, userId])
  );

  // Handle search
  const handleSearch = (text) => {
    setSearchQuery(text);
    if (text.trim()) {
      const filtered = maintenanceRequests.filter(
        (request) =>
          (request.workOrderTitle?.toLowerCase() || '').includes(text.toLowerCase()) ||
          (request.requestType?.toLowerCase() || '').includes(text.toLowerCase()) ||
          (request.equipment?.equipmentName?.toLowerCase() || '').includes(text.toLowerCase()) ||
          (request.requestNumber?.toLowerCase() || '').includes(text.toLowerCase())
      );
      setFilteredRequests(filtered);
    } else {
      setFilteredRequests(maintenanceRequests);
    }
  };

  // Handle refresh/sync
  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    fetchMaintenanceRequests().finally(() => {
      setIsRefreshing(false);
      setSyncModalVisible(false);
    });
  };

  // Defensive navigation handlers
  const handleApprove = (request) => {
    if (request && request.id) {
      setDropdownVisible(false);
      setTimeout(() => {
        try {
          navigation.navigate('RequestMApproveWO', { requestId: request.id });
        } catch (err) {
          console.error('Navigation to RequestMApproveWO failed:', err, request);
        }
      }, 100);
    } else {
      console.warn('Invalid request for approve:', request);
    }
  };

  const handleEdit = (request) => {
    if (request && request.id && hasEditPermission) {
      setSyncModalVisible(false);
      setDropdownVisible(false);
      setDeleteConfirmVisible(false);
      setPermissionsModalVisible(false);
      setTimeout(() => {
        try {
          navigation.navigate('UpdateRequestMaintenance', { requestId: request.id });
        } catch (err) {
          console.error('Navigation to UpdateRequestMaintenance failed:', err, request);
        }
      }, 100);
    } else {
      console.warn('Invalid request for edit:', request);
    }
  };

  const handleView = (request) => {
    if (request && request.id && hasViewPermission) {
      setDropdownVisible(false);
      setTimeout(() => {
        try {
          navigation.navigate('ViewMRequest', { requestId: request.id });
        } catch (err) {
          console.error('Navigation to ViewMRequest failed:', err, request);
        }
      }, 100);
    } else {
      console.warn('No view permission or invalid request', request);
    }
  };

  const handleDelete = (request) => {
    if (request && request.id && hasDeletePermission) {
      setSelectedRequest(request);
      setDeleteConfirmVisible(true);
    } else {
      console.warn('Invalid request for delete:', request);
    }
  };

  const handleDropdownAction = async (action, request) => {
    // Defensive: close all modals before any navigation
    setSyncModalVisible(false);
    setDropdownVisible(false);
    setDeleteConfirmVisible(false);
    setPermissionsModalVisible(false);
    try {
      switch (action) {
        case 'approve':
          await handleApprove(request);
          break;
        case 'edit':
          await handleEdit(request);
          break;
        case 'view':
          await handleView(request);
          break;
        case 'delete':
          await handleDelete(request);
          break;
        default:
          break;
      }
    } catch (error) {
      console.error('Error processing action:', error);
      showToast('error', 'Error', 'Failed to process action.');
    }
  };

  const confirmDelete = async () => {
    if (!selectedRequest || !userId || !hasDeletePermission) return;

    setDeleteConfirmVisible(false);
    setSyncModalVisible(true);

    try {
      const isOnline = await checkNetworkStatus();

      const updatedRequests = maintenanceRequests.filter(
        (request) => request.id !== selectedRequest.id
      );
      setMaintenanceRequests(updatedRequests);
      setFilteredRequests(updatedRequests);

      await AsyncStorage.setItem('globalMaintenanceRequests', JSON.stringify(updatedRequests));

      if (isOnline) {
        await database()
          .ref(`GlobalRequestMaintenancereport/admins/${userId}/${selectedRequest.id}`)
          .remove();
        showToast('success', 'Deleted', 'Maintenance request deleted successfully.');
      } else {
        showToast('info', 'Offline', 'Request deleted locally. Will sync when online.');
      }
    } catch (error) {
      console.error('Error deleting maintenance request:', error);
      let errorMessage = 'Failed to delete maintenance request.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Please contact support.';
      } else if (error.code === 'NETWORK_ERROR' || !(await checkNetworkStatus())) {
        errorMessage = 'Offline mode. Deletion saved locally and will sync when online.';
      }
      showToast('error', 'Error', errorMessage);
      await fetchMaintenanceRequests();
    } finally {
      setSyncModalVisible(false);
      setSelectedRequest(null);
    }
  };

  const cancelDelete = () => {
    setDeleteConfirmVisible(false);
    setSelectedRequest(null);
  };

  // Memoized empty state message
  const emptyStateMessage = useMemo(() => {
    return hasOnlyMineView 
      ? 'No maintenance requests found for your account.' 
      : 'No maintenance requests found for your assigned countries/projects.';
  }, [hasOnlyMineView]);

  // Render each maintenance request
  const renderMaintenanceRequest = useCallback(({ item }) => {
    const status = item.status || 'Requested';
    const statusColor = statusColors[status] || statusColors['Requested'];

    return (
      <View style={styles.cardContainer}>
        {/* Top Row: Icon, WO number, Status */}
        <View style={styles.cardTopRow}>
          <MaterialIcons name="local-fire-department" size={22} color="#4ECB71" style={{ marginRight: 6 }} />
          <Text style={styles.woNumber}>{item.requestNumber || 'WO-NA'}</Text>
          <View style={[styles.statusBadge, { backgroundColor: statusColor }]}> 
            <Text style={styles.statusText}>{status}</Text>
          </View>
        </View>
        {/* Table Rows */}
        <View style={styles.tableRow}>
          <Text style={styles.tableLabel}>Title</Text>
          <Text style={styles.tableValue}>{item.workOrderTitle || 'N/A'}</Text>
        </View>
        <View style={styles.tableRow}>
          <Text style={styles.tableLabel}>Equipments #</Text>
          <Text style={styles.tableValue}>{item.equipment?.equipmentName || 'N/A'}</Text>
        </View>
        <View style={styles.tableRow}>
          <Text style={styles.tableLabel}>Assign</Text>
          <Text style={styles.tableValue}>
            {item.assignTo?.map(a => a.fullName || a.name).join(', ') || 'Unassigned'}
          </Text>
        </View>
        <View style={[styles.tableRow, { alignItems: 'center' }]}> 
          <Text style={styles.tableLabel}>Create Date</Text>
          <Text style={styles.tableValue}>
            {item.createdAt
              ? new Date(item.createdAt).toLocaleString('en-GB', {
                  day: '2-digit',
                  month: 'short',
                  hour: '2-digit',
                  minute: '2-digit',
                  hour12: true,
                }).replace(',', '')
              : 'N/A'}
          </Text>
          <TouchableOpacity
            style={styles.menuButton}
            onPress={() => setDropdownRequest(item)}
          >
            <Dropdown
              style={styles.dropdown}
              data={DROPDOWN_OPTIONS}
              labelField="label"
              valueField="value"
              value={null}
              onChange={option => handleDropdownAction(option.value, item)}
              renderRightIcon={() => <MaterialIcons name="more-vert" size={22} color="#333" />}
              renderItem={option => (
                <View style={styles.dropdownItem}>
                  {option.icon}
                  <Text style={styles.dropdownText}>{option.label}</Text>
                </View>
              )}
              placeholder=""
              showsVerticalScrollIndicator={false}
              containerStyle={styles.dropdownContainer}
              itemTextStyle={styles.dropdownItemText}
              selectedTextStyle={{ display: 'none' }}
            />
          </TouchableOpacity>
        </View>
      </View>
    );
  }, []);

  const scale = (size) => (width / 375) * size;

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={scale(28)} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Request Maintenance</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={handleRefresh}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud.png')}
                style={styles.refreshIcon}
              />
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.iconButton}
              onPress={() => setPermissionsModalVisible(true)}
            >
              <MaterialIcons name="lock" size={scale(24)} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>

        {/* Search Bar */}
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by title, type, vehicle..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={scale(20)} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity style={styles.filterButton}>
            <MaterialIcons name="filter-list" size={scale(24)} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Main Content */}
      <FlatList
        data={filteredRequests.filter(item => item && item.id)}
        renderItem={renderMaintenanceRequest}
        keyExtractor={(item) => item.id}
        contentContainerStyle={styles.mainContent}
        ListEmptyComponent={
          <Text style={styles.defaultText}>{emptyStateMessage}</Text>
        }
        removeClippedSubviews={true}
        maxToRenderPerBatch={10}
        windowSize={10}
        initialNumToRender={10}
      />

      {/* Bottom Button - Only show if user has create permission */}
      {hasCreatePermission && (
        <View style={styles.buttonContainer}>
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
            <TouchableOpacity
              style={styles.startButton}
              onPress={() => navigation.navigate('CreateRequestMaintenance')}
            >
              <Ionicons name="add" size={scale(24)} color="#fff" />
              <Text style={styles.buttonText}>New Request</Text>
            </TouchableOpacity>
          </LinearGradient>
        </View>
      )}

      {/* Sync Modal */}
      <SyncModal
        visible={syncModalVisible}
        message={deleteConfirmVisible ? 'Deleting Request...' : 'Syncing Data...'}
      />

      {/* Delete Confirmation Modal */}
      <Modal
        visible={deleteConfirmVisible}
        transparent
        animationType="fade"
        onRequestClose={cancelDelete}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.confirmModalContent}>
            <Text style={styles.confirmModalText}>Are you sure?</Text>
            <Text style={styles.confirmModalSubText}>
              Do you really want to delete this maintenance request?
            </Text>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={cancelDelete}
              >
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.modalButton, styles.deleteButton]}
                onPress={confirmDelete}
              >
                <Text style={styles.deleteButtonText}>Delete</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Permissions Modal */}
      <Modal
        animationType="slide"
        transparent={true}
        visible={permissionsModalVisible}
        onRequestClose={() => setPermissionsModalVisible(false)}
      >
        <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.4)', justifyContent: 'center', alignItems: 'center' }}>
          <View style={{
            width: '85%',
            backgroundColor: '#fff',
            borderRadius: 18,
            padding: 28,
            alignItems: 'center',
            shadowColor: '#000',
            shadowOffset: { width: 0, height: 2 },
            shadowOpacity: 0.2,
            shadowRadius: 8,
            elevation: 8,
            maxHeight: '80%',
          }}>
            <Text style={{
              fontSize: 22,
              fontWeight: 'bold',
              color: '#025383',
              marginBottom: 18,
            }}>
              User Details
            </Text>
            <ScrollView style={{ width: '100%' }} contentContainerStyle={{ paddingBottom: 16 }} showsVerticalScrollIndicator={true}>
              <View style={{ marginBottom: 18 }}>
                <Text style={{ color: '#888', fontSize: 14 }}>Name</Text>
                <Text style={{ fontSize: 17, fontWeight: '600', color: '#025383', marginTop: 2 }}>{userData.name}</Text>
              </View>
              <View style={{ marginBottom: 18 }}>
                <Text style={{ color: '#888', fontSize: 14 }}>Countries</Text>
                <Text style={{ fontSize: 17, fontWeight: '600', color: userData.countries.length > 0 ? '#025383' : '#FF6B6B', marginTop: 2 }}>{userData.countries.length > 0 ? userData.countries.join(', ') : 'No countries assigned'}</Text>
              </View>
              <View style={{ marginBottom: 18 }}>
                <Text style={{ color: '#888', fontSize: 14 }}>Projects</Text>
                <Text style={{ fontSize: 17, fontWeight: '600', color: userData.projects.length > 0 ? '#025383' : '#FF6B6B', marginTop: 2 }}>{userData.projects.length > 0 ? userData.projects.join(', ') : 'No projects assigned'}</Text>
              </View>
              <View style={{ marginBottom: 18 }}>
                <Text style={{ color: '#888', fontSize: 14 }}>Request Maintenance Permissions</Text>
                <Text style={{ fontSize: 17, fontWeight: '600', color: userData.requestMaintenancePermissions.length > 0 ? '#025383' : '#FF6B6B', marginTop: 2 }}>{userData.requestMaintenancePermissions.length > 0 ? userData.requestMaintenancePermissions.join(', ') : 'No permissions assigned'}</Text>
              </View>
            </ScrollView>
            <TouchableOpacity
              style={{
                marginTop: 8,
                backgroundColor: '#025383',
                borderRadius: 8,
                paddingVertical: 10,
                paddingHorizontal: 32,
              }}
              onPress={() => setPermissionsModalVisible(false)}
              accessibilityLabel="Close user details"
              accessibilityRole="button"
            >
              <Text style={{
                color: '#fff',
                fontWeight: 'bold',
                fontSize: 16,
              }}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    padding: width * 0.012,
  },
  mainContent: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.2,
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  requestCard: {
    backgroundColor: '#f9f9f9',
    borderRadius: width * 0.02,
    padding: width * 0.04,
    marginBottom: width * 0.03,
    borderWidth: 1,
    borderColor: '#ddd',
    elevation: 2,
  },
  cardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  woNumber: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#3481BC',
    marginLeft: 6,
  },
  statusBadge: {
    borderRadius: 4,
    paddingHorizontal: 8,
    paddingVertical: 2,
    marginLeft: 10,
  },
  statusText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: 'bold',
  },
  menuButton: {
    padding: 4,
    hitSlop: { top: 10, bottom: 10, left: 10, right: 10 },
  },
  cardRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 4,
  },
  cardLabel: {
    color: '#333',
    fontWeight: '500',
    fontSize: width * 0.035,
    width: '30%',
  },
  cardValue: {
    color: '#222',
    fontSize: width * 0.035,
    fontWeight: '400',
    flex: 1,
  },
  dateContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  dropdownContainer: {
    width: 220,
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
    left: '40%',
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  dropdownItemText: {
    fontSize: 14,
    color: '#222',
    marginLeft: 16,
    fontWeight: '500',
  },
  buttonContainer: {
    position: 'absolute',
    bottom: width * 0.05,
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: 10,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  confirmModalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
    alignItems: 'center',
  },
  confirmModalText: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.025,
  },
  confirmModalSubText: {
    fontSize: width * 0.04,
    color: '#333',
    marginBottom: width * 0.05,
    textAlign: 'center',
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.05,
    borderRadius: width * 0.02,
    alignItems: 'center',
    flex: 1,
    marginHorizontal: width * 0.01,
  },
  cancelButton: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
  },
  deleteButton: {
    backgroundColor: '#C2185B',
  },
  modalButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  deleteButtonText: {
    color: '#FFFFFF',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  modalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 20,
  },
  modalScroll: {
    maxHeight: Dimensions.get('window').height * 0.5,
    width: '100%',
  },
  modalSection: {
    marginBottom: 20,
  },
  modalSectionTitle: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
  },
  chip: {
    backgroundColor: '#e6f0fa',
    borderRadius: 12,
    paddingHorizontal: 10,
    paddingVertical: 6,
    marginRight: 8,
    marginBottom: 8,
    borderWidth: 1,
    borderColor: '#015185',
  },
  chipText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '500',
  },
  noItemsText: {
    fontSize: 14,
    color: '#888',
    marginTop: 8,
  },
  modalCloseButton: {
    marginTop: 20,
    paddingVertical: 10,
    paddingHorizontal: 20,
    backgroundColor: '#ff4444',
    borderRadius: 5,
  },
  modalCloseText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  modernModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.25)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modernModalContent: {
    width: '90%',
    maxWidth: 400,
    height: '60%', // Set fixed height to 60% of the screen
    backgroundColor: '#fff',
    borderRadius: 22,
    padding: 24,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.18,
    shadowRadius: 12,
    elevation: 8,
    alignItems: 'center',
    justifyContent: 'flex-start', // Ensure content starts at the top
  },
  modernModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#222',
    marginBottom: 18,
    textAlign: 'center',
  },
  section: {
    width: '100%',
    marginBottom: 24,
  },
  sectionTitle: {
    color: '#015185',
    fontWeight: 'bold',
    fontSize: 17,
    marginBottom: 4,
  },
  sectionDivider: {
    height: 2,
    backgroundColor: '#e0e6ed',
    marginBottom: 12,
    borderRadius: 1,
  },
  chipRow: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    alignItems: 'center',
    marginBottom: 0,
  },
  chip: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f4f8fb',
    borderRadius: 20,
    borderWidth: 1,
    borderColor: '#3481BC',
    paddingHorizontal: 14,
    paddingVertical: 8,
    margin: 6,
  },
  chipText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '500',
  },
  chipEmpty: {
    fontSize: 14,
    color: '#bbb',
    margin: 6,
    fontStyle: 'italic',
  },
  closeButton: {
    backgroundColor: '#ff4444',
    borderRadius: 24,
    paddingHorizontal: 32,
    paddingVertical: 12,
    marginTop: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    elevation: 4,
  },
  closeButtonText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
    textAlign: 'center',
    letterSpacing: 0.5,
  },
  userInfoRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 8,
  },
  userInfoLabel: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  userInfoValue: {
    fontSize: width * 0.035,
    color: '#015185',
    fontWeight: '600',
  },
  summaryRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 8,
  },
  summaryLabel: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  summaryValue: {
    fontSize: width * 0.035,
    color: '#015185',
    fontWeight: '600',
  },
  dateContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  dropdownItemText: {
    fontSize: 14,
    color: '#222',
    marginLeft: 16,
    fontWeight: '500',
  },
  cardContainer: {
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    marginBottom: 16,
    overflow: 'hidden',
    paddingHorizontal: 0,
    paddingVertical: 0,
  },
  cardTopRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-start',
    paddingHorizontal: 16,
    paddingTop: 12,
    paddingBottom: 8,
    backgroundColor: '#fff',
  },
  woNumber: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#3481BC',
    marginLeft: 4,
    flex: 0,
  },
  statusBadge: {
    marginLeft: 'auto',
    borderRadius: 4,
    paddingHorizontal: 10,
    paddingVertical: 2,
    backgroundColor: '#3481BC',
    alignSelf: 'center',
  },
  statusText: {
    color: '#fff',
    fontSize: 13,
    fontWeight: 'bold',
  },
  tableRow: {
    flexDirection: 'row',
    alignItems: 'center',
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
    paddingHorizontal: 16,
    paddingVertical: 10,
    backgroundColor: '#fff',
  },
  tableLabel: {
    width: 110,
    color: '#333',
    fontWeight: '500',
    fontSize: 15,
  },
  tableValue: {
    flex: 1,
    color: '#222',
    fontSize: 15,
    fontWeight: '400',
    textAlign: 'right',
  },
  menuButton: {
    marginLeft: 8,
    padding: 4,
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
});

export default RequestmaintenanceHome;