import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const LaborCodeList = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [addModalVisible, setAddModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [partsInventory, setPartsInventory] = useState([]);
  const [filteredParts, setFilteredParts] = useState([]);
  const [laborCode, setLaborCode] = useState('');
  const [hourlyRate, setHourlyRate] = useState('');
  const [description, setDescription] = useState('');
  const [editingItem, setEditingItem] = useState(null);
  const [dropdownVisible, setDropdownVisible] = useState(false);
  const [selectedPartId, setSelectedPartId] = useState(null);
  const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0 });
  const [showDeleteConfirm, setShowDeleteConfirm] = useState(false);
  const [showErrors, setShowErrors] = useState(false);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const fetchPartsInventory = async () => {
    try {
      const parts = await AsyncStorage.getItem('partsInventory');
      const parsedParts = parts ? JSON.parse(parts) : [];
      setPartsInventory(parsedParts);
      setFilteredParts(parsedParts);
    } catch (error) {
      console.error('Error fetching parts inventory:', error);
    }
  };

  useEffect(() => {
    fetchPartsInventory();
  }, []);

  const handleSearch = (text) => {
    setSearchQuery(text);
    if (text) {
      const filtered = partsInventory.filter(
        (part) =>
          part.id?.toLowerCase().includes(text.toLowerCase()) ||
          part.laborCode?.toLowerCase().includes(text.toLowerCase()) ||
          part.description?.toLowerCase().includes(text.toLowerCase())
      );
      setFilteredParts(filtered);
    } else {
      setFilteredParts(partsInventory);
    }
  };

  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    setTimeout(() => {
      fetchPartsInventory();
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }, 2000);
  };

  const handleSaveLaborCode = async () => {
    setShowErrors(true);
    if (!laborCode || !hourlyRate) return;
    if (editingItem) {
      const updatedParts = partsInventory.map((part) =>
        part.id === editingItem.id
          ? { ...part, laborCode, hourlyRate, description }
          : part
      );
      await AsyncStorage.setItem('partsInventory', JSON.stringify(updatedParts));
      setPartsInventory(updatedParts);
      setFilteredParts(updatedParts);
    } else {
      const newLaborCode = {
        id: Date.now().toString(),
        laborCode,
        hourlyRate,
        description,
      };
      const updatedParts = [...partsInventory, newLaborCode];
      await AsyncStorage.setItem('partsInventory', JSON.stringify(updatedParts));
      setPartsInventory(updatedParts);
      setFilteredParts(updatedParts);
    }
    setAddModalVisible(false);
    setLaborCode('');
    setHourlyRate('');
    setDescription('');
    setEditingItem(null);
    setShowErrors(false);
  };

  const handleOpenAddModal = () => {
    setEditingItem(null);
    setLaborCode('');
    setHourlyRate('');
    setDescription('');
    setAddModalVisible(true);
  };

  const handleMorePress = (partId, event) => {
    const { pageX, pageY } = event.nativeEvent;
    setSelectedPartId(partId);
    setDropdownPosition({ x: pageX, y: pageY });
    setDropdownVisible(true);
  };

  const handleEdit = () => {
    const item = partsInventory.find((part) => part.id === selectedPartId);
    if (item) {
      setEditingItem(item);
      setLaborCode(item.laborCode);
      setHourlyRate(item.hourlyRate);
      setDescription(item.description);
      setAddModalVisible(true);
    }
    setDropdownVisible(false);
  };

  const handleDelete = async () => {
    setShowDeleteConfirm(false);
    try {
      const updatedParts = partsInventory.filter((part) => part.id !== selectedPartId);
      await AsyncStorage.setItem('partsInventory', JSON.stringify(updatedParts));
      setPartsInventory(updatedParts);
      setFilteredParts(updatedParts);
      setDropdownVisible(false);
    } catch (error) {
      console.error('Error deleting labor code:', error);
    }
  };

  const renderPartItem = ({ item }) => (
    <View style={styles.partItemCustom}>
      <View style={styles.partIdRowCustom}>
        <Text style={styles.partIdCustom}>Item#{item.id}</Text>
      </View>
      <View style={styles.detailRowWithLine}>
        <Text style={styles.partLabelCustom}>Labor Code</Text>
        <View style={styles.valueColWrapNoDivider}>
          <Text style={styles.partValueCustom}>{item.laborCode || 'N/A'}</Text>
        </View>
      </View>
      <View style={styles.detailRowWithLine}>
        <Text style={styles.partLabelCustom}>Hourly Rate</Text>
        <View style={styles.valueColWrapNoDivider}>
          <Text style={styles.partValueCustom}>{item.hourlyRate || 'N/A'}</Text>
        </View>
      </View>
      <View style={styles.detailRowLast}>
        <Text style={styles.partLabelCustom}>Description</Text>
        <View style={styles.valueColWrapNoDivider}>
          <View style={styles.supplierRow}>
            <Text style={styles.partValueCustom}>{item.description || 'N/A'}</Text>
            <TouchableOpacity onPress={(event) => handleMorePress(item.id, event)} style={styles.moreIconButton}>
              <MaterialIcons name="more-vert" size={width * 0.07} color="#222" />
            </TouchableOpacity>
          </View>
        </View>
      </View>
    </View>
  );

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Labor Code</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={handleRefresh}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud_1.png')}
                style={styles.icons12}
              />
            </TouchableOpacity>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search"
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity style={styles.filterButton}>
            <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      <View style={styles.buttonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
          <TouchableOpacity style={styles.startButton} onPress={handleOpenAddModal}>
            <Ionicons name="add" size={width * 0.06} color="#fff" />
            <Text style={styles.buttonText}> Add Labor Code</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      <Modal visible={addModalVisible} transparent animationType="fade">
        <View style={styles.addModalContainer}>
          <View style={styles.addModalContentCustom}>
            <Text style={styles.addModalTitle}>Add Labor Code</Text>
            <Text style={styles.inputLabel}>
              Labor Code
              <Text style={{ color: 'red', fontWeight: 'bold' }}> *</Text>
            </Text>
            <TextInput
              style={styles.input}
              value={laborCode}
              onChangeText={setLaborCode}
              placeholder="Enter Labor Code"
              placeholderTextColor="#888"
            />
            {showErrors && !laborCode && (
              <Text style={styles.errorText}>Labor Code is required</Text>
            )}
            <Text style={styles.inputLabel}>
              Hourly Rate
              <Text style={{ color: 'red', fontWeight: 'bold' }}> *</Text>
            </Text>
            <TextInput
              style={styles.input}
              value={hourlyRate}
              onChangeText={text => setHourlyRate(text.replace(/[^0-9]/g, ''))}
              placeholder="Enter Hourly Rate"
              placeholderTextColor="#888"
              keyboardType="numeric"
            />
            {showErrors && !hourlyRate && (
              <Text style={styles.errorText}>Hourly Rate is required</Text>
            )}
            <Text style={styles.inputLabel}>Description</Text>
            <TextInput
              style={[styles.input, styles.descriptionInput]}
              value={description}
              onChangeText={setDescription}
              placeholder="Enter Description"
              placeholderTextColor="#888"
              multiline
            />
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={styles.modalButton}
                onPress={() => { setAddModalVisible(false); setShowErrors(false); }}
              >
                <Text style={styles.modalButtonText}>Close</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.modalButton, styles.saveButton]}
                onPress={handleSaveLaborCode}
              >
                <Text style={styles.modalButtonText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={dropdownVisible}
        transparent
        animationType="none"
        onRequestClose={() => setDropdownVisible(false)}
      >
        <TouchableOpacity
          style={styles.dropdownOverlay}
          onPress={() => setDropdownVisible(false)}
        >
          <View
            style={[
              styles.dropdownMenu,
              {
                top: dropdownPosition.y,
                left: dropdownPosition.x - 120,
              },
            ]}
          >
            <TouchableOpacity style={styles.dropdownItem} onPress={handleEdit}>
              <MaterialIcons name="edit" size={width * 0.05} color="#1E90FF" />
              <Text style={styles.dropdownText}>Edit</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.dropdownItem}
              onPress={() => {
                setDropdownVisible(false);
                setShowDeleteConfirm(true);
              }}
            >
              <MaterialIcons name="delete" size={width * 0.05} color="#FF0000" />
              <Text style={styles.dropdownText}>Delete</Text>
            </TouchableOpacity>
          </View>
        </TouchableOpacity>
      </Modal>

      <Modal
        visible={showDeleteConfirm}
        transparent
        animationType="fade"
        onRequestClose={() => setShowDeleteConfirm(false)}
      >
        <View style={styles.confirmOverlay}>
          <View style={styles.alertBox}>
            <View style={styles.alertHeader}>
              <MaterialIcons name="error" size={36} color="#d32f2f" style={{ marginRight: 10 }} />
              <Text style={styles.alertTitle}>Alert!</Text>
            </View>
            <Text style={styles.alertMessage}>Do you want to remove?</Text>
            <View style={styles.alertActions}>
              <TouchableOpacity style={styles.alertButton} onPress={() => setShowDeleteConfirm(false)}>
                <Text style={styles.alertNo}>NO</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.alertButton} onPress={handleDelete}>
                <Text style={styles.alertYes}>YES</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {filteredParts.length === 0 ? (
        <View style={styles.noDataContainer}>
          <Text style={styles.noDataText}>No Labor Codes Available</Text>
        </View>
      ) : (
        <FlatList
          data={filteredParts}
          renderItem={renderPartItem}
          keyExtractor={(item) => item.id}
          contentContainerStyle={styles.listContent}
        />
      )}
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f0f0f0',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    padding: width * 0.012,
  },
  listContent: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.3,
  },
  partItemCustom: {
    backgroundColor: '#fff',
    borderRadius: width * 0.02,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    padding: 0,
    marginBottom: width * 0.03,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.05,
    shadowRadius: 2,
    elevation: 1,
    overflow: 'hidden',
  },
  partIdRowCustom: {
    backgroundColor: '#ededed',
    paddingVertical: width * 0.018,
    paddingHorizontal: width * 0.025,
    borderTopLeftRadius: width * 0.02,
    borderTopRightRadius: width * 0.02,
  },
  partIdCustom: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#222',
  },
  detailRowWithLine: {
    flexDirection: 'row',
    alignItems: 'stretch',
    borderBottomWidth: 1,
    borderBottomColor: '#ededed',
    minHeight: width * 0.1,
    paddingHorizontal: width * 0.025,
  },
  detailRowLast: {
    flexDirection: 'row',
    alignItems: 'stretch',
    paddingHorizontal: width * 0.025,
  },
  partLabelCustom: {
    fontSize: width * 0.035,
    color: '#888',
    fontWeight: 'bold',
    width: '38%',
    textAlignVertical: 'center',
    textAlign: 'left',
    paddingVertical: width * 0.03,
  },
  valueColWrapNoDivider: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
  },
  partValueCustom: {
    fontSize: width * 0.035,
    color: '#222',
    textAlign: 'left',
    textAlignVertical: 'center',
    flexShrink: 1,
  },
  supplierRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-start',
    flex: 1,
  },
  moreIconButton: {
    marginLeft: width * 0.3,
    padding: width * 0.01,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '5%',
    width: '100%',
    alignItems: 'center',
    zIndex: 10,
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  addModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  addModalContentCustom: {
    width: '85%',
    backgroundColor: 'white',
    borderRadius: 12,
    padding: 24,
    alignItems: 'center',
    borderWidth: 2,
    borderColor: '#e0e0e0',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  addModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 15,
  },
  inputLabel: {
    fontSize: 14,
    color: '#333',
    alignSelf: 'flex-start',
    marginBottom: 5,
  },
  input: {
    width: '100%',
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    padding: 10,
    marginBottom: 15,
    color: '#333',
  },
  descriptionInput: {
    height: 100,
    textAlignVertical: 'top',
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    flex: 1,
    backgroundColor: '#3481BC',
    padding: 10,
    borderRadius: 5,
    alignItems: 'center',
    marginHorizontal: 5,
  },
  saveButton: {
    backgroundColor: '#025383',
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'transparent',
  },
  dropdownMenu: {
    position: 'absolute',
    backgroundColor: '#fff',
    borderRadius: width * 0.02,
    paddingVertical: width * 0.02,
    width: width * 0.35,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 3.84,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  dropdownText: {
    marginLeft: width * 0.025,
    fontSize: width * 0.04,
    color: '#333',
  },
  confirmOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  alertBox: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 24,
    alignItems: 'center',
    elevation: 5,
  },
  alertHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
    alignSelf: 'flex-start',
  },
  alertTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#222',
  },
  alertMessage: {
    fontSize: width * 0.04,
    color: '#333',
    marginBottom: 24,
    alignSelf: 'flex-start',
    marginLeft: 46,
  },
  alertActions: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    width: '100%',
    marginTop: 8,
  },
  alertButton: {
    paddingHorizontal: 18,
    paddingVertical: 8,
  },
  alertNo: {
    color: '#234567',
    fontWeight: 'bold',
    fontSize: width * 0.04,
    letterSpacing: 1,
    marginRight: 10,
  },
  alertYes: {
    color: '#234567',
    fontWeight: 'bold',
    fontSize: width * 0.04,
    letterSpacing: 1,
    marginLeft: 10,
  },
  noDataContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: width * 0.1,
  },
  noDataText: {
    fontSize: width * 0.045,
    color: '#333',
    fontWeight: '500',
  },
  errorText: {
    color: 'red',
    fontSize: 13,
    marginBottom: 8,
    alignSelf: 'flex-start',
    marginLeft: 2,
  },
});

export default LaborCodeList;