import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  StatusBar,
  TouchableOpacity,
  Image,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';

const Mainheader = ({ navigation }) => {
  const TOTAL_HEADER_HEIGHT = 90; // Total height including status bar
  const [unreadCount, setUnreadCount] = useState(0);
  const [currentUserId, setCurrentUserId] = useState('');

  // Get current user ID and load notification count
  useEffect(() => {
    getCurrentUserId();
  }, []);

  // Listen for notification count changes
  useEffect(() => {
    if (currentUserId) {
      loadNotificationCount();
      // Set up real-time listener for notification count
      const notificationsRef = database().ref(`/Globalusers/admins/${currentUserId}/notifications`);
      const unsubscribe = notificationsRef.on('value', (snapshot) => {
        if (snapshot.exists()) {
          const notifications = snapshot.val();
          const unread = Object.values(notifications).filter(notification => !notification.isRead).length;
          setUnreadCount(unread);
        } else {
          setUnreadCount(0);
        }
      });

      return () => notificationsRef.off('value', unsubscribe);
    }
  }, [currentUserId]);

  const getCurrentUserId = async () => {
    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (userId) {
        setCurrentUserId(userId);
      }
    } catch (error) {
      console.error('Error getting current user ID:', error);
    }
  };

  const loadNotificationCount = async () => {
    if (!currentUserId) return;
    
    try {
      const notificationsRef = database().ref(`/Globalusers/admins/${currentUserId}/notifications`);
      const snapshot = await notificationsRef.once('value');
      
      if (snapshot.exists()) {
        const notifications = snapshot.val();
        const unread = Object.values(notifications).filter(notification => !notification.isRead).length;
        setUnreadCount(unread);
      } else {
        setUnreadCount(0);
      }
    } catch (error) {
      console.error('Error loading notification count:', error);
    }
  };

  const handleNotificationPress = () => {
    navigation.navigate('HeaderWellNotificaton');
  };

  
  return (
    <View style={styles.container}>
      {/* Single Gradient for Status Bar and Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']} // Single gradient across entire header
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}>
        {/* Status Bar */}
        <StatusBar
          translucent
          backgroundColor="transparent" // Transparent to show gradient
          barStyle="light-content" // White icons/text for contrast
        />

        {/* Header Content */}
        <View style={styles.headerContent}>
          {/* Left: Logo */}
          {/* <TouchableOpacity  onPress={() => navigation.toggleDrawer()}> */}
            <TouchableOpacity>
            <Image
              source={require('../../Images/adminhome/homeheadlogo.png')} // Replace with your logo path
              style={styles.logo} 
            />
          </TouchableOpacity>

          {/* Right: Bell Icon and Menu Icon */}
          <View style={styles.rightContainer}>
            <TouchableOpacity 
              style={styles.notificationButton} 
              onPress={handleNotificationPress}
              activeOpacity={0.7}
            >
              <MaterialIcons name="notifications" size={24} color="white" style={styles.icon} />
              {unreadCount > 0 && (
                <View style={styles.notificationBadge}>
                  <Text style={styles.notificationBadgeText}>
                    {unreadCount > 99 ? '99+' : unreadCount}
                  </Text>
                </View>
              )}
            </TouchableOpacity>
            <TouchableOpacity 
              onPress={() => navigation.toggleDrawer()}
              activeOpacity={0.7}
            >
              <Image
                source={require('../../Images/adminhome/menuicon.png')}
                style={styles.logomenu} 
              />
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    width: '100%',
  },
  header: {
    width: '100%',
  },
  headerContent: {
    flex: 1, // Takes up the full height minus status bar padding
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 15,
    paddingTop: StatusBar.currentHeight || 44, // Ensures content starts below status bar
  },
  logo: {
    width: 100,
    height: 80,
    resizeMode: 'contain',
  },
  rightContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  icon: {
    marginLeft: 15, // Space between bell and menu icons
  },

  logomenu:{
    width: 20,
    height: 30,
    resizeMode: 'contain',
    marginLeft:20
  },
  notificationButton: {
    position: 'relative',
    padding: 8,
  },
  notificationBadge: {
    position: 'absolute',
    top: 2,
    right: 2,
    backgroundColor: '#FF4444',
    borderRadius: 10,
    minWidth: 20,
    height: 20,
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 4,
  },
  notificationBadgeText: {
    color: 'white',
    fontSize: 10,
    fontWeight: 'bold',
  },
});

export default Mainheader;