import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Modal,
  FlatList,
  KeyboardAvoidingView,
  Platform,
  PermissionsAndroid,
  Animated,
  Alert,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import DropDownPicker from 'react-native-dropdown-picker';
import database from '@react-native-firebase/database';
import { Dimensions } from 'react-native';
import Geolocation from '@react-native-community/geolocation';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import { Dropdown } from 'react-native-element-dropdown';

const { width } = Dimensions.get('window');

// Custom Input Field Component
const CustomInputField = ({ label, value, onChangeText, isRequired = false, showError = false, errorMessage = '', ...props }) => (
  <View style={inputStyles.container}>
    <Text style={inputStyles.label}>
      {label}
      {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
    </Text>
    <TextInput
      style={[inputStyles.input, showError ? inputStyles.inputError : null]}
      value={value}
      onChangeText={onChangeText}
      {...props}
      placeholderTextColor="#888"
    />
    {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
  </View>
);

// SyncModal Component
const SyncModal = ({ visible, onClose, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.modalOverlay}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const InsInfostep1 = ({ setActiveStep, equipment }) => {
  const navigation = useNavigation();
  const route = useRoute();
  const [country, setCountry] = useState(null);
  const [countryOpen, setCountryOpen] = useState(false);
  const [countryItems] = useState([{ label: 'Zambia', value: 'Zambia' }, { label: 'DRC', value: 'DRC' }]);
  const [project, setProject] = useState(null);
  const [projectOpen, setProjectOpen] = useState(false);
  const [projectItems, setProjectItems] = useState([]);
  const [inspector, setInspector] = useState('');
  const [checklistModalVisible, setChecklistModalVisible] = useState(false);
  const [equipmentModalVisible, setEquipmentModalVisible] = useState(false);
  const [checklists, setChecklists] = useState([]);
  const [firebaseEquipment, setFirebaseEquipment] = useState([]);
  const [checklistSearchQuery, setChecklistSearchQuery] = useState('');
  const [isChecklistSearchActive, setIsChecklistSearchActive] = useState(false);
  const [selectedTab, setSelectedTab] = useState('ALL');
  const [selectedChecklistId, setSelectedChecklistId] = useState(null);
  const [checklistTitle, setChecklistTitle] = useState('');
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [meterReading, setMeterReading] = useState('');
  const [currentMeterReading, setCurrentMeterReading] = useState('');
  const [requestNumber, setRequestNumber] = useState('');
  const [equipmentSearchQuery, setEquipmentSearchQuery] = useState('');
  const [isEquipmentSearchActive, setIsEquipmentSearchActive] = useState(false);
  const [showErrors, setShowErrors] = useState(false);
  const [countryError, setCountryError] = useState(false);
  const [projectError, setProjectError] = useState(false);
  const [inspectorError, setInspectorError] = useState(false);
  const [equipmentError, setEquipmentError] = useState(false);
  const [checklistError, setChecklistError] = useState(false);
  const [coordinates, setCoordinates] = useState('');
  const [gpsAddress, setGpsAddress] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const equipmentSearchInputRef = useRef(null);
  const isMounted = useRef(true);
  const ADMIN_UID = 'cahj3jJqe5Z16pQdLGSWrNgm84o1';
  const [equipmentLoading, setEquipmentLoading] = useState(false); // Add loading state for equipment modal

  // Update: Use global counter for inspection request numbers
  const generateRequestNumber = async () => {
    try {
      let nextNumber = 1;
      // Read the global lastRequestNumber from Firebase
      if (isOnline) {
        const lastRequestRef = database().ref('/GlobalInspectionReport/admins/lastRequestNumber');
        const snapshot = await lastRequestRef.once('value');
        const lastNumberStr = snapshot.val();
        let lastNumber = 0;
        if (lastNumberStr && typeof lastNumberStr === 'string') {
          const match = lastNumberStr.match(/-IR-(\d{3})$/);
          if (match) lastNumber = parseInt(match[1], 10);
        }
        nextNumber = lastNumber + 1;
      }
      // Generate country and project codes
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      const projectCode = project ? project.slice(0, 3).toUpperCase() || 'XXX' : 'XXX';
      const newRequestNumber = `${countryCode}-${projectCode}-IR-${String(nextNumber).padStart(3, '0')}`;
      return newRequestNumber;
    } catch (error) {
      console.error('Error generating request number:', error);
      // Fallback for offline or error scenarios
      let nextNumber = 1;
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      const projectCode = project ? project.slice(0, 3).toUpperCase() || 'XXX' : 'XXX';
      const newRequestNumber = `${countryCode}-${projectCode}-IR-${String(nextNumber).padStart(3, '0')}`;
      return newRequestNumber;
    }
  };

  const saveInspectionData = async (data) => {
    try {
      await AsyncStorage.setItem('inspectionStep1', JSON.stringify(data));
      if (!isOnline) {
        const syncQueue = JSON.parse(await AsyncStorage.getItem('syncQueue') || '[]');
        syncQueue.push({ type: 'inspectionStep1', data });
        await AsyncStorage.setItem('syncQueue', JSON.stringify(syncQueue));
        Toast.show({ type: 'info', text1: 'Offline', text2: 'Data saved locally. Will sync when online.', position: 'top' });
      }
    } catch (error) {
      console.error('Error saving inspection data:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to save data locally.', position: 'top' });
    }
  };

  const syncDataToFirebase = async () => {
    if (!isOnline) return;

    try {
      const syncQueue = JSON.parse(await AsyncStorage.getItem('syncQueue') || '[]');
      if (syncQueue.length === 0) return;

      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        Toast.show({ type: 'error', text1: 'Authentication Error', text2: 'User email not found. Please log in.', position: 'top' });
        return;
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      const userData = userSnapshot.val();
      if (!userData) {
        Toast.show({ type: 'error', text1: 'Authentication Error', text2: 'User not found. Please log in.', position: 'top' });
        navigation.replace('AdminLoginScreen');
        return;
      }
      const userId = Object.keys(userData)[0];

      for (const item of syncQueue) {
        if (item.type === 'inspectionStep1') {
          await database().ref(`/GlobalInspectionReport/admins/${userId}/${item.data.id}`).set(item.data);
        }
      }
      await AsyncStorage.setItem('syncQueue', '[]');
      Toast.show({ type: 'success', text1: 'Sync Complete', text2: 'Offline data synced to server.', position: 'top' });
    } catch (error) {
      console.error('Error syncing data to Firebase:', error);
      Toast.show({ type: 'error', text1: 'Sync Error', text2: 'Failed to sync data. Will retry when online.', position: 'top' });
    }
  };

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected);
      if (state.isConnected) syncDataToFirebase();
    });

    NetInfo.fetch().then((state) => {
      setIsOnline(state.isConnected);
      if (state.isConnected) syncDataToFirebase();
    });

    return () => unsubscribe();
  }, []);

  // Update loadEquipmentData to accept showSyncModal argument
  const loadEquipmentData = async (showSyncModal = true) => {
    if (showSyncModal) setSyncModalVisible(true);
    try {
      const isOnline = await NetInfo.fetch().then((state) => state.isConnected);
      let data = [];
      if (isOnline) {
        const snapshot = await database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData').once('value');
        data = snapshot.val() || [];
        data = Array.isArray(data) ? data : [];
        data = data.map((item, index) => ({ ...item, firebaseKey: item.firebaseKey || `equip_${index}` }));
        data = data.filter((item, index, self) => self.findIndex((t) => t.id === item.id) === index);
        data.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        await AsyncStorage.setItem('equipment', JSON.stringify(data));
      } else {
        const storedEquipment = await AsyncStorage.getItem('equipment');
        data = storedEquipment ? JSON.parse(storedEquipment) : [];
        data.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        Toast.show({ type: 'info', text1: 'Offline', text2: 'Loaded equipment from local storage.', position: 'top' });
      }
      setFirebaseEquipment(data);
    } catch (error) {
      console.error('Error loading equipment:', error);
      const storedEquipment = await AsyncStorage.getItem('equipment');
      const parsedEquipment = storedEquipment ? JSON.parse(storedEquipment) : [];
      setFirebaseEquipment(parsedEquipment);
      Toast.show({
        type: error.code === 'NETWORK_ERROR' ? 'info' : 'error',
        text1: error.code === 'NETWORK_ERROR' ? 'Offline' : 'Error',
        text2: error.code === 'NETWORK_ERROR' ? 'Loaded from local storage.' : 'Failed to load equipment.',
        position: 'top',
      });
    } finally {
      if (showSyncModal) setSyncModalVisible(false);
      setEquipmentSearchQuery('');
      setIsEquipmentSearchActive(false);
    }
  };

  useEffect(() => {
    const loadData = async () => {
      try {
        await loadEquipmentData();
        const newRequestNumber = await generateRequestNumber();
        setRequestNumber(newRequestNumber);

        const savedInspection = await AsyncStorage.getItem('inspectionStep1');
        let inspectionData = savedInspection ? JSON.parse(savedInspection) : {};

        // Load checklists if we have checklist params or if checklists are not loaded yet
        if (route.params?.checklistId || route.params?.checklistData || checklists.length === 0) {
          await loadChecklists();
        }

        const savedChecklistIds = await AsyncStorage.getItem('checklistIds');
        const parsedChecklistIds = savedChecklistIds ? JSON.parse(savedChecklistIds) : [];
        if (parsedChecklistIds.length > 0) setSelectedChecklistId(parsedChecklistIds[0]);

        if (route.params?.checklistTitle) {
          setChecklistTitle(route.params.checklistTitle);
          inspectionData.checklistTitle = route.params.checklistTitle;
        } else if (inspectionData.checklistTitle) setChecklistTitle(inspectionData.checklistTitle);

        if (route.params?.checklistId) {
          setSelectedChecklistId(route.params.checklistId);
          inspectionData.selectedChecklistId = route.params.checklistId;
        } else if (inspectionData.selectedChecklistId) setSelectedChecklistId(inspectionData.selectedChecklistId);

        // Handle checklist data from route params
        if (route.params?.checklistData) {
          const checklistData = route.params.checklistData;
          // Convert checklist data to the expected format
          const convertedChecklist = {
            id: checklistData.id,
            title: checklistData.title,
            description: checklistData.description || '',
            items: checklistData.sections ? checklistData.sections.map(section => ({
              text: section.sectionName,
              subItems: section.subItems.map(subItem => ({
                itemName: subItem.itemName,
                inputType: subItem.inputType || 'Ok/Reject/N/A',
                description: subItem.description || '',
                instruction: subItem.instruction || '',
                sortOrder: subItem.sortOrder || 1,
                isInspectionRequired: subItem.isInspectionRequired || false,
              }))
            })) : [],
            status: 'Available',
            isUsed: false,
            isUserChecklist: false,
          };
          
          // Add to checklists if not already present
          setChecklists(prevChecklists => {
            const existing = prevChecklists.find(c => c.id === convertedChecklist.id);
            if (!existing) {
              return [...prevChecklists, convertedChecklist];
            }
            return prevChecklists;
          });
          
          setSelectedChecklistId(checklistData.id);
          setChecklistTitle(checklistData.title);
          inspectionData.selectedChecklistId = checklistData.id;
          inspectionData.checklistTitle = checklistData.title;
        }

        if (equipment) {
          setSelectedEquipment(equipment);
          setMeterReading(equipment.meterReading?.toString() || '');
          inspectionData.selectedEquipment = equipment;
        } else if (inspectionData.selectedEquipment) {
          setSelectedEquipment(inspectionData.selectedEquipment);
          setMeterReading(inspectionData.selectedEquipment.meterReading?.toString() || '');
        }

        setCountry(inspectionData.country || null);
        setProject(inspectionData.project || null);
        setInspector(inspectionData.inspector || '');
        setCurrentMeterReading(inspectionData.currentMeterReading || '');
        setCoordinates(inspectionData.coordinates || '');
        setGpsAddress(inspectionData.gpsAddress || '');

        await saveInspectionData(inspectionData);
      } catch (error) {
        console.error('Error loading data:', error);
        Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load initial data. Please try again.', position: 'top' });
      }
    };
    loadData();

    return () => {
      isMounted.current = false;
    };
  }, [equipment, route.params]);

  // Additional useEffect to ensure checklists are loaded when route params change
  useEffect(() => {
    if (route.params?.checklistId || route.params?.checklistData) {
      const handleChecklistFromParams = async () => {
        try {
          // Load checklists if not already loaded
          if (checklists.length === 0) {
            await loadChecklists();
          }
          
          // Handle checklist data from route params
          if (route.params?.checklistData) {
            const checklistData = route.params.checklistData;
            const convertedChecklist = {
              id: checklistData.id,
              title: checklistData.title,
              description: checklistData.description || '',
              items: checklistData.sections ? checklistData.sections.map(section => ({
                text: section.sectionName,
                subItems: section.subItems.map(subItem => ({
                  itemName: subItem.itemName,
                  inputType: subItem.inputType || 'Ok/Reject/N/A',
                  description: subItem.description || '',
                  instruction: subItem.instruction || '',
                  sortOrder: subItem.sortOrder || 1,
                  isInspectionRequired: subItem.isInspectionRequired || false,
                }))
              })) : [],
              status: 'Available',
              isUsed: false,
              isUserChecklist: false,
            };
            
            setChecklists(prevChecklists => {
              const existing = prevChecklists.find(c => c.id === convertedChecklist.id);
              if (!existing) {
                return [...prevChecklists, convertedChecklist];
              }
              return prevChecklists;
            });
            
            setSelectedChecklistId(checklistData.id);
            setChecklistTitle(checklistData.title);
          }
        } catch (error) {
          console.error('Error handling checklist from params:', error);
        }
      };
      
      handleChecklistFromParams();
    }
  }, [route.params?.checklistId, route.params?.checklistData]);

  useEffect(() => {
    const inspectionData = {
      id: Date.now().toString(),
      requestNumber,
      country,
      project,
      inspector,
      selectedEquipment,
      meterReading,
      currentMeterReading,
      selectedChecklistId,
      checklistTitle,
      coordinates,
      gpsAddress,
      createdAt: new Date().toISOString(),
    };
    saveInspectionData(inspectionData);
  }, [country, project, inspector, selectedEquipment, meterReading, currentMeterReading, selectedChecklistId, checklistTitle, coordinates, gpsAddress, requestNumber]);

  useEffect(() => {
    const updateRequestNumber = async () => {
      const newRequestNumber = await generateRequestNumber();
      setRequestNumber(newRequestNumber);
    };
    updateRequestNumber();
  }, [country, project]);

  useEffect(() => {
    if (country === 'Zambia') {
      setProjectItems([
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kimteto', value: 'Kimteto' },
        { label: 'Kobold', value: 'Kobold' },
        { label: 'FQM Exploration', value: 'FQM Exploration' },
        { label: 'Mimosa', value: 'Mimosa' },
      ]);
    } else if (country === 'DRC') {
      setProjectItems([
        { label: 'Lubumbashi', value: 'Lubumbashi' },
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
        { label: 'Kamoa', value: 'Kamoa' },
      ]);
    } else {
      setProjectItems([]);
    }
    if (!country || (project && !projectItems.some((item) => item.value === project))) setProject(null);
  }, [country]);

  const loadChecklists = async () => {
    setSyncModalVisible(true);
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        Toast.show({ type: 'info', text1: 'Offline', text2: 'Cannot fetch checklists. Please check your internet connection.', position: 'top' });
        const storedChecklists = await AsyncStorage.getItem('checklists');
        setChecklists(storedChecklists ? JSON.parse(storedChecklists) : []);
        return;
      }

      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        Toast.show({ type: 'error', text1: 'Authentication Error', text2: 'Please log in to access checklists.', position: 'top' });
        setChecklists([]);
        return;
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      const userData = userSnapshot.val();
      if (!userData) {
        Toast.show({ type: 'error', text1: 'Authentication Error', text2: 'User not found. Please log in.', position: 'top' });
        navigation.replace('AdminLoginScreen');
        setChecklists([]);
        return;
      }
      const userId = Object.keys(userData)[0];

      const inspectionSnapshot = await database().ref(`/GlobalInspectionReport/admins/${userId}`).once('value');
      const inspections = inspectionSnapshot.val() || {};
      const usedChecklistIds = new Set(Object.values(inspections).map((i) => i.selectedChecklistId).filter(Boolean));

      // Fetch user-created checklists from InspectionChecklistCreatebyusers
      let userChecklistsArr = [];
      const userChecklistsSnap = await database().ref(`/InspectionChecklistCreatebyusers/admin/${ADMIN_UID}/checklists`).once('value');
      const userChecklistsData = userChecklistsSnap.val();
      if (userChecklistsData) {
        userChecklistsArr = Object.keys(userChecklistsData).map((key) => ({
          id: key,
          ...userChecklistsData[key],
        }));
      }

      // Fetch global checklists from AdminChecklistJson path
      const snapshot = await database().ref('/AdminChecklistJson/GlobalChecklistJson/Checklistfile/documents').once('value');
      const data = snapshot.val();
      let documentsArr = [];
      if (data) {
        if (Array.isArray(data)) {
          documentsArr = data;
        } else if (typeof data === 'object') {
          documentsArr = Object.values(data);
        }
      }

      // Map user checklists  
      const mappedUserChecklists = userChecklistsArr.map((doc) => ({
        id: doc.id,
        title: `${doc.title || 'My Checklist'} (My Checklist)`,
        description: doc.description || '',
        items: doc.items || [],
        status: 'Available',
        isUsed: usedChecklistIds.has(doc.id),
        isUserChecklist: true,
      }));

      // Map global checklists from AdminChecklistJson path
      const globalChecklists = documentsArr.map((doc) => {
        const items = Object.entries(doc.checklist || {}).map(([sectionKey, sectionItems]) => ({
          text: `${sectionKey}`,
          // Handle when a section is an object with an `items` array
          subItems: Array.isArray(sectionItems)
            ? sectionItems.map((item, index) => ({
                itemName: typeof item === 'string' ? item : item.itemName || 'Unnamed Item',
                inputType: 'Ok/Reject/N/A',
                description: '',
                instruction: '',
                sortOrder: index + 1,
                isInspectionRequired: false,
              }))
            : (sectionItems && Array.isArray(sectionItems.items)
                ? sectionItems.items.map((item, index) => ({
                    itemName: typeof item === 'string' ? item : item.itemName || 'Unnamed Item',
                    inputType: 'Ok/Reject/N/A',
                    description: '',
                    instruction: '',
                    sortOrder: index + 1,
                    isInspectionRequired: false,
                  }))
                : []),
        }));
        return {
          id: doc.document_id,
          title: doc.document_name,
          description: '',
          items,
          status: 'Available',
          isUsed: usedChecklistIds.has(doc.document_id),
          isUserChecklist: false,
          // Include all metadata fields from the original doc
          version: doc.version || '',
          version_date: doc.version_date || '',
          approved_by: doc.approved_by || '',
          document_id: doc.document_id || '',
          document_name: doc.document_name || '',
          Header_title: doc.Header_title || '',
          // Include header_inputs and footer_inputs for rendering
          header_inputs: doc.header_inputs || {},
          footer_inputs: doc.footer_inputs || {},
          // Include the full checklist structure
          checklist: doc.checklist || {},
        };
      });

      // Merge and sort checklists: user checklists at the top
      const allChecklists = [...mappedUserChecklists, ...globalChecklists.filter(g => !mappedUserChecklists.some(u => u.id === g.id))];
      await AsyncStorage.setItem('checklists', JSON.stringify(allChecklists));
      setChecklists(allChecklists);
    } catch (error) {
      console.error('Error loading checklists:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load checklists.', position: 'top' });
    } finally {
      setSyncModalVisible(false);
      setChecklistSearchQuery('');
      setIsChecklistSearchActive(false);
    }
  };

  const getFilteredEquipment = () => {
    let allEquipment = firebaseEquipment;

    if (selectedTab !== 'ALL') {
      allEquipment = allEquipment.filter((item) => {
        switch (selectedTab) {
          case 'VEHICLE': return ['Light Vehicle', 'Heavy Vehicle'].includes(item.equipmentCategory);
          case 'MACHINERY': return ['Machinery', 'Workshop Machinery'].includes(item.equipmentCategory);
          case 'CONTAINER': return item.equipmentCategory === 'Container';
          case 'RIG': return item.equipmentCategory === 'Rig';
          default: return true;
        }
      });
    }

    if (!equipmentSearchQuery.trim()) return allEquipment;
    return allEquipment.filter((item) =>
      [item.equipmentName, item.mainCategory, item.model].some((field) =>
        field?.toLowerCase().includes(equipmentSearchQuery.toLowerCase())
      )
    );
  };

  const handleSelectPress = () => {
    setEquipmentModalVisible(true); // Open modal immediately
    setEquipmentLoading(true); // Show loading indicator
  };

  // Load equipment data when modal is opened and loading is true
  useEffect(() => {
    if (equipmentModalVisible && equipmentLoading) {
      (async () => {
        try {
          await loadEquipmentData(false); // Don't show SyncModal when opening equipment modal
        } catch (error) {
          console.error('Error opening equipment modal:', error);
          Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load equipment data. Please try again.', position: 'top' });
        } finally {
          setEquipmentLoading(false);
        }
      })();
    }
  }, [equipmentModalVisible, equipmentLoading]);

  const handleChecklistSelectPress = async () => {
    try {
      await loadChecklists();
      setSelectedTab('ALL'); // Always show ALL by default
      setChecklistSearchQuery(''); // Clear search
      if (isMounted.current) setChecklistModalVisible(true);
    } catch (error) {
      console.error('Error opening checklist modal:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load checklist data. Please try again.', position: 'top' });
      if (isMounted.current) setChecklistModalVisible(true);
    }
  };

  const handleBackPress = () => {
    Alert.alert('Exit Step', 'Do you want to exit this step? Your data will be saved.', [
      { text: 'Cancel', style: 'cancel' },
      { text: 'OK', onPress: () => navigation.goBack(), style: 'default' },
    ], { cancelable: false });
  };

  const requestLocationPermission = async () => {
    try {
      if (Platform.OS === 'android') {
        const granted = await PermissionsAndroid.request(PermissionsAndroid.PERMISSIONS.ACCESS_FINE_LOCATION);
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      } else {
        return new Promise((resolve) =>
          Geolocation.requestAuthorization(resolve, () => resolve(false))
        );
      }
    } catch (error) {
      console.error('Error requesting location permission:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to request location permission.', position: 'top' });
      return false;
    }
  };

  const fetchAddressFromCoordinates = async (latitude, longitude) => {
    const apiKey = 'AIzaSyDDEh8W8buXtPi7uhhwJx4e5iK7Gb-vbe8';
    const url = `https://maps.googleapis.com/maps/api/geocode/json?latlng=${latitude},${longitude}&key=${apiKey}`;

    try {
      const response = await fetch(url);
      const data = await response.json();
      if (data.status === 'OK' && data.results.length > 0) return data.results[0].formatted_address;
      throw new Error(data.status === 'ZERO_RESULTS' ? 'No address found' : 'Geocoding API error');
    } catch (error) {
      console.error('Error fetching address:', error.message);
      Toast.show({ type: 'error', text1: 'Address Fetch Failed', text2: error.message || 'Using coordinates only.', position: 'top' });
      return `Latitude: ${latitude}, Longitude: ${longitude} (Address unavailable)`;
    }
  };

  const handleGetLocation = async () => {
    let timeoutId;
    try {
      const hasPermission = await requestLocationPermission();
      if (!hasPermission) {
        Toast.show({ type: 'error', text1: 'Location Error', text2: 'Location permission denied.', position: 'top' });
        return;
      }
      setSyncModalVisible(true);
      timeoutId = setTimeout(() => {
        if (isMounted.current && syncModalVisible) {
          setSyncModalVisible(false);
          Toast.show({ type: 'error', text1: 'Error', text2: 'Location fetching timed out.', position: 'top' });
        }
      }, 15000);

      const position = await new Promise((resolve, reject) =>
        Geolocation.getCurrentPosition(resolve, reject, { enableHighAccuracy: true, timeout: 15000, maximumAge: 10000 })
      );
      const { latitude, longitude } = position.coords;
      const coords = `${latitude}, ${longitude}`;
      setCoordinates(coords);

      const address = await fetchAddressFromCoordinates(latitude, longitude);
      setGpsAddress(address);

      const savedInspection = await AsyncStorage.getItem('inspectionStep1');
      let inspectionData = savedInspection ? JSON.parse(savedInspection) : {};
      inspectionData.coordinates = coords;
      inspectionData.gpsAddress = address;
      await saveInspectionData(inspectionData);

      Toast.show({ type: 'success', text1: 'Location Fetched', text2: `Coordinates: ${coords}\nAddress: ${address}`, position: 'top', visibilityTime: 4000 });
    } catch (error) {
      console.error('Error in handleGetLocation:', error);
      let errorMsg = 'An unexpected error occurred.';
      if (error && error.message) {
        errorMsg = error.message;
      } else if (typeof error === 'string') {
        errorMsg = error;
      }
      Toast.show({ type: 'error', text1: 'Location Error', text2: errorMsg, position: 'top' });
    } finally {
      if (timeoutId) clearTimeout(timeoutId);
      if (isMounted.current) setSyncModalVisible(false);
    }
  };

  const validateChecklistItems = (checklistId) => {
    console.log('Validating checklist:', checklistId);
    console.log('Available checklists:', checklists.length);
    console.log('Selected checklist:', checklists.find(c => c.id === checklistId));
    
    if (!checklistId) {
      Toast.show({ type: 'error', text1: 'Checklist Required', text2: 'Please select a checklist.', position: 'top' });
      return false;
    }

    const checklist = checklists.find((c) => c.id === checklistId);
    console.log('Found checklist:', checklist);
    
    if (!checklist) {
      Toast.show({ type: 'error', text1: 'Checklist Not Found', text2: 'The selected checklist could not be found. Please try selecting it again.', position: 'top' });
      return false;
    }

    // For user-created checklists
    if (checklist.isUserChecklist) {
      if (!checklist.items || !Array.isArray(checklist.items) || checklist.items.length === 0) {
        Toast.show({ type: 'error', text1: 'Invalid Checklist', text2: 'The selected checklist is empty or invalid.', position: 'top' });
        return false;
      }
      
      return checklist.items.every(
        (item) =>
          item &&
          typeof item.text === 'string' &&
          Array.isArray(item.subItems) &&
          item.subItems.length > 0 &&
          item.subItems.every(
            (subItem) =>
              subItem &&
              typeof subItem.itemName === 'string' &&
              typeof subItem.inputType === 'string'
          )
      );
    }

    // For global checklists
    if (!checklist.items || !Array.isArray(checklist.items) || checklist.items.length === 0) {
      Toast.show({ type: 'error', text1: 'Invalid Checklist', text2: 'The selected checklist is empty or invalid.', position: 'top' });
      return false;
    }

    return checklist.items.every(
      (item) =>
        item &&
        typeof item.text === 'string' &&
        Array.isArray(item.subItems) &&
        item.subItems.length > 0 &&
        item.subItems.every(
          (subItem) =>
            subItem &&
            typeof subItem.itemName === 'string' &&
            typeof subItem.inputType === 'string'
        )
    );
  };

  // In handleNextPress, after saving the inspection, update the global lastRequestNumber
  const handleNextPress = async () => {
    setShowErrors(true);
    
    // Validate all required fields
    const hasCountry = !!country;
    const hasProject = !!project;
    const hasInspector = !!inspector.trim();
    const hasEquipment = !!selectedEquipment;
    const hasChecklist = !!selectedChecklistId;
    
    setCountryError(!hasCountry);
    setProjectError(!hasProject);
    setInspectorError(!hasInspector);
    setEquipmentError(!hasEquipment);
    setChecklistError(!hasChecklist);

    // Check if all basic fields are filled
    if (!hasCountry || !hasProject || !hasInspector || !hasEquipment || !hasChecklist) {
      const missingFields = [];
      if (!hasCountry) missingFields.push('Country');
      if (!hasProject) missingFields.push('Project');
      if (!hasInspector) missingFields.push('Inspector');
      if (!hasEquipment) missingFields.push('Equipment');
      if (!hasChecklist) missingFields.push('Checklist');
      
      Toast.show({ 
        type: 'error', 
        text1: 'Required Fields Missing', 
        text2: `Please fill: ${missingFields.join(', ')}`, 
        position: 'top' 
      });
      return;
    }

    // Validate meter reading
    if (selectedEquipment && meterReading && parseFloat(meterReading) <= parseFloat(selectedEquipment.meterReading)) {
      Toast.show({
        type: 'error',
        text1: 'Invalid Meter Reading',
        text2: `New meter reading (${meterReading}) must be greater than ${selectedEquipment.meterReading || '0'}.`,
        position: 'top',
      });
      return;
    }

    // Validate checklist structure
    console.log('Validating checklist:', selectedChecklistId);
    console.log('Available checklists:', checklists.length);
    console.log('Selected checklist:', checklists.find(c => c.id === selectedChecklistId));
    
    if (!validateChecklistItems(selectedChecklistId)) {
      setChecklistError(true);
      return;
    }

    setSyncModalVisible(true);
    try {
      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        Alert.alert('Authentication Error', 'User not authenticated. Please log in.', [
          { text: 'OK', onPress: () => navigation.replace('AdminLoginScreen') }
        ]);
        return;
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      const userData = userSnapshot.val();
      if (!userData) {
        Alert.alert('Authentication Error', 'User not found. Please log in.', [
          { text: 'OK', onPress: () => navigation.replace('AdminLoginScreen') }
        ]);
        return;
      }
      const userId = Object.keys(userData)[0];

      const inspectionData = {
        id: Date.now().toString(),
        requestNumber,
        country,
        project,
        inspector,
        selectedEquipment,
        meterReading,
        currentMeterReading,
        selectedChecklistId,
        checklistTitle,
        coordinates,
        gpsAddress,
        createdAt: new Date().toISOString(),
      };

      if (selectedEquipment && meterReading && parseFloat(meterReading) !== parseFloat(selectedEquipment.meterReading)) {
        const updatedEquipment = { ...selectedEquipment, meterReading: parseFloat(meterReading), updatedAt: new Date().toISOString() };
        if (isOnline) {
          const snapshot = await database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData').once('value');
          let globalData = snapshot.val() || [];
          const index = globalData.findIndex((item) => item.id === updatedEquipment.id);
          if (index !== -1) globalData[index] = updatedEquipment;
          else globalData.push(updatedEquipment);
          await database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData').set(globalData);
        } else {
          const storedEquipment = await AsyncStorage.getItem('equipment');
          let equipmentList = storedEquipment ? JSON.parse(storedEquipment) : [];
          const index = equipmentList.findIndex((item) => item.id === updatedEquipment.id);
          if (index !== -1) equipmentList[index] = { ...updatedEquipment, pendingSync: true };
          else equipmentList.push({ ...updatedEquipment, pendingSync: true });
          await AsyncStorage.setItem('equipment', JSON.stringify(equipmentList));
        }
        setFirebaseEquipment((prev) => prev.map((item) => (item.id === updatedEquipment.id ? updatedEquipment : item)));
        setSelectedEquipment(updatedEquipment);
        inspectionData.selectedEquipment = updatedEquipment;
      }

      await AsyncStorage.setItem('selectedChecklistIds', JSON.stringify([selectedChecklistId]));
      await saveInspectionData(inspectionData);

      // Generate a new request number for the next inspection
      const newRequestNumber = await generateRequestNumber();
      setRequestNumber(newRequestNumber);

      setShowErrors(false);
      setCountryError(false);
      setProjectError(false);
      setInspectorError(false);
      setEquipmentError(false);
      setChecklistError(false);
      setActiveStep(2);
    } catch (error) {
      console.error('Error saving inspection data:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to save inspection data locally.', position: 'top' });
    } finally {
      setSyncModalVisible(false);
    }
  };

  const handleChecklistSearch = (query) => setChecklistSearchQuery(query);
  const handleEquipmentSearch = (query) => setEquipmentSearchQuery(query);

  const filteredChecklists = Array.isArray(checklists)
    ? checklists.filter((checklist) =>
        (typeof checklist.title === 'string' ? checklist.title : '').toLowerCase().includes(checklistSearchQuery.toLowerCase())
      )
    : [];

  const toggleChecklistSelection = async (checklistId, title) => {
    console.log('toggleChecklistSelection called with:', checklistId, title);
    const selectedChecklist = checklists.find((c) => c.id === checklistId);
    console.log('Selected checklist data:', selectedChecklist);
    if (selectedChecklist) {
      // Save the full checklist object (with metadata) to AsyncStorage
      try {
        console.log('Saving checklist to AsyncStorage:', selectedChecklist);
        await AsyncStorage.setItem(`checklist_${checklistId}`, JSON.stringify(selectedChecklist));
        console.log('Checklist saved successfully');
        // Save footer_inputs separately for quick offline access in Step 3
        try {
          await AsyncStorage.setItem('selectedChecklistFooterInputs', JSON.stringify(selectedChecklist.footer_inputs || {}));
        } catch (e2) {
          console.warn('Unable to save selectedChecklistFooterInputs:', e2?.message);
        }
      } catch (e) {
        console.error('Error saving checklist metadata to AsyncStorage:', e);
      }
    }
    setSelectedChecklistId((prev) => {
      const newId = prev === checklistId ? null : checklistId;
      if (newId) {
        setChecklistTitle(title || 'Untitled Checklist');
        console.log('Selected checklistId:', newId, 'Title:', title);
      } else {
        setChecklistTitle('');
        console.log('Checklist deselected');
      }
      AsyncStorage.setItem('selectedChecklistIds', JSON.stringify(newId ? [newId] : []))
        .catch((error) => console.error('Error saving selected checklist:', error));
      return newId;
    });
    setChecklistModalVisible(false);
  };

  const selectEquipment = async (equipment) => {
    setSelectedEquipment(equipment);
    setMeterReading(equipment.meterReading?.toString() || '');
    setEquipmentModalVisible(false);
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Available': return '#2ecc71';
      case 'Maintenance Required': return '#f1c40f';
      case 'In Maintenance': return '#e67e22';
      case 'Breakdown': return '#e74c3c';
      case 'Deploy': return '#3498db';
      default: return '#666';
    }
  };

  const renderChecklistItem = ({ item }) => (
    <TouchableOpacity style={styles.checklistItem} onPress={() => toggleChecklistSelection(item?.id, item?.title)}>
      <View style={styles.checklistItemContent}>
        <MaterialIcons name={selectedChecklistId === item?.id ? 'check-box' : 'check-box-outline-blank'} size={24} color={selectedChecklistId === item?.id ? '#0078D4' : '#666'} />
        <View style={styles.checklistItemTextContainer}>
          <Text style={styles.checklistItemText}>{item?.title || 'Untitled Checklist'}</Text>
          {item?.document_id && (
            <Text style={styles.checklistItemSubText}>ID: {item.document_id}</Text>
          )}
        </View>
      </View>
    </TouchableOpacity>
  );

  const renderEquipmentItem = ({ item }) => (
    <View style={styles.equipmentRow}>
      <TouchableOpacity style={styles.equipmentInfo} onPress={() => selectEquipment(item)}>
        <View style={styles.equipmentDetailItemheader}>
          <Text style={styles.detailLabelheader}>{item.equipmentName || 'Tyre - Tyre'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Category</Text>
          <Text style={styles.detailValue}>{item.mainCategory || 'Sample Category'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Model</Text>
          <Text style={styles.detailValue}>{item.model || '326F'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Meter Reading</Text>
          <Text style={styles.detailValue}>{item.meterReading || '0'} KM</Text>
        </View>
        <View style={styles.equipmentDetailItemNoBorder}>
          <Text style={[styles.detailLabelstatus, { color: getStatusColor(item.status || 'Available') }]}>{item.status || 'Available'}</Text>
        </View>
      </TouchableOpacity>
      <TouchableOpacity style={styles.selectEquipmentButton} onPress={() => selectEquipment(item)}>
        <Text style={styles.selectEquipmentText}>{selectedEquipment && selectedEquipment.id === item.id ? 'Selected' : 'Select'}</Text>
      </TouchableOpacity>
    </View>
  );

  return (
    <>
      <KeyboardAvoidingView style={styles.container} behavior={Platform.OS === 'ios' ? 'padding' : 'height'} keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}>
        <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} keyboardShouldPersistTaps="handled">
          {checklists.length === 0 && (
            <View style={{flex: 1, justifyContent: 'center', alignItems: 'center', marginTop: 0}}>
              {/* <Text style={{color: '#888', fontSize: 18}}>No checklists available.</Text> */}
            </View>
          )}
          <View style={styles.headingContainer}><Text style={styles.headingText}>New Inspection</Text></View>
          <View style={styles.requestRow}><Text style={styles.requestLabel}>Request #</Text><Text style={styles.requestValue}>{requestNumber}</Text></View>
          <View style={inputStyles.container}>
            <Text style={inputStyles.label}>Country<Text style={inputStyles.requiredError}> *</Text></Text>
            <Dropdown
              style={[inputStyles.input, countryError && showErrors ? inputStyles.inputError : null]}
              placeholderStyle={styles.placeholderStyle}
              selectedTextStyle={styles.dropdownText}
              data={countryItems}
              maxHeight={300}
              labelField="label"
              valueField="value"
              placeholder="Select Country"
              value={country}
              onChange={item => setCountry(item.value)}
            />
            {countryError && showErrors && <Text style={inputStyles.errorText}>Country is required</Text>}
          </View>
          <View style={inputStyles.container}>
            <Text style={inputStyles.label}>Project<Text style={inputStyles.requiredError}> *</Text></Text>
            <Dropdown
              style={[inputStyles.input, !country && styles.disabledDropdown, projectError && showErrors ? inputStyles.inputError : null]}
              placeholderStyle={styles.placeholderStyle}
              selectedTextStyle={styles.dropdownText}
              data={projectItems}
              maxHeight={300}
              labelField="label"
              valueField="value"
              placeholder="Select Project"
              value={project}
              onChange={item => setProject(item.value)}
              disable={!country}
            />
            {projectError && showErrors && <Text style={inputStyles.errorText}>Project is required</Text>}
          </View>
          <CustomInputField label="Inspector" value={inspector} onChangeText={setInspector} placeholder="Enter Inspector" isRequired showError={inspectorError && showErrors} errorMessage="Inspector is required" />
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Equipments</Text>
            <TouchableOpacity onPress={handleSelectPress} style={styles.selectButton}>
              <Text style={styles.selectText}>{selectedEquipment ? 'Change equipment' : 'Select'}</Text>
              <MaterialIcons name="chevron-right" size={16} color="#0078D4" />
            </TouchableOpacity>
          </View>
          {equipmentError && showErrors && <Text style={[inputStyles.errorText, { marginHorizontal: 15 }]}>Equipment is required</Text>}
          {selectedEquipment && (
            <View style={styles.equipmentDetailsBox}>
              <View style={styles.vehicleEquipmentRow}><Text style={styles.vehicleEquipmentText}>Vehicle/Model: {selectedEquipment.vehicleNumber || selectedEquipment.model || 'N/A'}, Category: {selectedEquipment.mainCategory || 'N/A'}, Equipment Name: {selectedEquipment.equipmentName || 'N/A'}</Text></View>
              <View style={styles.meterReadingContainer}>
                <View style={styles.labelRow}><Text style={styles.meterReadingLabel}><Text style={styles.required}>*</Text>Meter Reading</Text></View>
                <View style={[styles.inputRow, showErrors && !meterReading && styles.inputError]}>
                  <TextInput
                    style={styles.meterReadingInput}
                    value={meterReading}
                    onChangeText={setMeterReading}
                    onBlur={() => {
                      if (meterReading) {
                        const currentReading = parseFloat(selectedEquipment.meterReading) || 0;
                        const newReading = parseFloat(meterReading);
                        if (!isNaN(newReading) && newReading < currentReading) {
                          Toast.show({ type: 'error', text1: 'Invalid Reading', text2: `Meter reading must be greater than ${currentReading}`, position: 'top' });
                        }
                      }
                    }}
                    placeholder="Enter meter reading"
                    placeholderTextColor="#888"
                    keyboardType="numeric"
                  />
                  <Text style={styles.meterUnit}>KM</Text>
                </View>
                {showErrors && !meterReading && <Text style={styles.errorText}>Meter reading is required</Text>}
                <View style={styles.currentMeterRow}>
                  <Text style={styles.currentMeterLabel}>Current Meter Reading: </Text>
                  <Text style={styles.currentMeterValue}>{selectedEquipment.meterReading || '0'} KM</Text>
                </View>
              </View>
            </View>
          )}
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Checklist</Text>
            <TouchableOpacity onPress={handleChecklistSelectPress} style={styles.selectButton}>
              <Text style={styles.selectText}>{checklistTitle ? 'Change checklist' : 'Select'}</Text>
              <MaterialIcons name="chevron-right" size={16} color="#0078D4" />
            </TouchableOpacity>
          </View>
          <View style={styles.selectedChecklistsBox}>
            {checklistTitle ? <Text style={styles.selectedChecklistText}>{checklistTitle}</Text> : <Text style={styles.noChecklistText}>No checklist selected</Text>}
          </View>
          {checklistError && showErrors && <Text style={[inputStyles.errorText, { marginHorizontal: 15 }]}>Checklist is required</Text>}
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>GPS Location</Text>
            <TouchableOpacity onPress={handleGetLocation} style={styles.selectButton}>
              <MaterialIcons name="location-on" size={16} color="#0078D4" />
              <Text style={styles.selectText}>Get Location</Text>
            </TouchableOpacity>
          </View>
          <View style={styles.gpsDetailsBox}>
            {coordinates && <View style={styles.gpsInfoRow}><Text style={styles.gpsLabel}>Coordinates: </Text><Text style={styles.gpsValue}>{coordinates}</Text></View>}
            {gpsAddress && <View style={styles.gpsInfoRow}><Text style={styles.gpsLabel}>GPS Address: </Text><Text style={styles.gpsValue}>{gpsAddress}</Text></View>}
            {!coordinates && !gpsAddress && <Text style={styles.noGpsText}>Click "Get Location" to fetch your current location</Text>}
          </View>
        </ScrollView>
        <View style={styles.buttonContainer}>
          <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
            <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
            <Text style={styles.cancelButtonText}>Cancel</Text>
          </TouchableOpacity>
          <TouchableOpacity style={styles.nextButton} onPress={handleNextPress}>
            <Text style={styles.nextButtonText}>Next</Text>
            <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
          </TouchableOpacity>
        </View>
        <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message="Processing" />
        <Toast />
      </KeyboardAvoidingView>
      {/* Render modals outside of KeyboardAvoidingView for iOS compatibility */}
      <Modal visible={checklistModalVisible} transparent animationType="fade" onRequestClose={() => setChecklistModalVisible(false)}>
        <View pointerEvents="box-none" style={styles.modalOverlay}>
          <View style={styles.checklistModalContent}>
            <View style={styles.checklistModalHeader}>
              <Text style={styles.checklistModalTitle}>Select Checklist</Text>
              <TouchableOpacity onPress={() => setChecklistModalVisible(false)} style={styles.modalCloseIcon}>
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            <View style={styles.checklistModalTabs}>
              <TouchableOpacity style={[styles.tabButton, selectedTab === 'ALL' && styles.activeTabButton]} onPress={() => setSelectedTab('ALL')}>
                <Text style={[styles.tabText, selectedTab === 'ALL' && styles.activeTabText]}>ALL</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.tabButton, selectedTab === 'Assign' && styles.activeTabButton]} onPress={() => setSelectedTab('Assign')}>
                <Text style={[styles.tabText, selectedTab === 'Assign' && styles.activeTabText]}>Assign</Text>
              </TouchableOpacity>
            </View>
            <View style={styles.checklistSearchContainer}>
              {isChecklistSearchActive ? (
                <TextInput
                  style={styles.checklistSearchInput}
                  placeholder="Search checklists..."
                  placeholderTextColor="#888"
                  value={checklistSearchQuery}
                  onChangeText={handleChecklistSearch}
                  autoFocus
                />
              ) : (
                <Text style={styles.checklistSelectText}>Search Checklists</Text>
              )}
              <TouchableOpacity onPress={() => { setIsChecklistSearchActive(!isChecklistSearchActive); if (isChecklistSearchActive) setChecklistSearchQuery(''); }}>
                <MaterialIcons name={isChecklistSearchActive ? 'close' : 'search'} size={24} color="#666" />
              </TouchableOpacity>
            </View>
            <View style={styles.divider} />
            <FlatList
              data={filteredChecklists}
              keyExtractor={(item, idx) => (item?.id ? item.id.toString() : `checklist-${idx}`)}
              renderItem={renderChecklistItem}
              ListEmptyComponent={<Text style={styles.checklistEmptyText}>No checklists found</Text>}
              style={styles.checklistList}
            />
            <View style={styles.divider} />
            <View style={styles.checklistModalFooter}>
              <TouchableOpacity onPress={() => { setChecklistModalVisible(false); navigation.navigate('Checklist_home', { checklist: null }); }}>
                <Text style={styles.createChecklistText}>Create New Checklist</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.checklistCloseButton} onPress={() => setChecklistModalVisible(false)}>
                <Text style={styles.checklistCloseText}>Close</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      <Modal visible={equipmentModalVisible} transparent animationType="fade" onRequestClose={() => setEquipmentModalVisible(false)}>
        <View pointerEvents="box-none" style={styles.modalOverlay}>
          <View style={styles.equipmentModalContent}>
            <View style={styles.equipmentModalHeader}>
              <Text style={styles.equipmentModalTitle}>Select Equipments</Text>
              <TouchableOpacity onPress={() => setEquipmentModalVisible(false)} style={styles.modalCloseIcon}>
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            {equipmentLoading ? (
              <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center', minHeight: 120 }}>
                <Text style={{ color: '#015185', fontSize: 16, marginBottom: 10 }}>Loading equipment...</Text>
                <Animated.View style={{ width: 40, height: 40 }}>
                  <MaterialIcons name="autorenew" size={40} color="#015185" style={{ opacity: 0.7 }} />
                </Animated.View>
              </View>
            ) : (
              <>
                <TouchableOpacity
                  style={[styles.equipmentSearchContainer, isEquipmentSearchActive && styles.equipmentSearchContainerActive]}
                  onPress={() => { setIsEquipmentSearchActive(true); equipmentSearchInputRef.current?.focus(); }}
                >
                  {isEquipmentSearchActive ? (
                    <TextInput
                      ref={equipmentSearchInputRef}
                      style={[styles.equipmentSearchInput, isEquipmentSearchActive && styles.equipmentSearchInputActive]}
                      placeholder="Search..."
                      placeholderTextColor="#888"
                      value={equipmentSearchQuery}
                      onChangeText={handleEquipmentSearch}
                      autoFocus
                    />
                  ) : (
                    <Text style={styles.equipmentSelectText}>Search Equipment</Text>
                  )}
                  <TouchableOpacity onPress={() => { setIsEquipmentSearchActive(!isEquipmentSearchActive); if (isEquipmentSearchActive) setEquipmentSearchQuery(''); equipmentSearchInputRef.current?.blur(); }}>
                    <MaterialIcons name={isEquipmentSearchActive ? 'close' : 'search'} size={24} color="#666" />
                  </TouchableOpacity>
                </TouchableOpacity>
                <View style={styles.divider} />
                <FlatList
                  data={getFilteredEquipment()}
                  keyExtractor={(item) => item.id.toString()}
                  renderItem={renderEquipmentItem}
                  ListEmptyComponent={<Text style={styles.equipmentEmptyText}>No equipment found</Text>}
                  style={styles.equipmentList}
                />
                <View style={styles.divider} />
                <View style={styles.equipmentModalFooter}>
                  <TouchableOpacity onPress={() => { setEquipmentModalVisible(false); navigation.navigate('Add_equipment'); }}>
                    <Text style={styles.addEquipmentText}>Add your own equipment</Text>
                  </TouchableOpacity>
                  <TouchableOpacity style={styles.equipmentCloseButton} onPress={() => setEquipmentModalVisible(false)}>
                    <Text style={styles.equipmentCloseText}>Close</Text>
                  </TouchableOpacity>
                </View>
              </>
            )}
          </View>
        </View>
      </Modal>
    </>
  );
};


const inputStyles = StyleSheet.create({
  container: { marginBottom: width * 0.04 },
  label: { fontSize: width * 0.038, color: '#015185', marginBottom: 4, fontWeight: '500' },
  requiredError: { color: 'red', fontWeight: 'bold' },
  input: { borderWidth: 1, borderColor: '#015185', borderRadius: 5, paddingHorizontal: 12, paddingVertical: 14, fontSize: width * 0.04, backgroundColor: '#fff' },
  inputError: { borderColor: 'red' },
  errorText: { color: 'red', fontSize: width * 0.032, marginTop: 2, marginLeft: 2 },
});

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#fff' },
  mainContent: { flex: 1, paddingHorizontal: width * 0.04, paddingVertical: width * 0.025 },
  scrollContent: { paddingBottom: 130 },
  headingContainer: { justifyContent: 'center', alignItems: 'center', paddingVertical: 10 },
  headingText: { fontSize: 20, fontWeight: '500', color: '#015185' },
  requestRow: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', height: 50, backgroundColor: '#f0f0f0', paddingHorizontal: 15, marginBottom: 20 },
  requestLabel: { fontSize: 16, color: '#015185' },
  requestValue: { fontSize: 16, color: '#015185' },
  dropdownContainer: { borderWidth: 1, borderColor: '#015185', borderRadius: 5, backgroundColor: '#fff' },
  dropdownText: { fontSize: 16, color: '#333' },
  placeholderStyle: { fontSize: 16, color: '#888' },
  disabledDropdown: { backgroundColor: '#e0e0e0' },
  selectRow: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', width: '100%', height: 50, backgroundColor: '#f0f0f0', paddingHorizontal: 15, borderRadius: 5, marginTop: 5 },
  selectLabel: { fontSize: 16, color: '#015185', fontWeight: '600' },
  selectButton: { flexDirection: 'row', alignItems: 'center' },
  selectText: { fontSize: 16, color: '#0078D4', marginLeft: 5 },
  equipmentDetailsBox: { width: '100%', backgroundColor: '#f0f0f0', paddingVertical: 10, paddingHorizontal: 15, marginBottom: 10, borderRadius: 5, alignSelf: 'center' },
  gpsDetailsBox: { width: '100%', backgroundColor: '#f0f0f0', paddingVertical: 10, paddingHorizontal: 15, marginBottom: 10, borderRadius: 5, alignSelf: 'center' },
  gpsInfoRow: { flexDirection: 'row', flexWrap: 'wrap', marginBottom: 5 },
  gpsLabel: { fontSize: 16, color: '#333', fontWeight: '600' },
  gpsValue: { fontSize: 16, color: '#333', flex: 1 },
  noGpsText: { fontSize: 14, color: '#888', textAlign: 'center' },
  vehicleEquipmentRow: { paddingVertical: 10 },
  vehicleEquipmentText: { fontSize: 16, color: '#333' },
  meterReadingContainer: { padding: 12, backgroundColor: '#F9F9F9', borderRadius: 6, marginTop: 8 },
  labelRow: { marginBottom: 8 },
  meterReadingLabel: { fontSize: 14, color: '#333', fontWeight: '500' },
  inputRow: { flexDirection: 'row', alignItems: 'center', backgroundColor: '#fff', borderWidth: 1, borderColor: '#ddd', borderRadius: 4, paddingHorizontal: 12, paddingVertical: 8 },
  meterReadingInput: { flex: 1, fontSize: 16, color: '#333', padding: 0 },
  meterUnit: { color: '#666', fontSize: 14, marginLeft: 8 },
  currentMeterRow: { flexDirection: 'row', marginTop: 8, alignItems: 'center' },
  currentMeterLabel: { fontSize: 13, color: '#666' },
  currentMeterValue: { fontSize: 13, color: '#333', fontWeight: '500' },
  buttonContainer: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', paddingHorizontal: 15, paddingVertical: 15, backgroundColor: '#F2F2F2', borderTopWidth: 1, borderTopColor: '#ddd', position: 'absolute', bottom: 0, left: 0, right: 0, elevation: 5, shadowColor: '#000', shadowOffset: { width: 0, height: -2 }, shadowOpacity: 0.1, shadowRadius: 2, paddingBottom: 30 },
  cancelButton: { flexDirection: 'row', alignItems: 'center', justifyContent: 'center', paddingBottom: 30, paddingHorizontal: 15 },
  cancelButtonText: { color: '#015185', fontSize: 16, fontWeight: '600' },
  nextButton: { flexDirection: 'row', alignItems: 'center', justifyContent: 'center', paddingBottom: 30, borderRadius: 5 },
  nextButtonText: { color: '#015185', fontSize: 16, fontWeight: '600' },
  modalOverlay: { flex: 1, justifyContent: 'center', alignItems: 'center', backgroundColor: 'rgba(0,0,0,0.5)' },
  checklistModalContent: { width: '90%', maxHeight: '70%', backgroundColor: '#fff', borderRadius: 10, padding: 20, elevation: 5, shadowColor: '#000', shadowOffset: { width: 0, height: 2 }, shadowOpacity: 0.3, shadowRadius: 4 },
  checklistModalHeader: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', marginBottom: 15 },
  checklistModalTitle: { fontSize: 18, fontWeight: '600', color: '#015185' },
  modalCloseIcon: { padding: 5 },
  checklistModalTabs: { flexDirection: 'row', justifyContent: 'space-between', marginBottom: 10 },
  tabButton: { flex: 1, alignItems: 'center', paddingVertical: 10, borderBottomWidth: 2, borderBottomColor: 'transparent' },
  activeTabButton: { borderBottomColor: '#0078D4' },
  tabText: { fontSize: 16, color: '#666', fontWeight: '600' },
  activeTabText: { color: '#0078D4' },
  checklistSearchContainer: { flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', paddingVertical: 10, borderWidth: 1, borderColor: '#ccc', borderRadius: 5, paddingHorizontal: 10 },
  checklistSelectText: { fontSize: 16, color: '#333', flex: 1 },
  checklistSearchInput: { flex: 1, height: 40, fontSize: 16, color: '#333' },
  divider: { height: 1, backgroundColor: '#eee', marginVertical: 10 },
  checklistList: { maxHeight: 200 },
  checklistItem: { paddingVertical: 10 },
  checklistItemContent: { flexDirection: 'row', alignItems: 'center' },
  checklistItemTextContainer: { flex: 1, marginLeft: 10 },
  checklistItemText: { fontSize: 16, color: '#333' },
  checklistItemSubText: { fontSize: 12, color: '#888', marginTop: 2 },
  checklistEmptyText: { fontSize: 16, color: '#888', textAlign: 'center', marginVertical: 20 },
  checklistModalFooter: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', paddingVertical: 10 },
  createChecklistText: { fontSize: 16, color: '#0078D4', fontWeight: '600' },
  checklistCloseButton: { padding: 10 },
  checklistCloseText: { fontSize: 16, color: '#3481BC', fontWeight: '600' },
  equipmentModalContent: { width: '90%', maxHeight: '70%', backgroundColor: '#fff', borderRadius: 10, padding: 20, elevation: 5, shadowColor: '#000', shadowOffset: { width: 0, height: 2 }, shadowOpacity: 0.3, shadowRadius: 4 },
  equipmentModalHeader: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', paddingVertical: 10 },
  equipmentModalTitle: { fontSize: 18, fontWeight: '600', color: '#015185' },
  equipmentSearchContainer: { flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', paddingVertical: 10, borderWidth: 1, borderColor: '#ccc', borderRadius: 5, paddingHorizontal: 10, marginBottom: 10 },
  equipmentSearchContainerActive: { paddingVertical: 2 },
  equipmentSelectText: { fontSize: 16, color: '#333', flex: 1 },
  equipmentSearchInput: { flex: 1, height: 40, fontSize: 16, color: '#333' },
  equipmentSearchInputActive: { flex: 1, height: 40, fontSize: 16, color: '#333' },
  equipmentList: { maxHeight: 300, paddingHorizontal: 12 },
  equipmentRow: { marginVertical: 5, backgroundColor: '#F2F2F2', borderRadius: 5, borderWidth: 1, borderColor: '#eee', paddingRight: 10 },
  equipmentInfo: { flex: 1 },
  equipmentDetailItem: { flexDirection: 'row', justifyContent: 'space-between', paddingVertical: 5, borderBottomWidth: 1, borderBottomColor: '#ddd', paddingHorizontal: '5%', paddingRight: '10%' },
  equipmentDetailItemNoBorder: { flexDirection: 'row', justifyContent: 'space-between', paddingVertical: 5, paddingLeft: '5%' },
  equipmentDetailItemheader: { flexDirection: 'row', justifyContent: 'space-between', paddingVertical: 5, backgroundColor: '#E9E9E9', paddingHorizontal: '5%' },
  detailLabel: { fontSize: 14, color: '#666', fontWeight: '400', width: '40%', marginVertical: 3 },
  detailLabelstatus: { fontSize: 14, fontWeight: '400', width: '60%', marginVertical: 3 },
  detailLabelheader: { fontSize: 18, color: '#015185', fontWeight: '600', width: '100%' },
  detailValue: { fontSize: 16, color: '#015185', width: '60%', textAlign: 'right', fontWeight: '500' },
  selectEquipmentButton: { position: 'absolute', right: 10, bottom: 10, backgroundColor: '#0078D4', paddingVertical: 5, paddingHorizontal: 10, borderRadius: 5 },
  selectEquipmentText: { color: '#fff', fontSize: 14, fontWeight: '600' },
  equipmentEmptyText: { fontSize: 16, color: '#888', textAlign: 'center', marginVertical: 20 },
  equipmentModalFooter: { flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center', paddingVertical: 10 },
  addEquipmentText: { fontSize: 16, color: '#0078D4', fontWeight: '600' },
  equipmentCloseButton: { padding: 10 },
  equipmentCloseText: { fontSize: 16, color: '#3481BC', fontWeight: '600' },
  selectedChecklistsBox: { width: '94%', paddingVertical: 10, paddingHorizontal: 15, marginBottom: 10, alignSelf: 'center' },
  selectedChecklistText: { fontSize: 14, color: '#333', marginBottom: 5 },
  noChecklistText: { fontSize: 14, color: '#888' },
  bottomicon: { marginTop: 2 },
  syncModalContent: { width: '80%', backgroundColor: '#fff', borderRadius: 10, padding: 20, alignItems: 'center', elevation: 5, shadowColor: '#000', shadowOffset: { width: 0, height: 2 }, shadowOpacity: 0.3, shadowRadius: 4 },
  syncModalText: { fontSize: 18, fontWeight: '600', color: '#015185', marginBottom: 10 },
  syncModalTextadd: { fontSize: 14, color: '#666', textAlign: 'center' },
  syncIcondiv: { alignItems: 'center' },
  syncIcon: { width: 30, height: 30, resizeMode: 'contain' },
});

export default InsInfostep1;