import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Dimensions,
  Image,
  Platform,
  SafeAreaView,
  ActivityIndicator,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import debounce from 'lodash.debounce';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import CheckBox from '@react-native-community/checkbox';
import Toast from 'react-native-toast-message';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import { SafeAreaProvider, useSafeAreaInsets } from 'react-native-safe-area-context';
import NetInfo from '@react-native-community/netinfo';
import Modal from 'react-native-modal';
import RNFS from 'react-native-fs';

const { width, height } = Dimensions.get('window');

const Signature_Step2 = ({
  report,
  setActiveStep,
  navigation,
  subItemStatuses,
  setSubItemStatuses,
  isUpdate,
  onRefresh,
}) => {
  const [sessionState, setSessionState] = useState({
    isSatisfactory: null,
    role: null,
    mechanicName: '',
    operatorName: '',
  });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [mechanicSignatureUri, setMechanicSignatureUri] = useState(null);
  const [operatorSignatureUri, setOperatorSignatureUri] = useState(null);
  const [signatureRefreshKey, setSignatureRefreshKey] = useState(0);
  const [savePdf, setSavePdf] = useState(true);
  const [errors, setErrors] = useState({
    isSatisfactory: '',
    role: '',
    mechanicName: '',
    operatorName: '',
  });
  const [isOnline, setIsOnline] = useState(true);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [combinedPdfDownloadUrl, setCombinedPdfDownloadUrl] = useState(null);
  const [isLoading, setIsLoading] = useState(false);
  const isMounted = useRef(true);
  const insets = useSafeAreaInsets();

  useEffect(() => {
    isMounted.current = true;
    const loadStoredData = async () => {
      if (isMounted.current) setIsLoading(true);
      if (!report?.id) {
        console.log('No report ID provided for loading data');
        if (isMounted.current) Toast.show({ type: 'error', text1: 'Error', text2: 'Missing report information.' });
        if (isMounted.current) setIsLoading(false);
        return;
      }
      try {
        const userId = await AsyncStorage.getItem('userUid');
        if (!userId) {
          console.warn('User UID not found');
          if (isMounted.current) {
            Toast.show({ type: 'error', text1: 'Authentication Error', text2: 'Please log in again.' });
            navigation.replace('LoginScreens');
          }
          return;
        }

        // Load from both possible paths to ensure we get the data
        const correctiveRef = database().ref(`/GlobalCorrectiveActions/admins/${userId}/${report.id}`);
        const globalCorrectiveRef = database().ref(`/GlobalCorrectiveActions/${report.id}`);

        const [correctiveSnapshot, globalSnapshot] = await Promise.all([
          correctiveRef.once('value'),
          globalCorrectiveRef.once('value'),
        ]);

        const correctiveData = correctiveSnapshot.val() || globalSnapshot.val();

        if (correctiveData) {
          console.log('Loaded corrective action data:', correctiveData);
          if (isMounted.current) {
            setSessionState({
              isSatisfactory: correctiveData.isSatisfactory !== undefined ? correctiveData.isSatisfactory : null,
              role: correctiveData.role || null,
              mechanicName: correctiveData.mechanicName || '',
              operatorName: correctiveData.operatorName || '',
            });
            setMechanicSignatureUri(correctiveData.mechanicSignatureUri || null);
            setOperatorSignatureUri(correctiveData.operatorSignatureUri || null);
            setSubItemStatuses(correctiveData.subItemStatuses || {});
            setSignatureRefreshKey((prev) => prev + 1);
            setCombinedPdfDownloadUrl(correctiveData.pdfDownloadUrl || null);
          }
        } else {
          console.log('No existing corrective action data found');
        }
      } catch (error) {
        console.error('Error loading corrective action data:', error);
        if (isMounted.current) {
          setIsOnline(false);
          Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load stored data. Working offline.' });
        }
      } finally {
        if (isMounted.current) setIsLoading(false);
      }
    };
    if (isUpdate) loadStoredData();

    const unsubscribe = NetInfo.addEventListener((state) => {
      if (isMounted.current) setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    // Prevent navigation (including iOS swipe back) while an operation is in progress
    const removeBeforeRemove = navigation.addListener('beforeRemove', (e) => {
      if (isLoading || isSubmitting) {
        e.preventDefault();
        Toast.show({ type: 'info', text1: 'Please wait', text2: 'Operation in progress' });
      }
    });

    return () => {
      isMounted.current = false;
      unsubscribe();
      if (removeBeforeRemove && typeof removeBeforeRemove === 'function') removeBeforeRemove();
    };
  }, [report, isUpdate, setSubItemStatuses, navigation, isLoading, isSubmitting]);

  const isValidSignatureUri = useCallback((uri) => uri && typeof uri === 'string' && uri.startsWith('data:image'), []);

  const validateFields = useCallback(() => {
    const newErrors = {
      isSatisfactory: sessionState.isSatisfactory === null ? 'Please select if the corrective action is satisfactory' : '',
      role: sessionState.role === null ? 'Please select your role' : '',
      mechanicName: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') && !sessionState.mechanicName.trim() ? 'Mechanic name is required' : '',
      operatorName: (sessionState.role === 'Operator' || sessionState.role === 'Both') && !sessionState.operatorName.trim() ? 'Operator name is required' : '',
    };
    if ((sessionState.role === 'Mechanic' || sessionState.role === 'Both') && !isValidSignatureUri(mechanicSignatureUri)) {
      newErrors.mechanicName = 'Mechanic signature is required';
    }
    if ((sessionState.role === 'Operator' || sessionState.role === 'Both') && !isValidSignatureUri(operatorSignatureUri)) {
      newErrors.operatorName = 'Operator signature is required';
    }
    setErrors(newErrors);
    return Object.values(newErrors).every((error) => error === '');
  }, [sessionState, mechanicSignatureUri, operatorSignatureUri]);

  const debouncedValidate = useCallback(debounce(validateFields, 300), [validateFields]);

  useEffect(() => {
    debouncedValidate();
    return () => debouncedValidate.cancel();
  }, [sessionState, mechanicSignatureUri, operatorSignatureUri, debouncedValidate]);

  const getCorrectiveItemsForPDF = () => {
    const items = [];
    if (report?.step2?.sections && report?.step2?.selections) {
      const selections = report.step2.selections;
      const sections = report.step2.sections;
      sections.forEach((section) => {
        if (Array.isArray(section.subItems)) {
          section.subItems.forEach((subItem) => {
            const sel = selections[subItem.key];
            if (sel === 'Faulty' || sel === 'Not Ok' || sel === 'Reject') {
              items.push({
                section: section.name || 'N/A',
                item: subItem.itemName || 'N/A',
                category: subItem.category || 'N/A',
                status: subItemStatuses[subItem.key] || 'Pending',
              });
            }
          });
        }
      });
    }
    return items;
  };

  const generateCombinedPDFContent = () => {
    // Generate inspection items HTML exactly like Summerystep3
    let checklistRows = '';
    const sections = report.step2?.sections || [];
    const selections = report.step2?.selections || {};
    const notes = report.step2?.notes || {};

    // Include items with selections of Ok, Okay, Reject, Not Ok, or N/A (same as Summerystep3)
    const selectedItems = sections.flatMap((section) =>
      (section.subItems || []).filter((item) => 
        selections[item.key] && 
        ['Ok', 'Okay', 'Reject', 'Not Ok', 'N/A'].includes(selections[item.key])
      )
    );

    if (selectedItems.length === 0) {
      checklistRows = `
        <tr>
          <td colspan="6">No selected inspection items available</td>
        </tr>
      `;
    } else {
      selectedItems.forEach((item) => {
        const selection = selections[item.key] || 'N/A';
        const isOkay = selection === 'Okay' || selection === 'Ok';
        const isFaulty = selection === 'Faulty' || selection === 'Reject' || selection === 'Not Ok';
        const isNA = selection === 'N/A';
        const comment = (notes && notes[item.key]) ? notes[item.key] : '';
        checklistRows += `
          <tr>
            <td>${item.itemName || 'Unknown'}</td>
            <td style="text-align: center;">${item.isInspectionRequired ? 'A' : 'B'}</td>
            <td style="text-align: center;">${isOkay ? '✓' : ''}</td>
            <td style="text-align: center;">${isFaulty ? 'X' : ''}</td>
            <td style="text-align: center;">${isNA ? '—' : ''}</td>
            <td style="text-align: left;">${comment || ''}</td>
          </tr>
        `;
      });
    }

    // Generate corrective action items HTML with S.No as first column
    const correctiveItems = getCorrectiveItemsForPDF();
    const correctiveItemsHtml = correctiveItems.length > 0
      ? correctiveItems.map((row, index) => `
        <tr>
          <td style="text-align: center;">${index + 1}</td>
          <td>${row.item}</td>
          <td>${row.status}</td>
        </tr>
      `).join('')
      : '<tr><td colspan="3">No corrective action items available</td></tr>';

    // Get step3 summary data
    const step3Data = report.step3 || {};
    const overallCondition = step3Data.overallCondition || 'N/A';
    const equipmentStatus = step3Data.equipmentStatus || 'N/A';
    const priority = step3Data.priority || 'N/A';
    const safeToUse = step3Data.safeToUse || 'N/A';

    return `
      <!DOCTYPE html>
      <html lang="en">
      <head>
        <meta charset="UTF-8"/>
        <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
        <title>Inspection and Corrective Action Report</title>
        <style>
          body {
            font-family: Arial, sans-serif;
            margin: 20px;
            padding: 0;
            font-size: 12px;
          }
          .header-row {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 10px;
          }
          .logo-left {
            display: flex;
            align-items: flex-start;
          }
          .logo-left img {
            width: 140px;
            height: 50px;
            object-fit: contain;
          }
          .doc-info-box {
            width: 340px;
          }
          .doc-info-box table {
            width: 100%;
            border: 1px solid #222;
            border-collapse: collapse;
            font-size: 12px;
          }
          .doc-info-box td {
            border: 1px solid #222;
            padding: 4px 8px;
            text-align: left;
          }
          .title-section {
            text-align: center;
            margin: 10px 0 0 0;
          }
          .title-section h2 {
            margin: 0;
            font-size: 18px;
            font-weight: bold;
          }
          .request-number {
            font-size: 14px;
            font-weight: bold;
            margin-top: 4px;
          }
          .info-boxes-row {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin: 16px 0 10px 0;
          }
          .info-box {
            width: 45%;
          }
          .info-box table {
            width: 100%;
            border: 1px solid #222;
            border-collapse: collapse;
            font-size: 12px;
          }
          .info-box td {
            border: 1px solid #222;
            padding: 4px 8px;
            text-align: left;
          }
          .classification {
            margin-bottom: 20px;
          }
          .classification table {
            width: 100%;
          }
          .classification td {
            border: 1px solid black;
            padding: 5px;
            text-align: left;
          }
          table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
          }
          th, td {
            border: 1px solid black;
            padding: 8px;
            text-align: center;
          }
          th {
            background-color: #f2f2f2;
            font-weight: bold;
          }
          .checklist th:nth-child(1), .checklist td:nth-child(1) {
            width: 25%;
            text-align: left;
          }
          .checklist th:nth-child(5), .checklist td:nth-child(5) {
            width: 15.5%;
          }
          .checklist th:nth-child(2), .checklist td:nth-child(2),
          .checklist th:nth-child(3), .checklist td:nth-child(3),
          .checklist th:nth-child(4), .checklist td:nth-child(4) {
            width: 13%;
          }
          .checklist th:nth-child(5), .checklist td:nth-child(5),
          .checklist th:nth-child(6), .checklist td:nth-child(6) {
            width: 15.5%;
          }
          .remarks {
            width: 100%;
            height: 50px;
            border: 1px solid black;
            padding: 5px;
            text-align: left;
          }
          .summary-section-table {
            width: 100%;
            margin-top: 30px;
            border: none;
          }
          .summary-section-table td {
            border: none;
            padding: 8px;
            font-size: 14px;
            text-align: left;
          }
          .summary-section-table .label {
            font-weight: bold;
            width: 180px;
          }
          .summary-section-table .signature-cell {
            text-align: right;
            vertical-align: middle;
            padding-right: 30px;
          }
          .section-break {
            margin-top: 20px;
            border-top: 2px solid #015185;
            padding-top: 10px;
          }
          .corrective-table th:nth-child(1), .corrective-table td:nth-child(1) {
            width: 15%;
            text-align: center;
          }
          .corrective-table th:nth-child(2), .corrective-table td:nth-child(2) {
            width: 60%;
            text-align: left;
          }
          .corrective-table th:nth-child(3), .corrective-table td:nth-child(3) {
            width: 25%;
            text-align: center;
          }
        </style>
      </head>
      <body>
        <div class="header-row">
          <div class="logo-left">
            <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo"/>
          </div>
          <div class="doc-info-box">
            <table>
              <tr><td>Document Name</td><td>${report.step1?.checklistTitle || 'Inspection and Corrective Action Report'}</td></tr>
              <tr><td>Document ID</td><td>CA-${report.step1?.requestNumber || 'N/A'}</td></tr>
              <tr><td>Version Date</td><td>${new Date().toLocaleDateString()}</td></tr>
              <tr><td>Version</td><td>1.0</td></tr>
              <tr><td>Approved</td><td>${report.step1?.inspector || 'N/A'}</td></tr>
              <tr><td>Country</td><td>${report.step1?.country || 'N/A'}</td></tr>
              <tr><td>Project</td><td>${report.step1?.project || 'N/A'}</td></tr>
            </table>
          </div>
        </div>

        <div class="title-section">
          <h2>${report.step1?.checklistTitle || 'INSPECTION AND CORRECTIVE ACTION REPORT'}</h2>
          <div class="request-number">NO: ${report.step1?.requestNumber || 'N/A'}</div>
        </div>

        <div class="info-boxes-row">
          <div class="info-box">
            <table>
              <tr><td>DATE</td><td>${new Date().toLocaleDateString()}</td></tr>
              <tr><td>SITE / PROJECT</td><td>${report.step1?.project || 'N/A'}</td></tr>
              <tr><td>INSPECTOR</td><td>${report.step1?.inspector || 'N/A'}</td></tr>
            </table>
          </div>
          <div class="info-box">
            <table>
              <tr><td>COMPRESSOR No</td><td>${report.step1?.selectedEquipment?.equipmentName || 'N/A'}</td></tr>
              <tr><td>HOURS</td><td>${report.step1?.meterReading || 'N/A'} ${report.step1?.selectedEquipment?.meterUnit || 'Hours'}</td></tr>
            </table>
          </div>
        </div>

        <div class="classification">
          <table>
            <tr><td>1. ALL DEFECTS UNDER "A" CLASSIFICATION TO BE ATTENDED IMMEDIATELY</td></tr>
            <tr><td>2. ALL DEFECTS UNDER "B" CLASSIFICATION TO BE ATTENDED WITHIN 48 HOURS</td></tr>
            <tr><td>3. ALL DEFECTS UNDER "C" CLASSIFICATION TO BE ATTENDED TO AS SOON AS POSSIBLE</td></tr>
          </table>
        </div>

        <table class="checklist">
          <tr>
            <th>DESCRIPTION</th>
            <th>CLASS</th>
            <th>OKAY</th>
            <th>FAULTY</th>
            <th>N/A</th>
            <th>COMMENT</th>
          </tr>
          ${checklistRows}
          <tr>
            <td>REMARKS</td>
            <td colspan="5">${step3Data.additionalNote || 'N/A'}</td>
          </tr>
        </table>

        <table class="summary-section-table">
          <tr>
            <td class="label">Overall Condition</td>
            <td>${overallCondition}</td>
            <td class="signature-cell" rowspan="3" colspan="2">
              ${step3Data.signatureDownloadUrl
                ? `<img src="${step3Data.signatureDownloadUrl}" alt="Signature" style="width: 120px; height: 60px;"/>`
                : 'Signature: Not Provided'}
            </td>
          </tr>
          <tr>
            <td class="label">Vehicle Safe To Use</td>
            <td style="color: ${safeToUse === 'Yes' ? '#28a745' : '#e74c3c'};">
              ${safeToUse}
            </td>
          </tr>
          <tr>
            <td class="label">Maintenance Priority</td>
            <td>${priority}</td>
          </tr>
        </table>

        <div class="section-break"></div>
        <h2>CORRECTIVE ACTION ITEMS</h2>
        <div style="margin-bottom: 15px;">
          <p><strong>Equipment Condition After Correction:</strong> ${sessionState.isSatisfactory !== null ? (sessionState.isSatisfactory ? 'Satisfactory' : 'Unsatisfactory') : 'Not Specified'}</p>
          <p><strong>Role:</strong> ${sessionState.role || 'N/A'}</p>
          <p><strong>Mechanic Name:</strong> ${sessionState.mechanicName || 'N/A'}</p>
          <p><strong>Operator Name:</strong> ${sessionState.operatorName || 'N/A'}</p>
        </div>
        <table class="corrective-table">
          <tr>
            <th>S.No</th>
            <th>Item</th>
            <th>Status</th>
          </tr>
          ${correctiveItemsHtml}
        </table>
        <table class="summary-section-table">
          <tr>
            <td class="label">Mechanic Signature</td>
            <td class="signature-cell">
              ${(sessionState.role === 'Mechanic' || sessionState.role === 'Both') && isValidSignatureUri(mechanicSignatureUri)
                ? `<img src="${mechanicSignatureUri}" alt="Mechanic Signature" style="width: 120px; height: 60px;"/>`
                : 'Not Provided'}
            </td>
            <td class="label">Operator Signature</td>
            <td class="signature-cell">
              ${(sessionState.role === 'Operator' || sessionState.role === 'Both') && isValidSignatureUri(operatorSignatureUri)
                ? `<img src="${operatorSignatureUri}" alt="Operator Signature" style="width: 120px; height: 60px;"/>`
                : 'Not Provided'}
            </td>
          </tr>
        </table>
      </body>
      </html>
    `;
  };

  const generateAndSavePDF = async () => {
    if (!validateFields()) {
      Toast.show({ type: 'error', text1: 'Incomplete Data', text2: 'Please fill all required fields.' });
      return null;
    }
    setIsLoading(true);
    try {
      // Debug: Log report data to ensure we have all the information
      console.log('Generating PDF with report data:', {
        step1: report.step1,
        step2: report.step2,
        step3: report.step3,
        subItemStatuses,
        sessionState
      });
      
      const combinedHtml = generateCombinedPDFContent();
      const options = {
        html: combinedHtml,
        fileName: `CombinedReport_${report.step1?.requestNumber || 'Unknown'}_${Date.now()}`,
        directory: Platform.OS === 'android' ? 'Documents' : 'DocumentDirectory',
      };
      const { filePath } = await RNHTMLtoPDF.convert(options);
      console.log('Combined PDF generated at:', filePath);

      const userId = await AsyncStorage.getItem('userUid');
      const storageRef = storage().ref(`GlobalCorrectiveActions/${report.id}/pdfs/CombinedReport_${report.step1?.requestNumber || 'Unknown'}_${Date.now()}.pdf`);
      await storageRef.putFile(filePath);
      const downloadUrl = await storageRef.getDownloadURL();
      await RNFS.unlink(filePath).catch((err) => console.warn('Failed to clean up combined PDF:', err));

      return downloadUrl;
    } catch (error) {
      console.error('Error generating or uploading PDF:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to generate or upload PDF.' });
      return null;
    } finally {
      setIsLoading(false);
    }
  };

  const saveToFirebase = async (pdfUrl) => {
    try {
      const userId = await AsyncStorage.getItem('userUid');
      const correctiveRef = database().ref(`/GlobalCorrectiveActions/admins/${userId}/${report.id}`);
      const inspectionRef = database().ref(`/GlobalInspectionReport/admins/${userId}/${report.id}`);
      
      const correctiveData = {
        reportId: report.id,
        subItemStatuses,
        updatedAt: new Date().toISOString(),
      };
      if (sessionState.isSatisfactory !== null) correctiveData.isSatisfactory = sessionState.isSatisfactory;
      if (sessionState.role) correctiveData.role = sessionState.role;
      if (sessionState.role === 'Mechanic' || sessionState.role === 'Both') {
        correctiveData.mechanicName = sessionState.mechanicName || null;
        correctiveData.mechanicSignatureUri = mechanicSignatureUri || null;
      }
      if (sessionState.role === 'Operator' || sessionState.role === 'Both') {
        correctiveData.operatorName = sessionState.operatorName || null;
        correctiveData.operatorSignatureUri = operatorSignatureUri || null;
      }
      if (pdfUrl) correctiveData.pdfDownloadUrl = pdfUrl;
      if (!isUpdate) correctiveData.createdAt = new Date().toISOString();

      await correctiveRef.update(correctiveData);
      await inspectionRef.update({
        correctiveActionNumber: `CA-${report.id}`,
        correctiveAction: { exists: true, pdfDownloadUrl: pdfUrl || null },
      });
    } catch (error) {
      console.error('Error saving to Firebase:', error);
      throw error;
    }
  };

  const saveToAsyncStorage = async (pdfUrl) => {
    try {
      const storedData = await AsyncStorage.getItem('inspectionReports');
      let reports = storedData ? JSON.parse(storedData) : [];
      const correctiveActionData = {
        subItemStatuses,
        isSatisfactory: sessionState.isSatisfactory,
        role: sessionState.role,
        mechanicName: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') ? sessionState.mechanicName : null,
        operatorName: (sessionState.role === 'Operator' || sessionState.role === 'Both') ? sessionState.operatorName : null,
        mechanicSignatureUri: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') ? mechanicSignatureUri : null,
        operatorSignatureUri: (sessionState.role === 'Operator' || sessionState.role === 'Both') ? operatorSignatureUri : null,
        pdfDownloadUrl: pdfUrl,
        timestamp: new Date().toISOString(),
      };
      const updatedReports = reports.map((r) =>
        r.id === report.id
          ? {
              ...r,
              correctiveAction: {
                ...r.correctiveAction,
                ...correctiveActionData,
              },
              correctiveActionNumber: `CA-${report.id}`,
            }
          : r
      );
      if (!reports.some((r) => r.id === report.id)) {
        updatedReports.push({
          ...report,
          correctiveAction: correctiveActionData,
          correctiveActionNumber: `CA-${report.id}`,
        });
      }
      await AsyncStorage.setItem('inspectionReports', JSON.stringify(updatedReports));
    } catch (error) {
      console.error('Error saving to AsyncStorage:', error);
      throw error;
    }
  };

  const handleDone = async () => {
    if (isSubmitting || !isMounted.current) return;
    if (!validateFields()) return;
    setIsSubmitting(true);
    setIsLoading(true);
    try {
      let pdfUrl = null;
      if (savePdf && isOnline) {
        pdfUrl = await generateAndSavePDF();
        setCombinedPdfDownloadUrl(pdfUrl);
      }
      if (isOnline) {
        await saveToFirebase(pdfUrl);
        await saveToAsyncStorage(pdfUrl);
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: pdfUrl ? 'Report saved and PDF uploaded' : 'Report saved successfully.',
        });
        if (isMounted.current && savePdf && pdfUrl) {
          setShowSuccessModal(true);
        }
      } else {
        await saveToAsyncStorage(pdfUrl);
        const pendingOperations = JSON.parse(await AsyncStorage.getItem('pendingOperations') || '[]');
        const correctiveData = {
          subItemStatuses,
          isSatisfactory: sessionState.isSatisfactory,
          role: sessionState.role,
          mechanicName: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') ? sessionState.mechanicName : null,
          operatorName: (sessionState.role === 'Operator' || sessionState.role === 'Both') ? sessionState.operatorName : null,
          mechanicSignatureUri: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') ? mechanicSignatureUri : null,
          operatorSignatureUri: (sessionState.role === 'Operator' || sessionState.role === 'Both') ? operatorSignatureUri : null,
          pdfDownloadUrl: pdfUrl,
          createdAt: isUpdate ? report.createdAt : new Date().toISOString(),
          updatedAt: new Date().toISOString(),
        };
        pendingOperations.push({
          type: 'create_corrective',
          userId: await AsyncStorage.getItem('userUid'),
          reportId: report.id,
          data: correctiveData,
          path: `/GlobalCorrectiveActions/${report.id}`,
        });
        await AsyncStorage.setItem('pendingOperations', JSON.stringify(pendingOperations));
        Toast.show({
          type: 'info',
          text1: 'Offline Mode',
          text2: 'Corrective action saved as draft and queued for sync when online.',
        });
        if (isMounted.current) {
          navigation.navigate('Inspection_Report_Home');
        }
      }
    } catch (error) {
      console.error('Error submitting corrective action:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to submit corrective action. Please try again.' });
    } finally {
      if (isMounted.current) {
        setIsSubmitting(false);
        setIsLoading(false);
      }
    }
  };

  const handleMechanicSignature = () => {
    navigation.navigate('SignatureCaptureScreen', {
      role: 'Mechanic',
      onSave: (signature) => {
        setMechanicSignatureUri(signature);
        setSignatureRefreshKey((prev) => prev + 1);
      },
    });
  };

  const handleOperatorSignature = () => {
    navigation.navigate('SignatureCaptureScreen', {
      role: 'Operator',
      onSave: (signature) => {
        setOperatorSignatureUri(signature);
        setSignatureRefreshKey((prev) => prev + 1);
      },
    });
  };

  const handleClearSignature = (type) => {
    if (type === 'Mechanic') {
      setMechanicSignatureUri(null);
      setSessionState((prev) => ({ ...prev, mechanicName: '' }));
      setErrors((prev) => ({ ...prev, mechanicName: '' }));
    } else if (type === 'Operator') {
      setOperatorSignatureUri(null);
      setSessionState((prev) => ({ ...prev, operatorName: '' }));
      setErrors((prev) => ({ ...prev, operatorName: '' }));
    }
    setSignatureRefreshKey((prev) => prev + 1);
  };

  const handleImageError = (type) => {
    console.error(`${type} signature image failed to load`);
    Toast.show({ type: 'error', text1: 'Error', text2: `Failed to load ${type.toLowerCase()} signature. Please try again.` });
    handleClearSignature(type);
  };

  const handleRolePress = useCallback((value) => {
    setSessionState((prev) => {
      const newRole = prev.role === value ? null : value;
      return {
        ...prev,
        role: newRole,
        mechanicName: newRole !== 'Mechanic' && newRole !== 'Both' ? '' : prev.mechanicName,
        operatorName: newRole !== 'Operator' && newRole !== 'Both' ? '' : prev.operatorName,
      };
    });
    setErrors((prev) => ({ ...prev, role: '' }));
    if (value !== 'Mechanic' && value !== 'Both') setMechanicSignatureUri(null);
    if (value !== 'Operator' && value !== 'Both') setOperatorSignatureUri(null);
    setSignatureRefreshKey((prev) => prev + 1);
  }, []);

  const handleSatisfactoryPress = useCallback((value) => {
    setSessionState((prev) => ({ ...prev, isSatisfactory: prev.isSatisfactory === value ? null : value }));
    setErrors((prev) => ({ ...prev, isSatisfactory: '' }));
  }, []);

  const showMechanicSection = sessionState.role === 'Mechanic' || sessionState.role === 'Both';
  const showOperatorSection = sessionState.role === 'Operator' || sessionState.role === 'Both';

  const handleViewPdf = () => {
    if (!combinedPdfDownloadUrl || typeof combinedPdfDownloadUrl !== 'string' || !combinedPdfDownloadUrl.startsWith('http')) {
      Toast.show({ type: 'error', text1: 'Error', text2: 'Combined PDF not available for viewing.' });
      return;
    }
    navigation.navigate('PdfviewPage', {
      pdfUrl: combinedPdfDownloadUrl,
      onGoBack: () => navigation.navigate('Inspection_Report_Home'),
    });
    setShowSuccessModal(false);
  };

  const ProcessorModal = ({ visible }) => {
    const rotateAnim = useRef(new Animated.Value(0)).current;

    useEffect(() => {
      if (visible) {
        Animated.loop(
          Animated.timing(rotateAnim, {
            toValue: 1,
            duration: 900,
            useNativeDriver: true,
          })
        ).start();
      } else {
        rotateAnim.setValue(0);
      }
    }, [visible]);

    const rotate = rotateAnim.interpolate({
      inputRange: [0, 1],
      outputRange: ['0deg', '360deg'],
    });

    return (
      <Modal isVisible={visible} style={{ margin: 0 }} backdropOpacity={0.4}>
        <View style={styles.processorModalOuterCenter}>
          <View style={styles.processorModalContainer}>
            <Text style={styles.processorModalTitle}>Please Wait</Text>
            <View style={styles.processorRowCenter}>
              <Animated.Image
                source={require('../../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate }] }]}
              />
              <Text style={styles.processorModalText}>Loading</Text>
            </View>
          </View>
        </View>
      </Modal>
    );
  };

  if (!report || !report.step1 || !report.step2) {
    return (
      <SafeAreaProvider>
        <SafeAreaView style={styles.container} edges={['top', 'left', 'right']}>
          <Text style={styles.noDataText}>No report data available</Text>
        </SafeAreaView>
      </SafeAreaProvider>
    );
  }

  return (
    <SafeAreaProvider>
      <SafeAreaView style={styles.container} edges={['top', 'left', 'right']}>
        <ScrollView
          contentContainerStyle={[styles.contentContainer, { paddingBottom: insets.bottom + 60 }]}
          showsVerticalScrollIndicator={false}
          keyboardShouldPersistTaps="handled"
        >
          <View style={styles.sectionNoBG}>
            <Text style={styles.sectionTitleLargeBlue}>Corrective of vehicle is satisfactory</Text>
            <View style={styles.buttonRowFullWidth}>
              <TouchableOpacity
                style={[
                  styles.fullWidthButton,
                  sessionState.isSatisfactory === true && styles.fullWidthButtonSelected,
                ]}
                onPress={() => handleSatisfactoryPress(true)}
              >
                <Text style={[
                  styles.fullWidthButtonText,
                  sessionState.isSatisfactory === true && styles.fullWidthButtonTextSelected,
                ]}>Yes</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[
                  styles.fullWidthButton,
                  sessionState.isSatisfactory === false && styles.fullWidthButtonSelected,
                ]}
                onPress={() => handleSatisfactoryPress(false)}
              >
                <Text style={[
                  styles.fullWidthButtonText,
                  sessionState.isSatisfactory === false && styles.fullWidthButtonTextSelected,
                ]}>No</Text>
              </TouchableOpacity>
            </View>
            {errors.isSatisfactory ? <Text style={styles.errorText}>{errors.isSatisfactory}</Text> : null}
          </View>

          <View style={styles.sectionNoBG}>
            <Text style={styles.sectionTitleLargeBlue}>Are you?</Text>
            <View style={styles.buttonRowFullWidth3}>
              {['Mechanic', 'Operator', 'Both'].map((role) => (
                <TouchableOpacity
                  key={role}
                  style={[
                    styles.fullWidthButton3,
                    sessionState.role === role && styles.fullWidthButtonSelected,
                  ]}
                  onPress={() => handleRolePress(role)}
                >
                  <Text style={[
                    styles.fullWidthButtonText,
                    sessionState.role === role && styles.fullWidthButtonTextSelected,
                  ]}>{role}</Text>
                </TouchableOpacity>
              ))}
            </View>
            {errors.role ? <Text style={styles.errorText}>{errors.role}</Text> : null}
          </View>

          {showMechanicSection && (
            <View style={styles.sectionNoBG}>
              <Text style={styles.signatureLabelCenteredBlue}>Mechanic Signature</Text>
              <TextInput
                style={styles.signatureInput}
                placeholder="Mechanic Name"
                placeholderTextColor="#888"
                value={sessionState.mechanicName}
                onChangeText={(text) => setSessionState((prev) => ({ ...prev, mechanicName: text }))}
              />
              {errors.mechanicName ? <Text style={styles.errorText}>{errors.mechanicName}</Text> : null}
              <View style={{ height: 12 }} />
              <View style={styles.signatureBoxImage}>
                {isValidSignatureUri(mechanicSignatureUri) ? (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleMechanicSignature}>
                    <Image
                      source={{ uri: mechanicSignatureUri }}
                      style={styles.signatureImage}
                      resizeMode="contain"
                      onError={() => handleImageError('Mechanic')}
                    />
                  </TouchableOpacity>
                ) : (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleMechanicSignature}>
                    <View style={styles.plusCircle}>
                      <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                    </View>
                  </TouchableOpacity>
                )}
              </View>
            </View>
          )}

          {showMechanicSection && (
            <View style={styles.acknowledgeRowCenteredColumn}>
              <View style={styles.acknowledgeRowCenteredRow}>
                <MaterialIcons name="check-box" size={22} color="#28a745" style={{ alignSelf: 'center' }} />
                <Text style={styles.acknowledgeTitleCentered}>MECHANIC'S ACKNOWLEDGEMENT</Text>
              </View>
              <Text style={styles.acknowledgeDescCentered}>By signing, I acknowledge that I have checked all corrective action taken on vehicle.</Text>
            </View>
          )}

          {showOperatorSection && (
            <View style={styles.sectionNoBG}>
              <Text style={styles.signatureLabelCenteredBlue}>Operator Signature</Text>
              <TextInput
                style={styles.signatureInput}
                placeholder="Operator Name"
                placeholderTextColor="#888"
                value={sessionState.operatorName}
                onChangeText={(text) => setSessionState((prev) => ({ ...prev, operatorName: text }))}
              />
              {errors.operatorName ? <Text style={styles.errorText}>{errors.operatorName}</Text> : null}
              <View style={{ height: 12 }} />
              <View style={styles.signatureBoxImage}>
                {isValidSignatureUri(operatorSignatureUri) ? (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleOperatorSignature}>
                    <Image
                      source={{ uri: operatorSignatureUri }}
                      style={styles.signatureImage}
                      resizeMode="contain"
                      onError={() => handleImageError('Operator')}
                    />
                  </TouchableOpacity>
                ) : (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleOperatorSignature}>
                    <View style={styles.plusCircle}>
                      <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                    </View>
                  </TouchableOpacity>
                )}
              </View>
            </View>
          )}

          <View style={styles.section}>
            <View style={styles.checkboxRow}>
              <CheckBox value={savePdf} onValueChange={setSavePdf} tintColors={{ true: '#015185', false: '#888' }} />
              <Text style={styles.checkboxLabel}>Save as PDF</Text>
            </View>
          </View>

          <View style={styles.buttonContainer}>
            <TouchableOpacity style={styles.cancelButton} onPress={() => setActiveStep(1)} disabled={isSubmitting}>
              <Text style={styles.buttonText}>Back</Text>
            </TouchableOpacity>
            <TouchableOpacity style={[styles.submitButton, isSubmitting && styles.disabledButton]} onPress={handleDone} disabled={isSubmitting}>
              <Text style={[styles.buttonText, { color: '#fff' }]}>Done</Text>
            </TouchableOpacity>
          </View>
        </ScrollView>

        <Toast />

        <Modal isVisible={showSuccessModal} style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Success</Text>
            <MaterialIcons name="check" size={width * 0.1} color="#015185" style={{ alignSelf: 'center', marginVertical: width * 0.02 }} />
            <Text style={styles.modalSubtitle}>Report Generated and Saved</Text>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity style={styles.modalButton}    onPress={() => {
                  setShowSuccessModal(false);
                  navigation.reset({
                    index: 1,
                    routes: [
                      { name: 'MainApp' },
                      { name: 'Inspection_Report_Home' },
                    ],
                  });
                }}>
                <Text style={styles.modalButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.modalButton} onPress={handleViewPdf}>
                <Text style={styles.modalButtonText}>View Report</Text>
              </TouchableOpacity>
            </View>
          </View>
        </Modal>

        <ProcessorModal visible={isLoading} />
      </SafeAreaView>
    </SafeAreaProvider>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  contentContainer: {
    paddingHorizontal: width * 0.04,
    paddingTop: width * 0.025,
    paddingBottom: width * 0.1,
  },
  section: {
    marginVertical: width * 0.025,
    padding: width * 0.025,
    backgroundColor: '#f0f0f0',
    borderRadius: width * 0.012,
  },
  sectionNoBG: {
    marginVertical: width * 0.025,
    padding: width * 0.025,
  },
  sectionTitleLargeBlue: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#1676b7',
    marginBottom: width * 0.01,
  },
  buttonRowFullWidth: {
    flexDirection: 'row',
    width: '100%',
    marginBottom: width * 0.01,
    gap: 8,
  },
  fullWidthButton: {
    flex: 1,
    backgroundColor: '#f2f6fa',
    borderRadius: 2,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: width * 0.025,
  },
  fullWidthButtonSelected: {
    backgroundColor: '#1676b7',
  },
  fullWidthButtonText: {
    color: '#222',
    fontWeight: 'bold',
    fontSize: width * 0.04,
  },
  fullWidthButtonTextSelected: {
    color: '#fff',
  },
  buttonRowFullWidth3: {
    flexDirection: 'row',
    width: '100%',
    marginBottom: width * 0.01,
    gap: 8,
  },
  fullWidthButton3: {
    flex: 1,
    backgroundColor: '#f2f6fa',
    borderRadius: 2,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: width * 0.025,
  },
  signatureLabelCenteredBlue: {
    color: '#1676b7',
    fontWeight: 'bold',
    fontSize: width * 0.045,
    textAlign: 'center',
    marginBottom: 4,
    marginTop: width * 0.04,
  },
  signatureInput: {
    backgroundColor: '#f0f0f0',
    borderRadius: 2,
    paddingHorizontal: width * 0.025,
    paddingVertical: width * 0.025,
    fontSize: width * 0.038,
    color: '#333',
    marginBottom: 4,
    borderWidth: 0,
    width: '100%',
    alignSelf: 'center',
  },
  signatureBoxImage: {
    height: height / 5,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: width * 0.012,
    justifyContent: 'center',
    alignItems: 'center',
    marginVertical: width * 0.025,
  },
  plusIconCircleContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    width: '100%',
    height: '100%',
  },
  plusCircle: {
    width: '100%',
    height: '100%',
    justifyContent: 'center',
    alignItems: 'center',
  },
  signatureImage: {
    width: '100%',
    height: '100%',
    borderRadius: 2,
    backgroundColor: '#f5f5f5',
  },
  acknowledgeRowCenteredColumn: {
    alignItems: 'center',
  },
  acknowledgeRowCenteredRow: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  acknowledgeTitleCentered: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#28a745',
    marginLeft: width * 0.02,
  },
  acknowledgeDescCentered: {
    fontSize: width * 0.035,
    color: '#333',
    textAlign: 'center',
    marginTop: width * 0.015,
  },
  checkboxRow: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
  },
  checkboxLabel: {
    fontSize: width * 0.04,
    color: '#666',
    marginLeft: width * 0.02,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.02,
  },
  cancelButton: {
    paddingVertical: width * 0.012,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.008,
    borderWidth: 1,
    borderColor: '#015185',
    alignItems: 'center',
    width: '40%',
  },
  submitButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.012,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.008,
    alignItems: 'center',
    width: '40%',
  },
  disabledButton: {
    backgroundColor: '#ccc',
  },
  buttonText: {
    fontSize: width * 0.032,
    fontWeight: '500',
    color: '#333',
  },
  modalOverlay: {
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    padding: width * 0.025,
    width: '80%',
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.015,
  },
  modalSubtitle: {
    fontSize: width * 0.035,
    color: '#333',
    textAlign: 'center',
    marginBottom: width * 0.025,
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
    flexWrap: 'nowrap',
    gap: 10,
  },
  modalButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.012,
    alignItems: 'center',
    flex: 1,
    marginHorizontal: 2,
    marginBottom: width * 0.015,
    minWidth: 0,
  },
  modalButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: '500',
  },
  processorModalContainer: {
    backgroundColor: '#fff',
    borderRadius: 14,
    padding: 28,
    alignItems: 'center',
    justifyContent: 'center',
    alignSelf: 'center',
    width: width * 0.7,
  },
  processorModalTitle: {
    fontSize: 20,
    fontWeight: '500',
    color: '#222',
    marginBottom: 10,
    textAlign: 'center',
  },
  processorModalText: {
    fontSize: 16,
    color: '#015185',
    marginTop: 6,
    textAlign: 'center',
  },
  processorRowCenter: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 12,
  },
  processorModalOuterCenter: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  errorText: {
    color: '#ff4444',
    fontSize: width * 0.035,
    marginLeft: width * 0.015,
    marginBottom: width * 0.015,
  },
  noDataText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
});

export default Signature_Step2;