import React, { useState, useImperativeHandle, forwardRef, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TextInput,
  ScrollView,
  TouchableOpacity,
  Dimensions,
  Alert,
  Image,
  KeyboardAvoidingView,
  Platform,
  Modal,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
import AsyncStorage from '@react-native-async-storage/async-storage';
import RNFS from 'react-native-fs';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import { check, request, PERMISSIONS, RESULTS } from 'react-native-permissions';
import Toast from 'react-native-toast-message';
import { useNavigation } from '@react-navigation/native';

const { width } = Dimensions.get('window');
const IMAGE_BOX_GAP = 8;
const IMAGE_BOX_HORIZONTAL_PADDING = 50;
const IMAGE_BOX_SIZE = (width - IMAGE_BOX_HORIZONTAL_PADDING - 2 * IMAGE_BOX_GAP - 12) / 3;

// SyncModal Component for Animated Loading
const SyncModal = ({ visible }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>Generating PDF...</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// SuccessModal Component
const SuccessModal = ({ visible, onClose, onNavigateBack, onViewPDF, pdfUrl }) => {
  const navigation = useNavigation();
  return (
    <Modal visible={visible} animationType="fade" transparent={true} onRequestClose={onClose}>
      <View style={styles.modalOverlay}>
        <View style={styles.modalContent}>
          <Text style={styles.modalTitle}>Success</Text>
          <Text style={styles.modalText}>Incident Report Generated and Saved</Text>
          <View style={styles.modalActions}>
            <TouchableOpacity
              style={[styles.modalButton, styles.viewButton]}
              onPress={() => {
                onClose && onClose();


                  navigation.reset({
                    index: 1,
                    routes: [
                      { name: 'MainApp' },
                      { name: 'IncodentReport' },
                    ],
                  });
                }}


               
            >
              <Text style={styles.modalButtonText}>Back</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={[styles.modalButton, styles.viewButton]}
              onPress={() => {
                onClose && onClose();
                onViewPDF && onViewPDF(pdfUrl);
              }}
            >
              <Text style={styles.modalButtonText}>View PDF</Text>
            </TouchableOpacity>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const IncidentReportStep2 = forwardRef(({ currentStep, formData, selectedIncidents, involvesEquipment }, ref) => {
  const navigation = useNavigation();
  const [incidentImages, setIncidentImages] = useState(Array(9).fill(null));
  const [incidentCause, setIncidentCause] = useState('');
  const [equipmentDamage, setEquipmentDamage] = useState('');
  const [additionalComment, setAdditionalComment] = useState('');
  const [immediateCorrectiveAction, setImmediateCorrectiveAction] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [successModalVisible, setSuccessModalVisible] = useState(false);
  const [pdfUrl, setPdfUrl] = useState(null);
  const [reportId, setReportId] = useState(null);

  useEffect(() => {
    loadPersistedState();
  }, []);

  const loadPersistedState = async () => {
    try {
      const state = await AsyncStorage.getItem('IncidentReportStep2State');
      if (state) {
        const parsed = JSON.parse(state);
        setIncidentImages(parsed.incidentImages || Array(9).fill(null));
        setIncidentCause(parsed.incidentCause || '');
        setEquipmentDamage(parsed.equipmentDamage || '');
        setAdditionalComment(parsed.additionalComment || '');
        setImmediateCorrectiveAction(parsed.immediateCorrectiveAction || '');
      }
    } catch (error) {
      console.error('Error loading persisted state:', error);
    }
  };

  const saveState = async () => {
    try {
      const state = {
        incidentImages,
        incidentCause,
        equipmentDamage,
        additionalComment,
        immediateCorrectiveAction,
      };
      await AsyncStorage.setItem('IncidentReportStep2State', JSON.stringify(state));
    } catch (error) {
      console.error('Error saving state:', error);
    }
  };

  const validateStep2 = () => {
    saveState();
    return true; // No required fields in Step 2
  };

  const requestStoragePermission = async () => {
    try {
      const permissions = Platform.OS === 'android'
        ? Platform.Version >= 33
          ? [PERMISSIONS.ANDROID.READ_MEDIA_IMAGES]
          : [PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE, PERMISSIONS.ANDROID.WRITE_EXTERNAL_STORAGE]
        : [];

      if (permissions.length === 0) return true;

      let allGranted = true;
      for (const permission of permissions) {
        const result = await check(permission);
        if (result === RESULTS.DENIED) {
          const requestResult = await request(permission);
          if (requestResult !== RESULTS.GRANTED) {
            allGranted = false;
            Alert.alert('Permission Denied', 'Storage permission is required to save the report.');
          }
        } else if (result === RESULTS.BLOCKED) {
          allGranted = false;
          Alert.alert('Permission Blocked', 'Please enable storage permission in Settings.');
        }
      }
      return allGranted;
    } catch (error) {
      console.error('Error requesting storage permission:', error);
      Alert.alert('Error', 'Failed to request storage permission.');
      return false;
    }
  };

  const uploadImageToFirebase = async (uri, reportId, index) => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) throw new Error('User not authenticated');

      const storagePath = `users/${userUid}/incidentImages/${reportId}_${index}.jpg`;
      const reference = storage().ref(storagePath);
      await reference.putFile(uri);
      const downloadUrl = await reference.getDownloadURL();
      return downloadUrl;
    } catch (error) {
      console.error('Error uploading image:', error);
      throw error;
    }
  };

  const generatePDFWithHtml = async (report) => {
    try {
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const selectedIncidents = report.selectedIncidents || {};

      const formatDate = (date) => {
        if (!date) return 'N/A';
        return new Date(date).toLocaleDateString('en-GB');
      };

      const formatTime = (date) => {
        if (!date) return 'N/A';
        return new Date(date).toLocaleTimeString('en-GB', { hour: '2-digit', minute: '2-digit' });
      };

      // Types of Incident checkboxes (4 columns per row)
      const incidentTypes = [
        'Brake Failure', 'Chemicals Spill', 'Collision', 'Drowned in Water',
        'Electrical Failure', 'Engine Failure', 'Equipment Failure', 'Explosion/fire',
        'Fall Attachments', 'Fall from elevation', 'Leakage in Vehicle', 'Loss of Stability',
        'Oil Spill', 'Person hit by Vehicle', 'Steering Failure', 'Stuck',
        'Swamping', 'Tire Burst', 'Turned Over', 'Other'
      ];
      let incidentTypesHtml = '';
      for (let i = 0; i < incidentTypes.length; i += 4) {
        incidentTypesHtml += '<tr>';
        for (let j = 0; j < 4; j++) {
          const idx = i + j;
          if (idx < incidentTypes.length) {
            const type = incidentTypes[idx];
            incidentTypesHtml += `<td style="padding:2px 6px;"><input type="checkbox" ${selectedIncidents[type] ? 'checked' : ''}/> ${type}</td>`;
          } else {
            incidentTypesHtml += '<td></td>';
          }
        }
        incidentTypesHtml += '</tr>';
      }

      // Severity checkboxes
      const severityOptions = ['High', 'Medium', 'Low'];
      let severityHtml = '<tr>';
      severityOptions.forEach((option) => {
        severityHtml += `<td style="padding:2px 6px;"><input type="checkbox" ${step1.severity === option ? 'checked' : ''}/> ${option}</td>`;
      });
      severityHtml += '</tr>';

      // Images
      let imagesHtml = '';
      const imageUrls = step2.incidentImages || [];
      if (imageUrls.length > 0 && imageUrls.some(Boolean)) {
        imagesHtml = '<div style="display:flex;flex-wrap:wrap;gap:8px;">';
        imageUrls.forEach((url, index) => {
          if (url) {
            imagesHtml += `<div style="width:120px;height:90px;border:1px solid #ccc;display:flex;align-items:center;justify-content:center;margin:4px;"><img src="${url}" style="max-width:100%;max-height:100%;object-fit:contain;"/></div>`;
          }
        });
        imagesHtml += '</div>';
      } else {
        imagesHtml = '<div style="color:#888;">No Images</div>';
      }

      const userName = await AsyncStorage.getItem('userName') || 'N/A';

      // Helper to show dash instead of N/A or empty
      const dashIfNA = (val) => {
        if (val === undefined || val === null || val === '' || val === 'N/A') return '-';
        return val;
      };

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Incident Report</title>
          <style>
            body { font-family: Arial, sans-serif; margin: 0; padding: 0; }
            .container { max-width: 900px; margin: 0 auto; padding: 12px; }
            .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px; }
            .header h1 { font-size: 22px; font-weight: bold; margin: 0; }
            .logo { font-size: 22px; font-weight: bold; color: #fff;  border-radius: 50%; width: 150px; height: 48px; display: flex; align-items: center; justify-content: center; }
            .section-title { font-weight: bold; font-size: 14px; margin: 10px 0 4px; background-color: #e6e6e6; padding: 4px; border: 1px solid #ccc; }
            table { width: 100%; border-collapse: collapse; margin-bottom: 8px; }
            th, td { border: 1px solid #ccc; padding: 6px; text-align: left; font-size: 13px; }
            th { background-color: #f2f2f2; font-weight: bold; }
            .checkbox-table td { border: none; padding: 2px 6px; font-size: 13px; }
            .checkbox-table input[type="checkbox"] { margin-right: 4px; }
            .text-section { border: 1px solid #ccc; padding: 8px; min-height: 32px; font-size: 13px; margin-bottom: 8px; }
            .footer { text-align: center; margin-top: 10px; font-size: 11px; color: #000; border-top: 1px solid #000; padding-top: 6px; }
            .reported-by { text-align: right; font-size: 13px; margin-top: 8px; }
            .reported-by span { border-bottom: 1px solid #000; padding-bottom: 2px; }
            .logo img { width: 100%; height: 100%; object-fit: contain; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>Incident Report</h1>
              <div class="logo">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo"/></div>
            </div>
            <table>
              <tr>
                <th colspan="2">Vehicle Details</th>
                <th colspan="2">Incident Details</th>
              </tr>
              <tr>
                <td>Vehicle Category</td>
                <td>${dashIfNA(step1.vehicleName)}</td>
                <td>Incident Number</td>
                <td>${dashIfNA(step1.incidentNumber)}</td>
              </tr>
              <tr>
                <td>Vehicle Name</td>
                <td>${dashIfNA(step1.vehicleName)}</td>
                <td>Incident Date</td>
                <td>${dashIfNA(formatDate(step1.incidentDate))}</td>
              </tr>
              <tr>
                <td>Vehicle #</td>
                <td>${dashIfNA(step1.vehicleId)}</td>
                <td>Incident Time</td>
                <td>${dashIfNA(formatTime(step1.incidentDate))}</td>
              </tr>
              <tr>
                <td>VIN #</td>
                <td>-</td>
                <td>Location</td>
                <td>${dashIfNA(step1.project)}</td>
              </tr>
              <tr>
                <td>Operator</td>
                <td>${dashIfNA(step1.incidentDate2)}</td>
                <td>Incident Area</td>
                <td>${dashIfNA(step1.incidentArea)}</td>
              </tr>
              <tr>
                <td>GPS Coordinates</td>
                <td colspan="3">${dashIfNA(step1.coordinates)}</td>
              </tr>
              <tr>
                <td>GPS Address</td>
                <td colspan="3">${dashIfNA(step1.gpsAddress)}</td>
              </tr>
            </table>
            <div class="section-title">Types of Incident</div>
            <table class="checkbox-table">${incidentTypesHtml}</table>
            <div class="section-title">Incident Severity Rating</div>
            <table class="checkbox-table">${severityHtml}</table>
            <div class="section-title">How incident occurred and what were the causes?</div>
            <div class="text-section">${step2.incidentCause || '-'}</div>
            <div class="section-title">What broke on the equipment, list all damages?</div>
            <div class="text-section">${step2.equipmentDamage || '-'}</div>
            <div class="section-title">Additional Comment</div>
            <div class="text-section">${step2.additionalComment || '-'}</div>
            <div class="section-title">Incident Images</div>
            ${imagesHtml}
            <div class="reported-by">
              <span>${userName}</span><br>
              Reported By
            </div>
            <div class="footer">
              Prepared By: Titan Drilling            </div>
          </div>
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: `Incident_Report_${report.id}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    }
  };

  const saveReportToFirebase = async (userUid, report, pdfDownloadUrl) => {
    try {
      // Save to GlobalIncidentReports to match inspection report pattern
      const reportRef = database().ref(`/GlobalIncidentReports/admins/${userUid}/${report.id}`);
      const reportData = {
        ...report,
        step2: {
          ...report.step2,
          pdfDownloadUrl,
        },
        // Ensure adminUid is set for permission filtering
        adminUid: userUid,
      };
      await reportRef.set(reportData);
      
      // Email notifications are automatically sent by Firebase function when report is created
      console.log('Incident report saved successfully. Email notifications will be sent automatically by Firebase function.');
      
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Incident report saved successfully',
      });
    } catch (error) {
      console.error('Error saving report to Firebase:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save incident report',
      });
      throw error;
    }
  };



  const generateReport = async () => {
    if (!validateStep2()) return;

    const hasPermission = await requestStoragePermission();
    if (!hasPermission) return;

    setSyncModalVisible(true);
    try {
      // Get user authentication
      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        setSyncModalVisible(false);
        return;
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      const userData = userSnapshot.val();
      if (!userData) {
        Alert.alert('Error', 'User not found in admin database.');
        setSyncModalVisible(false);
        return;
      }
      const userId = Object.keys(userData)[0];

      // Load step 1 data from AsyncStorage
      const step1DataStr = await AsyncStorage.getItem('IncidentReportStep1Data');
      if (!step1DataStr) {
        Alert.alert('Error', 'Incident details not found. Please restart the report.');
        setSyncModalVisible(false);
        return;
      }
      const step1Data = JSON.parse(step1DataStr);
      const incidentNumber = step1Data.incidentNumber;
      if (!incidentNumber) {
        Alert.alert('Error', 'Incident number not available.');
        setSyncModalVisible(false);
        return;
      }
      setReportId(incidentNumber);

      // Upload images to Firebase Storage
      const imageUrls = await Promise.all(
        incidentImages.map(async (uri, index) => {
          if (uri) {
            // Save images under GlobalIncidentReports/admins/{userId}/incidentReports/{incidentNumber}/images/
            const storagePath = `GlobalIncidentReports/admins/${userId}/incidentReports/${incidentNumber}/images/image_${index}.jpg`;
            const reference = storage().ref(storagePath);
            await reference.putFile(uri);
            return await reference.getDownloadURL();
          }
          return null;
        })
      );

      // Create flattened report structure for home page compatibility
      const report = {
        id: incidentNumber,
        // Flatten step1 data to top level for home page compatibility
        incidentNumber: step1Data.incidentNumber,
        incidentDate: step1Data.incidentDate,
        incidentArea: step1Data.incidentArea,
        severity: step1Data.severity,
        incidentCategory: step1Data.incidentCategory,
        country: step1Data.country,
        project: step1Data.project,
        vehicleId: step1Data.vehicleId,
        vehicleName: step1Data.vehicleName,
        personInvolved: step1Data.personInvolved,
        coordinates: step1Data.coordinates,
        gpsAddress: step1Data.gpsAddress,
        selectedIncidents: step1Data.selectedIncidents,
        involvesEquipment: step1Data.involvesEquipment,
        otherIncidentNote: step1Data.otherIncidentNote,
        createdAt: step1Data.createdAt || new Date().toISOString(),
        // Keep nested structure for PDF generation
        step1: step1Data,
        step2: {
          incidentImages: imageUrls,
          incidentCause,
          equipmentDamage,
          additionalComment,
          immediateCorrectiveAction,
        },
      };

      // Generate PDF and upload to GlobalIncidentReports/admins/{userId}/incidentReports/{incidentNumber}/IncidentReport.pdf
      const pdfPath = await generatePDFWithHtml({ step1: step1Data, step2: report.step2, selectedIncidents: report.selectedIncidents });
      const storagePath = `GlobalIncidentReports/admins/${userId}/incidentReports/${incidentNumber}/IncidentReport.pdf`;
      const reference = storage().ref(storagePath);
      await reference.putFile(pdfPath);
      const pdfDownloadUrl = await reference.getDownloadURL();

      // Save to GlobalIncidentReports using the same pattern as inspection reports
      await saveReportToFirebase(userId, report, pdfDownloadUrl);

      // Update lastIncidentNumber with only the 3-digit number
      const threeDigitNumber = incidentNumber.slice(-3);
      console.log('Updating lastIncidentNumber:', {
        incidentNumber,
        threeDigitNumber
      });
      await database().ref('/GlobalIncidentReports/lastIncidentNumber').set(threeDigitNumber);

      await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err));

      setSyncModalVisible(false);
      setTimeout(() => {
        setPdfUrl(pdfDownloadUrl);
        setSuccessModalVisible(true);
      }, 200);

      // Clear persisted state
      await AsyncStorage.removeItem('IncidentReportStep1Data');
      await AsyncStorage.removeItem('IncidentReportStep2State');
      // Do NOT auto-navigate here; only show modal
    } catch (error) {
      console.error('Error generating report:', error);
      Alert.alert('Error', 'Failed to generate or save report.');
    } finally {
      // setSyncModalVisible(false); // This line is now handled by the setTimeout
    }
  };

  // Only allow navigation from the modal's Back or View PDF buttons
  const handleViewPDF = async (pdfUrl) => {
    if (!pdfUrl || !reportId) {
      Alert.alert('Error', 'PDF or Report ID not available.');
      return;
    }
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        // Do not navigate away
        return;
      }
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        // Do not auto-navigate on return
      });
    } catch (error) {
      console.error('Error viewing PDF:', error);
      Alert.alert('Error', 'Could not view the PDF. Please try again.');
    }
    // Do not close the modal automatically here
  };

  const handleNavigateBack = () => {
    setSuccessModalVisible(false);
    navigation.navigate('IncodentReport');
  };

  useImperativeHandle(ref, () => ({
    validateStep2,
    getStep2Data: () => ({
      incidentImages,
      incidentCause,
      equipmentDamage,
      additionalComment,
      immediateCorrectiveAction,
    }),
    generatePDF: generateReport,
  }));

  const handleRemoveImage = (index) => {
    Alert.alert(
      'Remove Image',
      'Are you sure you want to remove this image?',
      [
        { text: 'Cancel', style: 'cancel' },
        { 
          text: 'Remove', 
          style: 'destructive',
          onPress: () => {
            const newImages = [...incidentImages];
            newImages[index] = null;
            setIncidentImages(newImages);
            saveState();
          }
        },
      ],
      { cancelable: true }
    );
  };

  const handleFilePick = (index) => {
    Alert.alert(
      'Select Image',
      'Choose an option',
      [
        { text: 'Cancel', style: 'cancel' },
        { text: 'Take Picture', onPress: () => pickImage(index, 'camera') },
        { text: 'Choose from Gallery', onPress: () => pickImage(index, 'gallery') },
      ],
      { cancelable: true }
    );
  };

  const pickImage = async (index, source) => {
    const options = {
      mediaType: 'photo',
      maxWidth: 500,
      maxHeight: 500,
      quality: 0.8,
    };

    const callback = (response) => {
      if (response.didCancel) {
        console.log('User cancelled image picker');
      } else if (response.errorCode) {
        console.error('Image Picker Error:', response.errorCode, response.errorMessage);
        if (response.errorCode === 'permission') {
          Alert.alert(
            'Permission Error',
            'Photo library access is denied. Please enable it in Settings > Privacy > Photos.'
          );
        } else if (response.errorCode === 'library') {
          Alert.alert('No Photos Available', 'No photos are available in your library.');
        } else {
          Alert.alert('Error', response.errorMessage || 'Failed to pick image');
        }
      } else if (response.assets && response.assets.length > 0) {
        const uri = response.assets[0].uri;
        const newImages = [...incidentImages];
        newImages[index] = uri;
        setIncidentImages(newImages);
        saveState();
      } else {
        Alert.alert('No Selection', 'No image was selected.');
      }
    };

    if (source === 'camera') {
      launchCamera(options, callback);
    } else {
      launchImageLibrary(options, callback);
    }
  };

  const chunkArray = (arr, size) => {
    const result = [];
    for (let i = 0; i < arr.length; i += size) {
      result.push(arr.slice(i, i + size));
    }
    return result;
  };

  const renderStep2 = () => (
    <KeyboardAvoidingView
      style={{ flex: 1 }}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 64 : 0}
      enabled
    >
      <ScrollView
        style={styles.formContainer}
        contentContainerStyle={{ paddingBottom: 100, flexGrow: 1 }}
        keyboardShouldPersistTaps="handled"
        showsVerticalScrollIndicator={true}
        bounces={true}
        automaticallyAdjustKeyboardInsets={true}
      >
        <TouchableOpacity style={styles.incidentImagesButton}>
          <Text style={styles.incidentImagesButtonText}>Click Incident Images</Text>
        </TouchableOpacity>
        <View style={styles.incidentImagesGridWrapper}>
          <View style={styles.incidentImagesGrid}>
            {incidentImages.map((img, idx) => (
              <View key={idx} style={styles.incidentImageBox}>
                {img ? (
                  <View style={styles.imageContainer}>
                    <Image
                      source={{ uri: img }}
                      style={styles.incidentImage}
                      onError={(e) => console.log('Image error:', e.nativeEvent.error)}
                    />
                    <TouchableOpacity 
                      style={styles.removeImageButton} 
                      onPress={() => handleRemoveImage(idx)}
                    >
                      <MaterialIcons name="close" size={16} color="#fff" />
                    </TouchableOpacity>
                    <TouchableOpacity 
                      style={styles.cameraOverlay} 
                      onPress={() => handleFilePick(idx)}
                    >
                      <MaterialIcons name="camera-alt" size={24} color="#fff" />
                    </TouchableOpacity>
                  </View>
                ) : (
                  <TouchableOpacity style={styles.emptyImageBox} onPress={() => handleFilePick(idx)}>
                    <MaterialIcons name="camera-alt" size={36} color="#015185" />
                  </TouchableOpacity>
                )}
              </View>
            ))}
          </View>
        </View>
      </ScrollView>
    </KeyboardAvoidingView>
  );

  const renderStep3 = () => (
    <KeyboardAvoidingView
      style={{ flex: 1 }}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 64 : 0}
      enabled
    >
      <ScrollView
        style={{ flex: 1, backgroundColor: '#F5F6FA' }}
        contentContainerStyle={{ paddingBottom: 100, flexGrow: 1 }}
        keyboardShouldPersistTaps="handled"
        showsVerticalScrollIndicator={true}
        bounces={true}
        automaticallyAdjustKeyboardInsets={true}
      >
        <View style={styles.summaryCard}>
          <View style={styles.incidentCauseGroup}>
            <Text style={styles.incidentCauseLabel}>Description of incident</Text>
            <View style={styles.incidentCauseRelative}>
              <TextInput
                style={styles.incidentCauseInput}
                multiline
                value={incidentCause}
                onChangeText={(text) => { setIncidentCause(text); saveState(); }}
                placeholder="Type here..."
                placeholderTextColor="#A0A0A0"
                textAlignVertical="top"
              />
              <MaterialIcons name="mic" size={22} color="#015185" style={styles.incidentCauseMic} />
            </View>
          </View>
          <View style={styles.equipmentDamageGroup}>
            <Text style={styles.equipmentDamageLabel}>Details of damage / injury / loss</Text>
            <View style={styles.equipmentDamageRelative}>
              <TextInput
                style={styles.equipmentDamageInput}
                multiline
                value={equipmentDamage}
                onChangeText={(text) => { setEquipmentDamage(text); saveState(); }}
                placeholder="Type here..."
                placeholderTextColor="#A0A0A0"
                textAlignVertical="top"
              />
              <MaterialIcons name="mic" size={22} color="#015185" style={styles.equipmentDamageMic} />
            </View>
          </View>
          <View style={styles.additionalCommentGroup}>
            <Text style={styles.additionalCommentLabel}>Possible direct causes</Text>
            <View style={styles.additionalCommentRelative}>
              <TextInput
                style={styles.additionalCommentInput}
                multiline
                value={additionalComment}
                onChangeText={(text) => { setAdditionalComment(text); saveState(); }}
                placeholder="Type here..."
                placeholderTextColor="#A0A0A0"
                textAlignVertical="top"
              />
              <MaterialIcons name="mic" size={22} color="#015185" style={styles.additionalCommentMic} />
            </View>
          </View>
          <View style={styles.immediateCorrectiveActionGroup}>
            <Text style={styles.immediateCorrectiveActionLabel}>Immediate Corrective Action taken</Text>
            <View style={styles.immediateCorrectiveActionRelative}>
              <TextInput
                style={styles.immediateCorrectiveActionInput}
                multiline
                value={immediateCorrectiveAction}
                onChangeText={(text) => { setImmediateCorrectiveAction(text); saveState(); }}
                placeholder="Type here..."
                placeholderTextColor="#A0A0A0"
                textAlignVertical="top"
              />
              <MaterialIcons name="mic" size={22} color="#015185" style={styles.immediateCorrectiveActionMic} />
            </View>
          </View>
        </View>
      </ScrollView>
    </KeyboardAvoidingView>
  );

  return (
    <View style={styles.container}>
      {currentStep === 2 && renderStep3()}
      {currentStep === 3 && renderStep2()}
      <SyncModal visible={syncModalVisible} />
      <SuccessModal
        visible={successModalVisible}
        onClose={() => setSuccessModalVisible(false)}
        onNavigateBack={handleNavigateBack}
        onViewPDF={handleViewPDF}
        pdfUrl={pdfUrl}
      />
      <Toast />
    </View>
  );
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  formContainer: {
    flex: 1,
    backgroundColor: '#fff',
    padding: 15,
  },
  incidentImagesButton: {
    backgroundColor: '#015185',
    borderRadius: 4,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 8,
    marginBottom: 18,
    marginHorizontal: 15,
  },
  incidentImagesButtonText: {
    color: '#fff',
    fontWeight: '600',
    fontSize: 14,
  },
  incidentImagesGridWrapper: {
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
  },
  incidentImagesGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'center',
    alignItems: 'center',
    width: 320, // or use width: '90%' for responsiveness
  },
  incidentImageBox: {
    width: 90,
    height: 90,
    margin: 8,
    backgroundColor: '#F5F8FA',
    borderRadius: 8,
    alignItems: 'center',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#B0C4DE',
    position: 'relative',
    overflow: 'hidden',
  },
  imageContainer: {
    position: 'relative',
    width: '100%',
    height: '100%',
  },
  incidentImage: {
    width: '100%',
    height: '100%',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#B0C4DE',
  },
  removeImageButton: {
    position: 'absolute',
    top: 5,
    right: 5,
    backgroundColor: 'rgba(255, 0, 0, 0.8)',
    borderRadius: 12,
    width: 24,
    height: 24,
    alignItems: 'center',
    justifyContent: 'center',
    zIndex: 2,
  },
  cameraOverlay: {
    position: 'absolute',
    bottom: 5,
    right: 5,
    backgroundColor: 'rgba(0, 0, 0, 0.6)',
    borderRadius: 12,
    padding: 5,
    zIndex: 1,
  },
  emptyImageBox: {
    width: '100%',
    height: '100%',
    backgroundColor: '#F5F8FA',
    borderRadius: 8,
    alignItems: 'center',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#B0C4DE',
  },
  summaryCard: {
    backgroundColor: '#fff',
    borderRadius: 14,
    margin: 18,
    padding: 18,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.08,
    shadowRadius: 8,
    elevation: 4,
  },
  incidentCauseGroup: {
    marginBottom: 18,
  },
  incidentCauseLabel: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  incidentCauseRelative: {
    position: 'relative',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 12,
    marginTop: 6,
    minHeight: 120,
  },
  incidentCauseInput: {
    flex: 1,
    minHeight: 120,
    maxHeight: 150,
    paddingHorizontal: 16,
    paddingVertical: 14,
    fontSize: 15,
    color: '#222',
    backgroundColor: 'transparent',
    borderRadius: 12,
    textAlignVertical: 'top',
  },
  incidentCauseMic: {
    position: 'absolute',
    right: 16,
    top: 14,
  },
  equipmentDamageGroup: {
    marginBottom: 18,
  },
  equipmentDamageLabel: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  equipmentDamageRelative: {
    position: 'relative',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 12,
    marginTop: 6,
    minHeight: 120,
  },
  equipmentDamageInput: {
    flex: 1,
    minHeight: 120,
    maxHeight: 150,
    paddingHorizontal: 15,
    paddingVertical: 13,
    fontSize: 15,
    color: '#222',
    backgroundColor: 'transparent',
    borderRadius: 12,
    textAlignVertical: 'top',
  },
  equipmentDamageMic: {
    position: 'absolute',
    right: 15,
    top: 13,
  },
  additionalCommentGroup: {
    marginBottom: 18,
  },
  additionalCommentLabel: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  additionalCommentRelative: {
    position: 'relative',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 12,
    marginTop: 6,
    minHeight: 120,
  },
  additionalCommentInput: {
    flex: 1,
    minHeight: 120,
    maxHeight: 150,
    paddingHorizontal: 14,
    paddingVertical: 14,
    fontSize: 14,
    color: '#333',
    backgroundColor: 'transparent',
    borderRadius: 12,
    textAlignVertical: 'top',
  },
  additionalCommentMic: {
    position: 'absolute',
    right: 14,
    top: 14,
  },
  immediateCorrectiveActionGroup: {
    marginBottom: 18,
  },
  immediateCorrectiveActionLabel: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  immediateCorrectiveActionRelative: {
    position: 'relative',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 12,
    marginTop: 6,
    minHeight: 120,
  },
  immediateCorrectiveActionInput: {
    flex: 1,
    minHeight: 120,
    maxHeight: 150,
    paddingHorizontal: 14,
    paddingVertical: 14,
    fontSize: 14,
    color: '#333',
    backgroundColor: 'transparent',
    borderRadius: 12,
    textAlignVertical: 'top',
  },
  immediateCorrectiveActionMic: {
    position: 'absolute',
    right: 14,
    top: 14,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
  },
  modalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  modalActions: {
    flexDirection: 'row',
    justifyContent: 'center',
  },
  modalButton: {
    backgroundColor: '#d3d3d3',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
    marginHorizontal: 5,
  },
  viewButton: {
    backgroundColor: '#015185',
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  successIcon: {
    alignSelf: 'center',
    marginBottom: 10,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
});

export default IncidentReportStep2;