import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  Image,
  StatusBar,
  Dimensions,
  RefreshControl,
  Alert,
  Platform,
  Animated,
  Modal,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import { useNavigation, useFocusEffect, useRoute } from '@react-navigation/native';
import notifee, { AndroidImportance } from '@notifee/react-native';

const { width, height } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;
const TOTAL_HEADER_HEIGHT = width * 0.25;
const NOTIFICATIONS_PER_PAGE = 16;

// SyncModal Component with animated sync.png (same as AdminLoginScreen)
const SyncModal = ({ visible, message = 'Loading Notifications...' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalOverlay}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HeaderWellNotificaton = ({ navigation }) => {
  const route = useRoute();
  const [notifications, setNotifications] = useState([]);
  const [loading, setLoading] = useState(true);
  const [refreshing, setRefreshing] = useState(false);
  const [currentUserId, setCurrentUserId] = useState('');
  const [unreadCount, setUnreadCount] = useState(0);
  const [filter, setFilter] = useState('all'); // all, unread, read
  const [recentNotices, setRecentNotices] = useState([]);
  const [noticesLoading, setNoticesLoading] = useState(false);
  const [highlightedNotificationId, setHighlightedNotificationId] = useState(null);
  const scrollViewRef = useRef(null);
  
  // Pagination states
  const [lastNotificationKey, setLastNotificationKey] = useState(null);
  const [hasMoreNotifications, setHasMoreNotifications] = useState(true);
  const [loadingMore, setLoadingMore] = useState(false);
  const [allNotifications, setAllNotifications] = useState([]);

  // Get current user ID on component mount
  useEffect(() => {
    getCurrentUserId();
  }, []);

  // Get highlighted notification ID from route parameters
  useEffect(() => {
    if (route.params?.highlightedNotificationId) {
      setHighlightedNotificationId(route.params.highlightedNotificationId);
      console.log('🔴 Highlighted notification ID:', route.params.highlightedNotificationId);
    }
  }, [route.params]);

  // Load recent notices when user ID is available
  useEffect(() => {
    if (currentUserId) {
      loadRecentNotices();
    }
  }, [currentUserId]);

  // Refresh notifications when screen comes into focus
  useFocusEffect(
    React.useCallback(() => {
      if (currentUserId) {
        loadNotifications();
      }
    }, [currentUserId])
  );

  const getCurrentUserId = async () => {
    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (userId) {
        setCurrentUserId(userId);
        console.log('Current user ID:', userId);
      }
    } catch (error) {
      console.error('Error getting current user ID:', error);
    }
  };

  const loadNotifications = async (loadMore = false) => {
    if (!currentUserId) return;
    
    if (loadMore && (!hasMoreNotifications || loadingMore)) return;
    
    if (loadMore) {
      setLoadingMore(true);
    } else {
      setLoading(true);
    }
    
    try {
      const notificationsRef = database().ref(`/Globalusers/admins/${currentUserId}/notifications`);
      
      let query = notificationsRef.orderByChild('timestamp');
      
      if (loadMore && lastNotificationKey) {
        // Load older notifications (pagination)
        query = query.endBefore(lastNotificationKey).limitToLast(NOTIFICATIONS_PER_PAGE);
      } else {
        // Initial load - get latest 16 notifications
        query = query.limitToLast(NOTIFICATIONS_PER_PAGE);
      }
      
      const snapshot = await query.once('value');
      
      if (snapshot.exists()) {
        const notificationsData = snapshot.val();
        const notificationsArray = Object.entries(notificationsData).map(([id, notification]) => ({
          id,
          ...notification,
          timestamp: notification.timestamp || Date.now(),
          isRead: notification.isRead || false,
        }));
        
        // Sort by timestamp (newest first)
        const sortedNotifications = notificationsArray.sort((a, b) => b.timestamp - a.timestamp);
        
        if (loadMore) {
          // Append to existing notifications
          const combinedNotifications = [...allNotifications, ...sortedNotifications];
          // Remove duplicates
          const uniqueNotifications = Array.from(
            new Map(combinedNotifications.map(item => [item.id, item])).values()
          );
          setAllNotifications(uniqueNotifications);
          setNotifications(uniqueNotifications);
        } else {
          // Initial load
          setAllNotifications(sortedNotifications);
          setNotifications(sortedNotifications);
        }
        
        // Update pagination state
        if (sortedNotifications.length > 0) {
          setLastNotificationKey(sortedNotifications[sortedNotifications.length - 1].timestamp);
        }
        
        // Check if there are more notifications
        setHasMoreNotifications(sortedNotifications.length === NOTIFICATIONS_PER_PAGE);
        
        updateUnreadCount(loadMore ? [...allNotifications, ...sortedNotifications] : sortedNotifications);
      } else {
        if (!loadMore) {
          setNotifications([]);
          setAllNotifications([]);
          setUnreadCount(0);
          setHasMoreNotifications(false);
        }
      }
    } catch (error) {
      console.error('Error loading notifications:', error);
      Alert.alert('Error', 'Failed to load notifications');
    } finally {
      if (loadMore) {
        setLoadingMore(false);
      } else {
        setLoading(false);
      }
    }
  };

  const updateUnreadCount = (notificationsList) => {
    const unread = notificationsList.filter(notification => !notification.isRead).length;
    setUnreadCount(unread);
  };

  const markAsRead = async (notificationId) => {
    try {
      const notificationRef = database().ref(`/Globalusers/admins/${currentUserId}/notifications/${notificationId}`);
      await notificationRef.update({ isRead: true });
      
      // Update local state
      setNotifications(prev => 
        prev.map(notification => 
          notification.id === notificationId 
            ? { ...notification, isRead: true }
            : notification
        )
      );
      
      // Update unread count
      updateUnreadCount(notifications);
    } catch (error) {
      console.error('Error marking notification as read:', error);
    }
  };

  const markAllAsRead = async () => {
    try {
      const notificationsRef = database().ref(`/Globalusers/admins/${currentUserId}/notifications`);
      const updates = {};
      
      notifications.forEach(notification => {
        if (!notification.isRead) {
          updates[`${notification.id}/isRead`] = true;
        }
      });
      
      await notificationsRef.update(updates);
      
      // Update local state
      setNotifications(prev => 
        prev.map(notification => ({ ...notification, isRead: true }))
      );
      
      setUnreadCount(0);
      Alert.alert('Success', 'All notifications marked as read');
    } catch (error) {
      console.error('Error marking all notifications as read:', error);
      Alert.alert('Error', 'Failed to mark all notifications as read');
    }
  };

  const deleteNotification = async (notificationId) => {
    Alert.alert(
      'Delete Notification',
      'Are you sure you want to delete this notification?',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            try {
              const notificationRef = database().ref(`/Globalusers/admins/${currentUserId}/notifications/${notificationId}`);
              await notificationRef.remove();
              
              // Update local state
              setNotifications(prev => prev.filter(notification => notification.id !== notificationId));
              
              // Update unread count
              updateUnreadCount(notifications.filter(notification => notification.id !== notificationId));
            } catch (error) {
              console.error('Error deleting notification:', error);
              Alert.alert('Error', 'Failed to delete notification');
            }
          }
        }
      ]
    );
  };

  const onRefresh = async () => {
    setRefreshing(true);
    // Reset pagination
    setLastNotificationKey(null);
    setHasMoreNotifications(true);
    setAllNotifications([]);
    await loadNotifications(false);
    await loadRecentNotices();
    setRefreshing(false);
  };
  
  // Handle scroll to load more
  const handleScroll = ({ nativeEvent }) => {
    const { layoutMeasurement, contentOffset, contentSize } = nativeEvent;
    const paddingToBottom = 20;
    
    // Check if user has scrolled to bottom
    const isCloseToBottom = layoutMeasurement.height + contentOffset.y >= contentSize.height - paddingToBottom;
    
    if (isCloseToBottom && hasMoreNotifications && !loadingMore && !loading) {
      console.log('Loading more notifications...');
      loadNotifications(true);
    }
  };

  // Load recent notices from all users
  const loadRecentNotices = async () => {
    if (!currentUserId) return;
    
    setNoticesLoading(true);
    try {
      const allUsersRef = database().ref('/users');
      const snapshot = await allUsersRef.once('value');
      
      if (snapshot.exists()) {
        const allUsersData = snapshot.val();
        const allNotices = [];
        
        Object.keys(allUsersData).forEach((userId) => {
          const userNotices = allUsersData[userId].notices;
          if (userNotices) {
            Object.values(userNotices).forEach((notice) => {
              allNotices.push({
                noticeId: notice.noticeId,
                title: notice.title,
                description: notice.description,
                imageUrl: notice.imageUrl || null,
                createdAt: notice.createdAt,
                createdBy: userId,
              });
            });
          }
        });
        
        // Sort by creation date (newest first) and take first 10
        const sortedNotices = allNotices
          .sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt))
          .slice(0, 10);
        
        setRecentNotices(sortedNotices);
      } else {
        setRecentNotices([]);
      }
    } catch (error) {
      console.error('Error loading recent notices:', error);
    } finally {
      setNoticesLoading(false);
    }
  };

  // Handle notice click
  const handleNoticeClick = (notice) => {
    navigation.navigate('ViewNotificationsadded', {
      title: notice.title,
      description: notice.description,
      imageUrl: notice.imageUrl,
      createdAt: notice.createdAt,
    });
  };

  const getFilteredNotifications = () => {
    switch (filter) {
      case 'unread':
        return notifications.filter(notification => !notification.isRead);
      case 'read':
        return notifications.filter(notification => notification.isRead);
      default:
        return notifications;
    }
  };

  const formatTimestamp = (timestamp) => {
    const now = Date.now();
    const diff = now - timestamp;
    const minutes = Math.floor(diff / (1000 * 60));
    const hours = Math.floor(diff / (1000 * 60 * 60));
    const days = Math.floor(diff / (1000 * 60 * 60 * 24));

    if (minutes < 1) return 'Just now';
    if (minutes < 60) return `${minutes}m ago`;
    if (hours < 24) return `${hours}h ago`;
    if (days < 7) return `${days}d ago`;
    
    return new Date(timestamp).toLocaleDateString();
  };

  const getNotificationIcon = (type) => {
    switch (type) {
      case 'welcome':
        return '🎉';
      case 'safety_alert':
        return '🚨';
      case 'maintenance':
        return '🔧';
      case 'update':
        return '📢';
      case 'test':
        return '🧪';
      case 'notice_board':
        return '📢';
      case 'dvir_report':
        return '🚛';
      case 'inspection_report':
        return '🔍';
      case 'incident_report':
        return '🚨';
      case 'request_maintenance_created':
        return '🔧';
      case 'work_order_created':
        return '📋';
      case 'job_card_created':
        return '📋';
      case 'service_schedule_created':
        return '📅';
      case 'hr_application_approval_rejection':
        return '🏢';
      case 'hr_application_supervisor_approved':
        return '📋';
      default:
        return '📱';
    }
  };

  const getNotificationColor = (type) => {
    switch (type) {
      case 'welcome':
        return '#4CAF50';
      case 'safety_alert':
        return '#F44336';
      case 'maintenance':
        return '#FF9800';
      case 'update':
        return '#2196F3';
      case 'test':
        return '#9C27B0';
      case 'notice_board':
        return '#3481BC';
      case 'dvir_report':
        return '#FF6B35';
      case 'inspection_report':
        return '#FF9800';
      case 'incident_report':
        return '#F44336';
      case 'request_maintenance_created':
        return '#015185';
      case 'work_order_created':
        return '#4CAF50';
      case 'job_card_created':
        return '#4CAF50';
      case 'service_schedule_created':
        return '#3481BC';
      case 'hr_application_approval_rejection':
        return '#015185';
      case 'hr_application_supervisor_approved':
        return '#FF9800';
      default:
        return '#607D8B';
    }
  };

  const renderNotification = (notification) => {
    const isHighlighted = highlightedNotificationId && 
      (notification.id === highlightedNotificationId || 
       notification.notificationId === highlightedNotificationId ||
       notification.data?.noticeId === highlightedNotificationId ||
       notification.data?.reportId === highlightedNotificationId);
    
    return (
    <TouchableOpacity
      key={notification.id}
      style={[
        styles.notificationItem,
        !notification.isRead && styles.unreadNotification,
        notification.type === 'inspection_report' && styles.inspectionNotification,
        notification.type === 'dvir_report' && styles.dvirNotification,
        notification.type === 'incident_report' && styles.incidentNotification,
        (notification.type === 'work_order_created' || notification.type === 'job_card_created') && styles.workOrderNotification,
        notification.type === 'service_schedule_created' && styles.serviceScheduleNotification,
        notification.type === 'hr_application_approval_rejection' && styles.hrApplicationNotification,
        notification.type === 'hr_application_supervisor_approved' && styles.hrManagerNotification,
        isHighlighted && styles.highlightedNotification
      ]}
      onPress={() => {
        // Mark as read first
        markAsRead(notification.id);
        
        // Then handle navigation based on type
        if (notification.type === 'inspection_report' || notification.data?.type === 'inspection_report') {
          console.log('🔍 Inspection report notification clicked:', notification);
          
          // Navigate to inspection report home
          navigation.navigate('Inspection_Report_Home', {
            notificationData: notification,
            fromNotification: true
          });
        } else if (notification.type === 'dvir_report' || notification.data?.type === 'dvir_report_created') {
          console.log('🚛 DVIR report notification clicked:', notification);
          
          // Navigate to DVIR report home
          navigation.navigate('DviranddotInHome', {
            notificationData: notification,
            fromNotification: true
          });
        } else if (notification.type === 'incident_report' || notification.data?.type === 'incident_report_created') {
          console.log('🚨 Incident report notification clicked:', notification);
          
          // Navigate to incident report home
          navigation.navigate('IncodentReport', {
            notificationData: notification,
            fromNotification: true
          });
        } else if (notification.type === 'work_order_created' || notification.data?.type === 'work_order_created' || 
                   notification.type === 'job_card_created' || notification.data?.type === 'job_card_created') {
          console.log('📋 Work order notification clicked:', notification);
          
          // Navigate directly to AllWotab (work order list)
          navigation.navigate('WorkOrdertab', {
            notificationData: notification,
            fromNotification: true
          });
        } else if (notification.type === 'service_schedule_created' || notification.data?.type === 'service_schedule_created') {
          console.log('📅 Service schedule notification clicked:', notification);
          
          // Navigate to service schedule home
          navigation.navigate('ServiceScheduleHome', {
            notificationData: notification,
            fromNotification: true
          });
        } else if (notification.type === 'hr_application_approval_rejection' || notification.data?.type === 'hr_application_approval_rejection') {
          console.log('🏢 HR Application approval/rejection notification clicked:', notification);
          
          // Navigate to HR application home
          navigation.navigate('AdminHrApplyHome', {
            notificationData: notification,
            fromNotification: true
          });
        } else if (notification.type === 'hr_application_supervisor_approved' || notification.data?.type === 'hr_application_supervisor_approved') {
          console.log('📋 HR Application supervisor approved notification clicked:', notification);
          
          // Navigate to HR application home
          navigation.navigate('AdminHrApplyHome', {
            notificationData: notification,
            fromNotification: true
          });
        } else {
          // Handle regular notifications
          navigation.navigate('ViewNotificationWell', {
            title: notification.title || 'No Title',
            description: notification.body || 'No Description',
            timestamp: notification.timestamp,
            type: notification.data?.type || 'general',
            notificationId: notification.id,
          });
        }
      }}
      activeOpacity={0.7}
    >
      <View style={styles.notificationHeader}>
        <View style={styles.notificationIconContainer}>
          <Text style={styles.notificationIcon}>
            {notification.type === 'inspection_report' ? '🔍' : 
             notification.type === 'dvir_report' ? '🚛' : 
             notification.type === 'incident_report' ? '🚨' :
             notification.type === 'request_maintenance_created' ? '🔧' :
             notification.type === 'work_order_created' ? '📋' :
             notification.type === 'job_card_created' ? '📋' :
             notification.type === 'service_schedule_created' ? '📅' :
             notification.type === 'hr_application_approval_rejection' ? '🏢' :
             notification.type === 'hr_application_supervisor_approved' ? '📋' :
             getNotificationIcon(notification.data?.type)}
          </Text>
        </View>
        <View style={styles.notificationContent}>
          <Text style={[
            styles.notificationTitle,
            !notification.isRead && styles.unreadTitle
          ]}>
            {notification.title ? notification.title.substring(0, 8) + (notification.title.length > 8 ? '...' : '') : 'No Title'}
          </Text>
          <Text style={styles.notificationBody} numberOfLines={1}>
            {notification.body ? notification.body.substring(0, 10) + (notification.body.length > 10 ? '...' : '') : 'No Description'}
          </Text>
          <Text style={styles.notificationTime}>
            {formatTimestamp(notification.timestamp)}
          </Text>
          
          {/* Show inspection report details if available */}
          {(notification.type === 'inspection_report' || notification.data?.type === 'inspection_report') && (
            <View style={styles.inspectionDetails}>
              <Text style={styles.inspectionDetailText}>
                Request: {notification.data?.requestNumber || notification.requestNumber || 'N/A'}
              </Text>
              {(notification.data?.project || notification.project) && (
                <Text style={styles.inspectionDetailText}>
                  Project: {notification.data?.project || notification.project}
                </Text>
              )}
              {(notification.data?.equipmentName || notification.equipmentName) && (
                <Text style={styles.inspectionDetailText}>
                  Equipment: {notification.data?.equipmentName || notification.equipmentName}
                </Text>
              )}
            </View>
          )}
          
          {/* Show DVIR report details if available */}
          {notification.type === 'dvir_report' && notification.requestNumber && (
            <View style={styles.dvirDetails}>
              <Text style={styles.dvirDetailText}>
                Request: {notification.requestNumber}
              </Text>
              {notification.project && (
                <Text style={styles.dvirDetailText}>
                  Project: {notification.project}
                </Text>
              )}
              {notification.equipmentName && (
                <Text style={styles.dvirDetailText}>
                  Equipment: {notification.equipmentName}
                </Text>
              )}
              {notification.inspectionType && (
                <Text style={styles.dvirDetailText}>
                  Type: {notification.inspectionType}
                </Text>
              )}
            </View>
          )}
          
          {/* Show incident report details if available */}
          {notification.type === 'incident_report' && notification.incidentNumber && (
            <View style={styles.incidentDetails}>
              <Text style={styles.incidentDetailText}>
                Incident: {notification.incidentNumber}
              </Text>
              {notification.country && (
                <Text style={styles.incidentDetailText}>
                  Country: {notification.country}
                </Text>
              )}
              {notification.project && (
                <Text style={styles.incidentDetailText}>
                  Project: {notification.project}
                </Text>
              )}
              {notification.adminName && (
                <Text style={styles.incidentDetailText}>
                  Created by: {notification.adminName}
                </Text>
              )}
            </View>
          )}
          
          {/* Show work order details if available */}
          {(notification.type === 'work_order_created' || notification.type === 'job_card_created') && notification.requestNumber && (
            <View style={styles.workOrderDetails}>
              <Text style={styles.workOrderDetailText}>
                Work Order: {notification.requestNumber}
              </Text>
              {notification.country && (
                <Text style={styles.workOrderDetailText}>
                  Country: {notification.country}
                </Text>
              )}
              {notification.project && (
                <Text style={styles.workOrderDetailText}>
                  Project: {notification.project}
                </Text>
              )}
              {notification.equipment && (
                <Text style={styles.workOrderDetailText}>
                  Equipment: {notification.equipment}
                </Text>
              )}
              {notification.creatorName && (
                <Text style={styles.workOrderDetailText}>
                  Created by: {notification.creatorName}
                </Text>
              )}
            </View>
          )}
          
          {/* Show HR application details if available */}
          {notification.type === 'hr_application_approval_rejection' && notification.data && (
            <View style={styles.hrApplicationDetails}>
              <Text style={styles.hrApplicationDetailText}>
                Application: {notification.data.applicationType || 'HR Application'}
              </Text>
              {notification.data.employeeName && (
                <Text style={styles.hrApplicationDetailText}>
                  Employee: {notification.data.employeeName}
                </Text>
              )}
              {notification.data.employeeNo && (
                <Text style={styles.hrApplicationDetailText}>
                  Employee No: {notification.data.employeeNo}
                </Text>
              )}
              {notification.data.action && (
                <Text style={styles.hrApplicationDetailText}>
                  Status: {notification.data.action}
                </Text>
              )}
              {notification.data.approverName && (
                <Text style={styles.hrApplicationDetailText}>
                  {notification.data.action} by: {notification.data.approverName}
                </Text>
              )}
            </View>
          )}
          
          {/* Show HR manager notification details if available */}
          {notification.type === 'hr_application_supervisor_approved' && notification.data && (
            <View style={styles.hrManagerDetails}>
              <Text style={styles.hrManagerDetailText}>
                Application: {notification.data.applicationType || 'HR Application'}
              </Text>
              {notification.data.employeeName && (
                <Text style={styles.hrManagerDetailText}>
                  Employee: {notification.data.employeeName}
                </Text>
              )}
              {notification.data.employeeNo && (
                <Text style={styles.hrManagerDetailText}>
                  Employee No: {notification.data.employeeNo}
                </Text>
              )}
              {notification.data.country && (
                <Text style={styles.hrManagerDetailText}>
                  Country: {notification.data.country}
                </Text>
              )}
              {notification.data.project && (
                <Text style={styles.hrManagerDetailText}>
                  Project: {notification.data.project}
                </Text>
              )}
            </View>
          )}
        </View>
        <View style={styles.notificationActions}>
          {!notification.isRead && (
            <View style={styles.unreadIndicator} />
          )}
          <TouchableOpacity
            style={styles.viewButton}
            onPress={() => {
              if (notification.type === 'inspection_report' || notification.data?.type === 'inspection_report') {
                navigation.navigate('Inspection_Report_Home', {
                  notificationData: notification,
                  fromNotification: true
                });
              } else if (notification.type === 'dvir_report' || notification.data?.type === 'dvir_report_created') {
                navigation.navigate('DviranddotInHome', {
                  notificationData: notification,
                  fromNotification: true
                });
              } else if (notification.type === 'incident_report' || notification.data?.type === 'incident_report_created') {
                navigation.navigate('IncidentReport', {
                  notificationData: notification,
                  fromNotification: true
                });
              } else if (notification.type === 'work_order_created' || notification.data?.type === 'work_order_created' || 
                         notification.type === 'job_card_created' || notification.data?.type === 'job_card_created') {
                navigation.navigate('WorkOrdertab', {
                  notificationData: notification,
                  fromNotification: true
                });
              } else if (notification.type === 'service_schedule_created' || notification.data?.type === 'service_schedule_created') {
                navigation.navigate('ServiceScheduleHome', {
                  notificationData: notification,
                  fromNotification: true
                });
              } else if (notification.type === 'hr_application_approval_rejection' || notification.data?.type === 'hr_application_approval_rejection') {
                navigation.navigate('AdminHrApplyHome', {
                  notificationData: notification,
                  fromNotification: true
                });
              } else if (notification.type === 'hr_application_supervisor_approved' || notification.data?.type === 'hr_application_supervisor_approved') {
                navigation.navigate('AdminHrApplyHome', {
                  notificationData: notification,
                  fromNotification: true
                });
              } else {
                navigation.navigate('ViewNotificationWell', {
                  title: notification.title || 'No Title',
                  description: notification.body || 'No Description',
                  timestamp: notification.timestamp,
                  type: notification.data?.type || 'general',
                  notificationId: notification.id,
                });
              }
            }}
          >
            <Text style={styles.viewButtonText}>
              {notification.type === 'inspection_report' || notification.data?.type === 'inspection_report' ? 'View Report' : 
               notification.type === 'dvir_report' || notification.data?.type === 'dvir_report_created' ? 'View DVIR' : 
               notification.type === 'incident_report' || notification.data?.type === 'incident_report_created' ? 'View Incident' : 
               notification.type === 'work_order_created' || notification.data?.type === 'work_order_created' || 
               notification.type === 'job_card_created' || notification.data?.type === 'job_card_created' ? 'View Job Order' : 
               notification.type === 'service_schedule_created' || notification.data?.type === 'service_schedule_created' ? 'View Schedule' : 
               notification.type === 'hr_application_approval_rejection' || notification.data?.type === 'hr_application_approval_rejection' ? 'View Application' : 
               notification.type === 'hr_application_supervisor_approved' || notification.data?.type === 'hr_application_supervisor_approved' ? 'View Application' : 'View'}
            </Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.deleteButton}
            onPress={() => deleteNotification(notification.id)}
          >
            <MaterialIcons name="delete-outline" size={20} color="#FF6B6B" />
          </TouchableOpacity>
        </View>
      </View>
    </TouchableOpacity>
    );
  };

  const renderEmptyState = () => (
    <View style={styles.emptyState}>
      <MaterialIcons name="notifications-none" size={80} color="#BDBDBD" />
      <Text style={styles.emptyStateTitle}>No Notifications</Text>
      <Text style={styles.emptyStateSubtitle}>
        {filter === 'all' 
          ? "You're all caught up! No notifications yet."
          : filter === 'unread'
          ? "No unread notifications."
          : "No read notifications."
        }
      </Text>
    </View>
  );

  return (
    <View style={styles.container}>
      <StatusBar
        translucent
        backgroundColor="transparent"
        barStyle="light-content"
      />
      
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity
            style={styles.backButton}
            onPress={() => navigation.goBack()}
          >
            <MaterialIcons name="arrow-back" size={width * 0.07} color="white" />
          </TouchableOpacity>
          
          <Text style={styles.headerTitle}>Notifications</Text>
          
          <View style={styles.headerActions}>
            {unreadCount > 0 && (
              <TouchableOpacity
                style={styles.markAllReadButton}
                onPress={markAllAsRead}
              >
                <MaterialIcons name="done-all" size={20} color="white" />
              </TouchableOpacity>
            )}
          </View>
        </View>
      </LinearGradient>

      {/* Filter Tabs */}
      <View style={styles.filterContainer}>
        <ScrollView horizontal showsHorizontalScrollIndicator={false}>
          <TouchableOpacity
            style={[styles.filterTab, filter === 'all' && styles.activeFilterTab]}
            onPress={() => setFilter('all')}
          >
            <Text style={[styles.filterText, filter === 'all' && styles.activeFilterText]}>
              All ({notifications.length})
            </Text>
          </TouchableOpacity>
          
          <TouchableOpacity
            style={styles.allNoticesFilterButton}
            onPress={() => navigation.navigate('Notice_BoardAC')}
            activeOpacity={0.7}
          >
            <MaterialIcons name="list-alt" size={16} color="#3481BC" />
            <Text style={styles.allNoticesFilterText}>All Notices</Text>
          </TouchableOpacity>
          
          <TouchableOpacity
            style={[styles.filterTab, filter === 'unread' && styles.activeFilterTab]}
            onPress={() => setFilter('unread')}
          >
            <Text style={[styles.filterText, filter === 'unread' && styles.activeFilterText]}>
              Unread ({unreadCount})
            </Text>
          </TouchableOpacity>
          
          <TouchableOpacity
            style={[styles.filterTab, filter === 'read' && styles.activeFilterTab]}
            onPress={() => setFilter('read')}
          >
            <Text style={[styles.filterText, filter === 'read' && styles.activeFilterText]}>
              Read ({notifications.filter(n => n.isRead).length})
            </Text>
          </TouchableOpacity>
        </ScrollView>
      </View>

      {/* Notifications List */}
      <ScrollView
        ref={scrollViewRef}
        style={styles.notificationsContainer}
        refreshControl={
          <RefreshControl refreshing={refreshing} onRefresh={onRefresh} />
        }
        showsVerticalScrollIndicator={false}
        onScroll={handleScroll}
        scrollEventThrottle={400}
      >
        {getFilteredNotifications().length > 0 ? (
          <>
            {getFilteredNotifications().map(renderNotification)}
            
            {/* Loading More Indicator */}
            {loadingMore && (
              <View style={styles.loadingMoreContainer}>
                <ActivityIndicator size="small" color="#3481BC" />
                <Text style={styles.loadingMoreText}>Loading more notifications...</Text>
              </View>
            )}
            
            {/* End of List Indicator */}
            {!hasMoreNotifications && notifications.length > 0 && (
              <View style={styles.endOfListContainer}>
                <Text style={styles.endOfListText}>No more notifications</Text>
              </View>
            )}
            
            {/* View All Notices Button */}
            <View style={styles.viewAllNoticesContainer}>
              <TouchableOpacity
                style={styles.viewAllNoticesButton}
                onPress={() => navigation.navigate('Notice_BoardAC')}
                activeOpacity={0.7}
              >
                <MaterialIcons name="list-alt" size={20} color="#3481BC" />
                <Text style={styles.viewAllNoticesText}>View All Notices</Text>
                <MaterialIcons name="arrow-forward" size={16} color="#3481BC" />
              </TouchableOpacity>
            </View>
          </>
        ) : (
          renderEmptyState()
        )}
      </ScrollView>
      
      {/* Sync Modal Loader */}
      <SyncModal visible={loading} message="Loading Notifications..." />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#F5F5F5',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'ios' ? width * 0.08 : width * 0.04,
    height: '100%',
    top: 7,
  },
  backButton: {
    padding: 8,
  },
  headerTitle: {
    fontSize: scale(20),
    fontWeight: 'bold',
    color: 'white',
  },
  headerActions: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  markAllReadButton: {
    padding: 8,
    backgroundColor: 'rgba(255, 255, 255, 0.2)',
    borderRadius: 20,
  },
  filterContainer: {
    backgroundColor: 'white',
    paddingVertical: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
  },
  filterTab: {
    paddingHorizontal: 20,
    paddingVertical: 10,
    marginHorizontal: 5,
    borderRadius: 20,
    backgroundColor: '#F5F5F5',
  },
  activeFilterTab: {
    backgroundColor: '#3481BC',
  },
  filterText: {
    fontSize: scale(14),
    color: '#666',
    fontWeight: '500',
  },
  activeFilterText: {
    color: 'white',
  },
  allNoticesFilterButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 16,
    paddingVertical: 10,
    marginHorizontal: 5,
    borderRadius: 20,
    borderWidth: 1,
    borderColor: '#3481BC',
    minWidth: 100,
  },
  allNoticesFilterText: {
    fontSize: scale(14),
    color: '#3481BC',
    fontWeight: '600',
    marginLeft: 4,
  },
  // All Notices Horizontal Row Styles
  allNoticesContainer: {
    backgroundColor: 'white',
    paddingVertical: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
  },
  allNoticesButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 16,
    paddingVertical: 10,
    marginHorizontal: 8,
    borderRadius: 25,
    borderWidth: 1,
    borderColor: '#3481BC',
    minWidth: 120,
  },
  allNoticesButtonText: {
    fontSize: scale(14),
    fontWeight: '600',
    color: '#3481BC',
    marginHorizontal: 8,
  },
  addNoticeButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 16,
    paddingVertical: 10,
    marginHorizontal: 8,
    borderRadius: 25,
    borderWidth: 1,
    borderColor: '#4CAF50',
    minWidth: 120,
  },
  addNoticeButtonText: {
    fontSize: scale(14),
    fontWeight: '600',
    color: '#4CAF50',
    marginHorizontal: 8,
  },
  viewNoticesButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 16,
    paddingVertical: 10,
    marginHorizontal: 8,
    borderRadius: 25,
    borderWidth: 1,
    borderColor: '#FF9800',
    minWidth: 120,
  },
  viewNoticesButtonText: {
    fontSize: scale(14),
    fontWeight: '600',
    color: '#FF9800',
    marginHorizontal: 8,
  },
  notificationsContainer: {
    flex: 1,
    paddingHorizontal: 20,
  },
  notificationItem: {
    backgroundColor: 'white',
    borderRadius: 12,
    marginVertical: 8,
    padding: 16,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  unreadNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#3481BC',
    backgroundColor: '#F8F9FF',
  },
  inspectionNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#FF9800', // Orange color for inspection reports
    backgroundColor: '#FFFBE6', // Light yellow background
  },
  dvirNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#FF6B35', // Orange-red color for DVIR reports
    backgroundColor: '#FFF5F0', // Light orange background
  },
  incidentNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#F44336', // Red color for incident reports
    backgroundColor: '#FFEBEE', // Light red background
  },
  workOrderNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#4CAF50', // Green color for work orders
    backgroundColor: '#F1F8E9', // Light green background
  },
  serviceScheduleNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#3481BC', // Blue color for service schedules
    backgroundColor: '#E3F2FD', // Light blue background
  },
  hrApplicationNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#015185', // Dark blue color for HR applications
    backgroundColor: '#E8F4FD', // Light blue background
  },
  hrManagerNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#FF9800', // Orange color for HR manager notifications
    backgroundColor: '#FFF3E0', // Light orange background
  },
  highlightedNotification: {
    borderWidth: 3,
    borderColor: '#3481BC', // Blue border for highlighted notification
    backgroundColor: '#F0F8FF', // Light blue background
    elevation: 8,
    shadowColor: '#3481BC',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
  },
  notificationHeader: {
    flexDirection: 'row',
    alignItems: 'flex-start',
  },
  notificationIconContainer: {
    width: 50,
    height: 50,
    borderRadius: 25,
    backgroundColor: '#F0F0F0',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 15,
  },
  notificationIcon: {
    fontSize: 24,
  },
  notificationContent: {
    flex: 1,
    marginRight: 10,
  },
  notificationTitle: {
    fontSize: scale(16),
    fontWeight: '600',
    color: '#333',
    marginBottom: 5,
  },
  unreadTitle: {
    fontWeight: '700',
    color: '#3481BC',
  },
  notificationBody: {
    fontSize: scale(14),
    color: '#666',
    lineHeight: 20,
    marginBottom: 8,
  },
  notificationTime: {
    fontSize: scale(12),
    color: '#999',
  },
  notificationActions: {
    alignItems: 'center',
  },
  unreadIndicator: {
    width: 8,
    height: 8,
    borderRadius: 4,
    backgroundColor: '#3481BC',
    marginBottom: 8,
  },
  deleteButton: {
    padding: 5,
  },
  notificationData: {
    marginTop: 12,
    paddingTop: 12,
    borderTopWidth: 1,
    borderTopColor: '#F0F0F0',
  },
  dataText: {
    fontSize: scale(12),
    color: '#888',
    fontFamily: 'monospace',
    marginBottom: 2,
  },
  emptyState: {
    flex: 1,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 60,
  },
  emptyStateTitle: {
    fontSize: scale(18),
    fontWeight: '600',
    color: '#666',
    marginTop: 20,
    marginBottom: 10,
  },
  emptyStateSubtitle: {
    fontSize: scale(14),
    color: '#999',
    textAlign: 'center',
    paddingHorizontal: 40,
    lineHeight: 20,
  },
  loadingContainer: {
    flex: 1,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 60,
  },
  loadingText: {
    fontSize: scale(14),
    color: '#999',
    marginTop: 15,
  },
  // Recent Notices Styles
  recentNoticesContainer: {
    backgroundColor: 'white',
    paddingVertical: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
  },
  recentNoticesHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 15,
    paddingHorizontal: 20,
  },
  recentNoticesTitle: {
    fontSize: scale(18),
    fontWeight: 'bold',
    color: '#333',
  },
  viewAllButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 20,
    borderWidth: 1,
    borderColor: '#3481BC',
  },
  viewAllButtonText: {
    fontSize: scale(14),
    fontWeight: '600',
    color: '#3481BC',
    marginRight: 4,
  },
  noticesScrollContainer: {
    paddingHorizontal: 20,
  },
  noticeCard: {
    width: 280,
    backgroundColor: '#f8f9fa',
    borderRadius: 12,
    marginRight: 15,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    overflow: 'hidden',
  },
  noticeImage: {
    width: '100%',
    height: 120,
  },
  noticeContent: {
    padding: 15,
  },
  noticeTitle: {
    fontSize: scale(16),
    fontWeight: '600',
    color: '#333',
    marginBottom: 8,
    lineHeight: 20,
  },
  noticeDescription: {
    fontSize: scale(14),
    color: '#666',
    lineHeight: 18,
    marginBottom: 10,
  },
  noticeDate: {
    fontSize: scale(12),
    color: '#999',
    fontWeight: '500',
  },
  noticesLoadingContainer: {
    paddingVertical: 20,
    alignItems: 'center',
  },
  noticesLoadingText: {
    fontSize: scale(14),
    color: '#999',
  },
  noNoticesText: {
    fontSize: scale(14),
    color: '#999',
    textAlign: 'center',
    paddingVertical: 20,
  },
  // View All Notices Button Styles
  viewAllNoticesContainer: {
    backgroundColor: 'white',
    marginTop: 20,
    marginHorizontal: 20,
    borderRadius: 12,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  viewAllNoticesButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 16,
    paddingHorizontal: 20,
    backgroundColor: '#f8f9ff',
    borderWidth: 1,
    borderColor: '#3481BC',
    borderRadius: 12,
  },
  viewAllNoticesText: {
    fontSize: scale(16),
    fontWeight: '600',
    color: '#3481BC',
    marginHorizontal: 12,
  },
  viewButton: {
    paddingHorizontal: 10,
    paddingVertical: 5,
    backgroundColor: '#3481BC',
    borderRadius: 15,
    marginRight: 10,
  },
  viewButtonText: {
    color: 'white',
    fontSize: scale(12),
    fontWeight: '600',
  },
  inspectionDetails: {
    marginTop: 8,
    paddingTop: 8,
    borderTopWidth: 1,
    borderTopColor: '#F0F0F0',
  },
  inspectionDetailText: {
    fontSize: scale(12),
    color: '#666',
    marginBottom: 2,
  },
  dvirDetails: {
    marginTop: 8,
    paddingTop: 8,
    borderTopWidth: 1,
    borderTopColor: '#F0F0F0',
  },
  dvirDetailText: {
    fontSize: scale(12),
    color: '#666',
    marginBottom: 2,
  },
  incidentDetails: {
    marginTop: 8,
    paddingTop: 8,
    borderTopWidth: 1,
    borderTopColor: '#F0F0F0',
  },
  incidentDetailText: {
    fontSize: scale(12),
    color: '#666',
    marginBottom: 2,
  },
  workOrderDetails: {
    marginTop: 8,
    paddingTop: 8,
    borderTopWidth: 1,
    borderTopColor: '#F0F0F0',
  },
  workOrderDetailText: {
    fontSize: scale(12),
    color: '#666',
    marginBottom: 2,
  },
  hrApplicationDetails: {
    marginTop: 8,
    paddingTop: 8,
    borderTopWidth: 1,
    borderTopColor: '#F0F0F0',
  },
  hrApplicationDetailText: {
    fontSize: scale(12),
    color: '#666',
    marginBottom: 2,
  },
  hrManagerDetails: {
    marginTop: 8,
    paddingTop: 8,
    borderTopWidth: 1,
    borderTopColor: '#F0F0F0',
  },
  hrManagerDetailText: {
    fontSize: scale(12),
    color: '#666',
    marginBottom: 2,
  },
  // Sync Modal Styles (same as AdminLoginScreen)
  syncModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: '#fff',
    borderRadius: scale(10),
    padding: scale(20),
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: scale(18),
    marginBottom: scale(10),
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: scale(24),
    height: scale(24),
    marginRight: scale(8),
  },
  syncModalTextadd: {
    fontSize: scale(14),
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  loadingMoreContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 20,
    backgroundColor: 'white',
    borderRadius: 12,
    marginVertical: 10,
  },
  loadingMoreText: {
    fontSize: scale(14),
    color: '#666',
    marginLeft: 10,
  },
  endOfListContainer: {
    alignItems: 'center',
    paddingVertical: 20,
    backgroundColor: 'white',
    borderRadius: 12,
    marginVertical: 10,
  },
  endOfListText: {
    fontSize: scale(14),
    color: '#999',
  },
});

export default HeaderWellNotificaton;
