import React, { useRef, useState } from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  StyleSheet,
  Dimensions,
  StatusBar,
} from 'react-native';
import SignatureCanvas from 'react-native-signature-canvas';
import { useNavigation, useRoute } from '@react-navigation/native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Toast from 'react-native-toast-message';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.33;

// Helper function to show toast messages
const showToast = (type, title, message) => {
  Toast.show({
    type: type,
    text1: title,
    text2: message,
    position: 'bottom',
  });
};

const HrDoceApprSignautre = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { signatureTitle, initialSignature, onSave } = route.params || {};

  const signatureRef = useRef(null);
  const [isSaving, setIsSaving] = useState(false);
  const [isClearing, setIsClearing] = useState(false);

  // Handle saving the signature
  const handleSave = () => {
    if (isSaving || isClearing) return; // Prevent multiple clicks
    setIsSaving(true);
    signatureRef.current.readSignature();
  };

  // Handle clearing the signature
  const handleClear = () => {
    if (isSaving || isClearing) return; // Prevent multiple clicks
    setIsClearing(true);
    signatureRef.current.clearSignature();
    setTimeout(() => {
      showToast('info', 'Signature Cleared', 'The signature area has been cleared.');
      setIsClearing(false);
    }, 500); // Small delay to ensure clear action completes
  };

  // Callback when signature is read (returns base64 data URL)
  const handleSignature = (signature) => {
    setIsSaving(false);
    if (signature) {
      onSave(signature); // Pass the signature back to the parent screen
      showToast('success', 'Success', 'Signature saved successfully.');
      navigation.goBack(); // Go back to the previous screen
    } else {
      showToast('error', 'Error', 'Failed to capture signature.');
    }
  };

  // Handle empty signature check
  const handleEmpty = () => {
    setIsSaving(false);
    showToast('error', 'Error', 'Signature is empty. Please sign before saving.');
  };

  // Signature canvas style
  const webStyle = `
    .m-signature-pad {
      width: ${width * 0.9}px;
      height: ${height * 0.3}px;
      border: 1px solid #ccc;
      border-radius: 8px;
      background-color: #fff;
    }
    .m-signature-pad--body {
      width: 100%;
      height: 100%;
    }
    canvas {
      width: 100%;
      height: 100%;
    }
  `;

  return (
    <View style={styles.container}>
      {/* Header Section */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity
              onPress={() => navigation.goBack()}
              accessibilityLabel="Go back"
              accessibilityRole="button"
            >
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>{signatureTitle || 'Approver Signature'}</Text>
          </View>
        </View>
      </LinearGradient>

      {/* Signature Area */}
      <View style={styles.signatureContainer}>
        <Text style={styles.title}>Please Sign Below</Text>
        <SignatureCanvas
          ref={signatureRef}
          onOK={handleSignature}
          onEmpty={handleEmpty}
          descriptionText="Sign here"
          clearText={isClearing ? 'Clearing...' : 'Clear'}
          confirmText={isSaving ? 'Saving...' : 'Save'}
          webStyle={webStyle}
          penColor="#000"
          backgroundColor="#fff"
          dataURL={initialSignature} // Load initial signature if provided
        />
        <View style={styles.buttonRow}>
          <TouchableOpacity
            style={[styles.button, styles.clearButton, isClearing && styles.disabledButton]}
            onPress={handleClear}
            disabled={isClearing || isSaving}
            accessibilityLabel="Clear signature"
            accessibilityRole="button"
          >
            <Text style={styles.buttonText}>Clear</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.button, styles.saveButton, isSaving && styles.disabledButton]}
            onPress={handleSave}
            disabled={isSaving || isClearing}
            accessibilityLabel="Save signature"
            accessibilityRole="button"
          >
            <Text style={styles.buttonText}>Save</Text>
          </TouchableOpacity>
        </View>
      </View>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f0f0f0',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  signatureContainer: {
    flex: 1,
    alignItems: 'center',
    paddingTop: height * 0.02,
    paddingHorizontal: width * 0.04,
  },
  title: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: height * 0.02,
  },
  buttonRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: width * 0.9,
    marginTop: height * 0.02,
  },
  button: {
    flex: 1,
    paddingVertical: height * 0.015,
    marginHorizontal: width * 0.02,
    borderRadius: width * 0.02,
    alignItems: 'center',
  },
  clearButton: {
    backgroundColor: '#F44336',
  },
  saveButton: {
    backgroundColor: '#4CAF50',
  },
  disabledButton: {
    opacity: 0.6,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
});

export default HrDoceApprSignautre;