import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  ScrollView,
  RefreshControl,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import FontAwesome from 'react-native-vector-icons/FontAwesome'; // Added for status icons
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.33;

// Helper function to show toast messages
const showToast = (type, title, message) => {
  Toast.show({
    type: type,
    text1: title,
    text2: message,
    position: 'bottom',
  });
};

const AdminHrDocument = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { updatedApplication } = route.params || {};

  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [applications, setApplications] = useState([]);
  const [filteredApplications, setFilteredApplications] = useState([]);
  const [searchQuery, setSearchQuery] = useState('');
  const [userUid, setUserUid] = useState('');
  const [selectedTab, setSelectedTab] = useState('All'); // 'All', 'Mine', or application type
  const [isLoading, setIsLoading] = useState(false);

  // Tab definitions
  const tabOptions = [
    { key: 'All', label: 'All' },
    { key: 'leave_application', label: 'Leave' },
    { key: 'funeral_grant_form', label: 'Funeral' },
    { key: 'salary_advance_application', label: 'Salary Advance' },
    { key: 'application_for_appeal', label: 'Appeal' },
    { key: 'grievance_record', label: 'Grievance' },
    { key: 'hospital_attendance_form', label: 'Hospital Attendance' },
  ];

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const [isRotating, setIsRotating] = useState(false);

  // Fetch all HR applications from Firebase
  const fetchApplications = async () => {
    setIsLoading(true);
    try {
      const userUidVal = await AsyncStorage.getItem('userUid');
      setUserUid(userUidVal);
      if (!userUidVal) {
        showToast('error', 'Error', 'Admin not authenticated.');
        return;
      }

      // Fetch from GlobalTeamMemberInformation
      const snapshot = await database()
        .ref(`/GlobalTeamMemberInformation`)
        .once('value');

      const applicationsData = [];
      if (snapshot.exists()) {
        snapshot.forEach((teamMemberSnapshot) => {
          const teamMemberId = teamMemberSnapshot.key;
          const hrModule = teamMemberSnapshot.child('HrmoduleTeam').val();
          if (hrModule) {
            Object.entries(hrModule).forEach(([applicationId, application]) => {
              applicationsData.push({
                teamMemberId,
                applicationId,
                ...application,
                fullName: application.fullName || 'N/A',
                employeeNo: application.employeeNo || 'N/A',
                applicationType: application.applicationType || 'N/A',
                submittedAt: application.submittedAt || 'N/A',
                status: application.status || 'Pending',
                approverName: application.approverName || 'N/A',
                approvalDate: application.approvalDate || 'N/A',
                rejectionReason: application.rejectionReason || '',
              });
            });
          }
        });
      }

      // Fetch from GlobalHrApplications/admins/*/{userUid}/*
      const adminTypes = [
        'leave_application',
        'funeral_grant_form',
        'salary_advance_application',
        'application_for_appeal',
        'grievance_record',
        'hospital_attendance_form',
      ];
      let adminApplications = [];
      for (const type of adminTypes) {
        const adminSnap = await database()
          .ref(`/GlobalHrApplications/admins/${type}/${userUidVal}`)
          .once('value');
        if (adminSnap.exists()) {
          adminSnap.forEach((appSnap) => {
            const app = appSnap.val();
            adminApplications.push({
              ...app,
              applicationId: appSnap.key,
              applicationType: type,
              userUid: userUidVal,
              fullName: app.fullName || 'N/A',
              employeeNo: app.employeeNo || 'N/A',
              submittedAt: app.submittedAt || 'N/A',
              status: app.status || 'Pending',
              approverName: app.approverName || 'N/A',
              approvalDate: app.approvalDate || 'N/A',
              rejectionReason: app.rejectionReason || '',
            });
          });
        }
      }

      // Merge and deduplicate applications (by applicationId + applicationType)
      const allApplications = [...applicationsData, ...adminApplications];
      const uniqueApplications = [];
      const seen = new Set();
      for (const app of allApplications) {
        const key = `${app.applicationId}_${app.applicationType}`;
        if (!seen.has(key)) {
          uniqueApplications.push(app);
          seen.add(key);
        }
      }
      setApplications(uniqueApplications);
      setFilteredApplications(uniqueApplications);
      setIsLoading(false);
    } catch (error) {
      console.error('Error fetching applications:', error);
      showToast('error', 'Error', 'Failed to load applications.');
      setIsLoading(false);
    }
  };

  // Initial fetch
  useEffect(() => {
    fetchApplications();
  }, []);

  // Update applications list when an application is updated
  useEffect(() => {
    if (updatedApplication) {
      setApplications((prevApplications) =>
        prevApplications.map((app) =>
          app.applicationId === updatedApplication.applicationId &&
          app.teamMemberId === updatedApplication.teamMemberId
            ? updatedApplication
            : app
        )
      );
      setFilteredApplications((prevFiltered) =>
        prevFiltered.map((app) =>
          app.applicationId === updatedApplication.applicationId &&
          app.teamMemberId === updatedApplication.teamMemberId
            ? updatedApplication
            : app
        )
      );
    }
  }, [updatedApplication]);

  // Search filter and tab filter
  useEffect(() => {
    let filtered = applications;
    if (selectedTab === 'Mine' && userUid) {
      filtered = filtered.filter(app => app.userUid === userUid);
    } else if (selectedTab !== 'All' && selectedTab !== 'Mine') {
      filtered = filtered.filter(app => app.applicationType === selectedTab);
    }
    filtered = filtered.filter(
      (app) =>
        app.fullName.toLowerCase().includes(searchQuery.toLowerCase()) ||
        app.employeeNo.toLowerCase().includes(searchQuery.toLowerCase()) ||
        app.applicationType.toLowerCase().includes(searchQuery.toLowerCase())
    );
    setFilteredApplications(filtered);
  }, [searchQuery, applications, selectedTab, userUid]);

  // Refresh handler
  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    fetchApplications().then(() => {
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      showToast('success', 'Success', 'Data synced successfully.');
    });
  };

  // Start/stop rotation animation for loader and sync modal
  useEffect(() => {
    if (isLoading || syncModalVisible) {
      if (!isRotating) {
        rotateAnim.setValue(0);
        setIsRotating(true);
        Animated.loop(
          Animated.timing(rotateAnim, {
            toValue: 1,
            duration: 1000,
            useNativeDriver: true,
          })
        ).start();
      }
    } else {
      setIsRotating(false);
      rotateAnim.stopAnimation();
      rotateAnim.setValue(0);
    }
  }, [isLoading, syncModalVisible]);

  // Format application type for display
  const formatApplicationType = (type) => {
    switch (type) {
      case 'leave_application':
        return 'Leave Application';
      case 'funeral_grant_form':
        return 'Funeral Grant Form';
      case 'salary_advance_application':
        return 'Salary Advance Application';
      case 'application_for_appeal':
        return 'Application for Appeal';
      default:
        return 'Unknown';
    }
  };

  // Helper function to format dates
  const formatDate = (dateString) => {
    if (!dateString || dateString === 'N/A') return 'N/A';
    const date = new Date(dateString);
    return isNaN(date.getTime()) ? 'N/A' : date.toLocaleString();
  };

  // Get status icon and color
  const getStatusIcon = (status) => {
    switch (status) {
      case 'Approved':
        return { icon: 'check-circle', color: '#4CAF50' };
      case 'Rejected':
        return { icon: 'times-circle', color: '#F44336' };
      case 'Pending':
      default:
        return { icon: 'clock-o', color: '#FFA500' };
    }
  };

  // Filter applications by status
  const pendingApplications = filteredApplications.filter((app) => app.status === 'Pending');
  const approvedApplications = filteredApplications.filter((app) => app.status === 'Approved' || app.status === 'Rejected');

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>HR Application</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={handleRefresh}>
              <Image
                source={require('../../../../Images/adminhome/equp/cloud_1.png')}
                style={styles.icons12}
              />
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      {/* Search Bar */}
      <View style={styles.searchContainer}>
        <View style={styles.searchWrapper}>
          <Ionicons name="search" size={width * 0.05} color="#888" />
          <TextInput
            style={styles.searchInput}
            placeholder="Search by name, employee no, or application type"
            placeholderTextColor="#888"
            value={searchQuery}
            onChangeText={setSearchQuery}
            accessibilityLabel="Search applications"
          />
        </View>
        {Platform.OS === 'ios' ? (
          <TouchableOpacity
            style={styles.iosApplyNowButton}
            onPress={() => navigation.navigate('AdminHrAppliApply')}
            accessibilityLabel="Apply Now"
          >
            <Text style={styles.iosApplyNowText}>Apply Now</Text>
          </TouchableOpacity>
        ) : (
          <TouchableOpacity
            style={styles.applyNowButton}
            onPress={() => navigation.navigate('AdminHrAppliApply')}
            accessibilityLabel="Apply Now"
          >
            <LinearGradient
              colors={['#025383', '#3481BC']}
              style={styles.applyNowGradient}
              start={{ x: 0, y: 0 }}
              end={{ x: 1, y: 0 }}
            >
              <Text style={styles.applyNowText}>Apply Now</Text>
            </LinearGradient>
          </TouchableOpacity>
        )}
      </View>

      {/* Sticky Horizontal Tab Bar */}
      <View style={styles.stickyTabBarWrapper}>
        <ScrollView
          horizontal
          showsHorizontalScrollIndicator={false}
          style={styles.tabBar}
          contentContainerStyle={{ paddingHorizontal: width * 0.04, alignItems: 'center' }}
        >
          {tabOptions.map((tab) => (
            <TouchableOpacity
              key={tab.key}
              style={[
                styles.tabButton,
                selectedTab === tab.key && styles.tabButtonActive,
              ]}
              onPress={() => setSelectedTab(tab.key)}
            >
              <Text
                style={[
                  styles.tabButtonText,
                  selectedTab === tab.key && styles.tabButtonTextActive,
                ]}
              >
                {tab.label}
              </Text>
            </TouchableOpacity>
          ))}
        </ScrollView>
      </View>

      {/* Applications List */}
      <ScrollView
        contentContainerStyle={[styles.listContent, { paddingTop: width * 0.16 }]}
        refreshControl={
          <RefreshControl refreshing={isRefreshing} onRefresh={handleRefresh} />
        }
      >
        {/* Pending Applications */}
        <View style={styles.applicationSection}>
          <Text style={styles.sectionTitle}>Pending Applications</Text>
          {pendingApplications.length > 0 ? (
            pendingApplications.map((item) => {
              const statusIcon = getStatusIcon(item.status);
              return (
                <TouchableOpacity
                  key={`${item.teamMemberId}_${item.applicationId}`}
                  style={styles.applicationItem}
                  onPress={() =>
                    navigation.navigate('HrDocViewApprove', {
                      application: item,
                    })
                  }
                  accessibilityLabel={`View details for ${item.fullName}'s ${formatApplicationType(item.applicationType)}`}
                >
                  <View style={styles.applicationRow}>
                    <FontAwesome
                      name={statusIcon.icon}
                      size={width * 0.05}
                      color={statusIcon.color}
                      style={styles.statusIcon}
                    />
                    <View style={styles.applicationDetails}>
                      <Text style={styles.applicationText}>
                        {item.fullName} - {formatApplicationType(item.applicationType)}
                      </Text>
                      <Text style={styles.applicationSubText}>Employee No: {item.employeeNo}</Text>
                      <Text style={styles.applicationSubText}>Submitted: {formatDate(item.submittedAt)}</Text>
                    </View>
                  </View>
                </TouchableOpacity>
              );
            })
          ) : (
            <Text style={styles.noDataText}>No pending applications</Text>
          )}

          {/* Approved/Rejected Applications */}
          <Text style={[styles.sectionTitle, { marginTop: width * 0.05 }]}>Approved/Rejected Applications</Text>
          {approvedApplications.length > 0 ? (
            approvedApplications.map((item) => {
              const statusIcon = getStatusIcon(item.status);
              return (
                <TouchableOpacity
                  key={`${item.teamMemberId}_${item.applicationId}`}
                  style={styles.applicationItem}
                  onPress={() =>
                    navigation.navigate('HrDocViewApprove', {
                      application: item,
                    })
                  }
                  accessibilityLabel={`View details for ${item.fullName}'s ${formatApplicationType(item.applicationType)}`}
                >
                  <View style={styles.applicationRow}>
                    <FontAwesome
                      name={statusIcon.icon}
                      size={width * 0.05}
                      color={statusIcon.color}
                      style={styles.statusIcon}
                    />
                    <View style={styles.applicationDetails}>
                      <Text style={styles.applicationText}>
                        {item.fullName} - {formatApplicationType(item.applicationType)}
                      </Text>
                      <Text style={styles.applicationSubText}>Employee No: {item.employeeNo}</Text>
                      <Text style={styles.applicationSubText}>Status: {item.status}</Text>
                      <Text style={styles.applicationSubText}>Approved by: {item.approverName}</Text>
                      <Text style={styles.applicationSubText}>Date: {formatDate(item.approvalDate)}</Text>
                      {item.status === 'Rejected' && (
                        <Text style={styles.applicationSubText}>Reason: {item.rejectionReason}</Text>
                      )}
                    </View>
                  </View>
                </TouchableOpacity>
              );
            })
          ) : (
            <Text style={styles.noDataText}>No approved/rejected applications</Text>
          )}
        </View>

        <View style={{ height: width * 0.1 }} />
      </ScrollView>

      {/* Loader/Sync Modal (unified) */}
      <Modal visible={isLoading || syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
                resizeMode="contain"
              />
              <Text style={styles.syncModalTextadd}>Loading Data from Cloud</Text>
            </View>
          </View>
        </View>
      </Modal>
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f0f0f0',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
    marginLeft: width * 0.02,
  },
  applyNowButton: {
    marginLeft: 10,
    borderRadius: 20,
    overflow: 'hidden',
    alignSelf: 'center',
  },
  applyNowGradient: {
    paddingVertical: 10,
    paddingHorizontal: 28,
    borderRadius: 20,
    alignItems: 'center',
    justifyContent: 'center',
  },
  applyNowText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: width * 0.045,
    letterSpacing: 0.5,
  },
  iosApplyNowButton: {
    backgroundColor: '#025383',
    paddingVertical: 12,
    paddingHorizontal: 24,
    borderRadius: 25,
    marginLeft: 10,
    alignSelf: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    elevation: 3,
  },
  iosApplyNowText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: width * 0.045,
    letterSpacing: 0.5,
    textAlign: 'center',
  },
  listContent: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.3,
  },
  applicationSection: {
    marginBottom: width * 0.05,
  },
  sectionTitle: {
    fontSize: width * 0.05,
    color: '#025383',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: width * 0.05,
  },
  applicationItem: {
    backgroundColor: '#fff',
    borderRadius: width * 0.02,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    marginBottom: width * 0.03,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.05,
    shadowRadius: 2,
    elevation: 1,
  },
  applicationRow: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.03,
  },
  statusIcon: {
    marginRight: width * 0.03,
  },
  applicationDetails: {
    flex: 1,
  },
  applicationText: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#333',
  },
  applicationSubText: {
    fontSize: width * 0.035,
    color: '#666',
    marginTop: width * 0.01,
  },
  noDataText: {
    fontSize: width * 0.04,
    color: '#888',
    textAlign: 'center',
    fontStyle: 'italic',
  },
  tabBar: {
    height: width * 0.1,
    backgroundColor: '#fff',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    paddingVertical: width * 0.01,
    flexDirection: 'row',
  },
  tabButton: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.012,
    borderRadius: 16,
    marginRight: width * 0.025,
    backgroundColor: '#f4f6f8',
    minWidth: width * 0.18,
    alignItems: 'center',
    justifyContent: 'center',
    elevation: 0,
  },
  tabButtonActive: {
    backgroundColor: '#025383',
    shadowColor: '#025383',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    borderWidth: 1.5,
    borderColor: '#025383',
    elevation: 2,
  },
  tabButtonText: {
    color: '#666',
    fontSize: width * 0.04,
    fontWeight: '600',
    textAlign: 'center',
  },
  tabButtonTextActive: {
    color: '#fff',
    fontWeight: '700',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  stickyTabBarWrapper: {
    position: 'absolute',
    top: TOTAL_HEADER_HEIGHT + width * 0.13, // below header and search bar
    left: 0,
    right: 0,
    zIndex: 10,
    backgroundColor: '#fff',
    height: width * 0.16, // 60px approx
    justifyContent: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
});

export default AdminHrDocument;