import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Dimensions,
  SafeAreaView,
  StatusBar,
  ScrollView,
  Platform,
  Modal,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');

const AdminCheckAllApli = () => {
  const navigation = useNavigation();
  const [unapprovedCounts, setUnapprovedCounts] = useState({
    leave_application: 0,
    application_for_appeal: 0,
    funeral_grant_form: 0,
    grievance_record: 0,
    hospital_attendance_form: 0,
    salary_advance_application: 0,
  });
  const [totalCounts, setTotalCounts] = useState({
    leave_application: 0,
    application_for_appeal: 0,
    funeral_grant_form: 0,
    grievance_record: 0,
    hospital_attendance_form: 0,
    salary_advance_application: 0,
  });
  const [isOnline, setIsOnline] = useState(true);
  const [isLoading, setIsLoading] = useState(true);
  const [modalVisible, setModalVisible] = useState(false);
  const [userDetails, setUserDetails] = useState(null);
  const [modalLoading, setModalLoading] = useState(false);
  const [modalError, setModalError] = useState(null);
  const [allowedCountries, setAllowedCountries] = useState([]);
  const [allowedProjects, setAllowedProjects] = useState([]);
  const [hrPermissions, setHrPermissions] = useState([]);

  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  const buttons = [
    {
      title: 'Leave Applications',
      icon: 'event-note', // Represents leave planning
      screen: 'AdminLeaveApli',
      type: 'leave_application',
    },
    {
      title: 'Appeal Applications',
      icon: 'gavel', // Represents legal/appeal process
      screen: 'AdminAllAppealApli',
      type: 'application_for_appeal',
    },
    {
      title: 'Funeral Applications',
      icon: 'sentiment-dissatisfied', // Represents condolence/funeral
      screen: 'AdminAllFuneralApl',
      type: 'funeral_grant_form',
    },
    {
      title: 'Grievance Applications',
      icon: 'report-problem', // Represents complaints/grievances
      screen: 'AdminGrievanceApl',
      type: 'grievance_record',
    },
    {
      title: 'Hospital Attendance',
      icon: 'local-hospital', // Represents medical/hospital
      screen: 'AdminHospitalAtteAppl',
      type: 'hospital_attendance_form',
    },
    {
      title: 'Salary Advance',
      icon: 'account-balance-wallet', // Represents financial transactions
      screen: 'AdminSalaryAdv',
      type: 'salary_advance_application',
    },
    {
      title: 'Apply New',
      icon: 'add-circle-outline', // Represents adding new application
      screen: 'AdminHrAppliApply',
      type: null, // No count for this button
    },
  ];

  const handleLockPress = async () => {
    setModalVisible(true);
    setModalLoading(true);
    setModalError(null);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        setModalError('User ID not found. Please log in again.');
        setModalLoading(false);
        return;
      }
      const userRef = database().ref(`/Globalusers/admins/${userUid}`);
      const snapshot = await userRef.once('value');
      const data = snapshot.val();
      if (!data) {
        setUserDetails(null);
        setModalError('User details not found.');
      } else {
        // Get country and project (handle array or string)
        const country = Array.isArray(data.countries) && data.countries.length > 0
          ? data.countries.join(', ')
          : (data.selectedCountry || data.country || 'N/A');
        const project = Array.isArray(data.projects) && data.projects.length > 0
          ? data.projects.join(', ')
          : (data.selectedProject || data.project || 'N/A');
        // Get role (try data.role, then data.roles[0], else 'N/A')
        let role = 'N/A';
        if (data.role) {
          role = data.role;
        } else if (Array.isArray(data.roles) && data.roles.length > 0) {
          role = data.roles[0];
        }
        // Find hrForm permission in modules array
        let hrFormPermission = 'N/A';
        if (Array.isArray(data.modules)) {
          const hrModule = data.modules.find(m => m.module === 'hrForm');
          if (hrModule && hrModule.permissions) {
            hrFormPermission = hrModule.permissions.length > 0
              ? hrModule.permissions.join(', ')
              : 'No Permissions';
          } else {
            hrFormPermission = 'No Permissions';
          }
        }
        setUserDetails({
          country,
          project,
          role,
          hrFormPermission,
        });
      }
    } catch (err) {
      setModalError('Failed to fetch user details.');
      setUserDetails(null);
    }
    setModalLoading(false);
  };

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected || false);
    });

    return () => unsubscribe();
  }, []);

  // Fetch allowed countries and projects on mount
  useEffect(() => {
    const fetchUserPermissions = async () => {
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) return;
        const userRef = database().ref(`/Globalusers/admins/${userUid}`);
        const snapshot = await userRef.once('value');
        const data = snapshot.val();
        if (data) {
          setAllowedCountries(Array.isArray(data.countries) ? data.countries : (data.countries ? [data.countries] : []));
          setAllowedProjects(Array.isArray(data.projects) ? data.projects : (data.projects ? [data.projects] : []));
          
          // Get HR permissions
          const hrModule = data.modules?.find((mod) => mod.module === 'hrForm');
          const permissions = hrModule?.permissions || [];
          setHrPermissions(permissions);
          
          // Debug log
          console.log('HR Permissions loaded:', permissions);
          console.log('onlyMineView present:', permissions.includes('onlyMineView'));
        }
      } catch (e) {}
    };
    fetchUserPermissions();
  }, []);

  useEffect(() => {
    const fetchUnapprovedCounts = async () => {
      // Wait for permissions to be loaded
      if (hrPermissions.length === 0 && allowedCountries.length === 0 && allowedProjects.length === 0) {
        return; // Don't fetch until permissions are loaded
      }
      
      setIsLoading(true);
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          showToast('error', 'Authentication Error', 'User ID not found. Please log in again.');
          navigation.reset({ index: 0, routes: [{ name: 'Login' }] });
          return;
        }

        const currentUser = auth().currentUser;
        if (!currentUser || currentUser.uid !== userUid) {
          showToast('error', 'Authentication Error', 'Invalid session. Please log in again.');
          await AsyncStorage.multiRemove(['userUid', 'userData']);
          navigation.reset({ index: 0, routes: [{ name: 'Login' }] });
          return;
        }

        if (!isOnline) {
          const cachedApplications = await AsyncStorage.getItem('offlineApplications');
          if (cachedApplications) {
            const parsedApplications = JSON.parse(cachedApplications);
            const counts = {};
            const totalCountsObj = {};
            buttons.forEach((button) => {
              if (button.type) {
                let filteredApps = parsedApplications.filter(
                  (app) => app.applicationType === button.type
                );

                // Apply onlyMineView filter
                if (hrPermissions.includes('onlyMineView')) {
                  console.log(`Filtering ${button.type}: onlyMineView mode`);
                  filteredApps = filteredApps.filter(app => app.userUid === userUid);
                } else {
                  console.log(`Filtering ${button.type}: country/project mode`);
                  // Apply country/project filter
                  filteredApps = filteredApps.filter(app => 
                    (allowedCountries.length === 0 || allowedCountries.includes(app.country)) &&
                    (allowedProjects.length === 0 || allowedProjects.includes(app.project))
                  );
                }

                counts[button.type] = filteredApps.filter(app => app.status !== 'Approved').length;
                totalCountsObj[button.type] = filteredApps.length;
              }
            });
            setUnapprovedCounts((prev) => ({ ...prev, ...counts }));
            setTotalCounts((prev) => ({ ...prev, ...totalCountsObj }));
            showToast('info', 'Offline Mode', 'Loaded cached unapproved application counts.');
          } else {
            showToast('info', 'Offline Mode', 'No cached applications available.');
          }
          setIsLoading(false);
          return;
        }

        for (const button of buttons) {
          if (button.type) {
            const applicationRef = database().ref(`/GlobalHrApplications/admins/${button.type}`);
            applicationRef.on('value', (snapshot) => {
              const data = snapshot.val();
              let total = 0;
              let unapproved = 0;
              if (data) {
                if (hrPermissions.includes('onlyMineView')) {
                  console.log(`Online filtering ${button.type}: onlyMineView mode`);
                  // Only fetch current user's applications
                  const userApps = data[userUid];
                  if (userApps) {
                    Object.values(userApps).forEach(app => {
                      total += 1;
                      if (app.status !== 'Approved') unapproved += 1;
                    });
                  }
                } else {
                  console.log(`Online filtering ${button.type}: country/project mode`);
                  // Fetch all users' applications filtered by country/project
                  Object.values(data).forEach(userApps => {
                    Object.values(userApps).forEach(app => {
                      // Filter by allowed countries and projects (AND logic)
                      const countryOk = allowedCountries.length === 0 || allowedCountries.includes(app.country);
                      const projectOk = allowedProjects.length === 0 || allowedProjects.includes(app.project);
                      if (countryOk && projectOk) {
                        total += 1;
                        if (app.status !== 'Approved') unapproved += 1;
                      }
                    });
                  });
                }
              }
              setTotalCounts(prev => ({ ...prev, [button.type]: total }));
              setUnapprovedCounts(prev => ({ ...prev, [button.type]: unapproved }));
            });
          }
        }
        setIsLoading(false);

        return () => {
          buttons.forEach((button) => {
            if (button.type) {
              database()
                .ref(`/GlobalHrApplications/admins/${button.type}`)
                .off();
            }
          });
        };
      } catch (error) {
        console.error('Error fetching unapproved counts:', error);
        showToast('error', 'Error', 'Failed to load unapproved application counts.');
        const cachedApplications = await AsyncStorage.getItem('offlineApplications');
        if (cachedApplications) {
          const parsedApplications = JSON.parse(cachedApplications);
          const counts = {};
          const totalCountsObj = {};
          buttons.forEach((button) => {
            if (button.type) {
              let filteredApps = parsedApplications.filter(
                (app) => app.applicationType === button.type
              );

              // Apply onlyMineView filter
              if (hrPermissions.includes('onlyMineView')) {
                filteredApps = filteredApps.filter(app => app.userUid === userUid);
              } else {
                // Apply country/project filter
                filteredApps = filteredApps.filter(app => 
                  (allowedCountries.length === 0 || allowedCountries.includes(app.country)) &&
                  (allowedProjects.length === 0 || allowedProjects.includes(app.project))
                );
              }

              counts[button.type] = filteredApps.filter(app => app.status !== 'Approved').length;
              totalCountsObj[button.type] = filteredApps.length;
            }
          });
          setUnapprovedCounts((prev) => ({ ...prev, ...counts }));
          setTotalCounts((prev) => ({ ...prev, ...totalCountsObj }));
          showToast('info', 'Offline Mode', 'Loaded cached unapproved counts due to error.');
        }
        setIsLoading(false);
      }
    };

    fetchUnapprovedCounts();
  }, [isOnline, navigation, allowedCountries, allowedProjects, hrPermissions]);

  return (
    <View style={styles.flex1}>
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.headerGradient}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <StatusBar
          translucent
          backgroundColor="transparent"
          barStyle="light-content"
        />
        <SafeAreaView style={styles.safeAreaHeader}>
          <View style={styles.headerContentFixed}>
            <TouchableOpacity
              style={styles.backButton}
              onPress={() => navigation.goBack()}
              accessibilityLabel="Go back"
              accessibilityRole="button"
            >
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitleFixed}>HR Admin Dashboard</Text>
            <TouchableOpacity
              style={styles.lockButton}
              onPress={handleLockPress}
              accessibilityLabel="Show user details"
              accessibilityRole="button"
            >
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </SafeAreaView>
      </LinearGradient>
      {/* Modal for user details */}
      <Modal
        visible={modalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>User Details</Text>
            {modalLoading ? (
              <ActivityIndicator size="large" color="#015185" />
            ) : modalError ? (
              <Text style={styles.modalError}>{modalError}</Text>
            ) : userDetails ? (
              <View style={styles.modalDetailsBox}>
                <Text style={styles.modalDetail}><Text style={styles.modalLabel}>Country:</Text> {userDetails.country}</Text>
                <Text style={styles.modalDetail}><Text style={styles.modalLabel}>Project:</Text> {userDetails.project}</Text>
                <Text style={styles.modalDetail}><Text style={styles.modalLabel}>Role:</Text> {userDetails.role}</Text>
                <Text style={styles.modalDetail}><Text style={styles.modalLabel}>HR Form Permission:</Text> {String(userDetails.hrFormPermission)}</Text>
              </View>
            ) : null}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setModalVisible(false)}
              accessibilityLabel="Close user details"
              accessibilityRole="button"
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
      <ScrollView contentContainerStyle={styles.scrollContent} bounces={false} showsVerticalScrollIndicator={false}>
        {/* Filter Mode Indicator */}
        <View style={styles.filterModeContainer}>
          <MaterialIcons name="filter-list" size={16} color="#035484" />
          <Text style={styles.filterModeText}>
            {hrPermissions.includes('onlyMineView')
              ? 'Showing only your HR applications'
              : `Filtered by: ${allowedCountries.join(', ') || 'All Countries'} | ${allowedProjects.join(', ') || 'All Projects'}`}
          </Text>
        </View>
        
        <View style={styles.buttonContainer}>
          {buttons.map((button, index) => (
            <View key={index} style={styles.buttonWrapper}>
              <TouchableOpacity
                style={styles.cardButton}
                onPress={() => navigation.navigate(button.screen)}
                accessibilityLabel={`Navigate to ${button.title}`}
                accessibilityRole="button"
                disabled={isLoading}
                activeOpacity={0.85}
              >
                <LinearGradient
                  colors={['#015185', '#3481BC']}
                  style={styles.gradientButton}
                  start={{ x: 0.5, y: 0 }}
                  end={{ x: 0.5, y: 1 }}
                >
                  <View style={styles.buttonContent}>
                    <MaterialIcons
                      name={button.icon}
                      size={width * 0.12}
                      color="#fff"
                      style={styles.buttonIcon}
                    />
                    <Text style={styles.buttonText}>{button.title}</Text>
                    {button.type && (
                      <View>
                        <Text style={styles.totalCount}>
                          {totalCounts[button.type]} Total
                        </Text>
                        {unapprovedCounts[button.type] > 0 && (
                          <Text style={styles.unapprovedCount}>
                            {unapprovedCounts[button.type]} Unapproved
                          </Text>
                        )}
                      </View>
                    )}
                  </View>
                </LinearGradient>
              </TouchableOpacity>
            </View>
          ))}
        </View>
      </ScrollView>
    </View>
  );
};

const styles = StyleSheet.create({
  flex1: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  headerGradient: {
    width: '100%',
    paddingBottom: Platform.OS === 'android' ? StatusBar.currentHeight || 24 : 0,
    // Ensures gradient covers status bar on Android
  },
  safeAreaHeader: {
    backgroundColor: 'transparent',
  },
  headerContentFixed: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'android' ? ((StatusBar.currentHeight || width * 0.4) - 30) : 0,
    paddingBottom: width * 0.03,
    minHeight: Platform.OS === 'android' ? width * 0.01 - 10 : width * 0.01,
  },
  headerTitleFixed: {
    flex: 1,
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#fff',
    textAlign: 'center',
    ...(Platform.OS === 'android' ? { top: 30 } : {}),
  },
  scrollContent: {
    flexGrow: 1,
    paddingBottom: width * 0.04,
  },
  buttonContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'space-between',
    padding: width * 0.04,
  },
  buttonWrapper: {
    width: '48%',
    marginBottom: width * 0.04,
    alignItems: 'center',
  },
  cardButton: {
    width: '100%',
    borderRadius: 16,
    backgroundColor: '#fff',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.12,
    shadowRadius: 6,
    elevation: 3,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    overflow: 'visible',
  },
  gradientButton: {
    borderRadius: 16,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.12,
    shadowRadius: 6,
    elevation: 3,
    paddingVertical: 0,
  },
  buttonContent: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 28,
  },
  buttonIcon: {
    marginBottom: 10,
  },
  buttonText: {
    color: '#fff',
    fontSize: 18,
    fontWeight: '600',
    textAlign: 'center',
    marginTop: 10,
    letterSpacing: 0.2,
  },
  unapprovedCount: {
    fontSize: 14,
    fontWeight: '600',
    color: '#FF6B6B',
    marginTop: 8,
    textAlign: 'center',
  },
  totalCount: {
    fontSize: 14,
    fontWeight: '600',
    color: '#fff',
    marginTop: 4,
    textAlign: 'center',
  },
  backButton:{
    padding: width * 0.02,
    ...(Platform.OS === 'android' ? { top: 30 } : {}),
  },
  lockButton: {
    padding: width * 0.02,
    ...(Platform.OS === 'android' ? { top: 30 } : {}),
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.4)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 24,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 5,
  },
  modalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    marginBottom: 16,
    color: '#015185',
  },
  modalDetailsBox: {
    marginBottom: 16,
    alignSelf: 'stretch',
  },
  modalDetail: {
    fontSize: 16,
    marginBottom: 8,
    color: '#222',
  },
  modalLabel: {
    fontWeight: 'bold',
    color: '#015185',
  },
  modalCloseButton: {
    marginTop: 8,
    backgroundColor: '#015185',
    borderRadius: 8,
    paddingVertical: 8,
    paddingHorizontal: 24,
  },
  modalCloseText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
  modalError: {
    color: '#FF6B6B',
    fontSize: 16,
    marginBottom: 12,
    textAlign: 'center',
  },
  filterModeContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#E0F2F7',
    borderRadius: 12,
    paddingVertical: 8,
    paddingHorizontal: 12,
    marginHorizontal: width * 0.04,
    marginBottom: width * 0.04,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 2,
  },
  filterModeText: {
    fontSize: 14,
    color: '#035484',
    marginLeft: 8,
    fontWeight: '500',
  },
});

export default AdminCheckAllApli;