import React, { useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  ScrollView,
  Dimensions
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.25;

const TABS = ['Today', '7 Days', '30 Days', '3 Months'];

const EquipmentOverview = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const [activeTab, setActiveTab] = useState('Today');

  // Extract equipment data from route params
  const equipment = route.params?.equipment || {};
  const equipmentName = equipment.equipmentName || 'Equipment Overview';
  const equipmentStatus = equipment.status || 'Unknown';

  // Function to calculate date range based on active tab
  const getDateRange = () => {
    const today = new Date();
    let startDate = new Date();
    const endDate = today;

    switch (activeTab) {
      case 'Today':
        startDate = new Date(today);
        break;
      case '7 Days':
        startDate = new Date(today);
        startDate.setDate(today.getDate() - 7);
        break;
      case '30 Days':
        startDate = new Date(today);
        startDate.setDate(today.getDate() - 30);
        break;
      case '3 Months':
        startDate = new Date(today);
        startDate.setMonth(today.getMonth() - 3);
        break;
      default:
        startDate = new Date(today);
    }

    // Format dates as DD-MMM-YYYY
    const formatDate = (date) => {
      const day = date.getDate().toString().padStart(2, '0');
      const month = date.toLocaleString('default', { month: 'short' });
      const year = date.getFullYear();
      return `${day}-${month}-${year}`;
    };

    return `From ${formatDate(startDate)} to ${formatDate(endDate)}`;
  };

  // Function to get status color
  const getStatusColor = (status) => {
    switch (status) {
      case 'Available':
        return '#2ecc71';
      case 'Maintenance Required':
        return '#f1c40f';
      case 'In Maintenance':
        return '#e67e22';
      case 'Breakdown':
        return '#e74c3c';
      case 'Deploy':
        return '#3498db';
      default:
        return '#666';
    }
  };

  // Dummy data for demonstration
  const statData = [
    { label: 'Deployed', value: 0, color: '#3481BC' },
    { label: 'Breakdown', value: 0, color: '#E74C3C' },
    { label: 'Repair Item', value: 0, color: '#F1C40F' },
    { label: 'Replace Item', value: 0, color: '#E74C3C' },
  ];

  const inspectionData = [
    { label: 'Inspection', value: 0, color: '#3481BC' },
    { label: 'Vehicle fault list', value: 0, color: '#E74C3C' },
    { label: 'Corrective Action', value: 0, color: '#2ECC71' },
    { label: 'Daily Reporting', value: 0, color: '#2ECC71' },
  ];

  const maintenanceData = [
    { label: 'Total Workorder', value: 0, color: '#3481BC' },
    { label: 'Open WO', value: 0, color: '#E74C3C' },
    { label: 'In Progress', value: 0, color: '#F1C40F' },
    { label: 'Complete', value: 0, color: '#2ECC71' },
    { label: 'Emergency', value: 0, color: '#E74C3C' },
    { label: 'Service Schedule', value: 0, color: '#3481BC' },
  ];

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()}>
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>{equipmentName}</Text>
          <View style={{ width: 24 }} /> {/* Placeholder for right icon */}
        </View>
      </LinearGradient>

      <ScrollView contentContainerStyle={{ paddingBottom: 30 }}>
        {/* Tabs */}
        <View style={styles.tabContainer}>
          {TABS.map(tab => (
            <TouchableOpacity
              key={tab}
              style={[styles.tab, activeTab === tab ? styles.activeTab : null]}
              onPress={() => setActiveTab(tab)}
            >
              <Text style={[styles.tabText, activeTab === tab ? styles.activeTabText : null]}>
                {tab}
              </Text>
            </TouchableOpacity>
          ))}
        </View>

        {/* Date Range */}
        <View style={styles.dateRangeBox}>
          <Text style={styles.dateRangeText}>{getDateRange()}</Text>
        </View>

        {/* Vehicle Status */}
        <View style={styles.statusRow}>
          <Text style={styles.statusLabel}>Vehicle Status </Text>
          <Text style={[styles.statusValue, { color: getStatusColor(equipmentStatus) }]}>
            {equipmentStatus}
          </Text>
        </View>

        {/* Stat Cards */}
        <View style={styles.statsGrid}>
          {statData.map((item, idx) => (
            <View key={item.label} style={styles.statCard}>
              <Text style={styles.statLabel}>{item.label}</Text>
              <Text style={[styles.statValue, { color: item.color }]}>
                {item.value}
              </Text>
            </View>
          ))}
        </View>

        {/* Inspection & Compliance */}
        <View style={styles.sectionCard}>
          <Text style={styles.sectionTitle}>Inspection & Compliance</Text>
          {inspectionData.map((item, idx) => (
            <View key={item.label} style={styles.sectionRow}>
              <Text style={styles.sectionLabel}>{item.label}</Text>
              <Text style={[styles.sectionValue, { color: item.color }]}>
                {item.value}
              </Text>
            </View>
          ))}
        </View>

        {/* Maintenance & PM Services */}
        <View style={styles.sectionCard}>
          <Text style={styles.sectionTitle}>Maintenance & PM Services</Text>
          {maintenanceData.map((item, idx) => (
            <View key={item.label} style={styles.sectionRow}>
              <Text style={styles.sectionLabel}>{item.label}</Text>
              <Text style={[styles.sectionValue, { color: item.color }]}>
                {item.value}
              </Text>
            </View>
          ))}
        </View>
      </ScrollView>
    </View>
  );
};

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#fff' },
  header: { width: '100%', justifyContent: 'center' },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 15,
    paddingTop: StatusBar.currentHeight || 44,
  },
  headerTitle: {
     color: '#fff',
    fontSize: width * 0.05,
    fontWeight: '400',
    //marginLeft: width * 0.025,
    marginRight: width * 0.4,
  },
  tabContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    paddingHorizontal: 20,
    marginVertical: 10,
  },
  tab: {
    paddingVertical: 10,
    paddingHorizontal: 15,
    borderRadius: 3,
    backgroundColor: '#FFF',
    borderWidth: 1,
    borderColor: '#3481BC',
  },
  activeTab: {
    backgroundColor: '#3481BC',
    borderWidth: 0,
  },
  tabText: {
    color: '#3481BC',
    fontSize: 14,
  },
  activeTabText: {
    color: '#FFF',
  },
  dateRangeBox: {
    backgroundColor: '#F5F7FA',
    padding: 10,
    marginHorizontal: 20,
    borderRadius: 10,
    alignItems: 'center',
    marginBottom: 10,
  },
  dateRangeText: {
    fontSize: 14,
    color: '#3481BC',
  },
  statusRow: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'center',
    marginTop: 10,
    marginBottom: 5,
  },
  statusLabel: {
    color: '#888',
    fontSize: 15,
  },
  statusValue: {
    fontWeight: 'bold',
    fontSize: 15,
  },
  statsGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginHorizontal: 10,
    marginTop: 10,
    marginBottom: 10,
    justifyContent: 'space-between',
  },
  statCard: {
    width: '47%',
    backgroundColor: '#fff',
    borderRadius: 8,
    paddingVertical: 18,
    alignItems: 'center',
    marginBottom: 10,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.07,
    shadowRadius: 2,
    borderWidth: 1,
    borderColor: '#F2F2F2',
  },
  statLabel: {
    color: '#888',
    fontSize: 15,
    marginBottom: 5,
    fontWeight: '500',
  },
  statValue: {
    fontSize: 20,
    fontWeight: 'bold',
  },
  sectionCard: {
    backgroundColor: '#fff',
    borderRadius: 8,
    marginHorizontal: 10,
    marginBottom: 15,
    padding: 12,
    elevation: 1,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.05,
    shadowRadius: 2,
    borderWidth: 1,
    borderColor: '#F2F2F2',
  },
  sectionTitle: {
    fontWeight: 'bold',
    color: '#015185',
    fontSize: 15,
    marginBottom: 8,
  },
  sectionRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 4,
  },
  sectionLabel: {
    color: '#888',
    fontSize: 14,
    fontWeight: '400',
  },
  sectionValue: {
    fontSize: 15,
    fontWeight: 'bold',
  },
});

export default EquipmentOverview;