import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  FlatList,
  Share,
  Modal,
  Alert,
  Animated,
  Platform,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useIsFocused, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import { Dropdown } from 'react-native-element-dropdown';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const DROPDOWN_OPTIONS = [
  {
    label: 'View Report',
    value: 'view',
    icon: <MaterialIcons name="picture-as-pdf" size={22} color="#2196f3" />,
  },
  {
    label: 'Share',
    value: 'share',
    icon: <MaterialIcons name="share" size={22} color="#2196f3" />,
  },
  {
    label: 'Corrective Action',
    value: 'corrective',
    icon: <MaterialIcons name="check-circle" size={22} color="#2196f3" />,
  },
  {
    label: 'Archive Inspection',
    value: 'archive',
    icon: <MaterialIcons name="archive" size={22} color="#2196f3" />,
  },
  {
    label: 'Delete',
    value: 'delete',
    icon: <MaterialIcons name="delete" size={22} color="#e74c3c" />,
  },
];

const DviranddotInHome = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const isFocused = useIsFocused();
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedFilter, setSelectedFilter] = useState('');
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [reports, setReports] = useState([]);
  const [filteredReports, setFilteredReports] = useState([]);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [userModalVisible, setUserModalVisible] = useState(false);
  const [userData, setUserData] = useState({ countries: [], projects: [], dvirPermissions: [] });
  const [userPermissions, setUserPermissions] = useState([]);
  const [userUid, setUserUid] = useState('');
  const reportItemRefs = useRef({});
  const [correctiveActionMap, setCorrectiveActionMap] = useState({});
  const [isLoadingCorrective, setIsLoadingCorrective] = useState(false);
  const [isInitialized, setIsInitialized] = useState(false);
  const isDataLoading = useRef(false);
  const [highlightedReportId, setHighlightedReportId] = useState(null);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Handle notification data from route parameters
  useEffect(() => {
    if (route.params?.notificationData) {
      const notificationData = route.params.notificationData;
      console.log('🔔 DVIR notification data received:', notificationData);
      
      // Extract report ID from notification data
      const reportId = notificationData.data?.reportId || notificationData.requestNumber;
      if (reportId) {
        setHighlightedReportId(reportId);
        console.log('🎯 Highlighting DVIR report:', reportId);
        
        // Clear highlight after 10 seconds
        setTimeout(() => {
          setHighlightedReportId(null);
        }, 10000);
      }
    }
  }, [route.params]);

  // Fetch user permissions and UID
  const fetchUserPerms = async () => {
    try {
      const uid = await AsyncStorage.getItem('userUid');
      if (!uid) {
        console.log('No user UID found in AsyncStorage');
        setUserUid('');
        setUserPermissions([]);
        setUserData({ countries: [], projects: [], dvirPermissions: [] });
        return;
      }
      setUserUid(uid);
      const userRef = database().ref(`/Globalusers/admins/${uid}`);
      const snapshot = await userRef.once('value');
      if (!snapshot.exists()) {
        console.log('User not found in admin database');
        setUserPermissions([]);
        setUserData({ countries: [], projects: [], dvirPermissions: [] });
        return;
      }
      const data = snapshot.val();
      const dvirModule = data?.modules?.find((mod) => mod.module === 'dvirReport') || {};
      setUserPermissions(dvirModule.permissions || []);
      setUserData({
        countries: data.countries || [],
        projects: data.projects || [],
        dvirPermissions: dvirModule.permissions || [],
      });
    } catch (error) {
      console.error('Error fetching user permissions:', error);
      setUserPermissions([]);
      setUserData({ countries: [], projects: [], dvirPermissions: [] });
    }
  };

  // Consolidated data loading function
  const loadAllData = async (showLoader = false) => {
    if (isDataLoading.current) {
      console.log('Data loading already in progress, skipping...');
      return;
    }
    
    isDataLoading.current = true;
    
    try {
      if (showLoader) {
        setSyncModalVisible(true);
        Animated.loop(
          Animated.timing(rotateAnim, {
            toValue: 1,
            duration: 1000,
            useNativeDriver: true,
          })
        ).start();
      }

      // Fetch user permissions first
      await fetchUserPerms();
      
      // Then load reports
      await loadReports();
      
      console.log('All data loaded successfully');
    } catch (error) {
      console.error('Error loading all data:', error);
      Alert.alert('Error', 'Failed to load data. Please try again.');
    } finally {
      isDataLoading.current = false;
      if (showLoader) {
        setTimeout(() => {
          setSyncModalVisible(false);
          rotateAnim.setValue(0);
        }, 1000);
      }
    }
  };

  // Initial data loading
  useEffect(() => {
    const initializeData = async () => {
      if (!isInitialized) {
        await loadAllData(true);
        setIsInitialized(true);
      }
    };

    initializeData();
    
    const unsubscribe = NetInfo.addEventListener((state) => {
      if (state.isConnected && isInitialized) {
        syncOfflineData();
        loadAllData(false); // Refresh without loader for network reconnection
      }
    });
    
    return () => unsubscribe();
  }, [isInitialized]);

  // Focus effect - only refresh if already initialized
  useEffect(() => {
    if (isFocused && isInitialized) {
      console.log('Screen focused, refreshing data...');
      loadAllData(false); // Refresh without loader on focus
    }
  }, [isFocused, isInitialized]);

  // Filter reports based on permissions, country, project, and onlyMineView
  const filterReports = (allReports) => {
    console.log('Filtering reports:', {
      totalReports: allReports.length,
      userUid,
      userDataCountries: userData.countries,
      userDataProjects: userData.projects,
      userPermissions
    });
    
    if (!userUid) {
      console.log('No userUid, returning all reports');
      return allReports; // Return all reports if no userUid
    }
    
    let filtered = allReports;
    
    // Only filter by countries if user has country restrictions
    if (userData.countries && userData.countries.length > 0) {
      const beforeCountryFilter = filtered.length;
      filtered = filtered.filter((r) => r.step1?.country && userData.countries.includes(r.step1.country));
      console.log(`Country filter: ${beforeCountryFilter} -> ${filtered.length} reports`);
    }
    
    // Only filter by projects if user has project restrictions
    if (userData.projects && userData.projects.length > 0) {
      const beforeProjectFilter = filtered.length;
      filtered = filtered.filter((r) => r.step1?.project && userData.projects.includes(r.step1.project));
      console.log(`Project filter: ${beforeProjectFilter} -> ${filtered.length} reports`);
    }
    
    // Only filter by onlyMineView if user has this permission
    if (userPermissions.includes('onlyMineView')) {
      const beforeMineFilter = filtered.length;
      filtered = filtered.filter((r) => {
        const createdBy = r.step1?.createdBy || r.userUid;
        return createdBy === userUid;
      });
      console.log(`OnlyMineView filter: ${beforeMineFilter} -> ${filtered.length} reports`);
    }
    
    console.log('Final filtered reports count:', filtered.length);
    return filtered;
  };

  const loadReports = async () => {
    try {
      const state = await NetInfo.fetch();
      console.log('Network state:', state.isConnected);
      
      let loadedReports = [];
      if (state.isConnected) {
        console.log('Fetching reports from Firebase...');
        const adminsSnapshot = await database().ref('/GlobalDvirReports/admins').once('value');
        const adminsData = adminsSnapshot.val();
        console.log('Admins data keys:', adminsData ? Object.keys(adminsData) : 'No data');
        
        if (adminsData) {
          Object.keys(adminsData).forEach((adminKey) => {
            if (adminKey === 'lastDvirInspectionNumber') return;
            const userReports = adminsData[adminKey]?.inspectionReports;
            console.log(`Admin ${adminKey} has ${userReports ? Object.keys(userReports).length : 0} reports`);
            if (userReports) {
              Object.keys(userReports).forEach((id) => {
                const reportData = userReports[id];
                console.log(`Report ${id} structure:`, {
                  hasStep1: !!reportData.step1,
                  hasStep2: !!reportData.step2,
                  hasStep3: !!reportData.step3,
                  step3Fields: reportData.step3 ? Object.keys(reportData.step3) : 'No step3',
                  createdAt: reportData.step3?.createdAt || reportData.createdAt
                });
                loadedReports.push({
                  id,
                  ...reportData,
                  userUid: adminKey,
                });
              });
            }
          });
        }
        console.log('Total reports loaded from Firebase:', loadedReports.length);
        await AsyncStorage.setItem('inspectionReports', JSON.stringify(loadedReports));
      } else {
        console.log('Loading reports from local storage...');
        const storedReports = await AsyncStorage.getItem('inspectionReports');
        loadedReports = storedReports ? JSON.parse(storedReports) : [];
        console.log('Total reports loaded from local storage:', loadedReports.length);
      }
      
      // Remove duplicates by unique report id
      const uniqueReports = [];
      const seenIds = new Set();
      for (const report of loadedReports) {
        if (!seenIds.has(report.id)) {
          uniqueReports.push(report);
          seenIds.add(report.id);
        }
      }
      console.log('Unique reports after deduplication:', uniqueReports.length);
      
      // Apply filtering
      const filteredReports = filterReports(uniqueReports);
      filteredReports.sort((a, b) => new Date(b.step3?.createdAt || 0) - new Date(a.step3?.createdAt || 0));
      
      // Only update state if reports have actually changed
      setReports(prevReports => {
        const prevIds = new Set(prevReports.map(r => r.id));
        const newIds = new Set(filteredReports.map(r => r.id));
        
        if (prevIds.size !== newIds.size || 
            !filteredReports.every(r => prevIds.has(r.id)) ||
            !prevReports.every(r => newIds.has(r.id))) {
          console.log('Reports changed, updating state');
          setFilteredReports(filteredReports);
          return filteredReports;
        }
        
        console.log('Reports unchanged, skipping state update');
        return prevReports;
      });
      
      console.log('Final reports to display:', filteredReports.length);
      
      if (filteredReports.length === 0) {
        console.log('No reports to display. Sample of loaded reports:', loadedReports.slice(0, 3));
      }
    } catch (error) {
      console.error('Error loading reports:', error);
      Alert.alert('Error', 'Failed to load reports.');
      setReports([]);
    }
  };

  const syncOfflineData = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) return;

      const storedReports = await AsyncStorage.getItem('inspectionReports');
      if (!storedReports) return;
      const localReports = JSON.parse(storedReports);

      for (const report of localReports) {
        const createdBy = report.step1?.createdBy || report.userUid;
        if (createdBy !== userUid) continue;
        const reportRef = database().ref(`/GlobalDvirReports/admins/${userUid}/inspectionReports/${report.id}`);
        const snapshot = await reportRef.once('value');
        if (!snapshot.exists()) {

          await reportRef.set(report);

          console.log(`Synced report ${report.id} to Firebase`);
          
        }
      }
      console.log('Offline data synced to Firebase Realtime Database');
    } catch (error) {
      console.error('Error syncing offline data:', error);
    }
  };

  const handleSearch = (text) => {
    setSearchQuery(text);
  };

  const handleRefresh = () => {
    setIsRefreshing(true);
    setSearchQuery('');
    setSelectedFilter('');
    loadAllData(true);
    setTimeout(() => {
      setIsRefreshing(false);
    }, 2000);
  };

  // Debug function to load all reports without filtering
  const loadAllReportsDebug = async () => {
    try {
      setIsRefreshing(true);
      const state = await NetInfo.fetch();
      let loadedReports = [];
      
      if (state.isConnected) {
        const adminsSnapshot = await database().ref('/GlobalDvirReports/admins').once('value');
        const adminsData = adminsSnapshot.val();
        
        if (adminsData) {
          Object.keys(adminsData).forEach((adminKey) => {
            if (adminKey === 'lastDvirInspectionNumber') return;
            const userReports = adminsData[adminKey]?.inspectionReports;
            if (userReports) {
              Object.keys(userReports).forEach((id) => {
                loadedReports.push({
                  id,
                  ...userReports[id],
                  userUid: adminKey,
                });
              });
            }
          });
        }
      } else {
        const storedReports = await AsyncStorage.getItem('inspectionReports');
        loadedReports = storedReports ? JSON.parse(storedReports) : [];
      }
      
      // Remove duplicates
      const uniqueReports = [];
      const seenIds = new Set();
      for (const report of loadedReports) {
        if (!seenIds.has(report.id)) {
          uniqueReports.push(report);
          seenIds.add(report.id);
        }
      }
      
      // Sort by date
      uniqueReports.sort((a, b) => new Date(b.step3?.createdAt || 0) - new Date(a.step3?.createdAt || 0));
      setReports(uniqueReports);
      console.log('Debug: Loaded all reports without filtering:', uniqueReports.length);
    } catch (error) {
      console.error('Error in debug load:', error);
    } finally {
      setIsRefreshing(false);
    }
  };

  const fetchUserData = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        Alert.alert('Error', 'User not authenticated.');
        return;
      }

      const userRef = database().ref(`/Globalusers/admins/${userUid}`);
      const snapshot = await userRef.once('value');

      if (!snapshot.exists()) {
        throw new Error('User not found in admin database');
      }

      const data = snapshot.val();
      const dvirModule = data.modules?.find((mod) => mod.module === 'dvirReport');
      setUserData({
        countries: data.countries || [],
        projects: data.projects || [],
        dvirPermissions: dvirModule?.permissions || [],
      });
      setUserPermissions(dvirModule?.permissions || []);
      setUserModalVisible(true);
    } catch (error) {
      console.error('Error fetching user data:', error);
      Alert.alert('Error', 'Failed to fetch user data.');
    }
  };

  const handleViewReport = async (report) => {
    if (!report?.step3?.pdfDownloadUrl) {
      Alert.alert('Error', 'No PDF found for this report.');
      return;
    }
    try {
      navigation.navigate('PdfviewPage', {
        pdfUrl: report.step3.pdfDownloadUrl,
        onGoBack: () => navigation.navigate('DviranddotInHome'),
      });
      console.log('Navigated to PdfviewPage with URL:', report.step3.pdfDownloadUrl);
    } catch (error) {
      console.error('Error viewing PDF:', error);
      Alert.alert('Error', `Failed to view PDF: ${error.message}`);
    }
  };

  const handleShare = async (report) => {
    try {
      if (!report.step3?.pdfDownloadUrl) {
        Alert.alert('Error', 'No PDF available to share for this report.');
        return;
      }
      await Share.share({
        url: report.step3.pdfDownloadUrl,
        title: `Inspection Report ${report.step1.requestNumber}`,
        message: `Inspection Report ${report.step1.requestNumber} generated on ${formatDate(report.step3?.createdAt)}`,
      });
      console.log('Report shared successfully');
    } catch (error) {
      console.error('Error sharing report:', error);
      Alert.alert('Error', 'Failed to share report.');
    }
  };

  const handleCorrectiveAction = (report) => {
    // Check if report has faulty items that need corrective action
    let faultCount = 0;
    
    // Check for faulty items in step2 data
    if (Array.isArray(report.step2?.items)) {
      faultCount = report.step2.items.filter(item => 
        item.status && item.status !== '' && item.status !== 'N/A' && 
        (item.status === 'No' || item.status === 'Fail' || item.status === 'faulty')
      ).length;
    } else if (report.step2?.itemStates) {
      faultCount = Object.values(report.step2.itemStates).filter(state => 
        state.status && state.status !== '' && state.status !== 'N/A' && 
        (state.status === 'No' || state.status === 'Fail' || state.status === 'faulty')
      ).length;
    }
    
    // Check if corrective action already exists
    const hasCorrectiveAction = report.correctiveAction && report.correctiveAction.hasCorrectiveAction;
    const correctiveActionData = correctiveActionMap[report.id] || report.correctiveAction;
    
    if (faultCount === 0 && !hasCorrectiveAction && !correctiveActionData) {
      Alert.alert('No Action Required', 'This report has no faulty items that require corrective action.');
      return;
    }
    
    navigation.navigate('DvirCorrectiveActionHeader', { 
      report: { 
        ...report, 
        correctiveAction: correctiveActionData 
      } 
    });
  };

  const handleArchive = () => {
    Alert.alert('Archive', 'This feature is not yet implemented.');
  };

  const handleDelete = async (report) => {
    Alert.alert(
      'Delete Report',
      `Are you sure you want to delete Inspection #${report.step1.requestNumber}?`,
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            try {
              const userUid = await AsyncStorage.getItem('userUid');
              if (!userUid) {
                Alert.alert('Error', 'User not authenticated.');
                return;
              }

              const updatedReports = reports.filter((r) => r.id !== report.id);
              await AsyncStorage.setItem('inspectionReports', JSON.stringify(updatedReports));

              const state = await NetInfo.fetch();
              if (state.isConnected) {
                await database()
                  .ref(`/GlobalDvirReports/admins/${userUid}/inspectionReports/${report.id}`)
                  .remove();
                if (report.step3?.pdfDownloadUrl) {
                  await storage().refFromURL(report.step3.pdfDownloadUrl).delete();
                }
                if (report.step3?.signatureDownloadUrl) {
                  await storage().refFromURL(report.step3.signatureDownloadUrl).delete();
                }
              }
              setReports(updatedReports);
              console.log('Report deleted:', report.step1.requestNumber);
            } catch (error) {
              console.error('Error deleting report:', error);
              Alert.alert('Error', 'Failed to delete report.');
            }
          },
        },
      ]
    );
  };

  const formatDate = (isoString) => {
    if (!isoString) return 'N/A';
    const date = new Date(isoString);
    return date.toLocaleString('en-US', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
      hour12: true,
    }).replace(',', '');
  };

  const handleDropdownSelect = (item, report) => {
    switch (item.value) {
      case 'view':
        handleViewReport(report);
        break;
      case 'share':
        handleShare(report);
        break;
      case 'corrective':
        handleCorrectiveAction(report);
        break;
      case 'archive':
        handleArchive();
        break;
      case 'delete':
        handleDelete(report);
        break;
      default:
        break;
    }
  };

  // Add a function to fetch corrective action data for a report
  const fetchCorrectiveActionData = async (reportId) => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) return null;
      
      // First check the new corrective action path
      const snapshot = await database()
        .ref(`/GlobalDVirCorectiveActionR/${userUid}/${reportId}`)
        .once('value');
      
      if (snapshot.exists()) {
        const data = snapshot.val();
        console.log('Found corrective action data in new path for:', reportId, data);
        return data;
      }
      
      // If not found in new path, check the main report's correctiveAction field
      const mainReportSnapshot = await database()
        .ref(`/GlobalDvirReports/admins/${userUid}/inspectionReports/${reportId}/correctiveAction`)
        .once('value');
      
      if (mainReportSnapshot.exists()) {
        const mainData = mainReportSnapshot.val();
        if (mainData.pdfDownloadUrl || mainData.hasCorrectiveAction) {
          console.log('Found corrective action data in main report for:', reportId, mainData);
          return mainData;
        }
      }
      
      console.log('No corrective action data found for report:', reportId);
      return null;
    } catch (error) {
      console.error('Error fetching corrective action data:', error);
      return null;
    }
  };

  // --- New: Corrective Action Data Map ---
  // Fetch corrective action data for all reports after reports are loaded
  useEffect(() => {
    const fetchAllCorrective = async () => {
      if (reports.length === 0) {
        setCorrectiveActionMap({});
        return;
      }

      // Prevent multiple simultaneous corrective action fetches
      if (isLoadingCorrective) {
        return;
      }

      setIsLoadingCorrective(true);
      const map = {};
      const userUid = await AsyncStorage.getItem('userUid');
      
      if (!userUid) {
        setIsLoadingCorrective(false);
        return;
      }

      console.log('Fetching corrective action data for', reports.length, 'reports...');

      try {
        // Process reports in batches to prevent UI blocking
        const batchSize = 5;
        for (let i = 0; i < reports.length; i += batchSize) {
          const batch = reports.slice(i, i + batchSize);
          
          await Promise.all(
            batch.map(async (item) => {
              // First check if corrective action data is already in the report object
              if (item.correctiveAction && (item.correctiveAction.pdfDownloadUrl || item.correctiveAction.hasCorrectiveAction)) {
                map[item.id] = item.correctiveAction;
                console.log('Found corrective action data in report object for:', item.id, 'PDF URL:', item.correctiveAction.pdfDownloadUrl);
              } else {
                // Fetch from Firebase - check both new path and main report
                const data = await fetchCorrectiveActionData(item.id);
                if (data) {
                  map[item.id] = data;
                  console.log('Fetched corrective action data from Firebase for:', item.id, 'PDF URL:', data.pdfDownloadUrl);
                } else {
                  console.log('No corrective action data found for report:', item.id);
                }
              }
            })
          );
          
          // Update map incrementally to prevent UI freezing
          setCorrectiveActionMap(prevMap => ({ ...prevMap, ...map }));
        }
        
        console.log('Corrective action map updated with', Object.keys(map).length, 'entries');
      } catch (error) {
        console.error('Error fetching all corrective action data:', error);
      } finally {
        setIsLoadingCorrective(false);
      }
    };
    
    // Add a small delay to prevent immediate execution after reports change
    const timeoutId = setTimeout(fetchAllCorrective, 100);
    return () => clearTimeout(timeoutId);
  }, [reports]);

  // Update filtered reports when reports change or filter changes
  useEffect(() => {
    let filtered = reports;
    
    if (searchQuery.trim() !== '') {
      filtered = reports.filter((report) => {
        const country = report.step1?.country?.toLowerCase() || '';
        const inspector = report.step1?.inspector?.toLowerCase() || '';
        const equipment = report.step1?.selectedEquipment?.equipmentName?.toLowerCase() || '';
        const query = searchQuery.toLowerCase();
        return (
          (selectedFilter === 'Location' && country.includes(query)) ||
          (selectedFilter === 'Inspector' && inspector.includes(query)) ||
          (selectedFilter === 'Equipment' && equipment.includes(query)) ||
          (!selectedFilter && (country.includes(query) || inspector.includes(query) || equipment.includes(query)))
        );
      });
    }
    
    // Sort to put highlighted report at the top
    if (highlightedReportId) {
      filtered = filtered.sort((a, b) => {
        const aHighlighted = a.id === highlightedReportId || a.requestNumber === highlightedReportId;
        const bHighlighted = b.id === highlightedReportId || b.requestNumber === highlightedReportId;
        if (aHighlighted && !bHighlighted) return -1;
        if (!aHighlighted && bHighlighted) return 1;
        return 0;
      });
    }
    
    setFilteredReports(filtered);
  }, [reports, searchQuery, selectedFilter, highlightedReportId]);

  // Clear corrective action map when reports change to force re-fetch
  useEffect(() => {
    if (reports.length > 0) {
      setCorrectiveActionMap({});
    }
  }, [reports]);

  // --- REPLACE renderReportItem with Inspection_Report_Home style ---
  const renderReportItem = ({ item, index }) => {
    const step1 = item.step1 || {};
    const step3 = item.step3 || {};
    const date = formatDate(step3.createdAt);
    const correctiveActionData = correctiveActionMap[item.id] || item.correctiveAction;
    const equipmentName = step1.selectedEquipment?.equipmentName || 'N/A';
    let faultCount = 0;
    let hasCorrective = false;
    let hasPdf = false;
    if (correctiveActionData) {
      hasCorrective = true;
      faultCount = correctiveActionData.faultyCount || 0;
      hasPdf = !!correctiveActionData.pdfDownloadUrl;
    } else {
      if (Array.isArray(item.step2?.items)) {
        faultCount = item.step2.items.filter(item =>
          item.status && item.status !== '' && item.status !== 'N/A' &&
          (item.status === 'No' || item.status === 'Fail' || item.status === 'faulty')
        ).length;
      } else if (item.step2?.itemStates) {
        faultCount = Object.values(item.step2.itemStates).filter(state =>
          state.status && state.status !== '' && state.status !== 'N/A' &&
          (state.status === 'No' || state.status === 'Fail' || state.status === 'faulty')
        ).length;
      }
    }
    const correctiveActionText = hasCorrective ? 'Taken' : 'Not taken';
    const correctiveActionColor = hasCorrective ? '#2ecc71' : '#e74c3c';
    const dropdownPosition = index > filteredReports.length - 4 ? 'top' : 'auto';
    const dropdownOptions = DROPDOWN_OPTIONS.filter((opt) => {
      if (opt.value === 'corrective') return userPermissions.includes('edit');
      if (opt.value === 'delete') return userPermissions.includes('delete');
      if (opt.value === 'view' || opt.value === 'share' || opt.value === 'archive') return true;
      return false;
    });
    // Check if this report should be highlighted
    const isHighlighted = highlightedReportId && (item.id === highlightedReportId || item.requestNumber === highlightedReportId);
    
    return (
      <View style={[
        styles.reportItemCard,
        isHighlighted && styles.highlightedReportCard
      ]}>
        <View style={styles.reportHeaderCard}>
          <Text style={styles.reportTitleCard}>{step1.requestNumber || 'N/A'}</Text>
          <View style={styles.headerRightContainer}>
            {isHighlighted && (
              <View style={styles.notifyBadge}>
                <Text style={styles.notifyText}>NOTIFY</Text>
              </View>
            )}
            <TouchableOpacity style={styles.statusButtonCard}>
              <Text style={styles.statusButtonTextCard}>{step3.equipmentStatus || 'N/A'}</Text>
            </TouchableOpacity>
          </View>
        </View>
        <View style={styles.reportDetailsCard}>
          <View style={{ flexDirection: 'row', alignItems: 'stretch' }}>
            <View style={{ flex: 1 }}>
              <View style={styles.detailItemCard}>
                <Text style={styles.detailLabelCard}>Inspector</Text>
                <Text style={[styles.detailValueCard, styles.detailValueLeftCard]}>{step1.inspector || 'N/A'}</Text>
              </View>
              <View style={styles.detailItemCard}>
                <Text style={styles.detailLabelCard}>Location</Text>
                <Text style={[styles.detailValueCard, styles.detailValueLeftCard]}>{step1.country || 'N/A'}</Text>
              </View>
            </View>
            <View style={{ justifyContent: 'center', alignItems: 'center', width: 70 }}>
              <View style={styles.faultBadgeContainerCard}>
                <Text style={styles.faultBadgeTextCard}>{faultCount}</Text>
                <Text style={styles.faultBadgeLabelCard}>Fault</Text>
              </View>
            </View>
          </View>
          <View style={styles.detailItemCard}>
            <Text style={styles.detailLabelCard}>Project</Text>
            <Text style={styles.detailValueCard}>{step1.project || 'N/A'}</Text>
          </View>
          <View style={styles.detailItemCard}>
            <Text style={styles.detailLabelCard}>Equipment</Text>
            <Text style={styles.detailValueCard}>{equipmentName}</Text>
          </View>
          <View style={styles.detailItemCard}>
            <Text style={styles.detailLabelCard}>Corrective Action</Text>
            <Text style={[styles.detailValueCard, { color: hasCorrective ? '#2ecc40' : '#e74c3c' }]}>{correctiveActionText}</Text>
          </View>
          {hasPdf && (
            <View style={styles.detailItemCard}>
              <Text style={styles.detailLabelCard}>Corrective PDF</Text>
              <TouchableOpacity onPress={() => {
                const pdfUrl = correctiveActionData?.pdfDownloadUrl;
                if (!pdfUrl) {
                  Alert.alert('Error', 'No Corrective Action PDF found for this report.');
                  return;
                }
                navigation.navigate('PdfviewPage', {
                  pdfUrl,
                  onGoBack: () => navigation.navigate('DviranddotInHome'),
                });
              }}>
                <Text style={[styles.detailValueCard, { color: '#0078D4', textDecorationLine: 'underline' }]}>View</Text>
              </TouchableOpacity>
            </View>
          )}
          <View style={styles.detailItemCard}>
            <Text style={[styles.detailLabelCard, { color: '#666', marginRight: 10 }]}>Date</Text>
            <View style={styles.dateContainerCard}>
              <Text style={styles.detailValueCard}>{date}</Text>
              <Dropdown
                style={styles.dropdown}
                data={dropdownOptions}
                labelField="label"
                valueField="value"
                value={null}
                onChange={(option) => handleDropdownSelect(option, item)}
                renderRightIcon={() => <MaterialIcons name="more-vert" size={22} color="#333" />}
                renderItem={(option) => (
                  <View style={styles.dropdownItem}>
                    {option.icon}
                    <Text
                      style={[
                        styles.dropdownText,
                        { color: option.value === 'delete' ? '#e74c3c' : '#015185' },
                      ]}
                    >
                      {option.label}
                    </Text>
                  </View>
                )}
                placeholder=""
                showsVerticalScrollIndicator={false}
                dropdownPosition={dropdownPosition}
                containerStyle={styles.dropdownContainer}
                itemTextStyle={styles.dropdownItemText}
                selectedTextStyle={{ display: 'none' }}
              />
            </View>
          </View>
        </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>DVIR Inspection</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={fetchUserData}>
              <MaterialIcons name="security" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity onPress={handleRefresh} style={styles.refreshButton}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud.png')}
                style={styles.refreshIcon}
              />
            </TouchableOpacity>
            <TouchableOpacity style={styles.iconButton} onPress={() => {
              console.log('Current state:', {
                userUid,
                userData,
                userPermissions,
                reportsCount: reports.length,
                filteredReportsCount: filteredReports.length
              });
              Alert.alert('Debug Info', 
                `User UID: ${userUid || 'Not set'}\n` +
                `Total Reports: ${reports.length}\n` +
                `Filtered Reports: ${filteredReports.length}\n` +
                `Countries: ${userData.countries?.length || 0}\n` +
                `Projects: ${userData.projects?.length || 0}\n` +
                `Permissions: ${userPermissions.join(', ') || 'None'}`
              );
            }}>
              <MaterialIcons name="info" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity style={styles.iconButton} onPress={loadAllReportsDebug}>
              <MaterialIcons name="bug-report" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder={`Search${selectedFilter ? ' by ' + selectedFilter : '...'}`}
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity
            style={styles.filterButton}
            onPress={() => setFilterModalVisible(true)}
          >
            <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      <View style={styles.mainContent}>
        {(isRefreshing || isLoadingCorrective) ? (
          <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center', paddingTop: 60 }}>
            <ActivityIndicator size="large" color="#015185" />
            <Text style={{ color: '#015185', marginTop: 12 }}>
              {isRefreshing ? 'Loading reports...' : 'Loading corrective action data...'}
            </Text>
          </View>
        ) : (
          <FlatList
            data={filteredReports}
            keyExtractor={(item) => item.id}
            renderItem={renderReportItem}
            ListEmptyComponent={
              <View style={styles.emptyContainer}>
                <Text style={styles.defaultText}>You don't have any inspection records</Text>
                <Text style={styles.emptySubText}>
                  Start a new inspection to create a report.
                </Text>
              </View>
            }
            contentContainerStyle={styles.listContent}
            refreshing={isRefreshing}
            onRefresh={handleRefresh}
          />
        )}
      </View>

      {userPermissions.includes('create') && (
        <View style={styles.buttonContainer}>
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
            <TouchableOpacity
              style={styles.startButton}
              onPress={() => navigation.navigate('DvirHomeheader')}
            >
              <Ionicons name="add" size={width * 0.06} color="#fff" />
              <Text style={styles.buttonText}>Start New Inspection</Text>
            </TouchableOpacity>
          </LinearGradient>
        </View>
      )}

      <Modal
        visible={filterModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setFilterModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Filter By</Text>
            {['Location', 'Inspector', 'Equipment'].map((filter) => (
              <TouchableOpacity
                key={filter}
                style={styles.modalOption}
                onPress={() => {
                  setSelectedFilter(filter);
                  setFilterModalVisible(false);
                }}
              >
                <Text style={styles.modalOptionText}>{filter}</Text>
              </TouchableOpacity>
            ))}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setFilterModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={userModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setUserModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>User Permissions</Text>
            <View style={styles.userDataContainer}>
              <Text style={styles.userDataLabel}>Countries:</Text>
              <Text style={styles.userDataText}>
                {userData.countries.length > 0 ? userData.countries.join(', ') : 'N/A'}
              </Text>
            </View>
            <View style={styles.userDataContainer}>
              <Text style={styles.userDataLabel}>Projects:</Text>
              <Text style={styles.userDataText}>
                {userData.projects.length > 0 ? userData.projects.join(', ') : 'N/A'}
              </Text>
            </View>
            <View style={styles.userDataContainer}>
              <Text style={styles.userDataLabel}>DVIR Permissions:</Text>
              <Text style={styles.userDataText}>
                {userData.dvirPermissions.length > 0 ? userData.dvirPermissions.join(', ') : 'N/A'}
              </Text>
            </View>
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setUserModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.08,
    height: width * 0.07,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    padding: width * 0.012,
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 20,
    marginTop: 50,
  },
  emptySubText: {
    fontSize: width * 0.04,
    color: '#888',
    textAlign: 'center',
    marginTop: 10,
  },
  reportItem: {
    backgroundColor: '#fff',
    borderRadius: 8,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.08,
    shadowRadius: 2,
    elevation: 2,
    marginTop: 12,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  reportTitle: {
    fontWeight: 'bold',
    fontSize: 16,
    color: '#222',
    marginBottom: 2,
  },
  reportRow: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 12,
    paddingTop: 2,
  },
  labelText: {
    color: '#888',
    fontSize: 13,
    fontWeight: '500',
    width: 70,
  },
  valueText: {
    color: '#222',
    fontSize: 14,
    flex: 1,
  },
  faultCount: {
    fontWeight: 'bold',
    fontSize: 16,
    marginRight: 2,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.025,
    textAlign: 'center',
  },
  modalOption: {
    paddingVertical: width * 0.025,
  },
  modalOptionText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  modalCloseButton: {
    marginTop: width * 0.025,
    alignItems: 'center',
  },
  modalCloseText: {
    fontSize: width * 0.04,
    color: '#0078D4',
    fontWeight: '600',
  },
  userDataContainer: {
    marginBottom: width * 0.025,
  },
  userDataLabel: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#015185',
  },
  userDataText: {
    fontSize: width * 0.035,
    color: '#333',
    marginTop: 4,
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  dropdownContainer: {
    width: 220,
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
    left: '40%',
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  dropdownItemText: {
    fontSize: 16,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  listContent: {
    paddingBottom: 100,
  },
  // --- New/override styles for report card ---
  reportItemCard: {
    marginVertical: width * 0.012,
    backgroundColor: '#F2F2F2',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#eee',
  },
  highlightedReportCard: {
    borderWidth: 3,
    borderColor: '#3481BC',
    backgroundColor: '#F8F9FF',
    marginTop: width * 0.008,
  },
  headerRightContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  notifyBadge: {
    backgroundColor: '#FF6B35',
    paddingHorizontal: width * 0.02,
    paddingVertical: width * 0.008,
    borderRadius: width * 0.008,
    marginRight: width * 0.02,
  },
  notifyText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: 'bold',
  },
  reportHeaderCard: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: width * 0.05,
    borderTopLeftRadius: width * 0.012,
    borderTopRightRadius: width * 0.012,
  },
  reportTitleCard: {
    fontSize: width * 0.045,
    color: '#015185',
    fontWeight: '500',
  },
  statusButtonCard: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.012,
  },
  statusButtonTextCard: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: '500',
  },
  reportDetailsCard: {
    paddingHorizontal: width * 0.05,
  },
  detailItemCard: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.012,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  detailLabelCard: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: width * 0.008,
  },
  detailValueCard: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  detailValueLeftCard: {
    textAlign: 'left',
    width: '60%',
  },
  dateContainerCard: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  faultBadgeContainerCard: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    marginLeft: width * 0.03,
    paddingHorizontal: width * 0.025,
    paddingVertical: width * 0.01,
    minWidth: width * 0.09,
    minHeight: width * 0.07,
  },
  faultBadgeTextCard: {
    color: '#e74c3c',
    fontWeight: 'bold',
    fontSize: width * 0.045,
    textAlign: 'center',
    lineHeight: width * 0.05,
  },
  faultBadgeLabelCard: {
    color: '#e74c3c',
    fontSize: width * 0.03,
    fontWeight: '600',
    textAlign: 'center',
    marginTop: -2,
  },
});

export default DviranddotInHome;