import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  TextInput,
  Dimensions,
  Image,
  Modal,
  Alert,
  Share,
  Platform,
  Linking,
  Animated,
  KeyboardAvoidingView,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import { useNavigation } from '@react-navigation/native';
import NetInfo from '@react-native-community/netinfo';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import { check, request, PERMISSIONS, RESULTS } from 'react-native-permissions';
import Voice from '@react-native-voice/voice';
import { Buffer } from 'buffer';

// Placeholder for DvirPChecklist
const DvirPChecklist = {
  requirements: {
    label: "Requirements",
    items: [
      { id: "req1", label: "I hold a valid driver's licence" },
      { id: "req2", label: "I will obey speed limits and road signs" },
      { id: "req3", label: "I will not overload the vehicle and ensure the load is secure" },
      { id: "req4", label: "I have completed the vehicle pre check list and satisfied all is order for my trip" },
      { id: "req5", label: "I am fit for work – (Not under the influence of alcohol, am well rested, am healthy)" },
      { id: "req6", label: "I have read and understand the Titan driving procedure, National driving rules and cliental rules" },
      { id: "req7", label: "I will ensure every passenger and myself wears a seat belt (No unauthorised passengers)" },
    ],
  },
  hazards: {
    label: "Hazards – Are you aware of:",
    items: [
      { id: "haz1", label: "Pedestrians crossing the road, school children and drunkards" },
      { id: "haz2", label: "Motor bikes and taxis stopping suddenly or pulling in to the road suddenly" },
      { id: "haz3", label: "Oncoming vehicles overtaking on blind rises or corners" },
      { id: "haz4", label: "Heavy trucks or plant taking up width of the road" },
      { id: "haz5", label: "Riots or unrest" },
      { id: "haz6", label: "Gravel/Sandy roads can be slippery – Loss of control" },
      { id: "haz7", label: "Tyre blowouts or punctures – Loss of control" },
    ],
  },
};

const { height, width } = Dimensions.get('window');

// SyncModal Component for Animated Loading
const SyncModal = ({ visible, onClose }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')} // Ensure this path is correct
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>Generating Report</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// ListeningModal Component for Voice Recognition
const ListeningModal = ({ visible, onClose }) => {
  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.listeningModalContainer}>
        <View style={styles.listeningModalContent}>
          <MaterialIcons name="mic" size={40} color="#015185" style={{ alignSelf: 'center', marginBottom: 10 }} />
          <Text style={styles.listeningModalText}>Listening...</Text>
          <TouchableOpacity style={styles.stopButton} onPress={onClose}>
            <Text style={styles.stopButtonText}>Stop</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );
};

const DvirSummeryStep3 = ({
  setActiveStep = () => {},
  signatureStatus = false,
  signatureData = null,
  onSignaturePress = () => {},
}) => {
  const navigation = useNavigation();
  const [equipmentStatus, setEquipmentStatus] = useState(null);
  const [priority, setPriority] = useState(null);
  const [safeToUse, setSafeToUse] = useState(null);
  const [additionalNote, setAdditionalNote] = useState('');
  const [overallCondition, setOverallCondition] = useState(null);
  const [modalVisible, setModalVisible] = useState(false);
  const [faultyItems, setFaultyItems] = useState(0);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [savedPdfPath, setSavedPdfPath] = useState(null);
  const [reportId, setReportId] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isListening, setIsListening] = useState(false);
  const [errors, setErrors] = useState({
    overallCondition: '',
    equipmentStatus: '',
    safeToUse: '',
    signature: '',
  });

  useEffect(() => {
    loadFaultyItems();
    loadPersistedState();
    Voice.onSpeechStart = onSpeechStart;
    Voice.onSpeechEnd = onSpeechEnd;
    Voice.onSpeechResults = onSpeechResults;
    Voice.onSpeechError = onSpeechError;

    return () => {
      Voice.destroy().then(Voice.removeAllListeners);
    };
  }, [signatureStatus, signatureData]);

  // Voice recognition event handlers
  const onSpeechStart = () => {
    console.log('Speech recognition started');
    setIsListening(true);
  };

  const onSpeechEnd = () => {
    console.log('Speech recognition ended');
    setIsListening(false);
  };

  const onSpeechResults = (e) => {
    const result = e.value[0];
    console.log('Speech recognition results:', result);
    setAdditionalNote((prev) => (prev ? prev + ' ' + result : result));
    saveState();
  };

  const onSpeechError = (e) => {
    console.error('Speech recognition error:', e.error);
    setIsListening(false);
    Alert.alert('Error', 'Failed to recognize speech. Please try again.');
  };

  const loadPersistedState = async () => {
    try {
      const persistedState = await AsyncStorage.getItem('Summerystep3State');
      if (persistedState) {
        const state = JSON.parse(persistedState);
        setEquipmentStatus(state.equipmentStatus || null);
        setPriority(state.priority || null);
        setSafeToUse(state.safeToUse || null);
        setAdditionalNote(state.additionalNote || '');
        setOverallCondition(state.overallCondition || null);
        console.log('Loaded persisted state:', state);
      }
    } catch (error) {
      console.error('Error loading persisted state:', error);
      Alert.alert('Error', 'Failed to load persisted state.');
    }
  };

  const saveState = async () => {
    try {
      const state = {
        equipmentStatus,
        priority,
        safeToUse,
        additionalNote,
        overallCondition,
      };
      await AsyncStorage.setItem('Summerystep3State', JSON.stringify(state));
      console.log('Persisted state saved:', state);
    } catch (error) {
      console.error('Error saving persisted state:', error);
    }
  };

  const loadFaultyItems = async () => {
    try {
      const allKeys = await AsyncStorage.getAllKeys();
      const step2Keys = allKeys.filter((k) => k.startsWith('step2-')); // Updated prefix to match Step 2
      if (step2Keys.length > 0) {
        const latestStep2Key = step2Keys.sort().reverse()[0];
        const step2Data = JSON.parse(await AsyncStorage.getItem(latestStep2Key));
        const itemStates = step2Data.itemStates || {};
        const faultyCount = Object.values(itemStates).filter(
          (state) => state.status === 'No' || state.status === 'Fail'
        ).length;
        setFaultyItems(faultyCount);
        console.log('Calculated faulty items:', faultyCount);
      } else {
        setFaultyItems(0);
        console.log('No Step 2 data found for faulty items');
      }
    } catch (error) {
      console.error('Error loading faulty items:', error);
      setFaultyItems(0);
      Alert.alert('Error', 'Failed to load faulty items.');
    }
  };

  const statusOptions = ['Fit to travel', 'Unfit to travel'];
  const priorityOptions = [
    { label: 'Low', image: require('../../../Images/adminhome/inspreport/Priority1.png') },
    { label: 'Medium', image: require('../../../Images/adminhome/inspreport/Priority2.png') },
    { label: 'High', image: require('../../../Images/adminhome/inspreport/Priority3.png') },
    { label: 'Emergency', image: require('../../../Images/adminhome/inspreport/Priority4.png') },
  ];
  const safeOptions = ['Yes', 'No'];
  const conditionOptions = [
    'Proceed with journey',
    'Do not proceed with journey',
    'Report to Supervisor',
  ];

  const handleStatusSelect = (option) => {
    setEquipmentStatus(option);
    if (option !== 'Unfit to travel') {
      setPriority(null);
    }
    setErrors((prev) => ({ ...prev, equipmentStatus: '' }));
    saveState();
    console.log('Driver status selected:', option);
  };

  const handlePrioritySelect = (option) => {
    setPriority(option);
    saveState();
    console.log('Priority selected:', option);
  };

  const handleSafeSelect = (option) => {
    setSafeToUse(option);
    setErrors((prev) => ({ ...prev, safeToUse: '' }));
    saveState();
    console.log('Safe to Use selected:', option);
  };

  const handleConditionSelect = (option) => {
    setOverallCondition(option);
    setModalVisible(false);
    setErrors((prev) => ({ ...prev, overallCondition: '' }));
    saveState();
    console.log('Overall Condition selected:', option);
  };

  const handleAdditionalNoteChange = (text) => {
    setAdditionalNote(text);
    saveState();
    console.log('Additional Note updated:', text);
  };

  const handleSignaturePress = () => {
    saveState();
    setErrors((prev) => ({ ...prev, signature: '' }));
    onSignaturePress();
    console.log('Signature pressed');
  };

  const requestMicrophonePermission = async () => {
    try {
      const permission = Platform.OS === 'android' ? PERMISSIONS.ANDROID.RECORD_AUDIO : PERMISSIONS.IOS.MICROPHONE;
      const result = await check(permission);
      if (result === RESULTS.DENIED) {
        const requestResult = await request(permission);
        if (requestResult !== RESULTS.GRANTED) {
          Alert.alert('Permission Denied', 'Microphone permission is required to use voice input.', [{ text: 'OK' }]);
          return false;
        }
      } else if (result === RESULTS.BLOCKED) {
        Alert.alert(
          'Permission Blocked',
          'Microphone permission is blocked. Please enable it in Settings.',
          [{ text: 'OK', onPress: () => Linking.openSettings() }]
        );
        return false;
      }
      return true;
    } catch (error) {
      console.error('Error requesting microphone permission:', error);
      Alert.alert('Error', 'Failed to request microphone permission.');
      return false;
    }
  };

  const handleVoiceInput = async () => {
    const hasPermission = await requestMicrophonePermission();
    if (!hasPermission) return;

    try {
      if (isListening) {
        await Voice.stop();
        setIsListening(false);
      } else {
        await Voice.start('en-US');
      }
    } catch (error) {
      console.error('Error starting/stopping voice recognition:', error);
      Alert.alert('Error', 'Failed to start voice recognition.');
      setIsListening(false);
    }
  };

  const showPriority = equipmentStatus === 'Unfit to travel';

  const validateFields = () => {
    const newErrors = {
      overallCondition: '',
      equipmentStatus: '',
      safeToUse: '',
      signature: '',
    };
    let hasError = false;

    if (!overallCondition) {
      newErrors.overallCondition = 'Overall Condition is required';
      hasError = true;
    }

    if (!equipmentStatus) {
      newErrors.equipmentStatus = 'Driver status is required';
      hasError = true;
    }

    if (!safeToUse) {
      newErrors.safeToUse = 'Equipment Safe to Use is required';
      hasError = true;
    }

    if (!signatureStatus || !signatureData) {
      newErrors.signature = 'Inspector Signature is required';
      hasError = true;
    }

    setErrors(newErrors);
    return !hasError;
  };

  const requestStoragePermission = async () => {
    try {
      const permissions = Platform.OS === 'android'
        ? Platform.Version >= 33
          ? [PERMISSIONS.ANDROID.READ_MEDIA_IMAGES]
          : [PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE, PERMISSIONS.ANDROID.WRITE_EXTERNAL_STORAGE]
        : [];

      if (permissions.length === 0) {
        return true;
      }

      let allGranted = true;
      for (const permission of permissions) {
        const result = await check(permission);
        if (result === RESULTS.DENIED) {
          const requestResult = await request(permission);
          if (requestResult !== RESULTS.GRANTED) {
            allGranted = false;
            Alert.alert('Permission Denied', 'Storage permission is required to save the report.', [{ text: 'OK' }]);
          }
        } else if (result === RESULTS.BLOCKED) {
          allGranted = false;
          Alert.alert(
            'Permission Blocked',
            'Storage permission is blocked. Please enable it in Settings.',
            [{ text: 'OK', onPress: () => Linking.openSettings() }]
          );
        }
      }
      return allGranted;
    } catch (error) {
      console.error('Error requesting permission:', error);
      Alert.alert('Error', 'Failed to request permissions.');
      return false;
    }
  };

  const uploadFileToFirebaseStorage = async (filePath, storagePath) => {
    try {
      const reference = storage().ref(storagePath);
      await reference.putFile(filePath);
      const downloadUrl = await reference.getDownloadURL();
      console.log('File uploaded to Firebase Storage:', downloadUrl);
      return downloadUrl;
    } catch (error) {
      console.error('Error uploading file to Firebase Storage:', error);
      throw error;
    }
  };

  const uploadSignatureToFirebaseStorage = async (signatureData, reportId) => {
    try {
      if (!signatureData || !signatureData.startsWith('data:image')) {
        throw new Error('Invalid signature data');
      }

      const base64Data = signatureData.split(',')[1];
      const buffer = Buffer.from(base64Data, 'base64');
      const tempPath = `${RNFS.TemporaryDirectoryPath}/signature_${reportId}.png`;
      await RNFS.writeFile(tempPath, buffer.toString('binary'), 'ascii');

      const storagePath = `signatures/signature_${reportId}.png`;
      const downloadUrl = await uploadFileToFirebaseStorage(tempPath, storagePath);

      await RNFS.unlink(tempPath).catch((err) => console.warn('Error deleting temp signature file:', err));
      console.log('Signature uploaded to Firebase Storage:', downloadUrl);
      return downloadUrl;
    } catch (error) {
      console.error('Error uploading signature to Firebase Storage:', error);
      throw error;
    }
  };

  const saveReportToFirebase = async (userUid, report, pdfDownloadUrl, signatureDownloadUrl) => {
    try {
      const state = await NetInfo.fetch();
      if (state.isConnected) {
        // Create the complete report structure that matches what the home page expects
        const optimizedReport = {
          id: report.id,
          dvirNumber: report.dvirNumber,
          step1: {
            requestNumber: report.step1?.requestNumber,
            country: report.step1?.country,
            project: report.step1?.project,
            inspector: report.step1?.inspector,
            selectedEquipment: {
              id: report.step1?.selectedEquipment?.id,
              equipmentName: report.step1?.selectedEquipment?.equipmentName,
              equipmentNumber: report.step1?.selectedEquipment?.equipmentNumber,
              mainCategory: report.step1?.selectedEquipment?.mainCategory,
              model: report.step1?.selectedEquipment?.model,
              meterReading: report.step1?.selectedEquipment?.meterReading,
              meterUnit: report.step1?.selectedEquipment?.meterUnit,
              status: report.step1?.selectedEquipment?.status,
            },
            meterReading: report.step1?.meterReading,
            currentMeterReading: report.step1?.currentMeterReading,
            inspectionTypes: report.step1?.inspectionTypes,
          },
          step2: {
            items: report.step2?.items?.map(item => ({
              section: item.section,
              label: item.label,
              status: item.status,
              note: item.note ? item.note.substring(0, 500) : '', // Limit note length for Android
            })) || [],
            checklistId: report.step2?.checklistId,
          },
          step3: {
            faultyItems: report.step3?.faultyItems || 0,
            overallCondition: report.step3?.overallCondition,
            equipmentStatus: report.step3?.equipmentStatus,
            priority: report.step3?.priority,
            safeToUse: report.step3?.safeToUse,
            additionalNote: report.step3?.additionalNote ? report.step3.additionalNote.substring(0, 1000) : '', // Limit note length
            signatureStatus: report.step3?.signatureStatus || false,
            signatureData: report.step3?.signatureData || null,
            pdfDownloadUrl,
            signatureDownloadUrl: signatureDownloadUrl || null,
            createdAt: report.step3?.createdAt || new Date().toISOString(),
            timestamp: report.step3?.timestamp || new Date().getTime(),
            dvirNumber: report.step3?.dvirNumber || report.dvirNumber,
          },
          createdAt: report.createdAt || new Date().toISOString(),
          updatedAt: new Date().toISOString(),
        };

        // Check data size for Android to prevent CursorWindow issues
        const reportSize = JSON.stringify(optimizedReport).length;
        console.log('Report data size:', reportSize, 'bytes');
        
        if (Platform.OS === 'android' && reportSize > 1000000) { // 1MB limit
          console.warn('Report data too large for Android, further optimizing...');
          // Further reduce data size
          optimizedReport.step2.items = optimizedReport.step2.items.map(item => ({
            section: item.section,
            label: item.label.substring(0, 100), // Further limit label length
            status: item.status,
            note: item.note ? item.note.substring(0, 200) : '', // Further limit note length
          }));
        }

        const reportRef = database().ref(`GlobalDvirReports/admins/${userUid}/inspectionReports/${report.id}`);
        
        // Save the complete report structure
        await reportRef.set(optimizedReport);
        console.log('Report saved to Firebase Realtime Database:', report.id);
      }

      // Save full report to AsyncStorage for offline access (with size limits)
      const existingReports = JSON.parse(await AsyncStorage.getItem('inspectionReports') || '[]');
      const updatedReports = existingReports.filter((r) => r.id !== report.id);
      
      // Limit the number of stored reports to prevent storage issues
      const maxStoredReports = 50;
      if (updatedReports.length >= maxStoredReports) {
        updatedReports.splice(0, updatedReports.length - maxStoredReports + 1);
      }
      
      updatedReports.push(report);
      await AsyncStorage.setItem('inspectionReports', JSON.stringify(updatedReports));
      console.log('Report saved to AsyncStorage:', report.id);
    } catch (error) {
      console.error('Error saving report to Firebase:', error);
      throw error;
    }
  };

  const generatePDFWithHtml = async (report) => {
    try {
      console.log('Generating PDF for report:', report.id);
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const step3 = report.step3 || {};

      let checklistRows = '';
      // Use the new step2.items array
      const items = Array.isArray(step2.items) ? step2.items : [];
      let lastSection = null;
      items.forEach((item, idx) => {
        if (!item.status || item.status === '' || item.status === 'N/A') return; // Only show selected/checked
        if (item.section !== lastSection) {
          checklistRows += `
            <tr style="background-color: #e0e0e0;">
              <td colspan="7"><strong>${item.section}</strong></td>
            </tr>
          `;
          lastSection = item.section;
        }
        // Determine class and checkmarks
        const isYes = item.status.toLowerCase() === 'yes' || item.status.toLowerCase() === 'fixed' || item.status.toLowerCase() === 'pass';
        const isNo = item.status.toLowerCase() === 'no' || item.status.toLowerCase() === 'not fixed' || item.status.toLowerCase() === 'fail';
        const isNA = item.status.toLowerCase() === 'na' || item.status.toLowerCase() === 'no need';
        checklistRows += `
          <tr>
            <td>${item.label}</td>
            <td>A</td>
            <td>${isYes ? '✓' : ''}</td>
            <td>${isNo ? '✓' : ''}</td>
            <td>${isYes ? '✓' : ''}</td>
            <td>${isNo ? '✓' : ''}</td>
            <td>${isNA ? 'N/A' : (item.note || '')}</td>
          </tr>
        `;
      });
      if (!checklistRows) {
        checklistRows = `
          <tr>
            <td colspan="7">No inspection items selected</td>
          </tr>
        `;
      }

      // Create signature HTML - use base64 data directly if available, otherwise use download URL
      let signatureHtml = '';
      if (step3.signatureData && step3.signatureData.startsWith('data:image')) {
        // Use the base64 signature data directly
        signatureHtml = `
          <div class="signature-container">
            <h4>Inspector Signature:</h4>
            <img src="${step3.signatureData}" class="signature-image" alt="Signature" style="border: 1px solid #ccc; padding: 5px;" />
          </div>
        `;
      } else if (step3.signatureDownloadUrl) {
        // Use the Firebase Storage URL
        signatureHtml = `
          <div class="signature-container">
            <h4>Inspector Signature:</h4>
            <img src="${step3.signatureDownloadUrl}" class="signature-image" alt="Signature" style="border: 1px solid #ccc; padding: 5px;" />
          </div>
        `;
      } else {
        signatureHtml = `
          <div class="signature-container">
            <h4>Inspector Signature:</h4>
            <p><strong>Signature:</strong> Not Provided</p>
          </div>
        `;
      }

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8" />
          <meta name="viewport" content="width=device-width, initial-scale=1.0" />
          <title>Daily DD Rig Pre-Start Checklist</title>
          <style>
            body { font-family: Arial, sans-serif; padding: 20px; font-size: 12px; }
            .header, .section { border: 1px solid #000; padding: 10px; margin-bottom: 10px; }
            .header-table, .checklist-table { width: 100%; border-collapse: collapse; margin-bottom: 10px; }
            .header-table td, .checklist-table th, .checklist-table td { border: 1px solid #000; padding: 5px; text-align: left; }
            .title { text-align: center; font-weight: bold; font-size: 16px; margin: 20px 0; }
            .notes { font-size: 10px; margin-bottom: 10px; }
            .checklist-table th { background-color: #ddd; font-weight: bold; }
            .signature-container { margin-top: 30px; text-align: center; border-top: 2px solid #000; padding-top: 20px; }
            .signature-image { width: 200px; height: 100px; object-fit: contain; }
            .summary-section { margin-top: 20px; }
            .summary-section p { margin: 5px 0; }
            .signature-container h4 { margin-bottom: 10px; color: #015185; }
          </style>
        </head>
        <body>
          <div class="header">
            <table class="header-table">
              <tr>
                <td rowspan="5"><img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo" width="100"></td>
                <td><strong>Document Name:</strong> HSE-FOR-8.1.1 DD Prestart Check</td>
              </tr>
              <tr><td><strong>Document ID:</strong> HSE-028</td></tr>
              <tr><td><strong>Version Date:</strong> 05/08/2021</td></tr>
              <tr><td><strong>Version:</strong> 2</td></tr>
              <tr><td><strong>Approved:</strong> HSM</td></tr>
            </table>
          </div>

          <div class="title">DAILY DD RIG PRE-START CHECKLIST</div>
          <div class="section">
            <strong>NO:</strong> ${step1.requestNumber || '0001'}<br />
            <strong>DATE:</strong> ${new Date().toLocaleDateString()}<br />
            <strong>RIG:</strong> ${step1.selectedEquipment?.equipmentName || 'N/A'}<br />
            <strong>ODOMETER READING:</strong> ${step1.meterReading || 'N/A'} ${step1.selectedEquipment?.meterUnit || 'Miles'}<br />
            <strong>FUEL LEVEL:</strong> N/A
          </div>

          <div class="section notes">
            1. ALL DEFECTS UNDER "A" CLASSIFICATION TO BE ATTENDED IMMEDIATELY<br />
            2. ALL DEFECTS UNDER "B" CLASSIFICATION TO BE ATTENDED WITHIN 48 HOURS<br />
            3. ALL DEFECTS UNDER "C" CLASSIFICATION TO BE ATTENDED AS SOON AS POSSIBLE
          </div>

          <table class="checklist-table">
            <tr>
              <th rowspan="2">Description</th>
              <th rowspan="2">Class</th>
              <th colspan="2">In Order (✓)</th>
              <th colspan="2">Defective (✓)</th>
              <th rowspan="2">Comment</th>
            </tr>
            <tr>
              <th>Yes</th>
              <th>No</th>
              <th>Yes</th>
              <th>No</th>
            </tr>
            ${checklistRows}
          </table>

          <div class="summary-section">
            <h3>Summary</h3>
            <p><strong>Country:</strong> ${step1.country || 'N/A'}</p>
            <p><strong>Project:</strong> ${step1.project || 'N/A'}</p>
            <p><strong>Inspector:</strong> ${step1.inspector || 'N/A'}</p>
            <p><strong>Faulty Items:</strong> ${faultyItems || '0'}</p>
            <p><strong>Overall Condition:</strong> ${step3.overallCondition || 'N/A'}</p>
            <p><strong>Driver status:</strong> ${step3.equipmentStatus || 'N/A'}</p>
            ${step3.priority ? `<p><strong>Priority:</strong> ${step3.priority}</p>` : ''}
            <p><strong>Safe to Use:</strong> ${step3.safeToUse || 'N/A'}</p>
            <p><strong>Additional Note:</strong> ${step3.additionalNote || 'N/A'}</p>
          </div>

          ${signatureHtml}
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: `Daily_DD_Rig_Prestart_Checklist_${report.id}`,
        directory: Platform.OS === 'ios' ? 'Documents' : 'Download',
        width: 595,
        height: 842,
        padding: 24,
        base64: false,
        bgColor: '#FFFFFF',
      };

      const file = await RNHTMLtoPDF.convert(options);
      console.log('PDF generated at:', file.filePath);
      
      // Verify file exists on Android
      if (Platform.OS === 'android') {
        const fileExists = await RNFS.exists(file.filePath);
        if (!fileExists) {
          throw new Error('PDF file was not created successfully on Android');
        }
      }
      
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      if (Platform.OS === 'android') {
        // Try alternative directory for Android
        try {
          const androidOptions = {
            ...options,
            directory: 'Documents',
          };
          const file = await RNHTMLtoPDF.convert(androidOptions);
          console.log('PDF generated at (Android fallback):', file.filePath);
          return file.filePath;
        } catch (fallbackError) {
          console.error('Android fallback PDF generation failed:', fallbackError);
          throw new Error(`PDF generation failed on Android: ${fallbackError.message}`);
        }
      }
      throw error;
    }
  };

  const handleViewPdf = async () => {
    if (!reportId) {
      Alert.alert('Error', 'Report ID not available.');
      return;
    }

    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('LoginScreens');
        return;
      }

      const reportRef = database().ref(`GlobalDvirReports/admins/${userUid}/inspectionReports/${reportId}`);
      const snapshot = await reportRef.once('value');
      const reportData = snapshot.val();

      if (!reportData) {
        Alert.alert('Error', 'Report not found in the database.');
        return;
      }

      const pdfUrl = reportData.step3?.pdfDownloadUrl;
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.');
        return;
      }

      // Navigate to PdfviewPage
      navigation.replace('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.replace('DviranddotInHome'),
      });
      console.log('Navigated to PdfviewPage with URL:', pdfUrl);
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      Alert.alert('Error', 'Could not fetch the PDF URL. Please try again.');
    } finally {
      setShowSuccessModal(false);
    }
  };

  const handleGenerateReport = async () => {
    if (!validateFields()) {
      return;
    }

    const hasPermission = await requestStoragePermission();
    if (!hasPermission) return;

    setSyncModalVisible(true);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('LoginScreens');
        setSyncModalVisible(false);
        return;
      }

      // Get latest Step 1 and Step 2 data
      const allKeys = await AsyncStorage.getAllKeys();
      const step1Keys = allKeys.filter((k) => k.startsWith('step1-'));
      const step2Keys = allKeys.filter((k) => k.startsWith('step2-'));
      if (!step1Keys.length || !step2Keys.length) {
        Alert.alert('Error', 'Data from previous steps is missing. Please complete all steps.');
        setSyncModalVisible(false);
        return;
      }
      const lastStep1Key = step1Keys.sort().reverse()[0];
      const lastStep2Key = step2Keys.sort().reverse()[0];
      const step1 = JSON.parse(await AsyncStorage.getItem(lastStep1Key)) || {};
      const step2Raw = JSON.parse(await AsyncStorage.getItem(lastStep2Key)) || {};

      console.log('Step 1 Data Retrieved:', step1);
      console.log('Step 2 Data Retrieved:', step2Raw);

      if (!step1.country || !step1.project || !step1.inspector || !step2Raw.itemStates) {
        Alert.alert('Error', 'Incomplete data from previous steps. Please complete all steps.');
        setSyncModalVisible(false);
        return;
      }

      // --- Build detailed Step 2 items array ---
      const itemStates = step2Raw.itemStates || {};
      const checklistId = step2Raw.checklistId || null;
      const timestamp = step2Raw.timestamp || null;
      const checklists = JSON.parse(await AsyncStorage.getItem('checklists') || '[]');
      const customChecklist = checklists.find((c) => c.id === checklistId);
      const step2Items = [];

      // Helper to push item to step2Items
      const pushItem = (section, itemId, label, status, note) => {
        step2Items.push({ section, itemId, label, status, note });
      };

      // Requirements
      DvirPChecklist.requirements.items.forEach((item) => {
        const key = `requirements_${item.id}_Status`;
        const state = itemStates[key] || {};
        pushItem(DvirPChecklist.requirements.label, item.id, item.label, state.status || '', state.note || '');
      });
      // Hazards
      DvirPChecklist.hazards.items.forEach((item) => {
        const key = `hazards_${item.id}_Status`;
        const state = itemStates[key] || {};
        pushItem(DvirPChecklist.hazards.label, item.id, item.label, state.status || '', state.note || '');
      });
      // Custom checklist sections (if any)
      if (customChecklist && Array.isArray(customChecklist.items)) {
        customChecklist.items.forEach((section) => {
          const sectionLabel = section.text || section.label || section.id || 'Custom Section';
          if (Array.isArray(section.subItems)) {
            section.subItems.forEach((item) => {
              const key = `${section.key || section.id}_${item.id}_${item.itemName}`;
              const state = itemStates[key] || {};
              pushItem(sectionLabel, item.id, item.itemName || 'Unknown', state.status || '', state.note || '');
            });
          }
        });
      }
      // --- End build detailed Step 2 items array ---

      // --- Increment the global DVIR number here ---
      let dvirNumber = '';
      const lastNumberRef = database().ref('GlobalDvirReports/admins/lastDvirInspectionNumber');
      const transactionResult = await lastNumberRef.transaction((current) => (current || 0) + 1);
      const newDvirNumber = transactionResult.committed ? transactionResult.snapshot.val() : (transactionResult.snapshot.val() || 1);
      dvirNumber = `DVIR-${String(newDvirNumber).padStart(3, '0')}`;

      const reportId = Date.now().toString(); // Generate a unique report ID
      setReportId(reportId);

      // Save Step 3 data
      const step3Data = {
        faultyItems,
        overallCondition,
        equipmentStatus,
        priority,
        safeToUse,
        additionalNote,
        signatureStatus,
        signatureData,
        createdAt: new Date().toISOString(),
        timestamp: new Date().getTime(),
        dvirNumber, // Store the generated DVIR number in step3
      };
      
      console.log('Step 3 data created:', step3Data);

      let signatureDownloadUrl = null;
      if (signatureStatus && signatureData) {
        signatureDownloadUrl = await uploadSignatureToFirebaseStorage(signatureData, reportId);
      }

      // Combine all data
      const step2 = {
        items: step2Items,
        checklistId,
        timestamp,
      };
      const report = {
        id: reportId,
        dvirNumber, // Store the generated DVIR number at the top level
        step1,
        step2,
        step3: step3Data,
        createdAt: new Date().toISOString(),
      };
      
      console.log('Final report structure before saving:', {
        id: report.id,
        dvirNumber: report.dvirNumber,
        hasStep1: !!report.step1,
        hasStep2: !!report.step2,
        hasStep3: !!report.step3,
        step3Fields: report.step3 ? Object.keys(report.step3) : 'No step3',
        createdAt: report.createdAt
      });

      // Generate PDF
      const pdfPath = await generatePDFWithHtml(report);
      const storagePath = `inspectionReports/Daily_DD_Rig_Prestart_Checklist_${reportId}.pdf`;
      const pdfDownloadUrl = await uploadFileToFirebaseStorage(pdfPath, storagePath);

      // Save report to Firebase and AsyncStorage
      try {
        await saveReportToFirebase(userUid, report, pdfDownloadUrl, signatureDownloadUrl);
        console.log('Report saved successfully');
      } catch (saveError) {
        console.error('Error saving report to database:', saveError);
        // Continue with PDF sharing even if database save fails
        Alert.alert(
          'Warning', 
          'Report generated but failed to save to database. PDF is still available.',
          [{ text: 'OK' }]
        );
      }

      // Share PDF
      try {
        await Share.share({
          url: `file://${pdfPath}`,
          title: `Inspection Report ${dvirNumber}`,
          message: `Inspection Report ${dvirNumber} generated on ${new Date().toLocaleDateString()}`,
        });
        console.log('PDF shared successfully');
      } catch (shareError) {
        console.error('Error sharing PDF:', shareError);
        // PDF is still generated, just sharing failed
        Alert.alert(
          'Info', 
          'Report generated successfully. You can find it in your device storage.',
          [{ text: 'OK' }]
        );
      }

      // Clean up temporary file
      await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err));

      // Clear temporary data
      await AsyncStorage.removeItem('Summerystep3State');
      await AsyncStorage.removeItem('selectedChecklistIds');
      // Remove all step1-* and step2-* keys
      const allKeysCleanup = await AsyncStorage.getAllKeys();
      const step1KeysCleanup = allKeysCleanup.filter((k) => k.startsWith('step1-'));
      const step2KeysCleanup = allKeysCleanup.filter((k) => k.startsWith('step2-'));
      await AsyncStorage.multiRemove([...step1KeysCleanup, ...step2KeysCleanup]);
      console.log('Cleared all step1 and step2 AsyncStorage data');

      setSavedPdfPath(pdfPath);
      setShowSuccessModal(true);
    } catch (error) {
      console.error('Error generating report:', error);
      Alert.alert('Error', `Failed to generate report: ${error.message}`);
    } finally {
      setSyncModalVisible(false);
    }
  };

  return (
    <KeyboardAvoidingView
      style={styles.container}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <Modal
        animationType="fade"
        transparent={true}
        visible={modalVisible}
        onRequestClose={() => setModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>
              Overall Condition <Text style={styles.requiredAsterisk}>*</Text>
            </Text>
            {conditionOptions.map((option) => (
              <View key={option}>
                <TouchableOpacity
                  style={styles.modalRow}
                  onPress={() => handleConditionSelect(option)}
                >
                  <Text style={styles.modalOptionText}>{option}</Text>
                  <View
                    style={[
                      styles.radioCircle,
                      overallCondition === option && styles.radioCircleSelected,
                    ]}
                  >
                    {overallCondition === option && <View style={styles.radioDot} />}
                  </View>
                </TouchableOpacity>
                <View style={styles.modalDivider} />
              </View>
            ))}
          </View>
        </View>
      </Modal>

      <Modal
        animationType="fade"
        transparent={true}
        visible={showSuccessModal}
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.popupModalContent}>
            <Text style={styles.popupTitle}>Success</Text>
            <MaterialIcons name="check" size={44} color="#015185" style={{ alignSelf: 'center', marginVertical: 8 }} />
            <Text style={styles.popupSubtitle}>Inspection Report Generated and Saved</Text>
            <View style={{ flexDirection: 'row', justifyContent: 'center', marginTop: 18 }}>
              <TouchableOpacity
                style={styles.popupButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.reset({
                    index: 1,
                    routes: [
                      { name: 'MainApp' },
                      { name: 'DviranddotInHome' },
                    ],
                  });
                }}
              >
                <Text style={styles.popupButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.popupButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.popupButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <ScrollView
        style={styles.mainContent}
        contentContainerStyle={styles.scrollContent}
        keyboardShouldPersistTaps="handled"
      >
        <View style={styles.rowContainer}>
          <Text style={styles.labelTextFault}>Faulty Items</Text>
          <Text style={styles.valueTextFault}>{faultyItems}</Text>
        </View>

        <View style={styles.rowContainer}>
          <Text style={styles.labelTextFault}>
            Overall Condition <Text style={styles.requiredAsterisk}>*</Text>
          </Text>
          <View style={styles.buttonContainer}>
            <TouchableOpacity style={styles.selectButton} onPress={() => setModalVisible(true)}>
              <Text style={styles.buttonText}>{overallCondition || 'Select'}</Text>
              <MaterialIcons name="chevron-right" size={20} color="#fff" style={styles.iconRight} />
            </TouchableOpacity>
          </View>
        </View>
        {errors.overallCondition ? (
          <Text style={styles.errorText}>{errors.overallCondition}</Text>
        ) : null}

        <Text style={styles.sectionLabelequip}>
          Driver status <Text style={styles.requiredAsterisk}>*</Text>
        </Text>
        <View style={styles.optionsRow}>
          {statusOptions.map((option) => {
            const isSelected = equipmentStatus === option;
            return (
              <TouchableOpacity
                key={option}
                style={[styles.statusOptionButton, isSelected && styles.statusOptionButtonSelected]}
                onPress={() => handleStatusSelect(option)}
              >
                <Text style={isSelected ? styles.optionTextSelected : styles.optionText}>{option}</Text>
              </TouchableOpacity>
            );
          })}
        </View>
        {errors.equipmentStatus ? (
          <Text style={styles.errorText}>{errors.equipmentStatus}</Text>
        ) : null}

        {showPriority && (
          <>
            <Text style={styles.sectionLabelequip}>Priority</Text>
            <View style={styles.priorityContainer}>
              {priorityOptions.map((option) => {
                const isSelected = priority === option.label;
                return (
                  <View key={option.label} style={styles.priorityItem}>
                    <TouchableOpacity
                      style={[
                        styles.priorityButton,
                        isSelected ? styles.priorityButtonSelected : styles.priorityButtonUnselected,
                      ]}
                      onPress={() => handlePrioritySelect(option.label)}
                    >
                      <Image source={option.image} style={styles.priorityImage} />
                      <Text style={isSelected ? styles.priorityTextSelected : styles.priorityText}>
                        {option.label}
                      </Text>
                    </TouchableOpacity>
                  </View>
                );
              })}
            </View>
          </>
        )}

        <Text style={styles.sectionLabelequip}>
          Equipment Safe to Use <Text style={styles.requiredAsterisk}>*</Text>
        </Text>
        <View style={styles.safeOptionsRow}>
          {safeOptions.map((option) => {
            const isSelected = safeToUse === option;
            return (
              <TouchableOpacity
                key={option}
                style={[styles.optionButton, isSelected && styles.optionButtonSelected, { marginRight: 20 }]}
                onPress={() => handleSafeSelect(option)}
              >
                <Text style={isSelected ? styles.optionTextSelected : styles.optionText}>{option}</Text>
              </TouchableOpacity>
            );
          })}
        </View>
        {errors.safeToUse ? (
          <Text style={styles.errorText}>{errors.safeToUse}</Text>
        ) : null}

        <View style={styles.inputContainer}>
          <TextInput
            style={styles.input}
            placeholder="Additional Note"
            value={additionalNote}
            onChangeText={handleAdditionalNoteChange}
            multiline={true}
            numberOfLines={4}
            textAlignVertical="top"
          />
          <TouchableOpacity style={styles.inputIconContainer} onPress={handleVoiceInput}>
            <MaterialIcons name="mic" size={24} color="#fff" />
          </TouchableOpacity>
        </View>

        <Text style={styles.signatureLabel}>
          Inspector Signature <Text style={styles.requiredAsterisk}>*</Text>
        </Text>
        {signatureStatus && signatureData ? (
          <View style={styles.signatureContainer}>
            <Text style={styles.signatureStatus}>Signature Done</Text>
            <Image
              source={{ uri: signatureData }}
              style={styles.signatureImage}
              resizeMode="contain"
              onError={(e) => console.log('Image loading error:', e.nativeEvent.error)}
            />
            <TouchableOpacity
              style={styles.uploadAgainButton}
              onPress={handleSignaturePress}
            >
              <Text style={styles.uploadAgainText}>Upload again</Text>
            </TouchableOpacity>
          </View>
        ) : (
          <TouchableOpacity onPress={handleSignaturePress}>
            <View style={styles.signatureBox}>
              <MaterialIcons name="add" size={40} color="#0078D4" />
            </View>
          </TouchableOpacity>
        )}
        {errors.signature ? (
          <Text style={styles.errorText}>{errors.signature}</Text>
        ) : null}

        <View style={styles.acknowledgmentContainer}>
          <View style={styles.acknowledgmentHeader}>
            <MaterialIcons name="check" size={20} color="#015185" />
            <Text style={styles.acknowledgmentTitle}>Operator's Acknowledgment</Text>
          </View>
          <Text style={styles.acknowledgmentText}>
            I hereby certify that the inspection has been done accurately and completely according to compliance with all applicable state and Federal Motor Carrier Safety Regulations.
          </Text>
        </View>
      </ScrollView>

      <View style={styles.navButtonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => setActiveStep(2)}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.navButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleGenerateReport}>
          <Text style={styles.navButtonText}>Generate Report</Text>
          <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} />
      <ListeningModal visible={isListening} onClose={handleVoiceInput} />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingVertical: 10,
    paddingHorizontal: '5%',
    paddingBottom: 120,
  },
  rowContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    width: '100%',
    padding: 10,
    borderRadius: 5,
    marginBottom: 10,
  },
  labelTextFault: {
    fontSize: 16,
    fontWeight: '500',
    color: '#015185',
  },
  valueTextFault: {
    fontSize: 16,
    fontWeight: '600',
    color: '#FF0000',
  },
  buttonContainer: {
    alignItems: 'center',
  },
  selectButton: {
    backgroundColor: '#015185',
    paddingVertical: 4,
    paddingHorizontal: 10,
    borderRadius: 5,
    justifyContent: 'center',
    flexDirection: 'row',
    alignItems: 'center',
  },
  buttonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  iconRight: {
    marginLeft: 5,
  },
  sectionLabelequip: {
    fontSize: 16,
    fontWeight: '500',
    color: '#015185',
    marginVertical: 10,
    marginLeft: '0%',
  },
  optionsRow: {
    flexDirection: 'row',
    marginBottom: 8,
  },
  safeOptionsRow: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginBottom: 15,
  },
  statusOptionButton: {
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#ECF0F3',
    paddingHorizontal: 10,
    paddingVertical: 9,
    borderRadius: 5,
    marginHorizontal: 5,
  },
  statusOptionButtonSelected: {
    backgroundColor: '#015185',
    paddingHorizontal: 10,
    paddingVertical: 9,
    borderRadius: 5,
    marginHorizontal: 5,
  },
  optionButton: {
    paddingHorizontal: 15,
    paddingVertical: 8,
    borderRadius: 5,
    alignItems: 'center',
    justifyContent: 'center',
    minWidth: 80,
    backgroundColor: '#ECF0F3',
  },
  optionButtonSelected: {
    paddingHorizontal: 15,
    paddingVertical: 8,
    borderRadius: 5,
    alignItems: 'center',
    justifyContent: 'center',
    minWidth: 80,
    backgroundColor: '#015185',
  },
  optionText: {
    fontSize: 13,
    fontWeight: '600',
  },
  optionTextSelected: {
    fontSize: 13,
    color: '#fff',
    fontWeight: '600',
  },
  priorityContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginBottom: 15,
    backgroundColor: '#fff',
    borderRadius: 5,
    width: '100%',
  },
  priorityItem: {
    alignItems: 'center',
  },
  priorityButton: {
    padding: 8,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    width: 85,
    height: 60,
    justifyContent: 'center',
    alignItems: 'center',
  },
  priorityButtonSelected: {
    backgroundColor: '#015185',
  },
  priorityButtonUnselected: {
    backgroundColor: '#FFFFFF',
  },
  priorityImage: {
    width: 24,
    height: 24,
  },
  priorityText: {
    fontSize: 10,
    color: '#333',
    marginTop: 3,
    textAlign: 'center',
  },
  priorityTextSelected: {
    fontSize: 10,
    color: '#fff',
    fontWeight: '600',
    marginTop: 3,
    textAlign: 'center',
  },
  inputContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#ECF0F3',
    borderRadius: 5,
    paddingHorizontal: 10,
    marginBottom: 20,
    width: '100%',
  },
  input: {
    flex: 1,
    fontSize: 14,
    color: '#333',
    paddingVertical: 10,
  },
  inputIconContainer: {
    backgroundColor: '#015185',
    padding: 5,
    left: 8,
  },
  signatureLabel: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    textAlign: 'center',
    marginVertical: 10,
  },
  signatureContainer: {
    alignItems: 'center',
  },
  signatureStatus: {
    fontSize: 16,
    color: '#015185',
    textAlign: 'center',
    marginVertical: 5,
  },
  signatureImage: {
    width: width * 0.6,
    height: height * 0.15,
    alignSelf: 'center',
    marginTop: 10,
    borderRadius: 10,
    backgroundColor: '#F5F5F5',
  },
  uploadAgainButton: {
    backgroundColor: '#d3d3d3',
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    marginTop: 10,
    alignSelf: 'center',
  },
  uploadAgainText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  signatureBox: {
    height: height / 3.5,
    backgroundColor: '#F5F5F5',
    borderRadius: 10,
    justifyContent: 'center',
    alignItems: 'center',
    width: '100%',
    marginTop: 10,
  },
  acknowledgmentContainer: {
    backgroundColor: '#fff',
    marginTop: 20,
    marginBottom: 20,
    padding: 16,
    borderRadius: 8,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.05,
    shadowRadius: 4,
    elevation: 2,
  },
  acknowledgmentHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
  },
  acknowledgmentTitle: {
    fontWeight: 'bold',
    color: '#015185',
    fontSize: 17,
    marginLeft: 6,
  },
  acknowledgmentText: {
    fontSize: 13,
    color: '#222',
    marginLeft: 2,
  },
  navButtonContainer: {
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: 15,
    paddingVertical: 10,
    backgroundColor: '#f0f0f0',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 15,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    borderRadius: 5,
  },
  navButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  modalRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
  },
  modalOptionText: {
    fontSize: 16,
    color: '#333',
  },
  modalDivider: {
    height: 1,
    backgroundColor: '#d3d3d3',
  },
  requiredAsterisk: {
    color: '#FF0000',
  },
  errorText: {
    color: '#FF0000',
    fontSize: 12,
    marginBottom: 10,
    marginLeft: 5,
  },
  radioCircle: {
    width: 24,
    height: 24,
    borderRadius: 12,
    borderWidth: 2,
    borderColor: '#015185',
    justifyContent: 'center',
    alignItems: 'center',
  },
  radioCircleSelected: {
    borderColor: '#015185',
  },
  radioDot: {
    width: 12,
    height: 12,
    borderRadius: 6,
    backgroundColor: '#015185',
  },
  popupModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
  },
  popupTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  popupSubtitle: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
  },
  popupButton: {
    backgroundColor: '#0078D4',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  popupButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  bottomicon: {
    marginTop: 2,
  },
  listeningModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  listeningModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  listeningModalText: {
    fontSize: 18,
    color: '#333',
    marginBottom: 15,
  },
  stopButton: {
    backgroundColor: '#FF0000',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  stopButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
});

export default DvirSummeryStep3;