import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Modal,
  FlatList,
  Alert,
  KeyboardAvoidingView,
  Platform,
  Animated,
  ActivityIndicator,
  Dimensions,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { Dropdown } from 'react-native-element-dropdown';
import NetInfo from '@react-native-community/netinfo';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');

const DvirInspectioninfoStep1 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [country, setCountry] = useState(null);
  const [countryItems] = useState([
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ]);
  const [project, setProject] = useState(null);
  const [projectItems, setProjectItems] = useState([]);
  const [inspector, setInspector] = useState('');
  const [checklistModalVisible, setChecklistModalVisible] = useState(false);
  const [equipmentModalVisible, setEquipmentModalVisible] = useState(false);
  const [checklists, setChecklists] = useState([]);
  const [firebaseEquipment, setFirebaseEquipment] = useState([]);
  const [savedEquipmentList, setSavedEquipmentList] = useState([]);
  const [checklistSearchQuery, setChecklistSearchQuery] = useState('');
  const [isChecklistSearchActive, setIsChecklistSearchActive] = useState(false);
  const [selectedTab, setSelectedTab] = useState('ALL');
  const [selectedChecklistId, setSelectedChecklistId] = useState(null);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [meterReading, setMeterReading] = useState('');
  const [currentMeterReading, setCurrentMeterReading] = useState('');
  const [requestNumber, setRequestNumber] = useState('');
  const [inspectionTypes, setInspectionTypes] = useState({
    DVIR: false,
    DOT: false,
  });
  const [equipmentSearchQuery, setEquipmentSearchQuery] = useState('');
  const [isEquipmentSearchActive, setIsEquipmentSearchActive] = useState(false);
  const [userInfo, setUserInfo] = useState({
    countries: [],
    equipmentPermissions: [],
    userUid: '',
  });
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isLoadingEquipment, setIsLoadingEquipment] = useState(false);
  const [selectedCountry, setSelectedCountry] = useState('All Countries');
  const equipmentSearchInputRef = useRef(null);
  const isMounted = useRef(true);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const showToast = (type, text1, text2) => {
    Toast.show({ type, text1, text2, position: 'top' });
  };

  const generateRequestNumber = async () => {
    try {
      let lastNumber = 0;
      const isOnline = await checkNetworkStatus();
      if (isOnline) {
        // Only read, do not increment
        const lastNumberRef = database().ref('/GlobalDvirReports/admins/lastDvirInspectionNumber');
        const snapshot = await lastNumberRef.once('value');
        lastNumber = snapshot.val() || 0;
      } else {
        // Offline fallback
        const lastRequestRef = await AsyncStorage.getItem('lastDvirInspectionNumber');
        lastNumber = parseInt(lastRequestRef, 10) || 0;
      }
      const nextNumber = lastNumber + 1;
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
      return `${countryCode}-${projectCode}-DVIR-${String(nextNumber).padStart(3, '0')}`;
    } catch (error) {
      console.error('Error generating request number:', error);
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
      return `${countryCode}-${projectCode}-DVIR-001`;
    }
  };

  const fetchUserInfo = async () => {
    try {
      const user = auth().currentUser;
      if (!user) {
        showToast('error', 'Error', 'User not authenticated. Please log in.');
        navigation.replace('AdminLogin');
        return null;
      }

      let userUid = await AsyncStorage.getItem('userUid');
      let userData = null;

      if (userUid) {
        const userRef = database().ref(`/Globalusers/admins/${userUid}`);
        const snapshot = await userRef.once('value');
        userData = snapshot.val();
      }

      if (!userData) {
        const email = user.email.replace(/\./g, '_');
        const userRef = database().ref(`/Globalusers/admins/${email}`);
        const snapshot = await userRef.once('value');
        userData = snapshot.val();
        if (userData && userData.userUid) {
          userUid = userData.userUid;
          await AsyncStorage.setItem('userUid', userUid);
        } else {
          userUid = email;
          await AsyncStorage.setItem('userUid', userUid);
        }
      }

      if (!userData) {
        showToast('error', 'Error', 'User data not found.');
        return null;
      }

      const equipmentModule = userData.modules?.find(
        (module) => module.module?.toLowerCase() === 'equipments'
      ) || { permissions: [] };

      const userInfo = {
        countries: userData.countries || [],
        equipmentPermissions: equipmentModule.permissions || [],
        userUid: userUid || user.uid,
      };

      await AsyncStorage.setItem('userInfo', JSON.stringify(userInfo));
      setUserInfo(userInfo);
      return userInfo;
    } catch (error) {
      console.error('Error fetching user info:', error);
      showToast('error', 'Error', 'Failed to fetch user permissions.');
      return null;
    }
  };

  const filterEquipmentByPermission = useCallback((equipmentArr, userInfoObj) => {
    if (!equipmentArr || !userInfoObj) return [];
    // Show all equipment the user has access to, regardless of category
    if (userInfoObj.equipmentPermissions.includes('onlyMineView')) {
      const creatorIdentifier = userInfoObj.userUid;
      return equipmentArr.filter(
        (item) =>
          (item.createdBy === creatorIdentifier || !item.createdBy) &&
          userInfoObj.countries.includes(item.country)
      );
    } else if (userInfoObj.equipmentPermissions.includes('view')) {
      return equipmentArr.filter(
        (item) => userInfoObj.countries.includes(item.country)
      );
    } else {
      return [];
    }
  }, []);

  const loadEquipment = async (isRefresh = false) => {
    setSyncModalVisible(true);
    setIsLoadingEquipment(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const user = auth().currentUser;
      if (!user) {
        setSyncModalVisible(false);
        setIsLoadingEquipment(false);
        rotateAnim.setValue(0);
        showToast('error', 'Error', 'User not authenticated. Please log in.');
        navigation.replace('AdminLogin');
        return;
      }

      const userInfoData = await fetchUserInfo();
      if (!userInfoData) {
        setSyncModalVisible(false);
        setIsLoadingEquipment(false);
        rotateAnim.setValue(0);
        return;
      }

      console.log('User Info:', userInfoData); // Debug: Check user info and permissions
      const isOnline = await checkNetworkStatus();
      let parsedData = [];

      if (isOnline) {
        if (userInfoData.equipmentPermissions.includes('onlyMineView')) {
          const equipmentRef = database().ref(`/equipment/${userInfoData.userUid}`);
          const snapshot = await equipmentRef.once('value');
          const equipmentData = snapshot.val() || {};

          console.log('Raw Equipment Data:', equipmentData); // Debug: Check raw data from Firebase

          parsedData = Object.keys(equipmentData).map((key) => ({
            ...equipmentData[key],
            firebaseKey: key,
          }));

          console.log('Parsed Equipment Data:', parsedData); // Debug: Check parsed data

          const userRef = database().ref(`/Globalusers/admins/${userInfoData.userUid}`);
          const userSnapshot = await userRef.once('value');
          const userData = userSnapshot.val();
          if (!userData || !userData.countries || userData.countries.length === 0) {
            parsedData = [];
            console.log('No user data or countries found, setting parsedData to empty');
          } else {
            parsedData = parsedData.filter((item) =>
              userData.countries.includes(item.country)
            );
            console.log('Filtered by countries:', parsedData);
          }
        } else {
          const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
          const snapshot = await globalRef.once('value');
          const data = snapshot.val() || [];

          parsedData = Array.isArray(data)
            ? data.map((item, index) => ({
                ...item,
                firebaseKey: `equip_${index}`,
              }))
            : [];
        }

        parsedData = parsedData
          .filter((item, index, self) => self.findIndex((t) => t.id === item.id) === index)
          .sort((a, b) => {
            const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
            const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
            return dateB - dateA;
          });

        const filteredData = filterEquipmentByPermission(parsedData, userInfoData);
        console.log('Data after filterEquipmentByPermission:', filteredData); // Debug: Check final filtered data

        parsedData = filteredData;
        await AsyncStorage.setItem('equipment', JSON.stringify(parsedData));
      } else {
        const savedData = await AsyncStorage.getItem('equipment');
        parsedData = savedData ? JSON.parse(savedData) : [];
        const filteredData = filterEquipmentByPermission(parsedData, userInfoData);
        console.log('Offline Data after filtering:', filteredData); // Debug: Check offline data
        parsedData = filteredData;
        parsedData.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        showToast('info', 'Offline', 'Equipment data refreshed from local storage.');
      }

      if (isRefresh) {
        setFirebaseEquipment(parsedData);
      } else {
        setFirebaseEquipment((prev) => [...prev, ...parsedData]);
      }

      setSavedEquipmentList(parsedData);
      
      // Additional state management like EquipmentOne
      const storedCountry = await AsyncStorage.getItem('selectedCountry');
      setSelectedCountry(storedCountry || 'All Countries');
    } catch (error) {
      console.error('Error loading equipment:', error.code, error.message);
      let errorMessage = 'Failed to load equipment.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Check Firebase security rules.';
      } else if (error.code === 'NETWORK_ERROR') {
        errorMessage = 'Offline: Loading from local storage.';
        const savedData = await AsyncStorage.getItem('equipment');
        const parsedData = savedData ? JSON.parse(savedData) : [];
        const userInfoData = await AsyncStorage.getItem('userInfo');
        const userInfoParsed = userInfoData ? JSON.parse(userInfoData) : userInfo;
        parsedData.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        const filteredData = filterEquipmentByPermission(parsedData, userInfoParsed);
        setFirebaseEquipment(filteredData);
        setSavedEquipmentList(filteredData);
      }
      showToast(
        error.code === 'NETWORK_ERROR' ? 'info' : 'error',
        error.code === 'NETWORK_ERROR' ? 'Offline' : 'Error',
        errorMessage
      );
    } finally {
      setSyncModalVisible(false);
      setIsLoadingEquipment(false);
      rotateAnim.setValue(0);
    }
  };

  const loadChecklists = async () => {
    try {
      const state = await NetInfo.fetch();
      let parsedChecklists = [];
      if (state.isConnected) {
        const snapshot = await database().ref('checklists').once('value');
        parsedChecklists = snapshot.val()
          ? Object.keys(snapshot.val()).map((id) => ({
              id,
              ...snapshot.val()[id],
            }))
          : [];
        await AsyncStorage.setItem('checklists', JSON.stringify(parsedChecklists));
      } else {
        const storedChecklists = await AsyncStorage.getItem('checklists');
        parsedChecklists = storedChecklists ? JSON.parse(storedChecklists) : [];
      }
      setChecklists(parsedChecklists);
    } catch (error) {
      console.error('Error loading checklists:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load checklists.',
        position: 'top',
      });
    }
  };

  const getFilteredEquipment = () => {
    let filtered = [...firebaseEquipment];
    if (selectedTab !== 'ALL') {
      filtered = filtered.filter((item) => {
        switch (selectedTab) {
          case 'LV':
            return item.mainCategory === 'Light Vehicles TDC';
          case 'HV':
            return item.mainCategory === 'Heavy Vehicles TDC';
          default:
            return true;
        }
      });
    }
    if (equipmentSearchQuery.trim()) {
      filtered = filtered.filter((item) =>
        [
          item.equipmentName || '',
          item.mainCategory || '',
          item.model || '',
          item.equipmentNumber || '',
        ].some((field) =>
          field.toLowerCase().includes(equipmentSearchQuery.toLowerCase())
        )
      );
    }
    return filtered;
  };

  useEffect(() => {
    const loadInitialData = async () => {
      await loadChecklists();
      await loadEquipment(true);
      const newRequestNumber = await generateRequestNumber();
      setRequestNumber(newRequestNumber);
    };
    loadInitialData();

    return () => {
      isMounted.current = false;
    };
  }, [country, project]);

  // Add sync functionality like EquipmentOne
  useEffect(() => {
    const syncPendingChanges = async () => {
      const isOnline = await checkNetworkStatus();
      if (!isOnline) return;

      const user = auth().currentUser;
      if (!user) return;

      try {
        const existing = await AsyncStorage.getItem('equipment');
        const equipment = existing ? JSON.parse(existing) : [];
        const pending = equipment.filter((item) => item.pendingSync);

        if (pending.length > 0) {
          setSyncModalVisible(true);
          Animated.loop(
            Animated.timing(rotateAnim, {
              toValue: 1,
              duration: 1000,
              useNativeDriver: true,
            })
          ).start();

          const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
          const globalSnapshot = await globalRef.once('value');
          let globalData = globalSnapshot.val() || [];

          for (const item of pending) {
            if (item.pendingDelete) {
              globalData = globalData.filter((g) => g.id !== item.id);
            } else {
              const updatedItem = { ...item, pendingSync: false, pendingDelete: false };
              const globalIndex = globalData.findIndex((g) => g.id === item.id);
              if (globalIndex !== -1) {
                globalData[globalIndex] = updatedItem;
              } else {
                globalData.push(updatedItem);
              }
            }
          }

          await globalRef.set(globalData);

          const updatedEquipment = equipment
            .filter((item) => !item.pendingDelete)
            .map((item) => ({
              ...item,
              pendingSync: false,
              pendingDelete: false,
            }));
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipment));

          setSyncModalVisible(false);
          rotateAnim.setValue(0);
          showToast('success', 'Success', 'Pending changes synced to database.');
        }
      } catch (error) {
        console.error('Error syncing pending changes:', error);
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        showToast('error', 'Error', 'Failed to sync pending changes.');
      }
    };

    const unsubscribe = NetInfo.addEventListener((state) => {
      if (state.isConnected) {
        syncPendingChanges();
      }
    });

    return () => unsubscribe();
  }, []);

  useEffect(() => {
    if (country === 'Zambia') {
      setProjectItems([
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kimteto', value: 'Kimteto' },
        { label: 'Kobold', value: 'Kobold' },
        { label: 'FQM Exploration', value: 'FQM Exploration' },
        { label: 'Mimosa', value: 'Mimosa' },
      ]);
    } else if (country === 'DRC') {
      setProjectItems([
        { label: 'Lubumbashi', value: 'Lubumbashi' },
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
        { label: 'Kamoa', value: 'Kamoa' },
      ]);
    } else {
      setProjectItems([]);
    }
    setProject(null);
  }, [country]);

  const handleSelectPress = async () => {
    setEquipmentModalVisible(true); // Open modal immediately
    setIsLoadingEquipment(true);
    setEquipmentSearchQuery('');
    setIsEquipmentSearchActive(false);

    try {
      await loadEquipment(true);
    } catch (error) {
      showToast('error', 'Error', 'Failed to load equipment data. Please try again.');
    } finally {
      setIsLoadingEquipment(false);
    }
  };

  const handleChecklistSelectPress = () => {
    setChecklistModalVisible(true);
  };

  const handleBackPress = () => {
    Alert.alert(
      'Exit Step',
      'Do you want to exit this step?',
      [
        { text: 'Cancel', style: 'cancel' },
        { text: 'OK', onPress: () => navigation.goBack(), style: 'default' },
      ],
      { cancelable: false }
    );
  };

  const isNextDisabled =
    !country ||
    !project ||
    !inspector.trim() ||
    !selectedEquipment ||
    (!inspectionTypes.DVIR && !inspectionTypes.DOT);

  const validateChecklistItems = (checklistId) => {
    const checklist = checklists.find((c) => c.id === checklistId);
    if (!checklist) {
      Toast.show({
        type: 'error',
        text1: 'Invalid Checklist',
        text2: 'No checklist found with the selected ID.',
        position: 'top',
      });
      return false;
    }
    if (!checklist.items || !Array.isArray(checklist.items)) {
      Toast.show({
        type: 'error',
        text1: 'Invalid Checklist',
        text2: 'Checklist items are missing or not an array.',
        position: 'top',
      });
      return false;
    }
    if (
      !checklist.items.every(
        (item) =>
          item &&
          typeof item.text === 'string' &&
          Array.isArray(item.subItems) &&
          item.subItems.every(
            (subItem) =>
              subItem &&
              typeof subItem.itemName === 'string' &&
              typeof subItem.inputType === 'string'
          )
      )
    ) {
      Toast.show({
        type: 'error',
        text1: 'Invalid Checklist',
        text2: 'Checklist contains invalid item or sub-item structure.',
        position: 'top',
      });
      return false;
    }
    return true;
  };

  const saveStep1Data = async () => {
    try {
      const step1Data = {
        requestNumber,
        country,
        project,
        inspector,
        selectedEquipment,
        meterReading,
        currentMeterReading,
        selectedChecklistId,
        inspectionTypes,
        timestamp: new Date().getTime(),
      };
      const key = `step1-${step1Data.timestamp}`;
      await AsyncStorage.setItem(key, JSON.stringify(step1Data));
      return step1Data;
    } catch (error) {
      console.error('Error saving Step 1 data:', error);
      throw error;
    }
  };

  const handleNextPress = async () => {
    if (isNextDisabled) {
      Alert.alert(
        'Error',
        'Please fill all required fields: Country, Project, Inspector, Equipment, and at least one Inspection Type.'
      );
      return;
    }
    if (meterReading && parseFloat(meterReading) <= parseFloat(selectedEquipment.meterReading)) {
      Alert.alert(
        'Error',
        `Meter reading must be greater than ${selectedEquipment.meterReading || '0'}.`
      );
      return;
    }
    try {
      if (meterReading && parseFloat(meterReading) !== parseFloat(selectedEquipment.meterReading)) {
        await updateMeterReading();
      }
      await saveStep1Data();
      setActiveStep(2);
    } catch (error) {
      console.error('Error saving inspection data:', error);
      Alert.alert('Error', 'Failed to save inspection data. Please try again.');
    }
  };

  const updateMeterReading = async () => {
    if (!selectedEquipment) return;
    if (!userInfo.equipmentPermissions.includes('update')) {
      Toast.show({
        type: 'error',
        text1: 'Permission Denied',
        text2: 'You do not have permission to update equipment.',
        position: 'top',
      });
      return;
    }

    const updatedEquipment = {
      ...selectedEquipment,
      meterReading: parseFloat(meterReading),
      updatedAt: new Date().toISOString(),
      createdBy: userInfo.userUid,
    };

    try {
      const state = await NetInfo.fetch();
      let updatedEquipmentList = [...firebaseEquipment];

      if (state.isConnected) {
        const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
        const globalSnapshot = await globalRef.once('value');
        let globalData = globalSnapshot.val() || [];
        const globalIndex = globalData.findIndex((g) => g.id === updatedEquipment.id);
        if (globalIndex !== -1) {
          globalData[globalIndex] = updatedEquipment;
        } else {
          globalData.push(updatedEquipment);
        }
        await globalRef.set(globalData);

        updatedEquipmentList = updatedEquipmentList.map((item) =>
          item.id === updatedEquipment.id ? updatedEquipment : item
        );
        await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipmentList));
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Meter reading updated in database.',
          position: 'top',
        });
      } else {
        updatedEquipmentList = updatedEquipmentList.map((item) =>
          item.id === updatedEquipment.id
            ? { ...updatedEquipment, pendingSync: true }
            : item
        );
        await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipmentList));
        Toast.show({
          type: 'info',
          text1: 'Offline',
          text2: 'Meter reading updated locally. Will sync when online.',
          position: 'top',
        });
      }

      setFirebaseEquipment(updatedEquipmentList);
      setSavedEquipmentList(updatedEquipmentList);
      setSelectedEquipment(updatedEquipment);
      setCurrentMeterReading(meterReading);
    } catch (error) {
      console.error('Error updating meter reading:', error);
      let errorMessage = 'Failed to update meter reading.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Check Firebase security rules.';
      } else if (error.code === 'NETWORK_ERROR') {
        errorMessage = 'Offline: Meter reading updated locally, will sync when online.';
        updatedEquipmentList = firebaseEquipment.map((item) =>
          item.id === updatedEquipment.id
            ? { ...updatedEquipment, pendingSync: true }
            : item
        );
        await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipmentList));
        setFirebaseEquipment(updatedEquipmentList);
        setSavedEquipmentList(updatedEquipmentList);
        setSelectedEquipment(updatedEquipment);
        setCurrentMeterReading(meterReading);
        Toast.show({
          type: 'info',
          text1: 'Offline',
          text2: errorMessage,
          position: 'top',
        });
        return;
      }
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
        position: 'top',
      });
    }
  };

  const handleChecklistSearch = (query) => {
    setChecklistSearchQuery(query);
  };

  const handleEquipmentSearch = (query) => {
    setEquipmentSearchQuery(query);
  };

  const filteredChecklists = checklists.filter((checklist) =>
    checklist.title?.toLowerCase().includes(checklistSearchQuery.toLowerCase())
  );

  const toggleChecklistSelection = async (checklistId) => {
    if (!validateChecklistItems(checklistId)) return;
    setSelectedChecklistId((prev) => {
      const newId = prev === checklistId ? null : checklistId;
      AsyncStorage.setItem('selectedChecklistIds', JSON.stringify(newId ? [newId] : []));
      return newId;
    });
    setChecklistModalVisible(false);
  };

  const selectEquipment = async (equipment) => {
    setSelectedEquipment(equipment);
    setMeterReading(equipment.meterReading?.toString() || '');
    setCurrentMeterReading(equipment.meterReading?.toString() || '');
    setEquipmentModalVisible(false);
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Available':
        return '#2ecc71';
      case 'Maintenance Required':
        return '#f1c40f';
      case 'In Maintenance':
        return '#e67e22';
      case 'Breakdown':
        return '#e74c3c';
      case 'Deploy':
        return '#3498db';
      default:
        return '#666';
    }
  };

  const renderChecklistItem = ({ item }) => (
    <TouchableOpacity
      style={styles.checklistItem}
      onPress={() => toggleChecklistSelection(item.id)}
    >
      <View style={styles.checklistItemContent}>
        <MaterialIcons
          name={selectedChecklistId === item.id ? 'check-box' : 'check-box-outline-blank'}
          size={24}
          color={selectedChecklistId === item.id ? '#0078D4' : '#666'}
        />
        <Text style={styles.checklistItemText}>{item.title || 'Untitled Checklist'}</Text>
      </View>
    </TouchableOpacity>
  );

  const renderEquipmentItem = ({ item }) => (
    <View style={styles.equipmentRow}>
      <TouchableOpacity
        style={styles.equipmentInfo}
        onPress={() => selectEquipment(item)}
      >
        <View style={styles.equipmentDetailItemheader}>
          <Text style={styles.detailLabelheader}>{item.equipmentName || 'Equipment'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Category</Text>
          <Text style={styles.detailValue}>{item.mainCategory || 'N/A'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Model</Text>
          <Text style={styles.detailValue}>{item.model || 'N/A'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Meter Reading</Text>
          <Text style={styles.detailValue}>
            {item.meterReading || '0'} {item.meterUnit || 'Miles'}
          </Text>
        </View>
        <View style={styles.equipmentDetailItemNoBorder}>
          <Text
            style={[styles.detailLabelstatuss, { color: getStatusColor(item.status || 'Available') }]}
          >
            {item.status || 'Available'}
          </Text>
        </View>
      </TouchableOpacity>
      <TouchableOpacity
        style={styles.selectEquipmentButton}
        onPress={() => selectEquipment(item)}
      >
        <Text style={styles.selectEquipmentText}>
          {selectedEquipment && selectedEquipment.id === item.id ? 'Selected' : 'Select'}
        </Text>
      </TouchableOpacity>
    </View>
  );

  return (
    <KeyboardAvoidingView
      style={{ flex: 1 }}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 0}
    >
      <View style={styles.container}>
        <ScrollView
          style={styles.mainContent}
          contentContainerStyle={styles.scrollContent}
          keyboardShouldPersistTaps="handled"
        >
          <View style={styles.headingContainer}>
            <Text style={styles.headingText}>New Inspection</Text>
          </View>
          <View style={styles.requestRow}>
            <Text style={styles.requestLabel}>Request #</Text>
            <Text style={styles.requestValue}>{requestNumber}</Text>
          </View>
          <View style={styles.fieldContainer}>
            <Text style={styles.label}>
              Country <Text style={styles.requiredLabel}>*</Text>
            </Text>
            <Dropdown
              style={styles.dropdown}
              placeholderStyle={styles.placeholderStyle}
              selectedTextStyle={styles.selectedTextStyle}
              data={countryItems}
              maxHeight={300}
              labelField="label"
              valueField="value"
              placeholder="Select Country"
              value={country}
              onChange={(item) => setCountry(item.value)}
            />
          </View>
          <View style={styles.fieldContainer}>
            <Text style={styles.label}>
              Project <Text style={styles.requiredLabel}>*</Text>
            </Text>
            <Dropdown
              style={[styles.dropdown, !country && styles.disabledDropdown]}
              placeholderStyle={styles.placeholderStyle}
              selectedTextStyle={styles.selectedTextStyle}
              data={projectItems}
              maxHeight={300}
              labelField="label"
              valueField="value"
              placeholder="Select Project"
              value={project}
              onChange={(item) => setProject(item.value)}
              disable={!country}
            />
          </View>
          <View style={styles.fieldContainer}>
            <Text style={styles.label}>
              Inspector <Text style={styles.requiredLabel}>*</Text>
            </Text>
            <TextInput
              style={styles.input}
              value={inspector}
              onChangeText={setInspector}
              placeholder="Enter Inspector"
              placeholderTextColor="#888"
            />
          </View>
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Equipment</Text>
            <TouchableOpacity onPress={handleSelectPress} style={styles.selectButton}>
              <Text style={styles.selectText}>
                {selectedEquipment ? 'Change equipment' : 'Select'}
              </Text>
              <MaterialIcons name="chevron-right" size={16} color="#0078D4" />
            </TouchableOpacity>
          </View>
          {selectedEquipment && (
            <View style={styles.equipmentDetailsBox}>
              <View style={styles.vehicleEquipmentRow}>
                <Text style={styles.vehicleEquipmentText}>
                  Equipment Number: {selectedEquipment.equipmentNumber || 'N/A'}, Category:{' '}
                  {selectedEquipment.mainCategory || 'N/A'}, Model: {selectedEquipment.model || 'N/A'}
                </Text>
              </View>
              <View style={styles.labelRow}>
                <Text style={styles.meterReadingLabel}>
                  Meter Reading <Text style={styles.requiredLabel}>*</Text>
                </Text>
              </View>
              <View style={styles.inputRow}>
                <TextInput
                  style={styles.meterReadingInput}
                  value={meterReading}
                  onChangeText={setMeterReading}
                  placeholder="Enter meter reading"
                  placeholderTextColor="#888"
                  keyboardType="numeric"
                />
                <Text style={styles.meterUnit}>{selectedEquipment.meterUnit || 'Miles'}</Text>
              </View>
              {meterReading && parseFloat(meterReading) <= parseFloat(selectedEquipment.meterReading) && (
                <Text style={styles.errorText}>
                  Meter reading must be greater than {selectedEquipment.meterReading || '0'}.
                </Text>
              )}
              <View style={styles.currentMeterRow}>
                <Text style={styles.currentMeterLabel}>Current Meter Reading: </Text>
                <Text style={styles.currentMeterValue}>
                  {selectedEquipment.meterReading || '0'} {selectedEquipment.meterUnit || 'Miles'}
                </Text>
              </View>
            </View>
          )}
          <View style={{ borderTopWidth: 1, borderTopColor: '#e0e0e0', marginTop: 2, marginBottom: 8 }} />
          <TouchableOpacity
            style={styles.checkboxRow}
            onPress={() =>
              setInspectionTypes((prev) => ({ ...prev, DVIR: !prev.DVIR }))
            }
          >
            <View style={[styles.checkbox, inspectionTypes.DVIR && styles.checkboxChecked]}>
              {inspectionTypes.DVIR && (
                <MaterialIcons name="check" size={16} color="#fff" />
              )}
            </View>
            <Text style={styles.checkboxLabel}>DVIR - Driver Vehicle Inspection Report</Text>
          </TouchableOpacity>
        </ScrollView>
        <View style={styles.buttonContainer}>
          <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
            <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
            <Text style={styles.cancelButtonText}>Cancel</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.nextButton, isNextDisabled && styles.disabledButton]}
            onPress={handleNextPress}
            disabled={isNextDisabled}
          >
            <Text style={styles.nextButtonText}>Next</Text>
            <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
          </TouchableOpacity>
        </View>
        <Modal
          visible={checklistModalVisible}
          transparent={true}
          animationType="fade"
          onRequestClose={() => setChecklistModalVisible(false)}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.checklistModalContent}>
              <View style={styles.checklistModalHeader}>
                <Text style={styles.checklistModalTitle}>Select Checklist</Text>
                <TouchableOpacity
                  onPress={() => setChecklistModalVisible(false)}
                  style={styles.modalCloseIcon}
                >
                  <MaterialIcons name="close" size={24} color="#666" />
                </TouchableOpacity>
              </View>
              <View style={styles.checklistModalTabs}>
                <TouchableOpacity
                  style={[styles.tabButton, selectedTab === 'ALL' && styles.activeTabButton]}
                  onPress={() => setSelectedTab('ALL')}
                >
                  <Text style={[styles.tabText, selectedTab === 'ALL' && styles.activeTabText]}>
                    ALL
                  </Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[styles.tabButton, selectedTab === 'Assign' && styles.activeTabButton]}
                  onPress={() => setSelectedTab('Assign')}
                >
                  <Text style={[styles.tabText, selectedTab === 'Assign' && styles.activeTabText]}>
                    Assign
                  </Text>
                </TouchableOpacity>
              </View>
              <View style={styles.checklistSearchContainer}>
                {isChecklistSearchActive ? (
                  <TextInput
                    style={styles.checklistSearchInput}
                    placeholder="Search checklists..."
                    placeholderTextColor="#888"
                    value={checklistSearchQuery}
                    onChangeText={handleChecklistSearch}
                    autoFocus
                  />
                ) : (
                  <Text style={styles.checklistSelectText}>Search Checklists</Text>
                )}
                <TouchableOpacity
                  onPress={() => {
                    setIsChecklistSearchActive(!isChecklistSearchActive);
                    if (isChecklistSearchActive) setChecklistSearchQuery('');
                  }}
                >
                  <MaterialIcons
                    name={isChecklistSearchActive ? 'close' : 'search'}
                    size={24}
                    color="#666"
                  />
                </TouchableOpacity>
              </View>
              <View style={styles.divider} />
              <FlatList
                data={filteredChecklists}
                keyExtractor={(item) => item.id.toString()}
                renderItem={renderChecklistItem}
                ListEmptyComponent={
                  <Text style={styles.checklistEmptyText}>No checklists found</Text>
                }
                style={styles.checklistList}
              />
              <View style={styles.divider} />
              <View style={styles.checklistModalFooter}>
                <TouchableOpacity
                  onPress={() => {
                    setChecklistModalVisible(false);
                  }}
                >
                  <Text style={styles.createChecklistText}>Create New Checklist</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={styles.checklistCloseButton}
                  onPress={() => setChecklistModalVisible(false)}
                >
                  <Text style={styles.checklistCloseText}>Close</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>
        <Modal
          visible={equipmentModalVisible}
          transparent={true}
          animationType="fade"
          onRequestClose={() => setEquipmentModalVisible(false)}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.equipmentModalContent}>
              <View style={styles.equipmentModalHeader}>
                <Text style={styles.equipmentModalTitle}>Select Equipment</Text>
                <TouchableOpacity
                  onPress={() => setEquipmentModalVisible(false)}
                  style={styles.modalCloseIcon}
                >
                  <MaterialIcons name="close" size={24} color="#666" />
                </TouchableOpacity>
              </View>
              <TouchableOpacity
                style={[
                  styles.equipmentSearchContainer,
                  isEquipmentSearchActive && styles.equipmentSearchContainerActive,
                ]}
                onPress={() => {
                  setIsEquipmentSearchActive(true);
                  equipmentSearchInputRef.current?.focus();
                }}
              >
                {isEquipmentSearchActive ? (
                  <TextInput
                    ref={equipmentSearchInputRef}
                    style={[
                      styles.equipmentSearchInput,
                      isEquipmentSearchActive && styles.equipmentSearchInputActive,
                    ]}
                    placeholder={isEquipmentSearchActive ? 'Search...' : 'Search equipment...'}
                    placeholderTextColor="#888"
                    value={equipmentSearchQuery}
                    onChangeText={handleEquipmentSearch}
                    autoFocus={isEquipmentSearchActive}
                  />
                ) : (
                  <Text style={styles.equipmentSelectText}>Search Equipment</Text>
                )}
                <TouchableOpacity
                  onPress={() => {
                    if (isEquipmentSearchActive) {
                      setIsEquipmentSearchActive(false);
                      setEquipmentSearchQuery('');
                      equipmentSearchInputRef.current?.blur();
                    } else {
                      setIsEquipmentSearchActive(true);
                      equipmentSearchInputRef.current?.focus();
                    }
                  }}
                >
                  <MaterialIcons
                    name={isEquipmentSearchActive ? 'close' : 'search'}
                    size={24}
                    color="#666"
                  />
                </TouchableOpacity>
              </TouchableOpacity>
              <View style={styles.divider} />
              {isLoadingEquipment ? (
                <View style={styles.loadingContainer}>
                  <ActivityIndicator size="large" color="#3481BC" />
                  <Text style={styles.loadingText}>Loading Equipment...</Text>
                </View>
              ) : (
                <FlatList
                  data={getFilteredEquipment()}
                  keyExtractor={(item) => item.id.toString()}
                  renderItem={renderEquipmentItem}
                  ListEmptyComponent={
                    <Text style={styles.equipmentEmptyText}>No equipment found</Text>
                  }
                  style={styles.equipmentList}
                />
              )}
              <View style={styles.divider} />
              <View style={styles.equipmentModalFooter}>
                <TouchableOpacity
                  style={styles.equipmentCloseButton}
                  onPress={() => setEquipmentModalVisible(false)}
                >
                  <Text style={styles.equipmentCloseText}>Close</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>
        <Modal
          visible={syncModalVisible}
          transparent
          animationType="fade"
        >
          <View style={styles.syncModalContainer}>
            <View style={styles.syncModalContent}>
              <Text style={styles.syncModalText}>Please Wait</Text>
              <View style={styles.syncIcondiv}>
                <Animated.Image
                  source={require('../../../Images/adminlogin/sync.png')}
                  style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
                />
                <Text style={styles.syncModalTextadd}>Processing Equipment Data</Text>
              </View>
            </View>
          </View>
        </Modal>
        <Toast />
      </View>
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: 15,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  headingContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: 10,
  },
  headingText: {
    fontSize: 20,
    fontWeight: '500',
    color: '#015185',
  },
  requestRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    height: 50,
    backgroundColor: '#f0f0f0',
    paddingHorizontal: 15,
    marginBottom: 20,
    borderRadius: 5,
  },
  requestLabel: {
    fontSize: 16,
    color: '#015185',
  },
  requestValue: {
    fontSize: 16,
    color: '#015185',
  },
  fieldContainer: {
    marginBottom: 15,
  },
  label: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 5,
    fontWeight: '500',
  },
  requiredLabel: {
    color: 'red',
    fontSize: 16,
  },
  input: {
    height: 40,
    borderWidth: 1,
    borderColor: '#0078D4',
    borderRadius: 5,
    paddingHorizontal: 10,
    fontSize: 16,
    color: '#000',
    backgroundColor: '#fff',
  },
  dropdown: {
    height: 40,
    borderWidth: 1,
    borderColor: '#0078D4',
    borderRadius: 5,
    backgroundColor: '#fff',
    paddingHorizontal: 10,
  },
  disabledDropdown: {
    backgroundColor: '#e0e0e0',
  },
  placeholderStyle: {
    fontSize: 16,
    color: '#888',
  },
  selectedTextStyle: {
    fontSize: 16,
    color: '#333',
  },
  selectRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    width: '100%',
    height: 50,
    backgroundColor: '#f0f0f0',
    paddingHorizontal: 15,
    marginTop: 10,
    borderRadius: 5,
  },
  selectLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
  },
  selectButton: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  selectText: {
    fontSize: 16,
    color: '#0078D4',
    marginLeft: 5,
  },
  equipmentDetailsBox: {
    width: '100%',
    backgroundColor: '#f0f0f0',
    paddingVertical: 10,
    paddingHorizontal: 15,
    marginBottom: 10,
    borderRadius: 5,
  },
  vehicleEquipmentRow: {
    paddingVertical: 10,
  },
  vehicleEquipmentText: {
    fontSize: 16,
    color: '#333',
  },
  labelRow: {
    paddingVertical: 5,
  },
  meterReadingLabel: {
    fontSize: 16,
    color: '#333',
    fontWeight: '600',
  },
  inputRow: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    paddingHorizontal: 10,
    paddingVertical: 5,
  },
  meterReadingInput: {
    flex: 1,
    fontSize: 16,
    color: '#333',
    padding: 0,
  },
  meterUnit: {
    color: '#666',
    fontSize: 14,
    marginLeft: 8,
  },
  currentMeterRow: {
    flexDirection: 'row',
    marginTop: 8,
    alignItems: 'center',
  },
  currentMeterLabel: {
    fontSize: 13,
    color: '#666',
  },
  currentMeterValue: {
    fontSize: 13,
    color: '#333',
    fontWeight: '500',
  },
  errorText: {
    color: 'red',
    fontSize: 12,
    marginTop: 5,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 15,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    borderRadius: 5,
  },
  disabledButton: {
    opacity: 0.6,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  checklistModalContent: {
    width: '90%',
    maxHeight: '70%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    elevation: 5,
  },
  checklistModalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 15,
  },
  checklistModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
  },
  modalCloseIcon: {
    padding: 5,
  },
  checklistModalTabs: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 10,
  },
  tabButton: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: 10,
    borderBottomWidth: 2,
    borderBottomColor: 'transparent',
  },
  activeTabButton: {
    borderBottomColor: '#0078D4',
  },
  tabText: {
    fontSize: 16,
    color: '#666',
    fontWeight: '600',
  },
  activeTabText: {
    color: '#0078D4',
  },
  checklistSearchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingVertical: 10,
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    paddingHorizontal: 10,
  },
  checklistSelectText: {
    fontSize: 16,
    color: '#333',
    flex: 1,
  },
  checklistSearchInput: {
    flex: 1,
    height: 40,
    fontSize: 16,
    color: '#333',
  },
  divider: {
    height: 1,
    backgroundColor: '#eee',
    marginVertical: 10,
  },
  checklistList: {
    maxHeight: 200,
  },
  checklistItem: {
    paddingVertical: 10,
  },
  checklistItemContent: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  checklistItemText: {
    fontSize: 16,
    color: '#333',
    marginLeft: 10,
  },
  checklistEmptyText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    marginVertical: 20,
  },
  checklistModalFooter: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
  },
  createChecklistText: {
    fontSize: 16,
    color: '#0078D4',
    fontWeight: '600',
  },
  checklistCloseButton: {
    padding: 10,
  },
  checkboxRow: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 20,
    marginBottom: 8,
    paddingVertical: 8,
  },
  checkbox: {
    width: 24,
    height: 24,
    borderWidth: 2,
    borderColor: '#0078D4',
    borderRadius: 4,
    marginRight: 12,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#fff',
  },
  checkboxChecked: {
    backgroundColor: '#0078D4',
    borderColor: '#0078D4',
  },
  checkboxLabel: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '500',
  },
  checklistCloseText: {
    fontSize: 16,
    color: '#3481BC',
    fontWeight: '600',
  },
  equipmentModalContent: {
    width: '90%',
    maxHeight: '70%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    elevation: 5,
  },
  equipmentModalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
  },
  equipmentModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
  },
  equipmentModalTabs: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 10,
  },
  equipmentSearchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingVertical: 10,
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    paddingHorizontal: 10,
    marginBottom: 10,
  },
  equipmentSearchContainerActive: {
    paddingVertical: 2,
  },
  equipmentSelectText: {
    fontSize: 16,
    color: '#333',
    flex: 1,
  },
  equipmentSearchInput: {
    flex: 1,
    height: 40,
    fontSize: 16,
    color: '#333',
  },
  equipmentSearchInputActive: {
    flex: 1,
    height: 40,
    fontSize: 16,
    color: '#333',
  },
  equipmentList: {
    maxHeight: 300,
    paddingHorizontal: 12,
  },
  equipmentRow: {
    marginVertical: 5,
    backgroundColor: '#F2F2F2',
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#eee',
    paddingRight: 10,
  },
  equipmentInfo: {
    flex: 1,
  },
  equipmentDetailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingHorizontal: '5%',
    paddingRight: '10%',
  },
  equipmentDetailItemNoBorder: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    paddingLeft: '5%',
  },
  equipmentDetailItemheader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: '5%',
  },
  detailLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: 3,
  },
  detailLabelstatuss: {
    fontSize: 14,
    fontWeight: '400',
    width: '60%',
    marginVertical: 3,
  },
  detailLabelheader: {
    fontSize: 18,
    color: '#015185',
    fontWeight: '600',
    width: '100%',
  },
  detailValue: {
    fontSize: 16,
    color: '#015185',
    width: '60%',
    textAlign: 'right',
    fontWeight: '500',
  },
  selectEquipmentButton: {
    position: 'absolute',
    right: 10,
    bottom: 10,
    backgroundColor: '#0078D4',
    paddingVertical: 5,
    paddingHorizontal: 10,
    borderRadius: 5,
  },
  selectEquipmentText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  equipmentEmptyText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    marginVertical: 20,
  },
  equipmentModalFooter: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
  },
  addEquipmentText: {
    fontSize: 16,
    color: '#0078D4',
    fontWeight: '600',
  },
  equipmentCloseButton: {
    padding: 10,
  },
  equipmentCloseText: {
    fontSize: 16,
    color: '#3481BC',
    fontWeight: '600',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 20,
  },
  loadingText: {
    marginTop: 10,
    color: '#666',
    fontSize: 16,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
});

export default DvirInspectioninfoStep1;