import React, { useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  StatusBar,
  ScrollView,
  TouchableOpacity,
  Dimensions,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import DvirDoinspectionStep2 from './DvirDoinspectionStep2';
import DvirSummeryStep3 from './DvirSummeryStep3';
import DvirInspectioninfoStep1 from './DvirInspectioninfoStep1';
import DvirInspecSignature from './DvirInspecSignature';

const { height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = height * 0.2 - 36;

const DvirHomeheader = () => {
  const [activeStep, setActiveStep] = useState(1);
  const [showSignature, setShowSignature] = useState(false);
  const [signatureStatus, setSignatureStatus] = useState('');
  const [signatureData, setSignatureData] = useState(null);

  const steps = [
    {
      id: 1,
      title: 'Step 1',
      subtitle: 'DVIR Info',
      component: <DvirInspectioninfoStep1 setActiveStep={setActiveStep} />,
    },
    {
      id: 2,
      title: 'Step 2',
      subtitle: 'Do DVIR',
      component: <DvirDoinspectionStep2 setActiveStep={setActiveStep} />,
    },
    {
      id: 3,
      title: 'Step 3',
      subtitle: 'Summary',
      component: (
        <DvirSummeryStep3
          setActiveStep={setActiveStep}
          signatureStatus={signatureStatus}
          signatureData={signatureData}
          onSignaturePress={() => setShowSignature(true)}
        />
      ),
    },
  ];

  const handleSaveSignature = (data) => {
    if (data) {
      setSignatureStatus(data.signatureStatus);
      setSignatureData(data.signatureData);
      setShowSignature(false);
      console.log('Signature saved, updating Step 3 with data:', data);
    } else {
      console.log('No data received in handleSaveSignature');
    }
  };

  const handleGoBack = () => {
    setShowSignature(false);
    console.log('Going back from DvirSignature');
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <ScrollView
          horizontal
          showsHorizontalScrollIndicator={false}
          contentContainerStyle={styles.stepsScrollContent}
        >
          {steps.map((step, index) => {
            const isActive = activeStep === step.id;
            const isCompleted = activeStep > step.id;
            return (
              <TouchableOpacity
                key={step.id}
                style={styles.stepWrapper}
                onPress={() => setActiveStep(step.id)}
              >
                <View
                  style={[
                    styles.stepCircle,
                    isActive && styles.activeStepCircle,
                    isCompleted && styles.completedStepCircle,
                  ]}
                >
                  {isCompleted ? (
                    <MaterialIcons name="check" size={20} color="#fff" />
                  ) : (
                    <Text style={[styles.stepNumber, isActive && styles.activeStepNumber]}>
                      {step.id}
                    </Text>
                  )}
                </View>
                <View style={styles.stepTextContainer}>
                  <Text style={[styles.stepTitle, isActive && styles.activeStepTitle]}>
                    {step.title}
                  </Text>
                  <Text style={[styles.stepSubtitle, isActive && styles.activeStepSubtitle]}>
                    {step.subtitle}
                  </Text>
                </View>
                {index < steps.length - 1 && (
                  <View
                    style={[styles.stepConnector, isCompleted && styles.completedStepConnector]}
                  />
                )}
              </TouchableOpacity>
            );
          })}
        </ScrollView>
      </LinearGradient>

      <View style={styles.mainContent}>
        {showSignature ? (
          <DvirInspecSignature
            onSave={handleSaveSignature}
            onGoBack={handleGoBack}
            setActiveStep={setActiveStep}
          />
        ) : (
          steps.find((step) => step.id === activeStep).component
        )}
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    paddingBottom: 15,
  },
  stepsScrollContent: {
    paddingHorizontal: 20,
    alignItems: 'center',
    marginTop: '12%',
  },
  stepWrapper: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 5,
  },
  stepCircle: {
    width: 30,
    height: 30,
    borderRadius: 15,
    backgroundColor: '#eee',
    justifyContent: 'center',
    alignItems: 'center',
    borderWidth: 2,
    borderColor: '#ccc',
  },
  activeStepCircle: {
    borderColor: '#fff',
    backgroundColor: '#fff',
  },
  completedStepCircle: {
    backgroundColor: '#4CAF50',
    borderColor: '#4CAF50',
  },
  stepNumber: {
    color: '#888',
    fontWeight: 'bold',
  },
  activeStepNumber: {
    color: '#025383',
  },
  stepTextContainer: {
    marginHorizontal: 10,
  },
  stepTitle: {
    fontSize: 14,
    color: '#ccc',
    fontWeight: 'bold',
  },
  activeStepTitle: {
    color: '#fff',
  },
  stepSubtitle: {
    fontSize: 12,
    color: '#ccc',
  },
  activeStepSubtitle: {
    color: '#fff',
  },
  stepConnector: {
    width: 30,
    height: 2,
    backgroundColor: '#ccc',
  },
  completedStepConnector: {
    backgroundColor: '#4CAF50',
  },
  mainContent: {
    flex: 1,
  },
});

export default DvirHomeheader;