import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  FlatList,
  TouchableOpacity,
  Dimensions,
  Alert,
  SafeAreaView,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { SafeAreaProvider, useSafeAreaInsets } from 'react-native-safe-area-context';
import database from '@react-native-firebase/database';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');

const BUTTONS = [
  { label: 'Fixed', value: 'Fixed' },
  { label: 'Not Fixed', value: 'Not Fixed' },
  { label: 'No Need', value: 'No Need' },
];

const DvirCorrectiveActionStep1 = ({
  report,
  setActiveStep,
  navigation,
  subItemStatuses,
  setSubItemStatuses,
  resetSubItemStatuses,
}) => {
  const [faultyItems, setFaultyItems] = useState([]);
  const [faultyCount, setFaultyCount] = useState(0);
  const [isSaving, setIsSaving] = useState(false);
  const [isOnline, setIsOnline] = useState(true);

  useEffect(() => {
    // Prefer new structure: report.step2.items (array)
    if (Array.isArray(report?.step2?.items)) {
      const items = report.step2.items
        .filter((item) => item.status && item.status !== '' && item.status !== 'N/A')
        .map((item, idx) => ({
          id: item.itemId || item.id || idx.toString(),
          name: item.label || item.name || '',
          fileName: item.fileName || 'No file',
          number: idx + 1,
          status: subItemStatuses[item.itemId || item.id || idx.toString()]?.status || '',
        }));
      setFaultyItems(items);
      setFaultyCount(items.length);
    } else if (report?.step2?.itemStates) {
      // Fallback for old structure
      const items = Object.keys(report.step2.itemStates)
        .filter((key) => {
          const status = report.step2.itemStates[key].status;
          return status === 'No' || status === 'Fail' || status === 'faulty';
        })
        .map((key, idx) => {
          const [sectionKey, itemId] = key.split('_');
          let itemName = report.step2.itemStates[key].name || `${sectionKey}_${itemId}`;
          return {
            id: key,
            name: itemName,
            fileName: report.step2.itemStates[key].fileName || 'No file',
            number: idx + 1,
            status: subItemStatuses[key]?.status || '',
          };
        });
      setFaultyItems(items);
      setFaultyCount(items.length);
    } else {
      setFaultyItems([]);
      setFaultyCount(0);
    }

    // Load stored corrective action data from Firebase
    const loadStoredData = async () => {
      if (!report?.id) {
        console.warn('No report ID provided for loading data');
        return;
      }
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          Toast.show({
            type: 'error',
            text1: 'Authentication Error',
            text2: 'Please log in again.',
          });
          navigation.replace('LoginScreens');
          return;
        }
        // Read from new path structure
        const correctiveRef = database().ref(`/GlobalDVirCorectiveActionR/${userUid}/${report.id}`);
        const snapshot = await correctiveRef.once('value');
        const correctiveData = snapshot.val();
        console.log('Loading corrective action data for report:', report.id, 'Data found:', !!correctiveData);
        if (correctiveData?.subItemStatuses) {
          console.log('Setting subItemStatuses:', correctiveData.subItemStatuses);
          setSubItemStatuses(correctiveData.subItemStatuses);
        }
        if (correctiveData?.checklistData) {
          console.log('Found checklist data:', correctiveData.checklistData);
        }
      } catch (error) {
        console.error('Error loading corrective action data:', error);
        setIsOnline(false);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to load stored data. Working offline.',
        });
      }
    };
    loadStoredData();

    let unsubscribe;
    const setupNetworkListener = async () => {
      try {
        const { NetInfo } = await import('@react-native-community/netinfo');
        unsubscribe = NetInfo.addEventListener((state) => {
          setIsOnline(state.isConnected ?? false);
        });
      } catch (error) {
        console.error('Error setting up network listener:', error);
        setIsOnline(false);
      }
    };
    setupNetworkListener();

    return () => {
      if (unsubscribe) unsubscribe();
    };
  }, [report, setSubItemStatuses, navigation]);

  const handleStatusChange = (subItemId, status) => {
    setSubItemStatuses((prev) => ({
      ...prev,
      [subItemId]: status,
    }));
  };

  const queueOperation = async (operation) => {
    try {
      let pendingOperations = [];
      const pendingOperationsRaw = await AsyncStorage.getItem('pendingOperations');
      pendingOperations = pendingOperationsRaw ? JSON.parse(pendingOperationsRaw) : [];
      pendingOperations.push({
        ...operation,
        path: `/GlobalDVirCorectiveActionR/${operation.userId}/${operation.reportId}`,
        timestamp: new Date().toISOString()
      });
      await AsyncStorage.setItem('pendingOperations', JSON.stringify(pendingOperations));
      Toast.show({
        type: 'info',
        text1: 'Offline Mode',
        text2: 'Corrective action data queued for sync when online.',
      });
    } catch (error) {
      console.error('Error queuing operation:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to queue corrective action data.',
      });
    }
  };

  const saveToFirebase = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid || !report?.id) return false;

      const correctiveActionData = {
        reportId: report.id,
        userUid: userUid,
        dvirNumber: report.dvirNumber || report.step1?.requestNumber,
        linkedDvirReportId: report.id, // Link to the original DVIR report
        subItemStatuses,
        updatedAt: new Date().toISOString(),
        faultyCount,
        status: 'step1_completed',
        // Include all checklist data with checked items
        checklistData: {
          items: faultyItems.map(item => ({
            id: item.id,
            name: item.name,
            fileName: item.fileName,
            number: item.number,
            status: subItemStatuses[item.id] || '',
            checked: !!subItemStatuses[item.id]
          })),
          totalItems: faultyItems.length,
          checkedItems: Object.keys(subItemStatuses).filter(key => subItemStatuses[key]).length
        },
        // Add tracking fields
        originalReportData: {
          step1: report.step1,
          step2: report.step2,
          step3: report.step3,
        }
      };

      // Save to new path structure
      await database()
        .ref(`/GlobalDVirCorectiveActionR/${userUid}/${report.id}`)
        .update(correctiveActionData);

      // Update local storage
      let storedReports = [];
      const storedData = await AsyncStorage.getItem('inspectionReports');
      storedReports = storedData ? JSON.parse(storedData) : [];
      const updatedReports = storedReports.map((r) =>
        r.id === report.id
          ? { 
              ...r, 
              correctiveAction: { 
                ...r.correctiveAction, 
                hasCorrectiveAction: true,
                correctiveActionId: report.id,
                status: 'step1_completed',
                updatedAt: new Date().toISOString(),
                faultyCount
              } 
            }
          : r
      );
      if (!storedReports.some((r) => r.id === report.id)) {
        updatedReports.push({ 
          ...report, 
          correctiveAction: {
            hasCorrectiveAction: true,
            correctiveActionId: report.id,
            status: 'step1_completed',
            updatedAt: new Date().toISOString(),
            faultyCount
          }
        });
      }
      await AsyncStorage.setItem('inspectionReports', JSON.stringify(updatedReports));

      // Process any pending operations
      let pendingOperations = [];
      const pendingData = await AsyncStorage.getItem('pendingOperations');
      pendingOperations = pendingData ? JSON.parse(pendingData) : [];
      
      for (const operation of pendingOperations) {
        if (operation.type === 'correctiveAction' && operation.reportId === report.id) {
          await database()
            .ref(`/GlobalDVirCorectiveActionR/${userUid}/${operation.reportId}`)
            .update(operation.correctiveAction);
        }
      }
      await AsyncStorage.setItem('pendingOperations', JSON.stringify([]));

      return true;
    } catch (error) {
      console.error('Error in saveToFirebase:', error);
      setIsOnline(false);
      await queueOperation({
        type: 'correctiveAction',
        reportId: report.id,
        userId: await AsyncStorage.getItem('userUid') || 'unknown',
        correctiveAction: { 
          reportId: report.id,
          userUid: await AsyncStorage.getItem('userUid'),
          dvirNumber: report.dvirNumber || report.step1?.requestNumber,
          linkedDvirReportId: report.id,
          subItemStatuses, 
          updatedAt: new Date().toISOString(), 
          faultyCount,
          status: 'step1_completed',
          checklistData: {
            items: faultyItems.map(item => ({
              id: item.id,
              name: item.name,
              fileName: item.fileName,
              number: item.number,
              status: subItemStatuses[item.id] || '',
              checked: !!subItemStatuses[item.id]
            })),
            totalItems: faultyItems.length,
            checkedItems: Object.keys(subItemStatuses).filter(key => subItemStatuses[key]).length
          }
        },
        path: `/GlobalDVirCorectiveActionR/${userUid}/${report.id}`,
      });
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save corrective action data. Saved offline.',
      });
      return false;
    }
  };

  const handleProceed = async () => {
    if (isSaving || !faultyItems.every((item) => subItemStatuses[item.id])) {
      Toast.show({
        type: 'error',
        text1: 'Incomplete Data',
        text2: 'Please select a status for all faulty items.',
      });
      return;
    }
    setIsSaving(true);
    try {
      const saveSuccess = await saveToFirebase();
      if (saveSuccess) {
        setActiveStep(2);
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Corrective action data saved successfully.',
        });
      }
    } catch (error) {
      console.error('Error in handleProceed:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'An unexpected error occurred. Data saved offline.',
      });
    } finally {
      setIsSaving(false);
    }
  };

  const handleCancel = () => {
    if (Object.values(subItemStatuses).some((status) => status)) {
      Alert.alert(
        'Cancel',
        'Are you sure you want to cancel? Your changes will not be saved.',
        [
          { text: 'No', style: 'cancel' },
          {
            text: 'Yes',
            style: 'destructive',
            onPress: () => {
              resetSubItemStatuses();
              navigation.goBack();
            },
          },
        ]
      );
    } else {
      navigation.goBack();
    }
  };

  const renderSubItem = (subItem, index) => {
    const currentStatus = subItemStatuses[subItem.id] || null;
    return (
      <View style={styles.subItemRow} key={subItem.id}>
        <View style={styles.subItemHeaderRow}>
          <Text style={styles.subItemNumber}>{index + 1}.</Text>
          <View style={styles.subItemDetails}>
            <Text style={styles.subItemText}>{subItem.name}</Text>
            <Text style={styles.subItemFileName}>File: {subItem.fileName}</Text>
          </View>
        </View>
        <View style={styles.statusOptionsRow}>
          {BUTTONS.map((btn) => {
            let buttonStyle = styles.statusButton;
            let textStyle = styles.statusButtonText;
            if (currentStatus === btn.value) {
              buttonStyle = [
                styles.statusButton,
                btn.value === 'Fixed'
                  ? styles.statusButtonSelected
                  : btn.value === 'Not Fixed'
                  ? styles.statusButtonNotFixed
                  : styles.statusButtonNoNeed,
              ];
              textStyle = [styles.statusButtonText, styles.statusButtonTextSelected];
            }
            return (
              <TouchableOpacity
                key={btn.value}
                style={buttonStyle}
                onPress={() => handleStatusChange(subItem.id, btn.value)}
              >
                <Text style={textStyle}>{btn.label}</Text>
              </TouchableOpacity>
            );
          })}
        </View>
      </View>
    );
  };

  const insets = useSafeAreaInsets();
  const isProceedEnabled = faultyItems.length > 0 && faultyItems.every((item) => subItemStatuses[item.id]);

  if (!report || !report.step1 || !report.step2) {
    return (
      <SafeAreaProvider>
        <SafeAreaView style={styles.container} edges={['top', 'left', 'right']}>
          <Text style={styles.noItemsText}>No report data available</Text>
        </SafeAreaView>
      </SafeAreaProvider>
    );
  }

  return (
    <SafeAreaProvider>
      <SafeAreaView style={styles.container} edges={['top', 'left', 'right']}>
        <View style={styles.infoRow}>
          <Text style={styles.infoLabel}>Report#</Text>
          <Text style={styles.infoValue}>{report.step1.requestNumber || 'N/A'}</Text>
        </View>
        <View style={styles.infoRow}>
          <Text style={styles.infoLabel}>Vehicle#</Text>
          <Text style={styles.infoValue}>{report.step1.selectedEquipment?.equipmentName || 'N/A'}</Text>
        </View>
        <View style={styles.blueHeaderBar}>
          <Text style={styles.blueHeaderText}>Corrective Action Items ({faultyCount})</Text>
        </View>
        {faultyItems.length > 0 ? (
          <FlatList
            data={faultyItems}
            renderItem={({ item, index }) => renderSubItem(item, index)}
            keyExtractor={(item) => item.id}
            style={styles.checklist}
          />
        ) : (
          <Text style={styles.noItemsText}>No faulty items found</Text>
        )}
        <View style={[styles.bottomButtonBar, { bottom: insets.bottom }]}>
          <TouchableOpacity
            style={styles.bottomBarButtonLeft}
            onPress={handleCancel}
            disabled={isSaving}
          >
            <MaterialIcons name="chevron-left" size={20} color="#1676b7" />
            <Text style={styles.bottomBarButtonText}>Cancel</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.bottomBarButtonRight, !isProceedEnabled && styles.disabledButton]}
            onPress={handleProceed}
            disabled={!isProceedEnabled || isSaving}
          >
            <Text style={[styles.bottomBarButtonText, !isProceedEnabled && styles.bottomBarButtonTextDisabled]}>
              Next
            </Text>
            <MaterialIcons name="chevron-right" size={20} color={isProceedEnabled ? '#1676b7' : '#ccc'} />
          </TouchableOpacity>
        </View>
      </SafeAreaView>
      <Toast />
    </SafeAreaProvider>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  infoRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: width * 0.06,
    paddingTop: width * 0.03,
    paddingBottom: 0,
    borderBottomWidth: 1,
    borderBottomColor: '#e5e5e5',
  },
  infoLabel: {
    color: '#1676b7',
    fontSize: width * 0.038,
    fontWeight: '500',
  },
  infoValue: {
    color: '#1676b7',
    fontSize: width * 0.038,
    fontWeight: '600',
  },
  blueHeaderBar: {
    backgroundColor: '#195c85',
    alignItems: 'center',
    justifyContent: 'center',
    height: width * 0.09,
    marginTop: width * 0.04,
    marginBottom: width * 0.04,
    width: '90%',
    alignSelf: 'center',
  },
  blueHeaderText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: width * 0.045,
  },
  checklist: {
    flex: 1,
    paddingHorizontal: width * 0.06,
  },
  subItemRow: {
    marginBottom: width * 0.09,
    paddingVertical: width * 0.025,
  },
  subItemHeaderRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: width * 0.02,
  },
  subItemNumber: {
    fontWeight: 'bold',
    fontSize: width * 0.04,
    color: '#222',
    marginRight: 4,
  },
  subItemDetails: {
    flex: 1,
  },
  subItemText: {
    fontSize: width * 0.04,
    color: '#222',
    fontWeight: '500',
  },
  subItemFileName: {
    fontSize: width * 0.035,
    color: '#666',
    fontStyle: 'italic',
  },
  statusOptionsRow: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    alignItems: 'center',
    marginTop: width * 0.01,
    marginBottom: width * 0.01,
  },
  statusButton: {
    flex: 1,
    backgroundColor: '#f2f6fa',
    paddingVertical: width * 0.025,
    marginRight: 6,
    alignItems: 'center',
    borderRadius: 2,
  },
  statusButtonSelected: {
    backgroundColor: '#195c85',
  },
  statusButtonNotFixed: {
    backgroundColor: '#dc3545',
  },
  statusButtonNoNeed: {
    backgroundColor: '#808080',
  },
  statusButtonText: {
    color: '#195c85',
    fontWeight: '600',
    fontSize: width * 0.038,
  },
  statusButtonTextSelected: {
    color: '#fff',
  },
  bottomButtonBar: {
    position: 'absolute',
    left: 0,
    right: 0,
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    backgroundColor: '#f5f5f5',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    height: 48,
    paddingHorizontal: width * 0.06,
    zIndex: 10,
  },
  bottomBarButtonLeft: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: 'transparent',
    borderWidth: 0,
    paddingVertical: 8,
    paddingHorizontal: 4,
  },
  bottomBarButtonRight: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: 'transparent',
    borderWidth: 0,
    paddingVertical: 8,
    paddingHorizontal: 4,
  },
  disabledButton: {
    opacity: 0.5,
  },
  bottomBarButtonText: {
    color: '#1676b7',
    fontSize: width * 0.045,
    fontWeight: 'bold',
    textAlign: 'center',
    marginHorizontal: 2,
  },
  bottomBarButtonTextDisabled: {
    color: '#ccc',
  },
  noItemsText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
});

export default DvirCorrectiveActionStep1;