import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  StatusBar,
  Dimensions,
  Modal,
  Animated,
  Image,
  PermissionsAndroid,
  Platform,
  Alert,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import LinearGradient from 'react-native-linear-gradient';
import { useNavigation, useRoute, useFocusEffect } from '@react-navigation/native';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
// Using react-native-image-picker for all file types
import RNFS from 'react-native-fs';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import AsyncStorage from '@react-native-async-storage/async-storage';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.25;

const DocumentuploadS = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { folder, subfolder, onDocumentCreated } = route.params || {};

  const [documentType, setDocumentType] = useState(folder || 'Equipments');
  const [showDocumentDropdown, setShowDocumentDropdown] = useState(false);
  const [type, setType] = useState('Invoice');
  const [showTypeDropdown, setShowTypeDropdown] = useState(false);
  const [description, setDescription] = useState('');
  const [subfolderName, setSubfolderName] = useState(subfolder || '');
  const [selectedFile, setSelectedFile] = useState(null);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [errors, setErrors] = useState({
    documentType: '',
    type: '',
    description: '',
    subfolderName: '',
    selectedFile: '',
    selectedEquipment: '',
  });
  const [isUploading, setIsUploading] = useState(false);
  const [showFileModal, setShowFileModal] = useState(false);
  const [showDeleteModal, setShowDeleteModal] = useState(false);
  const [documentToDelete, setDocumentToDelete] = useState(null);

  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Check internet connectivity and sync offline documents
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      if (state.isConnected && state.isInternetReachable) {
        syncOfflineDocuments();
      }
    });
    
    // Initial check
    NetInfo.fetch().then((state) => {
      if (state.isConnected && state.isInternetReachable) {
        syncOfflineDocuments();
      }
    });
    
    return () => unsubscribe();
  }, []);

  // Debug: Monitor selectedEquipment changes
  useEffect(() => {
    console.log('selectedEquipment state changed:', selectedEquipment);
  }, [selectedEquipment]);

  // Debug: Check route params when screen comes into focus
  useFocusEffect(
    React.useCallback(() => {
      console.log('DocumentUpload screen focused');
      console.log('Current route params:', route.params);
      console.log('Current selectedEquipment state:', selectedEquipment);
    }, [route.params, selectedEquipment])
  );

  // Permission request functions
  const requestCameraPermission = async () => {
    try {
      if (Platform.OS === 'android') {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.CAMERA,
          {
            title: 'Camera Permission',
            message: 'This app needs access to your camera to take photos.',
            buttonPositive: 'OK',
            buttonNegative: 'Cancel',
          }
        );
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      } else if (Platform.OS === 'ios') {
        // For iOS, camera permissions are handled automatically by the system
        // when launchCamera is called, but we can provide better user guidance
        return true; // Return true to allow the camera to launch and handle permissions
      }
      return false;
    } catch (error) {
      console.error('Camera permission error:', error);
      return false;
    }
  };

  const requestStoragePermission = async () => {
    try {
      if (Platform.OS === 'android') {
        const permission =
          Platform.Version >= 33
            ? PermissionsAndroid.PERMISSIONS.READ_MEDIA_IMAGES
            : PermissionsAndroid.PERMISSIONS.READ_EXTERNAL_STORAGE;
        const granted = await PermissionsAndroid.request(permission, {
          title: 'Storage Permission',
          message: 'This app needs access to your storage to select files.',
          buttonPositive: 'OK',
        });
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      }
      return true;
    } catch (error) {
      console.error('Storage permission error:', error);
      return false;
    }
  };

  // Copy file to persistent storage
  const copyFileToPersistentStorage = async (uri, fileName) => {
    try {
      const extension = fileName.split('.').pop().toLowerCase();
      const newFileName = `document_${Date.now()}.${extension}`;
      
      // Platform-specific path handling
      let destPath;
      if (Platform.OS === 'ios') {
        // For iOS, use the Documents directory
        destPath = `${RNFS.DocumentDirectoryPath}/${newFileName}`;
      } else {
        // For Android, use the Documents directory
        destPath = `${RNFS.DocumentDirectoryPath}/${newFileName}`;
      }
      
      // Ensure the destination directory exists
      const destDir = destPath.substring(0, destPath.lastIndexOf('/'));
      if (!(await RNFS.exists(destDir))) {
        await RNFS.mkdir(destDir);
      }
      
      await RNFS.copyFile(uri, destPath);
      return { uri: destPath, name: newFileName };
    } catch (error) {
      console.error('Error copying file to persistent storage:', error);
      throw new Error('Failed to copy file to storage');
    }
  };

  // Store document in AsyncStorage for offline mode
  const storeDocumentOffline = async (documentData, fileInfo) => {
    try {
      const offlineDocs = await AsyncStorage.getItem('offlineDocuments');
      const docs = offlineDocs ? JSON.parse(offlineDocs) : [];
      
      // Add timestamp for offline storage
      const offlineDoc = {
        ...documentData,
        fileUri: fileInfo.uri,
        fileName: fileInfo.name,
        storedAt: new Date().toISOString(),
        platform: Platform.OS,
      };
      
      docs.push(offlineDoc);
      await AsyncStorage.setItem('offlineDocuments', JSON.stringify(docs));
      
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Document stored offline',
      });
      return true;
    } catch (error) {
      console.error('Error storing document offline:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to store document offline',
      });
      return false;
    }
  };

  // Sync offline documents to Firebase
  const syncOfflineDocuments = async () => {
    try {
      const offlineDocs = await AsyncStorage.getItem('offlineDocuments');
      if (!offlineDocs) return;

      const docs = JSON.parse(offlineDocs);
      if (docs.length === 0) return;

      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        console.error('Admin UID not found during sync');
        return;
      }

      for (const doc of docs) {
        try {
          const fileUrl = await uploadFileToStorage(doc.fileUri, doc.documentType, doc.subfolderName);
          const documentData = {
            id: doc.id,
            documentType: doc.documentType,
            type: doc.type,
            description: doc.description,
            subfolderName: doc.subfolderName,
            equipment: doc.equipment,
            file: fileUrl,
            fileType: doc.fileName.split('.').pop().toLowerCase(),
            createdAt: doc.createdAt,
          };

          const documentRef = database().ref(
            `/Documents/${adminUid}/${doc.documentType}/${doc.subfolderName}/${doc.id}`
          );
          await documentRef.set(documentData);
        } catch (error) {
          console.error('Error syncing document:', error);
        }
      }

      await AsyncStorage.setItem('offlineDocuments', JSON.stringify([]));
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Offline documents uploaded to database',
      });
    } catch (error) {
      console.error('Error syncing offline documents:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to sync offline documents',
      });
    }
  };

  const handleEquipmentSelect = (equipmentData) => {
    console.log('Selected Equipment Data:', equipmentData);
    
    // Check if equipmentData exists and has required fields
    if (equipmentData && (
      equipmentData.equipmentName || 
      equipmentData.rigName || 
      equipmentData.vehicleNumber ||
      equipmentData.id
    )) {
      setSelectedEquipment(equipmentData);
      setErrors((prevErrors) => ({ ...prevErrors, selectedEquipment: '' }));
      
      // Get display name for success message
      const displayName = equipmentData.equipmentName || 
                         equipmentData.rigName || 
                         equipmentData.vehicleNumber || 
                         `Equipment ${equipmentData.id}`;
      
      // Show success message
      Toast.show({
        type: 'success',
        text1: 'Equipment Selected',
        text2: displayName,
      });
      
      console.log('Equipment set successfully:', equipmentData);
    } else {
      console.log('Invalid equipment data:', equipmentData);
      setSelectedEquipment(null);
      setErrors((prevErrors) => ({ ...prevErrors, selectedEquipment: 'Please select valid equipment' }));
    }
  };

  const handleFileSelection = async (option) => {
    try {
      let fileInfo;
      if (option === 'camera') {
        try {
          // For iOS, launchCamera will automatically handle permission requests
          const options = {
            mediaType: 'photo',
            maxWidth: Platform.OS === 'ios' ? 1024 : 500,
            maxHeight: Platform.OS === 'ios' ? 1024 : 500,
            quality: Platform.OS === 'ios' ? 0.9 : 0.8,
            includeBase64: false,
            saveToPhotos: false,
            // iOS-specific options
            ...(Platform.OS === 'ios' && {
              allowsEditing: false,
              aspect: [1, 1],
            }),
          };
          
          const response = await launchCamera(options);
          
          if (response.didCancel) {
            console.log('User cancelled camera');
            return;
          } else if (response.errorCode) {
            // Handle specific iOS permission errors
            if (response.errorCode === 'permission' || response.errorCode === 'camera_unavailable') {
              Alert.alert(
                'Camera Permission Required',
                'Please grant camera permission to take photos. Go to Settings > Privacy & Security > Camera and enable access for this app.',
                [
                  { text: 'Cancel', style: 'cancel' },
                  { text: 'Settings', onPress: () => {
                    Alert.alert(
                      'Camera Settings',
                      'Please go to Settings > Privacy & Security > Camera and enable camera access for this app.',
                      [{ text: 'OK' }]
                    );
                  }}
                ]
              );
              return;
            } else if (response.errorCode === 'others') {
              // Handle other iOS-specific errors
              console.log('Camera error details:', response.errorMessage);
              throw new Error(response.errorMessage || 'Camera access error');
            }
            throw new Error(response.errorMessage || 'Camera error');
          }
          
          if (!response.assets || response.assets.length === 0) {
            throw new Error('No image captured');
          }
          
          const asset = response.assets[0];
          const uri = asset.uri;
          const fileName = asset.fileName || `photo_${Date.now()}.jpg`;
          const extension = fileName.split('.').pop().toLowerCase();
          
          if (!['jpg', 'jpeg', 'png', 'gif'].includes(extension)) {
            throw new Error('Unsupported image format. Please select JPG, JPEG, PNG, or GIF.');
          }
          
          fileInfo = await copyFileToPersistentStorage(uri, fileName);
          fileInfo.type = 'image';
          
          // Show success message
          Toast.show({
            type: 'success',
            text1: 'Photo Captured',
            text2: 'Image captured successfully',
          });
        } catch (error) {
          console.error('Camera error:', error);
          
          // Handle iOS-specific permission errors
          if (Platform.OS === 'ios') {
            if (error.message?.includes('permission') || error.message?.includes('camera')) {
              Alert.alert(
                'Camera Permission Required',
                'Please grant camera permission in Settings > Privacy & Security > Camera to take photos.',
                [
                  { text: 'Cancel', style: 'cancel' },
                  { text: 'Open Settings', onPress: () => {
                    Alert.alert(
                      'Camera Settings',
                      'Please go to Settings > Privacy & Security > Camera and enable camera access for this app.',
                      [{ text: 'OK' }]
                    );
                  }}
                ]
              );
            } else {
              // Show generic iOS camera error
              Alert.alert(
                'Camera Error',
                'Unable to access camera. Please check your camera permissions in Settings > Privacy & Security > Camera.',
                [
                  { text: 'Cancel', style: 'cancel' },
                  { text: 'Settings', onPress: () => {
                    Alert.alert(
                      'Camera Settings',
                      'Please go to Settings > Privacy & Security > Camera and enable camera access for this app.',
                      [{ text: 'OK' }]
                    );
                  }}
                ]
              );
            }
          } else {
            Toast.show({
              type: 'error',
              text1: 'Camera Error',
              text2: error.message || 'Failed to capture photo',
            });
          }
          return;
        }
      } else if (option === 'gallery') {
        const granted = await requestStoragePermission();
        if (!granted) {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Storage permission denied',
          });
          return;
        }
        const options = {
          mediaType: 'photo',
          maxWidth: Platform.OS === 'ios' ? 1024 : 500, // Higher quality for iOS
          maxHeight: Platform.OS === 'ios' ? 1024 : 500,
          quality: Platform.OS === 'ios' ? 0.9 : 0.8, // Better quality for iOS
          includeBase64: false,
          selectionLimit: 1, // iOS specific
          // Additional iOS-specific options
          ...(Platform.OS === 'ios' && {
            allowsEditing: false,
            aspect: [1, 1],
          }),
        };
        const response = await launchImageLibrary(options);
        if (response.didCancel) {
          console.log('User cancelled gallery');
          return;
        } else if (response.errorCode) {
          throw new Error(response.errorMessage || 'Gallery error');
        }
        if (!response.assets || response.assets.length === 0) {
          throw new Error('No image selected');
        }
        const uri = response.assets[0].uri;
        const fileName = response.assets[0].fileName || `image_${Date.now()}.jpg`;
        const extension = fileName.split('.').pop().toLowerCase();
        if (!['jpg', 'jpeg', 'png', 'gif'].includes(extension)) {
          throw new Error('Unsupported image format. Please select JPG, JPEG, PNG, or GIF.');
        }
        fileInfo = await copyFileToPersistentStorage(uri, fileName);
        fileInfo.type = 'image';
      } else if (option === 'document') {
        const granted = await requestStoragePermission();
        if (!granted) {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Storage permission denied',
          });
          return;
        }
        try {
          const options = {
            mediaType: 'mixed',
            includeBase64: false,
            selectionLimit: 1,
            // Allow all file types
            ...(Platform.OS === 'ios' && {
              allowsEditing: false,
              quality: 1,
            }),
          };
          
          const response = await launchImageLibrary(options);
          if (response.didCancel) {
            console.log('User cancelled document selection');
            return;
          } else if (response.errorCode) {
            throw new Error(response.errorMessage || 'Document selection error');
          }
          
          if (!response.assets || response.assets.length === 0) {
            throw new Error('No document selected');
          }
          
          const asset = response.assets[0];
          const uri = asset.uri;
          const fileName = asset.fileName || asset.originalPath?.split('/').pop() || `document_${Date.now()}`;
          
          // Get file extension from fileName or URI
          let extension = 'pdf'; // default
          if (fileName.includes('.')) {
            extension = fileName.split('.').pop().toLowerCase();
          } else if (uri.includes('.')) {
            extension = uri.split('.').pop().toLowerCase();
          }
          
          // Check if file type is supported
          const supportedExtensions = ['pdf', 'docx', 'doc', 'xlsx', 'xls', 'ppt', 'pptx', 'txt'];
          if (!supportedExtensions.includes(extension)) {
            // Try to detect from MIME type if available
            if (asset.type) {
              const mimeToExt = {
                'application/pdf': 'pdf',
                'application/vnd.openxmlformats-officedocument.wordprocessingml.document': 'docx',
                'application/msword': 'doc',
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet': 'xlsx',
                'application/vnd.ms-excel': 'xls',
                'application/vnd.ms-powerpoint': 'ppt',
                'application/vnd.openxmlformats-officedocument.presentationml.presentation': 'pptx',
                'text/plain': 'txt',
              };
              extension = mimeToExt[asset.type] || extension;
            }
            
            if (!supportedExtensions.includes(extension)) {
              throw new Error(`File type .${extension} is not supported. Please select PDF, DOCX, XLSX, PPT, or TXT files.`);
            }
          }
          
          // Ensure fileName has proper extension
          const finalFileName = fileName.includes('.') ? fileName : `${fileName}.${extension}`;
          
          fileInfo = await copyFileToPersistentStorage(uri, finalFileName);
          fileInfo.type = extension;
          
          // Show file type info
          Toast.show({
            type: 'success',
            text1: 'Document Selected',
            text2: `${finalFileName} (${extension.toUpperCase()})`,
          });
        } catch (error) {
          if (error instanceof Error) {
            Alert.alert('Error', error.message);
          } else {
            console.log('Document selection error:', error);
            Alert.alert('Error', 'An error occurred while selecting the document.');
          }
          return;
        }
      }

      if (fileInfo) {
        setSelectedFile(fileInfo);
        setErrors((prevErrors) => ({ ...prevErrors, selectedFile: '' }));
        setShowFileModal(false);
        
        // Show success message
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: `File selected: ${fileInfo.name}`,
        });
      }
    } catch (error) {
      console.error('File selection error:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: error.message || 'Failed to select file',
      });
    }
  };

  const uploadFileToStorage = async (uri, docType, subfolder) => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        throw new Error('Admin UID not found');
      }
      
      // Platform-specific URI handling
      let uploadUri = uri;
      if (Platform.OS === 'ios' && uri.startsWith('file://')) {
        // Remove file:// prefix for iOS
        uploadUri = uri.replace('file://', '');
      }
      
      // Ensure file exists before upload
      if (Platform.OS === 'ios') {
        const fileExists = await RNFS.exists(uploadUri);
        if (!fileExists) {
          throw new Error('File not found on device');
        }
      }
      
      const extension = uploadUri.split('.').pop().toLowerCase();
      const sanitizedSubfolderName = subfolder.replace(/[^a-zA-Z0-9-_]/g, '_');
      const fileName = `documents/${adminUid}/${docType}/${sanitizedSubfolderName}/${Date.now()}.${extension}`;
      const reference = storage().ref(fileName);
      
      // Get proper MIME type for different file types
      const getMimeType = (ext) => {
        const mimeTypes = {
          'pdf': 'application/pdf',
          'docx': 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
          'doc': 'application/msword',
          'xlsx': 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
          'xls': 'application/vnd.ms-excel',
          'ppt': 'application/vnd.ms-powerpoint',
          'pptx': 'application/vnd.openxmlformats-officedocument.presentationml.presentation',
          'txt': 'text/plain',
        };
        return mimeTypes[ext] || `application/${ext}`;
      };
      
      // Platform-specific upload options
      const uploadOptions = Platform.OS === 'ios' ? {
        cacheControl: 'public,max-age=300',
        contentType: getMimeType(extension),
      } : {
        contentType: getMimeType(extension),
      };
      
      // Show upload progress
      Toast.show({
        type: 'info',
        text1: 'Uploading',
        text2: `Uploading ${extension.toUpperCase()} file...`,
      });
      
      await reference.putFile(uploadUri, uploadOptions);
      const downloadURL = await reference.getDownloadURL();
      return downloadURL;
    } catch (error) {
      console.error('Error uploading file:', error);
      throw error;
    }
  };

  const handleUpload = async () => {
    let hasErrors = false;
    const newErrors = {
      documentType: '',
      type: '',
      description: '',
      subfolderName: '',
      selectedFile: '',
      selectedEquipment: '',
    };

    if (!documentType) {
      newErrors.documentType = 'Document type is required';
      hasErrors = true;
    }
    if (!type) {
      newErrors.type = 'Type is required';
      hasErrors = true;
    }
    if (!description.trim()) {
      newErrors.description = 'Description is required';
      hasErrors = true;
    }
    if (!subfolderName.trim()) {
      newErrors.subfolderName = 'Subfolder name is required';
      hasErrors = true;
    }
    if (!selectedFile) {
      newErrors.selectedFile = 'File selection is required';
      hasErrors = true;
    }
    if (!selectedEquipment || !selectedEquipment.id) {
      newErrors.selectedEquipment = 'Equipment selection is required';
      hasErrors = true;
    }

    setErrors(newErrors);

    if (hasErrors) {
      // Show error toast
      Toast.show({
        type: 'error',
        text1: 'Validation Error',
        text2: 'Please fill in all required fields',
      });
      return;
    }

    // Additional validation
    if (description.trim().length < 3) {
      Toast.show({
        type: 'error',
        text1: 'Validation Error',
        text2: 'Description must be at least 3 characters long',
      });
      return;
    }

    if (subfolderName.trim().length < 2) {
      Toast.show({
        type: 'error',
        text1: 'Validation Error',
        text2: 'Subfolder name must be at least 2 characters long',
      });
      return;
    }

    setIsUploading(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    const documentData = {
      id: Date.now().toString(),
      documentType,
      type,
      description,
      subfolderName,
      equipment: selectedEquipment,
      fileType: selectedFile.type,
      fileName: selectedFile.name,
      createdAt: new Date().toISOString(),
    };

    try {
      const state = await NetInfo.fetch();
      if (state.isConnected && state.isInternetReachable) {
        const fileUrl = await uploadFileToStorage(selectedFile.uri, documentType, subfolderName);
        documentData.file = fileUrl;

        const adminUid = await AsyncStorage.getItem('userUid');
        if (!adminUid) {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Admin not authenticated. Please log in again.',
          });
          navigation.navigate('AdminLoginScreen');
          return;
        }

        const documentRef = database().ref(
          `/Documents/${adminUid}/${documentType}/${subfolderName}/${documentData.id}`
        );
        await documentRef.set(documentData);

        console.log('Document saved:', documentData);
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Document uploaded to database',
        });

        if (onDocumentCreated) {
          onDocumentCreated();
        }
        navigation.goBack();
      } else {
        const success = await storeDocumentOffline(documentData, selectedFile);
        if (success) {
          if (onDocumentCreated) {
            onDocumentCreated();
          }
          navigation.goBack();
        }
      }
    } catch (error) {
      console.error('Error uploading document:', error);
      
      // Platform-specific error handling
      let errorMessage = 'Failed to upload document. Stored offline.';
      if (Platform.OS === 'ios') {
        if (error.message.includes('network')) {
          errorMessage = 'Network error. Document stored offline.';
        } else if (error.message.includes('permission')) {
          errorMessage = 'Permission denied. Document stored offline.';
        }
      }
      
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
      });
      
      const success = await storeDocumentOffline(documentData, selectedFile);
      if (success) {
        if (onDocumentCreated) {
          onDocumentCreated();
        }
        navigation.goBack();
      }
    } finally {
      setIsUploading(false);
      rotateAnim.setValue(0);
    }
  };

  // Handle document deletion
  const handleDeleteDocument = () => {
    if (!selectedFile) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'No document selected to delete',
      });
      return;
    }
    
    setDocumentToDelete({
      fileName: selectedFile.name,
      fileType: selectedFile.type,
      description: description,
    });
    setShowDeleteModal(true);
  };

  // Confirm delete document
  const confirmDeleteDocument = () => {
    try {
      // Clear all form data
      setSelectedFile(null);
      setDescription('');
      setSubfolderName('');
      setType('Invoice');
      setDocumentType('Equipments');
      setSelectedEquipment(null);
      
      // Clear all errors
      setErrors({
        documentType: '',
        type: '',
        description: '',
        subfolderName: '',
        selectedFile: '',
        selectedEquipment: '',
      });

      // Delete the file from storage if it exists
      if (selectedFile && selectedFile.uri) {
        RNFS.unlink(selectedFile.uri).catch(error => {
          console.log('File deletion error:', error);
        });
      }

      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Document cleared successfully',
      });

      setShowDeleteModal(false);
      setDocumentToDelete(null);
    } catch (error) {
      console.error('Error deleting document:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to clear document',
      });
    }
  };

  // Cancel delete document
  const cancelDeleteDocument = () => {
    setShowDeleteModal(false);
    setDocumentToDelete(null);
  };

  return (
    <View style={styles.container}>
      <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Document Upload</Text>
          </View>
          <View style={styles.headerRight}>
            {selectedFile && (
              <TouchableOpacity style={styles.deleteButton} onPress={handleDeleteDocument}>
                <MaterialIcons name="delete" size={20} color="#fff" />
                <Text style={styles.deleteButtonText}>Clear</Text>
              </TouchableOpacity>
            )}
            <TouchableOpacity style={styles.uploadButton} onPress={handleUpload}>
              <Text style={styles.uploadText}>Upload</Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <View style={styles.formContainer}>
        {/* Document Type Dropdown */}
        <View style={styles.row}>
          <Text style={styles.label}>Document of</Text>
          <View style={styles.inputContainer}>
            <TouchableOpacity
              style={[styles.dropdownButton, errors.documentType ? { borderColor: '#FF0000' } : null]}
              onPress={() => setShowDocumentDropdown(!showDocumentDropdown)}
              disabled={!!folder} // Disable if folder is pre-filled
            >
              <Text style={styles.dropdownButtonText}>{documentType}</Text>
              <MaterialIcons
                name="keyboard-arrow-down"
                size={24}
                color={errors.documentType ? '#FF0000' : '#015185'}
              />
            </TouchableOpacity>
            {showDocumentDropdown && (
              <View style={styles.dropdownList}>
                <TouchableOpacity
                  style={styles.dropdownItem}
                  onPress={() => {
                    setDocumentType('Equipments');
                    setShowDocumentDropdown(false);
                    setErrors((prevErrors) => ({ ...prevErrors, documentType: '' }));
                  }}
                >
                  <Text style={[styles.dropdownItemText, { color: '#015185' }]}>Equipments</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={styles.dropdownItem}
                  onPress={() => {
                    setDocumentType('Team');
                    setShowDocumentDropdown(false);
                    setErrors((prevErrors) => ({ ...prevErrors, documentType: '' }));
                  }}
                >
                  <Text style={[styles.dropdownItemText, { color: '#015185' }]}>Team</Text>
                </TouchableOpacity>
              </View>
            )}
            {errors.documentType && <Text style={styles.errorText}>{errors.documentType}</Text>}
          </View>
        </View>

        {/* Type Dropdown */}
        <View style={styles.row}>
          <Text style={styles.label}>Type</Text>
          <View style={styles.inputContainer}>
            <TouchableOpacity
              style={[styles.dropdownButton, errors.type ? { borderColor: '#FF0000' } : null]}
              onPress={() => setShowTypeDropdown(!showTypeDropdown)}
            >
              <Text style={styles.dropdownButtonText}>{type}</Text>
              <MaterialIcons
                name="keyboard-arrow-down"
                size={24}
                color={errors.type ? '#FF0000' : '#015185'}
              />
            </TouchableOpacity>
            {showTypeDropdown && (
              <View style={styles.dropdownList}>
                {[
                  'Invoice',
                  'Insurance',
                  'Registration',
                  'User Manual',
                  'Warranty',
                  'Purchase Order',
                  'Pollution Certificate',
                  'Miscellaneous',
                ].map((item) => (
                  <TouchableOpacity
                    key={item}
                    style={styles.dropdownItem}
                    onPress={() => {
                      setType(item);
                      setShowTypeDropdown(false);
                      setErrors((prevErrors) => ({ ...prevErrors, type: '' }));
                    }}
                  >
                    <Text style={[styles.dropdownItemText, { color: '#015185' }]}>{item}</Text>
                  </TouchableOpacity>
                ))}
              </View>
            )}
            {errors.type && <Text style={styles.errorText}>{errors.type}</Text>}
          </View>
        </View>


           {/* Subfolder Name Input */}
        <View style={styles.inputGroup}>
          <Text style={styles.inputLabel}>Subfolder Name *</Text>
          <TextInput
            style={[styles.input, errors.subfolderName ? { borderColor: '#FF0000' } : null]}
            value={subfolderName}
            onChangeText={(text) => {
              setSubfolderName(text);
              if (text.trim()) {
                setErrors((prevErrors) => ({ ...prevErrors, subfolderName: '' }));
              }
            }}
            placeholder="Enter subfolder name (e.g., Maintenance, Repairs)"
            editable={!subfolder} // Disable if subfolder is pre-filled
            maxLength={50}
          />
          {errors.subfolderName && <Text style={styles.errorText}>{errors.subfolderName}</Text>}
        </View>
        
        {/* Description Input */}
        <View style={styles.inputGroup}>
          <Text style={styles.inputLabel}>Description *</Text>
          <TextInput
            style={[styles.input, styles.textArea, errors.description ? { borderColor: '#FF0000' } : null]}
            value={description}
            onChangeText={(text) => {
              setDescription(text);
              if (text.trim()) {
                setErrors((prevErrors) => ({ ...prevErrors, description: '' }));
              }
            }}
            placeholder="Enter document description (e.g., Monthly maintenance report, Equipment manual)"
            multiline
            numberOfLines={4}
            maxLength={200}
          />
          {errors.description && <Text style={styles.errorText}>{errors.description}</Text>}
        </View>

     

        {/* File Selection Button */}
        <View>
          <Text style={styles.inputLabel}>File *</Text>
          <TouchableOpacity
            style={[styles.selectionButton, errors.selectedFile ? { borderColor: '#FF0000' } : null]}
            onPress={() => setShowFileModal(true)}
          >
            <Text style={styles.selectionButtonText}>Select File, Picture, or Document</Text>
            <MaterialIcons
              name="attach-file"
              size={24}
              color={errors.selectedFile ? '#FF0000' : '#015185'}
            />
          </TouchableOpacity>
          {selectedFile && (
            <View style={styles.selectedFileInfo}>
              <View style={styles.fileInfoLeft}>
                <MaterialIcons name="check-circle" size={20} color="#00A100" />
                <View style={styles.fileInfoText}>
                  <Text style={styles.selectedText} numberOfLines={1} ellipsizeMode="middle">
                    {selectedFile.name}
                  </Text>
                  <Text style={styles.fileTypeText}>
                    Type: {selectedFile.type.toUpperCase()}
                  </Text>
                </View>
              </View>
              <TouchableOpacity
                style={styles.quickDeleteButton}
                onPress={handleDeleteDocument}
              >
                <MaterialIcons name="close" size={18} color="#FF4444" />
              </TouchableOpacity>
            </View>
          )}
          {errors.selectedFile && <Text style={styles.errorText}>{errors.selectedFile}</Text>}
        </View>

        {/* Equipment Selection Button */}
        <View>
          <Text style={styles.inputLabel}>Equipment *</Text>
          <TouchableOpacity
            style={[styles.selectionButton, errors.selectedEquipment ? { borderColor: '#FF0000' } : null]}
            onPress={() => {
              console.log('Navigating to EquipmentselectD with params:', {
                source: 'DocumentUpload',
                onSelect: handleEquipmentSelect,
              });
              navigation.navigate('EquipmentselectD', {
                source: 'DocumentUpload',
                onSelect: handleEquipmentSelect,
              });
            }}
          >
            <Text style={styles.selectionButtonText}>Select Equipment</Text>
            <MaterialIcons
              name="menu"
              size={24}
              color={errors.selectedEquipment ? '#FF0000' : '#015185'}
            />
          </TouchableOpacity>
          {selectedEquipment && (
            <View style={styles.selectedFileInfo}>
              <MaterialIcons name="check-circle" size={20} color="#00A100" />
              <Text style={styles.selectedText} numberOfLines={1} ellipsizeMode="middle">
                {selectedEquipment.equipmentName || 
                 selectedEquipment.rigName || 
                 selectedEquipment.vehicleNumber || 
                 `Equipment ${selectedEquipment.id}` || 
                 'Equipment Selected'}
              </Text>
              {/* Debug: Show equipment data */}
              <Text style={styles.debugText}>
                ID: {selectedEquipment.id || 'N/A'}
              </Text>
            </View>
          )}
          {errors.selectedEquipment && <Text style={styles.errorText}>{errors.selectedEquipment}</Text>}
        </View>
      </View>

      {/* File Selection Modal */}
      <Modal
        transparent
        visible={showFileModal}
        animationType="slide"
        onRequestClose={() => setShowFileModal(false)}
      >
        <TouchableOpacity
          style={styles.modalOverlay}
          activeOpacity={1}
          onPress={() => setShowFileModal(false)}
        >
          <View style={styles.fileModalContainer}>
            <TouchableOpacity style={styles.fileModalOption} onPress={() => handleFileSelection('camera')}>
              <View style={styles.fileModalOptionContent}>
                <MaterialIcons name="camera-alt" size={24} color="#015185" style={styles.fileModalIcon} />
                <View style={styles.cameraOptionText}>
                  <Text style={styles.fileModalText}>Take Picture</Text>
                  {Platform.OS === 'ios' && (
                    <Text style={styles.cameraPermissionHint}>
                      Camera permission will be requested
                    </Text>
                  )}
                </View>
              </View>
            </TouchableOpacity>
            <TouchableOpacity style={styles.fileModalOption} onPress={() => handleFileSelection('gallery')}>
              <View style={styles.fileModalOptionContent}>
                <MaterialIcons name="photo-library" size={24} color="#015185" style={styles.fileModalIcon} />
                <Text style={styles.fileModalText}>Choose from Gallery</Text>
              </View>
            </TouchableOpacity>
            <TouchableOpacity style={styles.fileModalOption} onPress={() => handleFileSelection('document')}>
              <View style={styles.fileModalOptionContent}>
                <MaterialIcons name="folder" size={24} color="#015185" style={styles.fileModalIcon} />
                <Text style={styles.fileModalText}>Choose Document</Text>
              </View>

              
            </TouchableOpacity>
            <TouchableOpacity style={styles.fileModalOption} onPress={() => setShowFileModal(false)}>
              <Text style={[styles.fileModalText, { color: '#FF0000' }]}>Cancel</Text>
            </TouchableOpacity>
          </View>
        </TouchableOpacity>
      </Modal>

      {/* Upload Loader Modal */}
      <Modal visible={isUploading} transparent animationType="fade">
        <View style={styles.loaderModalContainer}>
          <View style={styles.loaderModalContent}>
            <Text style={styles.loaderModalText}>Uploading Document</Text>
            <View style={styles.loaderIconDiv}>
              <Animated.View style={[styles.loaderIcon, { transform: [{ rotate: rotateInterpolate }] }]}>
                <MaterialIcons name="sync" size={30} color="#015185" />
              </Animated.View>
              <Text style={styles.loaderModalTextAdd}>Processing...</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* Delete Confirmation Modal */}
      <Modal visible={showDeleteModal} transparent animationType="fade">
        <View style={styles.modalContainer}>
          <View style={styles.modalContent}>
            <View style={styles.modalHeader}>
              <MaterialIcons name="delete-forever" size={40} color="#FF4444" />
              <Text style={styles.modalTitle}>Clear Document</Text>
            </View>
            <Text style={styles.modalSubText}>
              Are you sure you want to clear this document?{'\n'}
              <Text style={styles.modalHighlight}>
                "{documentToDelete?.fileName || 'Document'}"
              </Text>{'\n'}
              This will remove the file and clear all form data.
            </Text>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={cancelDeleteDocument}
              >
                <Text style={styles.cancelButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.modalButton, styles.deleteButton]}
                onPress={confirmDeleteDocument}
              >
                <MaterialIcons name="delete" size={20} color="#fff" />
                <Text style={styles.deleteButtonText}>Clear All</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  inputGroup: {
    marginBottom: 16,
  },
  inputLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
    marginBottom: 8,
  },
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  uploadButton: {
    backgroundColor: '#fff',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    borderRadius: 5,
  },
  uploadText: {
    color: '#015185',
    fontSize: width * 0.035,
    fontWeight: 'bold',
  },
  deleteButton: {
    backgroundColor: '#FF4444',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    borderRadius: 5,
    marginRight: 10,
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  deleteButtonText: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: 'bold',
    marginLeft: 5,
  },
  formContainer: {
    padding: 16,
  },
  row: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 16,
  },
  label: {
    width: 100,
    fontSize: 16,
    color: '#015185',
  },
  inputContainer: {
    flex: 1,
    position: 'relative',
  },
  dropdownButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 4,
    paddingHorizontal: 12,
    paddingVertical: 8,
    backgroundColor: '#fff',
  },
  dropdownButtonText: {
    fontSize: 16,
    color: '#333',
  },
  dropdownList: {
    position: 'absolute',
    top: '100%',
    left: 0,
    right: 0,
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 4,
    zIndex: 1000,
    elevation: 5,
  },
  dropdownItem: {
    paddingVertical: 8,
    paddingHorizontal: 12,
  },
  dropdownItemText: {
    fontSize: 16,
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 4,
    paddingHorizontal: 12,
    paddingVertical: 12,
    fontSize: 16,
    color: '#333',
    backgroundColor: '#fff',
  },
  textArea: {
    height: 100,
    textAlignVertical: 'top',
    marginBottom: 4,
  },
  selectionButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 4,
    paddingHorizontal: 12,
    paddingVertical: 12,
    marginBottom: 4,
    backgroundColor: '#fff',
  },
  selectionButtonText: {
    fontSize: 16,
    color: '#666',
  },
  selectedText: {
    fontSize: 14,
    color: '#015185',
    marginTop: 4,
    marginBottom: 4,
    marginLeft: 12,
  },
  fileInfoText: {
    flex: 1,
    marginLeft: 8,
  },
  fileTypeText: {
    fontSize: 12,
    color: '#666',
    marginLeft: 12,
    fontStyle: 'italic',
  },
  selectedFileInfo: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginTop: 8,
    marginBottom: 16,
    paddingHorizontal: 12,
    paddingVertical: 8,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  fileInfoLeft: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  quickDeleteButton: {
    padding: 8,
    borderRadius: 20,
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#FF4444',
    marginLeft: 10,
  },
  errorText: {
    fontSize: 12,
    color: '#FF0000',
    marginTop: 4,
    marginLeft: 0,
    fontWeight: '400',
  },
  loaderModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loaderModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  loaderModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  loaderIconDiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  loaderIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  loaderModalTextAdd: {
    fontSize: 14,
    color: '#333',
  },
  modalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '85%',
    backgroundColor: 'white',
    borderRadius: 20,
    padding: 25,
    alignItems: 'center',
    elevation: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
  },
  modalHeader: {
    alignItems: 'center',
    marginBottom: 20,
  },
  modalTitle: {
    fontSize: 22,
    fontWeight: 'bold',
    color: '#333',
    marginTop: 10,
    textAlign: 'center',
  },
  modalSubText: {
    fontSize: 16,
    color: '#666',
    marginBottom: 25,
    textAlign: 'center',
    lineHeight: 24,
  },
  modalHighlight: {
    color: '#FF4444',
    fontWeight: 'bold',
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
    gap: 15,
  },
  modalButton: {
    flex: 1,
    paddingVertical: 15,
    paddingHorizontal: 20,
    borderRadius: 10,
    alignItems: 'center',
    justifyContent: 'center',
    flexDirection: 'row',
    minHeight: 50,
  },
  cancelButton: {
    backgroundColor: '#f8f9fa',
    borderWidth: 2,
    borderColor: '#dee2e6',
  },
  cancelButtonText: {
    color: '#6c757d',
    fontSize: 16,
    fontWeight: '600',
  },
  deleteButton: {
    backgroundColor: '#FF4444',
  },
  deleteButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 5,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  fileModalContainer: {
    backgroundColor: '#fff',
    borderRadius: 20,
    padding: 20,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 3.84,
    width: '80%',
    maxWidth: 400,
  },
  fileModalOption: {
    paddingVertical: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    alignItems: 'center',
  },
  fileModalOptionContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  fileModalIcon: {
    marginRight: 10,
  },
  fileModalText: {
    fontSize: 18,
    color: '#015185',
  },
  cameraOptionText: {
    alignItems: 'center',
  },
  cameraPermissionHint: {
    fontSize: 12,
    color: '#666',
    fontStyle: 'italic',
    marginTop: 2,
  },
  debugText: {
    fontSize: 10,
    color: '#999',
    fontStyle: 'italic',
    marginTop: 2,
    marginLeft: 12,
  },
});

export default DocumentuploadS;