import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Dimensions,
  Image,
  Animated,
  Modal,
  FlatList,
  Alert,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const DailyIns_report_home = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [activeTab, setActiveTab] = useState('running');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [reports, setReports] = useState([]);
  const [filteredReports, setFilteredReports] = useState([]);
  const [isOnline, setIsOnline] = useState(true);
  const [menuVisible, setMenuVisible] = useState(null);
  const [menuPosition, setMenuPosition] = useState({ x: 0, y: 0 });
  const [menuWidth, setMenuWidth] = useState(200); // Default dropdown width
  const [permissionModalVisible, setPermissionModalVisible] = useState(false);
  const [userPermissions, setUserPermissions] = useState(null);
  const [isLoadingPermissions, setIsLoadingPermissions] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Fetch reports from Firebase
  useEffect(() => {
    let unsubscribeNetInfo;

    const fetchReports = async () => {
      try {
        const user = auth().currentUser;
        if (!user) {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'User not authenticated.',
            position: 'bottom',
          });
          return;
        }

        const userId = user.uid;
        const inspectionsRef = database().ref(`/GlobalDailyreportdata/admins/${userId}`);
        inspectionsRef.on('value', (snapshot) => {
          const inspections = snapshot.val() || {};
          const reportList = [];

          for (const inspId in inspections) {
            const inspection = inspections[inspId];
            const status = inspection.step3?.pdfDownloadUrl ? 'submitted' : 'running';

            reportList.push({
              id: inspId,
              requestNumber: inspection.step1?.requestNumber || 'N/A',
              equipmentName: inspection.step1?.selectedEquipment?.equipmentName || 'N/A',
              createdAt: inspection.step3?.createdAt || '',
              status: status,
              pdfUrl: inspection.step3?.pdfDownloadUrl || null,
              generatedAt: inspection.step3?.createdAt || null,
            });
          }

          reportList.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
          setReports(reportList);
          filterReports(reportList, searchQuery, activeTab);
        });

        // Check network status
        const netInfo = await NetInfo.fetch();
        setIsOnline(netInfo.isConnected);

        unsubscribeNetInfo = NetInfo.addEventListener((state) => {
          setIsOnline(state.isConnected);
        });
      } catch (error) {
        console.error('Error fetching reports:', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to load reports.',
          position: 'bottom',
        });
      }
    };

    fetchReports();

    return () => {
      const user = auth().currentUser;
      if (user) {
        database().ref(`/GlobalDailyreportdata/admins/${user.uid}`).off();
      }
      if (unsubscribeNetInfo) {
        unsubscribeNetInfo();
      }
    };
  }, []);

  // Filter reports based on search query and active tab
  const filterReports = (reportList, query, tab) => {
    const today = new Date();
    const thirtyDaysAgo = new Date(today);
    thirtyDaysAgo.setDate(today.getDate() - 30);

    let filtered = reportList.filter((report) => {
      // Search query matching
      const matchesQuery =
        report.requestNumber.toLowerCase().includes(query.toLowerCase()) ||
        report.equipmentName.toLowerCase().includes(query.toLowerCase());

      // Tab filtering
      let matchesTab = true;
      if (tab === 'submitted') {
        // Only show submitted reports older than 30 days
        matchesTab = report.status === 'submitted' && report.createdAt && new Date(report.createdAt) <= thirtyDaysAgo;
      } else if (tab === 'running') {
        // Show all reports (running and submitted) in the Running tab
        matchesTab = true;
      }

      return matchesQuery && matchesTab;
    });
    setFilteredReports(filtered);
  };

  // Handle search query change
  useEffect(() => {
    filterReports(reports, searchQuery, activeTab);
  }, [searchQuery, activeTab, reports]);

  // Handle cloud sync press
  const handleCloudPress = () => {
    if (!isOnline) {
      Toast.show({
        type: 'error',
        text1: 'Offline',
        text2: 'Cannot sync data while offline.',
        position: 'bottom',
      });
      return;
    }

    setSyncModalVisible(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    // Simulate sync (replace with actual sync logic if needed)
    setTimeout(() => {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      Toast.show({
        type: 'success',
        text1: 'Sync Complete',
        text2: 'Data synced successfully.',
        position: 'bottom',
      });
    }, 2000);
  };

  // Handle view PDF
  const handleViewPDF = (report) => {
    if (!isOnline) {
      Toast.show({
        type: 'error',
        text1: 'Offline',
        text2: 'Cannot view PDF while offline.',
        position: 'bottom',
      });
      return;
    }

    if (!report.pdfUrl) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'No PDF available.',
        position: 'bottom',
      });
      return;
    }

    setMenuVisible(null);
    navigation.navigate('PdfviewPage', {
      pdfUrl: report.pdfUrl,
      onGoBack: () => navigation.navigate('DailyIns_report_home'),
    });
  };

  // Handle delete report
  const handleDeleteReport = (reportId) => {
    setMenuVisible(null);
    Alert.alert(
      'Delete Report',
      'Are you sure you want to delete this report? This action cannot be undone.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            try {
              const user = auth().currentUser;
              if (!user) {
                Toast.show({
                  type: 'error',
                  text1: 'Error',
                  text2: 'User not authenticated.',
                  position: 'bottom',
                });
                return;
              }

              const userId = user.uid;
              await database().ref(`/GlobalDailyreportdata/admins/${userId}/${reportId}`).remove();
              Toast.show({
                type: 'success',
                text1: 'Success',
                text2: 'Report deleted successfully.',
                position: 'bottom',
              });
            } catch (error) {
              console.error('Error deleting report:', error);
              Toast.show({
                type: 'error',
                text1: 'Error',
                text2: 'Failed to delete report.',
                position: 'bottom',
              });
            }
          },
        },
      ]
    );
  };

  // Handle edit report
  const handleEditReport = (report) => {
    if (report.status !== 'running') {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Only running reports can be edited.',
        position: 'bottom',
      });
      return;
    }
    setMenuVisible(null);
    navigation.navigate('Daily_Ins_Header', { inspectionId: report.id });
  };

  // Fetch user permissions
  const fetchUserPermissions = async () => {
    try {
      setIsLoadingPermissions(true);
      const user = auth().currentUser;
      const userUid = await AsyncStorage.getItem('userUid'); // Retrieve userUid from AsyncStorage

      if (!user || !userUid) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'User not authenticated or UID not found.',
          position: 'bottom',
        });
        return;
      }

      // Fetch user profile and permissions from the correct Firebase path
      const userRef = database().ref(`/Globalusers/admins/${userUid}`);
      const snapshot = await userRef.once('value');
      const userData = snapshot.val() || {};

      // Transform modules array into a flat object
      const modules = {};
      (userData.modules || []).forEach((module) => {
        modules[module.module] = module.permissions && module.permissions.length > 0;
      });

      // Structure permissions data to include country, project, and daily report permission
      setUserPermissions({
        profile: {
          country: Array.isArray(userData.countries) && userData.countries.length > 0
            ? userData.countries.join(', ')
            : (userData.selectedCountry || userData.country || (userData.profile && userData.profile.country) || 'Not Set'),
          project: Array.isArray(userData.projects) && userData.projects.length > 0
            ? userData.projects.join(', ')
            : (userData.selectedProject || userData.project || (userData.profile && userData.profile.project) || 'Not Set'),
          email: user.email || 'Not Available',
          uid: userUid,
        },
        dailyReportEnabled: !!modules['dailyReport'],
      });
      setPermissionModalVisible(true);
    } catch (error) {
      console.error('Error fetching permissions:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load permissions.',
        position: 'bottom',
      });
    } finally {
      setIsLoadingPermissions(false);
    }
  };

  // Handle permission icon press
  const handlePermissionPress = () => {
    fetchUserPermissions();
  };

  // Helper: check if a report exists for today
  const today = new Date();
  const todayStr = today.toISOString().slice(0, 10);
  // Change here: check all reports, not just filteredReports
  const hasTodayReport = reports.some(r => {
    if (!r.createdAt) return false;
    const d = new Date(r.createdAt);
    return d.toISOString().slice(0, 10) === todayStr;
  });

  // Render report item
  const renderReportItem = ({ item }) => {
    const reportDate = item.createdAt ? new Date(item.createdAt) : null;
    const dayNames = ['M', 'T', 'W', 'T', 'F', 'S', 'S'];
    const dayIdx = reportDate ? (reportDate.getDay() === 0 ? 6 : reportDate.getDay() - 1) : null;
    const formattedDate = reportDate
      ? `${reportDate.getDate().toString().padStart(2, '0')}-${reportDate.toLocaleString('en-US', { month: 'short' })}-${reportDate.getFullYear()} ${reportDate.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit', hour12: true })}`
      : 'N/A';
    return (
      <View style={styles.reportCardCustom}>
        <View style={styles.reportRowCustom}>
          <Text style={styles.reportLabelCustom}>Report#:</Text>
          <Text style={styles.reportValueCustom}>{item.requestNumber}</Text>
        </View>
        <View style={styles.dividerCustom} />
        <View style={styles.reportRowCustom}>
          <Text style={styles.reportLabelCustom}>Vehicle</Text>
          <Text style={styles.reportValueCustom}>{item.equipmentName}</Text>
        </View>
        <View style={styles.dividerCustom} />
        <View style={[styles.reportRowCustom, { alignItems: 'center' }]}> 
          <Text style={styles.reportLabelCustom}>Date</Text>
          <Text style={styles.reportValueCustom}>{formattedDate}</Text>
          <TouchableOpacity
            style={styles.menuButtonCustom}
            onPress={event => {
              event.stopPropagation();
              event.target.measure((fx, fy, width, height, px, py) => {
                // Adjust for dropdown width and screen width
                let left = px;
                let top = py + height;
                if (left + menuWidth > Dimensions.get('window').width) {
                  left = Dimensions.get('window').width - menuWidth - 10;
                }
                if (top + 100 > Dimensions.get('window').height) {
                  top = Dimensions.get('window').height - 120;
                }
                setMenuPosition({ x: left, y: top });
                setMenuVisible(item.id);
              });
            }}
          >
            <MaterialIcons name="more-vert" size={22} color="#333" />
          </TouchableOpacity>
        </View>
        {/* Days of week bar */}
        <View style={styles.daysBarCustom}>
          {dayNames.map((d, idx) => (
            <View key={d} style={[styles.dayBoxCustom, dayIdx === idx && styles.dayBoxActiveCustom]}>
              <Text style={[styles.dayTextCustom, dayIdx === idx && styles.dayTextActiveCustom]}>{d}</Text>
            </View>
          ))}
        </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.navigate('MainApp')}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Daily Inspection</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={() => navigation.navigate('TradckDailyreport')}>
              <MaterialIcons name="track-changes" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity style={styles.iconButton} onPress={handleCloudPress}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud_1.png')}
                style={styles.icons12}
              />
            </TouchableOpacity>
            <TouchableOpacity style={styles.iconButton} onPress={handlePermissionPress}>
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
        {/* Search Bar */}
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by equipment or request number"
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={setSearchQuery}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.searchIcon} />
          </View>
        </View>
        {/* Tabs */}
        <View style={styles.tabsContainer}>
          <TouchableOpacity
            style={[styles.tab]}
            onPress={() => setActiveTab('running')}
          >
            <Text style={[styles.tabText, activeTab === 'running' && styles.activeTabText]}>Running</Text>
            {activeTab === 'running' && <View style={styles.tabIndicator} />}
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.tab]}
            onPress={() => setActiveTab('submitted')}
          >
            <Text style={[styles.tabText, activeTab === 'submitted' && styles.activeTabText]}>Submitted</Text>
            {activeTab === 'submitted' && <View style={styles.tabIndicator} />}
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Reports List */}
      <FlatList
        data={filteredReports}
        renderItem={renderReportItem}
        keyExtractor={(item) => item.id}
        contentContainerStyle={styles.reportsContainer}
        ListHeaderComponent={
          !hasTodayReport ? (
            <View style={styles.reportCardCustom}>
              <View style={styles.reportRowCustom}>
                <Text style={styles.reportLabelCustom}>Report#:</Text>
                <Text style={styles.reportValueCustom}>-</Text>
              </View>
              <View style={styles.dividerCustom} />
              <View style={styles.reportRowCustom}>
                <Text style={styles.reportLabelCustom}>Vehicle</Text>
                <Text style={styles.reportValueCustom}>-</Text>
              </View>
              <View style={styles.dividerCustom} />
              <View style={[styles.reportRowCustom, { alignItems: 'center' }]}> 
                <Text style={styles.reportLabelCustom}>Date</Text>
                <Text style={styles.reportValueCustom}>{`${today.getDate().toString().padStart(2, '0')}-${today.toLocaleString('en-US', { month: 'short' })}-${today.getFullYear()} ${today.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit', hour12: true })}`}</Text>
                <View style={styles.menuButtonCustom} />
              </View>
              {/* Days of week bar with button */}
              <View style={styles.daysBarCustom}>
                {['M', 'T', 'W', 'T', 'F', 'S', 'S'].map((d, idx) => {
                  const isToday = (today.getDay() === 0 ? 6 : today.getDay() - 1) === idx;
                  return (
                    <View key={d} style={[styles.dayBoxCustom, isToday && styles.dayBoxActiveCustom]}> 
                      <Text style={[styles.dayTextCustom, isToday && styles.dayTextActiveCustom]}>{d}</Text>
                    </View>
                  );
                })}
                <TouchableOpacity style={styles.todayBtnCustom} onPress={() => navigation.navigate('Daily_Ins_Header')}>
                  <Text style={styles.todayBtnTextCustom}>Today Start Inspection</Text>
                </TouchableOpacity>
              </View>
            </View>
          ) : null
        }
        ListEmptyComponent={
          <View style={styles.emptyContainer}>
            <Text style={styles.emptyText}>
              {activeTab === 'running'
                ? 'No inspections found.'
                : 'No submitted inspections found older than 30 days.'}
            </Text>
          </View>
        }
      />

      {/* Bottom Button */}
      <View style={styles.buttonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
          <TouchableOpacity
            style={styles.startButton}
            onPress={() => navigation.navigate('Daily_Ins_Header')}
          >
            <Ionicons name="add" size={width * 0.06} color="#fff" />
            <Text style={styles.buttonText}> Start New Inspection</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* Permission Modal */}
      <Modal
        visible={permissionModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setPermissionModalVisible(false)}
      >
        <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.4)', justifyContent: 'center', alignItems: 'center' }}>
          <View style={{
            width: '85%',
            backgroundColor: '#fff',
            borderRadius: 18,
            padding: 28,
            alignItems: 'center',
            shadowColor: '#000',
            shadowOffset: { width: 0, height: 2 },
            shadowOpacity: 0.2,
            shadowRadius: 8,
            elevation: 8,
          }}>
            <Text style={{
              fontSize: 22,
              fontWeight: 'bold',
              color: '#025383',
              marginBottom: 18,
            }}>
              User Details
            </Text>
            {isLoadingPermissions ? (
              <ActivityIndicator size="large" color="#025383" />
            ) : !userPermissions ? (
              <Text style={{ color: '#FF6B6B', fontSize: 16, marginBottom: 12, textAlign: 'center' }}>Failed to load permissions</Text>
            ) : (
              <View style={{ width: '100%', marginBottom: 18 }}>
                <View style={{ marginBottom: 12 }}>
                  <Text style={{ color: '#888', fontSize: 14 }}>Country</Text>
                  <Text style={{ fontSize: 17, fontWeight: '600', color: '#025383', marginTop: 2 }}>{userPermissions.profile.country}</Text>
                </View>
                <View style={{ marginBottom: 12 }}>
                  <Text style={{ color: '#888', fontSize: 14 }}>Project</Text>
                  <Text style={{ fontSize: 17, fontWeight: '600', color: '#025383', marginTop: 2 }}>{userPermissions.profile.project}</Text>
                </View>
                <View>
                  <Text style={{ color: '#888', fontSize: 14 }}>Daily Report Permission</Text>
                  <Text style={{
                    fontSize: 17,
                    fontWeight: '600',
                    color: userPermissions.dailyReportEnabled ? '#025383' : '#FF6B6B',
                    marginTop: 2,
                  }}>{userPermissions.dailyReportEnabled ? 'Enabled' : 'No Permissions'}</Text>
                </View>
              </View>
            )}
            <TouchableOpacity
              style={{
                marginTop: 8,
                backgroundColor: '#025383',
                borderRadius: 8,
                paddingVertical: 10,
                paddingHorizontal: 32,
              }}
              onPress={() => setPermissionModalVisible(false)}
              accessibilityLabel="Close user details"
              accessibilityRole="button"
            >
              <Text style={{
                color: '#fff',
                fontWeight: 'bold',
                fontSize: 16,
              }}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      {menuVisible && (
        <TouchableOpacity
          activeOpacity={1}
          style={{
            position: 'absolute',
            top: 0,
            left: 0,
            right: 0,
            bottom: 0,
            zIndex: 9999,
            elevation: 20,
          }}
          onPress={() => setMenuVisible(null)}
        >
          <View
            style={[
              styles.dropdownMenuCustom,
              {
                position: 'absolute',
                left: menuPosition.x,
                top: menuPosition.y,
                width: menuWidth,
                backgroundColor: '#fff',
                borderRadius: 8,
                shadowColor: '#000',
                shadowOffset: { width: 0, height: 2 },
                shadowOpacity: 0.2,
                shadowRadius: 4,
                elevation: 10,
              },
            ]}
            onLayout={e => setMenuWidth(e.nativeEvent.layout.width)}
          >
            {(() => {
              const item = filteredReports.find(r => r.id === menuVisible);
              if (!item) return null;
              return (
                <>
                  {item.pdfUrl && (
                    <TouchableOpacity style={styles.dropdownMenuItemCustom} onPress={() => { handleViewPDF(item); setMenuVisible(null); }}>
                      <MaterialIcons name="picture-as-pdf" size={22} color="#2196f3" />
                      <Text style={styles.dropdownMenuTextCustom}>View PDF</Text>
                    </TouchableOpacity>
                  )}
                  <TouchableOpacity style={styles.dropdownMenuItemCustom} onPress={() => { handleDeleteReport(item.id); setMenuVisible(null); }}>
                    <MaterialIcons name="delete" size={22} color="#e74c3c" />
                    <Text style={[styles.dropdownMenuTextCustom, { color: '#e74c3c' }]}>Delete</Text>
                  </TouchableOpacity>
                </>
              );
            })()}
          </View>
        </TouchableOpacity>
      )}

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    alignItems: 'center',
    width: '100%',
    paddingTop: width * 0.025,
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  searchIcon: {
    marginLeft: width * 0.012,
  },
  tabsContainer: {
    flexDirection: 'row',
    backgroundColor: 'transparent',
    marginTop: 0,
    paddingHorizontal: 20,
  },
  tab: {
    flex: 1,
    paddingVertical: 15,
    alignItems: 'center',
    position: 'relative',
  },
  tabText: {
    fontSize: 16,
    color: 'rgba(255, 255, 255, 0.7)',
    fontWeight: '500',
  },
  activeTabText: {
    color: '#fff',
  },
  tabIndicator: {
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    height: 3,
    backgroundColor: '#fff',
  },
  reportsContainer: {
    padding: 20,
    paddingBottom: 100, // Space for bottom button
  },
  reportCardCustom: {
    backgroundColor: '#fff',
    borderRadius: 1,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    marginBottom: 16,
    padding: 0,
    elevation: 1,
  },
  reportRowCustom: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'flex-start',
    paddingHorizontal: 16,
    paddingVertical: 10,
    backgroundColor: '#fff',
  },
  reportLabelCustom: {
    fontWeight: 'bold',
    color: '#333',
    fontSize: 14,
    minWidth: 70,
  },
  reportValueCustom: {
    color: '#222',
    fontSize: 14,
    flex: 1,
    textAlign: 'right',
    marginLeft: 10,
  },
  dividerCustom: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginHorizontal: 16,
  },
  menuButtonCustom: {
    marginLeft: 8,
    padding: 4,
  },
  daysBarCustom: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    backgroundColor: '#fff',
    paddingVertical: 8,
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
  },
  dayBoxCustom: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: 2,
  },
  dayBoxActiveCustom: {
    backgroundColor: '#7be0b0',
    borderRadius: 1,
    paddingHorizontal: 8,
    paddingVertical: 2,
  },
  dayTextCustom: {
    color: '#888',
    fontSize: 14,
    fontWeight: '500',
  },
  dayTextActiveCustom: {
    color: '#fff',
    fontWeight: 'bold',
  },
  menuDropdown: {
    position: 'absolute',
    right: 0,
    top: 30,
    backgroundColor: '#fff',
    borderRadius: 1,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 3,
    zIndex: 1000,
  },
  dropdownMenuCustom: {
    position: 'absolute',
    right: 0,
    top: 30,
    backgroundColor: '#fff',
    borderRadius: 6,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    zIndex: 1000,
    minWidth: 180,
    paddingVertical: 4,
    paddingHorizontal: 0,
  },
  dropdownMenuItemCustom: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 16,
  },
  dropdownMenuTextCustom: {
    marginLeft: 12,
    fontSize: 16,
    color: '#015185',
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 20,
  },
  emptyText: {
    fontSize: width * 0.04,
    color: '#888',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  todayBtnCustom: {
    backgroundColor: '#3481BC',
    borderRadius: 6,
    paddingVertical: 6,
    paddingHorizontal: 14,
    marginLeft: 8,
    alignSelf: 'center',
  },
  todayBtnTextCustom: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 13,
  },
  permissionModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  permissionModalContent: {
    width: '85%',
    maxHeight: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
  },
  permissionModalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 15,
  },
  closeIconButton: {
    padding: 5,
  },
  permissionModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
  },
  profileSection: {
    marginBottom: 15,
  },
  sectionTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
  },
  profileItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  profileLabel: {
    fontSize: 15,
    fontWeight: '600',
    color: '#015185',
  },
  profileValue: {
    fontSize: 15,
    color: '#666',
    fontWeight: '500',
  },
  divider: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginVertical: 15,
  },
  modulesSection: {
    marginTop: 15,
  },
  moduleItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  moduleName: {
    fontSize: 15,
    fontWeight: '600',
    color: '#015185',
  },
  statusIndicator: {
    paddingVertical: 4,
    paddingHorizontal: 10,
    borderRadius: 5,
  },
  statusText: {
    color: '#fff',
    fontSize: 13,
    fontWeight: 'bold',
  },
  noModulesText: {
    fontSize: 14,
    color: '#888',
    textAlign: 'center',
    marginTop: 10,
  },
  errorText: {
    fontSize: 16,
    color: '#e74c3c',
    textAlign: 'center',
    marginTop: 10,
  },
  loadingContainer: {
    flexDirection: 'column',
    alignItems: 'center',
    marginTop: 20,
  },
  loadingText: {
    fontSize: 15,
    color: '#333',
    marginTop: 10,
  },
  closeButton: {
    backgroundColor: '#3481BC',
    borderRadius: 8,
    paddingVertical: 12,
    paddingHorizontal: 24,
    marginTop: 20,
    alignSelf: 'center',
  },
  closeButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
});

export default DailyIns_report_home;