import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  FlatList,
  Modal,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import { useNavigation, useIsFocused } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const Create_Custom_c = () => {
  const navigation = useNavigation();
  const isFocused = useIsFocused();
  const [searchQuery, setSearchQuery] = useState('');
  const [checklists, setChecklists] = useState([]);
  const [isLoading, setIsLoading] = useState(false);
  const [dropdownVisible, setDropdownVisible] = useState(null);
  const [isAdminVerified, setIsAdminVerified] = useState(false);
  const user = auth().currentUser;
  const ADMIN_UID = 'cahj3jJqe5Z16pQdLGSWrNgm84o1';

  useEffect(() => {
    const verifyAdmin = async () => {
      if (!user?.uid) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'User not logged in.',
        });
        return;
      }

      if (user.uid !== ADMIN_UID) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Unauthorized user.',
        });
        return;
      }

      try {
        const userRef = database().ref(`/Globalusers/admins/${user.uid}`);
        const snapshot = await userRef.once('value');
        if (snapshot.exists()) {
          setIsAdminVerified(true);
        } else {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'User is not an admin.',
          });
        }
      } catch (error) {
        console.error('Error verifying admin:', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to verify admin status.',
        });
      }
    };

    verifyAdmin();
  }, [user?.uid]);

  useEffect(() => {
    if (isFocused && isAdminVerified) {
      loadChecklists();
    }
  }, [isFocused, isAdminVerified]);

  const loadChecklists = async () => {
    setIsLoading(true);
    try {
      const checklistRef = database().ref(`/InspectionChecklistCreatebyusers/admin/${ADMIN_UID}/checklists`);
      checklistRef.on('value', async (snapshot) => {
        const data = snapshot.val();
        let checklistArray = [];
        if (data) {
          checklistArray = Object.keys(data).map((key) => ({
            id: key,
            ...data[key],
          }));
        }
        setChecklists(checklistArray);
        try {
          await AsyncStorage.setItem('checklists', JSON.stringify(checklistArray));
          Toast.show({
            type: 'success',
            text1: 'Connected Successfully',
            text2: 'Checklists loaded from database.',
          });
        } catch (error) {
          console.error('Error saving to AsyncStorage:', error);
        }
        setIsLoading(false);
      });

      return () => checklistRef.off();
    } catch (error) {
      console.error('Error loading checklists:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load checklists.',
      });
      setIsLoading(false);
    }
  };

  const handleSearch = (text) => {
    setSearchQuery(text);
    if (text) {
      const filtered = checklists.filter((item) =>
        item.title.toLowerCase().includes(text.toLowerCase())
      );
      setChecklists(filtered);
    } else {
      loadChecklists();
    }
  };

  const handleDeleteChecklist = async (checklistId) => {
    try {
      const isConnected = (await NetInfo.fetch()).isConnected;
      if (isConnected) {
        await database().ref(`/InspectionChecklistCreatebyusers/admin/${ADMIN_UID}/checklists/${checklistId}`).remove();
      }
      const updatedChecklists = checklists.filter((item) => item.id !== checklistId);
      setChecklists(updatedChecklists);
      await AsyncStorage.setItem('checklists', JSON.stringify(updatedChecklists));
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Checklist deleted successfully.',
      });
    } catch (error) {
      console.error('Error deleting checklist:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to delete checklist.',
      });
    } finally {
      setDropdownVisible(null);
    }
  };

  const truncateDescription = (description) => {
    if (!description) return 'No description';
    const words = description.split(' ');
    return words.length > 4 ? `${words.slice(0, 4).join(' ')}...` : description;
  };

  const renderChecklist = ({ item }) => (
    <View style={styles.checklistItem}>
      <TouchableOpacity
        style={styles.checklistTitleContainer}
        onPress={() =>
          navigation.navigate('CreatechecklistHeader', { checklist: item })
        }
      >
        <Text style={styles.checklistTitle}>{item.title || 'Untitled Checklist'}</Text>
        <Text style={styles.checklistDescription}>
          {truncateDescription(item.description)}
        </Text>
        <View style={styles.itemsContainer}>
          <Text style={styles.itemsLabel}>Items:</Text>
          {item.items?.length > 0 ? (
            item.items.map((section, index) => (
              <View key={index} style={styles.sectionContainer}>
                <Text style={styles.sectionTitle}>{section.text || `Section ${index + 1}`}</Text>
                {section.subItems?.length > 0 ? (
                  <FlatList
                    data={section.subItems}
                    keyExtractor={(subItem, subIndex) => `${index}-${subIndex}`}
                    renderItem={({ item: subItem }) => (
                      <Text style={styles.subItemText}>
                        • {subItem.itemName || 'Unnamed Item'}
                      </Text>
                    )}
                    style={styles.subItemsList}
                  />
                ) : (
                  <Text style={styles.noSubItemsText}>No inspection items</Text>
                )}
              </View>
            ))
          ) : (
            <Text style={styles.noItemsText}>No sections available</Text>
          )}
        </View>
      </TouchableOpacity>
      <View style={styles.menuContainer}>
        <TouchableOpacity
          style={styles.menuButton}
          onPress={() =>
            setDropdownVisible(dropdownVisible === item.id ? null : item.id)
          }
        >
          <MaterialIcons name="more-vert" size={20} color="#0078D4" />
        </TouchableOpacity>
        {dropdownVisible === item.id && (
          <View style={styles.dropdown}>
            <TouchableOpacity
              style={styles.dropdownItem}
              activeOpacity={0.7}
              onPress={() => {
                navigation.navigate('CreatechecklistHeader', { checklist: item });
                setDropdownVisible(null);
              }}
            >
              <MaterialIcons name="edit" size={18} color="#0078D4" style={styles.dropdownIcon} />
              <Text style={styles.dropdownText}>Edit Checklist</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.dropdownItem}
              activeOpacity={0.7}
              onPress={() => handleDeleteChecklist(item.id)}
            >
              <MaterialIcons name="delete" size={18} color="#dc3545" style={styles.dropdownIcon} />
              <Text style={styles.dropdownText}>Delete Checklist</Text>
            </TouchableOpacity>
          </View>
        )}
      </View>
    </View>
  );

  if (!isAdminVerified) {
    return (
      <View style={styles.container}>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
        >
          <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
          <View style={styles.headerContent}>
            <View style={styles.headerLeft}>
              <TouchableOpacity onPress={() => navigation.goBack()}>
                <MaterialIcons name="chevron-left" size={28} color="#fff" />
              </TouchableOpacity>
              <Text style={styles.headerTitle}>Custom Checklist</Text>
            </View>
          </View>
        </LinearGradient>
        <View style={styles.mainContent}>
          <Text style={styles.errorText}>Verifying admin status...</Text>
        </View>
      </View>
    );
  }

  return (
    <View style={styles.container}>
      {/* Loader Modal */}
      <Modal
        transparent={true}
        animationType="fade"
        visible={isLoading}
        onRequestClose={() => {}}
      >
        <View style={styles.modalContainer}>
          <View style={styles.loaderContent}>
            <ActivityIndicator size="large" color="#0078D4" />
            <Text style={styles.loaderText}>Loading Checklists...</Text>
          </View>
        </View>
      </Modal>

      {/* Header with Gradient */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="chevron-left" size={28} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Custom Checklist</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud_1.png')}
                style={styles.icons12}
              />
            </TouchableOpacity>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search checklists..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
        </View>
      </LinearGradient>

      {/* Main Content */}
      <FlatList
        data={checklists}
        keyExtractor={(item) => item.id.toString()}
        renderItem={renderChecklist}
        ListEmptyComponent={
          <Text style={styles.emptyText}>No checklists found.</Text>
        }
        style={styles.list}
      />

      {/* Bottom Button */}
      <View style={styles.buttonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
          <TouchableOpacity
            style={styles.startButton}
            onPress={() => navigation.navigate('CreatechecklistHeader', { checklist: null })}
          >
            <Ionicons name="add" size={width * 0.06} color="#fff" />
            <Text style={styles.buttonText}>Create New Checklist</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      {/* Toast Configuration */}
      <Toast position="bottom" />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.06,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  list: {
    flex: 1,
    paddingHorizontal: width * 0.04,
  },
  checklistItem: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    paddingVertical: width * 0.03,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  checklistTitleContainer: {
    flex: 1,
  },
  checklistTitle: {
    fontSize: width * 0.045,
    color: '#333',
    fontWeight: '500',
  },
  checklistDescription: {
    fontSize: width * 0.035,
    color: '#666',
    marginTop: 4,
  },
  itemsContainer: {
    marginTop: width * 0.02,
  },
  itemsLabel: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '600',
    marginBottom: width * 0.01,
  },
  sectionContainer: {
    marginBottom: width * 0.02,
  },
  sectionTitle: {
    fontSize: width * 0.04,
    color: '#333',
    fontWeight: '500',
    marginBottom: width * 0.01,
  },
  subItemsList: {
    marginLeft: width * 0.04,
  },
  subItemText: {
    fontSize: width * 0.035,
    color: '#555',
    marginBottom: width * 0.005,
  },
  noSubItemsText: {
    fontSize: width * 0.035,
    color: '#888',
    marginLeft: width * 0.04,
  },
  noItemsText: {
    fontSize: width * 0.035,
    color: '#888',
  },
  menuContainer: {
    position: 'relative',
  },
  menuButton: {
    padding: width * 0.02,
  },
  dropdown: {
    position: 'absolute',
    right: 0,
    top: 40,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#ddd',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 3,
    width: width * 0.4,
    zIndex: 1000,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    paddingHorizontal: 16,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  dropdownIcon: {
    marginRight: 10,
  },
  dropdownText: {
    fontSize: width * 0.04,
    color: '#333',
    fontWeight: '400',
  },
  emptyText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  modalContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
  },
  loaderContent: {
    backgroundColor: '#fff',
    padding: 20,
    borderRadius: 10,
    alignItems: 'center',
  },
  loaderText: {
    marginTop: 10,
    fontSize: 16,
    color: '#333',
  },
  mainContent: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  errorText: {
    fontSize: 16,
    color: 'red',
    textAlign: 'center',
  },
});

export default Create_Custom_c;