import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  StatusBar,
  StyleSheet,
  Dimensions,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import Toast from 'react-native-toast-message';
import Inspection_info_Checklist from './Inspection_info_Checklist';
import Do_inspection_checklist from './Do_inspection_checklist';
import Add_inspection_item from './Add_inspection_item';

const { height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = height * 0.2 - 36;

const CreatechecklistHeader = ({ navigation, route }) => {
  const [activeStep, setActiveStep] = useState(1);
  const [addItemParams, setAddItemParams] = useState(null);
  const [checklistData, setChecklistData] = useState({
    id: route.params?.checklist?.id || Date.now().toString(),
    title: route.params?.checklist?.title || '',
    description: route.params?.checklist?.description || '',
    items: route.params?.checklist?.items || [],
    createdAt: route.params?.checklist?.createdAt || new Date().toISOString(),
    updatedAt: route.params?.checklist?.updatedAt || new Date().toISOString(),
    createdBy: route.params?.checklist?.createdBy || auth().currentUser?.uid || 'unknown',
  });
  const [isAdminVerified, setIsAdminVerified] = useState(false);
  const user = auth().currentUser;

  useEffect(() => {
    const verifyAdmin = async () => {
      if (!user?.uid) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'User not logged in.',
        });
        return;
      }

      try {
        const userRef = database().ref(`/Globalusers/admins/${user.uid}`);
        const snapshot = await userRef.once('value');
        if (snapshot.exists()) {
          setIsAdminVerified(true);
          setChecklistData((prev) => ({ ...prev, createdBy: user.uid }));
        } else {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'User is not an admin.',
          });
        }
      } catch (error) {
        console.error('Error verifying admin:', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to verify admin status.',
        });
      }
    };

    verifyAdmin();
  }, [user?.uid]);

  useEffect(() => {
    if (route.params?.checklist) {
      setChecklistData({
        id: route.params.checklist.id,
        title: route.params.checklist.title || '',
        description: route.params.checklist.description || '',
        items: route.params.checklist.items || [],
        createdAt: route.params.checklist.createdAt || new Date().toISOString(),
        updatedAt: route.params.checklist.updatedAt || new Date().toISOString(),
        createdBy: route.params.checklist.createdBy || user?.uid || 'unknown',
      });
    }
  }, [route.params?.checklist, user?.uid]);

  const handleChecklistDataUpdate = async (updatedData) => {
    if (!isAdminVerified) return;

    try {
      const isConnected = (await NetInfo.fetch()).isConnected;
      if (isConnected && user?.uid) {
        const checklistRef = database().ref(
          `/InspectionChecklistCreatebyusers/admin/${user.uid}/checklists/${updatedData.id}`
        );
        await checklistRef.set(updatedData);
      }

      let checklists = [];
      try {
        const storedChecklists = await AsyncStorage.getItem('checklists');
        checklists = storedChecklists ? JSON.parse(storedChecklists) : [];
      } catch (error) {
        console.error('Error reading AsyncStorage:', error);
      }

      const index = checklists.findIndex((c) => c.id === updatedData.id);
      if (index >= 0) {
        checklists[index] = updatedData;
      } else {
        checklists.push(updatedData);
      }

      await AsyncStorage.setItem('checklists', JSON.stringify(checklists));
      setChecklistData(updatedData);
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Checklist data updated.',
      });
    } catch (error) {
      console.error('Error saving checklist:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save checklist.',
      });
    }
  };

  const isStep1Complete = () => {
    return checklistData.title.trim() !== '' && checklistData.description.trim() !== '';
  };

  const steps = [
    {
      id: 1,
      title: 'Step 1',
      subtitle: 'Inspection Info',
      component: (
        <Inspection_info_Checklist
          checklistData={checklistData}
          setChecklistData={handleChecklistDataUpdate}
          setActiveStep={setActiveStep}
        />
      ),
    },
    {
      id: 2,
      title: 'Step 2',
      subtitle: 'Do Inspection',
      component: addItemParams ? (
        <Add_inspection_item
          {...addItemParams}
          onSave={(subItem) => {
            addItemParams.onSave(subItem);
            setAddItemParams(null);
          }}
          onCancel={() => {
            addItemParams.onCancel();
            setAddItemParams(null);
          }}
        />
      ) : (
        <Do_inspection_checklist
          checklistData={checklistData}
          setChecklistData={handleChecklistDataUpdate}
          setActiveStep={setActiveStep}
          setAddItemParams={setAddItemParams}
        />
      ),
    },
  ];

  if (!isAdminVerified) {
    return (
      <View style={styles.container}>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={styles.header}
        >
          <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
          <View style={styles.stepsRow}>
            <View style={styles.stepContainer}>
              <View style={styles.stepBadgeInactive}>
                <Text style={styles.stepBadgeTextInactive}>1</Text>
              </View>
              <View style={styles.stepTextBlock}>
                <Text style={styles.stepLabelWhite}>Step 1</Text>
                <Text style={styles.stepSubtitle}>Inspection Info</Text>
              </View>
            </View>
            <View style={styles.stepContainer}>
              <View style={styles.stepBadgeInactive}>
                <Text style={styles.stepBadgeTextInactive}>2</Text>
              </View>
              <View style={styles.stepTextBlock}>
                <Text style={styles.stepLabelWhite}>Step 2</Text>
                <Text style={styles.stepSubtitle}>Do Inspection</Text>
              </View>
            </View>
          </View>
        </LinearGradient>
        <View style={styles.mainContent}>
          <Text style={styles.errorText}>Verifying admin status...</Text>
        </View>
      </View>
    );
  }

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={styles.header}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <TouchableOpacity
          style={styles.homeButton}
          onPress={() => navigation.goBack()}
        >
          <MaterialIcons name="arrow-back" size={24} color="#fff" />
          <Text style={styles.headerHomeText}>Back</Text>
        </TouchableOpacity>
        <View style={styles.stepsRow}>
          <View style={styles.stepContainer}>
            {activeStep === 2 ? (
              <View style={styles.stepBadgeActive}>
                <MaterialIcons name="check" size={18} color="#fff" />
              </View>
            ) : (
              <View style={styles.stepBadgeActive}>
                <Text style={styles.stepBadgeTextActive}>1</Text>
              </View>
            )}
            <View style={styles.stepTextBlock}>
              <Text style={styles.stepLabelWhite}>Step 1</Text>
              <Text style={styles.stepSubtitle}>Inspection Info</Text>
            </View>
          </View>
          <View style={styles.stepContainer}>
            <View style={[styles.stepBadgeInactive, activeStep === 2 && { backgroundColor: '#4CAF50' }]}>
              <Text style={[styles.stepBadgeTextInactive, activeStep === 2 && { color: '#fff' }]}>2</Text>
            </View>
            <View style={styles.stepTextBlock}>
              <Text style={styles.stepLabelWhite}>Step 2</Text>
              <Text style={styles.stepSubtitle}>Do Inspection</Text>
            </View>
          </View>
        </View>
      </LinearGradient>
      <View style={styles.mainContent}>
        {steps.find((step) => step.id === activeStep).component}
      </View>
      <Toast position="bottom" />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    height: TOTAL_HEADER_HEIGHT,
    paddingTop: 32,
    paddingBottom: 0,
    paddingHorizontal: 0,
    justifyContent: 'flex-start',
    borderBottomLeftRadius: 0,
    borderBottomRightRadius: 0,
  },
  homeButton: {
    position: 'absolute',
    top: 24,
    left: 10,
    flexDirection: 'row',
    alignItems: 'center',
    zIndex: 2,
  },
  headerHomeText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
    marginLeft: 2,
  },
  stepsRow: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'flex-end',
    marginTop: 30,
    width: '100%',
  },
  stepContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginHorizontal: 32,
  },
  stepBadgeActive: {
    width: 28,
    height: 28,
    borderRadius: 14,
    backgroundColor: '#4CAF50',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 6,
  },
  stepBadgeInactive: {
    width: 28,
    height: 28,
    borderRadius: 14,
    backgroundColor: '#e0e6ed',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 6,
  },
  stepBadgeTextActive: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
  stepBadgeTextInactive: {
    color: '#3481BC',
    fontWeight: 'bold',
    fontSize: 16,
  },
  stepTextBlock: {
    flexDirection: 'column',
    alignItems: 'flex-start',
  },
  stepLabelWhite: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
    lineHeight: 18,
  },
  stepSubtitle: {
    color: '#fff',
    fontSize: 12,
    marginTop: -2,
  },
  mainContent: {
    flex: 1,
  },
  errorText: {
    fontSize: 16,
    color: 'red',
    textAlign: 'center',
    marginTop: 20,
  },
});

export default CreatechecklistHeader;