import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Dimensions,
  FlatList,
  ActivityIndicator,
  Modal,
  Animated,
  Image,
  Linking,
  Platform,
  PermissionsAndroid,
  Share,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';
import ReactNativeBlobUtil from 'react-native-blob-util';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const Checklist_library = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [documents, setDocuments] = useState([]);
  const [filteredDocuments, setFilteredDocuments] = useState([]);
  const [showLoaderModal, setShowLoaderModal] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [userId, setUserId] = useState(null);
  const [isLoading, setIsLoading] = useState(true);
  const [isLoadingMore, setIsLoadingMore] = useState(false);
  const [lastDocument, setLastDocument] = useState(null);
  const [hasMoreDocuments, setHasMoreDocuments] = useState(true);
  const [totalCount, setTotalCount] = useState(0);
  const [downloadingDocuments, setDownloadingDocuments] = useState(new Set());

  // Animation setup for loader icon
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Start animation when loader modal is shown
  useEffect(() => {
    if (showLoaderModal) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [showLoaderModal]);

  // Check authentication and admin status
  useEffect(() => {
    const unsubscribe = auth().onAuthStateChanged(async user => {
      if (user) {
        setUserId(user.uid);
        try {
          await AsyncStorage.setItem('userEmail', user.email);
          const userSnapshot = await database()
            .ref('/Globalusers/admins')
            .orderByChild('email')
            .equalTo(user.email)
            .once('value');
          const userData = userSnapshot.val();
          if (!userData) {
            Toast.show({
              type: 'error',
              text1: 'Access Denied',
              text2: 'You are not authorized as an admin.',
            });
            navigation.replace('AdminLoginScreen');
            setUserId(null);
          }
        } catch (error) {
          console.error('Error verifying admin status:', error);
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Failed to verify admin status.',
          });
          navigation.replace('AdminLoginScreen');
          setUserId(null);
        }
      } else {
        setUserId(null);
        Toast.show({
          type: 'error',
          text1: 'Authentication Error',
          text2: 'Please log in to access documents',
        });
        navigation.replace('AdminLoginScreen');
      }
    });
    return unsubscribe;
  }, [navigation]);

  // Network status listener
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener(state => {
      setIsOnline(state.isConnected);
    });
    return unsubscribe;
  }, []);

  // Load documents from Firebase Storage
  const loadDocuments = async (isLoadMore = false) => {
    if (!userId || !isOnline) return;

    try {
      if (!isLoadMore) {
        setShowLoaderModal(true);
        setIsLoading(true);
      } else {
        setIsLoadingMore(true);
      }

      const storageRef = storage().ref('AdminChecklistPDF');
      let listResult;

      if (isLoadMore && lastDocument) {
        // Load more documents after the last one
        listResult = await storageRef.list({
          maxResults: 10,
          pageToken: lastDocument
        });
      } else {
        // Initial load
        listResult = await storageRef.list({
          maxResults: 10
        });
      }

      const newDocuments = [];
      
      for (const item of listResult.items) {
        try {
          const metadata = await item.getMetadata();
          const downloadURL = await item.getDownloadURL();
          
          const document = {
            id: item.name,
            name: item.name,
            fullPath: item.fullPath,
            size: metadata.size,
            contentType: metadata.contentType,
            timeCreated: metadata.timeCreated,
            updated: metadata.updated,
            downloadURL: downloadURL,
            type: getFileType(item.name),
            icon: getFileIcon(item.name),
            selected: false,
            downloaded: false
          };
          
          newDocuments.push(document);
        } catch (error) {
          console.error(`Error getting metadata for ${item.name}:`, error);
        }
      }

      if (isLoadMore) {
        setDocuments(prev => [...prev, ...newDocuments]);
      } else {
        setDocuments(newDocuments);
      }

      setLastDocument(listResult.nextPageToken);
      setHasMoreDocuments(!!listResult.nextPageToken);
      
      // Get total count from Firebase Storage
      if (!isLoadMore) {
        try {
          const allItems = await storageRef.listAll();
          setTotalCount(allItems.items.length);
        } catch (error) {
          console.error('Error getting total count:', error);
          setTotalCount(prev => isLoadMore ? prev + newDocuments.length : newDocuments.length);
        }
      }

      // Check if documents are already downloaded
      const storedDownloads = await AsyncStorage.getItem('downloadedDocuments');
      const downloadedItems = storedDownloads ? JSON.parse(storedDownloads) : {};
      
      setDocuments(prev => prev.map(doc => ({
        ...doc,
        downloaded: !!downloadedItems[doc.id]
      })));

    } catch (error) {
      console.error('Error loading documents:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load documents from storage.',
      });
    } finally {
      if (!isLoadMore) {
        setShowLoaderModal(false);
        setIsLoading(false);
      } else {
        setIsLoadingMore(false);
      }
    }
  };

  // Get file type from filename
  const getFileType = (filename) => {
    const extension = filename.split('.').pop()?.toLowerCase();
    if (['pdf'].includes(extension)) return 'PDF';
    if (['doc', 'docx'].includes(extension)) return 'Word';
    if (['xls', 'xlsx'].includes(extension)) return 'Excel';
    if (['ppt', 'pptx'].includes(extension)) return 'PowerPoint';
    return 'Document';
  };

  // Get file icon based on file type
  const getFileIcon = (filename) => {
    const extension = filename.split('.').pop()?.toLowerCase();
    if (['pdf'].includes(extension)) return 'picture-as-pdf';
    if (['doc', 'docx'].includes(extension)) return 'description';
    if (['xls', 'xlsx'].includes(extension)) return 'table-chart';
    if (['ppt', 'pptx'].includes(extension)) return 'slideshow';
    return 'insert-drive-file';
  };

  // Load documents when component mounts or when online status changes
  useEffect(() => {
    if (userId && isOnline) {
      loadDocuments();
      getTotalCount();
    }
  }, [userId, isOnline]);

  // Get total count from Firebase Storage
  const getTotalCount = async () => {
    if (!userId || !isOnline) return;
    
    try {
      const storageRef = storage().ref('AdminChecklistPDF');
      const allItems = await storageRef.listAll();
      setTotalCount(allItems.items.length);
    } catch (error) {
      console.error('Error getting total count:', error);
      // Keep the current count if we can't get the total
    }
  };

  // Handle document view
  const handleViewDocument = async (document) => {
    try {
      // Navigate to ChecklistViewfile to view the document
      navigation.navigate('ChecklistViewfile', { 
        url: document.downloadURL,
        title: document.name,
        name: document.name
      });

    } catch (error) {
      console.error('Error navigating to document view:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to open document view.',
      });
    }
  };

  // Handle document download
  const handleDownload = async (document) => {
    try {
      setDownloadingDocuments(prev => new Set(prev).add(document.id));
      setShowLoaderModal(true);

      // Request storage permission for Android
      if (Platform.OS === 'android') {
        try {
          if (Platform.Version >= 33) {
            const granted = await PermissionsAndroid.request(
              PermissionsAndroid.PERMISSIONS.READ_MEDIA_IMAGES
            );
            if (granted !== PermissionsAndroid.RESULTS.GRANTED) {
              Toast.show({
                type: 'error',
                text1: 'Permission Denied',
                text2: 'Storage permission is required to download files',
                position: 'bottom',
                visibilityTime: 3000,
              });
              return;
            }
          } else {
            const granted = await PermissionsAndroid.request(
              PermissionsAndroid.PERMISSIONS.WRITE_EXTERNAL_STORAGE,
              {
                title: 'Storage Permission',
                message: 'App needs access to storage to download files.',
                buttonNeutral: 'Ask Me Later',
                buttonNegative: 'Cancel',
                buttonPositive: 'OK',
              }
            );
            if (granted !== PermissionsAndroid.RESULTS.GRANTED) {
              Toast.show({
                type: 'error',
                text1: 'Permission Denied',
                text2: 'Storage permission is required to download files',
                position: 'bottom',
                visibilityTime: 3000,
              });
              return;
            }
          }
        } catch (err) {
          console.warn(err);
          Toast.show({
            type: 'error',
            text1: 'Permission Error',
            text2: 'Failed to request storage permission',
            position: 'bottom',
            visibilityTime: 3000,
          });
          return;
        }
      }

      // Download the document using ReactNativeBlobUtil
      try {
        const fileName = `${document.name}`;
        let downloadDest;

        if (Platform.OS === 'android') {
          // On Android, use Download Manager to save to Downloads folder
          downloadDest = `${ReactNativeBlobUtil.fs.dirs.DownloadDir}/${fileName}`;
          const config = {
            fileCache: true,
            path: downloadDest,
            addAndroidDownloads: {
              useDownloadManager: true, // Use Android Download Manager
              notification: true, // Show notification
              title: fileName,
              description: 'Downloading document...',
              mime: document.contentType || 'application/pdf',
              mediaScannable: true, // Make file visible in Downloads app
            },
          };

          const res = await ReactNativeBlobUtil.config(config).fetch('GET', document.downloadURL);
          console.log('Document downloaded to:', res.path());

          Toast.show({
            type: 'success',
            text1: 'Download Complete',
            text2: `${document.name} downloaded to Downloads folder`,
            position: 'bottom',
            visibilityTime: 3000,
          });
        } else {
          // On iOS, save to DocumentDirectoryPath and offer to share
          downloadDest = `${ReactNativeBlobUtil.fs.dirs.DocumentDir}/${fileName}`;
          const config = {
            fileCache: true,
            path: downloadDest,
          };

          const res = await ReactNativeBlobUtil.config(config).fetch('GET', document.downloadURL);
          console.log('Document downloaded to:', res.path());

          // Share the file on iOS to allow saving to Files app
          await Share.share({
            url: `file://${res.path()}`,
            message: 'Here is your downloaded document.',
          });

          Toast.show({
            type: 'success',
            text1: 'Download Complete',
            text2: `${document.name} downloaded and ready to share`,
            position: 'bottom',
            visibilityTime: 3000,
          });
        }

        // Store download status in AsyncStorage
        const stored = await AsyncStorage.getItem('downloadedDocuments');
        const downloadedItems = stored ? JSON.parse(stored) : {};
        downloadedItems[document.id] = {
          downloadedAt: new Date().toISOString(),
          fileSize: document.size,
          fileType: document.contentType
        };
        await AsyncStorage.setItem(`document_${document.id}`, JSON.stringify(document));
        await AsyncStorage.setItem('downloadedDocuments', JSON.stringify(downloadedItems));

        // Update document status
        setDocuments(prev => prev.map(doc => 
          doc.id === document.id ? { ...doc, downloaded: true } : doc
        ));

        // Update filtered documents as well
        setFilteredDocuments(prev => prev.map(doc => 
          doc.id === document.id ? { ...doc, downloaded: true } : doc
        ));

      } catch (downloadError) {
        console.error('Error downloading file:', downloadError);
        Toast.show({
          type: 'error',
          text1: 'Download Failed',
          text2: 'Failed to download file from Firebase Storage.',
          position: 'bottom',
          visibilityTime: 3000,
        });
      }

    } catch (error) {
      console.error('Error in download process:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to download document.',
        position: 'bottom',
        visibilityTime: 3000,
      });
    } finally {
      setShowLoaderModal(false);
      setDownloadingDocuments(prev => {
        const newSet = new Set(prev);
        newSet.delete(document.id);
        return newSet;
      });
    }
  };

  // Handle search
  const handleSearch = text => {
    setSearchQuery(text);
    const filtered = documents.filter(
      document =>
        document.name.toLowerCase().includes(text.toLowerCase())
    );
    setFilteredDocuments(filtered);
  };

  // Handle document selection
  const toggleSelection = id => {
    const updatedDocuments = documents.map(document =>
      document.id === id ? { ...document, selected: !document.selected } : document
    );
    setDocuments(updatedDocuments);
    setFilteredDocuments(
      updatedDocuments.filter(
        item =>
          item.name.toLowerCase().includes(searchQuery.toLowerCase())
      )
    );
    // Persist selection to AsyncStorage
    const selectedDocuments = updatedDocuments.filter(item => item.selected);
    AsyncStorage.setItem('selectedDocuments', JSON.stringify(selectedDocuments.map(item => item.id)));
  };

  // Load more documents when scrolling
  const handleLoadMore = () => {
    if (hasMoreDocuments && !isLoadingMore && isOnline) {
      loadDocuments(true);
    }
  };

  // Format file size
  const formatFileSize = (bytes) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  // Format date
  const formatDate = (dateString) => {
    return new Date(dateString).toLocaleDateString();
  };

  const renderItem = ({ item, index }) => (
    <View style={styles.cardWrapper}>
      <View style={styles.cardRow}>
        <View style={styles.cardLeft}>
          <View style={styles.circleNumber}>
            <Text style={styles.circleNumberText}>{index + 1}</Text>
          </View>
          <View style={styles.fileTypeIndicator}>
            <MaterialIcons name={item.icon} size={20} color="#015185" />
          </View>
        </View>
        <View style={styles.cardCenter}>
          <Text style={styles.cardTitle}>{item.name}</Text>
          <Text style={styles.cardSubtitle}>{item.type} Document</Text>
          <Text style={styles.cardDetail}>Size: {formatFileSize(item.size)}</Text>
          <Text style={styles.cardDetail}>Created: {formatDate(item.timeCreated)}</Text>
          <Text style={styles.cardDetail}>Updated: {formatDate(item.updated)}</Text>
          {item.downloaded && (
            <Text style={styles.downloadedStatus}>✓ Downloaded</Text>
          )}
          {downloadingDocuments.has(item.id) && (
            <Text style={styles.downloadingStatus}>⏳ Downloading...</Text>
          )}
          
          <View style={styles.actionsRow}>
            <TouchableOpacity
              style={styles.viewButton}
              onPress={() => handleViewDocument(item)}
            >
              <Text style={styles.viewButtonText}>View</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={[
                styles.downloadButton, 
                item.downloaded && styles.downloadedButton,
                downloadingDocuments.has(item.id) && styles.downloadingButton
              ]}
              onPress={() => handleDownload(item)}
              disabled={downloadingDocuments.has(item.id)}
            >
              <MaterialIcons
                name={
                  item.downloaded ? "check-circle" : 
                  downloadingDocuments.has(item.id) ? "hourglass-empty" : "file-download"
                }
                size={20}
                color={
                  item.downloaded ? "#4CAF50" : 
                  downloadingDocuments.has(item.id) ? "#888" : "#015185"
                }
              />
            </TouchableOpacity>
          </View>
        </View>
        <View style={styles.cardRight}>
          <TouchableOpacity onPress={() => toggleSelection(item.id)}>
            {item.selected ? (
              <MaterialIcons name="check" size={22} color="#015185" style={styles.checkIcon} />
            ) : (
              <View style={styles.checkIcon} />
            )}
          </TouchableOpacity>
        </View>
      </View>
    </View>
  );

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient colors={['#3481BC', '#025383']} style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}>
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle} numberOfLines={1} ellipsizeMode="tail">
              Checklist Library
            </Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity>
              <MaterialIcons name="cloud-download" size={width * 0.06} color={isOnline ? '#fff' : '#888'} />
            </TouchableOpacity>
          </View>
        </View>
        <Text style={styles.subHeaderText}>
        Total Checklists: {totalCount} 
        </Text>

        {/* Search Bar */}
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search documents..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.searchIcon} />
          </View>
        </View>
      </LinearGradient>

      {/* Custom Loader Modal */}
      <Modal
        transparent={true}
        visible={showLoaderModal}
        animationType="fade"
        onRequestClose={() => setShowLoaderModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalText}>Please Wait</Text>
            <View style={styles.syncIconDiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.modalTextAdd}>Processing Documents</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* Offline Indicator */}
      {!isOnline && (
        <View style={styles.offlineIndicator}>
          <Text style={styles.offlineText}>Offline Mode - Data may not be up to date</Text>
        </View>
      )}

      {/* Document List */}
      {isLoading ? (
        <View style={styles.loadingContainer}>
          <ActivityIndicator size="large" color="#015185" />
          <Text style={styles.loadingText}>Loading Documents...</Text>
        </View>
      ) : (
        <FlatList
          data={searchQuery ? filteredDocuments : documents}
          renderItem={renderItem}
          keyExtractor={item => item.id}
          contentContainerStyle={styles.mainContent}
          onEndReached={handleLoadMore}
          onEndReachedThreshold={0.1}
          ListFooterComponent={() => 
            isLoadingMore ? (
              <View style={styles.loadMoreContainer}>
                <ActivityIndicator size="small" color="#015185" />
                <Text style={styles.loadMoreText}>Loading more documents...</Text>
              </View>
            ) : null
          }
          ListEmptyComponent={
            <Text style={styles.defaultText}>
              No documents found
            </Text>
          }
        />
      )}

      {/* Toast Component */}
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
    marginTop: 20,
    flex: 1,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
    flex: 1,
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
    marginTop: 20,
  },
  subHeaderText: {
    color: '#fff',
    fontSize: width * 0.035,
    paddingHorizontal: width * 0.04,
    paddingTop: 5,
    paddingBottom: 10,
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginHorizontal: 16,
    marginTop: 8,
    marginBottom: 12,
    backgroundColor: '#fff',
    borderRadius: 25,
    paddingHorizontal: 16,
    borderWidth: 1,
    borderColor: '#E0E0E0',
    height: 44,
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: 25,
    paddingHorizontal: width * 0.025,
  },
  searchInput: {
    flex: 1,
    fontSize: 16,
    color: '#222',
    backgroundColor: 'transparent',
    paddingVertical: 0,
  },
  searchIcon: {
    marginLeft: 8,
  },
  mainContent: {
    paddingHorizontal: 0,
    paddingVertical: 0,
    paddingBottom: 30,
  },
  defaultText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    marginVertical: 20,
  },
  cardWrapper: {
    backgroundColor: '#fff',
    borderTopWidth: 2,
    borderTopColor: '#fff',
    borderBottomWidth: 2,
    borderBottomColor: '#fff',
    marginBottom: 0,
  },
  cardRow: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    borderRadius: 12,
    marginHorizontal: 16,
    marginTop: 18,
    paddingVertical: 18,
    paddingHorizontal: 12,
    minHeight: 120,
  },
  cardLeft: {
    alignItems: 'center',
    justifyContent: 'center',
    width: 40,
  },
  cardCenter: {
    flex: 1,
    alignItems: 'center',
    justifyContent: 'center',
  },
  cardRight: {
    alignItems: 'center',
    justifyContent: 'center',
    width: 48,
  },
  circleNumber: {
    width: 32,
    height: 32,
    borderRadius: 16,
    backgroundColor: '#015185',
    alignItems: 'center',
    justifyContent: 'center',
  },
  circleNumberText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 18,
  },
  fileTypeIndicator: {
    marginTop: 8,
  },
  checkIcon: {
    marginBottom: 8,
  },
  cardTitle: {
    fontSize: 15,
    fontWeight: 'bold',
    color: '#222',
    textAlign: 'center',
    marginTop: 8,
  },
  cardSubtitle: {
    fontSize: 13,
    color: '#888',
    textAlign: 'center',
    marginBottom: 5,
  },
  cardDetail: {
    fontSize: 11,
    color: '#666',
    textAlign: 'center',
    marginBottom: 2,
  },
  downloadedStatus: {
    fontSize: 11,
    color: '#4CAF50',
    textAlign: 'center',
    marginBottom: 2,
    fontWeight: 'bold',
  },
  downloadingStatus: {
    fontSize: 11,
    color: '#888',
    textAlign: 'center',
    marginBottom: 2,
    fontStyle: 'italic',
  },
  actionsRow: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 4,
  },
  viewButton: {
    backgroundColor: '#015185',
    borderRadius: 4,
    paddingHorizontal: 10,
    paddingVertical: 4,
    marginHorizontal: 4,
  },
  viewButtonText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 13,
  },
  downloadButton: {
    backgroundColor: '#fff',
    borderRadius: 4,
    paddingHorizontal: 10,
    paddingVertical: 4,
    marginHorizontal: 4,
    borderWidth: 1,
    borderColor: '#015185',
  },
  downloadedButton: {
    borderColor: '#4CAF50',
    backgroundColor: '#E8F5E8',
  },
  downloadingButton: {
    borderColor: '#888',
    backgroundColor: '#f5f5f5',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  modalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  modalTextAdd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncIconDiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  offlineIndicator: {
    backgroundColor: '#f8d7da',
    padding: 10,
    alignItems: 'center',
  },
  offlineText: {
    color: '#721c24',
    fontSize: 14,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: 20,
  },
  loadingText: {
    fontSize: 16,
    color: '#666',
    marginTop: 10,
  },
  loadMoreContainer: {
    paddingVertical: 20,
    alignItems: 'center',
  },
  loadMoreText: {
    fontSize: 14,
    color: '#666',
    marginTop: 8,
  },
});

export default Checklist_library;