import React, { useState, useCallback, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  Platform,
  Modal,
  Animated,
  PermissionsAndroid,
  Dimensions,
  StatusBar,
  KeyboardAvoidingView,
  ScrollView,
  Alert,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
import RNFS from 'react-native-fs'; // Added for persistent storage
import sanitizeHtml from 'sanitize-html';
import Toast from 'react-native-toast-message';
import ToastAndroid from 'react-native/Libraries/Components/ToastAndroid/ToastAndroid';
import { useSafeAreaInsets } from 'react-native-safe-area-context';

// Move Dimensions outside component to prevent hooks order issues
const { width, height } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;
const TOTAL_HEADER_HEIGHT = width * 0.25;

// SyncModal Component
const SyncModal = ({ visible, onClose }) => {
  const rotateAnim = React.useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  React.useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>Saving Notice</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function to show toast messages
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.show(message, ToastAndroid.LONG, ToastAndroid.BOTTOM);
  } else {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

// Request camera permission
const requestCameraPermission = async () => {
  if (Platform.OS === 'android') {
    try {
      const granted = await PermissionsAndroid.request(
        PermissionsAndroid.PERMISSIONS.CAMERA,
        {
          title: 'Camera Permission',
          message: 'This app needs access to your camera to take photos.',
          buttonNeutral: 'Ask Me Later',
          buttonNegative: 'Cancel',
          buttonPositive: 'OK',
        }
      );
      return granted === PermissionsAndroid.RESULTS.GRANTED;
    } catch (err) {
      console.warn(err);
      return false;
    }
  }
  return true; // iOS handles permissions via ImagePicker
};

// Request gallery permission
const requestGalleryPermission = async () => {
  if (Platform.OS === 'android') {
    try {
      const permission = Platform.Version >= 33
        ? PermissionsAndroid.PERMISSIONS.READ_MEDIA_IMAGES
        : PermissionsAndroid.PERMISSIONS.READ_EXTERNAL_STORAGE;
      const granted = await PermissionsAndroid.request(
        permission,
        {
          title: 'Gallery Permission',
          message: 'This app needs access to your gallery to upload images.',
          buttonNeutral: 'Ask Me Later',
          buttonNegative: 'Cancel',
          buttonPositive: 'OK',
        }
      );
      return granted === PermissionsAndroid.RESULTS.GRANTED;
    } catch (err) {
      console.warn(err);
      return false;
    }
  }
  return true; // iOS handles permissions via ImagePicker
};

// Copy file to persistent storage
const copyFileToPersistentStorage = async (uri, fileName) => {
  try {
    const extension = fileName.split('.').pop().toLowerCase();
    const newFileName = `image_${Date.now()}.${extension}`;
    const destPath = `${RNFS.DocumentDirectoryPath}/${newFileName}`;
    await RNFS.copyFile(uri, destPath);
    return { uri: destPath, name: newFileName };
  } catch (error) {
    console.error('Error copying file to persistent storage:', error);
    throw new Error('Failed to copy file to storage');
  }
};

const Add_NoticeONNB = () => {
  const navigation = useNavigation();
  const insets = useSafeAreaInsets();
  const [title, setTitle] = useState('');
  const [description, setDescription] = useState('');
  const [selectedImage, setSelectedImage] = useState(null); // Updated to match DocumentuploadS
  const [errors, setErrors] = useState({ title: '', description: '', image: '' });
  const [loading, setLoading] = useState(false);
  const [showImageModal, setShowImageModal] = useState(false);

  const slideAnim = useRef(new Animated.Value(Dimensions.get('window').height)).current;

  // Animate modal slide-in/slide-out
  React.useEffect(() => {
    if (showImageModal) {
      Animated.spring(slideAnim, {
        toValue: 0,
        useNativeDriver: true,
      }).start();
    } else {
      Animated.timing(slideAnim, {
        toValue: Dimensions.get('window').height,
        duration: 300,
        useNativeDriver: true,
      }).start();
    }
  }, [showImageModal]);

  // Sanitize input
  const sanitizeInput = useCallback((text) => {
    return sanitizeHtml(text, {
      allowedTags: [],
      allowedAttributes: {},
    });
  }, []);

  // Validate form
  const validateForm = useCallback(() => {
    const newErrors = {
      title: title.trim() ? '' : 'Title is required',
      description: description.trim() ? '' : 'Description is required',
      image: '',
    };
    setErrors(newErrors);
    return Object.values(newErrors).every((error) => !error);
  }, [title, description]);

  // Handle image selection (camera or gallery)
  const handleImageSelection = async (option) => {
    try {
      let imageInfo;
      if (option === 'camera') {
        const granted = await requestCameraPermission();
        if (!granted) {
          showToast('error', 'Permission Denied', 'Camera permission denied');
          return;
        }
        const options = {
          mediaType: 'photo',
          maxWidth: 1024,
          maxHeight: 1024,
          quality: 0.8,
        };
        const response = await launchCamera(options);
        if (response.didCancel) {
          console.log('User cancelled camera');
          return;
        } else if (response.errorCode) {
          throw new Error(response.errorMessage || 'Camera error');
        }
        const uri = response.assets[0].uri;
        const fileName = response.assets[0].fileName || `photo_${Date.now()}.jpg`;
        const extension = fileName.split('.').pop().toLowerCase();
        if (!['jpg', 'jpeg', 'png', 'gif'].includes(extension)) {
          throw new Error('Unsupported image format. Please select JPG, JPEG, PNG, or GIF.');
        }
        // Validate file size (max 5MB)
        if (response.assets[0].fileSize > 5 * 1024 * 1024) {
          throw new Error('Image size exceeds 5MB limit.');
        }
        imageInfo = await copyFileToPersistentStorage(uri, fileName);
      } else if (option === 'gallery') {
        const granted = await requestGalleryPermission();
        if (!granted) {
          showToast('error', 'Permission Denied', 'Gallery permission denied');
          return;
        }
        const options = {
          mediaType: 'photo',
          maxWidth: 1024,
          maxHeight: 1024,
          quality: 0.8,
        };
        const response = await launchImageLibrary(options);
        if (response.didCancel) {
          console.log('User cancelled gallery');
          return;
        } else if (response.errorCode) {
          throw new Error(response.errorMessage || 'Gallery error');
        }
        const uri = response.assets[0].uri;
        const fileName = response.assets[0].fileName || `image_${Date.now()}.jpg`;
        const extension = fileName.split('.').pop().toLowerCase();
        if (!['jpg', 'jpeg', 'png', 'gif'].includes(extension)) {
          throw new Error('Unsupported image format. Please select JPG, JPEG, PNG, or GIF.');
        }
        // Validate file size (max 5MB)
        if (response.assets[0].fileSize > 5 * 1024 * 1024) {
          throw new Error('Image size exceeds 5MB limit.');
        }
        imageInfo = await copyFileToPersistentStorage(uri, fileName);
      }

      setSelectedImage(imageInfo);
      setErrors((prevErrors) => ({ ...prevErrors, image: '' }));
      setShowImageModal(false);
      showToast('success', 'Success', 'Image selected successfully.');
    } catch (error) {
      console.error('Image selection error:', error);
      showToast('error', 'Error', error.message || 'Failed to select image');
    }
  };

  // Upload image to Firebase Storage
  const uploadImageToFirebase = useCallback(async (uri, userUid, noticeId) => {
    try {
      const fileExtension = uri.split('.').pop().toLowerCase();
      const fileName = `${noticeId}.${fileExtension}`;
      const reference = storage().ref(`notices/${userUid}/${fileName}`);
      
      const task = reference.putFile(uri);
      
      task.on('state_changed', 
        (snapshot) => {
          const progress = (snapshot.bytesTransferred / snapshot.totalBytes) * 100;
          console.log(`Upload is ${progress}% done`);
        },
        (error) => {
          console.error('Upload error:', error);
          throw error;
        }
      );

      await task;
      const downloadURL = await reference.getDownloadURL();
      return downloadURL;
    } catch (error) {
      console.error('Error uploading image:', error);
      throw error;
    }
  }, []);

  // Save notice to Firebase
  const handleAddNotice = useCallback(async () => {
    if (!validateForm()) return;

    setLoading(true);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        showToast('error', 'Error', 'No user logged in. Please log in again.');
        navigation.replace('LoginScreens');
        return;
      }

      const noticeId = Date.now().toString();
      let imageUrl = null;

      if (selectedImage) {
        imageUrl = await uploadImageToFirebase(selectedImage.uri, userUid, noticeId);
      }

      const notice = {
        noticeId,
        title: sanitizeInput(title),
        description: sanitizeInput(description),
        imageUrl,
        createdAt: new Date().toISOString(),
      };

      await database()
        .ref(`/users/${userUid}/notices/${noticeId}`)
        .set(notice);

      // Send push notification to all users with FCM tokens
      try {
        console.log('📢 Sending notice push notification to all users...');
        
        // First, get all users with FCM tokens
        const getAllUsersResponse = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/getAllUsersWithFCMTokens', {
          method: 'GET',
          headers: {
            'Content-Type': 'application/json',
          },
        });

        const getAllUsersResult = await getAllUsersResponse.json();
        console.log('📋 Get all users result:', getAllUsersResult);

        if (getAllUsersResult.success && getAllUsersResult.users && getAllUsersResult.users.length > 0) {
          console.log(`📊 Found ${getAllUsersResult.users.length} users with FCM tokens`);
          
          let totalSuccess = 0;
          let totalFailed = 0;
          let totalInvalidTokens = 0;
          const notificationResults = [];

          // Send notification to each user
          for (const user of getAllUsersResult.users) {
            try {
              console.log(`📤 Sending notification to user: ${user.name} (${user.email})`);
              
              const notificationResponse = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/sendNotificationToAllUserDevices', {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                  userId: user.userId,
                  title: `📢 ${notice.title}`,
                  body: notice.description.length > 100 ? notice.description.substring(0, 100) + '...' : notice.description,
                  data: {
                    type: 'notice_board',
                    noticeId: notice.noticeId,
                    title: notice.title,
                    description: notice.description,
                    imageUrl: notice.imageUrl,
                    createdAt: notice.createdAt,
                    senderUserId: userUid,
                    action: 'view_notice',
                  },
                }),
              });

              const notificationResult = await notificationResponse.json();
              console.log(`📋 Notification result for ${user.name}:`, notificationResult);

              if (notificationResult.success) {
                totalSuccess += notificationResult.successfulDevices || 0;
                totalFailed += notificationResult.failedDevices || 0;
                totalInvalidTokens += notificationResult.invalidTokens || 0;
                notificationResults.push({
                  user: user.name,
                  email: user.email,
                  status: 'success',
                  devices: notificationResult.successfulDevices || 0,
                  failed: notificationResult.failedDevices || 0,
                  invalidTokens: notificationResult.invalidTokens || 0,
                });
              } else {
                totalFailed += user.totalDevices || 0;
                notificationResults.push({
                  user: user.name,
                  email: user.email,
                  status: 'failed',
                  error: notificationResult.error,
                });
              }
            } catch (userError) {
              console.error(`❌ Error sending notification to user ${user.name}:`, userError);
              totalFailed += user.totalDevices || 0;
              notificationResults.push({
                user: user.name,
                email: user.email,
                status: 'error',
                error: userError.message,
              });
            }
          }

          // Log comprehensive results
          console.log('📊 Notice push notification results:', {
            totalUsers: getAllUsersResult.users.length,
            totalSuccess,
            totalFailed,
            totalInvalidTokens,
            results: notificationResults,
          });

          // Show success message with notification details
          const successMessage = `✅ Notice added successfully!\n\n📢 Push notifications sent:\n📱 Success: ${totalSuccess} devices\n❌ Failed: ${totalFailed} devices\n🔄 Invalid tokens: ${totalInvalidTokens} devices\n👥 Users notified: ${getAllUsersResult.users.length}`;
          
          showToast('success', 'Notice Added', `Notice added and notifications sent to ${getAllUsersResult.users.length} users!`);
          
          // Log detailed results for debugging
          console.log('🎉 Notice creation and notification summary:', successMessage);
          
        } else {
          console.log('⚠️ No users with FCM tokens found for push notifications');
          showToast('success', 'Notice Added', 'Notice added successfully! (No users to notify)');
        }

      } catch (notificationError) {
        console.error('❌ Error sending push notifications:', notificationError);
        // Don't fail the notice creation if notification fails
        showToast('warning', 'Notice Added', 'Notice added successfully, but push notifications failed');
      }

      // Clear form and navigate back
      setTitle('');
      setDescription('');
      setSelectedImage(null);
      navigation.goBack();
      
    } catch (error) {
      console.error('Error adding notice:', error);
      showToast('error', 'Error', `Failed to add notice: ${error.message}`);
    } finally {
      setLoading(false);
    }
  }, [validateForm, title, description, selectedImage, sanitizeInput, uploadImageToFirebase, navigation]);

  // Test push notification system for notices
  const testNoticePushNotification = async () => {
    if (!title.trim() || !description.trim()) {
      showToast('error', 'Error', 'Please fill in title and description first to test notifications');
      return;
    }

    setLoading(true);
    try {
      console.log('🧪 Testing notice push notification system...');
      
      // First, get all users with FCM tokens
      const getAllUsersResponse = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/getAllUsersWithFCMTokens', {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      const getAllUsersResult = await getAllUsersResponse.json();
      console.log('📋 Get all users result:', getAllUsersResult);

      if (getAllUsersResult.success && getAllUsersResult.users && getAllUsersResult.users.length > 0) {
        console.log(`📊 Found ${getAllUsersResult.users.length} users with FCM tokens`);
        
        let totalSuccess = 0;
        let totalFailed = 0;
        const testResults = [];

        // Send test notification to each user
        for (const user of getAllUsersResult.users) {
          try {
            console.log(`🧪 Sending test notification to user: ${user.name} (${user.email})`);
            
            const notificationResponse = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/sendNotificationToAllUserDevices', {
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
              },
              body: JSON.stringify({
                userId: user.userId,
                title: `🧪 Test Notice: ${title}`,
                body: `Test notification: ${description}`,
                data: {
                  type: 'test_notice',
                  noticeId: `test_${Date.now()}`,
                  title: title,
                  description: description,
                  isTest: true,
                  timestamp: new Date().toISOString(),
                  action: 'test_notice',
                },
              }),
            });

            const notificationResult = await notificationResponse.json();
            console.log(`📋 Test notification result for ${user.name}:`, notificationResult);

            if (notificationResult.success) {
              totalSuccess += notificationResult.successfulDevices || 0;
              totalFailed += notificationResult.failedDevices || 0;
              testResults.push({
                user: user.name,
                email: user.email,
                status: 'success',
                devices: notificationResult.successfulDevices || 0,
                failed: notificationResult.failedDevices || 0,
              });
            } else {
              totalFailed += user.totalDevices || 0;
              testResults.push({
                user: user.name,
                email: user.email,
                status: 'failed',
                error: notificationResult.error,
              });
            }
          } catch (userError) {
            console.error(`❌ Error sending test notification to user ${user.name}:`, userError);
            totalFailed += user.totalDevices || 0;
            testResults.push({
              user: user.name,
              email: user.email,
              status: 'error',
              error: userError.message,
            });
          }
        }

        // Show comprehensive test results
        const testMessage = `🧪 Test Notification Results:\n\n📊 Total Users: ${getAllUsersResult.users.length}\n📱 Success: ${totalSuccess} devices\n❌ Failed: ${totalFailed} devices\n\n✅ Test completed! Check console for detailed results.`;
        
        Alert.alert(
          'Test Results', 
          testMessage,
          [{ text: 'OK' }]
        );

        // Log detailed results for debugging
        console.log('🧪 Notice push notification test results:', {
          totalUsers: getAllUsersResult.users.length,
          totalSuccess,
          totalFailed,
          results: testResults,
        });
        
      } else {
        console.log('⚠️ No users with FCM tokens found for testing');
        Alert.alert('Test Result', 'No users with FCM tokens found. Users need to login first to receive notifications.');
      }

    } catch (error) {
      console.error('❌ Error testing notice push notifications:', error);
      Alert.alert('Test Error', `Failed to test notifications: ${error.message}`);
    } finally {
      setLoading(false);
    }
  };

  // Check FCM status and show user count
  const checkFCMStatus = async () => {
    setLoading(true);
    try {
      console.log('🔍 Checking FCM status for notice notifications...');
      
      const response = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/getAllUsersWithFCMTokens', {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      const result = await response.json();
      console.log('📋 FCM status check result:', result);

      if (result.success && result.users && result.users.length > 0) {
        const totalDevices = result.users.reduce((sum, user) => sum + user.totalDevices, 0);
        const statusMessage = `📱 FCM Status: Ready for Notifications\n\n👥 Users with FCM tokens: ${result.users.length}\n📱 Total devices: ${totalDevices}\n\n✅ Your notice will be sent to all registered users when published.`;
        
        Alert.alert('FCM Status', statusMessage, [{ text: 'OK' }]);
        
        // Log detailed status
        console.log('📊 FCM Status for notices:', {
          totalUsers: result.users.length,
          totalDevices,
          users: result.users.map(u => ({ name: u.name, email: u.email, devices: u.totalDevices }))
        });
        
      } else {
        Alert.alert('FCM Status', 'No users with FCM tokens found. Users need to login to the app first to receive notifications.', [{ text: 'OK' }]);
      }

    } catch (error) {
      console.error('❌ Error checking FCM status:', error);
      Alert.alert('Error', `Failed to check FCM status: ${error.message}`);
    } finally {
      setLoading(false);
    }
  };

  return (
    <View style={styles.screen}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity
            style={styles.backButton}
            onPress={() => navigation.goBack()}
            accessibilityRole="button"
            accessibilityLabel="Go Back"
          >
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Add Notice</Text>
          <View style={{ width: 24 }} />
        </View>
      </LinearGradient>
      <KeyboardAvoidingView
        style={{ flex: 1 }}
        behavior={Platform.OS === 'ios' ? 'padding' : undefined}
        keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 0}
      >
        <ScrollView contentContainerStyle={styles.content} keyboardShouldPersistTaps="handled">
          {/* Title Input */}
          <Text style={styles.inputLabel}>Title</Text>
          <TextInput
            style={styles.input}
            value={title}
            onChangeText={setTitle}
            placeholder="Enter notice title"
            placeholderTextColor="#888"
            accessibilityLabel="Notice Title"
          />
          {errors.title ? <Text style={styles.errorText}>{errors.title}</Text> : null}

          {/* Description Editor */}
          <Text style={styles.inputLabel}>Description</Text>
          <TextInput
            style={styles.editor}
            value={description}
            onChangeText={setDescription}
            placeholder="Write your notice description..."
            placeholderTextColor="#888"
            multiline
            numberOfLines={6}
            textAlignVertical="top"
            accessibilityLabel="Notice Description"
          />
          {errors.description ? <Text style={styles.errorText}>{errors.description}</Text> : null}

          {/* Image Upload Button */}
          <TouchableOpacity
            style={[styles.imageUploadBtn, errors.image ? { borderColor: '#FF0000' } : null]}
            onPress={() => setShowImageModal(true)}
            accessibilityRole="button"
            accessibilityLabel="Upload Image"
          >
            <MaterialIcons name="image" size={32} color="#3481BC" />
            <Text style={styles.imageUploadText}>
              {selectedImage ? 'Image Selected' : 'Upload Image'}
            </Text>
          </TouchableOpacity>
          {selectedImage && (
            <Text style={styles.selectedText} numberOfLines={1} ellipsizeMode="middle">
              {selectedImage.name}
            </Text>
          )}
          {errors.image ? <Text style={styles.errorText}>{errors.image}</Text> : null}

          {/* Add Notice Button - moved out of ScrollView for fixed position */}
        </ScrollView>
        <View style={[styles.addNoticeBtnWrapFixed, { paddingBottom: Math.max(insets.bottom || 0, 16) + (Platform.OS === 'android' ? 60 : 0) }]}>
          <TouchableOpacity
            style={styles.addNoticeBtn}
            activeOpacity={0.85}
            onPress={handleAddNotice}
            accessibilityRole="button"
            accessibilityLabel="Add Notice"
          >
            <LinearGradient colors={['#3481BC', '#025383']} style={styles.addNoticeGradient}>
              <Text style={styles.addNoticeText}>Add Notice</Text>
            </LinearGradient>
          </TouchableOpacity>
        </View>
      </KeyboardAvoidingView>

      {/* Image Selection Modal */}
      <Modal
        transparent
        visible={showImageModal}
        animationType="none"
        onRequestClose={() => setShowImageModal(false)}
      >
        <TouchableOpacity
          style={styles.modalOverlay}
          activeOpacity={1}
          onPress={() => setShowImageModal(false)}
        >
          <Animated.View
            style={[styles.imageModalContainer, { transform: [{ translateY: slideAnim }] }]}
          >
            <TouchableOpacity style={styles.imageModalOption} onPress={() => handleImageSelection('camera')}>
              <View style={styles.imageModalOptionContent}>
                <MaterialIcons name="camera-alt" size={24} color="#3481BC" style={styles.imageModalIcon} />
                <Text style={styles.imageModalText}>Take Picture</Text>
              </View>
            </TouchableOpacity>
            <TouchableOpacity style={styles.imageModalOption} onPress={() => handleImageSelection('gallery')}>
              <View style={styles.imageModalOptionContent}>
                <MaterialIcons name="photo-library" size={24} color="#3481BC" style={styles.imageModalIcon} />
                <Text style={styles.imageModalText}>Choose from Gallery</Text>
              </View>
            </TouchableOpacity>
            <TouchableOpacity style={styles.imageModalOption} onPress={() => setShowImageModal(false)}>
              <Text style={[styles.imageModalText, { color: '#FF0000' }]}>Cancel</Text>
            </TouchableOpacity>
          </Animated.View>
        </TouchableOpacity>
      </Modal>

      <SyncModal visible={loading} onClose={() => setLoading(false)} />
    </View>
  );
};

const styles = StyleSheet.create({
  screen: {
    flex: 1,
    backgroundColor: '#f6f7f9',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'ios' ? width * 0.08 : width * 0.04,
    height: '100%',
  },
  backButton: {
    padding: 4,
    marginRight: 8,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    letterSpacing: 0.5,
    flex: 1,
    textAlign: 'center',
    
  },
  content: {
    padding: 22,
    alignItems: 'stretch',
    justifyContent: 'flex-start',
    paddingBottom: 120,
  },
  inputLabel: {
    fontSize: 15,
    color: '#29486A',
    fontWeight: 'bold',
    marginBottom: 6,
    marginLeft: 2,
  },
  input: {
    borderWidth: 1,
    borderColor: '#bcd2ee',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    color: '#222',
    backgroundColor: '#fff',
    marginBottom: 18,
  },
  editor: {
    borderWidth: 1,
    borderColor: '#bcd2ee',
    borderRadius: 8,
    paddingHorizontal: 12,
    paddingVertical: 12,
    fontSize: 16,
    color: '#222',
    backgroundColor: '#fff',
    minHeight: 120,
    marginBottom: 22,
  },
  imageUploadBtn: {
    flexDirection: 'row',
    alignItems: 'center',
    alignSelf: 'flex-start',
    backgroundColor: '#eaf2fb',
    borderRadius: 8,
    paddingVertical: 10,
    paddingHorizontal: 18,
    borderWidth: 1,
    borderColor: '#3481BC',
  },
  imageUploadText: {
    color: '#3481BC',
    fontSize: 16,
    fontWeight: 'bold',
    marginLeft: 10,
  },
  selectedText: {
    fontSize: 14,
    color: '#3481BC',
    marginTop: 4,
    marginBottom: 16,
    marginLeft: 12,
  },
  addNoticeBtnWrap: {
    marginTop: 32,
    alignItems: 'center',
    width: '100%',
  },
  addNoticeBtnWrapFixed: {
    position: 'absolute',
    left: 0,
    right: 0,
    bottom: 0,
    backgroundColor: 'transparent',
    paddingHorizontal: 16,
    alignItems: 'center',
    zIndex: 10,
  },
  addNoticeBtn: {
    width: '100%',
    borderRadius: 1,
    overflow: 'hidden',
    height: 35,
  },
  addNoticeGradient: {
    height: 35,
    alignItems: 'center',
    justifyContent: 'center',
    borderRadius: 1,
    width: '100%',
  },
  addNoticeText: {
    color: '#fff',
    fontSize: 17,
    fontWeight: 'bold',
    letterSpacing: 0.5,
  },
  errorText: {
    color: '#FF0000',
    fontSize: 12,
    marginBottom: 10,
    marginLeft: 5,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'flex-end',
  },
  imageModalContainer: {
    backgroundColor: '#fff',
    borderTopLeftRadius: 20,
    borderTopRightRadius: 20,
    padding: 20,
  },
  imageModalOption: {
    paddingVertical: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    alignItems: 'center',
  },
  imageModalOptionContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  imageModalIcon: {
    marginRight: 10,
  },
  imageModalText: {
    fontSize: 18,
    color: '#3481BC',
  },
});

export default Add_NoticeONNB;
