import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  Alert,
  Dimensions,
  Modal,
  ScrollView,

  Image,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';

import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';

const { width, height } = Dimensions.get('window');

const SyncModal = ({ visible, onClose, message = 'Downloading PDF...' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalText}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_155_Step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [formData, setFormData] = useState({
    vehicleFleetNumber: '',
    vehicleType: '',
    artisan: '',
    plannedAction: '',
  });
  const [signature, setSignature] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [requestNumber, setRequestNumber] = useState('');

  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);

  useEffect(() => {
    loadSavedData();
  }, []);

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse155Step3');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        setFormData(parsedData.formData || {
          vehicleFleetNumber: '',
          vehicleType: '',
          artisan: '',
          plannedAction: '',
        });
        setSignature(parsedData.signature || '');
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async () => {
    try {
      const dataToSave = {
        formData,
        signature,
        timestamp: new Date().toISOString()
      };
      await AsyncStorage.setItem('hse155Step3', JSON.stringify(dataToSave));
    } catch (error) {
      console.error('Error saving data:', error);
    }
  };

  const handleInputChange = (field, value) => {
    const updatedData = { ...formData, [field]: value };
    setFormData(updatedData);
    // Save data after a short delay to avoid too many saves
    setTimeout(() => saveData(), 100);
  };

  const handleSubmit = async () => {
    if (!formData.vehicleFleetNumber || !formData.vehicleType || !formData.artisan) {
      Alert.alert('Error', 'Please fill in all required fields');
      return;
    }
    if (!signature) {
      Alert.alert('Error', 'Please add your signature');
      return;
    }
    
    try {
      await saveData();
      
      // Generate PDF
      await generatePDF();
      
      // Show success modal with options
      setShowSuccessModal(true);
    } catch (error) {
      console.error('Error submitting form:', error);
      Alert.alert('Error', 'Failed to submit form. Please try again.');
    }
  };

  const handleBack = () => {
    setActiveStep(2);
  };

  const generatePDF = async () => {
    try {
      setIsLoading(true);
      setSyncModalVisible(true);
      setUploadProgress(10);
      
      // Load data from all steps
      const step1Data = await loadStep1Data();
      const step2Data = await loadStep2Data();
      const step3Data = {
        ...formData,
        date: new Date().toLocaleDateString(), // Auto-pick today's date
        signature: signature, // Include the signature
      };
      
      // Get the request number from step1 data
      const dbId = step1Data.requestNumber || `${step1Data.countryCode || 'ZMB'}-${step1Data.projectCode || 'PRJ'}-HSE-001`;

      // Set the request number for this report
      setRequestNumber(dbId);

      const html = generatePDFHTML(step1Data, step2Data, step3Data);
      
      const options = {
        html,
        fileName: `HSE_155_${dbId}`,
        directory: 'Documents',
      };

      setUploadProgress(30);
      const file = await RNHTMLtoPDF.convert(options);
      
      if (file.filePath) {
        setUploadProgress(50);
        
        // Get admin user ID
        const userId = await getAdminUserId();
        
        // Upload PDF to Firebase Storage
        const storagePath = `GlobalHSEInspections/admins/${userId}/reports/${dbId}.pdf`;
        const pdfDownloadUrl = await uploadFileToFirebaseStorage(file.filePath, storagePath);
        
        setUploadProgress(70);
        
        // Save report data to Firebase Database
        const reportData = {
          id: dbId,
          userId,
          rig: step1Data.selectedEquipment ? (step1Data.selectedEquipment.equipmentName || step1Data.selectedEquipment.equipmentNumber || 'N/A') : 'N/A',
          location: step1Data.gpsAddress || step1Data.coordinates || 'N/A',
          date: step3Data.date,
          artisan: step3Data.artisan,
          selectedRole: 'Artisan',
          pdfDownloadUrl: pdfDownloadUrl,
          createdAt: new Date().toISOString(),
          status: 'completed',
          step1: step1Data,
          step2: step2Data,
          step3: {
            date: step3Data.date,
            vehicleFleetNumber: step3Data.vehicleFleetNumber,
            vehicleType: step3Data.vehicleType,
            artisan: step3Data.artisan,
            signature: step3Data.signature,
          },
          images: step2Data.images || {}
        };
        
        const dbRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${dbId}`);
        await dbRef.set(reportData);
        
        setUploadProgress(90);
        
        // Increment the global report counter
        await incrementReportCounter();
        
        setUploadProgress(100);
        
        // Remove data from all steps after successful generation
        await flushStepData(dbId);
        
        Toast.show({
          type: 'success',
          text1: 'PDF Generated & Saved',
          text2: `Report saved with ID: ${dbId}`,
          position: 'bottom'
        });
      }
    } catch (error) {
      console.error('Error generating PDF:', error);
      Toast.show({
        type: 'error',
        text1: 'PDF Generation Failed',
        text2: error.message || 'Please try again',
        position: 'bottom'
      });
    } finally {
      setIsLoading(false);
      setSyncModalVisible(false);
      setUploadProgress(0);
    }
  };

  const loadStep1Data = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hseStep1');
      return savedData ? JSON.parse(savedData) : {};
    } catch (error) {
      console.error('Error loading step 1 data:', error);
      return {};
    }
  };

  const loadStep2Data = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse155Step2');
      return savedData ? JSON.parse(savedData) : {};
    } catch (error) {
      console.error('Error loading step 2 data:', error);
      return {};
    }
  };

  const getAdminUserId = async () => {
    const userEmail = await AsyncStorage.getItem('userEmail');
    if (!userEmail) throw new Error('User email not found. Please log in again.');
    const userSnapshot = await database()
      .ref('/Globalusers/admins')
      .orderByChild('email')
      .equalTo(userEmail)
      .once('value');
    const userData = userSnapshot.val();
    if (!userData) throw new Error('User not found in admin database.');
    return Object.keys(userData)[0];
  };

  const incrementReportCounter = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      let currentNumber = snapshot.val();
      
      // Initialize counter if it doesn't exist
      if (currentNumber === null || currentNumber === undefined) {
        currentNumber = 0;
      }
      
      currentNumber++;
      
      // Update the counter in Firebase
      await counterRef.set(currentNumber);
      
      return currentNumber.toString().padStart(3, '0');
    } catch (error) {
      console.error('Error incrementing report counter:', error);
      return '001';
    }
  };

  const flushStepData = async (hseId) => {
    try {
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hse155Step2');
      await AsyncStorage.removeItem('hse155Step3');
      await AsyncStorage.removeItem(`hse155_offline_${hseId}`);
      console.log('Successfully cleared all step data for HSE155');
    } catch (error) {
      console.error('Error flushing step data:', error);
      throw new Error('Failed to clear local data');
    }
  };

  const uploadFileToFirebaseStorage = async (filePath, storagePath) => {
    try {
      const reference = storage().ref(storagePath);
      await reference.putFile(filePath);
      const downloadUrl = await reference.getDownloadURL();
      return downloadUrl;
    } catch (error) {
      console.error('Error uploading file to Firebase Storage:', error);
      throw error;
    }
  };

  const handleViewPDF = async () => {
    if (!requestNumber) {
      Alert.alert('Error', 'Report ID not available.');
      return;
    }

    try {
      const userId = await getAdminUserId();
      const reportRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${requestNumber}`);
      const snapshot = await reportRef.once('value');
      const reportData = snapshot.val();

      if (!reportData) {
        Alert.alert('Error', 'Report not found in the database.');
        return;
      }

      const pdfUrl = reportData.pdfDownloadUrl;
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.');
        return;
      }

      // Close modal first
      setShowSuccessModal(false);
      
      // Navigate to HSEViewPDF screen
      navigation.navigate('HSEViewPDF', {
        pdfUrl: pdfUrl,
        onGoBack: () => {
          // Handle going back from PDF view
          navigation.navigate('HSEInspectionHome');
        }
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      Alert.alert('Error', 'Could not fetch the PDF URL. Please try again.');
    }
  };

  const generatePDFHTML = (step1, step2, step3) => {
    const { selections = {}, notes = {}, images = {} } = step2;
    
    // Generate checklist rows with complete structure
    let checklistRows = '';
    
    try {
      // Import HSE_155_JSON to get section names
      const HSE_155_JSON = require('./HSE_155_json.js').default;
      
      // Generate checklist rows for each section
      HSE_155_JSON.sections.forEach((section) => {
        // Add items for this section (no section headers)
        section.subItems.forEach((item) => {
          const selection = selections[item.key] || '';
          const note = notes[item.key] || '';
          const image = images[item.key] || '';
          
          // Create image HTML if image exists
          let imageHTML = '';
          if (image) {
            // Always show image indicator for any type of image
            imageHTML = `<div style="width: 60px; height: 60px; border-radius: 4px; border: 2px solid #015185; background-color: #e8f4fd; display: flex; align-items: center; justify-content: center; font-size: 9px; color: #015185; text-align: center; font-weight: bold;">
              📸<br/>PHOTO<br/>TAKEN
            </div>`;
          }
          
          checklistRows += `
            <tr>
              <td>${item.itemName}</td>
              <td style="text-align: center;">${selection === 'YES/OK' ? '✓' : ''}</td>
              <td style="text-align: center;">${selection === 'NO/DEF' ? '✓' : ''}</td>
              <td>${note || ''}</td>
              <td style="text-align: center;">${imageHTML}</td>
            </tr>
          `;
        });
      });
    } catch (error) {
      console.error('Error generating checklist rows:', error);
      checklistRows = '';
    }

    return `
      <!DOCTYPE html>
      <html lang="en">
      <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>HSE-FOR-8.1.1 Workshop Vehicle Pre-Release</title>
          <style>
              body {
                  font-family: Arial, sans-serif;
                  margin: 10px;
                  padding: 0;
                  font-size: 14px;
              }
              .page-title {
                  border-bottom: 3px solid #015185;
                  margin-bottom: 20px;
                  padding-bottom: 15px;
              }
              h1 {
                  text-align: center;
                  margin-bottom: 10px;
                  font-size: 1.5em;
              }
              .header-container {
                  display: flex;
                  flex-direction: row;
                  justify-content: space-between;
                  align-items: flex-start;
                  margin-bottom: 20px;
              }
              .logo-section {
                  flex: 0 0 auto;
                  padding: 10px;
              }
              .header-info-section {
                  flex: 0 0 auto;
                  display: flex;
                  justify-content: flex-end;
              }
              .header-table {
                  border-collapse: collapse;
                  width: 300px;
                  margin: 0;
              }
              .header-table th, .header-table td {
                  border: 1px solid #ddd;
                  padding: 8px;
                  text-align: left;
                  font-size: 0.9em;
              }
              .header-table th {
                  background-color: #f2f2f2;
                  font-weight: bold;
                  color: #015185;
              }
              .header-table td {
                  background-color: #f8f8f8;
              }
              h2 {
                  text-align: center;
                  font-size: 1.2em;
                  margin: 15px 0;
              }
              table.checklist {
                  width: 100%;
                  border-collapse: collapse;
                  margin-bottom: 15px;
              }
              .checklist th, .checklist td {
                  border: 1px solid #ddd;
                  padding: 6px;
                  text-align: left;
                  font-size: 0.9em;
              }
              .checklist th {
                  background-color: #f2f2f2;
              }

              .info-row td {
                  padding: 8px;
                  background-color: #f8f8f8;
              }
              .checklist th:nth-child(1), .checklist td:nth-child(1) {
                  width: 40%;
                  padding: 8px;
              }
              .checklist th:nth-child(2), .checklist td:nth-child(2) {
                  width: 12%;
                  text-align: center;
                  padding: 8px;
              }
              .checklist th:nth-child(3), .checklist td:nth-child(3) {
                  width: 12%;
                  text-align: center;
                  padding: 8px;
              }
              .checklist th:nth-child(4), .checklist td:nth-child(4) {
                  width: 20%;
                  padding: 8px;
              }
              .checklist th:nth-child(5), .checklist td:nth-child(5) {
                  width: 16%;
                  text-align: center;
                  padding: 8px;
              }
              .checklist th {
                  background-color: #f2f2f2;
                  font-weight: bold;
                  color: #015185;
              }
          </style>
      </head>
      <body>
         
          <div class="header-container">
              <div class="logo-section">
                  <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" style="width: 200px; height: auto;">
              </div>
              <div class="header-info-section">
                  <table class="header-table">
                      <tr>
                          <td><strong>Document ID:</strong></td>
                          <td>HSE-155</td>
                      </tr>
                      <tr>
                          <td><strong>Version Date:</strong></td>
                          <td>21/04/2025</td>
                      </tr>
                      <tr>
                          <td><strong>Version:</strong></td>
                          <td>1</td>
                      </tr>
                      <tr>
                          <td><strong>Approved By:</strong></td>
                          <td>HSM</td>
                      </tr>
                  </table>
              </div>
          </div>

          <table class="checklist">
              <tr class="info-row">
                  <td><strong>Date:</strong> ${step3.date || new Date().toLocaleDateString()}</td>
                  <td><strong>Vehicle Fleet Number:</strong> ${step3.vehicleFleetNumber || ''}</td>
                  <td><strong>Vehicle Type:</strong> ${step3.vehicleType || ''}</td>
                  <td><strong>Artisan:</strong> ${step3.artisan || ''}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Country:</strong> ${step1.country || ''}</td>
                  <td><strong>Project:</strong> ${step1.project || ''}</td>
                  <td><strong>Equipment:</strong> ${step1.selectedEquipment ? (step1.selectedEquipment.equipmentName || step1.selectedEquipment.equipmentNumber || 'N/A') : ''}</td>
                  <td><strong>Request #:</strong> ${step1.requestNumber || ''}</td>
              </tr>
          </table>
          

        

          <table class="checklist">
              <tr>
                  <th>Component</th>
                  <th>Yes/OK</th>
                  <th>No/Def</th>
                  <th>Comments</th>
                  <th>Signature</th>
              </tr>
              ${checklistRows}
          </table>
          
          
          
          <!-- Planned Action Section -->
          ${step3.plannedAction ? `
          <div style="margin-top: 30px; padding: 20px; background-color: #f8f8f8; border-radius: 8px; border: 2px solid #015185;">
              <h3 style="margin: 0 0 15px 0; color: #015185; font-size: 18px; text-align: center; font-weight: bold;">PLANNED ACTION IF ANY:</h3>
              <p style="margin: 0; color: #333; line-height: 1.6; font-size: 14px; text-align: justify;">${step3.plannedAction}</p>
          </div>
          ` : ''}
          
          <!-- Artisan Signature and Date Section -->
          <div style="margin-top: 20px; padding: 20px; border: 2px solid #015185; border-radius: 8px;">
              <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                  <div style="flex: 1; text-align: center;">
                      <h3 style="margin: 0 0 15px 0; color: #015185; font-size: 16px;">ARTISAN SIGNATURE:</h3>
                      <div style="width: 200px; height: 80px; border: 2px solid #015185; border-radius: 8px; margin: 0 auto; display: flex; align-items: center; justify-content: center; background-color: #f8f8f8;">
                          ${step3.signature ? `<img src="${step3.signature}" style="width: 180px; height: 70px; object-fit: contain;" />` : '<span style="color: #666; font-size: 14px;">No Signature</span>'}
                      </div>
                  </div>
                  <div style="flex: 1; text-align: center;">
                      <h3 style="margin: 0 0 15px 0; color: #015185; font-size: 16px;">DATE:</h3>
                      <div style="width: 200px; height: 80px; border: 2px solid #015185; border-radius: 8px; margin: 0 auto; display: flex; align-items: center; justify-content: center; background-color: #f8f8f8;">
                          <span style="color: #333; font-size: 16px; font-weight: bold;">${step3.date || new Date().toLocaleDateString()}</span>
                      </div>
                  </div>
              </div>
          </div>

        </body>
        </html>
      `;
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <Text style={styles.pageTitle}>Workshop Vehicle Pre-Release - Review & Submit</Text>
        
        <View style={styles.section}>
          <View style={styles.sectionHeader}>
            <Text style={styles.sectionTitle}>FORM DETAILS</Text>
          </View>
          
          <View style={styles.formFields}>
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>PLANNED ACTION IF ANY:</Text>
              <TextInput
                style={[styles.input, styles.textArea]}
                value={formData.plannedAction}
                onChangeText={(value) => handleInputChange('plannedAction', value)}
                placeholder="Enter planned action or description..."
                placeholderTextColor="#999"
                multiline
                numberOfLines={4}
                textAlignVertical="top"
              />
            </View>

            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>VEHICLE FLEET NUMBER:</Text>
              <TextInput
                style={styles.input}
                value={formData.vehicleFleetNumber}
                onChangeText={(value) => handleInputChange('vehicleFleetNumber', value)}
                placeholder="Enter vehicle fleet number"
                placeholderTextColor="#999"
              />
            </View>

            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>VEHICLE TYPE:</Text>
              <TextInput
                style={styles.input}
                value={formData.vehicleType}
                onChangeText={(value) => handleInputChange('vehicleType', value)}
                placeholder="Enter vehicle type"
                placeholderTextColor="#999"
              />
            </View>

            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>ARTISAN:</Text>
              <TextInput
                style={styles.input}
                value={formData.artisan}
                onChangeText={(value) => handleInputChange('artisan', value)}
                placeholder="Enter artisan name"
                placeholderTextColor="#999"
              />
            </View>

            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>SIGNATURE:</Text>
              <TouchableOpacity style={styles.signatureBox} onPress={() => navigation.navigate('HSE41_42_signature', {
                role: 'Artisan',
                onSave: (signature) => setSignature(signature)
              })}>
                {signature ? (
                  <Image source={{ uri: signature }} style={styles.signatureImage} resizeMode="contain" />
                ) : (
                  <View style={styles.signaturePlaceholder}>
                    <MaterialIcons name="add" size={24} color="#015185" />
                    <Text style={styles.signaturePlaceholderText}>Add Signature</Text>
                  </View>
                )}
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleSubmit}>
          <Text style={styles.nextButtonText}>Generate Report</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* Sync Modal */}
      <SyncModal 
        visible={syncModalVisible} 
        onClose={() => setSyncModalVisible(false)} 
        message="Downloading PDF..." 
      />

      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalTitle}>Success</Text>
            <Text style={styles.successModalText}>
              HSE Workshop Vehicle Pre-Release Report Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.navigate('HSEInspectionHome');
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPDF}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>



      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  pageTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    paddingVertical: 20,
    backgroundColor: '#F2F2F2',
  },
  section: {
    marginBottom: 30,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    overflow: 'hidden',
    margin: 20,
  },
  sectionHeader: {
    backgroundColor: '#f0f0f0',
    padding: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
  },
  formFields: {
    padding: 20,
  },
  fieldContainer: {
    marginBottom: 20,
  },
  fieldLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 8,
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  textArea: {
    minHeight: 100,
    paddingTop: 12,
    paddingBottom: 12,
  },
  pickerButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    backgroundColor: '#fff',
    minHeight: 40,
    width: '100%',
  },
  pickerButtonText: {
    flex: 1,
    fontSize: 16,
    color: '#333',
  },
  signatureBox: {
    height: height / 5,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#f8f9fa',
  },
  signatureImage: {
    width: '100%',
    height: '100%',
    borderRadius: 6,
  },
  signaturePlaceholder: {
    alignItems: 'center',
    justifyContent: 'center',
  },
  signaturePlaceholderText: {
    fontSize: 14,
    color: '#015185',
    marginTop: 5,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 25,
    width: width * 0.7,
    alignItems: 'center',
    elevation: 8,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
  },
  syncModalText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  syncIcondiv: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginBottom: 12,
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
});

export default HSE_155_Step3;                               