import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  Alert,
  Dimensions,
  Modal,
  ScrollView,
  Platform,
  Image,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';
import DateTimePicker from '@react-native-community/datetimepicker';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';
import { Linking } from 'react-native';

const { width, height } = Dimensions.get('window');

const SyncModal = ({ visible, onClose, message = 'Loading Data' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_125Step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [formData, setFormData] = useState({
    date: '',
    servicedBy: '',
    fleetNumber: '',
    mileage: '',
    comments: '',
    nextServiceDueKm: '',
  });
  const [signature, setSignature] = useState('');
  const [signatureRefreshKey, setSignatureRefreshKey] = useState(0);
  const [isLoading, setIsLoading] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [pdfDownloadUrl, setPdfDownloadUrl] = useState(null);
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [selectedDate, setSelectedDate] = useState(new Date());
  const [generatedPdfPath, setGeneratedPdfPath] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);

  // Monitor network connectivity
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    NetInfo.fetch().then((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    return () => unsubscribe();
  }, []);

  const isValidSignatureUri = (uri) => uri && typeof uri === 'string' && uri.startsWith('data:image');

  const handleSignaturePress = () => {
    navigation.navigate('HSE41_42_signature', {
      role: 'Technician',
      onSave: (signature) => {
        setSignature(signature);
        setSignatureRefreshKey((prev) => prev + 1);
      },
    });
  };

  const handleClearSignature = () => {
    setSignature(null);
    setSignatureRefreshKey((prev) => prev + 1);
  };

  const handleImageError = () => {
    console.error('Signature image failed to load');
    Toast.show({ 
      type: 'error', 
      text1: 'Error', 
      text2: 'Failed to load signature. Please try again.', 
      position: 'bottom' 
    });
    handleClearSignature();
  };

  const handleDateChange = (event, selectedDate) => {
    setShowDatePicker(false);
    if (selectedDate) {
      setSelectedDate(selectedDate);
      setFormData(prev => ({ ...prev, date: selectedDate.toLocaleDateString() }));
    }
  };

  const showDatePickerModal = () => {
    setShowDatePicker(true);
  };

  useEffect(() => {
    loadSavedData();
  }, []);

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse125Step3');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        setFormData(parsedData.formData || {});
        setSignature(parsedData.signature || '');
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async () => {
    try {
      const dataToSave = {
        formData,
        signature,
        timestamp: new Date().toISOString()
      };
      await AsyncStorage.setItem('hse125Step3', JSON.stringify(dataToSave));
    } catch (error) {
      console.error('Error saving data:', error);
    }
  };

  const handleInputChange = (field, value) => {
    const updatedData = { ...formData, [field]: value };
    setFormData(updatedData);
    saveData();
  };

  const handleSubmit = async () => {
    // Validate required fields
    if (!formData.date || !formData.servicedBy || !formData.fleetNumber || !formData.mileage) {
      Alert.alert('Error', 'Please fill in all required fields');
      return;
    }

    // Validate signature
    if (!signature) {
      Alert.alert('Error', 'Please add your signature');
      return;
    }

    try {
      await saveData();
      
      // Generate PDF
      await generatePDF();
      
      // Show success modal with options
      setShowSuccessModal(true);
    } catch (error) {
      console.error('Error submitting form:', error);
      Alert.alert('Error', 'Failed to submit form. Please try again.');
    }
  };

  const handleBack = () => {
    setActiveStep(2);
  };

  const handleGoToHSEHome = () => {
    setShowSuccessModal(false);
    navigation.navigate('HSEInspectionHome');
  };

  const handleViewPDF = () => {
    if (generatedPdfPath) {
      setShowSuccessModal(false);
      // For local file paths, we need to convert to file:// URL format
      const pdfUrl = generatedPdfPath.startsWith('file://') ? generatedPdfPath : `file://${generatedPdfPath}`;
      navigation.navigate('HSEViewPDF', {
        pdfUrl: pdfUrl,
        onGoBack: () => navigation.navigate('HSEInspectionHome')
      });
    } else {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'PDF not generated yet. Please try again.',
        position: 'bottom'
      });
    }
  };

  const generatePDF = async () => {
    try {
      setSyncModalVisible(true);
      
      // Load data from all steps
      const step1Data = await loadStep1Data();
      const step2Data = await loadStep2Data();
      const step3Data = formData;
      
      // Ensure signature is available
      let signatureData = step3Data.signature;
      if (!signatureData) {
        try {
          const savedStep3 = await AsyncStorage.getItem('hse125Step3');
          if (savedStep3) {
            const parsedStep3 = JSON.parse(savedStep3);
            signatureData = parsedStep3.signature || signatureData;
          }
        } catch (error) {
          console.log('Could not load signature from AsyncStorage:', error);
        }
      }
      
      // Update step3Data with signature
      const finalStep3Data = {
        ...step3Data,
        signature: signatureData
      };
      
      const html = generatePDFHTML(step1Data, step2Data, finalStep3Data);
      
      const options = {
        html,
        fileName: `HSE_125_${step1Data.requestNumber || 'Report'}_${new Date().toISOString().split('T')[0]}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      
      if (file.filePath) {
        setGeneratedPdfPath(file.filePath);
        Toast.show({
          type: 'success',
          text1: 'PDF Generated',
          text2: `Saved to: ${file.filePath}`,
          position: 'center'
        });
      }
    } catch (error) {
      console.error('Error generating PDF:', error);
      Toast.show({
        type: 'error',
        text1: 'PDF Generation Failed',
        text2: error.message || 'Please try again',
        position: 'bottom'
      });
    } finally {
      setSyncModalVisible(false);
    }
  };

  const loadStep1Data = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hseStep1');
      return savedData ? JSON.parse(savedData) : {};
    } catch (error) {
      console.error('Error loading step 1 data:', error);
      return {};
    }
  };

  const loadStep2Data = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse125Step2');
      return savedData ? JSON.parse(savedData) : {};
    } catch (error) {
      console.error('Error loading step 2 data:', error);
      return {};
    }
  };

  const generatePDFHTML = (step1, step2, step3) => {
    const { selections = {}, notes = {} } = step2;
    
    // Helper function to get position-specific data
    const getPositionData = (baseKey, positions) => {
      let positionData = '';
      positions.forEach(position => {
        const positionKey = `${baseKey}_${position.replace(/\s+/g, '_')}`;
        const positionValue = selections[positionKey] || '';
        const positionNote = notes[positionKey] || '';
        if (positionValue || positionNote) {
          positionData += `<div class="sub-comment"><strong>${position}:</strong> ${positionValue}${positionNote ? ` - ${positionNote}` : ''}</div>`;
        }
      });
      return positionData;
    };
    
    // Generate checklist rows with complete structure
    let checklistRows = '';
    
    // 5000KM Service
    checklistRows += `
      <tr class="section-header">
        <td colspan="4">5000KM</td>
      </tr>
      <tr>
        <td>ENGINE OIL CHANGED</td>
        <td>${selections['engineOilChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['engineOilChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['engineOilChanged'] || ''}</td>
      </tr>
      <tr>
        <td>ENGINE OIL FILTER CHANGED</td>
        <td>${selections['engineOilFilterChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['engineOilFilterChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['engineOilFilterChanged'] || ''}</td>
      </tr>
      <tr>
        <td>ENGINE FUEL FILTER CHANGED</td>
        <td>${selections['engineFuelFilterChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['engineFuelFilterChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['engineFuelFilterChanged'] || ''}</td>
      </tr>
      <tr>
        <td>AIR FILTER</td>
        <td>${selections['airFilter'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['airFilter'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['airFilter'] || ''}</td>
      </tr>
      <tr>
        <td>POWER STEERING FLUID CHECKED</td>
        <td>${selections['powerSteeringFluidChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['powerSteeringFluidChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['powerSteeringFluidChecked'] || ''}</td>
      </tr>
      <tr>
        <td>POWER STEERING FLUID CHANGED</td>
        <td>${selections['powerSteeringFluidChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['powerSteeringFluidChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['powerSteeringFluidChanged'] || ''}</td>
      </tr>
    `;
    
    // 30000KM /12 MONTHS
    checklistRows += `
      <tr class="section-header">
        <td colspan="4">30000KM /12 MONTHS</td>
      </tr>
      <tr>
        <td>RADIATOR COOLANT CHECKED</td>
        <td>${selections['radiatorCoolantChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['radiatorCoolantChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['radiatorCoolantChecked'] || ''}</td>
      </tr>
      <tr>
        <td>RADIATOR COOLANT CHANGED</td>
        <td>${selections['radiatorCoolantChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['radiatorCoolantChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['radiatorCoolantChanged'] || ''}</td>
      </tr>
    `;
    
    // 50000KM
    checklistRows += `
      <tr class="section-header">
        <td colspan="4">50000KM</td>
      </tr>
      <tr>
        <td>GEARBOX OIL CHANGE</td>
        <td>${selections['gearboxOilChange'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['gearboxOilChange'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['gearboxOilChange'] || ''}</td>
      </tr>
    `;
    
    // 10000KM
    checklistRows += `
      <tr class="section-header">
        <td colspan="4">10000KM</td>
      </tr>
      <tr>
        <td>TIMING BELT</td>
        <td>${selections['timingBelt'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['timingBelt'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['timingBelt'] || ''}</td>
      </tr>
      <tr>
        <td>IDLER BEARINGS CHECKED</td>
        <td>${selections['idlerBearingsChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['idlerBearingsChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['idlerBearingsChecked'] || ''}</td>
      </tr>
      <tr>
        <td>VISCOUS FAN/FAN BLADES CHECKED</td>
        <td>${selections['viscousFanFanBladesChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['viscousFanFanBladesChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['viscousFanFanBladesChecked'] || ''}</td>
      </tr>
    `;
    
    // 50000KM/18 MONTHS
    checklistRows += `
      <tr class="section-header">
        <td colspan="4">50000KM/18 MONTHS</td>
      </tr>
      <tr>
        <td>GEARBOX OIL LEVEL CHECK</td>
        <td>${selections['gearboxOilLevelCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['gearboxOilLevelCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['gearboxOilLevelCheck'] || ''}</td>
      </tr>
    `;
    
    // 50000/18 MONTHS
    checklistRows += `
      <tr class="section-header">
        <td colspan="4">50000/18 MONTHS</td>
      </tr>
      <tr>
        <td>TRANSFER BOX OIL LEVEL CHECK</td>
        <td>${selections['transferBoxOilLevelCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['transferBoxOilLevelCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['transferBoxOilLevelCheck'] || ''}</td>
      </tr>
      <tr>
        <td>DIFF OIL LEVEL CHECK</td>
        <td>${selections['diffOilLevelCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['diffOilLevelCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['diffOilLevelCheck'] || ''}</td>
      </tr>
      <tr>
        <td>DIFF OIL CHANGE</td>
        <td>${selections['diffOilChange'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['diffOilChange'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['diffOilChange'] || ''}</td>
      </tr>
    `;
    
    // 5 CLICKS
    checklistRows += `
      <tr class="section-header">
        <td colspan="4">5 CLICKS</td>
      </tr>
      <tr>
        <td>BRAKE PADS/SHOES CHECK</td>
        <td>${selections['brakePadsShoesCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['brakePadsShoesCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('brakePadsShoesCheck', ['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT', 'SPARE'])}
          ${notes['brakePadsShoesCheck'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['brakePadsShoesCheck']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>HANDBRAKE SHOES CHECK/ADJUSTMENT</td>
        <td>${selections['handbrakeShoesCheckAdjustment'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['handbrakeShoesCheckAdjustment'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['handbrakeShoesCheckAdjustment'] || ''}</td>
      </tr>
      <tr>
        <td>KING PIN PLAY CHECK</td>
        <td>${selections['kingPinPlayCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['kingPinPlayCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('kingPinPlayCheck', ['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT'])}
          ${notes['kingPinPlayCheck'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['kingPinPlayCheck']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>WHEEL BEARING CHECK</td>
        <td>${selections['wheelBearingCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['wheelBearingCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('wheelBearingCheck', ['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT', 'SPARE'])}
          ${notes['wheelBearingCheck'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['wheelBearingCheck']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>SHOCK ABSORBERS/DAMPERS CHECK</td>
        <td>${selections['shockAbsorbersDampersCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['shockAbsorbersDampersCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['shockAbsorbersDampersCheck'] || ''}</td>
      </tr>
      <tr>
        <td>SUSPENSION SPRINGS SPLAYED/CHECKED</td>
        <td>${selections['suspensionSpringsSplayedChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['suspensionSpringsSplayedChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['suspensionSpringsSplayedChecked'] || ''}</td>
      </tr>
      <tr>
        <td>HANGERS CHECKED</td>
        <td>${selections['hangersChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['hangersChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['hangersChecked'] || ''}</td>
      </tr>
      <tr>
        <td>SHACKLE PINS/BUSHES</td>
        <td>${selections['shacklePinsBushes'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['shacklePinsBushes'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['shacklePinsBushes'] || ''}</td>
      </tr>
      <tr>
        <td>TIE ROD END/BALL JOINT CHANGE/CHECK</td>
        <td>${selections['tieRodEndBallJointChangeCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['tieRodEndBallJointChangeCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['tieRodEndBallJointChangeCheck'] || ''}</td>
      </tr>
      <tr>
        <td>WHEEL NUTS SECURE</td>
        <td>${selections['wheelNutsSecure'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['wheelNutsSecure'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('wheelNutsSecure', ['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT', 'SPARE'])}
          ${notes['wheelNutsSecure'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['wheelNutsSecure']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>TYRE WEAR/PRESSURE</td>
        <td>${selections['tyreWearPressure'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['tyreWearPressure'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('tyreWearPressure', ['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT', 'SPARE'])}
          ${notes['tyreWearPressure'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['tyreWearPressure']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>HEAD/TAIL/SIGNAL/BRAKE LIGHTS/HAZARDS</td>
        <td>${selections['headTailSignalBrakeLightsHazards'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['headTailSignalBrakeLightsHazards'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('headTailSignalBrakeLightsHazards', ['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT', 'SPARE'])}
          ${notes['headTailSignalBrakeLightsHazards'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['headTailSignalBrakeLightsHazards']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>BEACON</td>
        <td>${selections['beacon'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['beacon'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['beacon'] || ''}</td>
      </tr>
      <tr>
        <td>HORN</td>
        <td>${selections['horn'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['horn'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['horn'] || ''}</td>
      </tr>
      <tr>
        <td>DOOR HANDLES</td>
        <td>${selections['doorHandles'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['doorHandles'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('doorHandles', ['LEFT FRONT', 'LEFT REAR', 'RIGHT FRONT', 'RIGHT REAR', 'REAR/TAIL GATE'])}
          ${notes['doorHandles'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['doorHandles']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>WINDOW WINDING MECHANISM</td>
        <td>${selections['windowWindingMechanism'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['windowWindingMechanism'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('windowWindingMechanism', ['LEFT FRONT', 'LEFT REAR', 'RIGHT FRONT', 'RIGHT REAR'])}
          ${notes['windowWindingMechanism'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['windowWindingMechanism']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>EXTERIOR (DENTS/PAINTWORK)</td>
        <td>${selections['exteriorDentsPaintwork'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['exteriorDentsPaintwork'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('exteriorDentsPaintwork', ['1ST PROPSHAFT', '2ND PROPSHAFT', 'AXLES', 'SHACKLES'])}
          ${notes['exteriorDentsPaintwork'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['exteriorDentsPaintwork']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>GREASED</td>
        <td>${selections['greased'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['greased'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('greased', ['1ST PROPSHAFT', '2ND PROPSHAFT', 'AXLES', 'SHACKLES'])}
          ${notes['greased'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['greased']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>BUGGY WHIP</td>
        <td>${selections['buggyWhip'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['buggyWhip'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['buggyWhip'] || ''}</td>
      </tr>
      <tr>
        <td>FIRE EXTINGUISHER CHECK/PRESSURE CHECK</td>
        <td>${selections['fireExtinguisherCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['fireExtinguisherCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['fireExtinguisherCheck'] || ''}</td>
      </tr>
      <tr>
        <td>WINDSCREEN WASHER FLUID CHECKED</td>
        <td>${selections['windscreenWasherFluidChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['windscreenWasherFluidChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['windscreenWasherFluidChecked'] || ''}</td>
      </tr>
      <tr>
        <td>BATTERY WATER CHECKED</td>
        <td>${selections['batteryWaterChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['batteryWaterChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['batteryWaterChecked'] || ''}</td>
      </tr>
      <tr>
        <td>BATTERY CLAMP SECURED</td>
        <td>${selections['batteryClampSecured'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['batteryClampSecured'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['batteryClampSecured'] || ''}</td>
      </tr>
      <tr>
        <td>BRAKE FLUID CHECKED</td>
        <td>${selections['brakeFluidChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['brakeFluidChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['brakeFluidChecked'] || ''}</td>
      </tr>
      <tr>
        <td>CLUTCH FLUID CHECKED</td>
        <td>${selections['clutchFluidChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['clutchFluidChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['clutchFluidChecked'] || ''}</td>
      </tr>
      <tr>
        <td>AIRCON FILTER CHECKED</td>
        <td>${selections['airconFilterChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['airconFilterChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['airconFilterChecked'] || ''}</td>
      </tr>
      <tr>
        <td>AIRCON FILTER CHANGED</td>
        <td>${selections['airconFilterChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['airconFilterChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['airconFilterChanged'] || ''}</td>
      </tr>
      <tr>
        <td>AIRCON/HEATER WORKING</td>
        <td>${selections['airconHeaterWorking'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['airconHeaterWorking'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['airconHeaterWorking'] || ''}</td>
      </tr>
      <tr>
        <td>ENGINE BAY LEAKS/LOOSE ITEMS CHECK</td>
        <td>${selections['engineBayLeaksLooseItemsCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['engineBayLeaksLooseItemsCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['engineBayLeaksLooseItemsCheck'] || ''}</td>
      </tr>
      <tr>
        <td>WINDSCREEN WIPERS CHECKED/CHANGED</td>
        <td>${selections['windscreenWipersCheckedChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['windscreenWipersCheckedChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['windscreenWipersCheckedChanged'] || ''}</td>
      </tr>
    `;

    return `
      <!DOCTYPE html>
      <html lang="en">
      <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Titan Drilling LV Service Sheet</title>
          <style>
              body {
                  font-family: Arial, sans-serif;
                  margin: 5px;
                  padding: 0;
                  font-size: 14px;
              }
              h1 {
                  text-align: center;
                  margin-bottom: 10px;
                  font-size: 1.5em;
              }
              .header-container {
                  display: flex;
                  flex-wrap: wrap;
                  justify-content: space-between;
                  align-items: flex-start;
                  margin-bottom: 15px;
              }
              .header-table {
                  border-collapse: collapse;
                  width: 100%;
                  max-width: 250px;
              }
              .header-table th, .header-table td {
                  border: 1px solid #ddd;
                  padding: 6px;
                  text-align: left;
                  font-size: 0.9em;
              }
              .header-table th {
                  background-color: #f2f2f2;
              }
              h2 {
                  text-align: center;
                  font-size: 1.2em;
                  margin: 15px 0;
              }
              table.checklist {
                  width: 100%;
                  border-collapse: collapse;
                  margin-bottom: 15px;
              }
              .checklist th, .checklist td {
                  border: 1px solid #ddd;
                  padding: 6px;
                  text-align: left;
                  font-size: 0.9em;
                  word-wrap: break-word;
              }
              .checklist th {
                  background-color: #f2f2f2;
              }
              .section-header td {
                  background-color: #e0e0e0;
                  font-weight: bold;
                  text-align: center;
              }
              .info-row td {
                  padding: 6px;
              }
              .sub-comment {
                  margin: 4px 0;
                  padding: 4px;
                  border-top: 1px solid #eee;
                  font-size: 0.85em;
              }
              .checklist th:nth-child(1), .checklist td:nth-child(1) {
                  width: 50%;
              }
              .checklist th:nth-child(2), .checklist td:nth-child(2) {
                  width: 15%;
                  text-align: center;
              }
              .checklist th:nth-child(3), .checklist td:nth-child(3) {
                  width: 15%;
                  text-align: center;
              }
              .checklist th:nth-child(4), .checklist td:nth-child(4) {
                  width: 20%;
              }
              .footer-table {
                  width: 100%;
                  border-collapse: collapse;
                  margin-top: 15px;
              }
              .footer-table td {
                  border: 1px solid #ddd;
                  padding: 8px;
                  font-size: 0.9em;
              }

              @media screen and (max-width: 600px) {
                  body {
                      margin: 2px;
                      font-size: 10px;
                  }
                  h1 {
                      font-size: 1em;
                      margin-bottom: 5px;
                  }
                  .header-container {
                      flex-direction: column;
                      align-items: center;
                  }
                  .header-table {
                      max-width: 100%;
                      margin-top: 5px;
                  }
                  .header-table th, .header-table td {
                      padding: 3px;
                      font-size: 0.7em;
                  }
                  h2 {
                      font-size: 0.9em;
                      margin: 10px 0;
                  }
                  table.checklist {
                      font-size: 0.8em;
                  }
                  .checklist th, .checklist td {
                      padding: 3px;
                      font-size: 0.7em;
                      word-wrap: break-word;
                  }
                  .checklist th:nth-child(1), .checklist td:nth-child(1) {
                      width: 50%;
                  }
                  .checklist th:nth-child(2), .checklist td:nth-child(2) {
                      width: 15%;
                  }
                  .checklist th:nth-child(3), .checklist td:nth-child(3) {
                      width: 15%;
                  }
                  .checklist th:nth-child(4), .checklist td:nth-child(4) {
                      width: 20%;
                  }
                  .info-row td {
                      padding: 4px;
                      font-size: 0.7em;
                  }
                  .sub-comment {
                      font-size: 0.6em;
                      padding: 2px;
                  }
                  .footer-table td {
                      padding: 4px;
                      font-size: 0.7em;
                  }
              }
          </style>
      </head>
      <body>
          <div class="header-container">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" style="width: 200px; height: auto;">
              <table class="header-table">
                  <tr>
                      <th colspan="2">Header Information</th>
                  </tr>
                  <tr>
                      <td><strong>Document Name:</strong></td>
                      <td>HSE-FOR-8.1.1 Workshop LV Check Sheet</td>
                  </tr>
                  <tr>
                      <td><strong>Document ID:</strong></td>
                      <td>HSE-125</td>
                  </tr>
                  <tr>
                      <td><strong>Version Date:</strong></td>
                      <td>19/01/2024</td>
                  </tr>
                  <tr>
                      <td><strong>Version:</strong></td>
                      <td>1</td>
                  </tr>
                  <tr>
                      <td><strong>Approved:</strong></td>
                      <td>HSM</td>
                  </tr>
              </table>
          </div>

          <table class="checklist">
              <tr class="info-row">
                  <td><strong>DATE:</strong> ${step3.date || ''}</td>
                  <td><strong>FLEET #:</strong> ${step3.fleetNumber || ''}</td>
                  <td><strong>MILEAGE:</strong> ${step3.mileage || ''}</td>
                  <td><strong>SERVICED BY:</strong> ${step3.servicedBy || ''}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Country:</strong> ${step1.country || ''}</td>
                  <td><strong>Project:</strong> ${step1.project || ''}</td>
                  <td><strong>Equipment:</strong> ${step1.selectedEquipment ? (step1.selectedEquipment.equipmentName || step1.selectedEquipment.equipmentNumber || 'N/A') : ''}</td>
                  <td><strong>Request #:</strong> ${step1.requestNumber || ''}</td>
              </tr>
          </table>

          <h2>LV SERVICE SHEET</h2>

          <table class="checklist">
              <tr>
                  <th>Task</th>
                  <th>Yes</th>
                  <th>No</th>
                  <th>Comments</th>
              </tr>
              ${checklistRows || '<tr><td colspan="4" style="text-align: center;">No checklist items available</td></tr>'}
          </table>

          <table class="footer-table">
              <tr class="info-row">
                  <td><strong>COMMENTS:</strong> ${step3.comments || ''}</td>
                  <td><strong>NEXT SERVICE DUE:</strong> ${step3.nextServiceDueKm || ''}</td>
                  <td><strong>SIGNATURE:</strong> ${step3.signature && step3.signature.length > 0 ? `
                      <div style="text-align: center;">
                          <img src="${step3.signature}" alt="Technician Signature" style="max-width: 80px; max-height: 40px; border: 1px solid #015185; border-radius: 4px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);" />
                          <br><small style="color: #4CAF50; font-weight: bold;">✓ Signed</small>
                      </div>
                  ` : `
                      <span style="color: #FF6B6B; font-weight: bold;">⚠ Not Signed</span>
                  `}</td>
              </tr>
          </table>
      </body>
      </html>
    `;
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <Text style={styles.pageTitle}>Workshop LV Check Sheet - Review & Submit</Text>

        <SyncModal 
          visible={syncModalVisible} 
          onClose={() => setSyncModalVisible(false)} 
          message="Generating Report"
        />

        {/* Form Section */}
        <View style={styles.section}>
          <View style={styles.sectionHeader}>
            <Text style={styles.sectionTitle}>FORM DETAILS</Text>
          </View>
          
          <View style={styles.formFields}>
            {/* DATE Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>DATE:</Text>
              <TouchableOpacity style={styles.pickerButton} onPress={showDatePickerModal}>
                <Text style={styles.pickerButtonText}>
                  {formData.date || selectedDate.toLocaleDateString()}
                </Text>
                <MaterialIcons name="calendar-today" size={20} color="#015185" />
              </TouchableOpacity>
            </View>

            {/* SERVICED BY Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>SERVICED BY:</Text>
              <TextInput
                style={styles.input}
                value={formData.servicedBy}
                onChangeText={(value) => handleInputChange('servicedBy', value)}
                placeholder="Enter service technician name"
                placeholderTextColor="#999"
              />
            </View>

            {/* FLEET NUMBER Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>FLEET NUMBER:</Text>
              <TextInput
                style={styles.input}
                value={formData.fleetNumber}
                onChangeText={(value) => handleInputChange('fleetNumber', value)}
                placeholder="Enter fleet number"
                placeholderTextColor="#999"
              />
            </View>

            {/* MILEAGE Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>MILEAGE:</Text>
              <TextInput
                style={styles.input}
                value={formData.mileage}
                onChangeText={(value) => handleInputChange('mileage', value)}
                placeholder="Enter current mileage"
                placeholderTextColor="#999"
                keyboardType="numeric"
              />
            </View>

            {/* COMMENTS Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>COMMENTS:</Text>
              <TextInput
                style={styles.input}
                value={formData.comments}
                onChangeText={(value) => handleInputChange('comments', value)}
                placeholder="Enter any additional comments"
                placeholderTextColor="#999"
                multiline
                numberOfLines={4}
                textAlignVertical="top"
              />
            </View>

            {/* NEXT SERVICE DUE KM Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>NEXT SERVICE DUE (KM):</Text>
              <TextInput
                style={styles.input}
                value={formData.nextServiceDueKm}
                onChangeText={(value) => handleInputChange('nextServiceDueKm', value)}
                placeholder="Enter next service due mileage"
                placeholderTextColor="#999"
                keyboardType="numeric"
              />
            </View>

            {/* SIGNATURE Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>SIGNATURE:</Text>
              <View style={styles.signatureBoxImage}>
                {isValidSignatureUri(signature) ? (
                  <TouchableOpacity 
                    style={styles.plusIconCircleContainer} 
                    onPress={handleSignaturePress}
                    key={signatureRefreshKey}
                  >
                    <Image
                      source={{ uri: signature }}
                      style={styles.signatureImage}
                      resizeMode="contain"
                      onError={handleImageError}
                    />
                  </TouchableOpacity>
                ) : (
                  <TouchableOpacity 
                    style={styles.plusIconCircleContainer} 
                    onPress={handleSignaturePress}
                    key={signatureRefreshKey}
                  >
                    <View style={styles.plusCircle}>
                      <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                    </View>
                  </TouchableOpacity>
                )}
              </View>
            </View>
          </View>
        </View>
      </ScrollView>

      {/* Date Picker Modal */}
      {showDatePicker && (
        <DateTimePicker
          value={selectedDate}
          mode="date"
          display={Platform.OS === 'ios' ? 'spinner' : 'default'}
          onChange={handleDateChange}
        />
      )}

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalTitle}>Success</Text>
            <Text style={styles.successModalText}>
              HSE-125 Service Report Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.navigate('HSEInspectionHome');
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPDF}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={[styles.nextButton, syncModalVisible && styles.disabledButton]} onPress={handleSubmit} disabled={syncModalVisible}>
          <Text style={[styles.nextButtonText, syncModalVisible && styles.disabledButtonText]}>
            {syncModalVisible ? 'Generating PDF...' : 'Generate Report'}
          </Text>
          <MaterialIcons style={styles.bottomicon1} name={syncModalVisible ? "hourglass-empty" : "check"} size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      <SyncModal 
        visible={syncModalVisible} 
        onClose={() => setSyncModalVisible(false)} 
        message="Generating Report"
      />
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  pageTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    paddingVertical: 20,
    backgroundColor: '#F2F2F2',
  },
  section: {
    marginBottom: 30,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    overflow: 'hidden',
    margin: 20,
  },
  sectionHeader: {
    backgroundColor: '#f0f0f0',
    padding: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
  },
  formFields: {
    padding: 20,
  },
  fieldContainer: {
    marginBottom: 20,
  },
  fieldLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 8,
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  pickerButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    backgroundColor: '#fff',
    minHeight: 40,
    width: '100%',
  },
  pickerButtonText: {
    flex: 1,
    fontSize: 16,
    color: '#333',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  signatureBoxImage: {
    height: height / 5,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: width * 0.012,
    justifyContent: 'center',
    alignItems: 'center',
  },
  plusIconCircleContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    width: '100%',
    height: '100%',
  },
  plusCircle: {
    width: '100%',
    height: '100%',
    justifyContent: 'center',
    alignItems: 'center',
  },
  signatureImage: {
    width: '100%',
    height: '100%',
    borderRadius: 2,
    backgroundColor: '#f5f5f5',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 30,
    width: width * 0.85,
    alignItems: 'center',
  },
  successIconContainer: {
    marginBottom: 20,
  },
  successModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  successModalSubtitle: {
    fontSize: 16,
    color: '#666',
    marginBottom: 25,
    textAlign: 'center',
  },
  successModalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  successModalButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 8,
    flex: 1,
    marginHorizontal: 5,
  },
  viewPdfButton: {
    backgroundColor: '#28a745',
  },
  successModalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 8,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(255, 255, 255, 0.9)',
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    bottom: 0,
    zIndex: 10,
  },
  loadingText: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginTop: 10,
  },
  loadingSubtext: {
    fontSize: 14,
    color: '#666',
    marginTop: 5,
  },
  syncModalContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  syncModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '70%',
    maxWidth: 300,
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
    lineHeight: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  disabledButton: {
    opacity: 0.5,
  },
  disabledButtonText: {
    opacity: 0.5,
  },
});

export default HSE_125Step3;