import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  TextInput,
  Modal,
  Dimensions,
  Image,
  Animated,
  Alert,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import DateTimePicker from '@react-native-community/datetimepicker';
import Toast from 'react-native-toast-message';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';
import { useNavigation } from '@react-navigation/native';

const { width, height } = Dimensions.get('window');

const SyncModal = ({ visible, onClose, message = 'Loading Data' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_122Step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [formData, setFormData] = useState({
    fleetNumber: '',
    engineHours: '',
    dateServiceStarted: '',
    dateServiceEnded: '',
  });

  const [selectedPersonnel, setSelectedPersonnel] = useState('');
  const [personnelName, setPersonnelName] = useState('');
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [datePickerField, setDatePickerField] = useState('');
  const [selectedDate, setSelectedDate] = useState(new Date());
  const [signatureModal, setSignatureModal] = useState({
    visible: false,
    personnelType: '',
  });

  // New state variables for PDF functionality
  const [step1Data, setStep1Data] = useState({});
  const [step2Data, setStep2Data] = useState({});
  const [isLoading, setIsLoading] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [pdfDownloadUrl, setPdfDownloadUrl] = useState(null);
  const [generatedPdfPath, setGeneratedPdfPath] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);

  const personnelOptions = [
    'MECHANIC',
    'HYDRAULIC TECH',
    'BOILER MAKER',
    'AUTO ELECTRICIAN'
  ];

  useEffect(() => {
    loadSavedData();
  }, []);

  // Monitor network connectivity
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    NetInfo.fetch().then((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    return () => unsubscribe();
  }, []);

  // Load data from step1 and step2
  useEffect(() => {
    loadStepData();
  }, []);

  const loadStepData = async () => {
    try {
      // Load step1 data
      const savedStep1 = await AsyncStorage.getItem('hseStep1');
      if (savedStep1) {
        const parsedStep1 = JSON.parse(savedStep1);
        setStep1Data(parsedStep1);
      }

      // Load step2 data
      const savedStep2 = await AsyncStorage.getItem('hse122Step2');
      if (savedStep2) {
        const parsedStep2 = JSON.parse(savedStep2);
        setStep2Data(parsedStep2);
      }
    } catch (error) {
      console.error('Error loading step data:', error);
    }
  };

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse122Step3');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        setFormData(parsedData.formData || {});
        setSelectedPersonnel(parsedData.selectedPersonnel || '');
        setPersonnelName(parsedData.personnelName || '');
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async () => {
    try {
      const dataToSave = {
        formData,
        selectedPersonnel,
        personnelName,
        timestamp: new Date().toISOString()
      };
      await AsyncStorage.setItem('hse122Step3', JSON.stringify(dataToSave));
    } catch (error) {
      console.error('Error saving data:', error);
    }
  };

  const handleInputChange = (field, value) => {
    const updatedData = { ...formData, [field]: value };
    setFormData(updatedData);
    saveData();
  };

  const handlePersonnelSelection = (personnel) => {
    setSelectedPersonnel(personnel);
    setPersonnelName('');
    saveData();
  };

  const handlePersonnelNameChange = (value) => {
    setPersonnelName(value);
    saveData();
  };

  const showDatePickerModal = (field) => {
    setDatePickerField(field);
    setShowDatePicker(true);
  };

  const handleDateChange = (event, selectedDate) => {
    setShowDatePicker(false);
    if (selectedDate) {
      setSelectedDate(selectedDate);
      const dateString = selectedDate.toLocaleDateString('en-GB');
      handleInputChange(datePickerField, dateString);
    }
  };

  const formatDate = (date) => {
    if (!date) return 'Select Date';
    if (typeof date === 'string') {
      try {
        date = new Date(date);
      } catch (error) {
        return 'Select Date';
      }
    }
    if (!(date instanceof Date) || isNaN(date.getTime())) {
      return 'Select Date';
    }
    return date.toLocaleDateString('en-GB');
  };

  const openSignatureModal = (personnelType) => {
    setSignatureModal({
      visible: true,
      personnelType,
    });
  };

  const closeSignatureModal = () => {
    setSignatureModal({
      visible: false,
      personnelType: '',
    });
  };

  const handleNext = async () => {
    // Validate required fields
    const requiredFields = ['fleetNumber', 'engineHours', 'dateServiceStarted', 'dateServiceEnded'];
    const missingFields = requiredFields.filter(field => !formData[field]);
    
    if (missingFields.length > 0) {
      Alert.alert('Error', 'Please fill in all required fields');
      return;
    }
    
    // Validate personnel selection
    if (!selectedPersonnel || !personnelName) {
      Alert.alert('Error', 'Please select personnel type and enter name');
      return;
    }

    try {
      // Save data before generating PDF
      await saveData();
      // Generate PDF immediately after successful validation
      await generatePdf();
    } catch (error) {
      console.error('Error submitting form:', error);
      Alert.alert('Error', 'Failed to submit form. Please try again.');
    }
  };

  const generatePdf = async () => {
    try {
      setSyncModalVisible(true);
      
      // Generate the HTML content for the PDF
      const htmlContent = generatePDFHTML();
      
      const options = {
        html: htmlContent,
        fileName: `HSE_122_${formData.fleetNumber || 'Report'}_${new Date().toISOString().split('T')[0]}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      
      if (file.filePath) {
        setGeneratedPdfPath(file.filePath);
        Toast.show({
          type: 'success',
          text1: 'PDF Generated',
          text2: `Saved to: ${file.filePath}`,
          position: 'bottom'
        });
        
        // Show success modal instead of alert
        setShowSuccessModal(true);
      }
    } catch (error) {
      console.error('Error generating PDF:', error);
      Toast.show({
        type: 'error',
        text1: 'PDF Generation Failed',
        text2: error.message || 'Please try again',
        position: 'bottom'
      });
    } finally {
      setSyncModalVisible(false);
    }
  };

  const handleBackToHome = () => {
    setShowSuccessModal(false);
    navigation.navigate('HSEInspectionHome');
  };

  const handleViewPDF = () => {
    setShowSuccessModal(false);
    if (generatedPdfPath) {
      // For local file paths, we need to convert to file:// URL format
      const pdfUrl = generatedPdfPath.startsWith('file://') ? generatedPdfPath : `file://${generatedPdfPath}`;
      navigation.navigate('HSEViewPDF', {
        pdfUrl: pdfUrl,
        onGoBack: () => navigation.navigate('HSEInspectionHome')
      });
    } else {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'PDF not available for viewing.',
        position: 'bottom'
      });
    }
  };

  const generatePDFHTML = () => {
    // Generate checklist items with their selections and notes
    let checklistRows = '';
    if (step2Data.selections) {
      const sections = [
        {
          key: 'engine',
          name: 'ENGINE',
          items: [
            { key: 'engineOil', name: 'Engine oil (CHANGED/LEVEL)' },
            { key: 'coolant', name: 'Coolant (LEAKS/LEVEL)' },
            { key: 'airFilter', name: 'Air filter / intake pipes (CLEANED HOUSING/CHANGED)' },
            { key: 'fuelFilter', name: 'Fuel filter (CHANGED)' },
            { key: 'fuelSystem', name: 'Fuel system (LEAKS)' },
            { key: 'vBelts', name: 'V belts (CONDITION/TENSION)' },
            { key: 'radiator', name: 'Radiator (CLEANED/ PRESSURE TESTED)' },
            { key: 'radiatorCap', name: 'Radiator cap (INSPECTED)' },
            { key: 'engineMountings', name: 'Engine mountings (WORN/SECURED)' },
          ]
        },
        {
          key: 'every2000hrs',
          name: 'EVERY 2000 HRS',
          items: [
            { key: 'valveClearance', name: 'Valve clearance (ADJUSTED/CHECKED)' },
          ]
        },
        {
          key: 'electrical',
          name: 'ELECTRICAL',
          items: [
            { key: 'batteries', name: 'Batteries (POLE CONDITION/ JELLED)' },
            { key: 'dateEngraved', name: 'Date engraved' },
            { key: 'voltageCranking', name: 'Voltage at cranking (VOLTS)' },
            { key: 'alternator', name: 'Alternator (MOUNTING/CHECKED)' },
            { key: 'starter', name: 'Starter (MOUNTING/TESTED)' },
            { key: 'lights', name: 'Lights (WORKING/WIRING)' },
            { key: 'chargingSystem', name: 'Charging system (CHARGING/WIRES/FUSES)' },
            { key: 'rpmGauge', name: 'RPM/Volt/Temp/Oil gauge (WORKING/CALIBRATED)' },
            { key: 'eStops', name: 'E stops (CHECKED / TESTED)' },
            { key: 'limitSwitches', name: 'Limit switches (FUNCTION TEST/ADJUST)' },
            { key: 'engineProtection', name: 'Engine protection cut out' },
            { key: 'horn', name: 'Horn (CHECKED / TESTED)' },
          ]
        },
        {
          key: 'mast',
          name: 'MAST',
          items: [
            { key: 'pins', name: 'Pins (SECURE)' },
            { key: 'slides', name: 'Slides (NO FREE PLAY)' },
            { key: 'inspectionPlate', name: 'Inspection plate (DATED)' },
            { key: 'cracks', name: 'Cracks' },
            { key: 'sheaveWheels', name: 'Sheave wheels (INSPECTED/CHANGED)' },
            { key: 'wireline', name: 'Wireline (INSPECTED/CHANGED)' },
            { key: 'winchCable', name: 'Winch cable (INSPECTED/CHANGED)' },
            { key: 'rotationGuard', name: 'Rotation guard switch (FUNCTION TEST/ADJUST)' },
            { key: 'rodKicker', name: 'Rod kicker (FREE/SECURE)' },
            { key: 'fineFeedRam', name: 'Fine feed ram (SECURE/LEAKS)' },
          ]
        },
        {
          key: 'drillHead',
          name: 'DRILL HEAD',
          items: [
            { key: 'chainTension', name: 'Chain tension (IF APPLICABLE)' },
            { key: 'gearboxBreather', name: 'Check gearbox breather (CLEAN)' },
            { key: 'headLockBolts', name: 'Head lock bolts (INSPECT)' },
            { key: 'slideBushes', name: 'Slide bushes (INSPECT/CHANGE)' },
            { key: 'strainer', name: 'Strainer (CLEAN)' },
          ]
        },
        {
          key: 'hydraulics',
          name: 'HYDRAULICS',
          items: [
            { key: 'hydraulicPressure', name: 'Check hydraulic pressure (TEST PRESSURES)' },
            { key: 'hydraulicHoses', name: 'Hydraulic hoses (LEAKS/CONDITION)' },
            { key: 'hydraulicOil', name: 'Hydraulic oil (LEVEL/QUALITY)' },
            { key: 'hydraulicTankBreather', name: 'Hydraulic tank breather (CLEAN)' },
            { key: 'hydraulicOilFilter', name: 'Hydraulic oil filter (CLEAN/CHECK)' },
            { key: 'rotationMotor', name: 'Rotation motor (LEAKS/CONDITION)' },
            { key: 'wirelineDrum', name: 'Wireline drum/motor (SECURE/LEAKS)' },
            { key: 'winchDrum', name: 'Winch drum/motor (SECURE/LEAKS)' },
            { key: 'hydraulicOilCoolers', name: 'Hydraulic oil coolers (CLEAN/LEAKS)' },
          ]
        },
        {
          key: 'beanPump',
          name: 'BEAN PUMP',
          items: [
            { key: 'pistonsLiners', name: 'Pistons/liners (INSPECT/HANGE)' },
            { key: 'ballsCradles', name: 'Balls and cradles (CONDITION)' },
            { key: 'seals', name: 'Seals' },
            { key: 'oil', name: 'Oil (LEVEL/CHANGE)' },
          ]
        },
        {
          key: 'tyresAndHubs',
          name: 'TYRES AND HUBS',
          items: [
            { key: 'tyre', name: 'Tyre (CONDITION)' },
            { key: 'hubs', name: 'Hubs (BEARING FREEPLAY)' },
            { key: 'bolts', name: 'Bolts (SECURE)' },
          ]
        }
      ];

      sections.forEach(section => {
        checklistRows += `
          <tr class="section-header">
            <td colspan="4">${section.name}</td>
          </tr>
        `;
        
        section.items.forEach(item => {
          const selection = step2Data.selections[item.key];
          const note = step2Data.notes[item.key];
          let yesChecked = '';
          let noChecked = '';
          let comments = note || '';

          // Handle voltage input for voltage at cranking
          if (item.key === 'voltageCranking') {
            const voltageValue = step2Data.voltageInputs?.[item.key];
            if (voltageValue) {
              comments = `Voltage: ${voltageValue}V`;
            }
          }

          // Handle charging system inputs
          if (item.key === 'chargingSystem') {
            const runningVoltage = step2Data.chargingInputs?.[`${item.key}_running`];
            const notRunningVoltage = step2Data.chargingInputs?.[`${item.key}_notRunning`];
            if (runningVoltage || notRunningVoltage) {
              const voltageValues = [];
              if (runningVoltage) voltageValues.push(`V RUNNING: ${runningVoltage}V`);
              if (notRunningVoltage) voltageValues.push(`V NOT RUNNING: ${notRunningVoltage}V`);
              comments = voltageValues.join(' | ');
            }
          }
          
          // Standard Yes/No items
          yesChecked = selection === 'YES/OK' ? '✓' : '';
          noChecked = selection === 'NO/DEF' ? '✓' : '';
          
          checklistRows += `
            <tr>
              <td>${item.name}</td>
              <td style="text-align: center;">${yesChecked}</td>
              <td style="text-align: center;">${noChecked}</td>
              <td>${comments}</td>
            </tr>
          `;
        });
      });
    }

    return `
      <!DOCTYPE html>
      <html lang="en">
      <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>HSE-FOR-8-1.1 Workshop RC Rig Check Sheet</title>
          <style>
              body {
                  font-family: Arial, sans-serif;
                  margin: 5px;
                  padding: 0;
                  font-size: 14px;
              }
              h1 {
                  text-align: center;
                  margin-bottom: 10px;
                  font-size: 1.5em;
              }
              .header-container {
                  display: flex;
                  flex-wrap: wrap;
                  justify-content: space-between;
                  align-items: flex-start;
                  margin-bottom: 15px;
              }
              .header-table {
                  border-collapse: collapse;
                  width: 100%;
                  max-width: 250px;
              }
              .header-table th, .header-table td {
                  border: 1px solid #ddd;
                  padding: 6px;
                  text-align: left;
                  font-size: 0.9em;
              }
              .header-table th {
                  background-color: #f2f2f2;
              }
              h2 {
                  text-align: center;
                  font-size: 1.2em;
                  margin: 15px 0;
              }
              table.checklist {
                  width: 100%;
                  border-collapse: collapse;
                  margin-bottom: 15px;
              }
              .checklist th, .checklist td {
                  border: 1px solid #ddd;
                  padding: 6px;
                  text-align: left;
                  font-size: 0.9em;
                  word-wrap: break-word;
              }
              .checklist th {
                  background-color: #f2f2f2;
              }
              .section-header td {
                  background-color: #e0e0e0;
                  font-weight: bold;
                  text-align: center;
              }
              .info-row td {
                  padding: 6px;
              }
              .checklist th:nth-child(1), .checklist td:nth-child(1) {
                  width: 50%;
              }
              .checklist th:nth-child(2), .checklist td:nth-child(2) {
                  width: 15%;
                  text-align: center;
              }
              .checklist th:nth-child(3), .checklist td:nth-child(3) {
                  width: 15%;
                  text-align: center;
              }
              .checklist th:nth-child(4), .checklist td:nth-child(4) {
                  width: 20%;
              }
              .footer-table {
                  width: 100%;
                  border-collapse: collapse;
                  margin-top: 15px;
              }
              .footer-table td {
                  border: 1px solid #ddd;
                  padding: 8px;
                  font-size: 0.9em;
              }
              .project-info-table {
                  margin-bottom: 20px;
              }
              .project-info-table .section-header td {
                  background-color: #015185;
                  color: white;
                  font-weight: bold;
                  text-align: center;
                  font-size: 16px;
              }
              .project-info-table .info-row td {
                  padding: 8px;
                  border: 1px solid #ddd;
                  vertical-align: top;
              }
              .project-info-table .info-row td:first-child {
                  background-color: #f8f9fa;
                  font-weight: 600;
                  color: #015185;
                  width: 25%;
              }
              .project-info-table .info-row td:nth-child(2),
              .project-info-table .info-row td:nth-child(4) {
                  width: 25%;
              }
              .project-info-table .info-row td:nth-child(3) {
                  width: 25%;
              }
              .project-info-table .info-row td[colspan="3"] {
                  width: 75%;
              }

              @media screen and (max-width: 600px) {
                  body {
                      margin: 2px;
                      font-size: 10px;
                  }
                  h1 {
                      font-size: 1em;
                      margin-bottom: 5px;
                  }
                  .header-container {
                      flex-direction: column;
                      align-items: center;
                  }
                  .header-table {
                      max-width: 100%;
                      margin-top: 5px;
                  }
                  .header-table th, .header-table td {
                      padding: 3px;
                      font-size: 0.7em;
                  }
                  h2 {
                      font-size: 0.9em;
                      margin: 10px 0;
                  }
                  table.checklist {
                      font-size: 0.8em;
                  }
                  .checklist th, .checklist td {
                      padding: 3px;
                      font-size: 0.7em;
                      word-wrap: break-word;
                  }
                  .info-row td {
                      padding: 4px;
                      font-size: 0.7em;
                  }
                  .footer-table td {
                      padding: 4px;
                      font-size: 0.7em;
                  }
              }
          </style>
      </head>
      <body>
          <div class="header-container">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" style="width: 200px; height: auto;">
              <table class="header-table">
                  <tr>
                      <th colspan="2">Header Information</th>
                  </tr>
                  <tr>
                      <td><strong>Document ID:</strong></td>
                      <td>HSE-122</td>
                  </tr>
                  <tr>
                      <td><strong>Version Date:</strong></td>
                      <td>19/01/2024</td>
                  </tr>
                  <tr>
                      <td><strong>Version:</strong></td>
                      <td>HSM</td>
                  </tr>
                  <tr>
                      <td><strong>Approved By:</strong></td>
                      <td></td>
                  </tr>
              </table>
          </div>

          <table class="checklist">
              <tr class="info-row">
                  <td><strong>Date Service Started:</strong> ${formData.dateServiceStarted || ''}</td>
                  <td><strong>Fleet Number:</strong> ${formData.fleetNumber || ''}</td>
                  <td><strong>Engine Hours:</strong> ${formData.engineHours || ''}</td>
                  <td><strong>Serviced By:</strong></td>
              </tr>
              <tr class="info-row">
                  <td><strong>Date Service Ended:</strong> ${formData.dateServiceEnded || ''}</td>
                  <td><strong>Work Carried Out:</strong></td>
                  <td></td>
                  <td></td>
              </tr>
              <tr class="info-row">
                  <td><strong>Personnel Type:</strong> ${selectedPersonnel || ''}</td>
                  <td><strong>Name:</strong> ${personnelName || ''}</td>
                  <td><strong>Signature:</strong></td>
                  <td></td>
              </tr>
          </table>

          <!-- Project Information Table -->
          <h2>Project Information</h2>
          <table class="checklist project-info-table">
              <tr class="section-header">
                  <td colspan="4">Project Details</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Request Number:</strong></td>
                  <td colspan="3">${step1Data.requestNumber || 'N/A'}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Country:</strong></td>
                  <td>${step1Data.country || 'N/A'}</td>
                  <td><strong>Project:</strong></td>
                  <td>${step1Data.project || 'N/A'}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Equipment Name:</strong></td>
                  <td colspan="3">${step1Data.selectedEquipment?.equipmentName || step1Data.selectedEquipment?.equipmentNumber || 'N/A'}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Equipment Number:</strong></td>
                  <td>${step1Data.selectedEquipment?.equipmentNumber || step1Data.selectedEquipment?.entryNumber || 'N/A'}</td>
                  <td><strong>Equipment Category:</strong></td>
                  <td>${step1Data.selectedEquipment?.mainCategory || 'N/A'}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Equipment Model:</strong></td>
                  <td>${step1Data.selectedEquipment?.model || 'N/A'}</td>
                  <td><strong>Equipment Status:</strong></td>
                  <td>${step1Data.selectedEquipment?.status || 'N/A'}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>GPS Coordinates:</strong></td>
                  <td colspan="3">${step1Data.coordinates || 'N/A'}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>GPS Address:</strong></td>
                  <td colspan="3">${step1Data.gpsAddress || 'N/A'}</td>
              </tr>
          </table>

          <h2>Checklist</h2>

          <table class="checklist">
              <tr>
                  <th>Task</th>
                  <th>Yes</th>
                  <th>No</th>
                  <th>Comments</th>
              </tr>
              ${checklistRows}
          </table>

          <table class="footer-table">
              <tr class="info-row">
                  <td><strong>Comments:</strong></td>
                  <td><strong>Next Service Due (Hours):</strong></td>
                  <td><strong>Signature:</strong></td>
              </tr>
          </table>
      </body>
      </html>
    `;
  };

  const handleBack = () => {
    setActiveStep(2);
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.scrollView} contentContainerStyle={styles.scrollContent}>
        
        <Text style={styles.pageTitle}>RC Drill Rig Checklist - Review & Submit</Text>

       

        {/* Fleet Information Section */}
        <View style={styles.section}>
          <View style={styles.sectionHeader}>
            <Text style={styles.sectionTitle}>FLEET INFORMATION</Text>
          </View>
          
          <View style={styles.formFields}>
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>FLEET NUMBER:</Text>
              <TextInput
                style={styles.input}
                value={formData.fleetNumber}
                onChangeText={(value) => handleInputChange('fleetNumber', value)}
                placeholder="Enter fleet number"
                placeholderTextColor="#999"
              />
            </View>

            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>ENGINE HOURS:</Text>
              <TextInput
                style={styles.input}
                value={formData.engineHours}
                onChangeText={(value) => handleInputChange('engineHours', value)}
                placeholder="Enter engine hours"
                placeholderTextColor="#999"
                keyboardType="numeric"
              />
            </View>

            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>DATE SERVICE STARTED:</Text>
              <TouchableOpacity
                style={styles.dateInput}
                onPress={() => showDatePickerModal('dateServiceStarted')}
              >
                <Text style={[
                  styles.dateInputText,
                  !formData.dateServiceStarted && styles.datePlaceholderText
                ]}>
                  {formatDate(formData.dateServiceStarted)}
                </Text>
                <MaterialIcons name="calendar-today" size={20} color="#015185" />
              </TouchableOpacity>
            </View>

            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>DATE SERVICE ENDED:</Text>
              <TouchableOpacity
                style={styles.dateInput}
                onPress={() => showDatePickerModal('dateServiceEnded')}
              >
                <Text style={[
                  styles.dateInputText,
                  !formData.dateServiceEnded && styles.datePlaceholderText
                ]}>
                  {formatDate(formData.dateServiceEnded)}
                </Text>
                <MaterialIcons name="calendar-today" size={20} color="#015185" />
              </TouchableOpacity>
            </View>
          </View>
        </View>

        {/* Signature Information Section */}
        <View style={styles.section}>
          <View style={styles.sectionHeader}>
            <Text style={styles.sectionTitle}>SIGNATURE INFORMATION</Text>
          </View>
          
          <View style={styles.formFields}>
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>SELECT PERSONNEL:</Text>
              <View style={styles.selectorContainer}>
                {personnelOptions.map((option) => (
                  <TouchableOpacity
                    key={option}
                    style={[
                      styles.selectorOption,
                      selectedPersonnel === option && styles.selectorOptionSelected
                    ]}
                    onPress={() => handlePersonnelSelection(option)}
                  >
                    <Text style={[
                      styles.selectorOptionText,
                      selectedPersonnel === option && styles.selectorOptionTextSelected
                    ]}>
                      {option}
                    </Text>
                  </TouchableOpacity>
                ))}
              </View>
            </View>

            {selectedPersonnel && (
              <>
                <View style={styles.fieldContainer}>
                  <Text style={styles.fieldLabel}>NAME:</Text>
                  <TextInput
                    style={styles.input}
                    value={personnelName}
                    onChangeText={handlePersonnelNameChange}
                    placeholder={`Enter ${selectedPersonnel} name`}
                    placeholderTextColor="#999"
                  />
                </View>

                <View style={styles.fieldContainer}>
                  <Text style={styles.fieldLabel}>SIGNATURE:</Text>
                  <TouchableOpacity
                    style={styles.signatureButton}
                    onPress={() => openSignatureModal(selectedPersonnel)}
                  >
                    <MaterialIcons name="edit" size={20} color="#015185" />
                    <Text style={styles.signatureButtonText}>Add Signature</Text>
                  </TouchableOpacity>
                </View>
              </>
            )}
          </View>
        </View>

        <SyncModal 
          visible={syncModalVisible} 
          onClose={() => setSyncModalVisible(false)} 
          message="Generating Report"
        />
      </ScrollView>

      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={[styles.nextButton, syncModalVisible && styles.disabledButton]} onPress={handleNext} disabled={syncModalVisible}>
          <Text style={[styles.nextButtonText, syncModalVisible && styles.disabledButtonText]}>
            {syncModalVisible ? 'Generating PDF...' : 'Generate Report'}
          </Text>
          <MaterialIcons style={styles.bottomicon1} name={syncModalVisible ? "hourglass-empty" : "chevron-right"} size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* Date Picker Modal */}
      {showDatePicker && (
        <DateTimePicker
          value={selectedDate}
          mode="date"
          display="default"
          onChange={handleDateChange}
        />
      )}

      {/* Signature Modal */}
      <Modal
        visible={signatureModal.visible}
        transparent={true}
        animationType="slide"
        onRequestClose={closeSignatureModal}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Add Signature for {signatureModal.personnelType}</Text>
            <View style={styles.signatureArea}>
              <Text style={styles.signaturePlaceholder}>Signature Pad Area</Text>
              <Text style={styles.signatureNote}>Tap to add signature</Text>
            </View>
            <View style={styles.modalButtons}>
              <TouchableOpacity
                style={styles.modalButton}
                onPress={closeSignatureModal}
              >
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.modalButton, styles.modalButtonPrimary]} onPress={closeSignatureModal}>
                <Text style={[styles.modalButtonText, styles.modalButtonTextPrimary]}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalTitle}>Success</Text>
            <Text style={styles.successModalText}>
              HSE-122 Workshop RC Rig Check Sheet Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.navigate('HSEInspectionHome');
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPDF}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <SyncModal 
        visible={syncModalVisible} 
        onClose={() => setSyncModalVisible(false)} 
        message="Generating Report"
      />
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  scrollView: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  header: {
    backgroundColor: '#f8f9fa',
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  headerTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
  },
  section: {
    marginBottom: 30,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    overflow: 'hidden',
    margin: 20,
  },
  sectionHeader: {
    backgroundColor: '#f0f0f0',
    padding: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
  },
  formFields: {
    padding: 20,
  },
  fieldContainer: {
    marginBottom: 20,
  },
  fieldLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 8,
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    minHeight: 40,
    width: '100%',
  },
  dateInput: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    backgroundColor: '#fff',
    minHeight: 40,
    width: '100%',
  },
  dateInputText: {
    fontSize: 16,
    color: '#333',
    flex: 1,
    marginRight: 10,
  },
  datePlaceholderText: {
    color: '#999',
  },
  selectorContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: 10,
  },
  selectorOption: {
    paddingHorizontal: 15,
    paddingVertical: 10,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    backgroundColor: '#fff',
    minWidth: 120,
    alignItems: 'center',
  },
  selectorOptionSelected: {
    backgroundColor: '#015185',
  },
  selectorOptionText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
  },
  selectorOptionTextSelected: {
    color: '#fff',
  },
  signatureButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 15,
    paddingVertical: 10,
    backgroundColor: '#fff',
    minHeight: 40,
    width: '100%',
  },
  signatureButtonText: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginLeft: 8,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 20,
    width: width * 0.9,
    maxHeight: height * 0.6,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  signatureArea: {
    borderWidth: 2,
    borderColor: '#ddd',
    borderStyle: 'dashed',
    borderRadius: 8,
    padding: 40,
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#f8f9fa',
    marginBottom: 20,
    minHeight: 150,
  },
  signaturePlaceholder: {
    fontSize: 18,
    color: '#666',
    fontWeight: '600',
  },
  signatureNote: {
    fontSize: 14,
    color: '#999',
    marginTop: 5,
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalButton: {
    flex: 1,
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 6,
    marginHorizontal: 5,
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#ddd',
    backgroundColor: '#fff',
  },
  modalButtonPrimary: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  modalButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
  },
  modalButtonTextPrimary: {
    color: '#fff',
  },
  successIconContainer: {
    alignItems: 'center',
    marginBottom: 15,
  },
  modalSubtitle: {
    fontSize: 14,
    color: '#555',
    textAlign: 'center',
    marginBottom: 20,
  },
  successModalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginTop: 20,
  },
  successModalButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 12,
    paddingHorizontal: 25,
    borderRadius: 6,
    width: '45%',
  },
  backButton: {
    backgroundColor: '#f8f9fa',
    borderWidth: 1,
    borderColor: '#015185',
  },
  backButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 10,
  },
  viewPdfButton: {
    backgroundColor: '#015185',
  },
  viewPdfButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 10,
  },
  disabledButton: {
    opacity: 0.7,
  },
  disabledButtonText: {
    color: '#999',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 20,
    backgroundColor: '#fff',
    borderRadius: 8,
    margin: 20,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  loadingText: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginTop: 10,
    textAlign: 'center',
  },
  loadingSubtext: {
    fontSize: 14,
    color: '#666',
    marginTop: 5,
    textAlign: 'center',
  },
  pageTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginTop: 20,
    marginBottom: 15,
  },
  dataSummaryContainer: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    padding: 15,
    margin: 20,
    marginBottom: 30,
  },
  dataSummaryTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  dataSummaryRow: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginBottom: 8,
  },
  dataSummaryLabel: {
    fontSize: 16,
    color: '#333',
    fontWeight: '600',
  },
  dataSummaryValue: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
  },
  projectInfoPreviewContainer: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    padding: 15,
    margin: 20,
    marginBottom: 30,
  },
  projectInfoPreviewTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  projectInfoPreviewRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 8,
  },
  projectInfoPreviewLabel: {
    fontSize: 16,
    color: '#333',
    fontWeight: '600',
  },
  projectInfoPreviewValue: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  syncModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '70%',
    maxWidth: 300,
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
    lineHeight: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  disabledButton: {
    opacity: 0.5,
  },
  disabledButtonText: {
    opacity: 0.5,
  },
});

export default HSE_122Step3;