import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TextInput,
  ScrollView,
  TouchableOpacity,
  Dimensions,
  KeyboardAvoidingView,
  Platform,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import DateTimePicker from '@react-native-community/datetimepicker';

const { width, height } = Dimensions.get('window');

const HSE_051step2 = ({ setActiveStep }) => {
  const [formData, setFormData] = useState({
    projectSite: '',
    area: '',
    drillerPicSupervisor: '',
    date: '',
    requiredPpe: '',
    listAssociatedProcedures: '',
  });

  const [showDatePicker, setShowDatePicker] = useState(false);
  const [selectedDate, setSelectedDate] = useState(new Date());

  useEffect(() => {
    loadSavedData();
  }, []);

  const loadSavedData = async () => {
    try {
      // Load step2 saved data first
      const savedData = await AsyncStorage.getItem('hse051Step2');
      let next = { ...formData };
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        next = { ...next, ...parsedData };
        if (parsedData.date) {
          setSelectedDate(new Date(parsedData.date));
        }
      }
      // Also load Step 1 to auto-fill Project / Site
      const step1Raw = await AsyncStorage.getItem('hseStep1');
      if (step1Raw) {
        const step1 = JSON.parse(step1Raw);
        if (step1.project) {
          next.projectSite = step1.project;
        }
      }
      setFormData(next);
      // Persist the merged data for consistency
      await AsyncStorage.setItem('hse051Step2', JSON.stringify(next));
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async (data) => {
    try {
      await AsyncStorage.setItem('hse051Step2', JSON.stringify(data));
    } catch (error) {
      console.error('Error saving data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save data',
        position: 'top',
      });
    }
  };

  const handleInputChange = (field, value) => {
    const updatedData = { ...formData, [field]: value };
    setFormData(updatedData);
    saveData(updatedData);
  };

  const showDatePickerModal = () => {
    setShowDatePicker(true);
  };

  const handleDateChange = (event, selectedDate) => {
    setShowDatePicker(false);
    if (selectedDate) {
      setSelectedDate(selectedDate);
      const dateString = selectedDate.toLocaleDateString('en-GB');
      handleInputChange('date', dateString);
    }
  };

  const formatDate = (date) => {
    if (!date) return 'Select Date';
    if (typeof date === 'string') {
      try {
        date = new Date(date);
      } catch (error) {
        return 'Select Date';
      }
    }
    if (!(date instanceof Date) || isNaN(date.getTime())) {
      return 'Select Date';
    }
    return date.toLocaleDateString('en-GB');
  };

  const handleNext = () => {
    // Validate required fields
    const requiredFields = ['projectSite', 'area', 'drillerPicSupervisor'];
    const missingFields = requiredFields.filter(field => !formData[field]);
    
    if (missingFields.length > 0) {
      Toast.show({
        type: 'error',
        text1: 'Required Fields Missing',
        text2: `Please fill: ${missingFields.join(', ')}`,
        position: 'top',
      });
      return;
    }
    
    setActiveStep(3);
  };

  const handleBack = () => {
    setActiveStep(1);
  };

  return (
    <KeyboardAvoidingView 
      style={styles.container} 
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <ScrollView style={styles.scrollView} contentContainerStyle={styles.scrollContent}>
        {/* Header */}
       

        {/* Form Content */}
        <View style={styles.formContainer}>
          {/* Task Being Analysed Section */}
          <View style={styles.section}>
            <View style={styles.sectionHeader}>
              <Text style={styles.sectionTitle}>TASK BEING ANALYSED</Text>
            </View>
            
            <View style={styles.formFields}>
              <View style={styles.fieldContainer}>
                <Text style={styles.fieldLabel}>PROJECT / SITE:</Text>
                <TextInput
                  style={[styles.input, styles.readonly]}
                  value={formData.projectSite}
                  editable={false}
                  selectTextOnFocus={false}
                  placeholderTextColor="#999"
                />
              </View>

              <View style={styles.fieldContainer}>
                <Text style={styles.fieldLabel}>AREA:</Text>
                <TextInput
                  style={styles.input}
                  value={formData.area}
                  onChangeText={(value) => handleInputChange('area', value)}
                  placeholder="Enter specific area"
                  placeholderTextColor="#999"
                />
              </View>

              <View style={styles.fieldContainer}>
                <Text style={styles.fieldLabel}>DRILLER / PIC / SUPERVISOR:</Text>
                <TextInput
                  style={styles.input}
                  value={formData.drillerPicSupervisor}
                  onChangeText={(value) => handleInputChange('drillerPicSupervisor', value)}
                  placeholder="Enter name"
                  placeholderTextColor="#999"
                />
              </View>

              {/* DATE field removed as per requirement */}

              <View style={styles.fieldContainer}>
                <Text style={styles.fieldLabel}>REQUIRED PPE:</Text>
                <TextInput
                  style={styles.input}
                  value={formData.requiredPpe}
                  onChangeText={(value) => handleInputChange('requiredPpe', value)}
                  placeholder="List required personal protective equipment"
                  placeholderTextColor="#999"
                />
              </View>

             
            </View>
          </View>
        </View>
      </ScrollView>

      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>Assessment Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* Date Picker Modal */}
      {showDatePicker && (
        <DateTimePicker
          value={selectedDate}
          mode="date"
          display="default"
          onChange={handleDateChange}
        />
      )}
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  scrollView: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  header: {
    backgroundColor: '#f8f9fa',
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  headerTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: 20,
  },
  formContainer: {
    padding: 20,
  },
  section: {
    marginBottom: 30,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    overflow: 'hidden',
  },
  sectionHeader: {
    backgroundColor: '#f0f0f0',
    padding: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
  },
  formFields: {
    padding: 20,
  },
  fieldContainer: {
    marginBottom: 20,
  },
  fieldLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 8,
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    minHeight: 40,
  },
  textArea: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    minHeight: 80,
    textAlignVertical: 'top',
  },
  dateInput: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    backgroundColor: '#fff',
    minHeight: 40,
  },
  dateInputText: {
    fontSize: 16,
    color: '#333',
    flex: 1,
    marginRight: 10,
  },
  datePlaceholderText: {
    color: '#999',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
});

export default HSE_051step2;