import React, { useEffect, useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  TextInput,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import HSE_085_JSON from './HSE_085_json.json';

const HSE_085_step2 = ({ setActiveStep }) => {
  const [summaryText, setSummaryText] = useState('');
  const [selections, setSelections] = useState({});

  useEffect(() => {
    loadSaved();
  }, []);

  const loadSaved = async () => {
    try {
      // Prefer the common key used by 43
      let saved = await AsyncStorage.getItem('hseStep2');
      if (!saved) saved = await AsyncStorage.getItem('hse085Step2');
      if (saved) {
        const data = JSON.parse(saved);
        setSummaryText(data.summaryText || '');
        setSelections(data.selections || {});
      }
    } catch (e) {
      console.error('Load step2 error:', e);
    }
  };

  const save = async (payload = {}) => {
    try {
      const toSave = {
        summaryText,
        selections,
        ...payload,
        updatedAt: new Date().toISOString(),
      };
      await AsyncStorage.setItem('hseStep2', JSON.stringify(toSave));
      // Clean old module-specific key if it exists
      await AsyncStorage.removeItem('hse085Step2');
    } catch (e) {
      console.error('Save step2 error:', e);
    }
  };

  const handleSelect = async (key, value) => {
    const next = { ...selections, [key]: value };
    setSelections(next);
    await save({ selections: next });
  };

  const handleNext = async () => {
    await save();
    setActiveStep(3);
  };

  // Build checklist items from JSON
  const checklistItems = HSE_085_JSON.checklist || [];

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <Text style={styles.title}>Lift Plan - Pre-task Checks</Text>

        {/* Summary row */}
        <View style={styles.block}>
          <Text style={styles.label}>Summary of Key control identified as part of the pre-task risk assessment</Text>
          <TextInput
            style={[styles.input, styles.textArea]}
            value={summaryText}
            onChangeText={(t) => {
              setSummaryText(t);
              save({ summaryText: t });
            }}
            placeholder="Type summary..."
            multiline
          />
        </View>

        {/* YES/NO items */}
        <View style={styles.block}>
          {checklistItems
            .filter((i) => i.type === 'yes_no')
            .map((item) => {
              const value = selections[item.key];
              return (
                <View key={item.key} style={styles.row}>
                  <Text style={styles.rowLabel}>{item.question}</Text>
                  <View style={styles.rowButtons}>
                    <TouchableOpacity
                      style={[styles.choiceBtn, value === 'YES' && styles.choiceSelectedOk]}
                      onPress={() => handleSelect(item.key, 'YES')}
                    >
                      <Text style={[styles.choiceText, value === 'YES' && styles.choiceTextSelected]}>YES</Text>
                    </TouchableOpacity>
                    <TouchableOpacity
                      style={[styles.choiceBtn, value === 'NO' && styles.choiceSelectedNo]}
                      onPress={() => handleSelect(item.key, 'NO')}
                    >
                      <Text style={[styles.choiceText, value === 'NO' && styles.choiceTextSelected]}>NO</Text>
                    </TouchableOpacity>
                  </View>
                </View>
              );
            })}
        </View>

        {/* Loading Sketch section removed */}
      </ScrollView>

      {/* Footer */}
      <View style={styles.footer}>
        <TouchableOpacity style={styles.backBtn} onPress={() => setActiveStep(1)}>
          <MaterialIcons name="chevron-left" size={16} color="#015185" />
          <Text style={styles.backText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextBtn} onPress={handleNext}>
          <Text style={styles.nextText}>Next</Text>
          <MaterialIcons name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#fff' },
  mainContent: { flex: 1 },
  scrollContent: { paddingBottom: 100, paddingHorizontal: 16, paddingTop: 10 },
  title: { fontSize: 18, fontWeight: '700', color: '#015185', textAlign: 'center', marginBottom: 10 },
  block: { backgroundColor: '#f8f9fa', borderWidth: 1, borderColor: '#e9ecef', borderRadius: 8, padding: 12, marginBottom: 14 },
  label: { fontSize: 14, color: '#015185', fontWeight: '600', marginBottom: 6 },
  input: { borderWidth: 1, borderColor: '#ddd', borderRadius: 6, paddingHorizontal: 12, paddingVertical: 10, backgroundColor: '#fff', color: '#333' },
  textArea: { minHeight: 90, textAlignVertical: 'top' },
  row: { marginBottom: 10 },
  rowLabel: { fontSize: 14, color: '#333', marginBottom: 8 },
  rowButtons: { flexDirection: 'row' },
  choiceBtn: { paddingVertical: 8, paddingHorizontal: 12, borderWidth: 1, borderColor: '#ccc', borderRadius: 6, marginRight: 8, backgroundColor: '#fff' },
  choiceSelectedOk: { backgroundColor: '#e8f5e9', borderColor: '#4CAF50' },
  choiceSelectedNo: { backgroundColor: '#ffebee', borderColor: '#FF4B2B' },
  choiceText: { color: '#333', fontWeight: '600' },
  choiceTextSelected: { color: '#000' },
  sectionTitle: { fontSize: 15, color: '#015185', fontWeight: '700', marginBottom: 8 },
  footer: { position: 'absolute', left: 0, right: 0, bottom: 0, backgroundColor: '#F2F2F2', paddingHorizontal: 15, paddingTop: 12, paddingBottom: 20, borderTopWidth: 1, borderTopColor: '#ddd', flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center' },
  backBtn: { flexDirection: 'row', alignItems: 'center' },
  backText: { color: '#015185', fontSize: 16, fontWeight: '600', marginLeft: 4 },
  nextBtn: { flexDirection: 'row', alignItems: 'center' },
  nextText: { color: '#015185', fontSize: 16, fontWeight: '600', marginRight: 6 },
});

export default HSE_085_step2;