<?php $__env->startSection('title', 'Request Maintenance'); ?>

<?php $__env->startSection('styles'); ?>
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<?php $__env->stopSection(); ?>

<?php
    function getStatusColor($status) {
        $status = strtolower($status);
        switch ($status) {
            case 'completed':
            case 'closed':
            case 'approved':
                return 'success';
            case 'in progress':
            case 'processing':
            case 'pending':
                return 'primary';
            case 'on hold':
                return 'warning';
            case 'initiated':
            case 'requested':
                return 'info';
            case 'rejected':
            case 'cancelled':
                return 'danger';
            case 'archived':
                return 'secondary';
            default:
                return 'info';
        }
    }

    function getPriorityColor($priority) {
        $priority = strtolower($priority);
        switch ($priority) {
            case 'high':
            case 'urgent':
                return 'danger';
            case 'medium':
                return 'warning';
            case 'low':
                return 'success';
            default:
                return 'secondary';
        }
    }
?>

<?php $__env->startSection('content'); ?>
<div class="request-maintenance-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="javascript:history.back()" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Request Maintenance</h1>
            </div>
            <div class="header-right">
                <div class="search-container">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="search-icon">
                        <circle cx="11" cy="11" r="8"/>
                        <path d="M21 21l-4.35-4.35"/>
                    </svg>
                    <input type="text" class="search-input" placeholder="Search maintenance requests..." id="requestSearch">
                </div>
                <a href="/create-new-request-maintenance" class="create-btn">
                    <i class="fas fa-plus"></i>
                    Create Request
                </a>
            </div>
        </div>
    </div>
    
    <!-- Main Content -->
    <div class="main-content">
        <div class="request-maintenance-content">
        <div class="table-section">
            <div class="table-container">
                <table class="request-maintenance-table">
                    <thead class="table-header">
                        <tr>
                            <th style="width: 60px;">S.No</th>
                            <th style="width: 120px;">Action</th>
                            <th>WO Number</th>
                            <th>Created Date</th>
                            <th>Vehicle</th>
                            <th>Status</th>
                            <th>Priority</th>
                            <th>Assigned</th>
                            <th>Memo</th>
                            <th>Requested By</th>
                        </tr>
                    </thead>
                    <tbody class="table-body" id="requestTableBody">
                        <tr class="loading-row">
                            <td colspan="10" class="loading-state">
                                <div class="loading-content">
                                    <div class="loading-spinner"></div>
                                    <h3>Loading Maintenance Requests...</h3>
                                    <p>Please wait while we fetch your data.</p>
                                </div>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <div class="pagination-container" id="paginationContainer" style="display: none;">
                <div class="pagination-info">
                    Showing <span id="showingStart">0</span> to <span id="showingEnd">0</span> of <span id="totalRecords">0</span> entries
                </div>
                <div class="pagination-buttons" id="paginationButtons">
                    <!-- Pagination buttons will be inserted here -->
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div id="deleteConfirmModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content" style="max-width: 640px; width: 100%;">
        <div class="custom-modal-header">
            <h3>Confirm Delete</h3>
            <button type="button" class="close-btn" onclick="closeDeleteConfirmModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div style="text-align: center; padding: 20px;">
                <i class="fas fa-exclamation-triangle" style="font-size: 48px; color: #dc3545; margin-bottom: 20px;"></i>
                <h4 style="margin-bottom: 10px; color: #374151;">Are you sure?</h4>
                <p style="color: #6b7280; margin-bottom: 0;">This maintenance request will be permanently deleted. This action cannot be undone.</p>
            </div>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeDeleteConfirmModal()">Cancel</button>
            <button type="button" class="btn btn-danger" onclick="confirmDelete()">Delete</button>
        </div>
    </div>
</div>

<!-- View Request Details Modal -->
<div id="viewDetailsModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content" style="max-width: 960px; width: 100%;">
        <div class="custom-modal-header">
            <h3>Maintenance Request Details</h3>
            <button type="button" class="close-btn" onclick="closeViewDetailsModal()">&times;</button>
        </div>
        <div class="custom-modal-body" id="viewDetailsBody" style="max-height: 70vh; overflow-y: auto;">
            <!-- Details will be populated here -->
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-primary" onclick="closeViewDetailsModal()">Close</button>
        </div>
    </div>
</div>

<!-- Approve/Reject Modal -->
<div id="approveRejectModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content" style="max-width: 720px; width: 100%;">
        <div class="custom-modal-header">
            <h3>Approve or Reject Request</h3>
            <button type="button" class="close-btn" onclick="closeApproveRejectModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="form-group" style="margin-bottom: 20px;">
                <label class="form-label" style="display: block; margin-bottom: 8px; font-weight: 600; color: #374151;">Decision <span style="color: #dc3545;">*</span></label>
                <select id="approveRejectStatus" class="form-input" style="width: 100%; padding: 12px; border: 1px solid #d1d5db; border-radius: 6px; font-size: 14px;">
                    <option value="">Select Decision</option>
                    <option value="Approved">Approve</option>
                    <option value="Rejected">Reject</option>
                </select>
                </div>
            <div class="form-group" style="margin-bottom: 0;">
                <label class="form-label" style="display: block; margin-bottom: 8px; font-weight: 600; color: #374151;">Note</label>
                <textarea id="approveRejectNote" class="form-input" rows="4" placeholder="Add a note (optional)" style="width: 100%; padding: 12px; border: 1px solid #d1d5db; border-radius: 6px; font-size: 14px; resize: vertical;"></textarea>
            </div>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeApproveRejectModal()">Cancel</button>
            <button type="button" class="btn btn-primary" onclick="submitApproveReject()">Submit</button>
        </div>
    </div>
</div>

<script>
(function() {
    'use strict';

    // Configuration object for centralized settings
    const CONFIG = {
        selectors: {
            requestSearch: '#requestSearch',
            requestTableBody: '#requestTableBody',
            sidebar: '.sidebar'
        },
        messages: {
            loading: 'Loading...',
            error: 'An error occurred. Please try again.',
            success: 'Operation completed successfully!'
        },
        timing: {
            debounceDelay: 300,
            notificationDuration: 3000
        },
        pagination: {
            itemsPerPage: 10,
            maxVisiblePages: 5
        }
    };

    // DOM element caching
    let elements = {};
    
    // Pagination variables
    let allRequests = [];
    let currentPage = 1;
    let totalPages = 0;
    
    // Modal state variables
    let currentDeleteId = null;
    let currentApproveRejectId = null;
    let requestDropdownInitialized = false;

    // Initialize DOM elements
    function initElements() {
        elements = {
            requestSearch: document.querySelector(CONFIG.selectors.requestSearch),
            requestTableBody: document.querySelector(CONFIG.selectors.requestTableBody),
            sidebar: document.querySelector(CONFIG.selectors.sidebar)
        };
    }

    // Initialize the application
    function init() {
        initElements();
        initEventListeners();
        loadMaintenanceRequests();
        initializeRequestDropdowns();
    }

    // Initialize event listeners
    function initEventListeners() {
        // Search functionality
        if (elements.requestSearch) {
            elements.requestSearch.addEventListener('input', function() {
                filterTable(CONFIG.selectors.requestTableBody, this.value);
            });
        }
    }

    // Toggle sidebar
    function toggleSidebar() {
        if (elements.sidebar) {
            elements.sidebar.classList.toggle('collapsed');
        }
    }

    // Load maintenance requests from MongoDB
    async function loadMaintenanceRequests() {
        try {
            console.log('🔄 Loading maintenance requests from MongoDB...');
            
            const response = await fetch('https://api.titandrillingzm.com:6009/request-maintenance');
            const result = await response.json();
            
            console.log('Raw MongoDB data:', result);
            
            if (result.success && result.data && Array.isArray(result.data)) {
                allRequests = result.data;
                
                console.log(`Found ${allRequests.length} total maintenance requests`);
                
                // Sort by creation date (newest first)
                allRequests.sort((a, b) => {
                    const dateA = new Date(a.createdAt || 0).getTime();
                    const dateB = new Date(b.createdAt || 0).getTime();
                    return dateB - dateA;
                });
                
                // Calculate total pages
                totalPages = Math.ceil(allRequests.length / CONFIG.pagination.itemsPerPage);
                currentPage = 1;
                
                // Display first page
                displayCurrentPage();
                
                // Show pagination
                updatePagination();
                
                if (allRequests.length > 0) {
                    showNotification(`Loaded ${allRequests.length} maintenance requests successfully`, 'success');
                } else {
                    showNotification('No maintenance requests found', 'info');
                }
            } else {
                console.log('No data found in MongoDB or invalid format');
                displayEmptyState();
            }
        } catch (error) {
            console.error('❌ Error loading maintenance requests:', error);
            showNotification('Failed to load maintenance requests: ' + error.message, 'error');
            displayEmptyState();
        }
    }

    // Display current page of requests
    function displayCurrentPage() {
        if (!elements.requestTableBody) return;
        
        if (allRequests.length === 0) {
            displayEmptyState();
            return;
        }
        
        const startIndex = (currentPage - 1) * CONFIG.pagination.itemsPerPage;
        const endIndex = Math.min(startIndex + CONFIG.pagination.itemsPerPage, allRequests.length);
        const pageRequests = allRequests.slice(startIndex, endIndex);
        
        const rows = pageRequests.map((request, index) => {
            const serialNumber = startIndex + index + 1;
            const requestIdValue = request.id || request._id || `request-${serialNumber}`;
            const safeRequestId = String(requestIdValue).replace(/\\/g, '\\\\').replace(/'/g, "\\'");
            const dropdownLabel = `request-actions-${serialNumber}`;
            
            // Format date
            let formattedDate = 'N/A';
            if (request.createdAt) {
                try {
                    const date = new Date(request.createdAt);
                    formattedDate = date.toLocaleDateString('en-GB', {
                        day: '2-digit',
                        month: '2-digit',
                        year: 'numeric'
                    });
                } catch (error) {
                    console.error('Error formatting date:', error);
                }
            }
            
            // Extract vehicle information
            let vehicleInfo = 'N/A';
            if (request.equipment) {
                vehicleInfo = request.equipment.equipmentName || request.equipment.model || 'Unknown Equipment';
            }
            
            // Extract assigned users
            let assignedUsers = 'N/A';
            if (request.assignTo && Array.isArray(request.assignTo) && request.assignTo.length > 0) {
                assignedUsers = request.assignTo.map(user => user.fullName || user.name || 'Unknown').join(', ');
            }
            
            // Extract memo/description
            const memo = request.description || request.memo || 'No memo';
            
            // Extract requested by (using adminUid or userId)
            const requestedBy = request.adminUid || request.userId || 'Unknown';
            
            return `
                <tr class="table-row">
                    <td style="text-align: center; font-weight: 600;">${serialNumber}</td>
                    <td>
                        <div class="dropdown request-dropdown">
                            <button class="action-btn" type="button" aria-haspopup="true" aria-expanded="false" aria-label="Actions for request ${serialNumber}" id="${dropdownLabel}">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="12" cy="12" r="3"></circle>
                                    <path d="M12 1v6m0 6v6m11-7h-6m-6 0H1"></path>
                                </svg>
                                Actions
                            </button>
                            <ul class="dropdown-menu" role="menu" aria-labelledby="${dropdownLabel}">
                                <li role="none">
                                    <a class="dropdown-item" href="#" role="menuitem" onclick="event.preventDefault(); closeRequestDropdowns(); viewRequestDetails('${safeRequestId}')">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                                        <polyline points="14,2 14,8 20,8"></polyline>
                                    </svg>
                                    View Maintenance
                                    </a>
                                </li>
                                <li role="none">
                                    <a class="dropdown-item" href="#" role="menuitem" onclick="event.preventDefault(); closeRequestDropdowns(); openApproveRejectModal('${safeRequestId}', '${(request.status || '').replace(/'/g, "\\'")}')">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <polyline points="20 6 9 17 4 12"></polyline>
                                    </svg>
                                    Approve/Reject
                                    </a>
                                </li>
                                <li><hr class="dropdown-divider"></li>
                                <li role="none">
                                    <a class="dropdown-item danger" href="#" role="menuitem" onclick="event.preventDefault(); closeRequestDropdowns(); openDeleteConfirmModal('${safeRequestId}')">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <polyline points="3,6 5,6 21,6"></polyline>
                                        <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                                    </svg>
                                    Delete
                                    </a>
                                </li>
                            </ul>
                        </div>
                    </td>
                    <td class="wo-number">${request.requestNumber || 'N/A'}</td>
                    <td>${formattedDate}</td>
                    <td>${vehicleInfo}</td>
                    <td>
                        <span class="status-badge status-${getStatusColorClass(request.status)}">${request.status || 'Pending'}</span>
                    </td>
                    <td>
                        <span class="priority-badge priority-${getPriorityColorClass(request.priority)}">${request.priority || 'Medium'}</span>
                    </td>
                    <td>${assignedUsers}</td>
                    <td class="memo-text">${memo}</td>
                    <td>${requestedBy}</td>
                </tr>
            `;
        }).join('');
        
        elements.requestTableBody.innerHTML = rows;
    }

    // Display empty state
    function displayEmptyState() {
        if (!elements.requestTableBody) return;
        
        elements.requestTableBody.innerHTML = `
            <tr class="empty-row">
                <td colspan="10" class="empty-state">
                    <div class="empty-content">
                        <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M12 2L2 7l10 5 10-5-10-5z"/>
                            <path d="M2 17l10 5 10-5"/>
                            <path d="M2 12l10 5 10-5"/>
                        </svg>
                        <h3>No Maintenance Requests</h3>
                        <p>There are currently no maintenance requests to display.</p>
                    </div>
                </td>
            </tr>
        `;
        
        // Hide pagination
        const paginationContainer = document.getElementById('paginationContainer');
        if (paginationContainer) {
            paginationContainer.style.display = 'none';
        }
    }
    
    // Pagination functions
    function updatePagination() {
        const paginationContainer = document.getElementById('paginationContainer');
        const paginationButtons = document.getElementById('paginationButtons');
        
        if (!paginationContainer || !paginationButtons) return;
        
        if (totalPages <= 1) {
            paginationContainer.style.display = 'none';
            return;
        }
        
        paginationContainer.style.display = 'flex';
        
        // Update info
        const startIndex = (currentPage - 1) * CONFIG.pagination.itemsPerPage + 1;
        const endIndex = Math.min(currentPage * CONFIG.pagination.itemsPerPage, allRequests.length);
        
        document.getElementById('showingStart').textContent = startIndex;
        document.getElementById('showingEnd').textContent = endIndex;
        document.getElementById('totalRecords').textContent = allRequests.length;
        
        // Generate pagination buttons
        let buttonsHTML = '';
        
        // Previous button
        buttonsHTML += `
            <button class="pagination-btn ${currentPage === 1 ? 'disabled' : ''}" 
                    onclick="goToPage(${currentPage - 1})" 
                    ${currentPage === 1 ? 'disabled' : ''}>
                <i class="fas fa-chevron-left"></i>
            </button>
        `;
        
        // Page number buttons
        const maxVisible = CONFIG.pagination.maxVisiblePages;
        let startPage = Math.max(1, currentPage - Math.floor(maxVisible / 2));
        let endPage = Math.min(totalPages, startPage + maxVisible - 1);
        
        if (endPage - startPage < maxVisible - 1) {
            startPage = Math.max(1, endPage - maxVisible + 1);
        }
        
        if (startPage > 1) {
            buttonsHTML += `<button class="pagination-btn" onclick="goToPage(1)">1</button>`;
            if (startPage > 2) {
                buttonsHTML += `<span class="pagination-ellipsis">...</span>`;
            }
        }
        
        for (let i = startPage; i <= endPage; i++) {
            buttonsHTML += `
                <button class="pagination-btn ${i === currentPage ? 'active' : ''}" 
                        onclick="goToPage(${i})">
                    ${i}
                </button>
            `;
        }
        
        if (endPage < totalPages) {
            if (endPage < totalPages - 1) {
                buttonsHTML += `<span class="pagination-ellipsis">...</span>`;
            }
            buttonsHTML += `<button class="pagination-btn" onclick="goToPage(${totalPages})">${totalPages}</button>`;
        }
        
        // Next button
        buttonsHTML += `
            <button class="pagination-btn ${currentPage === totalPages ? 'disabled' : ''}" 
                    onclick="goToPage(${currentPage + 1})" 
                    ${currentPage === totalPages ? 'disabled' : ''}>
                <i class="fas fa-chevron-right"></i>
            </button>
        `;
        
        paginationButtons.innerHTML = buttonsHTML;
    }
    
    function goToPage(page) {
        if (page < 1 || page > totalPages || page === currentPage) return;
        
        currentPage = page;
        displayCurrentPage();
        updatePagination();
        
        // Scroll to top of table
        const tableSection = document.querySelector('.table-section');
        if (tableSection) {
            tableSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }
    }

    function closeRequestDropdowns() {
        document.querySelectorAll('.request-dropdown .dropdown-menu.show').forEach(menu => menu.classList.remove('show'));
        document.querySelectorAll('.request-dropdown.show').forEach(dropdown => {
            dropdown.classList.remove('show');
            const toggle = dropdown.querySelector('.action-btn');
            if (toggle) {
                toggle.setAttribute('aria-expanded', 'false');
            }
        });
    }

    // Delete Modal Functions
    function openDeleteConfirmModal(requestId) {
        currentDeleteId = requestId;
        const modal = document.getElementById('deleteConfirmModal');
        modal.style.display = 'flex';
        document.body.classList.add('modal-open');
        
        // Close any open dropdowns
        closeRequestDropdowns();
    }
    
    function closeDeleteConfirmModal() {
        const modal = document.getElementById('deleteConfirmModal');
        modal.style.display = 'none';
        document.body.classList.remove('modal-open');
        currentDeleteId = null;
    }
    
    async function confirmDelete() {
        if (!currentDeleteId) return;
        
        try {
            const response = await fetch(`https://api.titandrillingzm.com:6009/request-maintenance/${currentDeleteId}`, {
                method: 'DELETE'
            });
            
            const result = await response.json();
            
            if (result.success) {
                showNotification('Maintenance request deleted successfully', 'success');
                closeDeleteConfirmModal();
                // Reload requests
                await loadMaintenanceRequests();
            } else {
                showNotification('Failed to delete request: ' + result.error, 'error');
            }
        } catch (error) {
            console.error('Error deleting request:', error);
            showNotification('Failed to delete request: ' + error.message, 'error');
        }
    }

    // View Details Modal Functions
    function viewRequestDetails(requestId) {
        const request = allRequests.find(r => r.id === requestId || r._id === requestId);
        
        if (!request) {
            showNotification('Request not found', 'error');
            return;
        }
        
        // Format date
        let formattedDate = 'N/A';
        if (request.createdAt) {
            try {
            const date = new Date(request.createdAt);
                formattedDate = date.toLocaleString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
            } catch (error) {
                console.error('Error formatting date:', error);
            }
        }
        
        // Extract equipment information
        let equipmentDetails = 'N/A';
        if (request.equipment) {
            equipmentDetails = `
                <div style="background: #f9fafb; padding: 15px; border-radius: 8px; margin-top: 10px;">
                    <div style="margin-bottom: 8px;"><strong>Name:</strong> ${request.equipment.equipmentName || 'N/A'}</div>
                    <div style="margin-bottom: 8px;"><strong>Model:</strong> ${request.equipment.model || 'N/A'}</div>
                    <div style="margin-bottom: 8px;"><strong>Category:</strong> ${request.equipment.mainCategory || 'N/A'}</div>
                    <div style="margin-bottom: 8px;"><strong>Serial Number:</strong> ${request.equipment.serialNumber || 'N/A'}</div>
                    <div><strong>Status:</strong> <span class="status-badge status-${getStatusColorClass(request.equipment.status)}">${request.equipment.status || 'N/A'}</span></div>
                        </div>
            `;
        }
        
        // Extract assigned users
        let assignedUsersDetails = 'N/A';
        if (request.assignTo && Array.isArray(request.assignTo) && request.assignTo.length > 0) {
            assignedUsersDetails = request.assignTo.map(user => `
                <div style="background: #f9fafb; padding: 10px; border-radius: 6px; margin-bottom: 8px;">
                    <div style="font-weight: 600; color: #0055a5; margin-bottom: 4px;">${user.fullName || user.name || 'Unknown'}</div>
                    <div style="font-size: 13px; color: #6b7280;">
                        ${user.email ? `<div>Email: ${user.email}</div>` : ''}
                        ${user.role ? `<div>Role: ${user.role}</div>` : ''}
                        </div>
                        </div>
            `).join('');
        }
        
        // Extract tasks
        let tasksDetails = '<div style="color: #6b7280;">No tasks assigned</div>';
        if (request.tasks && Array.isArray(request.tasks) && request.tasks.length > 0) {
            tasksDetails = request.tasks.map((task, index) => `
                <div style="background: #f9fafb; padding: 15px; border-radius: 8px; margin-bottom: 10px; border-left: 4px solid #0055a5;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 8px;">${index + 1}. ${task.title || 'Untitled Task'}</div>
                    <div style="font-size: 13px; color: #6b7280; margin-bottom: 6px;">
                        <strong>Type:</strong> ${task.description || task.type || 'N/A'}
                        </div>
                    <div style="font-size: 13px; margin-bottom: 6px;">
                        <strong>Status:</strong> <span class="status-badge status-${getStatusColorClass(task.status)}">${task.status || 'Pending'}</span>
                        </div>
                    ${task.note ? `<div style="font-size: 13px; color: #6b7280; font-style: italic;">Note: ${task.note}</div>` : ''}
                    ${task.assignees && task.assignees.length > 0 ? `
                        <div style="font-size: 13px; color: #6b7280; margin-top: 6px;">
                            <strong>Assigned to:</strong> ${task.assignees.map(a => a.fullName).join(', ')}
                        </div>
                    ` : ''}
                        </div>
            `).join('');
        }
        
        const detailsHTML = `
            <div style="padding: 20px;">
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 25px;">
                    <div>
                        <div style="font-weight: 600; color: #6b7280; margin-bottom: 5px; font-size: 13px;">Request Number</div>
                        <div style="font-size: 16px; font-weight: 600; color: #0055a5;">${request.requestNumber || 'N/A'}</div>
                        </div>
                    <div>
                        <div style="font-weight: 600; color: #6b7280; margin-bottom: 5px; font-size: 13px;">Created Date</div>
                        <div style="font-size: 15px; color: #374151;">${formattedDate}</div>
                        </div>
                    <div>
                        <div style="font-weight: 600; color: #6b7280; margin-bottom: 5px; font-size: 13px;">Status</div>
                        <div><span class="status-badge status-${getStatusColorClass(request.status)}">${request.status || 'Pending'}</span></div>
                        </div>
                    <div>
                        <div style="font-weight: 600; color: #6b7280; margin-bottom: 5px; font-size: 13px;">Priority</div>
                        <div><span class="priority-badge priority-${getPriorityColorClass(request.priority)}">${request.priority || 'Medium'}</span></div>
                    </div>
                </div>
                
                <div style="margin-bottom: 25px;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 15px;">Work Order Title</div>
                    <div style="color: #6b7280;">${request.workOrderTitle || 'N/A'}</div>
                            </div>
                
                <div style="margin-bottom: 25px;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 15px;">Description / Memo</div>
                    <div style="color: #6b7280;">${request.description || request.memo || 'No description provided'}</div>
                    </div>
                
                <div style="margin-bottom: 25px;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 15px;">Equipment Details</div>
                    ${equipmentDetails}
                </div>
                
                <div style="margin-bottom: 25px;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 15px;">Assigned To</div>
                    ${assignedUsersDetails}
                                </div>
                
                <div style="margin-bottom: 0;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 15px;">Tasks (${request.tasks ? request.tasks.length : 0})</div>
                    ${tasksDetails}
                    </div>
            </div>
        `;
        
        document.getElementById('viewDetailsBody').innerHTML = detailsHTML;
        
        const modal = document.getElementById('viewDetailsModal');
        modal.style.display = 'flex';
        document.body.classList.add('modal-open');
        
        // Close any open dropdowns
        closeRequestDropdowns();
    }
    
    function closeViewDetailsModal() {
        const modal = document.getElementById('viewDetailsModal');
        modal.style.display = 'none';
        document.body.classList.remove('modal-open');
    }

    // Approve/Reject Modal Functions
    function openApproveRejectModal(requestId, currentStatus) {
        currentApproveRejectId = requestId;
        
        // Reset form
        document.getElementById('approveRejectStatus').value = '';
        document.getElementById('approveRejectNote').value = '';
        
        const modal = document.getElementById('approveRejectModal');
        modal.style.display = 'flex';
        document.body.classList.add('modal-open');
        
        // Close any open dropdowns
        closeRequestDropdowns();
    }

    function closeApproveRejectModal() {
        const modal = document.getElementById('approveRejectModal');
        modal.style.display = 'none';
        document.body.classList.remove('modal-open');
        currentApproveRejectId = null;
    }
    
    async function submitApproveReject() {
        if (!currentApproveRejectId) return;
        
        const status = document.getElementById('approveRejectStatus').value;
        const note = document.getElementById('approveRejectNote').value;
        
        if (!status) {
            showNotification('Please select a decision', 'warning');
            return;
        }
        
        try {
            const updateData = {
                status: status,
                approvalNote: note,
                approvedAt: new Date().toISOString(),
                approvedBy: 'Admin' // Should be from logged-in user
            };
            
            const response = await fetch(`https://api.titandrillingzm.com:6009/request-maintenance/${currentApproveRejectId}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(updateData)
            });
            
            const result = await response.json();
                
            if (result.success) {
                showNotification(`Request ${status.toLowerCase()} successfully`, 'success');
                closeApproveRejectModal();
                // Reload requests
                await loadMaintenanceRequests();
            } else {
                showNotification('Failed to update request: ' + result.error, 'error');
            }
        } catch (error) {
            console.error('Error updating request:', error);
            showNotification('Failed to update request: ' + error.message, 'error');
        }
    }

    // Helper functions for status and priority colors
    function getStatusColorClass(status) {
        if (!status) return 'primary';
        const statusLower = status.toLowerCase();
        if (['completed', 'closed', 'approved'].includes(statusLower)) return 'success';
        if (['in progress', 'processing', 'pending'].includes(statusLower)) return 'primary';
        if (['on hold'].includes(statusLower)) return 'warning';
        if (['initiated', 'requested'].includes(statusLower)) return 'info';
        if (['rejected', 'cancelled'].includes(statusLower)) return 'danger';
        if (['archived'].includes(statusLower)) return 'secondary';
        return 'primary';
    }

    function getPriorityColorClass(priority) {
        if (!priority) return 'warning';
        const priorityLower = priority.toLowerCase();
        if (['high', 'urgent', 'emergency'].includes(priorityLower)) return 'danger';
        if (['medium'].includes(priorityLower)) return 'warning';
        if (['low'].includes(priorityLower)) return 'success';
        return 'warning';
    }

    // Filter table function
    function filterTable(tableBodyId, searchTerm) {
        const tableBody = document.querySelector(tableBodyId);
        if (!tableBody) return;
        
        const rows = tableBody.getElementsByTagName('tr');
        
        for (let row of rows) {
            const cells = row.getElementsByTagName('td');
            let found = false;
            
            for (let cell of cells) {
                if (cell.textContent.toLowerCase().includes(searchTerm.toLowerCase())) {
                    found = true;
                    break;
                }
            }
            
            row.style.display = found ? '' : 'none';
        }
    }

    // Notification system
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `request-notification request-notification-${type}`;
        notification.textContent = message;
        
        notification.style.cssText = `
            position: fixed;
            top: 80px;
            right: 20px;
            padding: 12px 20px;
            border-radius: 0;
            color: white;
            font-weight: 500;
            font-size: 14px;
            z-index: 10001;
            opacity: 0;
            transform: translateX(100%);
            transition: all 0.3s ease;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            font-family: 'Inter', sans-serif;
        `;
        
        if (type === 'success') {
            notification.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
        } else if (type === 'error') {
            notification.style.background = 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)';
        } else if (type === 'warning') {
            notification.style.background = 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)';
        } else {
            notification.style.background = 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)';
        }
        
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.style.opacity = '1';
            notification.style.transform = 'translateX(0)';
        }, 100);
        
        setTimeout(() => {
            notification.style.opacity = '0';
            notification.style.transform = 'translateX(100%)';
            setTimeout(() => {
                if (notification.parentNode) {
                    document.body.removeChild(notification);
                }
            }, 300);
        }, CONFIG.timing.notificationDuration);
    }

    // Make functions globally accessible
    window.toggleSidebar = toggleSidebar;
    window.goToPage = goToPage;
    window.closeRequestDropdowns = closeRequestDropdowns;
     
    // Modal functions
    window.openDeleteConfirmModal = openDeleteConfirmModal;
    window.closeDeleteConfirmModal = closeDeleteConfirmModal;
    window.confirmDelete = confirmDelete;
    window.viewRequestDetails = viewRequestDetails;
    window.closeViewDetailsModal = closeViewDetailsModal;
    window.openApproveRejectModal = openApproveRejectModal;
    window.closeApproveRejectModal = closeApproveRejectModal;
    window.submitApproveReject = submitApproveReject;

    // Close modals when clicking outside
    window.onclick = function(event) {
        const deleteModal = document.getElementById('deleteConfirmModal');
        const viewModal = document.getElementById('viewDetailsModal');
        const approveRejectModal = document.getElementById('approveRejectModal');
        
        if (event.target === deleteModal) {
            closeDeleteConfirmModal();
        } else if (event.target === viewModal) {
            closeViewDetailsModal();
        } else if (event.target === approveRejectModal) {
            closeApproveRejectModal();
        }
    };

    function initializeRequestDropdowns() {
        if (requestDropdownInitialized) return;

        document.addEventListener('click', function(event) {
            const toggle = event.target.closest('.request-dropdown .action-btn');
            const dropdown = toggle ? toggle.closest('.request-dropdown') : null;

            if (toggle && dropdown) {
                event.preventDefault();
                const menu = dropdown.querySelector('.dropdown-menu');
                const isOpen = menu && menu.classList.contains('show');
                closeRequestDropdowns();
                if (!isOpen && menu) {
                    dropdown.classList.add('show');
                    menu.classList.add('show');
                    toggle.setAttribute('aria-expanded', 'true');
                }
                return;
            }

            if (event.target.closest('.request-dropdown .dropdown-item')) {
                closeRequestDropdowns();
                return;
            }

            if (!event.target.closest('.request-dropdown')) {
                closeRequestDropdowns();
            }
        }, true);

        document.addEventListener('keydown', function(event) {
            const toggle = event.target.closest('.request-dropdown .action-btn');
            if (!toggle) return;

            if (event.key === ' ' || event.key === 'Enter') {
                event.preventDefault();
                toggle.click();
            } else if (event.key === 'Escape') {
                closeRequestDropdowns();
                toggle.focus();
            }
        }, true);

        requestDropdownInitialized = true;
    }

    // Initialize the application
    init();
})();

</script>

<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --secondary-hover: #5a6268;
    --success-color: #28a745;
    --success-hover: #218838;
    --warning-color: #ffc107;
    --warning-hover: #e0a800;
    --danger-color: #dc3545;
    --danger-hover: #c82333;
    --info-color: #17a2b8;
    --info-hover: #138496;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --text-muted: #6c757d;
    --text-light: #ffffff;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --bg-hover: #e9ecef;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --transition-slow: all 0.3s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-modal: 1050;
    --z-index-dropdown: 1000;
}

/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== REQUEST MAINTENANCE CONTAINER ===== */
.request-maintenance-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* ===== HEADER STYLES ===== */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    color: var(--primary-color);
    text-decoration: none;
}

/* Search Container */
.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #9ca3af;
    z-index: 1;
}

.search-input {
    padding: 10px 12px 10px 40px;
    border: 1px solid #d1d5db;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 400;
    color: #374151;
    background: white;
    width: 300px;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.search-input::placeholder {
    color: #9ca3af;
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* ===== CONTENT SECTION ===== */
.request-maintenance-content {
    background: var(--bg-primary);
    border-radius: 0;
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

/* ===== ACTION BAR ===== */
.action-bar {
    background: var(--bg-light);
    border-bottom: 1px solid var(--border-light);
    padding: 20px 24px;
}

.action-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 20px;
}

.search-section {
    flex: 1;
    max-width: 400px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: var(--text-muted);
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid var(--border-color);
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: var(--bg-primary);
    transition: var(--transition-base);
}

.search-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.action-buttons {
    display: flex;
    align-items: center;
    gap: 12px;
}

.btn-create {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    background: #0055a5;
    color: white;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-create:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-2px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    color: black;
    text-decoration: none;
}

/* ===== TABLE SECTION ===== */
.table-section {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
    margin-bottom: 24px;
}

.table-container {
    overflow-x: auto;
    padding-bottom: 400px;
}

.request-maintenance-table {
    width: 100%;
    border-collapse: collapse;
    background: white;
}

.table-header {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.table-header th {
    padding: 16px 12px;
    text-align: left;
    font-size: 14px;
    font-weight: 600;
    color: #374151;
    border-bottom: 1px solid #e5e7eb;
    border-radius: 0;
}

.table-body {
    background: white;
}

.table-row {
    border-bottom: 1px solid #f3f4f6;
    transition: all 0.2s ease;
}

.table-row:hover {
    background: #f8fafc;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

.table-row td {
    padding: 16px 12px;
    font-size: 14px;
    color: #374151;
    vertical-align: middle;
}

/* ===== BUTTON STYLES ===== */
.btn-view {
    display: flex;
    align-items: center;
    gap: 6px;
    padding: 6px 12px;
    background: #0055a5;
    color: white;
    border: none;
    border-radius: 0;
    font-size: 12px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(0, 85, 165, 0.2);
}

.btn-view:hover {
    background: linear-gradient(135deg, #003f7d 0%, #002a5c 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
}

/* ===== STATUS BADGES ===== */
.status-badge {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 0;
    font-size: 12px;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-success {
    color: green;
}

.status-primary {
    /* background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%); */
    /* color: white; */
}

.status-warning {
    /* background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); */
    color: red;
}

.status-info {
    /* background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%); */
    color: #0055a5;
}

.status-danger {
    /* background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); */
    color: red;
}

.status-secondary {
    /* background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%); */
    /* color: white; */
}

/* ===== PRIORITY BADGES ===== */
.priority-badge {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 0;
    font-size: 12px;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.priority-danger {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
}

.priority-warning {
    /* background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); */
    color: green;
}

.priority-success {
    /* background: linear-gradient(135deg, #10b981 0%, #059669 100%); */
    color: #0055a5;
}

.priority-secondary {
    /* background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%); */
    /* color: white; */
}

/* ===== LOADING STATE ===== */
.loading-row {
    background: white;
}

.loading-state {
    text-align: center;
    padding: 48px 24px;
    background: white;
}

.loading-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid #e5e7eb;
    border-top: 4px solid #0055a5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-content h3 {
    font-size: 18px;
    font-weight: 600;
    color: #374151;
    margin: 0;
}

.loading-content p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

/* ===== EMPTY STATE ===== */
.empty-row {
    background: white;
}

.empty-state {
    text-align: center;
    padding: 48px 24px;
    background: white;
}

.empty-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.empty-content svg {
    color: #9ca3af;
}

.empty-content h3 {
    font-size: 18px;
    font-weight: 600;
    color: #374151;
    margin: 0;
}

.empty-content p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .main-content {
        padding: 20px;
    }
    
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .search-input {
        width: 250px;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
        flex-direction: column;
        gap: 15px;
        align-items: stretch;
    }
    
    .header-left {
        justify-content: center;
    }
    
    .header-right {
        justify-content: center;
        flex-direction: column;
        gap: 10px;
    }
    
    .search-input {
        width: 100%;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 992px) {
    .request-maintenance-container {
        padding: 16px;
    }
    
    .request-maintenance-header {
        padding: 20px;
        margin-bottom: 20px;
    }
    
    .page-title h1 {
        font-size: 26px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .action-bar {
        padding: 16px 20px;
    }
    
    .action-buttons {
        flex-wrap: wrap;
        gap: 8px;
    }
    
    .btn-create {
        padding: 10px 16px;
        font-size: 13px;
        flex: 1;
        min-width: 150px;
    }
    
    .table-container {
        overflow-x: auto;
    }
    
    .request-maintenance-table {
        min-width: 800px;
    }
}

@media (max-width: 768px) {
    .request-maintenance-container {
        padding: 12px;
    }
    
    .request-maintenance-header {
        padding: 16px;
        margin-bottom: 16px;
    }
    
    .page-title h1 {
        font-size: 22px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .action-bar {
        padding: 12px 16px;
    }
    
    .action-content {
        flex-direction: column;
        gap: 12px;
        align-items: stretch;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 8px;
    }
    
    .btn-create {
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
        min-height: 48px;
        justify-content: center;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        border: 1px solid #e2e8f0;
    }
    
    .request-maintenance-table {
        min-width: 700px;
    }
    
    .table-header th,
    .table-row td {
        padding: 10px 8px;
        font-size: 13px;
    }
    
    .table-header th {
        font-size: 12px;
        font-weight: 600;
    }
    
    .status-badge,
    .priority-badge {
        padding: 4px 8px;
        font-size: 11px;
    }
    
    .btn-view {
        padding: 6px 10px;
        font-size: 11px;
    }
    
    .memo-text {
        max-width: 150px;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
    }
}

@media (max-width: 576px) {
    .request-maintenance-container {
        padding: 8px;
    }
    
    .request-maintenance-header {
        padding: 12px;
        margin-bottom: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .action-bar {
        padding: 8px 12px;
    }
    
    .action-content {
        gap: 8px;
    }
    
    .btn-create {
        padding: 14px 16px;
        font-size: 15px;
        font-weight: 700;
        min-height: 52px;
    }
    
    .search-input {
        padding: 12px 12px 12px 40px;
        font-size: 14px;
    }
    
    .search-icon {
        left: 12px;
        width: 16px;
        height: 16px;
    }
    
    .table-container {
        margin-bottom: 16px;
    }
    
    .request-maintenance-table {
        min-width: 600px;
    }
    
    .table-header th,
    .table-row td {
        padding: 8px 6px;
        font-size: 12px;
    }
    
    .table-header th {
        font-size: 11px;
        padding: 10px 6px;
    }
    
    .status-badge,
    .priority-badge {
        padding: 3px 6px;
        font-size: 10px;
    }
    
    .btn-view {
        padding: 5px 8px;
        font-size: 10px;
    }
    
    .memo-text {
        max-width: 120px;
    }
    
    .empty-state {
        padding: 32px 16px;
    }
    
    .empty-content h3 {
        font-size: 16px;
    }
    
    .empty-content p {
        font-size: 13px;
    }
}

@media (max-width: 480px) {
    .request-maintenance-container {
        padding: 4px;
    }
    
    .request-maintenance-header {
        padding: 8px;
        margin-bottom: 8px;
    }
    
    .page-title h1 {
        font-size: 18px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .action-bar {
        padding: 6px 8px;
    }
    
    .btn-create {
        padding: 16px 12px;
        font-size: 16px;
        font-weight: 700;
        min-height: 56px;
    }
    
    .search-input {
        padding: 14px 14px 14px 44px;
        font-size: 15px;
        min-height: 48px;
    }
    
    .search-icon {
        left: 14px;
        width: 18px;
        height: 18px;
    }
    
    .table-container {
        margin-bottom: 12px;
    }
    
    .request-maintenance-table {
        min-width: 500px;
    }
    
    .table-header th,
    .table-row td {
        padding: 6px 4px;
        font-size: 11px;
    }
    
    .table-header th {
        font-size: 10px;
        padding: 8px 4px;
    }
    
    .status-badge,
    .priority-badge {
        padding: 2px 4px;
        font-size: 9px;
    }
    
    .btn-view {
        padding: 4px 6px;
        font-size: 9px;
    }
    
    .memo-text {
        max-width: 100px;
    }
    
    .empty-state {
        padding: 24px 12px;
    }
    
    .empty-content h3 {
        font-size: 14px;
    }
    
    .empty-content p {
        font-size: 12px;
    }
}

/* ===== TABLET LANDSCAPE OPTIMIZATION ===== */
@media (max-width: 1024px) and (orientation: landscape) {
    .request-maintenance-container {
        padding: 16px;
    }
    
    .action-content {
        flex-direction: row;
        align-items: center;
    }
    
    .search-section {
        max-width: 300px;
    }
    
    .action-buttons {
        flex-wrap: wrap;
        gap: 6px;
    }
    
    .btn-create {
        padding: 8px 12px;
        font-size: 12px;
    }
}

/* ===== MOBILE LANDSCAPE OPTIMIZATION ===== */
@media (max-width: 896px) and (orientation: landscape) {
    .request-maintenance-header {
        padding: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .action-bar {
        padding: 8px 12px;
    }
    
    .btn-create {
        padding: 10px 12px;
        font-size: 13px;
    }
}

/* ===== TABLE RESPONSIVE IMPROVEMENTS ===== */
@media (max-width: 768px) {
    .request-maintenance-table {
        font-size: 12px;
    }
    
    .table-header th:first-child,
    .table-row td:first-child {
        position: sticky;
        left: 0;
        background: #f8fafc;
        z-index: 10;
        border-right: 1px solid #e2e8f0;
    }
    
    .table-header th:first-child {
        background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    }
    
    .wo-number {
        font-weight: 600;
        color: #0055a5;
        font-size: 12px;
    }
}

/* ===== ACCESSIBILITY IMPROVEMENTS ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        transition: none !important;
        animation: none !important;
    }
}

/* ===== HIGH DPI DISPLAYS ===== */
@media (-webkit-min-device-pixel-ratio: 2), (min-resolution: 192dpi) {
    .search-icon,
    .btn-create svg,
    .btn-view svg {
        image-rendering: -webkit-optimize-contrast;
        image-rendering: crisp-edges;
    }
}

/* ===== FOCUS STATES ===== */
.search-input:focus,
.btn-create:focus,
.btn-view:focus {
    outline: 2px solid rgba(0, 85, 165, 0.5);
    outline-offset: 2px;
}

/* ===== SMOOTH ANIMATIONS ===== */
* {
    transition: all 0.2s ease;
}

/* ===== ADDITIONAL WORKORDER-STYLE IMPROVEMENTS ===== */
.request-maintenance-container {
    background: #f8fafc;
}

.request-maintenance-content {
    border-radius: 0;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
}

.table-header th {
    font-weight: 600;
    color: #374151;
    border-bottom: 2px solid #e5e7eb;
}

.action-bar {
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
    padding: 20px 24px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: white;
    transition: all 0.2s ease;
}

.action-buttons button {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    background: white;
    color: #374151;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.action-buttons button:hover {
    background: #f3f4f6;
    border-color: #9ca3af;
    transform: translateY(-1px);
}

.wo-number {
    font-weight: 600;
    color: #0055a5;
}

/* ===== HOVER EFFECTS ===== */
.btn-create:active,
.btn-view:active {
    transform: translateY(0);
}

/* ===== LOADING STATES ===== */
.btn-create:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none;
}

.btn-create:disabled:hover {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    border-color: #059669;
    transform: none;
}

/* ===== ACTION DROPDOWN ===== */
.request-dropdown {
    position: relative;
    display: inline-block;
}

.request-dropdown .action-btn {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 8px 16px;
    background: var(--success-color);
    color: white;
    border: none;
    border-radius: var(--radius-sm);
    font-size: 12px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
}

.request-dropdown .action-btn:hover {
    background: rgba(0, 85, 165, 0.65);
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
    color: white;
}

.request-dropdown .action-btn svg {
    transition: transform 0.2s ease;
}

.request-dropdown.show .action-btn svg {
    transform: rotate(180deg);
}

.request-dropdown .dropdown-menu {
    position: absolute;
    top: calc(100% + 6px);
    left: 110px;
    background: white;
    border: 1px solid #e2e8f0;
    border-radius: 8px;
    box-shadow: 0 10px 30px rgba(15, 23, 42, 0.18);
    min-width: 200px;
    z-index: 1000;
    overflow: hidden;
    display: none;
    padding: 8px 0;
    list-style: none;
    margin: 0;
}

.request-dropdown .dropdown-menu.show {
    display: block;
}

.request-dropdown .dropdown-menu li {
    list-style: none;
}

.request-dropdown .dropdown-item {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    color: var(--text-primary);
    text-decoration: none;
    font-size: 13px;
    font-weight: 500;
    transition: var(--transition-base);
    background: none;
    border: none;
    width: 100%;
    text-align: left;
}

.request-dropdown .dropdown-item svg {
    color: #64748b;
}

.request-dropdown .dropdown-item:hover {
    background: #f8fafc;
    color: #0055a5;
}

.request-dropdown .dropdown-item.danger {
    color: #dc2626;
}

.request-dropdown .dropdown-item.danger svg {
    color: #dc2626;
}

.request-dropdown .dropdown-item.danger:hover {
    background: #fef2f2;
    color: #b91c1c;
}

.request-dropdown .dropdown-divider {
    height: 1px;
    margin: 8px 0;
    background: #f1f5f9;
}

/* ===== PAGINATION ===== */
.pagination-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 24px;
    background: white;
    border-top: 1px solid #e5e7eb;
}

.pagination-info {
    font-size: 14px;
    color: #6b7280;
}

.pagination-buttons {
    display: flex;
    gap: 5px;
    align-items: center;
}

.pagination-btn {
    min-width: 36px;
    height: 36px;
    padding: 0 12px;
    border: 1px solid #d1d5db;
    background: white;
    color: #374151;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s ease;
    font-size: 14px;
    font-weight: 500;
    display: flex;
    align-items: center;
    justify-content: center;
}

.pagination-btn:hover:not(.disabled) {
    background: #f9fafb;
    border-color: #0055a5;
    color: #0055a5;
}

.pagination-btn.active {
    background: #0055a5;
    color: white;
    border-color: #0055a5;
}

.pagination-btn.disabled {
    opacity: 0.5;
    cursor: not-allowed;
    background: #f9fafb;
}

.pagination-ellipsis {
    padding: 0 8px;
    color: #9ca3af;
    font-size: 14px;
}

/* ===== RESPONSIVE PAGINATION ===== */
@media (max-width: 768px) {
    .pagination-container {
        flex-direction: column;
        gap: 15px;
        padding: 15px;
    }
    
    .pagination-info {
        font-size: 13px;
    }
    
    .pagination-btn {
        min-width: 32px;
        height: 32px;
        padding: 0 10px;
        font-size: 13px;
    }
}

@media (max-width: 480px) {
    .pagination-buttons {
        flex-wrap: wrap;
        justify-content: center;
    }
    
    .pagination-btn {
        min-width: 28px;
        height: 28px;
        padding: 0 8px;
        font-size: 12px;
    }
}

/* ===== MODAL STYLES ===== */
body.modal-open {
    overflow: hidden;
}

.custom-modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 60px;
    width: 100vw;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    overflow-y: auto;
    justify-content: center;
    align-items: center;
    padding: 30px 16px;
    box-sizing: border-box;
}

.custom-modal-content {
    background-color: #ffffff;
    margin: 0;
    padding: 0;
    border: 1px solid #e5e7eb;
    border-radius: 12px;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2);
    animation: modalSlideIn 0.3s ease;
    position: relative;
    max-height: calc(100vh - 60px);
    display: flex;
    flex-direction: column;
}

@keyframes modalSlideIn {
    from {
        opacity: 0;
        transform: scale(0.95) translateY(-20px);
    }
    to {
        opacity: 1;
        transform: scale(1) translateY(0);
    }
}

.custom-modal-header {
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
    color: white;
    padding: 20px 24px;
    border-radius: 12px 12px 0 0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.custom-modal-header h3 {
    margin: 0;
    font-size: 18px;
    font-weight: 600;
}

.close-btn {
    color: white;
    font-size: 28px;
    font-weight: bold;
    background: none;
    border: none;
    cursor: pointer;
    padding: 0;
    line-height: 1;
    transition: all 0.2s ease;
}

.close-btn:hover {
    opacity: 0.7;
    transform: scale(1.1);
}

.custom-modal-body {
    padding: 24px;
    flex: 1;
    overflow-y: auto;
}

.modal-footer {
    padding: 16px 24px;
    background-color: #f9fafb;
    border-top: 1px solid #e5e7eb;
    display: flex;
    justify-content: flex-end;
    align-items: center;
    gap: 12px;
    border-radius: 0 0 12px 12px;
}

.modal-footer .btn {
    padding: 10px 20px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 500;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
}

.modal-footer .btn-primary {
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
    color: white;
}

.modal-footer .btn-primary:hover {
    background: linear-gradient(135deg, #003f7d 0%, #002a5c 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.modal-footer .btn-secondary {
    background: #f3f4f6;
    color: #374151;
    border: 1px solid #d1d5db;
}

.modal-footer .btn-secondary:hover {
    background: #e5e7eb;
    border-color: #9ca3af;
}

.modal-footer .btn-danger {
    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
    color: white;
}

.modal-footer .btn-danger:hover {
    background: linear-gradient(135deg, #c82333 0%, #a71d2a 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(220, 53, 69, 0.3);
}

.form-label {
    display: block;
    font-size: 14px;
    font-weight: 600;
    color: #374151;
    margin-bottom: 8px;
}

.form-input {
    width: 100%;
    padding: 12px;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    font-size: 14px;
    font-family: 'Inter', sans-serif;
    transition: all 0.2s ease;
}

.form-input:focus {
    outline: none;
    border-color: #0055a5;
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

/* ===== RESPONSIVE MODAL ===== */
@media (max-width: 768px) {
    .custom-modal-content {
        width: 95%;
        margin: 20px auto;
    }
    
    .custom-modal-header {
        padding: 16px 20px;
    }
    
    .custom-modal-header h3 {
        font-size: 16px;
    }
    
    .custom-modal-body {
        padding: 20px;
    }
    
    .modal-footer {
        padding: 12px 20px;
        flex-direction: column-reverse;
    }
    
    .modal-footer .btn {
        width: 100%;
    }
}

@media (max-width: 480px) {
    .custom-modal-content {
        width: 98%;
        margin: 10px auto;
    }
    
    .custom-modal-header {
        padding: 12px 16px;
    }
    
    .custom-modal-body {
        padding: 16px;
    }
}

</style>
<?php $__env->stopSection(); ?>
<?php echo $__env->make('welcome', array_diff_key(get_defined_vars(), ['__data' => 1, '__path' => 1]))->render(); ?><?php /**PATH /home/titandrillingzm/public_html/admin.titandrillingzm.com/resources/views/maintenance/Request_Maintenance.blade.php ENDPATH**/ ?>