<?php $__env->startSection('title', 'New Fuel Entry'); ?>

<?php $__env->startSection('styles'); ?>
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<?php $__env->stopSection(); ?>

<?php $__env->startSection('content'); ?>
<div class="workorder-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="<?php echo e(route('fuel.logger')); ?>" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Fuel Log Entry</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" onclick="saveFuelLog()">
                    <i class="fas fa-plus"></i>
                    Create
                </button>
            </div>
        </div>
    </div>
    
    <!-- Main Content -->
    <div class="main-content">
        <div class="form-container">
            <!-- Request Number Section -->
            <div class="form-section">
                <div class="request-number-display">
                    <span class="request-number-label">Request #</span>
                    <span class="request-number-value" id="requestNumber">Will be generated on save</span>
                        </div>
                    </div>

            <!-- Basic Information Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-info-circle"></i>
                    Basic Information
                </h2>
                
                <form id="fuelEntryForm">
                    <?php echo csrf_field(); ?>
                    
                    <!-- Country Selection -->
                    <div class="form-group">
                        <label for="country" class="form-label">Country <span class="required">*</span></label>
                        <select id="country" name="country" class="form-input" required>
                            <option value="">Select Country</option>
                            <option value="Zambia">Zambia</option>
                            <option value="DRC">DRC</option>
                        </select>
                </div>
                
                    <!-- Project Selection -->
                    <div class="form-group">
                        <label for="project" class="form-label">Project <span class="required">*</span></label>
                        <select id="project" name="project" class="form-input" required disabled>
                            <option value="">Select Project</option>
                            <!-- Projects will be populated dynamically based on country selection -->
                        </select>
                    </div>
                </form>
                </div>
                
            <!-- Equipment Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-tools"></i>
                    Equipment
                </h2>
                
                <!-- Equipment Selection -->
                    <div class="form-group">
                    <label for="equipment" class="form-label">Equipment <span class="required">*</span></label>
                    <div class="equipment-select-container">
                        <div class="equipment-select" onclick="openEquipmentModal(event)">
                            <span id="equipmentText">Select Equipment</span>
                            <i class="fas fa-chevron-down"></i>
                    </div>
                        <input type="hidden" id="equipmentId" name="equipmentId">
                    </div>
                    <div id="selectedEquipmentInfo" class="selected-equipment-info" style="display: none;">
                        <div class="equipment-details">
                            <p><strong>Selected:</strong> <span id="selectedEquipmentName"></span></p>
                            <p><strong>Equipment No:</strong> <span id="selectedEquipmentNo"></span></p>
                            <p><strong>Meter Reading:</strong> <span id="selectedMeterReading"></span></p>
                        </div>
                    </div>
                </div>
                
                <!-- Equipment Number -->
                    <div class="form-group">
                    <label for="equipmentNo" class="form-label">Equipment Number <span class="required">*</span></label>
                    <input type="text" id="equipmentNo" name="equipmentNo" class="form-input" placeholder="Equipment Number" readonly>
                    </div>
                
                <!-- Equipment Name -->
                    <div class="form-group">
                    <label for="equipmentName" class="form-label">Equipment Name <span class="required">*</span></label>
                    <input type="text" id="equipmentName" name="equipmentName" class="form-input" placeholder="Equipment Name" readonly>
                </div>
                
                <!-- Meter Reading -->
                <div class="form-group">
                    <label for="meterReading" class="form-label">Meter Reading <span class="required">*</span></label>
                    <div class="input-with-select">
                        <input type="number" id="meterReading" name="meterReading" class="form-input" placeholder="Enter meter reading" required>
                        <select id="meterUnit" name="meterUnit" class="form-input">
                            <option value="Miles">Miles</option>
                            <option value="KM">KM</option>
                            <option value="Hour">Hour</option>
                        </select>
                    </div>
                    </div>
                </div>
                
            <!-- Fuel Information Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-gas-pump"></i>
                    Fuel Information
                </h2>
                
                <!-- Filling Source -->
                    <div class="form-group">
                    <label class="form-label">Filling Source <span class="required">*</span></label>
                        <div class="radio-group">
                            <label class="radio-label">
                            <input type="radio" name="fillingSource" value="Fuel Bunker" checked>
                                <span class="radio-custom"></span>
                            Fuel Bunker
                            </label>
                            <label class="radio-label">
                            <input type="radio" name="fillingSource" value="Fuel Truck">
                                <span class="radio-custom"></span>
                            Fuel Truck
                            </label>
                            <label class="radio-label">
                            <input type="radio" name="fillingSource" value="Out Station">
                                <span class="radio-custom"></span>
                            Out Station
                            </label>
                        </div>
                    </div>
                
                <!-- Select Bunker -->
                    <div class="form-group">
                    <label for="selectedBunker" class="form-label">Select Bunker <span class="required">*</span></label>
                    <select id="selectedBunker" name="selectedBunker" class="form-input" required>
                        <option value="">Select Bunker</option>
                        <option value="Fuel Bunker">Fuel Bunker</option>
                        <option value="Fuel Truck">Fuel Truck</option>
                        <option value="Out Station">Out Station</option>
                    </select>
                    </div>
                
                <!-- Fuel Type -->
                <div class="form-group">
                    <label for="fuelType" class="form-label">Fuel Type</label>
                    <select id="fuelType" name="fuelType" class="form-input">
                        <option value="">Select Fuel Type</option>
                        <option value="Diesel">Diesel</option>
                        <option value="Petrol">Petrol</option>
                        <option value="Gas">Gas</option>
                        <option value="CNG">CNG</option>
                        <option value="LPG">LPG</option>
                        <option value="AdBlue">AdBlue</option>
                    </select>
                </div>
                
                <!-- Fuel Quantity -->
                    <div class="form-group">
                    <label for="fuelQuantity" class="form-label">Fuel Quantity <span class="required">*</span></label>
                    <div class="input-with-select">
                        <input type="number" id="fuelQuantity" name="fuelQuantity" class="form-input" placeholder="Enter fuel quantity" required>
                        <select id="fuelUnit" name="fuelUnit" class="form-input">
                            <option value="Gallon">Gallon</option>
                            <option value="Liter">Liter</option>
                            <option value="KG">KG</option>
                        </select>
                    </div>
                </div>
                
                <!-- Fuel Cost -->
                    <div class="form-group">
                    <label for="fuelCost" class="form-label">Fuel Cost</label>
                    <input type="number" id="fuelCost" name="fuelCost" class="form-input" placeholder="Enter fuel cost" step="0.01">
                    </div>
                </div>
                
            <!-- Additional Information Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-clipboard-list"></i>
                    Additional Information
                </h2>
                
                <!-- Note -->
                    <div class="form-group">
                    <label for="note" class="form-label">Note</label>
                    <textarea id="note" name="note" class="form-input form-textarea" rows="3" placeholder="Enter note"></textarea>
                    </div>
                
                <!-- Receiver Name -->
                    <div class="form-group">
                    <label for="receiverName" class="form-label">Receiver Name</label>
                    <input type="text" id="receiverName" name="receiverName" class="form-input" placeholder="Enter receiver name">
                    </div>
                </div>
                
            <!-- Photo Upload Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-camera"></i>
                    Photos
                </h2>
                
                <div class="form-group">
                    <div class="photo-upload-container">
                        <button type="button" class="btn-upload" onclick="openPhotoModal('Receipt', event)">
                            <i class="fas fa-camera"></i>
                            Upload Receipt
                    </button>
                        <button type="button" class="btn-upload" onclick="openPhotoModal('Fuel Browser', event)">
                            <i class="fas fa-camera"></i>
                            Upload Fuel Browser
                        </button>
                </div>
                    <div id="photoPreview" class="photo-preview" style="display: none;">
                        <div id="receiptPreview" class="photo-item" style="display: none;">
                            <img id="receiptImage" src="" alt="Receipt">
                            <span>Receipt</span>
        </div>
                        <div id="bowserPreview" class="photo-item" style="display: none;">
                            <img id="bowserImage" src="" alt="Fuel Browser">
                            <span>Fuel Browser</span>
    </div>
</div>
            </div>
                    </div>
        </div>
    </div>
</div>

<!-- Equipment Selection Modal -->
<div id="equipmentModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Equipment</h3>
            <div style="display: flex; gap: 10px; align-items: center;">
                <button type="button" class="refresh-btn" onclick="loadEquipmentFromMongoDB()" title="Refresh equipment list" style="background: rgba(255,255,255,0.2); color: white; border: none; padding: 5px 10px; border-radius: 4px; cursor: pointer; font-size: 16px;">
                    <i class="fas fa-sync-alt"></i>
                </button>
                <button type="button" class="close-btn" onclick="closeEquipmentModal()">&times;</button>
            </div>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="equipmentSearch" class="search-input" placeholder="Search equipment..." onkeyup="filterEquipment()">
                <button type="button" class="search-btn" onclick="filterEquipment()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div class="equipment-list" id="equipmentList">
                <div class="loading-state">
                    <div class="loading-spinner"></div>
                    Loading equipment...
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Photo Upload Modal -->
<div id="photoModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Add Photo</h3>
            <button type="button" class="close-btn" onclick="closePhotoModal()">&times;</button>
            </div>
        <div class="custom-modal-body">
            <div class="photo-options">
                <button type="button" class="photo-option-btn" onclick="handleChooseFromGallery()">
                    <i class="fas fa-image"></i>
                    Choose from Gallery
                </button>
                <input type="file" id="photoFile" accept="image/*" style="display: none;" onchange="handleFileSelect(event)">
                    </div>
        </div>
    </div>
</div>

<!-- Success Modal -->
<div id="successModal" class="success-modal-overlay" style="display: none;">
    <div class="success-modal-container">
        <div class="success-modal-header">
            <h2 class="success-title">Success</h2>
        </div>
        <div class="success-modal-body">
            <p class="success-message" id="successMessage">Fuel log entry created successfully!</p>
            <div class="success-buttons">
                <button type="button" class="success-btn back-btn" onclick="goBackToFuelLogger()">
                    Back
                        </button>
                <button type="button" class="success-btn create-new-btn" onclick="createNewEntry()">
                    Create New Entry
                        </button>
                    </div>
            </div>
    </div>
</div>

<!-- Sync Modal -->
<div id="syncModal" class="sync-modal" style="display: none;">
    <div class="sync-modal-content">
        <h3 class="sync-modal-text">Please Wait</h3>
        <div class="sync-icon-container">
            <div class="sync-icon">🔄</div>
            <span class="sync-modal-subtext">Saving Fuel Log Entry</span>
        </div>
    </div>
</div>

<script>
// Global variables
let selectedEquipment = null;
let equipmentList = [];
let currentPhotoType = '';
let receiptImageUri = null;
let fuelBrowserImageUri = null;
let isSavingFuelLog = false;

// Project data based on country - Same as Request Maintenance
const projectData = {
    'Zambia': ['Kansanshi', 'Kalumbila', 'Kimteto', 'Kobold', 'FQM Exploration', 'Mimosa'],
    'DRC': ['Lubumbashi', 'Musompo', 'IME', 'Kamoa']
};

document.addEventListener('DOMContentLoaded', function() {
    initializeForm();
    setupEventListeners();
    loadEquipmentFromMongoDB();
});

function initializeForm() {
    const meterUnitSelect = document.getElementById('meterUnit');
    if (meterUnitSelect) {
        meterUnitSelect.value = 'Miles';
    }

    const fuelUnitSelect = document.getElementById('fuelUnit');
    if (fuelUnitSelect) {
        fuelUnitSelect.value = 'Gallon';
    }

    const fuelTypeSelect = document.getElementById('fuelType');
    if (fuelTypeSelect) {
        fuelTypeSelect.value = '';
    }

    const bunkerSelect = document.getElementById('selectedBunker');
    if (bunkerSelect) {
        bunkerSelect.value = '';
    }

    const projectSelect = document.getElementById('project');
    if (projectSelect) {
        projectSelect.innerHTML = '<option value="">Select Project</option>';
        projectSelect.disabled = true;
    }

    const countrySelect = document.getElementById('country');
    if (countrySelect) {
        countrySelect.value = '';
    }

    updateRequestPreview();
}

function setupEventListeners() {
    const countrySelect = document.getElementById('country');
    const projectSelect = document.getElementById('project');
    
    // Country change handler
    if (countrySelect) {
        countrySelect.addEventListener('change', function() {
            updateProjectItems(this.value);
            updateRequestPreview();
        });
    }
    
    // Project change handler
    if (projectSelect) {
        projectSelect.addEventListener('change', function() {
            updateRequestPreview();
        });
    }
    
    // Real-time validation
    const inputs = document.querySelectorAll('.form-input');
    if (inputs) {
        inputs.forEach(input => {
            input.addEventListener('blur', function() {
                validateField(this);
            });
            
            input.addEventListener('input', function() {
                // Clear error styling on input
                if (this.style.borderColor === 'rgb(220, 53, 69)') {
                    this.style.borderColor = '#ced4da';
                    this.style.boxShadow = 'none';
                }
            });
        });
    }
}

function updateProjectItems(country) {
    const projectSelect = document.getElementById('project');
    projectSelect.innerHTML = '<option value="">Select Project</option>';
    projectSelect.disabled = true;
    updateRequestPreview();
    
    console.log('Country changed to:', country);
    
    if (country && projectData[country]) {
        projectData[country].forEach(project => {
            const option = document.createElement('option');
            option.value = project;
            option.textContent = project;
            projectSelect.appendChild(option);
        });
        projectSelect.disabled = false;
        console.log(`✅ Loaded ${projectData[country].length} projects for ${country}`);
    }
}

function updateRequestPreview() {
    const requestNumberEl = document.getElementById('requestNumber');
    if (!requestNumberEl) return;

    const countrySelect = document.getElementById('country');
    const projectSelect = document.getElementById('project');
    const country = countrySelect ? countrySelect.value : '';
    const project = projectSelect ? projectSelect.value : '';

    if (!country || !project) {
        requestNumberEl.textContent = 'Will be generated on save';
        return;
    }

    let countryCode = '';
    if (country === 'Zambia') {
        countryCode = 'ZMB';
    } else if (country === 'DRC') {
        countryCode = 'DRC';
    } else {
        countryCode = country.substring(0, 3).toUpperCase();
    }

    const projectCode = project.substring(0, 3).toUpperCase();
    requestNumberEl.textContent = `${countryCode}-${projectCode}-FL-???`;
}

// Equipment modal functions
function openEquipmentModal(event) {
    const modal = document.getElementById('equipmentModal');
    if (!modal) {
        console.error('Equipment modal not found');
        return;
    }
    
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    // Reload equipment if list is empty
    if (equipmentList.length === 0) {
        loadEquipmentFromMongoDB();
    } else {
        updateEquipmentDisplay();
    }
    
    // Position modal content with proper scrolling
    setTimeout(() => {
        const modalContent = modal.querySelector('.custom-modal-content');
        if (modalContent && event) {
            const clickY = event.clientY;
            const scrollY = window.scrollY;
            const viewportHeight = window.innerHeight;
            const contentHeight = modalContent.offsetHeight;
            
            let topPosition = scrollY + clickY - (contentHeight / 2);
            const minTop = scrollY + 20;
            const maxTop = scrollY + viewportHeight - contentHeight - 20;
            
            if (topPosition < minTop) topPosition = minTop;
            if (topPosition > maxTop) topPosition = maxTop;
            
            modalContent.style.top = `${topPosition}px`;
            
            // Ensure modal is visible by scrolling to it if needed
            const modalRect = modalContent.getBoundingClientRect();
            if (modalRect.top < 0 || modalRect.bottom > viewportHeight) {
                modalContent.scrollIntoView({ 
                    behavior: 'smooth', 
                    block: 'center',
                    inline: 'center'
                });
            }
        } else if (modalContent) {
            const scrollY = window.scrollY;
            const viewportHeight = window.innerHeight;
            const contentHeight = modalContent.offsetHeight;
            const topPosition = scrollY + (viewportHeight - contentHeight) / 2;
            modalContent.style.top = `${topPosition}px`;
            
            // Center the modal in viewport
            modalContent.scrollIntoView({ 
                behavior: 'smooth', 
                block: 'center',
                inline: 'center'
            });
        }
        
        const searchInput = document.getElementById('equipmentSearch');
        if (searchInput) {
            searchInput.focus();
        }
    }, 10);
}

function closeEquipmentModal() {
    const modal = document.getElementById('equipmentModal');
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function updateEquipmentDisplay() {
    const equipmentListEl = document.getElementById('equipmentList');
    if (!equipmentListEl) {
        console.error('Equipment list element not found');
        return;
    }
    
    if (!equipmentList || equipmentList.length === 0) {
        equipmentListEl.innerHTML = '<div style="text-align: center; padding: 40px; color: #888;"><i class="fas fa-box-open" style="font-size: 48px; color: #ddd; margin-bottom: 15px; display: block;"></i><p>No equipment found. Please check MongoDB connection.</p></div>';
        return;
    }
    
    console.log(`Rendering ${equipmentList.length} equipment items`);
    
    equipmentListEl.innerHTML = equipmentList.map((equipment, index) => {
        // Use equipment name or model as display name
        const displayName = equipment.name !== 'Unknown Equipment' && equipment.name && equipment.name.trim() !== '' 
            ? equipment.name 
            : equipment.model;
        
        return `
        <div class="equipment-item" onclick='selectEquipment(${JSON.stringify(equipment).replace(/'/g, "&apos;")})'>
            <div class="equipment-header">
                <div class="equipment-name">${displayName}</div>
                <div class="equipment-status ${equipment.status.toLowerCase().replace(/\s+/g, '-')}">${equipment.status}</div>
            </div>
            <div class="equipment-details">
                <div class="equipment-row">
                    <span class="equipment-label">Category:</span>
                    <span class="equipment-value">${equipment.category}</span>
                </div>
                <div class="equipment-row">
                    <span class="equipment-label">Model:</span>
                    <span class="equipment-value">${equipment.model}</span>
                </div>
                <div class="equipment-row">
                    <span class="equipment-label">Registration:</span>
                    <span class="equipment-value">${equipment.registration}</span>
                </div>
                <div class="equipment-row">
                    <span class="equipment-label">Meter Reading:</span>
                    <span class="equipment-value">${equipment.meterReading} ${equipment.meterUnit}</span>
                </div>
            </div>
        </div>
        `;
    }).join('');
    
    console.log('✅ Equipment list rendered successfully');
}

function filterEquipment() {
    const searchTerm = document.getElementById('equipmentSearch').value.toLowerCase();
    
    if (!searchTerm.trim()) {
        updateEquipmentDisplay();
        return;
    }
    
    const filteredEquipment = equipmentList.filter(equipment => {
        const displayName = equipment.name !== 'Unknown Equipment' && equipment.name && equipment.name.trim() !== '' 
            ? equipment.name 
            : equipment.model;
        
        return displayName.toLowerCase().includes(searchTerm) ||
               equipment.category.toLowerCase().includes(searchTerm) ||
               equipment.model.toLowerCase().includes(searchTerm) ||
               equipment.registration.toLowerCase().includes(searchTerm) ||
               equipment.type.toLowerCase().includes(searchTerm) ||
               equipment.location.toLowerCase().includes(searchTerm);
    });
    
    const equipmentListEl = document.getElementById('equipmentList');
    if (filteredEquipment.length === 0) {
        equipmentListEl.innerHTML = '<div style="text-align: center; padding: 40px; color: #888;"><i class="fas fa-search" style="font-size: 48px; color: #ddd; margin-bottom: 15px; display: block;"></i><p>No equipment matches your search.</p></div>';
        return;
    }
    
    equipmentListEl.innerHTML = filteredEquipment.map((equipment, index) => {
        const displayName = equipment.name !== 'Unknown Equipment' && equipment.name && equipment.name.trim() !== '' 
            ? equipment.name 
            : equipment.model;
        
        return `
        <div class="equipment-item" onclick='selectEquipment(${JSON.stringify(equipment).replace(/'/g, "&apos;")})'>
            <div class="equipment-header">
                <div class="equipment-name">${displayName}</div>
                <div class="equipment-status ${equipment.status.toLowerCase().replace(/\s+/g, '-')}">${equipment.status}</div>
            </div>
            <div class="equipment-details">
                <div class="equipment-row">
                    <span class="equipment-label">Category:</span>
                    <span class="equipment-value">${equipment.category}</span>
                </div>
                <div class="equipment-row">
                    <span class="equipment-label">Model:</span>
                    <span class="equipment-value">${equipment.model}</span>
                </div>
                <div class="equipment-row">
                    <span class="equipment-label">Registration:</span>
                    <span class="equipment-value">${equipment.registration}</span>
                </div>
                <div class="equipment-row">
                    <span class="equipment-label">Meter:</span>
                    <span class="equipment-value">${equipment.meterReading} ${equipment.meterUnit}</span>
                </div>
            </div>
        </div>
        `;
    }).join('');
}

function selectEquipment(equipment) {
    selectedEquipment = equipment;
    
    console.log('Equipment selected:', selectedEquipment);
    
    // Use equipment name or model as display name
    const displayName = equipment.name !== 'Unknown Equipment' && equipment.name && equipment.name.trim() !== '' 
        ? equipment.name 
        : equipment.model;
    
    // Update the display
    document.getElementById('equipmentText').textContent = displayName;
    document.getElementById('equipmentId').value = equipment.id;
    document.getElementById('equipmentNo').value = equipment.registration || equipment.regNum || equipment.vehicleNumber || '';
    document.getElementById('equipmentName').value = displayName;
    document.getElementById('meterReading').value = equipment.meterReading || '';
    document.getElementById('meterUnit').value = equipment.meterUnit || 'Hours';
    
    // Update selected equipment info
    document.getElementById('selectedEquipmentName').textContent = displayName;
    document.getElementById('selectedEquipmentNo').textContent = equipment.registration || equipment.regNum || '';
    document.getElementById('selectedMeterReading').textContent = `${equipment.meterReading || '0'} ${equipment.meterUnit || 'Hours'}`;
    document.getElementById('selectedEquipmentInfo').style.display = 'block';
    
    // Close modal
    closeEquipmentModal();
}

// Load equipment from MongoDB - Using same endpoint as Create Request Maintenance
async function loadEquipmentFromMongoDB() {
    try {
        console.log('🔄 Loading equipment from MongoDB...');
        
        // Show loading state
        const equipmentListEl = document.getElementById('equipmentList');
        if (equipmentListEl) {
            equipmentListEl.innerHTML = '<div style="text-align: center; padding: 40px; color: #888;"><div class="loading-spinner" style="width: 30px; height: 30px; border: 3px solid #f3f3f3; border-top: 3px solid #0055a5; border-radius: 50%; animation: spin 1s linear infinite; margin: 0 auto 15px;"></div>Loading equipment...</div>';
        }
        
        // Fetch equipment data from MongoDB API - SAME ENDPOINT AS REQUEST MAINTENANCE
        const response = await fetch('https://api.titandrillingzm.com:6001/EquipmentsDataJSON/GlobalEquipmentJsonData');
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('✅ MongoDB equipment response:', result);
        console.log('Success:', result.success);
        console.log('Data count:', result.data?.length || 0);
        
        if (result.success && Array.isArray(result.data)) {
            console.log(`Found ${result.data.length} equipment items`);
            
            equipmentList = result.data.map((equipment, index) => ({
                id: equipment._id || equipment.id || index.toString(),
                name: equipment.vehicleNumber || equipment.equipmentName || equipment.name || equipment.model || 'Unknown Equipment',
                type: equipment.vehicleType || equipment.equipmentCategory || equipment.equipmentType || equipment.category || 'N/A',
                location: equipment.location || equipment.site || equipment.country || 'N/A',
                model: equipment.model || equipment.equipmentModel || 'N/A',
                meterReading: equipment.meterReading || equipment.odometerReading || '0',
                meterUnit: equipment.meterUnit || equipment.meterType || 'Hours',
                status: equipment.status || 'Active',
                manufacturer: equipment.make || equipment.manufacturer || 'N/A',
                year: equipment.year || equipment.yearOfManufacture || 'N/A',
                regNum: equipment.regNum || equipment.registrationNumber || equipment.vehicleNumber || 'N/A',
                vehicleNumber: equipment.vehicleNumber || equipment.regNum || 'N/A',
                registration: equipment.regNum || equipment.registrationNumber || 'N/A',
                serialNumber: equipment.vin || equipment.serialNumber || equipment.vinNumber || 'N/A',
                engineNumber: equipment.engineNumber || 'N/A',
                category: equipment.mainCategory || equipment.category || 'N/A',
                country: equipment.country || 'N/A',
                project: equipment.project || 'N/A'
            }));
            
            console.log(`✅ Mapped ${equipmentList.length} equipment items`);
            console.log('Sample equipment:', equipmentList[0]);
            updateEquipmentDisplay();
        } else {
            console.error('Failed to load equipment data from MongoDB:', result);
            equipmentList = [];
            updateEquipmentDisplay();
        }
    } catch (error) {
        console.error('❌ Error loading equipment from MongoDB:', error);
        console.error('Error details:', error.message);
        equipmentList = [];
        const equipmentListEl = document.getElementById('equipmentList');
        if (equipmentListEl) {
            equipmentListEl.innerHTML = `<div style="text-align: center; padding: 40px; color: #dc3545;">
                <i class="fas fa-exclamation-triangle" style="font-size: 48px; margin-bottom: 15px; display: block;"></i>
                <p style="font-weight: 600; margin-bottom: 10px;">Error loading equipment</p>
                <p style="font-size: 12px; color: #6c757d; margin-bottom: 15px;">${error.message}</p>
                <button onclick="loadEquipmentFromMongoDB()" style="padding: 10px 20px; background: #0055a5; color: white; border: none; border-radius: 4px; cursor: pointer; font-size: 14px;">
                    <i class="fas fa-sync-alt"></i> Retry
                </button>
            </div>`;
        }
    }
}

// Photo modal functions
function openPhotoModal(type, event) {
    currentPhotoType = type;
    const modal = document.getElementById('photoModal');
    modal.style.display = 'block';
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    // Position modal content with proper scrolling
    setTimeout(() => {
        const modalContent = modal.querySelector('.custom-modal-content');
        if (modalContent && event) {
            const clickY = event.clientY;
            const scrollY = window.scrollY;
            const viewportHeight = window.innerHeight;
            const contentHeight = modalContent.offsetHeight;
            
            let topPosition = scrollY + clickY - (contentHeight / 2);
            const minTop = scrollY + 20;
            const maxTop = scrollY + viewportHeight - contentHeight - 20;
            
            if (topPosition < minTop) topPosition = minTop;
            if (topPosition > maxTop) topPosition = maxTop;
            
            modalContent.style.top = `${topPosition}px`;
            
            // Ensure modal is visible by scrolling to it if needed
            const modalRect = modalContent.getBoundingClientRect();
            if (modalRect.top < 0 || modalRect.bottom > viewportHeight) {
                modalContent.scrollIntoView({ 
                    behavior: 'smooth', 
                    block: 'center',
                    inline: 'center'
                });
            }
        } else if (modalContent) {
            const scrollY = window.scrollY;
            const viewportHeight = window.innerHeight;
            const contentHeight = modalContent.offsetHeight;
            const topPosition = scrollY + (viewportHeight - contentHeight) / 2;
            modalContent.style.top = `${topPosition}px`;
            
            // Center the modal in viewport
            modalContent.scrollIntoView({ 
                behavior: 'smooth', 
                block: 'center',
                inline: 'center'
            });
        }
    }, 10);
}

function closePhotoModal() {
    const modal = document.getElementById('photoModal');
    modal.style.display = 'none';
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function handleChooseFromGallery() {
    document.getElementById('photoFile').click();
}

function handleFileSelect(event) {
    const file = event.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            if (currentPhotoType === 'Receipt') {
                receiptImageUri = e.target.result;
                document.getElementById('receiptImage').src = e.target.result;
                document.getElementById('receiptPreview').style.display = 'block';
            } else if (currentPhotoType === 'Fuel Browser') {
                fuelBrowserImageUri = e.target.result;
                document.getElementById('bowserImage').src = e.target.result;
                document.getElementById('bowserPreview').style.display = 'block';
            }
            document.getElementById('photoPreview').style.display = 'flex';
            closePhotoModal();
            showNotification(`${currentPhotoType} image selected successfully!`, 'success');
        };
        reader.readAsDataURL(file);
    }
}

// Validation function for required fields
function validateRequiredFields() {
    const errors = [];
    
    // Required field mappings
    const requiredFields = [
        { id: 'country', name: 'Country', element: document.getElementById('country') },
        { id: 'project', name: 'Project', element: document.getElementById('project') },
        { id: 'equipmentId', name: 'Equipment', element: document.getElementById('equipmentId') },
        { id: 'equipmentNo', name: 'Equipment Number', element: document.getElementById('equipmentNo') },
        { id: 'equipmentName', name: 'Equipment Name', element: document.getElementById('equipmentName') },
        { id: 'meterReading', name: 'Meter Reading', element: document.getElementById('meterReading') },
        { id: 'fillingSource', name: 'Filling Source', element: document.querySelector('input[name="fillingSource"]:checked') },
        { id: 'selectedBunker', name: 'Select Bunker', element: document.getElementById('selectedBunker') },
        { id: 'fuelQuantity', name: 'Fuel Quantity', element: document.getElementById('fuelQuantity') }
    ];
    
    // Check each required field
    requiredFields.forEach(field => {
        if (!field.element || !field.element.value.trim()) {
            errors.push(`${field.name} is required`);
            // Highlight the field in red
            if (field.element) {
                field.element.style.borderColor = '#dc3545';
                field.element.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
            }
        } else {
            // Reset field styling if valid
            if (field.element) {
                field.element.style.borderColor = '#ced4da';
                field.element.style.boxShadow = 'none';
            }
        }
    });
    
    // Check if equipment is selected
    if (!selectedEquipment || !document.getElementById('equipmentId').value) {
        errors.push('Please select an equipment from the equipment list');
        // Highlight equipment select area
        const equipmentSelect = document.querySelector('.equipment-select');
        if (equipmentSelect) {
            equipmentSelect.style.borderColor = '#dc3545';
            equipmentSelect.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
        }
    } else {
        const equipmentSelect = document.querySelector('.equipment-select');
        if (equipmentSelect) {
            equipmentSelect.style.borderColor = '#ced4da';
            equipmentSelect.style.boxShadow = 'none';
        }
    }
    
    return {
        isValid: errors.length === 0,
        message: errors.length > 0 ? errors.join('. ') + '.' : 'All required fields are valid'
    };
}

// Real-time field validation function
function validateField(field) {
    const fieldId = field.id;
    const fieldValue = field.value.trim();
    
    // Reset styling
    field.style.borderColor = '#ced4da';
    field.style.boxShadow = 'none';
    
    // Required field validation
    const requiredFields = ['country', 'project', 'equipmentId', 'equipmentNo', 'equipmentName', 'meterReading', 'selectedBunker', 'fuelQuantity'];
    
    if (requiredFields.includes(fieldId) && !fieldValue) {
        field.style.borderColor = '#dc3545';
        field.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
        return false;
    }
    
    // Numeric validation for reading and quantity fields
    if ((fieldId === 'meterReading' || fieldId === 'fuelQuantity' || fieldId === 'fuelCost') && fieldValue) {
        const numValue = parseFloat(fieldValue);
        if (isNaN(numValue) || numValue < 0) {
            field.style.borderColor = '#dc3545';
            field.style.boxShadow = '0 0 0 3px rgba(220, 53, 69, 0.1)';
            return false;
        }
    }
    
    return true;
}

// Save fuel log function
async function saveFuelLog() {
    if (isSavingFuelLog) {
        return;
    }

    const validationResult = validateRequiredFields();
    if (!validationResult.isValid) {
        showNotification(validationResult.message, 'error');
        return;
    }

    const saveButton = document.querySelector('.create-btn');
    const syncModal = document.getElementById('syncModal');

    if (saveButton) {
        saveButton.disabled = true;
    }

    if (syncModal) {
        syncModal.style.display = 'flex';
    }

    isSavingFuelLog = true;

    try {
        const fillingSourceElement = document.querySelector('input[name="fillingSource"]:checked');
        if (!fillingSourceElement) {
            throw new Error('Please select a filling source');
        }

        const nowIso = new Date().toISOString();

        const payload = {
            adminUid: 'cahj3jJqe5Z16pQdLGSWrNgm84o1', // TODO: replace with logged-in user
            country: document.getElementById('country').value.trim(),
            project: document.getElementById('project').value.trim(),
            equipmentNo: document.getElementById('equipmentNo').value.trim(),
            equipmentName: document.getElementById('equipmentName').value.trim(),
            meterReading: parseFloat(document.getElementById('meterReading').value) || 0,
            meterUnit: document.getElementById('meterUnit').value,
            selectedSource: fillingSourceElement.value,
            selectedBunker: document.getElementById('selectedBunker').value,
            fuelQuantity: parseFloat(document.getElementById('fuelQuantity').value) || 0,
            fuelUnit: document.getElementById('fuelUnit').value,
            createdAt: nowIso,
            updatedAt: nowIso
        };

        const fuelTypeValue = document.getElementById('fuelType').value;
        if (fuelTypeValue) {
            payload.fuelType = fuelTypeValue;
        }

        const fuelCostValue = document.getElementById('fuelCost').value;
        if (fuelCostValue !== '') {
            payload.fuelCost = parseFloat(fuelCostValue) || 0;
        }

        const noteValue = document.getElementById('note').value.trim();
        if (noteValue) {
            payload.note = noteValue;
        }

        const receiverNameValue = document.getElementById('receiverName').value.trim();
        if (receiverNameValue) {
            payload.receiverName = receiverNameValue;
        }

        if (receiptImageUri) {
            payload.receiptImageUri = receiptImageUri;
        }

        if (fuelBrowserImageUri) {
            payload.fuelBrowserImageUri = fuelBrowserImageUri;
        }

        console.log('💾 Saving fuel log data:', payload);

        const response = await fetch('https://api.titandrillingzm.com:6011/fuel-logs', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            body: JSON.stringify(payload)
        });

        let responseBody = null;
        try {
            responseBody = await response.json();
        } catch (parseError) {
            console.warn('⚠️ Unable to parse response JSON:', parseError);
        }

        if (!response.ok) {
            const errorMessage = responseBody?.message || `Request failed with status ${response.status}`;
            throw new Error(errorMessage);
        }

        if (!responseBody?.success) {
            throw new Error(responseBody?.message || 'Failed to save fuel log');
        }

        const savedRequestNumber = responseBody?.data?.requestNumber;

        if (syncModal) {
            syncModal.style.display = 'none';
        }

        if (savedRequestNumber) {
            const requestNumberEl = document.getElementById('requestNumber');
            if (requestNumberEl) {
                requestNumberEl.textContent = savedRequestNumber;
            }
        }

        showSuccessModal(savedRequestNumber);
    } catch (error) {
        console.error('❌ Error saving data to MongoDB:', error);
        showNotification(error.message || 'Failed to save fuel log entry', 'error');
    } finally {
        if (syncModal) {
            syncModal.style.display = 'none';
        }
        if (saveButton) {
            saveButton.disabled = false;
        }
        isSavingFuelLog = false;
    }
}

// Success Modal Functions
function showSuccessModal(requestNumber) {
    const successModal = document.getElementById('successModal');
    const successMessageEl = document.getElementById('successMessage');

    if (successMessageEl) {
        if (requestNumber) {
            successMessageEl.innerHTML = `Fuel log entry <span class="success-request-number">${requestNumber}</span> saved successfully!`;
        } else {
            successMessageEl.textContent = 'Fuel log entry created successfully!';
        }
    }

    if (successModal) {
        successModal.style.display = 'flex';
        successModal.style.opacity = '0';
        successModal.style.transition = 'opacity 0.3s ease';
        
        // Fade in the success modal
        setTimeout(() => {
            successModal.style.opacity = '1';
        }, 10);
    }
}

function hideSuccessModal() {
    const successModal = document.getElementById('successModal');
    if (successModal) {
        successModal.style.display = 'none';
    }
}

function closeSuccessModal() {
    document.getElementById('successModal').style.display = 'none';
}

// Go back to Fuel Logger list
function goBackToFuelLogger() {
    closeSuccessModal();
    // Navigate to fuel logger page
    window.location.href = "<?php echo e(route('fuel.logger')); ?>";
}

// Create new entry
function createNewEntry() {
    closeSuccessModal();
    // Reset form for new entry
    document.getElementById('fuelEntryForm').reset();
    initializeForm();
    updateProjectItems('');
    updateRequestPreview();
    document.getElementById('selectedEquipmentInfo').style.display = 'none';
    selectedEquipment = null;
    receiptImageUri = null;
    fuelBrowserImageUri = null;
    document.getElementById('photoPreview').style.display = 'none';
    document.getElementById('receiptPreview').style.display = 'none';
    document.getElementById('bowserPreview').style.display = 'none';
    // Scroll to top
    window.scrollTo({ top: 0, behavior: 'smooth' });
    }
    
    // Enhanced notification system
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `fuel-notification fuel-notification-${type}`;
        notification.textContent = message;
        
        const colors = {
            success: 'linear-gradient(135deg, #10b981 0%, #059669 100%)',
            error: 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)',
            warning: 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)',
            info: 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)'
        };
        
        notification.style.cssText = `
            position: fixed;
            top: 80px;
            right: 20px;
            padding: 12px 20px;
            border-radius: 0;
            color: white;
            font-weight: 500;
            font-size: 14px;
            z-index: 9999;
            opacity: 0;
            transform: translateX(100%);
            transition: all 0.3s ease;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            background: ${colors[type] || colors.info};
        `;
        
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.style.opacity = '1';
            notification.style.transform = 'translateX(0)';
        }, 100);
        
        setTimeout(() => {
            notification.style.opacity = '0';
            notification.style.transform = 'translateX(100%)';
            setTimeout(() => {
                if (notification.parentNode) {
                    document.body.removeChild(notification);
                }
            }, 300);
        }, 3000);
    }
        
        // Make functions globally accessible
window.saveFuelLog = saveFuelLog;
window.openEquipmentModal = openEquipmentModal;
window.closeEquipmentModal = closeEquipmentModal;
window.filterEquipment = filterEquipment;
window.selectEquipment = selectEquipment;
window.loadEquipmentFromMongoDB = loadEquipmentFromMongoDB;
window.openPhotoModal = openPhotoModal;
window.closePhotoModal = closePhotoModal;
window.handleChooseFromGallery = handleChooseFromGallery;
window.handleFileSelect = handleFileSelect;
window.goBackToFuelLogger = goBackToFuelLogger;
window.createNewEntry = createNewEntry;
        window.showNotification = showNotification;
</script>

<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.workorder-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Form Styles */
.form-container {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

.form-section {
    padding: 30px;
    border-bottom: 1px solid var(--border-light);
}

.form-section:last-child {
    border-bottom: none;
}

.section-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.section-title i {
    color: var(--primary-color);
}

/* Form Groups */
.form-group {
    margin-bottom: 25px;
}

.form-group:last-child {
    margin-bottom: 0;
}

.form-label {
    display: block;
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.required {
    color: var(--danger-color);
    font-weight: bold;
}

.form-input {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
    transition: var(--transition-base);
    background: white;
}

.form-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.form-input.error {
    border-color: var(--danger-color);
}

.form-textarea {
    min-height: 100px;
    resize: vertical;
}

.error-message {
    color: var(--danger-color);
    font-size: 12px;
    margin-top: 5px;
    display: none;
}

.error-message.show {
    display: block;
}

/* Request Number Display */
.request-number-display {
    background: var(--bg-secondary);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 15px 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 25px;
}

.request-number-label {
    font-weight: 500;
    color: var(--text-primary);
}

.request-number-value {
    font-weight: 700;
    color: var(--primary-color);
    font-size: 16px;
}

/* Equipment Selection */
.equipment-select-container {
    position: relative;
}

.equipment-select {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    background: white;
    cursor: pointer;
    transition: var(--transition-base);
    min-height: 48px;
}

.equipment-select:hover {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.selected-equipment-info {
    margin-top: 10px;
    padding: 15px;
    background: var(--bg-secondary);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
}

.equipment-details p {
    margin: 5px 0;
    font-size: 14px;
    color: var(--text-secondary);
}

/* Input with Select */
.input-with-select {
    display: flex;
    gap: 10px;
}

.input-with-select .form-input {
    flex: 1;
}

/* Radio Group */
.radio-group {
    display: flex;
    gap: 20px;
    flex-wrap: wrap;
}

.radio-label {
    display: flex;
    align-items: center;
    gap: 8px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
}

.radio-label input[type="radio"] {
    display: none;
}

.radio-custom {
    width: 18px;
    height: 18px;
    border: 2px solid var(--border-color);
    border-radius: 50%;
    position: relative;
    transition: var(--transition-base);
}

.radio-label input[type="radio"]:checked + .radio-custom {
    border-color: var(--primary-color);
    background: var(--primary-color);
}

.radio-label input[type="radio"]:checked + .radio-custom::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 6px;
    height: 6px;
    background: white;
    border-radius: 50%;
}

/* Photo Upload */
.photo-upload-container {
    display: flex;
    gap: 15px;
    margin-top: 10px;
}

.photo-preview {
    display: flex;
    gap: 15px;
    margin-top: 15px;
}

.photo-item {
    text-align: center;
}

.photo-item img {
    width: 80px;
    height: 80px;
    object-fit: cover;
    border-radius: var(--radius-md);
    border: 2px solid var(--border-light);
}

.photo-item span {
    display: block;
    margin-top: 5px;
    font-size: 12px;
    color: var(--text-secondary);
}

/* Button Styles */
.btn-upload {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    background: var(--primary-color);
    color: white;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
}

.btn-upload:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

/* Body scroll lock when modal is open */
body.modal-open {
    overflow: hidden;
}

/* Modal Styles - Matching Create New Workorder */
.custom-modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    overflow-y: auto;
}

.custom-modal.show {
    display: block;
}

.custom-modal-content {
    background-color: #fefefe;
    margin: 0;
    padding: 0;
    border: 1px solid #888;
    width: 80%;
    max-width: 600px;
    border-radius: 8px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    position: absolute;
    left: 50%;
    transform: translateX(-50%);
    max-height: 80vh;
    overflow-y: auto;
    animation: modalSlideIn 0.3s ease;
}

@keyframes modalSlideIn {
    from {
        opacity: 0;
        transform: translateX(-50%) scale(0.9);
    }
    to {
        opacity: 1;
        transform: translateX(-50%) scale(1);
    }
}

.custom-modal-header {
    background-color: #0055A5;
    color: white;
    padding: 15px 20px;
    border-radius: 8px 8px 0 0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.custom-modal-header h3 {
    margin: 0;
    font-size: 18px;
    font-weight: 500;
}

.close-btn {
    color: white;
    font-size: 28px;
    font-weight: bold;
    background: none;
    border: none;
    cursor: pointer;
    padding: 0;
    line-height: 1;
}

.close-btn:hover {
    opacity: 0.7;
}

.custom-modal-body {
    padding: 20px;
    max-height: 60vh;
    overflow-y: auto;
}

/* Equipment Modal Styles */
.search-container {
    display: flex;
    gap: 10px;
    margin-bottom: 20px;
}

.search-input {
    flex: 1;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
}

.search-btn {
    padding: 12px 16px;
    background: var(--primary-color);
    color: white;
    border: none;
    border-radius: var(--radius-md);
    cursor: pointer;
    transition: background-color 0.2s;
}

.search-btn:hover {
    background: var(--primary-hover);
}

.equipment-list {
    max-height: 400px;
    overflow-y: auto;
}

.equipment-item {
    padding: 15px;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    margin-bottom: 10px;
    cursor: pointer;
    transition: var(--transition-base);
}

.equipment-item:hover {
    border-color: var(--primary-color);
    background: var(--bg-secondary);
}

.equipment-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 10px;
}

.equipment-name {
    font-weight: 600;
    color: var(--primary-color);
    font-size: 16px;
}

.equipment-status {
    padding: 4px 8px;
    border-radius: var(--radius-sm);
    font-size: 12px;
    font-weight: 500;
}

.equipment-status.available {
    background: #d4edda;
    color: #155724;
}

.equipment-status.inuse {
    background: #fff3cd;
    color: #856404;
}

.equipment-status.maintenance {
    background: #f8d7da;
    color: #721c24;
}

.equipment-row {
    display: flex;
    justify-content: space-between;
    margin-bottom: 5px;
}

.equipment-label {
    font-weight: 500;
    color: var(--text-secondary);
}

.equipment-value {
    color: var(--text-primary);
}

.loading-state {
    text-align: center;
    padding: 40px;
    color: var(--text-secondary);
}

.loading-spinner {
    width: 20px;
    height: 20px;
    border: 2px solid #f3f3f3;
    border-top: 2px solid var(--primary-color);
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin: 0 auto 10px;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.no-data {
    text-align: center;
    padding: 40px;
    color: var(--text-secondary);
    font-style: italic;
}

/* Photo Modal Styles */
.photo-options {
    display: flex;
    flex-direction: column;
    gap: 15px;
}

.photo-option-btn {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 15px 20px;
    background: var(--bg-secondary);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    cursor: pointer;
    transition: var(--transition-base);
    font-size: 16px;
    color: var(--text-primary);
}

.photo-option-btn:hover {
    background: var(--border-light);
    border-color: var(--primary-color);
}

/* Success Modal Styles */
.success-modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10000;
}

.success-modal-container {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
    max-width: 500px;
    width: 90%;
    max-height: 80vh;
    overflow: hidden;
    position: relative;
}

.success-modal-header {
    background: var(--primary-color);
    color: white;
    padding: 20px 24px;
    text-align: center;
}

.success-title {
    margin: 0;
    font-size: 24px;
    font-weight: 600;
}

.success-modal-body {
    padding: 30px 24px;
    text-align: center;
}

.success-message {
    font-size: 18px;
    color: var(--text-primary);
    margin-bottom: 30px;
}

.success-request-number {
    color: var(--primary-color);
    font-weight: 700;
    margin-left: 4px;
}

.success-buttons {
    display: flex;
    gap: 15px;
    justify-content: center;
}

.success-btn {
    padding: 12px 24px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    color: white;
    background: var(--primary-color);
}

.success-btn:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
}

/* Sync Modal Styles */
.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10000;
}

.sync-modal-content {
    background: white;
    padding: 40px;
    border-radius: var(--radius-lg);
    text-align: center;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
}

.sync-modal-text {
    font-size: 20px;
    font-weight: 600;
    color: var(--primary-color);
    margin-bottom: 10px;
}

.sync-icon-container {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 10px;
}

.sync-icon {
    font-size: 24px;
    animation: spin 1s linear infinite;
}

.sync-modal-subtext {
    color: var(--text-secondary);
    font-size: 14px;
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .main-content {
        padding: 20px 15px;
    }
    
    .form-section {
        padding: 20px;
    }
    
    .photo-upload-container {
        flex-direction: column;
    }
    
    .success-buttons {
        flex-direction: column;
    }
    
    .success-btn {
        width: 100%;
    }
    
    .custom-modal-content {
        width: 95%;
        max-width: 95%;
        margin: 10px;
    }
    
    .equipment-list {
        max-height: 300px;
    }
    
    .custom-modal-header {
        padding: 12px 15px;
    }
    
    .custom-modal-body {
        padding: 15px;
        max-height: 50vh;
    }
}

@media (max-width: 576px) {
    .header-title {
        font-size: 20px;
    }
    
    .main-content {
        padding: 15px 10px;
    }
    
    .form-section {
        padding: 15px;
    }
    
    .request-number-display {
        flex-direction: column;
        gap: 10px;
        text-align: center;
    }
    
    .equipment-select {
        padding: 10px 12px;
    }
    
    .custom-modal-content {
        width: 98%;
        margin: 5px;
    }
    
    .custom-modal-header {
        padding: 15px 20px;
    }
    
    .custom-modal-body {
        padding: 20px;
    }
    
    .success-modal-container {
        width: 95%;
        margin: 10px;
    }
    
    .success-modal-header {
        padding: 15px 20px;
    }
    
    .success-title {
        font-size: 20px;
    }
    
    .success-modal-body {
        padding: 20px;
    }
    
    .success-message {
        font-size: 16px;
    }
}

/* Accessibility */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* Focus Styles */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid var(--primary-color);
    outline-offset: 2px;
}
</style>
<?php $__env->stopSection(); ?>

<?php echo $__env->make('welcome', array_diff_key(get_defined_vars(), ['__data' => 1, '__path' => 1]))->render(); ?><?php /**PATH /home/titandrillingzm/public_html/admin.titandrillingzm.com/resources/views/fuel_management/new-fuel-entry.blade.php ENDPATH**/ ?>