<?php $__env->startSection('title', 'Application Details'); ?>

<?php $__env->startSection('styles'); ?>
<meta name="csrf-token" content="<?php echo e(csrf_token()); ?>">
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<style>
:root {
    --primary-color: #0055a5;
    --success-color: #34C759;
    --danger-color: #FF3B30;
    --warning-color: #F57C00;
    --gray-color: #e0e0e0;
}

* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    background-color: #f5f6f8;
    color: #333;
}

.container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
}

.header-section {
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
    color: white;
    padding: 20px;
    margin-bottom: 30px;
    border-radius: 8px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    position: relative;
}

.header-logo {
    position: absolute;
    left: 50%;
    transform: translateX(-50%);
    height: 60px;
    width: auto;
}

.header-section h1 {
    flex: 1;
    text-align: right;
    font-size: 20px;
    font-weight: 600;
}

.back-button {
    background: rgba(255, 255, 255, 0.2);
    border: none;
    color: white;
    padding: 10px 20px;
    border-radius: 6px;
    cursor: pointer;
    font-size: 16px;
    display: flex;
    align-items: center;
    gap: 8px;
    transition: background 0.3s;
}

.back-button:hover {
    background: rgba(255, 255, 255, 0.3);
}

.detail-card {
    background: white;
    border-radius: 12px;
    padding: 30px;
    margin-bottom: 20px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
}

.card-title {
    font-size: 24px;
    font-weight: 700;
    color: var(--primary-color);
    margin-bottom: 20px;
    padding-bottom: 10px;
    border-bottom: 2px solid #f0f0f0;
}

.detail-row {
    display: flex;
    justify-content: space-between;
    padding: 12px 0;
    border-bottom: 1px solid #f5f5f5;
}

.detail-label {
    font-weight: 600;
    color: #666;
    flex: 0 0 40%;
}

.detail-value {
    color: #333;
    flex: 1;
    text-align: right;
}

.status {
    display: inline-block;
    padding: 6px 16px;
    border-radius: 20px;
    font-weight: 600;
    font-size: 14px;
    text-transform: uppercase;
}

.status.pending {
    background: #fff3cd;
    color: #856404;
}

.status.approved {
    background: #d4edda;
    color: #155724;
}

.status.rejected {
    background: #f8d7da;
    color: #721c24;
}

.approval-section {
    margin-top: 30px;
}

.approval-timeline {
    margin-top: 20px;
}

.approval-item {
    border-left: 3px solid var(--gray-color);
    padding-left: 15px;
    margin-bottom: 20px;
    position: relative;
}

.approval-item.approved {
    border-left-color: var(--success-color);
}

.approval-item.rejected {
    border-left-color: var(--danger-color);
}

.approval-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 8px;
}

.approval-role {
    font-weight: 700;
    font-size: 16px;
    color: #333;
}

.approval-name {
    font-size: 14px;
    color: #666;
}

.approval-reason {
    margin-top: 8px;
    padding: 10px;
    background: #f9f9f9;
    border-radius: 4px;
    font-size: 14px;
    color: #666;
}

.approval-buttons {
    margin-top: 12px;
    display: flex;
    gap: 10px;
}

.btn {
    padding: 10px 20px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-weight: 600;
    font-size: 14px;
    display: flex;
    align-items: center;
    gap: 8px;
    transition: all 0.3s;
}

.btn-approve {
    background: var(--success-color);
    color: white;
}

.btn-approve:hover {
    background: #28a745;
}

.btn-reject {
    background: var(--danger-color);
    color: white;
}

.btn-reject:hover {
    background: #dc3545;
}

.waiting-message {
    color: var(--warning-color);
    font-style: italic;
    font-size: 14px;
    margin-top: 12px;
}

.loading {
    text-align: center;
    padding: 40px;
    color: #666;
}

.error {
    background: #f8d7da;
    color: #721c24;
    padding: 20px;
    border-radius: 8px;
    margin: 20px 0;
}

.table {
    width: 100%;
    margin-top: 15px;
    border-collapse: collapse;
}

.table th,
.table td {
    padding: 12px;
    text-align: left;
    border-bottom: 1px solid #f0f0f0;
}

.table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #333;
}

/* Modal for approval reason */
.modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.7);
    z-index: 10000;
    align-items: center;
    justify-content: center;
}

.modal.active {
    display: flex;
}

.modal-content {
    background: white;
    border-radius: 12px;
    padding: 30px;
    max-width: 600px;
    width: 90%;
    max-height: 90vh;
    overflow-y: auto;
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
}

.modal-title {
    font-size: 22px;
    font-weight: 700;
    color: var(--primary-color);
}

.modal-close {
    background: none;
    border: none;
    font-size: 28px;
    color: #999;
    cursor: pointer;
}

.modal-close:hover {
    color: #333;
}

.form-group {
    margin-bottom: 20px;
}

.form-label {
    display: block;
    font-weight: 600;
    margin-bottom: 8px;
    color: #333;
}

.form-control {
    width: 100%;
    padding: 12px;
    border: 1px solid #e0e0e0;
    border-radius: 6px;
    font-size: 14px;
    font-family: inherit;
    resize: vertical;
}

.char-count {
    text-align: right;
    font-size: 12px;
    color: #999;
    margin-top: 4px;
}

.modal-actions {
    display: flex;
    justify-content: flex-end;
    gap: 12px;
    margin-top: 20px;
}

.btn-cancel {
    background: #f3f4f6;
    color: #374151;
    padding: 10px 20px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-weight: 600;
}

.btn-cancel:hover {
    background: #e5e7eb;
}

.btn-submit {
    padding: 10px 20px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-weight: 600;
    color: white;
}

.notification {
    position: fixed;
    top: 20px;
    right: 20px;
    padding: 15px 20px;
    border-radius: 8px;
    color: white;
    z-index: 10001;
    display: flex;
    align-items: center;
    gap: 10px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    animation: slideIn 0.3s ease-out;
}

.notification.success {
    background: var(--success-color);
}

.notification.error {
    background: var(--danger-color);
}

@keyframes slideIn {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

.info-box {
    background: #f0f7ff;
    border-left: 4px solid var(--primary-color);
    padding: 15px;
    margin: 20px 0;
    border-radius: 4px;
}
</style>
<?php $__env->stopSection(); ?>

<?php $__env->startSection('content'); ?>
<div class="container">
    <div class="header-section">
        <button class="back-button" onclick="goBack()">
            <i class="fas fa-arrow-left"></i>
            Back to Applications
        </button>
        <img src="/images/Sidebar-logo.png" alt="Titan Drilling Logo" class="header-logo">
        <h1 id="pageTitle">Application Details</h1>
    </div>

    <div id="loadingState" class="loading">
        <i class="fas fa-spinner fa-spin" style="font-size: 48px; color: var(--primary-color);"></i>
        <p style="margin-top: 20px;">Loading application details...</p>
    </div>

    <div id="errorState" class="error" style="display: none;">
        <h3><i class="fas fa-exclamation-triangle"></i> Error Loading Application</h3>
        <p id="errorMessage"></p>
    </div>

    <div id="contentArea" style="display: none;">
        <!-- Application details will be loaded here -->
    </div>
</div>

<!-- Approval Modal -->
<div id="approvalModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title" id="modalTitle">Approve Application</h2>
            <button class="modal-close" onclick="closeApprovalModal()">&times;</button>
        </div>
        <div id="modalBody">
            <div class="info-box" id="approvalInfo"></div>
            <div class="form-group">
                <label class="form-label">
                    Reason <span style="color: var(--danger-color);">*</span>
                </label>
                <textarea 
                    id="approvalReason" 
                    class="form-control" 
                    rows="5" 
                    placeholder="Enter detailed reason for your decision..."
                    maxlength="500"></textarea>
                <div class="char-count">
                    <span id="charCount">0</span>/500 characters
                </div>
            </div>
        </div>
        <div class="modal-actions">
            <button class="btn-cancel" onclick="closeApprovalModal()">Cancel</button>
            <button class="btn-submit" id="submitBtn" onclick="submitApprovalDecision()">Submit</button>
        </div>
    </div>
</div>

<script>
// Get application ID from URL
const urlParams = new URLSearchParams(window.location.search);
const applicationId = urlParams.get('id');
const applicationType = urlParams.get('type');
const HR_REMOTE_UID = encodeURIComponent(<?php echo json_encode(config('services.hr_module.admin_uid'), 15, 512) ?>);

let currentApplication = null;
let currentApprovalRole = null;
let currentApprovalAction = null;

// Load application on page load
document.addEventListener('DOMContentLoaded', function() {
    checkUserRole();
    if (applicationId && applicationType) {
        loadApplicationDetails();
    } else {
        showError('Invalid application ID or type');
    }
});

// Check user role
function checkUserRole() {
    const currentUser = JSON.parse(localStorage.getItem('currentUser') || '{}');
    if (!currentUser.role) {
        showRoleSelector();
    }
}

// Role selector (same as before)
function showRoleSelector() {
    const modal = document.createElement('div');
    modal.style.cssText = 'position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.8); z-index: 999999; display: flex; align-items: center; justify-content: center;';
    modal.innerHTML = `
        <div style="background: white; padding: 30px; border-radius: 12px; max-width: 500px; width: 90%;">
            <h2 style="color: #0055a5; margin-bottom: 20px; font-size: 24px;">👤 Select Your Role</h2>
            <p style="margin-bottom: 20px; color: #666;">Please select your role to test the approval system:</p>
            <div style="display: flex; flex-direction: column; gap: 12px;">
                <button onclick="setUserRole('ProAdmin', 'Admin User')" style="padding: 15px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 8px; cursor: pointer; font-weight: 600; font-size: 16px;">
                    👑 ProAdmin (Can approve all levels)
                </button>
                <button onclick="setUserRole('Supervisor', 'Test Supervisor')" style="padding: 15px; background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); color: white; border: none; border-radius: 8px; cursor: pointer; font-weight: 600; font-size: 16px;">
                    ⭐ Supervisor
                </button>
                <button onclick="setUserRole('HrManager', 'Test HR Manager')" style="padding: 15px; background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); color: white; border: none; border-radius: 8px; cursor: pointer; font-weight: 600; font-size: 16px;">
                    ⭐ HR Manager
                </button>
                <button onclick="setUserRole('FinanceManager', 'Test Finance Manager')" style="padding: 15px; background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%); color: white; border: none; border-radius: 8px; cursor: pointer; font-weight: 600; font-size: 16px;">
                    ⭐ Finance Manager
                </button>
            </div>
        </div>
    `;
    document.body.appendChild(modal);
}

function setUserRole(role, name) {
    localStorage.setItem('currentUser', JSON.stringify({
        _id: 'test_' + Date.now(),
        name: name,
        role: role
    }));
    location.reload();
}

// Load application details
async function loadApplicationDetails() {
    try {
        console.log('🔍 Loading application:', applicationId, 'Type:', applicationType);
        
        // Try to get from sessionStorage first (faster)
        const cachedApps = JSON.parse(sessionStorage.getItem(getStorageKey(applicationType)) || '{}');
        if (cachedApps[applicationId]) {
            console.log('✅ Found in sessionStorage cache');
            currentApplication = cachedApps[applicationId];
            renderApplicationDetails(cachedApps[applicationId]);
            return;
        }
        
        console.log('📡 Not in cache, fetching from MongoDB...');
        
        // Fetch all applications of this type from MongoDB
        const response = await fetch(`https://api.titandrillingzm.com:6013/hr-applications/${applicationType}/${HR_REMOTE_UID}`);
        
        console.log('📡 Response status:', response.status);
        
        if (!response.ok) {
            throw new Error('Failed to load applications from MongoDB');
        }
        
        const result = await response.json();
        console.log('📦 MongoDB result:', result);
        console.log('📊 Applications count:', result.data?.length || 0);
        
        if (result.success && result.data && result.data.length > 0) {
            // Find the specific application by ID
            const application = result.data.find(app => 
                app.id === applicationId || 
                app._id === applicationId || 
                app.applicationId === applicationId
            );
            
            if (application) {
                console.log('✅ Application found:', application);
                currentApplication = application;
                renderApplicationDetails(application);
            } else {
                console.log('❌ Application not found in list');
                console.log('Looking for ID:', applicationId);
                console.log('Available IDs:', result.data.map(a => a.id || a._id).slice(0, 5));
                throw new Error('Application not found in the list');
            }
        } else {
            console.log('❌ No applications returned from MongoDB');
            throw new Error('No applications found for this type');
        }
    } catch (error) {
        console.error('❌ Error loading application:', error);
        showError(error.message || 'Failed to load application details');
    }
}

// Get sessionStorage key based on application type
function getStorageKey(type) {
    const keys = {
        'leave_application': 'leaveApplications',
        'grievance_record': 'grievanceApplications',
        'salary_advance_application': 'salaryAdvanceApplications',
        'hospital_attendance_form': 'hospitalAttendanceApplications',
        'funeral_grant_form': 'funeralApplications',
        'application_for_appeal': 'appealApplications'
    };
    return keys[type] || 'applications';
}

// Render application details
function renderApplicationDetails(application) {
    console.log('🎨 Rendering application details:', application);
    console.log('📋 Available fields:', Object.keys(application));
    
    document.getElementById('loadingState').style.display = 'none';
    document.getElementById('contentArea').style.display = 'block';
    
    const currentUser = JSON.parse(localStorage.getItem('currentUser') || '{}');
    const userRole = (currentUser.role || '').toLowerCase();
    const isSupervisor = userRole === 'supervisor' || userRole === 'supervisormanager';
    const isHrManager = userRole === 'hrmanager' || userRole === 'hr manager';
    const isProAdmin = userRole === 'proadmin' || userRole === 'pro admin';
    
    let html = '';
    
    // Basic Information Card
    html += `
        <div class="detail-card">
            <h2 class="card-title">${getApplicationTypeName(applicationType)}</h2>
            ${renderBasicInfo(application)}
        </div>
    `;
    
    // Application-specific details
    html += renderSpecificDetails(application);
    
    // Approval Status
    html += `
        <div class="detail-card">
            <h2 class="card-title">Approval Status</h2>
            <div class="approval-timeline">
                ${renderApprovalStatus(application, { isSupervisor, isHrManager, isProAdmin })}
            </div>
        </div>
    `;
    
    document.getElementById('contentArea').innerHTML = html;
}

// Render basic info
function renderBasicInfo(app) {
    console.log('📊 Rendering basic info for:', app);
    
    // Safe date formatting
    const formatDate = (dateValue) => {
        if (!dateValue) return 'N/A';
        try {
            const date = new Date(dateValue);
            if (isNaN(date.getTime())) return 'N/A';
            return date.toLocaleString('en-US', { 
                year: 'numeric', 
                month: 'short', 
                day: 'numeric', 
                hour: '2-digit', 
                minute: '2-digit' 
            });
        } catch (e) {
            return 'N/A';
        }
    };
    
    return `
        <div class="detail-row">
            <span class="detail-label">Application No:</span>
            <span class="detail-value">${app.applicationNumber || app.applicationNo || 'N/A'}</span>
        </div>
        <div class="detail-row">
            <span class="detail-label">Full Name:</span>
            <span class="detail-value">${app.fullName || app.name || app.applicantName || 'N/A'}</span>
        </div>
        <div class="detail-row">
            <span class="detail-label">Employee No:</span>
            <span class="detail-value">${app.employeeNo || app.employeeNumber || app.empNo || 'N/A'}</span>
        </div>
        <div class="detail-row">
            <span class="detail-label">Department:</span>
            <span class="detail-value">${app.department || app.dept || 'N/A'}</span>
        </div>
        <div class="detail-row">
            <span class="detail-label">Country:</span>
            <span class="detail-value">${Array.isArray(app.country) ? app.country.join(', ') : (app.country || app.countries || 'N/A')}</span>
        </div>
        <div class="detail-row">
            <span class="detail-label">Project:</span>
            <span class="detail-value">${Array.isArray(app.project) ? app.project.join(', ') : (app.project || app.projects || 'N/A')}</span>
        </div>
        <div class="detail-row">
            <span class="detail-label">NRC No:</span>
            <span class="detail-value">${app.nrcNo || app.nrcIdNumber || app.nrc || 'N/A'}</span>
        </div>
        <div class="detail-row">
            <span class="detail-label">Submitted At:</span>
            <span class="detail-value">${formatDate(app.submittedAt || app.submissionDate || app.createdAt)}</span>
        </div>
    `;
}

// Render specific details based on type
function renderSpecificDetails(app) {
    console.log('📋 Rendering specific details for type:', applicationType);
    
    // Safe date formatting
    const formatDate = (dateValue) => {
        if (!dateValue) return 'N/A';
        try {
            const date = new Date(dateValue);
            if (isNaN(date.getTime())) return 'N/A';
            return date.toLocaleDateString('en-US', { 
                year: 'numeric', 
                month: 'short', 
                day: 'numeric'
            });
        } catch (e) {
            console.error('Date formatting error:', e);
            return 'N/A';
        }
    };
    
    let html = '<div class="detail-card"><h2 class="card-title">Application Details</h2>';
    
    switch (applicationType) {
        case 'leave_application':
            const leaveStart = app.leaveStartDate || app.dateFrom;
            const leaveEnd = app.leaveEndDate || app.dateTo;
            const startDate = leaveStart ? new Date(leaveStart) : null;
            const endDate = leaveEnd ? new Date(leaveEnd) : null;
            const reportDate = formatDate(app.reportDate);
            
            let duration = 'N/A';
            if (startDate && endDate && !isNaN(startDate.getTime()) && !isNaN(endDate.getTime())) {
                duration = Math.ceil((endDate - startDate) / (1000 * 60 * 60 * 24)) + 1;
                duration = `${duration} day${duration > 1 ? 's' : ''}`;
            }
            
            html += `
                <div class="detail-row">
                    <span class="detail-label">Leave Start Date:</span>
                    <span class="detail-value">${formatDate(leaveStart)}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Leave End Date:</span>
                    <span class="detail-value">${formatDate(leaveEnd)}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Report Date:</span>
                    <span class="detail-value">${reportDate}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Duration:</span>
                    <span class="detail-value">${duration}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Reason:</span>
                    <span class="detail-value">${app.reasonOfLeave || app.reason || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Address While On Leave:</span>
                    <span class="detail-value">${app.addressWhileOnLeave || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Other Contact Name:</span>
                    <span class="detail-value">${app.otherContactName || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Other Contact Cell:</span>
                    <span class="detail-value">${app.otherContactCell || 'N/A'}</span>
                </div>
            `;
            break;
            
        case 'grievance_record':
            const grievanceDate = formatDate(app.grievanceDate);
            let complainantsHtml = '';
            if (app.grievanceComplaints && app.grievanceComplaints.length > 0) {
                complainantsHtml = '<table class="table"><thead><tr><th>Name</th><th>Mine No</th><th>Occupation/Grade</th></tr></thead><tbody>';
                app.grievanceComplaints.forEach(c => {
                    complainantsHtml += `<tr><td>${c.name || 'N/A'}</td><td>${c.mineNo || 'N/A'}</td><td>${c.occupationGrade || 'N/A'}</td></tr>`;
                });
                complainantsHtml += '</tbody></table>';
            } else {
                complainantsHtml = 'N/A';
            }
            
            html += `
                <div class="detail-row">
                    <span class="detail-label">Grievance Date:</span>
                    <span class="detail-value">${grievanceDate}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Grievance Type:</span>
                    <span class="detail-value">${app.grievanceType || 'General Grievance'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Priority:</span>
                    <span class="detail-value"><span class="status ${(app.priority || 'normal').toLowerCase()}">${app.priority || app.grievancePriority || 'Normal'}</span></span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Place of Work:</span>
                    <span class="detail-value">${app.grievancePlaceOfWork || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Supervisor Name:</span>
                    <span class="detail-value">${app.grievanceSupervisorName || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Details:</span>
                    <span class="detail-value">${app.description || app.grievanceDescription || app.grievanceDetails || 'N/A'}</span>
                </div>
                ${app.grievanceComplaints && app.grievanceComplaints.length > 0 ? `
                    <div style="margin-top: 20px;">
                        <h3 style="margin-bottom: 10px; color: #333;">Complainants:</h3>
                        ${complainantsHtml}
                    </div>
                ` : ''}
            `;
            break;
            
        case 'salary_advance_application':
            html += `
                <div class="detail-row">
                    <span class="detail-label">Amount:</span>
                    <span class="detail-value">K ${app.salaryAdvanceAmount || app.amount || app.advanceAmount || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Sites:</span>
                    <span class="detail-value">${app.selectedSites?.join(', ') || app.sites || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Recovery Period:</span>
                    <span class="detail-value">${app.selectedRecoveryPeriod || app.recoveryPeriod || app.repaymentPeriod || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Reasons:</span>
                    <span class="detail-value">${app.selectedReasons?.join(', ') || app.reasons || app.purpose || 'N/A'}</span>
                </div>
            `;
            break;
            
        case 'hospital_attendance_form':
            html += `
                <div class="detail-row">
                    <span class="detail-label">Clinics:</span>
                    <span class="detail-value">${app.selectedClinics?.join(', ') || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Shift:</span>
                    <span class="detail-value">${app.selectedShift || app.shift || 'N/A'}</span>
                </div>
            `;
            break;
            
        case 'funeral_grant_form':
            html += `
                <div class="detail-row">
                    <span class="detail-label">From Date:</span>
                    <span class="detail-value">${formatDate(app.funeralFromDate)}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">To Date:</span>
                    <span class="detail-value">${formatDate(app.funeralToDate)}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Relationship Type:</span>
                    <span class="detail-value">${app.selectedFuneralRelationshipType || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Deceased Name:</span>
                    <span class="detail-value">${app.deceasedName || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Deceased Relationship:</span>
                    <span class="detail-value">${app.deceasedRelationship || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Deceased Date:</span>
                    <span class="detail-value">${formatDate(app.deceasedDate)}</span>
                </div>
            `;
            break;
            
        case 'application_for_appeal':
            html += `
                <div class="detail-row">
                    <span class="detail-label">Form Date:</span>
                    <span class="detail-value">${formatDate(app.appealFormDate || app.formDate)}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Sanction:</span>
                    <span class="detail-value">${app.appealSanction || app.sanction || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Offence:</span>
                    <span class="detail-value">${app.appealOffence || app.offence || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Offence Date:</span>
                    <span class="detail-value">${formatDate(app.appealOffenceDate || app.offenceDate)}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Grounds for Appeal:</span>
                    <span class="detail-value">${app.appealGrounds || app.grounds || app.reason || 'N/A'}</span>
                </div>
            `;
            break;
    }
    
    html += '</div>';
    return html;
}

// Render approval status
function renderApprovalStatus(app, userContext) {
    const { isSupervisor, isHrManager, isProAdmin } = userContext;
    const currentUser = JSON.parse(localStorage.getItem('currentUser') || '{}');
    const userRole = (currentUser.role || '').toLowerCase();
    const isFinanceManager = userRole === 'financemanager' || userRole === 'finance manager';
    
    let html = '';
    
    // Supervisor Section
    if (app.Supervisor || isSupervisor || isProAdmin) {
        const supervisorStatus = app.Supervisor?.status || 'Pending';
        const supervisorName = app.Supervisor?.name && app.Supervisor.name !== 'Pending:Supervisor' ? app.Supervisor.name : 'N/A';
        const supervisorReason = app.Supervisor?.reason || app.Supervisor?.comments || '';
        const canApprove = (isSupervisor || isProAdmin) && (!app.Supervisor || !app.Supervisor.status || app.Supervisor.status === 'Pending');
        
        html += `
            <div class="approval-item ${supervisorStatus.toLowerCase()}">
                <div class="approval-header">
                    <strong class="approval-role">Supervisor${isProAdmin ? ' (Admin can approve)' : ''}</strong>
                    <span class="approval-name">${supervisorName}${isSupervisor ? ' ⭐' : ''}${isProAdmin && !isSupervisor ? ' 👑' : ''}</span>
                </div>
                <span class="status ${supervisorStatus.toLowerCase()}">${supervisorStatus}</span>
                ${supervisorReason ? `<div class="approval-reason">💬 ${supervisorReason}</div>` : ''}
                ${canApprove ? `
                    <div class="approval-buttons">
                        <button class="btn btn-approve" onclick="openApprovalModal('Supervisor', 'Approved')">
                            <i class="fas fa-check"></i> Approve
                        </button>
                        <button class="btn btn-reject" onclick="openApprovalModal('Supervisor', 'Rejected')">
                            <i class="fas fa-times"></i> Reject
                        </button>
                    </div>
                ` : ''}
            </div>
        `;
    }
    
    // HR Manager Section
    if (app.HrManager || isHrManager || isProAdmin) {
        const hrStatus = app.HrManager?.status || 'Pending';
        const hrName = app.HrManager?.name && app.HrManager.name !== 'Pending:HrManager' ? app.HrManager.name : 'N/A';
        const hrReason = app.HrManager?.reason || app.HrManager?.comments || '';
        const supervisorApproved = app.Supervisor?.status === 'Approved';
        const canApprove = (isHrManager || isProAdmin) && (!app.HrManager || !app.HrManager.status || app.HrManager.status === 'Pending');
        
        html += `
            <div class="approval-item ${hrStatus.toLowerCase()}">
                <div class="approval-header">
                    <strong class="approval-role">HR Manager${isProAdmin ? ' (Admin can approve)' : ''}</strong>
                    <span class="approval-name">${hrName}${isHrManager ? ' ⭐' : ''}${isProAdmin && !isHrManager ? ' 👑' : ''}</span>
                </div>
                <span class="status ${hrStatus.toLowerCase()}">${hrStatus}</span>
                ${hrReason ? `<div class="approval-reason">💬 ${hrReason}</div>` : ''}
                ${!supervisorApproved && (isHrManager || isProAdmin) && !isProAdmin ? '<div class="waiting-message">⏳ Waiting for Supervisor approval</div>' : ''}
                ${!supervisorApproved && isProAdmin ? '<div class="waiting-message">⏳ Supervisor not approved yet (Admin can approve anyway)</div>' : ''}
                ${canApprove && (supervisorApproved || isProAdmin) ? `
                    <div class="approval-buttons">
                        <button class="btn btn-approve" onclick="openApprovalModal('HrManager', 'Approved')">
                            <i class="fas fa-check"></i> Approve
                        </button>
                        <button class="btn btn-reject" onclick="openApprovalModal('HrManager', 'Rejected')">
                            <i class="fas fa-times"></i> Reject
                        </button>
                    </div>
                ` : ''}
            </div>
        `;
    }
    
    // Finance Manager Section (only for salary_advance_application)
    if (applicationType === 'salary_advance_application' && (app.FinanceManager || isFinanceManager || isProAdmin)) {
        const financeStatus = app.FinanceManager?.status || 'Pending';
        const financeName = app.FinanceManager?.name && app.FinanceManager.name !== 'Pending:FinanceManager' ? app.FinanceManager.name : 'N/A';
        const financeReason = app.FinanceManager?.reason || app.FinanceManager?.comments || '';
        const hrApproved = app.HrManager?.status === 'Approved';
        const canApprove = (isFinanceManager || isProAdmin) && (!app.FinanceManager || !app.FinanceManager.status || app.FinanceManager.status === 'Pending');
        
        html += `
            <div class="approval-item ${financeStatus.toLowerCase()}">
                <div class="approval-header">
                    <strong class="approval-role">Finance Manager${isProAdmin ? ' (Admin can approve)' : ''}</strong>
                    <span class="approval-name">${financeName}${isFinanceManager ? ' ⭐' : ''}${isProAdmin && !isFinanceManager ? ' 👑' : ''}</span>
                </div>
                <span class="status ${financeStatus.toLowerCase()}">${financeStatus}</span>
                ${financeReason ? `<div class="approval-reason">💬 ${financeReason}</div>` : ''}
                ${!hrApproved && (isFinanceManager || isProAdmin) && !isProAdmin ? '<div class="waiting-message">⏳ Waiting for HR Manager approval</div>' : ''}
                ${!hrApproved && isProAdmin ? '<div class="waiting-message">⏳ HR Manager not approved yet (Admin can approve anyway)</div>' : ''}
                ${canApprove && (hrApproved || isProAdmin) ? `
                    <div class="approval-buttons">
                        <button class="btn btn-approve" onclick="openApprovalModal('FinanceManager', 'Approved')">
                            <i class="fas fa-check"></i> Approve
                        </button>
                        <button class="btn btn-reject" onclick="openApprovalModal('FinanceManager', 'Rejected')">
                            <i class="fas fa-times"></i> Reject
                        </button>
                    </div>
                ` : ''}
            </div>
        `;
    }
    
    return html;
}

// Open approval modal
function openApprovalModal(role, action) {
    currentApprovalRole = role;
    currentApprovalAction = action;
    
    const modal = document.getElementById('approvalModal');
    const modalTitle = document.getElementById('modalTitle');
    const approvalInfo = document.getElementById('approvalInfo');
    const submitBtn = document.getElementById('submitBtn');
    const reasonTextarea = document.getElementById('approvalReason');
    
    modalTitle.textContent = action === 'Approved' ? '✓ Approve Application' : '✕ Reject Application';
    approvalInfo.innerHTML = `
        <strong>${action === 'Approved' ? 'Approving' : 'Rejecting'} as ${role === 'HrManager' ? 'HR Manager' : role}</strong><br>
        Employee: ${currentApplication.fullName} (#${currentApplication.employeeNo})<br>
        Application: ${currentApplication.applicationNumber}
    `;
    approvalInfo.style.background = action === 'Approved' ? '#f0fdf4' : '#fef2f2';
    approvalInfo.style.borderLeftColor = action === 'Approved' ? '#34C759' : '#FF3B30';
    
    submitBtn.style.background = action === 'Approved' ? '#34C759' : '#FF3B30';
    reasonTextarea.value = '';
    document.getElementById('charCount').textContent = '0';
    
    modal.classList.add('active');
}

// Close approval modal
function closeApprovalModal() {
    document.getElementById('approvalModal').classList.remove('active');
}

// Character counter
document.getElementById('approvalReason').addEventListener('input', function() {
    document.getElementById('charCount').textContent = this.value.length;
});

// Submit approval decision
async function submitApprovalDecision() {
    const reason = document.getElementById('approvalReason').value.trim();
    
    if (!reason) {
        showNotification('Please enter a reason', 'error');
        return;
    }
    
    const currentUser = JSON.parse(localStorage.getItem('currentUser') || '{}');
    const userName = currentUser.name || currentUser.fullName || currentApprovalRole;
    const userUid = currentUser._id || currentUser.id || 'unknown';
    
    try {
        document.getElementById('submitBtn').disabled = true;
        document.getElementById('submitBtn').textContent = 'Processing...';
        
        const response = await fetch(`https://api.titandrillingzm.com:6013/hr-applications/${applicationId}/status`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                status: currentApprovalAction,
                comments: reason,
                approverUid: userUid,
                approverName: userName,
                approverRole: currentApprovalRole,
                reason: reason,
                timestamp: new Date().toISOString()
            }),
        });
        
        if (!response.ok) {
            throw new Error('Failed to update application status');
        }
        
        const result = await response.json();
        
        if (result.success) {
            setTimeout(() => triggerNotificationWorkflow(currentApprovalAction, currentApprovalRole, reason, userName), 300);

            applyLocalApprovalPatch(currentApprovalRole, currentApprovalAction, reason, userName);
            clearApplicationCache();
            
            showNotification(`Application ${currentApprovalAction.toLowerCase()} successfully`, 'success');
            closeApprovalModal();
            
            // Re-render immediately with the updated local data
            renderApplicationDetails(currentApplication);
            
            // Reload after 1 second to sync with backend
            setTimeout(() => {
                loadApplicationDetails();
            }, 1000);
        } else {
            throw new Error(result.error || 'Failed to update status');
        }
    } catch (error) {
        console.error('Error:', error);
        showNotification('Failed to update application status', 'error');
        document.getElementById('submitBtn').disabled = false;
        document.getElementById('submitBtn').textContent = 'Submit';
    }
}

// Helper functions
function getApplicationTypeName(type) {
    const names = {
        'leave_application': 'Leave Application',
        'grievance_record': 'Grievance Record',
        'salary_advance_application': 'Salary Advance Application',
        'hospital_attendance_form': 'Hospital Attendance Form',
        'funeral_grant_form': 'Funeral Grant Form',
        'application_for_appeal': 'Application for Appeal'
    };
    return names[type] || 'Application Details';
}

function applyLocalApprovalPatch(role, action, reason, approverName) {
    const keyMap = {
        'Supervisor': 'Supervisor',
        'HrManager': 'HrManager',
        'FinanceManager': 'FinanceManager'
    };
    const blockKey = keyMap[role];
    if (blockKey) {
        if (!currentApplication[blockKey] || typeof currentApplication[blockKey] !== 'object') {
            currentApplication[blockKey] = {};
        }
        currentApplication[blockKey].status = action;
        currentApplication[blockKey].reason = reason;
        currentApplication[blockKey].comments = reason;
        currentApplication[blockKey].name = approverName;
        currentApplication[blockKey].updatedAt = new Date().toISOString();
    }

    currentApplication.status = action;

    if (action === 'Rejected') {
        ['Supervisor', 'HrManager', 'FinanceManager'].forEach(section => {
            if (currentApplication[section]) {
                currentApplication[section].status = 'Rejected';
                currentApplication[section].reason = currentApplication[section].reason || reason;
            }
        });
    }
}

function clearApplicationCache() {
    const storageKey = getStorageKey(applicationType);
    sessionStorage.removeItem(storageKey);
}

async function triggerNotificationWorkflow(action, role, reason, approverName) {
    try {
        const applicationPayload = buildNotificationApplicationData();

        if (role === 'Supervisor' && action === 'Approved') {
            await postNotificationPayload('/api/hr-notifications/supervisor-approval', {
                applicationData: applicationPayload,
                approverName
            });
        }

        await postNotificationPayload('/api/hr-notifications/application-decision', {
            applicationData: applicationPayload,
            status: action,
            approverName,
            approverRole: role,
            reason,
            applicantId: currentApplication.adminUid || currentApplication.userUid
        });
    } catch (error) {
        console.error('❌ Notification workflow failed:', error);
    }
}

async function postNotificationPayload(path, payload) {
    try {
        const response = await fetch(`https://api.titandrillingzm.com:6027${path}`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(payload)
        });
        if (!response.ok) {
            const body = await response.text();
            console.warn(`⚠️ Notification service ${path} responded with ${response.status}:`, body);
        }
    } catch (error) {
        console.error(`❌ Failed to call notification service ${path}:`, error);
    }
}

function buildNotificationApplicationData() {
    return {
        _id: currentApplication._id || currentApplication.id,
        id: currentApplication.id || currentApplication._id,
        applicationNumber: currentApplication.applicationNumber,
        applicationType: currentApplication.applicationType || applicationType,
        fullName: currentApplication.fullName,
        employeeNo: currentApplication.employeeNo,
        department: currentApplication.department,
        country: currentApplication.country,
        project: currentApplication.project,
        status: currentApplication.status || currentApprovalAction || 'Pending',
        adminUid: currentApplication.adminUid || currentApplication.userUid
    };
}

function showError(message) {
    document.getElementById('loadingState').style.display = 'none';
    document.getElementById('errorState').style.display = 'block';
    document.getElementById('errorMessage').textContent = message;
}

function showNotification(message, type) {
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-circle'}"></i>
        ${message}
    `;
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.remove();
    }, 3000);
}

function goBack() {
    const typeMap = {
        'leave_application': '/leave-applications',
        'grievance_record': '/grievance-applications',
        'salary_advance_application': '/salary-advance',
        'hospital_attendance_form': '/hospital-attendance',
        'funeral_grant_form': '/funeral-applications',
        'application_for_appeal': '/appeal-applications'
    };
    window.location.href = typeMap[applicationType] || '/hr-admin-dashboard';
}
</script>
<?php $__env->stopSection(); ?>


<?php echo $__env->make('welcome', array_diff_key(get_defined_vars(), ['__data' => 1, '__path' => 1]))->render(); ?><?php /**PATH /home/titandrillingzm/public_html/admin.titandrillingzm.com/resources/views/hr/application-detail.blade.php ENDPATH**/ ?>