@extends('welcome')

@section('title', 'Create Incident Report')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<!-- HTML to PDF library -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js"></script>
<style>
/* CSS Variables for consistent theming - Matching Workorder File */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

body.modal-open {
    overflow: hidden;
}

/* Main Container */
.incident-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
    text-decoration: none;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-subtitle {
    color: rgba(255, 255, 255, 0.9);
    font-size: 14px;
    font-weight: 400;
    margin: 4px 0 0 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.generate-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.generate-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    text-decoration: none;
    color: var(--primary-color);
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Form Styles */
.form-container {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

.form-section {
    padding: 30px;
    border-bottom: 1px solid var(--border-light);
}

.form-section:last-child {
    border-bottom: none;
}

.section-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.section-title i {
    color: var(--primary-color);
}

/* Form Groups */
.form-group {
    margin-bottom: 25px;
}

.form-group:last-child {
    margin-bottom: 0;
}

.form-label {
    display: block;
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.required {
    color: var(--danger-color);
    font-weight: bold;
}

.form-input, .form-select, .form-textarea {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
    transition: var(--transition-base);
    background: white;
}

.form-input:focus, .form-select:focus, .form-textarea:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.form-input.error, .form-select.error {
    border-color: var(--danger-color);
}

.form-textarea {
    min-height: 100px;
    resize: vertical;
}

.error-message {
    color: var(--danger-color);
    font-size: 12px;
    margin-top: 5px;
    display: none;
}

.error-message.show {
    display: block;
}

/* Incident Number Display */
.incident-number-display {
    background: var(--bg-secondary);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 15px 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 25px;
}

.incident-number-label {
    font-weight: 500;
    color: var(--text-primary);
}

.incident-number-value {
    font-weight: 700;
    color: var(--primary-color);
    font-size: 16px;
}

/* Form Row Layout */
.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
}

/* Incident Types Grid */
.incident-types-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
    gap: 12px;
    margin-top: 10px;
}

.form-check {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 8px 30px;
    background: var(--bg-light);
    border-radius: var(--radius-sm);
    transition: var(--transition-base);
}

.form-check:hover {
    background: #e6f0ff;
}

.form-check-input {
    width: 18px;
    height: 18px;
    cursor: pointer;
}

.form-check-label {
    cursor: pointer;
    font-size: 14px;
    color: var(--text-primary);
    margin: 0;
}

/* Severity Rating */
.severity-options {
    display: flex;
    gap: 15px;
    margin-top: 10px;
}

.severity-options.severity-rating {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
    gap: 15px;
    max-width: 52%;
}

.severity-option {
    flex: 1;
    padding: 15px;
    border: 2px solid var(--border-color);
    border-radius: var(--radius-md);
    cursor: pointer;
    transition: var(--transition-base);
    text-align: center;
    background: white;
    display: flex;
    flex-direction: column;
    align-items: center;
}

.severity-option.severity-card {
    flex: unset;
}

.severity-option:hover {
    border-color: var(--primary-color);
    transform: translateY(-2px);
    box-shadow: var(--shadow-sm);
}

.severity-option.selected {
    border-color: var(--primary-color);
    background: var(--primary-color);
    color: white;
}

.severity-icon {
    font-size: 24px;
    margin-bottom: 8px;
}

.severity-icon.image {
    width: 36px;
    height: 36px;
    margin-bottom: 8px;
    object-fit: contain;
    display: block;
}

.severity-options.equipment-rating {
    max-width: 40%;
}

.severity-icon.equipment-icon {
    font-size: 32px;
    margin-bottom: 8px;
}

.severity-option[data-equipment="yes"] .equipment-icon {
    color: #28a745;
}

.severity-option[data-equipment="no"] .equipment-icon {
    color: #dc3545;
}

.severity-option.selected .equipment-icon {
    color: #fff;
}

.severity-label {
    font-size: 14px;
    font-weight: 500;
}

/* Equipment Selection */
.select-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 15px 20px;
    background: var(--bg-secondary);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    cursor: pointer;
    transition: var(--transition-base);
    margin-bottom: 10px;
}

.select-row:hover {
    background: #e6f0ff;
    border-color: var(--primary-color);
}

.select-label {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
}

.select-right {
    display: flex;
    align-items: center;
}

.select-text {
    color: var(--primary-color);
    font-size: 14px;
    font-weight: 500;
    margin-right: 8px;
}

.select-arrow {
    color: var(--primary-color);
    font-size: 16px;
}

/* Equipment Preview */
.equipment-preview {
    margin-top: 15px;
    padding: 15px;
    background: var(--bg-light);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
}

.preview-card {
    background: white;
    border-radius: var(--radius-md);
    padding: 15px;
    box-shadow: var(--shadow-sm);
}

.preview-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 15px;
    padding-bottom: 10px;
    border-bottom: 1px solid var(--border-light);
}

.preview-header h4 {
    margin: 0;
    color: var(--text-primary);
    font-size: 16px;
}

.preview-status {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 500;
    text-transform: uppercase;
    background: var(--success-color);
    color: white;
}

.preview-details {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 10px;
}

.preview-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 5px 0;
}

.preview-label {
    font-weight: 500;
    color: var(--text-secondary);
    font-size: 12px;
}

.preview-value {
    color: var(--text-primary);
    font-size: 12px;
    text-align: right;
    font-weight: 500;
}

/* Image Upload Grid */
.images-upload-section {
    margin-top: 20px;
}

.images-grid {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 12px;
    margin: 15px auto 0;
    max-width: 600px;
    justify-content: center;
}

.image-upload-box {
    width: 100%;
    max-width: 180px;
    aspect-ratio: 1;
    border: 2px dashed var(--border-color);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: var(--transition-base);
    background: var(--bg-light);
    position: relative;
    overflow: hidden;
    margin: 0 auto;
}

.image-upload-box:hover {
    border-color: var(--primary-color);
    background: #e6f0ff;
}

.image-upload-box.has-image {
    border-style: solid;
    border-color: var(--success-color);
}

.image-upload-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 8px;
    text-align: center;
    padding: 20px;
}

.image-upload-content i {
    font-size: 32px;
    color: var(--primary-color);
}

.image-upload-text {
    font-size: 13px;
    font-weight: 500;
    color: var(--text-primary);
}

.image-upload-hint {
    font-size: 11px;
    color: var(--text-secondary);
}

.image-preview {
    width: 100%;
    height: 100%;
    object-fit: cover;
    border-radius: var(--radius-md);
}

.image-remove-btn {
    position: absolute;
    top: 8px;
    right: 8px;
    background: var(--danger-color);
    color: white;
    border: none;
    width: 28px;
    height: 28px;
    border-radius: 50%;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: var(--transition-base);
    z-index: 10;
}

.image-remove-btn:hover {
    background: #c82333;
    transform: scale(1.1);
}

/* Modal Styles - Matching Workorder */
.custom-modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    backdrop-filter: blur(4px);
    overflow-y: auto;
}

.custom-modal.show {
    display: block;
}

.custom-modal-content {
    background-color: #fefefe;
    margin: 20px 0;
    padding: 0;
    border: 1px solid #888;
    width: 80%;
    max-width: 600px;
    border-radius: 8px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    position: absolute;
    left: 50%;
    transform: translateX(-50%);
    max-height: 80vh;
    overflow: hidden;
    animation: modalSlideIn 0.3s ease;
}

@keyframes modalSlideIn {
    from {
        opacity: 0;
        transform: translateX(-50%) scale(0.9);
    }
    to {
        opacity: 1;
        transform: translateX(-50%) scale(1);
    }
}

.custom-modal-header {
    background-color: #0055A5;
    color: white;
    padding: 15px 20px;
    border-radius: 8px 8px 0 0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.custom-modal-header h3 {
    margin: 0;
    font-size: 18px;
    font-weight: 500;
}

.close-btn {
    color: white;
    font-size: 28px;
    font-weight: bold;
    background: none;
                border: none;
    cursor: pointer;
    padding: 0;
    line-height: 1;
}

.close-btn:hover {
    opacity: 0.7;
}

.custom-modal-body {
    padding: 20px;
    overflow-y: auto;
    max-height: calc(80vh - 140px);
}

.search-container {
    display: flex;
    gap: 10px;
    margin-bottom: 20px;
}

.search-input {
    flex: 1;
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
}

.search-btn {
    padding: 10px 15px;
    background-color: #0055A5;
    color: white;
    border: none;
    border-radius: 4px;
    cursor: pointer;
}

.search-btn:hover {
    background-color: #003f7d;
}

.equipment-list {
    max-height: 400px;
    overflow-y: auto;
    border: 1px solid #eee;
    border-radius: 4px;
}

.equipment-item {
    padding: 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    margin-bottom: 12px;
    cursor: pointer;
    transition: all 0.3s ease;
    background: white;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

.equipment-item:hover {
    border-color: var(--primary-color);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    transform: translateY(-2px);
}

.equipment-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 12px;
    padding-bottom: 8px;
    border-bottom: 1px solid var(--border-color);
}

.equipment-name {
    font-weight: 700;
    color: var(--text-primary);
    font-size: 16px;
    margin: 0;
}

.equipment-details {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.equipment-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 4px 0;
    border-bottom: 1px solid #f0f0f0;
}

.equipment-row:last-child {
    border-bottom: none;
}

.equipment-label {
    font-weight: 600;
    color: var(--text-secondary);
    font-size: 12px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.equipment-value {
    color: var(--text-primary);
    font-size: 13px;
    text-align: right;
}

/* Loading & Success Modals */
.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    backdrop-filter: blur(4px);
    z-index: 10001;
    overflow-y: auto;
    display: none;
}

.sync-modal.show {
    display: flex;
}

.sync-content {
    background: white;
    border-radius: var(--radius-lg);
    padding: 40px;
    text-align: center;
    max-width: 300px;
    position: absolute;
    left: 50%;
    transform: translateX(-50%);
    margin: 20px 0;
    animation: modalSlideIn 0.3s ease;
}

.sync-icon {
    width: 40px;
    height: 40px;
    margin: 0 auto 20px;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

.sync-text {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
}

.sync-subtext {
    font-size: 14px;
    color: var(--text-secondary);
}

.success-modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    backdrop-filter: blur(4px);
    z-index: 10000;
    overflow-y: auto;
    display: none;
}

.success-modal-overlay.show {
    display: block;
}

.success-modal-container {
    background: white;
    border-radius: 12px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
    max-width: 500px;
    width: 90%;
    overflow: hidden;
    position: absolute;
    left: 50%;
    transform: translateX(-50%);
    margin: 20px 0;
    animation: modalSlideIn 0.3s ease;
}

.success-modal-header {
    background: #0055A5;
    padding: 20px 30px;
    text-align: center;
}

.success-title {
    color: white;
    font-size: 24px;
    font-weight: 700;
    margin: 0;
}

.success-modal-body {
    padding: 40px 30px;
    text-align: center;
}

.success-message {
    color: #333;
    font-size: 18px;
    font-weight: 500;
    margin: 0 0 30px 0;
}

.success-buttons {
    display: flex;
    gap: 15px;
    justify-content: center;
}

.success-btn {
    background: #0055A5;
    color: white;
    border: none;
    padding: 12px 24px;
    border-radius: 8px;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    min-width: 120px;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
}

.success-btn:hover {
    background: #003d7a;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
    text-decoration: none;
    color: white;
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .main-content {
        padding: 20px 15px;
    }
    
    .form-section {
        padding: 20px;
    }
    
    .form-row {
        grid-template-columns: 1fr;
    }
    
    .images-grid {
        grid-template-columns: repeat(2, 1fr);
    }
    
    .severity-options {
        flex-direction: column;
    }
}

@media (max-width: 480px) {
    .images-grid {
        grid-template-columns: 1fr;
    }
}

/* Utility Classes */
.hidden {
    display: none !important;
}

.text-center {
    text-align: center;
}
</style>
@endsection

@section('content')
<div class="incident-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('inspection.incident_report') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <div>
                    <h1 class="header-title">Create Incident Report</h1>
                    <p class="header-subtitle">Report and document incidents with detailed information</p>
                </div>
            </div>
            <div class="header-right">
                <button type="button" class="generate-btn" onclick="generateReport()">
                    <i class="fas fa-file-alt"></i>
                    Generate Report
                </button>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="form-container">
            <!-- Incident Number Section -->
            <div class="form-section">
                <div class="incident-number-display">
                    <span class="incident-number-label">Incident Number</span>
                    <span class="incident-number-value" id="incidentNumber">Generating...</span>
                </div>
            </div>

            <!-- Basic Information Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-info-circle"></i>
                    Basic Information
                </h2>

                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">
                            Country
                            <span class="required">*</span>
                        </label>
                        <select class="form-select" id="country" onchange="updateProjects()">
                            <option value="">Select Country</option>
                            <option value="Zambia">Zambia</option>
                            <option value="DRC">DRC</option>
                        </select>
                        <div class="error-message" id="countryError">Country is required</div>
                    </div>

                    <div class="form-group">
                        <label class="form-label">
                            Project
                            <span class="required">*</span>
                        </label>
                        <select class="form-select" id="project" disabled onchange="generateIncidentNumber()">
                            <option value="">Select Project</option>
                        </select>
                        <div class="error-message" id="projectError">Project is required</div>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">
                            Incident Date & Time
                            <span class="required">*</span>
                        </label>
                        <input type="datetime-local" class="form-input" id="incidentDate">
                        <div class="error-message" id="incidentDateError">Incident date is required</div>
                </div>

                    <div class="form-group">
                        <label class="form-label">Incident Area</label>
                        <input type="text" class="form-input" id="incidentArea" placeholder="Enter incident area">
                </div>
            </div>

                <div class="form-group">
                    <label class="form-label">
                        Incident Category
                        <span class="required">*</span>
                    </label>
                    <select class="form-select" id="incidentCategory">
                        <option value="">Select Category</option>
                        <option value="Personnel">Personnel</option>
                        <option value="Equipment">Equipment</option>
                        <option value="Environmental">Environmental</option>
                        <option value="Other">Other</option>
                    </select>
                    <div class="error-message" id="categoryError">Incident category is required</div>
            </div>

                <div class="form-group">
                    <label class="form-label">
                        Weather Condition
                        <span class="required">*</span>
                    </label>
                    <select class="form-select" id="weatherCondition">
                        <option value="">Select Weather</option>
                        <option value="Clear">Clear</option>
                        <option value="Cloudy">Cloudy</option>
                        <option value="Rainy">Rainy</option>
                        <option value="Foggy">Foggy</option>
                        <option value="Stormy">Stormy</option>
                    </select>
                    <div class="error-message" id="weatherError">Weather condition is required</div>
                </div>
            </div>

            <!-- Incident Types Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-exclamation-triangle"></i>
                    Types of Incident
                    <span class="required">*</span>
                </h2>

                <div class="incident-types-grid">
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Brake Failure" id="type1">
                        <label class="form-check-label" for="type1">Brake Failure</label>
                </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Chemicals Spill" id="type2">
                        <label class="form-check-label" for="type2">Chemicals Spill</label>
            </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Collision" id="type3">
                        <label class="form-check-label" for="type3">Collision</label>
                </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Drowned in Water" id="type4">
                        <label class="form-check-label" for="type4">Drowned in Water</label>
            </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Electrical Failure" id="type5">
                        <label class="form-check-label" for="type5">Electrical Failure</label>
                </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Engine Failure" id="type6">
                        <label class="form-check-label" for="type6">Engine Failure</label>
            </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Equipment Failure" id="type7">
                        <label class="form-check-label" for="type7">Equipment Failure</label>
                </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Explosion/fire" id="type8">
                        <label class="form-check-label" for="type8">Explosion/fire</label>
            </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Fall Attachments" id="type9">
                        <label class="form-check-label" for="type9">Fall Attachments</label>
                        </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Fall from elevation" id="type10">
                        <label class="form-check-label" for="type10">Fall from elevation</label>
                        </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Leakage in Vehicle" id="type11">
                        <label class="form-check-label" for="type11">Leakage in Vehicle</label>
                    </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Loss of Stability" id="type12">
                        <label class="form-check-label" for="type12">Loss of Stability</label>
                </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Oil Spill" id="type13">
                        <label class="form-check-label" for="type13">Oil Spill</label>
            </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Person hit by Vehicle" id="type14">
                        <label class="form-check-label" for="type14">Person hit by Vehicle</label>
            </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Steering Failure" id="type15">
                        <label class="form-check-label" for="type15">Steering Failure</label>
            </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Stuck" id="type16">
                        <label class="form-check-label" for="type16">Stuck</label>
                    </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Swamping" id="type17">
                        <label class="form-check-label" for="type17">Swamping</label>
                    </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Tire Burst" id="type18">
                        <label class="form-check-label" for="type18">Tire Burst</label>
                    </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Turned Over" id="type19">
                        <label class="form-check-label" for="type19">Turned Over</label>
                    </div>
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="incidentTypes[]" value="Other" id="type20">
                        <label class="form-check-label" for="type20">Other</label>
                    </div>
                </div>
                <div class="error-message" id="typesError">Please select at least one incident type</div>
            </div>

            <!-- Severity Rating Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-thermometer-half"></i>
                    Incident Severity Rating
                    <span class="required">*</span>
                </h2>

                <div class="severity-options severity-rating">
                    <div class="severity-option severity-card" data-severity="High" onclick="selectSeverity('High')">
                        <img class="severity-icon image" src="{{ asset('images/insvecico/High_vectorIcon_1.png') }}" alt="High severity icon">
                        <div class="severity-label">High</div>
                    </div>
                    <div class="severity-option severity-card" data-severity="Medium" onclick="selectSeverity('Medium')">
                        <img class="severity-icon image" src="{{ asset('images/insvecico/Medium_vectorIcon_1.png') }}" alt="Medium severity icon">
                        <div class="severity-label">Medium</div>
                    </div>
                    <div class="severity-option severity-card" data-severity="Low" onclick="selectSeverity('Low')">
                        <img class="severity-icon image" src="{{ asset('images/insvecico/Low_vectorIcon_1.png') }}" alt="Low severity icon">
                        <div class="severity-label">Low</div>
                    </div>
                </div>
                <div class="error-message" id="severityError">Severity rating is required</div>
            </div>

            <!-- Equipment Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-tools"></i>
                    Equipment Involvement
                </h2>

                <div class="form-group">
                    <label class="form-label">Does the incident involve any equipment?</label>
                    <div class="severity-options severity-rating equipment-rating">
                        <div class="severity-option severity-card" data-equipment="yes" onclick="selectEquipmentInvolvement('yes')">
                            <i class="severity-icon equipment-icon bi bi-check-circle-fill"></i>
                            <div class="severity-label">Yes</div>
                        </div>
                        <div class="severity-option severity-card" data-equipment="no" onclick="selectEquipmentInvolvement('no')">
                            <i class="severity-icon equipment-icon bi bi-x-circle-fill"></i>
                            <div class="severity-label">No</div>
                        </div>
                    </div>
                </div>

                <div class="form-group" id="equipmentSelectionSection" style="display: none;">
                    <div class="select-row" onclick="openEquipmentModal(event)">
                        <span class="select-label">Select Equipment</span>
                        <div class="select-right">
                            <span class="select-text" id="equipmentText">Select Equipment</span>
                            <i class="fas fa-chevron-right select-arrow"></i>
                        </div>
                    </div>
                    
                    <!-- Equipment Preview -->
                    <div class="equipment-preview" id="equipmentPreview" style="display: none;">
                        <div class="preview-card">
                            <div class="preview-header">
                                <h4 id="equipmentPreviewName">Equipment Name</h4>
                                <span class="preview-status" id="equipmentPreviewStatus">Active</span>
                        </div>
                            <div class="preview-details">
                                <div class="preview-row">
                                    <span class="preview-label">Type:</span>
                                    <span class="preview-value" id="equipmentPreviewType">-</span>
                                </div>
                                <div class="preview-row">
                                    <span class="preview-label">Model:</span>
                                    <span class="preview-value" id="equipmentPreviewModel">-</span>
                                </div>
                                <div class="preview-row">
                                    <span class="preview-label">Location:</span>
                                    <span class="preview-value" id="equipmentPreviewLocation">-</span>
                                </div>
                                <div class="preview-row">
                                    <span class="preview-label">Registration:</span>
                                    <span class="preview-value" id="equipmentPreviewReg">-</span>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Equipment Detail Fields -->
                    <div id="equipmentDetailsSection" style="margin-top: 20px; display: none;">
                        <h3 style="font-size: 16px; font-weight: 600; color: var(--primary-color); margin-bottom: 15px;">
                            <i class="fas fa-info-circle"></i> Equipment Details
                        </h3>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Vehicle ID</label>
                                <input type="text" class="form-input" id="vehicleId" placeholder="Vehicle ID">
                        </div>
                            <div class="form-group">
                                <label class="form-label">Vehicle Name</label>
                                <input type="text" class="form-input" id="vehicleName" placeholder="Vehicle Name">
                    </div>
                </div>
                </div>
            </div>
            </div>

            <!-- Incident Images Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-images"></i>
                    Incident Images
                </h2>

                <div class="images-grid" id="imagesGrid">
                    <!-- 9 image upload boxes -->
                    <div class="image-upload-box" id="imageBox1" onclick="document.getElementById('imageInput1').click()">
                <div class="image-upload-content">
                            <i class="fas fa-camera"></i>
                            <div class="image-upload-text">Image 1</div>
                            <div class="image-upload-hint">Click to upload</div>
                </div>
                        <input type="file" id="imageInput1" accept="image/*" style="display: none;" onchange="handleImageUpload(1)">
                    </div>
                    <div class="image-upload-box" id="imageBox2" onclick="document.getElementById('imageInput2').click()">
                        <div class="image-upload-content">
                            <i class="fas fa-camera"></i>
                            <div class="image-upload-text">Image 2</div>
                            <div class="image-upload-hint">Click to upload</div>
                        </div>
                        <input type="file" id="imageInput2" accept="image/*" style="display: none;" onchange="handleImageUpload(2)">
                    </div>
                    <div class="image-upload-box" id="imageBox3" onclick="document.getElementById('imageInput3').click()">
                        <div class="image-upload-content">
                            <i class="fas fa-camera"></i>
                            <div class="image-upload-text">Image 3</div>
                            <div class="image-upload-hint">Click to upload</div>
                        </div>
                        <input type="file" id="imageInput3" accept="image/*" style="display: none;" onchange="handleImageUpload(3)">
                    </div>
                    <div class="image-upload-box" id="imageBox4" onclick="document.getElementById('imageInput4').click()">
                        <div class="image-upload-content">
                            <i class="fas fa-camera"></i>
                            <div class="image-upload-text">Image 4</div>
                            <div class="image-upload-hint">Click to upload</div>
                        </div>
                        <input type="file" id="imageInput4" accept="image/*" style="display: none;" onchange="handleImageUpload(4)">
                    </div>
                    <div class="image-upload-box" id="imageBox5" onclick="document.getElementById('imageInput5').click()">
                        <div class="image-upload-content">
                            <i class="fas fa-camera"></i>
                            <div class="image-upload-text">Image 5</div>
                            <div class="image-upload-hint">Click to upload</div>
                        </div>
                        <input type="file" id="imageInput5" accept="image/*" style="display: none;" onchange="handleImageUpload(5)">
                    </div>
                    <div class="image-upload-box" id="imageBox6" onclick="document.getElementById('imageInput6').click()">
                        <div class="image-upload-content">
                            <i class="fas fa-camera"></i>
                            <div class="image-upload-text">Image 6</div>
                            <div class="image-upload-hint">Click to upload</div>
                        </div>
                        <input type="file" id="imageInput6" accept="image/*" style="display: none;" onchange="handleImageUpload(6)">
                    </div>
                    <div class="image-upload-box" id="imageBox7" onclick="document.getElementById('imageInput7').click()">
                        <div class="image-upload-content">
                            <i class="fas fa-camera"></i>
                            <div class="image-upload-text">Image 7</div>
                            <div class="image-upload-hint">Click to upload</div>
                        </div>
                        <input type="file" id="imageInput7" accept="image/*" style="display: none;" onchange="handleImageUpload(7)">
                    </div>
                    <div class="image-upload-box" id="imageBox8" onclick="document.getElementById('imageInput8').click()">
                        <div class="image-upload-content">
                            <i class="fas fa-camera"></i>
                            <div class="image-upload-text">Image 8</div>
                            <div class="image-upload-hint">Click to upload</div>
                        </div>
                        <input type="file" id="imageInput8" accept="image/*" style="display: none;" onchange="handleImageUpload(8)">
                    </div>
                    <div class="image-upload-box" id="imageBox9" onclick="document.getElementById('imageInput9').click()">
                        <div class="image-upload-content">
                            <i class="fas fa-camera"></i>
                            <div class="image-upload-text">Image 9</div>
                            <div class="image-upload-hint">Click to upload</div>
                        </div>
                        <input type="file" id="imageInput9" accept="image/*" style="display: none;" onchange="handleImageUpload(9)">
                    </div>
                </div>
            </div>

            <!-- Detailed Description Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-file-alt"></i>
                    Detailed Description
                </h2>

                <div class="form-group">
                    <label class="form-label">Description of incident</label>
                    <textarea class="form-textarea" id="incidentCause" placeholder="Type here..." rows="5"></textarea>
                </div>

                <div class="form-group">
                    <label class="form-label">Details of damage / injury / loss</label>
                    <textarea class="form-textarea" id="equipmentDamage" placeholder="Type here..." rows="5"></textarea>
                </div>

                <div class="form-group">
                    <label class="form-label">Possible direct causes</label>
                    <textarea class="form-textarea" id="additionalComment" placeholder="Type here..." rows="5"></textarea>
                </div>

                <div class="form-group">
                    <label class="form-label">Immediate Corrective Action taken</label>
                    <textarea class="form-textarea" id="immediateCorrectiveAction" placeholder="Type here..." rows="5"></textarea>
                </div>

                <div class="form-group">
                    <label class="form-label">Reported By</label>
                    <input type="text" class="form-input" id="reportedBy" placeholder="Enter reporter name">
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Equipment Selection Modal -->
<div id="equipmentModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Equipment</h3>
            <button type="button" class="close-btn" onclick="closeEquipmentModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="equipmentSearch" class="search-input" placeholder="Search equipment..." onkeyup="filterEquipment()">
                <button type="button" class="search-btn" onclick="filterEquipment()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div class="equipment-list" id="equipmentList">
                <div style="text-align: center; padding: 20px; color: #666;">
                    Loading equipment...
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Loading Modal -->
<div class="sync-modal" id="loadingModal" style="display: none;">
    <div class="sync-content">
        <div class="sync-icon">⏳</div>
        <div class="sync-text">Please Wait</div>
        <div class="sync-subtext">Generating Incident Report...</div>
                </div>
</div>

<!-- Success Modal -->
<div id="successModal" class="success-modal-overlay" style="display: none;">
    <div class="success-modal-container">
        <div class="success-modal-header">
            <h2 class="success-title">Success</h2>
                    </div>
        <div class="success-modal-body">
            <p class="success-message">Incident Report Generated and Saved Successfully!<br><small style="font-size: 14px; color: #666;">Notifications sent to all managers and supervisors</small></p>
            <div class="success-buttons">
                <button type="button" class="success-btn" onclick="goBackToIncidentReports()">
                    <i class="fas fa-arrow-left"></i>
                    Back
                        </button>
                <button type="button" class="success-btn" onclick="viewIncidentPDF()">
                    <i class="fas fa-file-pdf"></i>
                    View PDF
                        </button>
                    </div>
                </div>
                </div>
                </div>

<script>
// MongoDB API Configuration
const API_BASE_URL = 'https://api.titandrillingzm.com:6007';
const EQUIPMENT_API_URL = 'https://api.titandrillingzm.com:6001/EquipmentsDataJSON/GlobalEquipmentJsonData';
const NOTIFICATIONS_API_URL = 'https://api.titandrillingzm.com:6021';
const ADMIN_UID = 'web_admin_' + Date.now(); // Generate unique admin ID for web

// Global variables
let selectedEquipment = null;
let selectedSeverity = null;
let involvesEquipment = null;
let equipmentList = [];
let incidentImages = Array(9).fill(null);
let generatedPDFUrl = null;
let currentIncidentNumber = null;

// Project options based on country
const projectOptions = {
    'Zambia': ['Kansanshi', 'Kalumbila', 'Kimiteto', 'Kobold', 'Mimosa'],
    'DRC': ['Musompo', 'IME']
};

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    console.log('Incident Report Form initialized');
    loadEquipmentFromMongoDB();
});

// Update project options based on country
    function updateProjects() {
        const country = document.getElementById('country').value;
        const projectSelect = document.getElementById('project');
        
        projectSelect.innerHTML = '<option value="">Select Project</option>';
        projectSelect.disabled = true;
        
    if (country && projectOptions[country]) {
            projectSelect.disabled = false;
        projectOptions[country].forEach(project => {
                const option = document.createElement('option');
                option.value = project;
                option.textContent = project;
                projectSelect.appendChild(option);
            });
        }

        generateIncidentNumber();
}

// Generate incident number from MongoDB (global sequential)
async function generateIncidentNumber() {
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    
    if (!country || !project) {
        document.getElementById('incidentNumber').textContent = 'Select Country & Project';
        return;
    }
    
    try {
        // Country code mapping
        const countryCodeMap = {
            'Zambia': 'ZAM',
            'DRC': 'DRC'
        };
        
        // Project code mapping
        const projectCodeMap = {
            'Kobold': 'KOB',
            'Kalumbila': 'KAL',
            'Kansanshi': 'KAN',
            'Kimiteto': 'KIM',
            'Mimosa': 'MIM',
            'Musompo': 'MUS',
            'IME': 'IME'
        };
        
        const countryCode = countryCodeMap[country] || country.toUpperCase().slice(0, 3);
        const projectCode = projectCodeMap[project] || project.toUpperCase().slice(0, 3);
        const prefix = `${countryCode}-${projectCode}-INC-`;
        
        // Fetch all incidents from MongoDB to find highest number
        const response = await fetch(`${API_BASE_URL}/incident-reports`);
        const result = await response.json();
        
        let maxGlobalNumber = 0;
        if (result.success && result.data) {
            // Find highest number across ALL incidents (global counter)
            result.data.forEach(report => {
                if (report.incidentNumber && !report.incidentNumber.includes('-OFFLINE')) {
                    const match = report.incidentNumber.match(/INC-(\d+)/);
                    if (match) {
                        const num = parseInt(match[1], 10);
                        if (!isNaN(num)) {
                            maxGlobalNumber = Math.max(maxGlobalNumber, num);
                        }
                    }
                }
            });
        }
        
        const nextNumber = (maxGlobalNumber + 1).toString().padStart(3, '0');
        const incidentNumber = `${prefix}${nextNumber}`;
        currentIncidentNumber = incidentNumber;
        
        document.getElementById('incidentNumber').textContent = incidentNumber;
        console.log('✅ Generated incident number:', incidentNumber);
    } catch (error) {
        console.error('❌ Error generating incident number:', error);
        // Fallback to timestamp-based number
        const countryCode = country === 'Zambia' ? 'ZAM' : 'DRC';
        const projectCode = project.slice(0, 3).toUpperCase();
        const timestamp = Date.now().toString().slice(-6);
        const fallbackNumber = `${countryCode}-${projectCode}-INC-${timestamp}`;
        currentIncidentNumber = fallbackNumber;
        document.getElementById('incidentNumber').textContent = fallbackNumber;
        console.warn('⚠️ Using fallback number:', fallbackNumber);
    }
}

// Select severity
function selectSeverity(severity) {
    document.querySelectorAll('.severity-option[data-severity]').forEach(opt => {
        opt.classList.remove('selected');
    });
    event.currentTarget.classList.add('selected');
    selectedSeverity = severity;
    document.getElementById('severityError').classList.remove('show');
}

// Select equipment involvement
function selectEquipmentInvolvement(value) {
    document.querySelectorAll('.severity-option[data-equipment]').forEach(opt => {
        opt.classList.remove('selected');
    });
    event.currentTarget.classList.add('selected');
    involvesEquipment = value;
    
    const equipmentSection = document.getElementById('equipmentSelectionSection');
    const equipmentDetailsSection = document.getElementById('equipmentDetailsSection');
    
    if (value === 'yes') {
        equipmentSection.style.display = 'block';
    } else {
        equipmentSection.style.display = 'none';
        equipmentDetailsSection.style.display = 'none';
        selectedEquipment = null;
        document.getElementById('equipmentText').textContent = 'Select Equipment';
        document.getElementById('equipmentPreview').style.display = 'none';
        document.getElementById('vehicleId').value = '';
        document.getElementById('vehicleName').value = '';
    }
}

// Modal Functions with click positioning
function openModal(modalId, clickEvent = null) {
    const modal = document.getElementById(modalId);
    if (modal) {
        // Lock body scrolling
        document.body.classList.add('modal-open');
        
        // Add show class for consistent styling
        modal.classList.add('show');
        modal.style.display = 'block';
        
        // Position modal based on click or center of viewport
        const modalContent = modal.querySelector('.custom-modal-content, .success-modal-container, .sync-content');
        if (modalContent) {
            let topPosition = 0;
            
            if (clickEvent && clickEvent.clientY) {
                // Get click position relative to viewport
                const clickY = clickEvent.clientY + window.scrollY;
                const viewportHeight = window.innerHeight;
                const modalHeight = modalContent.offsetHeight || 400; // fallback height
                
                // Calculate top position centered on click point
                topPosition = clickY - (modalHeight / 2);
                
                // Ensure modal doesn't go off-screen at top
                if (topPosition < window.scrollY + 20) {
                    topPosition = window.scrollY + 20;
                }
                
                // Ensure modal doesn't go off-screen at bottom
                if (topPosition + modalHeight > window.scrollY + viewportHeight - 20) {
                    topPosition = window.scrollY + viewportHeight - modalHeight - 20;
                }
                
                // If modal is still too tall, center it in viewport
                if (topPosition < window.scrollY + 20) {
                    topPosition = window.scrollY + Math.max(20, (viewportHeight - modalHeight) / 2);
                }
            } else {
                // Center vertically in viewport if no click event
                const viewportHeight = window.innerHeight;
                const modalHeight = modalContent.offsetHeight || 400;
                topPosition = window.scrollY + Math.max(20, (viewportHeight - modalHeight) / 2);
            }
            
            modalContent.style.top = topPosition + 'px';
        }
        
        // Add click outside to close
        modal.onclick = function(event) {
            if (event.target === modal) {
                closeModal(modalId);
            }
        };
        
        // Add escape key to close
        const handleEscape = function(event) {
            if (event.key === 'Escape') {
                closeModal(modalId);
                document.removeEventListener('keydown', handleEscape);
            }
        };
        document.addEventListener('keydown', handleEscape);
    }
}

function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        // Unlock body scrolling
        document.body.classList.remove('modal-open');
        
        // Remove show class and hide modal
        modal.classList.remove('show');
        modal.style.display = 'none';
    }
}

// Equipment Modal Functions
function openEquipmentModal(event) {
    openModal('equipmentModal', event);
}

function closeEquipmentModal() {
    closeModal('equipmentModal');
}

// Loading Modal Functions
function showLoadingModal() {
    const loadingModal = document.getElementById('loadingModal');
    if (loadingModal) {
        document.body.classList.add('modal-open');
        loadingModal.classList.add('show');
        loadingModal.style.display = 'flex';
        
        // Position loading modal in center of viewport
        const modalContent = loadingModal.querySelector('.sync-content');
        if (modalContent) {
            const viewportHeight = window.innerHeight;
            const modalHeight = modalContent.offsetHeight || 200;
            const topPosition = window.scrollY + Math.max(20, (viewportHeight - modalHeight) / 2);
            modalContent.style.top = topPosition + 'px';
        }
    }
}

function hideLoadingModal() {
    const loadingModal = document.getElementById('loadingModal');
    if (loadingModal) {
        document.body.classList.remove('modal-open');
        loadingModal.classList.remove('show');
        loadingModal.style.display = 'none';
    }
}

// Success Modal Functions
function showSuccessModal() {
    const successModal = document.getElementById('successModal');
    if (successModal) {
        document.body.classList.add('modal-open');
        successModal.classList.add('show');
        successModal.style.display = 'block';
        
        // Position success modal in center of viewport
        const modalContent = successModal.querySelector('.success-modal-container');
        if (modalContent) {
            const viewportHeight = window.innerHeight;
            const modalHeight = modalContent.offsetHeight || 300;
            const topPosition = window.scrollY + Math.max(20, (viewportHeight - modalHeight) / 2);
            modalContent.style.top = topPosition + 'px';
        }
        
        // Add click outside to close (but prevent for success modal - user must click buttons)
        successModal.onclick = function(event) {
            // Don't close on outside click for success modal
            if (event.target === successModal) {
                // Optionally allow closing:
                // hideSuccessModal();
            }
        };
    }
}

function hideSuccessModal() {
    const successModal = document.getElementById('successModal');
    if (successModal) {
        document.body.classList.remove('modal-open');
        successModal.classList.remove('show');
        successModal.style.display = 'none';
    }
}

async function loadEquipmentFromMongoDB() {
    try {
        console.log('🔄 Loading equipment from MongoDB...');
        console.log('   API URL:', EQUIPMENT_API_URL);
        
        const response = await fetch(EQUIPMENT_API_URL);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('✅ Equipment API Response received');
        console.log('   Response structure:', {
            hasSuccess: 'success' in result,
            hasData: 'data' in result,
            dataType: result.data ? (Array.isArray(result.data) ? 'array' : typeof result.data) : 'none',
            count: result.count || 'unknown'
        });
        
        equipmentList = [];
        
        // Handle new API format {success: true, data: [...]}
        if (result.success && Array.isArray(result.data)) {
            console.log(`📦 Processing ${result.data.length} equipment items from array format...`);
            
            result.data.forEach(equipment => {
                if (!equipment || typeof equipment !== 'object') {
                    return;
                }
                
                const equipmentItem = {
                    id: equipment.id || equipment._id || `eq_${Date.now()}_${Math.random()}`,
                    name: equipment.equipmentName || 
                          equipment.vehicleNumber || 
                          equipment.customName ||
                          equipment.rigName || 
                          equipment.equipmentNumber ||
                          'Unknown Equipment',
                    type: equipment.mainCategory || 
                          equipment.category ||
                          equipment.vehicleType || 
                          equipment.equipmentType ||
                          'Unknown Type',
                    status: equipment.status || 'Active',
                    location: equipment.location ||
                             equipment.site || 
                             equipment.country ||
                             'Unknown Location',
                    model: equipment.model || 
                           equipment.rigModel || 
                           equipment.vehicleModel ||
                           'N/A',
                    regNum: equipment.serialNumber ||
                            equipment.registrationNumber || 
                            equipment.regNum || 
                            equipment.equipmentNumber ||
                            equipment.vehicleNumber ||
                            'N/A',
                    // Keep all original fields
                    ...equipment
                };
                
                equipmentList.push(equipmentItem);
            });
        } 
        // Handle old API format (object with keys)
        else if (result && typeof result === 'object' && !result.success) {
            console.log('📦 Processing equipment from object format...');
            
            Object.keys(result).forEach(equipmentId => {
                const equipment = result[equipmentId];
                
                if (!equipment || typeof equipment !== 'object') {
                    return;
                }
                
                const equipmentItem = {
                    id: equipmentId,
                    name: equipment.equipmentName || 
                          equipment.vehicleNumber || 
                          equipment.customName ||
                          equipment.rigName || 
                          equipment.equipmentNumber ||
                          'Unknown Equipment',
                    type: equipment.mainCategory || 
                          equipment.category ||
                          equipment.vehicleType || 
                          equipment.equipmentType ||
                          'Unknown Type',
                    status: equipment.status || 'Active',
                    location: equipment.location ||
                             equipment.site || 
                             equipment.country ||
                             'Unknown Location',
                    model: equipment.model || 
                           equipment.rigModel || 
                           equipment.vehicleModel ||
                           'N/A',
                    regNum: equipment.serialNumber ||
                            equipment.registrationNumber || 
                            equipment.regNum || 
                            equipment.equipmentNumber ||
                            equipment.vehicleNumber ||
                            'N/A',
                    ...equipment
                };
                
                equipmentList.push(equipmentItem);
            });
        }
        
        console.log(`✅ Loaded ${equipmentList.length} equipment items`);
        
        if (equipmentList.length > 0) {
            console.log('   Sample equipment:', {
                id: equipmentList[0].id,
                name: equipmentList[0].name,
                type: equipmentList[0].type,
                model: equipmentList[0].model,
                location: equipmentList[0].location,
                regNum: equipmentList[0].regNum
            });
        } else {
            console.warn('⚠️ No equipment items found in response');
        }
        
        updateEquipmentDisplay();
        
    } catch (error) {
        console.error('❌ Error loading equipment:', error);
        console.error('   Error details:', error.message);
        console.error('   Stack:', error.stack);
        equipmentList = [];
        updateEquipmentDisplay();
        showNotification('Failed to load equipment list. Please try again.', 'warning');
    }
}

function updateEquipmentDisplay() {
    filterEquipment();
}

function filterEquipment() {
    const searchTerm = document.getElementById('equipmentSearch').value.toLowerCase();
    const equipmentListDiv = document.getElementById('equipmentList');
    
    if (!equipmentListDiv) {
        console.error('❌ Equipment list div not found');
        return;
    }
    
    equipmentListDiv.innerHTML = '';
    
    // Show loading state if no equipment loaded yet
    if (equipmentList.length === 0) {
        equipmentListDiv.innerHTML = `
            <div style="text-align: center; padding: 40px 20px; color: #666;">
                <div style="font-size: 48px; margin-bottom: 15px;">🔄</div>
                <h3 style="margin: 0 0 10px 0; font-size: 16px; color: #333;">Loading Equipment...</h3>
                <p style="margin: 0; font-size: 14px; color: #999;">Please wait while we fetch equipment data.</p>
            </div>
        `;
        return;
    }
    
    // Filter equipment based on search term
    const filteredEquipment = equipmentList.filter(equipment => 
        (equipment.name && equipment.name.toLowerCase().includes(searchTerm)) ||
        (equipment.type && equipment.type.toLowerCase().includes(searchTerm)) ||
        (equipment.location && equipment.location.toLowerCase().includes(searchTerm)) ||
        (equipment.model && equipment.model.toLowerCase().includes(searchTerm)) ||
        (equipment.regNum && equipment.regNum.toLowerCase().includes(searchTerm))
    );
    
    // Show no results message
    if (filteredEquipment.length === 0) {
        equipmentListDiv.innerHTML = `
            <div style="text-align: center; padding: 40px 20px; color: #666;">
                <div style="font-size: 48px; margin-bottom: 15px;">🔍</div>
                <h3 style="margin: 0 0 10px 0; font-size: 16px; color: #333;">No Equipment Found</h3>
                <p style="margin: 0; font-size: 14px; color: #999;">
                    ${searchTerm ? `No equipment matches "${searchTerm}"` : 'No equipment available'}
                </p>
            </div>
        `;
        return;
    }
    
    console.log(`📋 Displaying ${filteredEquipment.length} of ${equipmentList.length} equipment items`);
    
    // Display filtered equipment
    filteredEquipment.forEach((equipment, index) => {
        const equipmentItem = document.createElement('div');
        equipmentItem.className = 'equipment-item';
        
        // Extract and display correct fields with better fallbacks
        const displayName = equipment.name || equipment.equipmentName || 'Unknown Equipment';
        const displayType = equipment.type || equipment.mainCategory || 'N/A';
        const displayLocation = equipment.location || equipment.site || equipment.country || 'N/A';
        const displayModel = equipment.model || 'N/A';
        const displayRegNum = equipment.regNum || equipment.serialNumber || 'N/A';
        const displayStatus = equipment.status || 'Active';
        
        equipmentItem.innerHTML = `
            <div class="equipment-header">
                <div class="equipment-name">${displayName}</div>
                <span class="preview-status" style="font-size: 10px; padding: 3px 6px;">${displayStatus}</span>
            </div>
            <div class="equipment-details">
                <div class="equipment-row">
                    <span class="equipment-label">Type:</span>
                    <span class="equipment-value">${displayType}</span>
                </div>
                <div class="equipment-row">
                    <span class="equipment-label">Location:</span>
                    <span class="equipment-value">${displayLocation}</span>
                </div>
                <div class="equipment-row">
                    <span class="equipment-label">Model:</span>
                    <span class="equipment-value">${displayModel}</span>
                </div>
                <div class="equipment-row">
                    <span class="equipment-label">Registration:</span>
                    <span class="equipment-value">${displayRegNum}</span>
                </div>
            </div>
        `;
        
        equipmentItem.onclick = () => selectEquipment(equipment);
        equipmentListDiv.appendChild(equipmentItem);
        
        // Log first 3 items for debugging
        if (index < 3) {
            console.log(`   Equipment #${index + 1}:`, {
                name: displayName,
                type: displayType,
                model: displayModel,
                location: displayLocation,
                regNum: displayRegNum
            });
        }
    });
    
    console.log('✅ Equipment list displayed successfully');
}

function selectEquipment(equipment) {
    selectedEquipment = equipment;
    
    console.log('🚗 Equipment selected:', equipment);
    
    const equipmentName = equipment.name || 'Unknown';
    document.getElementById('equipmentText').textContent = equipmentName;
    
    // Show equipment preview
    const preview = document.getElementById('equipmentPreview');
    preview.style.display = 'block';
    
    // Update preview with all fields
    const previewName = equipment.name || 'Unknown';
    const previewStatus = equipment.status || 'Active';
    const previewType = equipment.type || 'N/A';
    const previewModel = equipment.model || 'N/A';
    const previewLocation = equipment.location || 'N/A';
    const previewRegNum = equipment.regNum || 'N/A';
    
    document.getElementById('equipmentPreviewName').textContent = previewName;
    document.getElementById('equipmentPreviewStatus').textContent = previewStatus;
    document.getElementById('equipmentPreviewType').textContent = previewType;
    document.getElementById('equipmentPreviewModel').textContent = previewModel;
    document.getElementById('equipmentPreviewLocation').textContent = previewLocation;
    document.getElementById('equipmentPreviewReg').textContent = previewRegNum;
    
    console.log('   Preview updated:', {
        name: previewName,
        type: previewType,
        model: previewModel,
        location: previewLocation,
        regNum: previewRegNum,
        status: previewStatus
    });
    
    // Show and auto-fill equipment detail fields
    const equipmentDetailsSection = document.getElementById('equipmentDetailsSection');
    equipmentDetailsSection.style.display = 'block';
    
    const vehicleIdValue = equipment.regNum || equipment.id || '';
    const vehicleNameValue = equipment.name || '';
    
    document.getElementById('vehicleId').value = vehicleIdValue;
    document.getElementById('vehicleName').value = vehicleNameValue;
    
    console.log('   Form fields updated:', {
        vehicleId: vehicleIdValue,
        vehicleName: vehicleNameValue
    });
    
    closeEquipmentModal();
}

// Image Upload Functions
function handleImageUpload(index) {
    const fileInput = document.getElementById(`imageInput${index}`);
    const file = fileInput.files[0];
    
    if (file) {
        if (!file.type.startsWith('image/')) {
            showNotification('Please select an image file', 'error');
            return;
        }
        
        if (file.size > 5 * 1024 * 1024) {
            showNotification('Image size should be less than 5MB', 'error');
            return;
        }
        
        const reader = new FileReader();
        reader.onload = function(e) {
            incidentImages[index - 1] = {
                file: file,
                dataUrl: e.target.result,
                name: file.name
            };
            showImagePreview(index, e.target.result);
        };
        reader.readAsDataURL(file);
    }
}

function showImagePreview(index, dataUrl) {
    const imageBox = document.getElementById(`imageBox${index}`);
    imageBox.classList.add('has-image');
    imageBox.onclick = null;
    imageBox.innerHTML = `
        <img src="${dataUrl}" class="image-preview" alt="Incident Image ${index}">
        <button class="image-remove-btn" onclick="removeImage(${index}, event)">
            <i class="fas fa-times"></i>
        </button>
    `;
}

function removeImage(index, event) {
    event.stopPropagation();
    incidentImages[index - 1] = null;
    const imageBox = document.getElementById(`imageBox${index}`);
    const fileInput = document.getElementById(`imageInput${index}`);
    fileInput.value = '';
    
    imageBox.classList.remove('has-image');
    imageBox.onclick = () => document.getElementById(`imageInput${index}`).click();
    imageBox.innerHTML = `
        <div class="image-upload-content">
            <i class="fas fa-camera"></i>
            <div class="image-upload-text">Image ${index}</div>
            <div class="image-upload-hint">Click to upload</div>
        </div>
        <input type="file" id="imageInput${index}" accept="image/*" style="display: none;" onchange="handleImageUpload(${index})">
    `;
}

// Send Incident Notifications (Push, Email, In-App)
async function sendIncidentNotifications(incidentData, pdfUrl) {
    try {
        console.log('🔔 Sending incident notifications...');
        
        const createdByName = incidentData.personInvolved || 'Web Admin';
        const reporterEmail = ''; // Web admin doesn't have email in this context
        
        const notificationPayload = {
            incidentData: {
                _id: incidentData._id,
                id: incidentData.id,
                incidentNumber: incidentData.incidentNumber,
                severity: incidentData.severity,
                incidentCategory: incidentData.incidentCategory,
                country: incidentData.country,
                project: incidentData.project,
                incidentArea: incidentData.incidentArea
            },
            createdByName: createdByName,
            reporterEmail: reporterEmail,
            pdfUrl: pdfUrl
        };
        
        console.log('📤 Notification payload:', notificationPayload);
        
        const response = await fetch(`${NOTIFICATIONS_API_URL}/api/incident-notifications/new-incident`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(notificationPayload)
        });
        
        if (response.ok) {
            const result = await response.json();
            console.log('✅ Notifications sent successfully!');
            console.log('📊 Recipients:', result.recipients?.total || 0);
            console.log('📱 Push:', result.results?.push?.sent || 0);
            console.log('📧 Emails:', result.results?.emails?.sent || 0);
            return result;
        } else {
            console.warn('⚠️ Notification API returned error:', response.status);
            return { success: false };
        }
    } catch (error) {
        console.error('❌ Error sending notifications:', error);
        // Don't throw - notifications are non-critical
        return { success: false, error: error.message };
    }
}

// Validate Form
function validateForm() {
    let isValid = true;
    
    // Clear previous errors
    document.querySelectorAll('.error-message').forEach(error => {
        error.classList.remove('show');
    });
    
    // Validate country
    if (!document.getElementById('country').value) {
        document.getElementById('countryError').classList.add('show');
        isValid = false;
    }
    
    // Validate project
    if (!document.getElementById('project').value) {
        document.getElementById('projectError').classList.add('show');
        isValid = false;
    }
    
    // Validate incident date
    if (!document.getElementById('incidentDate').value) {
        document.getElementById('incidentDateError').classList.add('show');
        isValid = false;
    }
    
    // Validate category
    if (!document.getElementById('incidentCategory').value) {
        document.getElementById('categoryError').classList.add('show');
        isValid = false;
    }
    
    // Validate weather
    if (!document.getElementById('weatherCondition').value) {
        document.getElementById('weatherError').classList.add('show');
        isValid = false;
    }
    
    // Validate incident types
    const selectedTypes = document.querySelectorAll('input[name="incidentTypes[]"]:checked');
    if (selectedTypes.length === 0) {
        document.getElementById('typesError').classList.add('show');
        isValid = false;
    }
    
    // Validate severity
    if (!selectedSeverity) {
        document.getElementById('severityError').classList.add('show');
        isValid = false;
    }
    
    return isValid;
}

// Generate Report and Save to MongoDB
async function generateReport() {
    if (!validateForm()) {
        showNotification('Please fill in all required fields', 'error');
        return;
    }
    
    // Show loading modal
    showLoadingModal();
    
    try {
        const adminUid = ADMIN_UID;
        const incidentNumber = currentIncidentNumber || document.getElementById('incidentNumber').textContent;
        const createdAt = new Date().toISOString();
        const incidentDate = new Date(document.getElementById('incidentDate').value).toISOString();
        
        // Get selected incident types as object with boolean values
        const selectedIncidents = {};
        const allIncidentTypes = [
            'Brake Failure', 'Chemicals Spill', 'Collision', 'Drowned in Water',
            'Electrical Failure', 'Engine Failure', 'Equipment Failure', 'Explosion/fire',
            'Fall Attachments', 'Fall from elevation', 'Leakage in Vehicle', 'Loss of Stability',
            'Oil Spill', 'Person hit by Vehicle', 'Steering Failure', 'Stuck',
            'Swamping', 'Tire Burst', 'Turned Over', 'Other'
        ];
        
        allIncidentTypes.forEach(type => {
            const checkbox = document.querySelector(`input[name="incidentTypes[]"][value="${type}"]`);
            selectedIncidents[type] = checkbox ? checkbox.checked : false;
        });
        
        // Create complete incident data for MongoDB
        const incidentData = {
            _id: incidentNumber,
            id: incidentNumber,
            adminUid: adminUid,
            incidentNumber: incidentNumber,
            incidentDate: incidentDate,
            country: document.getElementById('country').value,
            project: document.getElementById('project').value,
            incidentArea: document.getElementById('incidentArea').value || '',
            incidentCategory: document.getElementById('incidentCategory').value,
            weatherCondition: document.getElementById('weatherCondition').value || '',
            severity: selectedSeverity,
            selectedIncidents: selectedIncidents,
            involvesEquipment: involvesEquipment === 'yes',
            vehicleId: document.getElementById('vehicleId') ? document.getElementById('vehicleId').value : '',
            vehicleName: document.getElementById('vehicleName') ? document.getElementById('vehicleName').value : '',
            personInvolved: document.getElementById('reportedBy').value || '',
            coordinates: '',
            gpsAddress: '',
            otherIncidentNote: '',
            createdAt: createdAt,
            isArchived: false,
            step1: {
                adminUid: adminUid,
                incidentNumber: incidentNumber,
                incidentDate: incidentDate,
                country: document.getElementById('country').value,
                project: document.getElementById('project').value,
                incidentArea: document.getElementById('incidentArea').value || '',
                incidentCategory: document.getElementById('incidentCategory').value,
                weatherCondition: document.getElementById('weatherCondition').value || '',
                severity: selectedSeverity,
                selectedIncidents: selectedIncidents,
                involvesEquipment: involvesEquipment === 'yes',
                vehicleId: document.getElementById('vehicleId') ? document.getElementById('vehicleId').value : '',
                vehicleName: document.getElementById('vehicleName') ? document.getElementById('vehicleName').value : '',
                personInvolved: document.getElementById('reportedBy').value || '',
                coordinates: '',
                gpsAddress: '',
                createdAt: createdAt
            },
            step2: {
                incidentCause: document.getElementById('incidentCause').value || '',
                equipmentDamage: document.getElementById('equipmentDamage').value || '',
                additionalComment: document.getElementById('additionalComment').value || '',
                immediateCorrectiveAction: document.getElementById('immediateCorrectiveAction').value || '',
                incidentImages: []
            }
        };
        
        console.log('📋 Incident Data to be saved:', incidentData);
        
        // 1. Save incident to MongoDB first (without files)
        console.log('Step 1/4: Saving incident to MongoDB...');
        await saveIncidentToMongoDB(incidentData);
        console.log('✅ Step 1 complete');
        
        // 2. Upload images if any
        if (incidentImages.some(img => img !== null)) {
            console.log('Step 2/4: Uploading images...');
            const imageUrls = await uploadImagesToMongoDB(incidentNumber, adminUid);
            incidentData.step2.incidentImages = imageUrls;
            console.log('✅ Step 2 complete:', imageUrls.length, 'images');
        } else {
            console.log('Step 2/4: No images to upload');
        }
        
        // 3. Generate and upload PDF
        console.log('Step 3/4: Generating and uploading PDF...');
        const pdfUrl = await generateAndUploadPDFToMongoDB(incidentData, adminUid, incidentNumber);
        incidentData.pdfDownloadUrl = pdfUrl;
        incidentData.step2.pdfDownloadUrl = pdfUrl;
        console.log('✅ Step 3 complete');
        
        // 4. Update incident with file URLs
        console.log('Step 4/4: Updating incident with file URLs...');
        await updateIncidentInMongoDB(incidentData, adminUid, incidentNumber);
        console.log('✅ Step 4 complete');
        
        // 5. Send notifications (emails & push notifications) - Fire and forget
        console.log('Step 5/5: Sending notifications...');
        sendIncidentNotifications(incidentData, pdfUrl).catch(err => {
            console.warn('⚠️ Notifications failed (non-critical):', err.message);
        });
        
        console.log('');
        console.log('✅ ========================================');
        console.log('✅ INCIDENT REPORT SAVED SUCCESSFULLY!');
        console.log('✅ ========================================');
        console.log('🆔 Incident Number:', incidentNumber);
        console.log('🏢 Country:', incidentData.country);
        console.log('📂 Project:', incidentData.project);
        console.log('⚠️ Severity:', incidentData.severity);
        console.log('📄 PDF URL:', pdfUrl);
        console.log('📸 Images Uploaded:', incidentData.step2.incidentImages.length);
        console.log('✅ ========================================');
        console.log('');
        
        generatedPDFUrl = pdfUrl;
        
        // Hide loading modal
        hideLoadingModal();
        
        // Show success notification
        showNotification('✅ Incident Report Saved & Notifications Sent!', 'success');
        
        // Show success modal
        setTimeout(() => {
            showSuccessModal();
        }, 300);
        
    } catch (error) {
        console.error('❌ Error generating report:', error);
        console.error('Error stack:', error.stack);
        hideLoadingModal();
        
        // Provide detailed error message
        let errorMsg = 'Error generating report';
        if (error.message.includes('Step 1')) {
            errorMsg = 'Failed to save incident to database';
        } else if (error.message.includes('Step 2') || error.message.includes('images')) {
            errorMsg = 'Failed to upload images';
        } else if (error.message.includes('Step 3') || error.message.includes('PDF')) {
            errorMsg = 'Failed to generate or upload PDF';
        } else if (error.message.includes('Step 4')) {
            errorMsg = 'Failed to update incident record';
        } else {
            errorMsg = error.message;
        }
        
        showNotification(errorMsg, 'error');
    }
}

// Save incident to MongoDB
async function saveIncidentToMongoDB(incidentData) {
    try {
        console.log('💾 Saving incident to MongoDB...');
        console.log('   Incident Number:', incidentData.incidentNumber);
        console.log('   API Endpoint:', `${API_BASE_URL}/incident-reports`);
        
        const response = await fetch(`${API_BASE_URL}/incident-reports`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(incidentData)
        });
        
        console.log('   Response status:', response.status);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('❌ Save failed:', errorText);
            throw new Error(`Failed to save - HTTP ${response.status}`);
        }
        
        const result = await response.json();
        console.log('   Server response:', result);
        
        if (!result.success) {
            throw new Error(result.error || 'Failed to save incident report');
        }
        
        console.log('✅ Incident saved to MongoDB successfully!');
        
        return result.data;
    } catch (error) {
        console.error('❌ Error saving to MongoDB:', error);
        throw new Error(`Step 1 Save Error: ${error.message}`);
    }
}

// Update incident in MongoDB
async function updateIncidentInMongoDB(incidentData, adminUid, incidentNumber) {
    try {
        console.log('🔄 Updating incident in MongoDB with file URLs...');
        console.log('   Incident Number:', incidentNumber);
        console.log('   PDF URL:', incidentData.pdfDownloadUrl);
        console.log('   Images:', incidentData.step2.incidentImages.length);
        
        const response = await fetch(`${API_BASE_URL}/incident-reports/${adminUid}/${incidentNumber}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(incidentData)
        });
        
        console.log('   Response status:', response.status);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('❌ Update failed:', errorText);
            throw new Error(`Failed to update - HTTP ${response.status}`);
        }
        
        const result = await response.json();
        console.log('   Server response:', result);
        
        if (!result.success) {
            throw new Error(result.error || 'Failed to update incident report');
        }
        
        console.log('✅ Incident updated successfully with file URLs!');
        return result.data;
    } catch (error) {
        console.error('❌ Error updating incident:', error);
        throw new Error(`Step 4 Update Error: ${error.message}`);
    }
}

// Upload images to MongoDB server
async function uploadImagesToMongoDB(incidentNumber, adminUid) {
    try {
        console.log('📸 Uploading images to MongoDB server...');
        console.log('   Incident Number:', incidentNumber);
        
        const formData = new FormData();
        
        // Add all non-null images to formData
        let imageCount = 0;
        for (let i = 0; i < incidentImages.length; i++) {
            if (incidentImages[i] && incidentImages[i].file) {
                formData.append('images', incidentImages[i].file, `incident_image_${i}_${Date.now()}.jpg`);
                imageCount++;
                console.log(`   Added image ${i + 1}:`, incidentImages[i].name);
            }
        }
        
        if (imageCount === 0) {
            console.log('⚠️ No images to upload');
            return [];
        }
        
        console.log(`   Uploading ${imageCount} image(s)...`);
        
        const response = await fetch(`${API_BASE_URL}/incident-reports/${adminUid}/${incidentNumber}/upload-images`, {
            method: 'POST',
            body: formData
        });
        
        console.log('   Response status:', response.status);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('❌ Image upload failed:', errorText);
            throw new Error(`Image upload failed - HTTP ${response.status}`);
        }
        
        const result = await response.json();
        console.log('   Server response:', result);
        
        if (!result.success) {
            throw new Error(result.error || 'Failed to upload images');
        }
        
        console.log(`✅ ${result.imageUrls.length} images uploaded successfully!`);
        return result.imageUrls;
    } catch (error) {
        console.error('❌ Error uploading images:', error);
        showNotification('Warning: Image upload failed. Report will be saved without images.', 'warning');
        return [];
    }
}

// Generate PDF and upload to MongoDB server
async function generateAndUploadPDFToMongoDB(incidentData, adminUid, incidentNumber) {
    try {
        console.log('📄 Step 3.1: Generating PDF HTML template...');
        
        // Generate PDF HTML
        const htmlContent = generatePDFHTML(incidentData.step1, incidentData.step2);
        console.log('✅ HTML template generated');
        console.log('   HTML length:', htmlContent.length, 'characters');
        
        // Create a temporary div to hold the HTML content
        console.log('📄 Step 3.2: Creating temporary DOM element...');
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = htmlContent;
        
        // Make sure the element is rendered but invisible
        tempDiv.style.position = 'fixed';
        tempDiv.style.left = '-9999px';
        tempDiv.style.top = '0';
        tempDiv.style.width = '900px';
        tempDiv.style.visibility = 'hidden';
        tempDiv.style.pointerEvents = 'none';
        
        document.body.appendChild(tempDiv);
        console.log('✅ DOM element created and appended');
        
        // Wait for DOM to fully render
        await new Promise(resolve => setTimeout(resolve, 100));
        
        // Configure html2pdf options with better settings
        const opt = {
            margin: [10, 10, 10, 10],
            filename: `IncidentReport_${incidentNumber}.pdf`,
            image: { type: 'jpeg', quality: 0.98 },
            html2canvas: { 
                scale: 2, 
                useCORS: true, 
                logging: true,
                letterRendering: true,
                allowTaint: false,
                backgroundColor: '#ffffff'
            },
            jsPDF: { 
                unit: 'mm', 
                format: 'a4', 
                orientation: 'portrait',
                compress: true
            },
            pagebreak: { mode: ['avoid-all', 'css', 'legacy'] }
        };
        
        // Generate PDF as blob (promise-based)
        console.log('📄 Step 3.3: Converting HTML to PDF...');
        console.log('   Using html2pdf with options:', opt);
        
        const pdfBlob = await new Promise((resolve, reject) => {
            const timeoutId = setTimeout(() => {
                reject(new Error('PDF generation timeout after 30 seconds'));
            }, 30000);
            
            try {
                // Use the first child (the styled div we created)
                const contentDiv = tempDiv.firstElementChild || tempDiv;
                
                console.log('   Converting element:', contentDiv.tagName);
                console.log('   Element innerHTML length:', contentDiv.innerHTML?.length || 0);
                console.log('   Element has children:', contentDiv.children?.length || 0);
                
                // Verify content before PDF generation
                if (!contentDiv.innerHTML || contentDiv.innerHTML.length < 100) {
                    clearTimeout(timeoutId);
                    reject(new Error('PDF content is empty or too small'));
                    return;
                }
                
                html2pdf()
                    .set(opt)
                    .from(contentDiv)
                    .toPdf()
                    .output('blob')
                    .then(blob => {
                        clearTimeout(timeoutId);
                        console.log('✅ PDF blob generated successfully!');
                        console.log('   Blob size:', blob.size, 'bytes (', (blob.size / 1024).toFixed(2), 'KB )');
                        console.log('   Blob type:', blob.type);
                        
                        if (blob.size < 1000) {
                            console.warn('⚠️ PDF size is suspiciously small:', blob.size, 'bytes');
                        }
                        
                        resolve(blob);
                    })
                    .catch(err => {
                        clearTimeout(timeoutId);
                        console.error('❌ html2pdf conversion error:', err);
                        reject(err);
                    });
            } catch (err) {
                clearTimeout(timeoutId);
                console.error('❌ html2pdf setup error:', err);
                reject(err);
            }
        });
        
        // Remove temporary div after PDF generation
        console.log('📄 Step 3.3.5: Cleaning up temporary element...');
        try {
            if (tempDiv && tempDiv.parentNode) {
                document.body.removeChild(tempDiv);
                console.log('   ✅ Temporary element removed');
            }
        } catch (cleanupErr) {
            console.warn('⚠️ Cleanup warning:', cleanupErr.message);
        }
        
        console.log('📄 Step 3.4: Uploading PDF to server...');
        console.log('   PDF Size:', (pdfBlob.size / 1024).toFixed(2), 'KB');
        
        // Validate PDF blob
        if (pdfBlob.size < 1000) {
            console.error('❌ PDF too small:', pdfBlob.size, 'bytes - likely empty or corrupted');
            throw new Error('Generated PDF is too small or empty. Please check browser console for details.');
        }
        
        console.log('   ✅ PDF validation passed');
        
        // Create FormData for PDF upload
        const formData = new FormData();
        formData.append('pdf', pdfBlob, `IncidentReport_${incidentNumber}_${Date.now()}.pdf`);
        
        // Upload PDF to server
        const response = await fetch(`${API_BASE_URL}/incident-reports/${adminUid}/${incidentNumber}/upload-pdf`, {
            method: 'POST',
            body: formData
        });
        
        console.log('   Response status:', response.status);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('❌ PDF upload failed:', errorText);
            throw new Error(`PDF upload failed - HTTP ${response.status}`);
        }
        
        const result = await response.json();
        console.log('   Server response:', result);
        
        if (!result.success) {
            throw new Error(result.error || 'Failed to upload PDF');
        }
        
        console.log('✅ PDF uploaded successfully!');
        console.log('📎 PDF URL:', result.pdfUrl);
        
        return result.pdfUrl;
    } catch (error) {
        console.error('❌ Error in PDF generation/upload:', error);
        console.error('   Error type:', error.constructor.name);
        console.error('   Error message:', error.message);
        console.error('   Error stack:', error.stack);
        throw new Error(`Step 3 PDF Error: ${error.message}`);
    }
}

// Generate PDF HTML (Matching exact demo design) - Returns styled container div
function generatePDFHTML(step1Data, step2Data) {
    const formatDate = (date) => {
        if (!date) return '-';
        const d = new Date(date);
        const day = String(d.getDate()).padStart(2, '0');
        const month = String(d.getMonth() + 1).padStart(2, '0');
        const year = d.getFullYear();
        return `${day}/${month}/${year}`;
    };

    const formatTime = (date) => {
        if (!date) return '-';
        const d = new Date(date);
        let hours = d.getHours();
        const minutes = String(d.getMinutes()).padStart(2, '0');
        const ampm = hours >= 12 ? 'PM' : 'AM';
        hours = hours % 12;
        hours = hours ? hours : 12;
        return `${String(hours).padStart(2, '0')}:${minutes} ${ampm}`;
    };

    const dashIfNA = (val) => {
        if (val === undefined || val === null || val === '' || val === 'N/A') return '-';
        return val;
    };

    // Generate incident types checkboxes (4 columns per row)
    const allIncidentTypes = [
        'Brake Failure', 'Chemicals Spill', 'Collision', 'Drowned in Water',
        'Electrical Failure', 'Engine Failure', 'Equipment Failure', 'Explosion/fire',
        'Fall Attachments', 'Fall from elevation', 'Leakage in Vehicle', 'Loss of Stability',
        'Oil Spill', 'Person hit by Vehicle', 'Steering Failure', 'Stuck',
        'Swamping', 'Tire Burst', 'Turned Over', 'Other'
    ];
    
    let incidentTypesHtml = '';
    for (let i = 0; i < allIncidentTypes.length; i += 4) {
        incidentTypesHtml += '<tr>';
        for (let j = 0; j < 4; j++) {
            const idx = i + j;
            if (idx < allIncidentTypes.length) {
                const type = allIncidentTypes[idx];
                const isChecked = step1Data.selectedIncidents && step1Data.selectedIncidents[type] === true;
                incidentTypesHtml += `<td><input type="checkbox" ${isChecked ? 'checked' : ''} /> ${type}</td>`;
            } else {
                incidentTypesHtml += '<td></td>';
            }
        }
        incidentTypesHtml += '</tr>';
    }

    // Severity checkboxes
    const severityOptions = ['High', 'Medium', 'Low'];
    let severityHtml = '<tr>';
    severityOptions.forEach((option) => {
        severityHtml += `<td><input type="checkbox" ${step1Data.severity === option ? 'checked' : ''} /> ${option}</td>`;
    });
    severityHtml += '</tr>';

    // Images HTML - Show actual uploaded images or placeholders
    let imagesHtml = '';
    const validImages = step2Data.incidentImages ? step2Data.incidentImages.filter(url => url && url.trim() !== '') : [];
    
    if (validImages.length > 0) {
        imagesHtml = '<div style="display:flex;flex-wrap:wrap;gap:8px;">';
        validImages.forEach((url, index) => {
            imagesHtml += `<div style="width:120px;height:90px;border:1px solid #ccc;display:flex;align-items:center;justify-content:center;margin:4px;"><img src="${url}" style="max-width:100%;max-height:100%;object-fit:contain;" crossorigin="anonymous"/></div>`;
        });
        imagesHtml += '</div>';
    } else {
        // Show placeholder boxes like in demo
        imagesHtml = '<div style="display:flex;flex-wrap:wrap;gap:8px;">';
        for (let i = 1; i <= 3; i++) {
            imagesHtml += `<div style="width:120px;height:90px;border:1px solid #ccc;display:flex;align-items:center;justify-content:center;margin:4px;background-color:#f5f5f5;color:#999;font-size:12px;">Image ${i}</div>`;
        }
        imagesHtml += '</div>';
    }

    const vehicleName = step1Data.vehicleName || '-';
    const vehicleId = step1Data.vehicleId || '-';

    // Return styled div (not full HTML doc) - html2pdf works better with div elements
    return `
<div style="font-family: Arial, sans-serif; max-width: 900px; margin: 0 auto; padding: 20px; background: white;">
  <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; border-bottom: 2px solid #333; padding-bottom: 10px;">
    <h1 style="font-size: 22px; font-weight: bold; margin: 0;">Incident Report</h1>
    <div style="width: 150px; height: 48px; display: flex; align-items: center; justify-content: center;">
      <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo" style="width: 100%; height: 100%; object-fit: contain;"/>
    </div>
  </div>
  
  <table style="width: 100%; border-collapse: collapse; margin-bottom: 15px;">
    <tr>
      <th colspan="2" style="border: 1px solid #ccc; padding: 8px; background-color: #f2f2f2; font-weight: bold; font-size: 13px;">Vehicle Details</th>
      <th colspan="2" style="border: 1px solid #ccc; padding: 8px; background-color: #f2f2f2; font-weight: bold; font-size: 13px;">Incident Details</th>
    </tr>
    <tr>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">Vehicle Category</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">${dashIfNA(vehicleName)}</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">Incident Number</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">${dashIfNA(step1Data.incidentNumber)}</td>
    </tr>
    <tr>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">Vehicle Name</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">${dashIfNA(vehicleName)}</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">Incident Date</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">${formatDate(step1Data.incidentDate)}</td>
    </tr>
    <tr>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">Vehicle #</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">${dashIfNA(vehicleId)}</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">Incident Time</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">${formatTime(step1Data.incidentDate)}</td>
    </tr>
    <tr>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">VIN #</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">-</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">Location</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">${dashIfNA(step1Data.project)}</td>
    </tr>
    <tr>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">Operator</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">${dashIfNA(step1Data.personInvolved)}</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">Incident Area</td>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">${dashIfNA(step1Data.incidentArea)}</td>
    </tr>
    <tr>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">GPS Coordinates</td>
      <td colspan="3" style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">${dashIfNA(step1Data.coordinates)}</td>
    </tr>
    <tr>
      <td style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">GPS Address</td>
      <td colspan="3" style="border: 1px solid #ccc; padding: 8px; font-size: 13px;">${dashIfNA(step1Data.gpsAddress)}</td>
    </tr>
  </table>
  
  <div style="font-weight: bold; font-size: 14px; margin: 15px 0 8px; background-color: #e6e6e6; padding: 6px; border: 1px solid #ccc;">Types of Incident</div>
  <table style="width: 100%; border-collapse: collapse; margin-bottom: 15px;">
    ${incidentTypesHtml}
  </table>
  
  <div style="font-weight: bold; font-size: 14px; margin: 15px 0 8px; background-color: #e6e6e6; padding: 6px; border: 1px solid #ccc;">Incident Severity Rating</div>
  <table style="width: 100%; border-collapse: collapse; margin-bottom: 15px;">
    ${severityHtml}
  </table>
  
  <div style="font-weight: bold; font-size: 14px; margin: 15px 0 8px; background-color: #e6e6e6; padding: 6px; border: 1px solid #ccc;">How incident occurred and what were the causes?</div>
  <div style="border: 1px solid #ccc; padding: 10px; min-height: 40px; font-size: 13px; margin-bottom: 15px; background: white;">${dashIfNA(step2Data.incidentCause)}</div>
  
  <div style="font-weight: bold; font-size: 14px; margin: 15px 0 8px; background-color: #e6e6e6; padding: 6px; border: 1px solid #ccc;">What broke on the equipment, list all damages?</div>
  <div style="border: 1px solid #ccc; padding: 10px; min-height: 40px; font-size: 13px; margin-bottom: 15px; background: white;">${dashIfNA(step2Data.equipmentDamage)}</div>
  
  <div style="font-weight: bold; font-size: 14px; margin: 15px 0 8px; background-color: #e6e6e6; padding: 6px; border: 1px solid #ccc;">Additional Comment</div>
  <div style="border: 1px solid #ccc; padding: 10px; min-height: 40px; font-size: 13px; margin-bottom: 15px; background: white;">${dashIfNA(step2Data.additionalComment)}</div>
  
  <div style="font-weight: bold; font-size: 14px; margin: 15px 0 8px; background-color: #e6e6e6; padding: 6px; border: 1px solid #ccc;">Immediate Corrective Action Taken</div>
  <div style="border: 1px solid #ccc; padding: 10px; min-height: 40px; font-size: 13px; margin-bottom: 15px; background: white;">${dashIfNA(step2Data.immediateCorrectiveAction)}</div>
  
  <div style="font-weight: bold; font-size: 14px; margin: 15px 0 8px; background-color: #e6e6e6; padding: 6px; border: 1px solid #ccc;">Incident Images</div>
  ${imagesHtml}
  
  <div style="text-align: right; font-size: 13px; margin-top: 20px;">
    <span style="border-bottom: 2px solid #000; padding-bottom: 2px; display: inline-block;">${dashIfNA(step1Data.personInvolved)}</span><br>
    <span style="display: inline-block; margin-top: 5px;">Reported By</span>
  </div>
  
  <div style="text-align: center; margin-top: 20px; font-size: 11px; color: #000; border-top: 2px solid #000; padding-top: 8px;">
    Prepared By: Titan Drilling
  </div>
</div>`;
}

// Success Modal Functions
function viewIncidentPDF() {
    if (generatedPDFUrl) {
        window.open(generatedPDFUrl, '_blank');
    } else {
        showNotification('PDF not available', 'error');
    }
}

function goBackToIncidentReports() {
    hideSuccessModal();
    window.location.href = "{{ route('inspection.incident_report') }}";
}

// Notification function
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.style.cssText = `
        position: fixed;
        top: 80px;
        right: 20px;
        padding: 12px 20px;
        border-radius: 8px;
        color: white;
        font-weight: 500;
        font-size: 14px;
        z-index: 10001;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        font-family: 'Inter', sans-serif;
    `;
    
    if (type === 'success') {
        notification.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
    } else if (type === 'error') {
        notification.style.background = 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)';
    } else if (type === 'warning') {
        notification.style.background = 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)';
            } else {
        notification.style.background = 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)';
    }
    
    notification.textContent = message;
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.opacity = '0';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

// Close modal when clicking outside (handled by openModal function)
// This is now integrated into the openModal function above
</script>
@endsection

