@extends('welcome')

@section('title', 'DVIR DOT Inspection')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="dvir-inspection-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="javascript:history.back()" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">DVIR DOT Inspection</h1>
            </div>
            <div class="header-right">
                <div class="search-container">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="search-icon">
                        <circle cx="11" cy="11" r="8"/>
                        <path d="M21 21l-4.35-4.35"/>
                    </svg>
                    <input type="text" class="search-input" placeholder="Search DVIR reports..." id="searchInput">
                </div>
                <a href="/dvir-dot-inspection/create-dvir-report-step1" class="create-btn">
                    <i class="fas fa-plus"></i>
                    Create New DVIR
                </a>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="table-section">
            <div class="table-container">
            <table class="dvir-table" id="dvirTable">
                <thead class="table-header">
                    <tr>
                        <th class="sno-column">S.No</th>
                        <th class="action-column">Action</th>
                        <th>Report Number</th>
                        <th>Date</th>
                        <th>Vehicle</th>
                        <th>Location</th>
                        <th>Inspector</th>
                        <th>Status</th>
                        <th>Fault</th>
                    </tr>
                </thead>
                <tbody class="table-body" id="dvirTableBody">
                    <tr class="loading-row">
                        <td colspan="9" class="loading-state">
                            <div class="loading-content">
                                <div class="loading-spinner"></div>
                                <h3>Loading DVIR Reports...</h3>
                                <p>Please wait while we fetch your data from MongoDB.</p>
                                </div>
                            </td>
                        </tr>
                </tbody>
            </table>
            </div>
        </div>
    </div>

    <!-- Pagination Section -->
    <div class="pagination-section" id="paginationSection" style="display: none;">
        <div class="pagination-content">
            <div class="pagination-info" id="paginationInfo">
                <span class="pagination-text">Showing 0 to 0 of 0 entries</span>
            </div>
            
            <div class="pagination-controls">
                <div class="pagination-buttons" id="paginationButtons">
                    <!-- Pagination buttons will be generated here -->
                </div>
            </div>
            </div>
        </div>

    <!-- Loading Spinner -->
    <div id="loadingSpinner" class="loading-section d-none">
        <div class="loading-content">
            <div class="spinner">
                <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="spinner-icon">
                    <line x1="12" y1="2" x2="12" y2="6"/>
                    <line x1="12" y1="18" x2="12" y2="22"/>
                    <line x1="4.93" y1="4.93" x2="7.76" y2="7.76"/>
                    <line x1="16.24" y1="16.24" x2="19.07" y2="19.07"/>
                    <line x1="2" y1="12" x2="6" y2="12"/>
                    <line x1="18" y1="12" x2="22" y2="12"/>
                    <line x1="4.93" y1="19.07" x2="7.76" y2="16.24"/>
                    <line x1="16.24" y1="7.76" x2="19.07" y2="4.93"/>
                </svg>
            </div>
            <h3>Loading DVIR Reports</h3>
            <p>Fetching the latest inspection data...</p>
        </div>
    </div>

    <!-- Error Message -->
    <div id="errorMessage" class="error-section d-none">
        <div class="error-content">
            <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round" class="error-icon">
                <circle cx="12" cy="12" r="10"/>
                <line x1="15" y1="9" x2="9" y2="15"/>
                <line x1="9" y1="9" x2="15" y2="15"/>
            </svg>
            <h3>Error Loading Data</h3>
            <p id="errorText">An error occurred while loading DVIR reports.</p>
            <button class="btn-retry" onclick="refreshData()">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <polyline points="23 4 23 10 17 10"/>
                    <polyline points="1 20 1 14 7 14"/>
                    <path d="M20.49 9A9 9 0 0 0 5.64 5.64L1 10m22 4l-4.64 4.36A9 9 0 0 1 3.51 15"/>
                </svg>
                <span>Try Again</span>
            </button>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div id="deleteConfirmModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content" style="max-width: 500px;">
        <div class="custom-modal-header">
            <h3>Confirm Delete</h3>
            <button type="button" class="close-btn" onclick="closeDeleteConfirmModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div style="text-align: center; padding: 20px;">
                <i class="fas fa-exclamation-triangle" style="font-size: 48px; color: #dc3545; margin-bottom: 20px;"></i>
                <h4 style="margin-bottom: 10px; color: #374151;">Are you sure?</h4>
                <p style="color: #6b7280; margin-bottom: 0;">This DVIR report will be permanently deleted. This action cannot be undone.</p>
            </div>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn-secondary" onclick="closeDeleteConfirmModal()">Cancel</button>
            <button type="button" class="btn-danger" onclick="confirmDeleteDvir()">Delete Report</button>
        </div>
    </div>
</div>

<!-- PDF Viewer Modal -->
<div id="pdfViewerModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content pdf-modal-large">
        <div class="custom-modal-header">
            <h3 id="pdfModalTitle">View DVIR Report PDF</h3>
            <button type="button" class="close-btn" onclick="closePdfModal()">&times;</button>
        </div>
        <div class="custom-modal-body pdf-modal-body">
            <div class="pdf-loader" id="pdfLoader">
                <div class="pdf-loader-spinner"></div>
                <p>Loading PDF...</p>
            </div>
            <iframe id="pdfIframe" style="width: 100%; height: 600px; border: none;"></iframe>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn-secondary" onclick="closePdfModal()">Close</button>
            <button type="button" class="btn-primary" onclick="downloadCurrentPdf()">
                <i class="fas fa-download"></i> Download PDF
            </button>
        </div>
    </div>
</div>

<script>
// DVIR DOT Inspection with MongoDB Integration
(function() {
    'use strict';
    
    // Configuration object
    const CONFIG = {
        mongodb: {
            reportsUrl: 'https://api.titandrillingzm.com:6006/reports',
            deleteUrl: 'https://api.titandrillingzm.com:6006/reports'
        },
        pagination: {
            itemsPerPage: 10,
            maxVisiblePages: 5
        }
    };
    
    // Global state
    let allReports = [];
    let currentPage = 1;
    let totalPages = 0;
    let filteredReports = [];
    
    // Cache DOM elements
    let elements = {};
    
    function initElements() {
        elements = {
            searchInput: document.getElementById('searchInput'),
            dvirTableBody: document.getElementById('dvirTableBody'),
            paginationSection: document.getElementById('paginationSection'),
            paginationInfo: document.getElementById('paginationInfo'),
            paginationButtons: document.getElementById('paginationButtons')
        };
    }
    
    // Load DVIR reports from MongoDB
    async function loadDVIRReports() {
        try {
            console.log('🔄 Loading DVIR reports from MongoDB...');
            
            const response = await fetch(CONFIG.mongodb.reportsUrl);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const result = await response.json();
            console.log('✅ MongoDB DVIR response:', result);
            
            if (result.success && Array.isArray(result.data)) {
                // Sort by createdAt descending (newest first)
                allReports = result.data.sort((a, b) => {
                    const dateA = new Date(b.createdAt || b.step3?.createdAt || 0);
                    const dateB = new Date(a.createdAt || a.step3?.createdAt || 0);
                    return dateA - dateB;
                });
                
                filteredReports = [...allReports];
                
                console.log(`✅ Loaded ${allReports.length} DVIR reports`);
                
                // Calculate pagination
                totalPages = Math.ceil(filteredReports.length / CONFIG.pagination.itemsPerPage);
                currentPage = 1;
                
                // Display data
                displayCurrentPage();
                updatePagination();
                
                } else {
                throw new Error('Invalid data format from MongoDB');
            }
        } catch (error) {
            console.error('❌ Error loading DVIR reports:', error);
            displayEmptyState();
        }
    }
    
    // Display current page of reports
    function displayCurrentPage() {
        const tableBody = elements.dvirTableBody;
        if (!tableBody) return;
        
        if (filteredReports.length === 0) {
            displayEmptyState();
            return;
        }
        
        const startIndex = (currentPage - 1) * CONFIG.pagination.itemsPerPage;
        const endIndex = Math.min(startIndex + CONFIG.pagination.itemsPerPage, filteredReports.length);
        const pageReports = filteredReports.slice(startIndex, endIndex);
        
        const rows = pageReports.map((report, index) => {
            const serialNumber = startIndex + index + 1;
            const dvirNumber = report.dvirNumber || report.step3?.dvirNumber || 'N/A';
            const reportDate = formatDate(report.createdAt || report.step3?.createdAt);
            const vehicle = report.step1?.selectedEquipment?.equipmentName || 'N/A';
            const location = `${report.step1?.country || 'N/A'} - ${report.step1?.project || 'N/A'}`;
            const inspector = report.step1?.inspector || 'N/A';
            const faultyItems = report.step3?.faultyItems || 0;
            const status = faultyItems > 0 ? 'Issues Found' : 'Completed';
            const reportId = report._id || report.id || report.reportId;
            
            return `
                <tr class="table-row">
                    <td class="sno-cell">${serialNumber}</td>
                    <td class="action-cell">
                        <div class="action-dropdown">
                            <button class="btn-action" onclick="toggleActionDropdown(event, '${reportId}')">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="12" cy="12" r="1"/>
                                    <circle cx="19" cy="12" r="1"/>
                                    <circle cx="5" cy="12" r="1"/>
                                </svg>
                            </button>
                            <div class="dropdown-menu" id="dropdown-${reportId}">
                                <button class="dropdown-item" onclick="viewPdfInModal('${reportId}')">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                        <polyline points="14,2 14,8 20,8"/>
                                    </svg>
                                    <span>PDF</span>
                                </button>
                                <button class="dropdown-item delete-action" onclick="openDeleteConfirmModal('${reportId}')">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <polyline points="3,6 5,6 21,6"/>
                                        <path d="M19,6v14a2,2 0 0,1 -2,2H7a2,2 0 0,1 -2,-2V6m3,0V4a2,2 0 0,1 2,-2h4a2,2 0 0,1 2,2v2"/>
                                    </svg>
                                    <span>Delete</span>
                                </button>
                            </div>
                        </div>
                    </td>
                    <td class="data-cell">${dvirNumber}</td>
                    <td class="data-cell">${reportDate}</td>
                    <td class="data-cell">${vehicle}</td>
                    <td class="data-cell">${location}</td>
                    <td class="data-cell">${inspector}</td>
                    <td class="data-cell">
                        <span class="status-badge ${status === 'Completed' ? 'completed' : 'issues'}">${status}</span>
                    </td>
                    <td class="data-cell">
                        ${faultyItems > 0 ? 
                            `<div class="fault-info">
                                <span class="fault-badge">${faultyItems} faults</span>
                            </div>` : 
                            `<span class="no-faults">No faults</span>`
                        }
                    </td>
                </tr>
            `;
        }).join('');
        
        tableBody.innerHTML = rows;
        
        // Show pagination if needed
        if (totalPages > 1) {
            elements.paginationSection.style.display = 'block';
        }
    }
    
    // Display empty state
    function displayEmptyState() {
        const tableBody = elements.dvirTableBody;
        if (!tableBody) return;
        
        tableBody.innerHTML = `
            <tr class="empty-row">
                <td colspan="9" class="empty-state">
                    <div class="empty-content">
                        <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round" class="empty-icon">
                            <path d="M9 12l2 2 4-4"/>
                            <path d="M21 12c-1 0-3-1-3-3s2-3 3-3 3 1 3 3-2 3-3 3"/>
                            <path d="M3 12c1 0 3-1 3-3s-2-3-3-3-3 1-3 3 2 3 3 3"/>
                        </svg>
                        <h3>No DVIR Reports Found</h3>
                        <p>No DVIR data is currently available. Please check your connection or try refreshing the page.</p>
                        <button class="btn-refresh-empty" onclick="refreshData()">
                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <polyline points="23 4 23 10 17 10"/>
                                <polyline points="1 20 1 14 7 14"/>
                                <path d="M20.49 9A9 9 0 0 0 5.64 5.64L1 10m22 4l-4.64 4.36A9 9 0 0 1 3.51 15"/>
                            </svg>
                            <span>Refresh Data</span>
                        </button>
                    </div>
                </td>
            </tr>
        `;
        elements.paginationSection.style.display = 'none';
    }
    
    // Format date helper
    function formatDate(dateStr) {
        if (!dateStr) return 'N/A';
        try {
            const date = new Date(dateStr);
            return date.toLocaleDateString('en-GB', {
                day: '2-digit',
                month: 'short',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
        } catch (e) {
            return 'Invalid Date';
        }
    }
    
    // Update pagination controls
    function updatePagination() {
        if (!elements.paginationInfo || !elements.paginationButtons) return;
        
        const startRecord = (currentPage - 1) * CONFIG.pagination.itemsPerPage + 1;
        const endRecord = Math.min(currentPage * CONFIG.pagination.itemsPerPage, filteredReports.length);
        
        // Update info text
        elements.paginationInfo.innerHTML = `
            <span class="pagination-text">Showing ${startRecord} to ${endRecord} of ${filteredReports.length} entries</span>
        `;
        
        // Generate pagination buttons
        let paginationHTML = '';
        
        // Previous button
        paginationHTML += `
            <button class="pagination-btn ${currentPage === 1 ? 'disabled' : ''}" 
                    onclick="goToPage(${currentPage - 1})" 
                    ${currentPage === 1 ? 'disabled' : ''}>
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <polyline points="15,18 9,12 15,6"/>
                </svg>
                Previous
            </button>
        `;
        
        // Page numbers
        const startPage = Math.max(1, currentPage - Math.floor(CONFIG.pagination.maxVisiblePages / 2));
        const endPage = Math.min(totalPages, startPage + CONFIG.pagination.maxVisiblePages - 1);
        
        if (startPage > 1) {
            paginationHTML += `<button class="pagination-btn page-number" onclick="goToPage(1)">1</button>`;
            if (startPage > 2) {
                paginationHTML += `<span class="page-ellipsis">...</span>`;
            }
        }
        
        for (let i = startPage; i <= endPage; i++) {
            paginationHTML += `
                <button class="pagination-btn page-number ${i === currentPage ? 'active' : ''}" 
                        onclick="goToPage(${i})">
                    ${i}
                </button>
            `;
        }
        
        if (endPage < totalPages) {
            if (endPage < totalPages - 1) {
                paginationHTML += `<span class="page-ellipsis">...</span>`;
            }
            paginationHTML += `<button class="pagination-btn page-number" onclick="goToPage(${totalPages})">${totalPages}</button>`;
        }
        
        // Next button
        paginationHTML += `
            <button class="pagination-btn ${currentPage === totalPages ? 'disabled' : ''}" 
                    onclick="goToPage(${currentPage + 1})" 
                    ${currentPage === totalPages ? 'disabled' : ''}>
                Next
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <polyline points="9,18 15,12 9,6"/>
                </svg>
            </button>
        `;
        
        elements.paginationButtons.innerHTML = paginationHTML;
    }
    
    // Go to specific page
    function goToPage(page) {
        if (page < 1 || page > totalPages || page === currentPage) return;
        
        currentPage = page;
        displayCurrentPage();
        updatePagination();
        
        // Scroll to top of table
        window.scrollTo({ top: 0, behavior: 'smooth' });
    }
    
    // Refresh data function
    function refreshData() {
        loadDVIRReports();
    }
    
    // Search functionality
    function initSearch() {
        if (!elements.searchInput) return;
        
        elements.searchInput.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            
            if (!searchTerm) {
                filteredReports = [...allReports];
                } else {
                filteredReports = allReports.filter(report => {
                    const dvirNumber = (report.dvirNumber || report.step3?.dvirNumber || '').toLowerCase();
                    const vehicle = (report.step1?.selectedEquipment?.equipmentName || '').toLowerCase();
                    const inspector = (report.step1?.inspector || '').toLowerCase();
                    const country = (report.step1?.country || '').toLowerCase();
                    const project = (report.step1?.project || '').toLowerCase();
                    
                    return dvirNumber.includes(searchTerm) ||
                           vehicle.includes(searchTerm) ||
                           inspector.includes(searchTerm) ||
                           country.includes(searchTerm) ||
                           project.includes(searchTerm);
                });
            }
            
            // Recalculate pagination
            totalPages = Math.ceil(filteredReports.length / CONFIG.pagination.itemsPerPage);
            currentPage = 1;
            
            displayCurrentPage();
            updatePagination();
        });
    }
    
    // Toggle action dropdown
    function toggleActionDropdown(event, reportId) {
        event.stopPropagation();
        event.preventDefault();
        
        console.log('🔘 Toggle dropdown for report:', reportId);
        
        const dropdown = document.getElementById(`dropdown-${reportId}`);
        const allDropdowns = document.querySelectorAll('.dropdown-menu');
        
        console.log('Dropdown element found:', !!dropdown);
        
        // Close other dropdowns
        allDropdowns.forEach(dd => {
            if (dd.id !== `dropdown-${reportId}`) {
                dd.style.setProperty('display', 'none', 'important');
            }
        });
        
        // Toggle current dropdown with important flag to override CSS
        if (dropdown) {
            const isVisible = dropdown.style.display === 'block';
            console.log('Current display:', isVisible ? 'visible' : 'hidden');
            
            if (isVisible) {
                dropdown.style.setProperty('display', 'none', 'important');
                console.log('✅ Dropdown closed');
            } else {
                dropdown.style.setProperty('display', 'block', 'important');
                console.log('✅ Dropdown opened');
            }
        } else {
            console.error('❌ Dropdown not found:', `dropdown-${reportId}`);
        }
    }

    // Modal state
    let currentDeleteId = null;
    let currentPdfUrl = null;
    let currentPdfObjectUrl = null;
    
    // View PDF in modal
    function viewPdfInModal(reportId) {
        const report = allReports.find(r => (r._id || r.id || r.reportId) === reportId);
        
        if (!report) {
            showNotification('Report not found', 'error');
            return;
        }
        
        // Close dropdown
        const dropdown = document.getElementById(`dropdown-${reportId}`);
        if (dropdown) dropdown.style.setProperty('display', 'none', 'important');
        
        const pdfUrl = report.step3?.pdfDownloadUrl;
        
        if (pdfUrl) {
            console.log('Opening PDF in modal:', pdfUrl);
            currentPdfUrl = pdfUrl;
            
            const iframe = document.getElementById('pdfIframe');
            const loader = document.getElementById('pdfLoader');
            
            if (iframe) {
                iframe.src = '';
            }
            
            if (loader) {
                loader.style.display = 'flex';
            }
            
            // Update modal title
            document.getElementById('pdfModalTitle').textContent = `View PDF - ${report.dvirNumber || 'DVIR Report'}`;
            
            // Show modal
            const modal = document.getElementById('pdfViewerModal');
            if (modal) {
                modal.style.display = 'block';
            }
            document.body.classList.add('modal-open');
            
            fetch(pdfUrl)
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Failed to load PDF');
                    }
                    return response.blob();
                })
                .then(blob => {
                    if (currentPdfObjectUrl) {
                        URL.revokeObjectURL(currentPdfObjectUrl);
                    }
                    currentPdfObjectUrl = URL.createObjectURL(blob);
                    if (iframe) {
                        iframe.src = `${currentPdfObjectUrl}#toolbar=0`;
                    }
                })
                .catch(error => {
                    console.error('Error loading PDF:', error);
                    showNotification('Failed to load PDF. Please try again.', 'error');
                    closePdfModal();
                })
                .finally(() => {
                    if (loader) {
                        loader.style.display = 'none';
                    }
                });
        } else {
            showNotification('PDF not available for this report', 'error');
        }
    }
    
    // Close PDF modal
    function closePdfModal() {
        const modal = document.getElementById('pdfViewerModal');
        const iframe = document.getElementById('pdfIframe');
        const loader = document.getElementById('pdfLoader');
        
        if (modal) {
            modal.style.display = 'none';
        }
        if (iframe) {
            iframe.src = '';
        }
        if (loader) {
            loader.style.display = 'none';
        }
        if (currentPdfObjectUrl) {
            URL.revokeObjectURL(currentPdfObjectUrl);
            currentPdfObjectUrl = null;
        }
        document.body.classList.remove('modal-open');
        currentPdfUrl = null;
    }
    
    // Download current PDF
    function downloadCurrentPdf() {
        if (currentPdfUrl) {
            window.open(currentPdfUrl, '_blank');
            showNotification('Downloading PDF...', 'success');
        }
    }
    
    // Open delete confirmation modal
    function openDeleteConfirmModal(reportId) {
        currentDeleteId = reportId;
        
        // Close dropdown
        const dropdown = document.getElementById(`dropdown-${reportId}`);
        if (dropdown) dropdown.style.setProperty('display', 'none', 'important');
        
        // Show confirmation modal
        document.getElementById('deleteConfirmModal').style.display = 'block';
        document.body.classList.add('modal-open');
    }
    
    // Close delete confirmation modal
    function closeDeleteConfirmModal() {
        document.getElementById('deleteConfirmModal').style.display = 'none';
        document.body.classList.remove('modal-open');
        currentDeleteId = null;
    }
    
    // Confirm delete DVIR
    async function confirmDeleteDvir() {
        if (!currentDeleteId) return;
        
        try {
            console.log('🗑️ Deleting DVIR report:', currentDeleteId);
            
            // Close modal first
            closeDeleteConfirmModal();
            
            const response = await fetch(`${CONFIG.mongodb.deleteUrl}/${currentDeleteId}`, {
                method: 'DELETE'
            });
            
            const result = await response.json();
            
            if (result.success) {
                showNotification('DVIR report deleted successfully', 'success');
                // Reload reports
                await loadDVIRReports();
                } else {
                showNotification('Failed to delete report: ' + result.error, 'error');
            }
        } catch (error) {
            console.error('❌ Error deleting DVIR report:', error);
            showNotification('Failed to delete report: ' + error.message, 'error');
        } finally {
            currentDeleteId = null;
        }
    }

    // Enhanced notification system
    function showNotification(message, type = 'info') {
        const typeColors = {
            success: '#28a745',
            error: '#dc3545',
            warning: '#ffc107',
            info: '#17a2b8'
        };
        
        const notification = document.createElement('div');
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 10000;
            background: white;
            border-left: 4px solid ${typeColors[type] || typeColors.info};
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            border-radius: 8px;
            padding: 16px;
            min-width: 300px;
            max-width: 400px;
            animation: slideIn 0.3s ease;
        `;
        
        notification.innerHTML = `
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <span style="color: #374151; font-size: 14px;">${message}</span>
                <button onclick="this.parentElement.parentElement.remove()" style="background: none; border: none; font-size: 20px; cursor: pointer; color: #9ca3af;">×</button>
            </div>
        `;
        
        document.body.appendChild(notification);
        
        setTimeout(() => {
            if (notification.parentElement) {
                notification.remove();
            }
        }, 5000);
    }

    // Close all dropdowns when clicking outside
    document.addEventListener('click', function(event) {
        if (!event.target.closest('.action-dropdown')) {
            const allDropdowns = document.querySelectorAll('.dropdown-menu');
            allDropdowns.forEach(dd => {
                dd.style.setProperty('display', 'none', 'important');
                });
            }
        });

    // Close modals when clicking outside
    window.onclick = function(event) {
        const deleteModal = document.getElementById('deleteConfirmModal');
        const pdfModal = document.getElementById('pdfViewerModal');
        const viewModal = document.getElementById('viewReportModal');
        
        if (event.target === deleteModal) {
            closeDeleteConfirmModal();
        } else if (event.target === pdfModal) {
            closePdfModal();
        } else if (event.target === viewModal) {
            closeViewReportModal();
        }
    };
    
    // Initialize all functionality
    function init() {
        initElements();
        initSearch();
        
        // Load DVIR reports from MongoDB
        loadDVIRReports();
        
        // Add click event for search input focus
        if (elements.searchInput) {
            elements.searchInput.addEventListener('focus', function() {
                this.select();
            });
        }
    }
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

    // Make functions globally accessible
    window.refreshData = refreshData;
    window.loadDVIRReports = loadDVIRReports;
    window.goToPage = goToPage;
    window.toggleActionDropdown = toggleActionDropdown;
    window.viewPdfInModal = viewPdfInModal;
    window.closePdfModal = closePdfModal;
    window.downloadCurrentPdf = downloadCurrentPdf;
    window.openDeleteConfirmModal = openDeleteConfirmModal;
    window.closeDeleteConfirmModal = closeDeleteConfirmModal;
    window.confirmDeleteDvir = confirmDeleteDvir;
    window.showNotification = showNotification;
})();
</script>

<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.dvir-inspection-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    color: var(--primary-color);
    text-decoration: none;
}

/* Search Container */
.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #9ca3af;
    z-index: 1;
}

.search-input {
    padding: 10px 12px 10px 40px;
    border: 1px solid #d1d5db;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 400;
    color: #374151;
    background: white;
    width: 300px;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.search-input::placeholder {
    color: #9ca3af;
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Table Section */
.table-section {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
    margin-bottom: 24px;
}

.table-container {
    overflow-x: auto;
    padding-bottom: 400px;
}

.dvir-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
}

.table-header th {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    color: #374151;
    font-weight: 600;
    font-size: 14px;
    padding: 16px 12px;
    text-align: left;
    border-bottom: 1px solid #e2e8f0;
    border-radius: 0;
}

.sno-column {
    min-width: 60px;
    width: 60px;
    text-align: center;
}

.action-column {
    min-width: 120px;
    width: 120px;
}

.table-body td {
    padding: 12px;
    border-bottom: 1px solid #f1f5f9;
    vertical-align: middle;
}

.sno-cell {
    text-align: center;
    font-weight: 600;
    color: #6b7280;
    background-color: #f8fafc;
}

.table-row:hover {
    background: #f8fafc;
}

/* Action Cell */
.action-cell {
    text-align: center;
    padding: 12px 8px;
    border-bottom: 1px solid #e2e8f0;
    vertical-align: middle;
    position: relative !important;
    width: 120px;
}

/* Action Dropdown */
.action-dropdown {
    position: relative !important;
    display: inline-block;
}

.btn-action {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 6px;
    padding: 8px;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    background: white;
    color: #6b7280;
    font-size: 13px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    min-width: 36px;
    min-height: 36px;
}

.btn-action span {
    display: none;
}

.btn-action:hover {
    background: #f9fafb;
    border-color: #0055a5;
    color: #0055a5;
}

.action-dropdown .dropdown-menu {
    position: absolute !important;
    top: calc(100% + 8px) !important;
    right: -47px !important;
    z-index: 9999 !important;
    display: none !important;
    min-width: 180px;
    padding: 8px 0;
    background: white;
    border: 1px solid #e2e8f0;
    border-radius: 8px;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
}

.dropdown-item {
    display: flex !important;
    align-items: center;
    gap: 12px;
    padding: 10px 16px;
    font-size: 14px;
    color: #374151;
    transition: all 0.2s ease;
    text-decoration: none;
    cursor: pointer;
    border: none;
    background: none;
    width: 100%;
    text-align: left;
}

.dropdown-item:hover {
    background: #f8fafc;
    color: #0f172a;
    text-decoration: none;
}

.dropdown-item svg {
    flex-shrink: 0;
}

.dropdown-item.delete-action {
    color: #dc2626;
}

.dropdown-item.delete-action:hover {
    background: #fef2f2;
    color: #dc2626;
}

/* Status Badges */
.status-badge {
    display: inline-flex;
    align-items: center;
    padding: 4px 12px;
    border-radius: 0;
    font-size: 12px;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.025em;
}

.status-badge.completed {
    background: #dcfce7;
    color: #166534;
    border: 1px solid #bbf7d0;
}

.status-badge.pending {
    background: #fef3c7;
    color: #92400e;
    border: 1px solid #fde68a;
}

.status-badge.issues {
    background: #fee2e2;
    color: #991b1b;
    border: 1px solid #fecaca;
}

.status-badge.default {
    background: #f3f4f6;
    color: #374151;
    border: 1px solid #d1d5db;
}

/* Data Cell Styling */
.data-cell {
    color: #374151;
    font-weight: 400;
}

.fault-text {
    color: #dc2626;
    font-weight: 500;
}

.no-fault {
    color: #059669;
    font-weight: 500;
}

.fault-info {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.fault-badge {
    background-color: #dc2626;
    color: white;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 12px;
    font-weight: 600;
    display: inline-block;
    width: fit-content;
}

.fault-details {
    color: #dc2626;
    font-weight: 500;
    font-size: 14px;
}

.no-faults {
    color: #059669;
    font-weight: 500;
}

.action-text {
    color: #0ea5e9;
    font-weight: 500;
}

.no-action {
    color: #6b7280;
    font-weight: 400;
}

/* Empty State */
.empty-state {
    padding: 60px 20px;
    text-align: center;
}

.empty-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.empty-icon {
    color: #9ca3af;
}

.empty-content h3 {
    color: #374151;
    font-size: 18px;
    font-weight: 600;
    margin: 0;
}

.empty-content p {
    color: #6b7280;
    font-size: 14px;
    margin: 0;
    max-width: 400px;
}

.btn-refresh-empty {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border: 1px solid #3b82f6;
    border-radius: 0;
    background: white;
    color: #3b82f6;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-refresh-empty:hover {
    background: #3b82f6;
    color: white;
}

/* Loading Section */
.loading-section {
    background: white;
    border: 1px solid #e2e8f0;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    border-radius: 0;
    margin-bottom: 24px;
}

.loading-content {
    padding: 60px 20px;
    text-align: center;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.spinner {
    animation: spin 1s linear infinite;
}

.spinner-icon {
    color: #3b82f6;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

.loading-content h3 {
    color: #374151;
    font-size: 18px;
    font-weight: 600;
    margin: 0;
}

.loading-content p {
    color: #6b7280;
    font-size: 14px;
    margin: 0;
}

/* Error Section */
.error-section {
    background: white;
    border: 1px solid #e2e8f0;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    border-radius: 0;
    margin-bottom: 24px;
}

.error-content {
    padding: 60px 20px;
    text-align: center;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.error-icon {
    color: #dc2626;
}

.error-content h3 {
    color: #374151;
    font-size: 18px;
    font-weight: 600;
    margin: 0;
}

.error-content p {
    color: #6b7280;
    font-size: 14px;
    margin: 0;
    max-width: 400px;
}

.btn-retry {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border: 1px solid #dc2626;
    border-radius: 0;
    background: white;
    color: #dc2626;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-retry:hover {
    background: #dc2626;
    color: white;
}

/* ===== LOADING STATE ===== */
.loading-state {
    text-align: center;
    padding: 60px 20px;
}

.loading-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid #f3f3f3;
    border-top: 4px solid #0055a5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-content h3 {
    color: #374151;
    font-size: 18px;
    font-weight: 600;
    margin: 0;
}

.loading-content p {
    color: #6b7280;
    font-size: 14px;
    margin: 0;
}

/* ===== PAGINATION STYLES ===== */
.pagination-section {
    background: white;
    border: 1px solid #e2e8f0;
    box-shadow: 0 1px 3px rgba(0,0,0,0.1);
    border-radius: 0;
    margin-bottom: 24px;
    padding: 20px 32px;
}

.pagination-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 20px;
}

.pagination-info {
    color: #64748b;
    font-size: 14px;
    font-weight: 500;
}

.pagination-buttons {
    display: flex;
    align-items: center;
    gap: 8px;
    flex-wrap: wrap;
}

.pagination-btn {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 8px 16px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    background: white;
    color: #374151;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.pagination-btn:hover:not(.disabled) {
    background: #0055a5;
    color: white;
    border-color: #0055a5;
}

.pagination-btn.disabled {
    background: #f3f4f6;
    color: #9ca3af;
    cursor: not-allowed;
    opacity: 0.6;
}

.pagination-btn.page-number {
    min-width: 40px;
    justify-content: center;
}

.pagination-btn.page-number.active {
    background: #0055a5;
    color: white;
    border-color: #0055a5;
}

.page-ellipsis {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 40px;
    height: 40px;
    color: #9ca3af;
    font-size: 14px;
    font-weight: 500;
}

/* ===== MODAL STYLES ===== */
body.modal-open {
    overflow: hidden;
}

.custom-modal {
    display: none;
    position: fixed;
    z-index: 1050;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    overflow: auto;
}

.custom-modal-content {
    background-color: #fff;
    margin: 50px auto;
    border-radius: 12px;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2);
    max-width: 600px;
    width: 90%;
    animation: modalSlideIn 0.3s ease;
}

.pdf-modal-large {
    max-width: 90%;
    width: 1200px;
}

@keyframes modalSlideIn {
    from {
        transform: translateY(-50px);
        opacity: 0;
    }
    to {
        transform: translateY(0);
        opacity: 1;
    }
}

.custom-modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 24px;
    border-bottom: 1px solid #e2e8f0;
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.custom-modal-header h3 {
    margin: 0;
    font-size: 20px;
    font-weight: 600;
    color: #0055a5;
}

.close-btn {
    background: none;
    border: none;
    font-size: 28px;
    font-weight: 300;
    line-height: 1;
    color: #6b7280;
    cursor: pointer;
    padding: 0;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 6px;
    transition: all 0.2s ease;
}

.close-btn:hover {
    background: #f3f4f6;
    color: #374151;
}

.custom-modal-body {
    padding: 0;
    max-height: 70vh;
    overflow-y: auto;
}

.pdf-modal-body {
    padding: 0;
    max-height: none;
    position: relative;
    background: #f5f5f5;
}

.pdf-loader {
    position: absolute;
    inset: 0;
    display: none;
    align-items: center;
    justify-content: center;
    flex-direction: column;
    gap: 12px;
    background: rgba(255, 255, 255, 0.9);
    z-index: 2;
}

.pdf-loader-spinner {
    width: 48px;
    height: 48px;
    border: 4px solid #e5e7eb;
    border-top: 4px solid #0055a5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

.pdf-loader p {
    margin: 0;
    font-size: 14px;
    color: #374151;
    font-weight: 500;
}

.modal-footer {
    display: flex;
    justify-content: flex-end;
    gap: 12px;
    padding: 20px 24px;
    border-top: 1px solid #e2e8f0;
    background: #f8fafc;
}

.modal-footer button {
    padding: 10px 20px;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    border: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.btn-primary {
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
    color: white;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #003f7d 0%, #002a5c 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.btn-secondary {
    background: #f3f4f6;
    color: #374151;
    border: 1px solid #d1d5db;
}

.btn-secondary:hover {
    background: #e5e7eb;
    border-color: #9ca3af;
}

.btn-danger {
    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
    color: white;
}

.btn-danger:hover {
    background: linear-gradient(135deg, #c82333 0%, #a71d2a 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(220, 53, 69, 0.3);
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .main-content {
        padding: 20px;
    }
    
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .search-input {
        width: 250px;
    }
    
    .pagination-content {
        flex-direction: column;
        gap: 15px;
    }
    
    .custom-modal-content {
        margin: 20px auto;
        width: 95%;
    }
    
    .pdf-modal-large {
        max-width: 95%;
        width: 95%;
    }
    
    .custom-modal-body {
        max-height: 60vh;
    }
    
    #pdfIframe {
        height: 500px;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
        flex-direction: column;
        gap: 15px;
        align-items: stretch;
    }
    
    .header-left {
        justify-content: center;
    }
    
    .header-right {
        justify-content: center;
        flex-direction: column;
        gap: 10px;
    }
    
    .search-input {
        width: 100%;
        max-width: 400px;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
    }
    
    .main-content {
        padding: 15px;
    }
    
    .custom-modal-content {
        margin: 10px;
        width: calc(100% - 20px);
    }
    
    .modal-footer {
        flex-direction: column;
        gap: 8px;
    }
    
    .modal-footer button {
        width: 100%;
        justify-content: center;
    }
    
    #pdfIframe {
        height: 400px;
    }
}

@media (max-width: 992px) {
    .dvir-inspection-container {
        padding: 12px;
    }
    
    .header-content {
        padding: 16px;
    }
    
    .page-title h1 {
        font-size: 22px;
        margin-bottom: 6px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .action-content {
        padding: 12px 16px;
        flex-direction: column;
        align-items: stretch;
        gap: 12px;
    }
    
    .action-buttons {
        width: 100%;
        justify-content: center;
        flex-wrap: wrap;
        gap: 8px;
    }
    
    .btn-create,
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort,
    .btn-export {
        padding: 10px 14px;
        font-size: 13px;
        min-height: 44px;
        flex: 1;
        min-width: 120px;
        justify-content: center;
    }
    
    .search-section {
        align-self: center;
    }
    
    .search-input {
        width: 100%;
        max-width: 400px;
        padding: 12px 16px;
        font-size: 16px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .dvir-table {
        min-width: 800px;
    }
    
    .table-header th {
        padding: 12px 8px;
        font-size: 13px;
    }
    
    .table-body td {
        padding: 10px 8px;
        font-size: 13px;
    }
    
    .sno-column {
        min-width: 50px;
        width: 50px;
    }
    
    .action-column {
        min-width: 100px;
        width: 100px;
    }
    
    .btn-action {
        padding: 6px 10px;
        font-size: 12px;
    }
    
    .dropdown-menu {
        min-width: 180px;
    }
    
    .dropdown-item {
        padding: 8px 12px;
        font-size: 13px;
    }
}

@media (max-width: 768px) {
    .dvir-inspection-container {
        padding: 0;
    }
    
    .dvir-header {
        margin-bottom: 16px;
    }
    
    .header-content {
        padding: 16px;
    }
    
    .page-title h1 {
        font-size: 20px;
        margin-bottom: 4px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .action-bar {
        margin-bottom: 16px;
    }
    
    .action-content {
        padding: 12px 16px;
        flex-direction: column;
        align-items: stretch;
        gap: 12px;
    }
    
    .action-buttons {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 8px;
        width: 100%;
    }
    
    .btn-create {
        grid-column: 1 / -1;
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
        min-height: 48px;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 10px 8px;
        font-size: 12px;
        min-height: 44px;
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 4px;
    }
    
    .btn-export {
        grid-column: 1 / -1;
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
        min-height: 48px;
    }
    
    .search-section {
        width: 100%;
    }
    
    .search-container {
        width: 100%;
    }
    
    .search-input {
        width: 100%;
        padding: 12px 16px;
        font-size: 16px;
        min-height: 48px;
    }
    
    .table-section {
        margin-bottom: 16px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        border: 1px solid #e2e8f0;
    }
    
    .dvir-table {
        min-width: 700px;
        font-size: 12px;
    }
    
    .table-header th {
        padding: 10px 6px;
        font-size: 11px;
        background-color: #f8fafc;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .table-body td {
        padding: 8px 6px;
        font-size: 11px;
        white-space: nowrap;
    }
    
    .sno-column {
        min-width: 40px;
        width: 40px;
    }
    
    .action-column {
        min-width: 80px;
        width: 80px;
    }
    
    .btn-action {
        padding: 6px 8px;
        font-size: 11px;
        min-width: 32px;
        min-height: 32px;
    }
    
    .dropdown-menu {
        min-width: 160px;
    }
    
    .dropdown-item {
        padding: 8px 12px;
        font-size: 12px;
    }
    
    .status-badge {
        padding: 4px 8px;
        font-size: 10px;
    }
    
    .empty-state {
        padding: 40px 16px;
    }
    
    .empty-content h3 {
        font-size: 16px;
    }
    
    .empty-content p {
        font-size: 13px;
    }
    
    .loading-content,
    .error-content {
        padding: 40px 16px;
    }
    
    .loading-content h3,
    .error-content h3 {
        font-size: 16px;
    }
    
    .loading-content p,
    .error-content p {
        font-size: 13px;
    }
}

@media (max-width: 576px) {
    .dvir-inspection-container {
        padding: 0;
    }
    
    .header-content {
        padding: 12px;
    }
    
    .page-title h1 {
        font-size: 18px;
        margin-bottom: 3px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .action-content {
        padding: 10px 12px;
        gap: 10px;
    }
    
    .action-buttons {
        display: grid;
        grid-template-columns: 1fr;
        gap: 6px;
        width: 100%;
    }
    
    .btn-create,
    .btn-export {
        width: 100%;
        padding: 14px 16px;
        font-size: 15px;
        font-weight: 700;
        min-height: 52px;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        width: 100%;
        padding: 12px 8px;
        font-size: 11px;
        min-height: 48px;
        flex-direction: row;
        justify-content: center;
        gap: 6px;
    }
    
    .search-input {
        padding: 14px 16px;
        font-size: 15px;
        min-height: 52px;
    }
    
    .table-container {
        border: 1px solid #d1d5db;
    }
    
    .dvir-table {
        min-width: 600px;
        font-size: 11px;
    }
    
    .table-header th {
        padding: 8px 4px;
        font-size: 10px;
        background-color: #f3f4f6;
    }
    
    .table-body td {
        padding: 6px 4px;
        font-size: 10px;
    }
    
    .sno-column {
        min-width: 35px;
        width: 35px;
    }
    
    .action-column {
        min-width: 70px;
        width: 70px;
    }
    
    .btn-action {
        padding: 4px 6px;
        font-size: 10px;
        min-width: 28px;
        min-height: 28px;
    }
    
    .dropdown-menu {
        min-width: 140px;
    }
    
    .dropdown-item {
        padding: 6px 10px;
        font-size: 11px;
    }
    
    .status-badge {
        padding: 3px 6px;
        font-size: 9px;
    }
    
    .empty-state {
        padding: 30px 12px;
    }
    
    .empty-content h3 {
        font-size: 14px;
    }
    
    .empty-content p {
        font-size: 12px;
    }
    
    .loading-content,
    .error-content {
        padding: 30px 12px;
    }
    
    .loading-content h3,
    .error-content h3 {
        font-size: 14px;
    }
    
    .loading-content p,
    .error-content p {
        font-size: 12px;
    }
}

@media (max-width: 480px) {
    .header-content {
        padding: 8px;
    }
    
    .page-title h1 {
        font-size: 16px;
        margin-bottom: 2px;
    }
    
    .page-title p {
        font-size: 11px;
    }
    
    .action-content {
        padding: 8px;
        gap: 8px;
    }
    
    .action-buttons {
        gap: 4px;
    }
    
    .btn-create,
    .btn-export {
        padding: 10px 6px !important;
        font-size: 16px;
        font-weight: 700;
        min-height: 56px;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 14px 6px;
        font-size: 10px;
        min-height: 52px;
        gap: 4px;
    }
    
    .search-input {
        padding: 16px 12px;
        font-size: 16px;
        min-height: 56px;
    }
    
    .dvir-table {
        min-width: 500px;
        font-size: 10px;
    }
    
    .table-header th {
        padding: 6px 3px;
        font-size: 9px;
        background-color: #f3f4f6;
    }
    
    .table-body td {
        padding: 4px 3px;
        font-size: 9px;
    }
    
    .sno-column {
        min-width: 30px;
        width: 30px;
    }
    
    .action-column {
        min-width: 60px;
        width: 60px;
    }
    
    .btn-action {
        padding: 3px 4px;
        font-size: 9px;
        min-width: 24px;
        min-height: 24px;
    }
    
    .dropdown-menu {
        min-width: 120px;
    }
    
    .dropdown-item {
        padding: 4px 8px;
        font-size: 10px;
    }
    
    .status-badge {
        padding: 2px 4px;
        font-size: 8px;
    }
    
    .empty-state {
        padding: 20px 8px;
    }
    
    .empty-content h3 {
        font-size: 13px;
    }
    
    .empty-content p {
        font-size: 11px;
    }
    
    .loading-content,
    .error-content {
        padding: 20px 8px;
    }
    
    .loading-content h3,
    .error-content h3 {
        font-size: 13px;
    }
    
    .loading-content p,
    .error-content p {
        font-size: 11px;
    }
}

@media (max-width: 360px) {
    .header-content {
        padding: 6px;
    }
    
    .page-title h1 {
        font-size: 14px;
        margin-bottom: 1px;
    }
    
    .page-title p {
        font-size: 10px;
    }
    
    .action-content {
        padding: 6px;
        gap: 6px;
    }
    
    .action-buttons {
        gap: 3px;
    }
    
    .btn-create,
    .btn-export {
        padding: 18px 8px;
        font-size: 14px;
        font-weight: 700;
        min-height: 60px;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 16px 4px;
        font-size: 9px;
        min-height: 56px;
        gap: 2px;
    }
    
    .search-input {
        padding: 18px 8px;
        font-size: 14px;
        min-height: 60px;
    }
    
    .dvir-table {
        min-width: 450px;
        font-size: 9px;
    }
    
    .table-header th {
        padding: 4px 2px;
        font-size: 8px;
        background-color: #f3f4f6;
    }
    
    .table-body td {
        padding: 3px 2px;
        font-size: 8px;
    }
    
    .action-column {
        min-width: 50px;
        width: 50px;
    }
    
    .btn-action {
        padding: 2px 3px;
        font-size: 8px;
        min-width: 20px;
        min-height: 20px;
    }
    
    .dropdown-menu {
        min-width: 100px;
    }
    
    .dropdown-item {
        padding: 3px 6px;
        font-size: 9px;
    }
    
    .status-badge {
        padding: 1px 2px;
        font-size: 7px;
    }
    
    .empty-state {
        padding: 15px 6px;
    }
    
    .empty-content h3 {
        font-size: 12px;
    }
    
    .empty-content p {
        font-size: 10px;
    }
    
    .loading-content,
    .error-content {
        padding: 15px 6px;
    }
    
    .loading-content h3,
    .error-content h3 {
        font-size: 12px;
    }
    
    .loading-content p,
    .error-content p {
        font-size: 10px;
    }
}

/* Focus states for accessibility */
.btn-create:focus,
.btn-refresh:focus,
.btn-test:focus,
.btn-structure:focus,
.btn-sort:focus,
.btn-export:focus,
.btn-action:focus,
.search-input:focus,
.btn-refresh-empty:focus,
.btn-retry:focus {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}

/* Pagination Styles */
.pagination-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
    padding: 20px 32px;
}

.pagination-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 20px;
    flex-wrap: wrap;
}

.pagination-info {
    flex-shrink: 0;
}

.pagination-text {
    color: var(--text-muted);
    font-size: 14px;
    font-weight: 500;
}

.pagination-controls {
    display: flex;
    align-items: center;
    gap: 8px;
    flex-wrap: wrap;
}

.pagination-btn {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 8px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    background: var(--bg-white);
    color: var(--text-secondary);
    font-size: 14px;
    font-weight: 500;
    text-decoration: none;
    cursor: pointer;
    transition: var(--transition-base);
}

.pagination-btn:hover:not(.disabled) {
    background: var(--primary-color);
    color: var(--bg-white);
    border-color: var(--primary-color);
    text-decoration: none;
}

.pagination-btn.disabled {
    background: var(--bg-primary);
    color: var(--text-light);
    cursor: not-allowed;
    opacity: 0.6;
}

.page-numbers {
    display: flex;
    align-items: center;
    gap: 4px;
    margin: 0 12px;
}

.page-number {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 40px;
    height: 40px;
    padding: 0 8px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    background: var(--bg-white);
    color: var(--text-secondary);
    font-size: 14px;
    font-weight: 500;
    text-decoration: none;
    cursor: pointer;
    transition: var(--transition-base);
}

.page-number:hover:not(.current) {
    background: var(--primary-color);
    color: var(--bg-white);
    border-color: var(--primary-color);
    text-decoration: none;
}

.page-number.current {
    background: var(--primary-color);
    color: var(--bg-white);
    border-color: var(--primary-color);
    cursor: default;
}

.page-ellipsis {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 40px;
    height: 40px;
    color: var(--text-muted);
    font-size: 14px;
    font-weight: 500;
}

.records-per-page {
    display: flex;
    align-items: center;
    gap: 8px;
    flex-shrink: 0;
}

.records-per-page label {
    color: var(--text-secondary);
    font-size: 14px;
    font-weight: 500;
}

.per-page-select {
    padding: 6px 12px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    background: var(--bg-white);
    color: var(--text-secondary);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.per-page-select:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.records-per-page span {
    color: var(--text-muted);
    font-size: 14px;
    font-weight: 500;
}

/* Responsive Pagination */
@media (max-width: 768px) {
    .pagination-content {
        flex-direction: column;
        gap: 16px;
        align-items: stretch;
    }
    
    .pagination-controls {
        justify-content: center;
    }
    
    .page-numbers {
        margin: 0 8px;
    }
    
    .page-number {
        min-width: 36px;
        height: 36px;
        font-size: 13px;
    }
    
    .records-per-page {
        justify-content: center;
    }
}

@media (max-width: 576px) {
    .pagination-section {
        padding: 16px 20px;
    }
    
    .pagination-btn {
        padding: 6px 12px;
        font-size: 13px;
    }
    
    .page-number {
        min-width: 32px;
        height: 32px;
        font-size: 12px;
    }
    
    .pagination-text {
        font-size: 13px;
    }
    
    .per-page-select {
        font-size: 13px;
    }
}

/* Smooth animations */
* {
    transition: all 0.2s ease;
}

/* Icon improvements */
.btn-create svg,
.btn-refresh svg,
.btn-test svg,
.btn-structure svg,
.btn-sort svg,
.btn-export svg,
.btn-action svg,
.search-icon,
.dropdown-item svg,
.btn-refresh-empty svg,
.btn-retry svg,
.empty-icon,
.spinner-icon,
.error-icon {
    flex-shrink: 0;
}

/* ===== ENHANCED RESPONSIVE DESIGN ===== */
@media (max-width: 1400px) {
    .dvir-inspection-container {
        padding: 18px;
    }
    
    .dvir-header {
        margin-bottom: 20px;
    }
    
    .header-content {
        padding: 28px;
    }
}

@media (max-width: 1200px) {
    .dvir-inspection-container {
        padding: 16px;
    }
    
    .header-content {
        padding: 24px;
    }
    
    .action-bar {
        padding: 20px;
    }
    
    .action-content {
        flex-direction: column;
        gap: 16px;
        align-items: flex-start;
    }
    
    .action-buttons {
        width: 100%;
        justify-content: flex-start;
        flex-wrap: wrap;
        gap: 12px;
    }
}

@media (max-width: 992px) {
    .dvir-inspection-container {
        padding: 14px;
    }
    
    .dvir-header {
        margin-bottom: 16px;
    }
    
    .header-content {
        padding: 20px;
    }
    
    .page-title h1 {
        font-size: 28px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .action-bar {
        padding: 16px;
    }
    
    .action-buttons {
        flex-direction: column;
        width: 100%;
    }
    
    .btn-create,
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort,
    .btn-export {
        width: 100%;
        justify-content: center;
    }
    
    .table-responsive {
        font-size: 14px;
    }
    
    .table th,
    .table td {
        padding: 8px;
    }
}

@media (max-width: 768px) {
    .dvir-inspection-container {
        padding: 8px;
    }
    
    .header-content {
        padding: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
        margin-bottom: 4px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .action-bar {
        padding: 8px;
        margin-bottom: 12px;
    }
    
    .action-content {
        flex-direction: column;
        gap: 8px;
        align-items: stretch;
    }
    
    .action-buttons {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 8px;
        width: 100%;
    }
    
    .btn-create {
        grid-column: 1 / -1;
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 10px 8px;
        font-size: 12px;
        min-height: 44px;
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 4px;
    }
    
    .btn-export {
        grid-column: 1 / -1;
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
    }
    
    .btn-create svg,
    .btn-refresh svg,
    .btn-test svg,
    .btn-structure svg,
    .btn-sort svg,
    .btn-export svg {
        width: 16px;
        height: 16px;
    }
    
    .search-container {
        flex-direction: column;
        gap: 8px;
        margin-bottom: 12px;
    }
    
    .search-input {
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        border-radius: 0;
    }
    
    .search-btn {
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        min-height: 44px;
    }
    
    .table-responsive {
        font-size: 12px;
        border: 1px solid #e2e8f0;
        border-radius: 0;
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .table {
        margin-bottom: 0;
        min-width: 600px;
    }
    
    .table th,
    .table td {
        padding: 8px 4px;
        white-space: nowrap;
        vertical-align: middle;
    }
    
    .table th {
        background-color: #f8fafc;
        font-weight: 600;
        font-size: 11px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .table td {
        font-size: 11px;
    }
    
    .status-badge {
        padding: 4px 8px;
        font-size: 10px;
        border-radius: 0;
    }
    
    .action-btn {
        padding: 6px 8px;
        font-size: 11px;
        min-width: 32px;
        min-height: 32px;
    }
}

@media (max-width: 576px) {
    .dvir-inspection-container {
        padding: 4px;
    }
    
    .header-content {
        padding: 8px;
    }
    
    .page-title h1 {
        font-size: 18px;
        margin-bottom: 2px;
    }
    
    .page-title p {
        font-size: 11px;
    }
    
    .action-bar {
        padding: 6px;
        margin-bottom: 8px;
    }
    
    .action-content {
        gap: 6px;
    }
    
    .action-buttons {
        grid-template-columns: 1fr;
        gap: 6px;
    }
    
    .btn-create,
    .btn-export {
        padding: 14px 16px;
        font-size: 15px;
        font-weight: 700;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 12px 8px;
        font-size: 11px;
        min-height: 48px;
        flex-direction: row;
        justify-content: center;
        gap: 6px;
    }
    
    .btn-create svg,
    .btn-refresh svg,
    .btn-test svg,
    .btn-structure svg,
    .btn-sort svg,
    .btn-export svg {
        width: 18px;
        height: 18px;
    }
    
    .search-container {
        gap: 6px;
        margin-bottom: 8px;
    }
    
    .search-input {
        padding: 14px 16px;
        font-size: 15px;
    }
    
    .search-btn {
        padding: 14px 16px;
        font-size: 15px;
        min-height: 48px;
    }
    
    .table-responsive {
        font-size: 11px;
        margin-bottom: 16px;
    }
    
    .table {
        min-width: 500px;
    }
    
    .table th,
    .table td {
        padding: 6px 3px;
    }
    
    .table th {
        font-size: 10px;
        padding: 8px 3px;
    }
    
    .table td {
        font-size: 10px;
    }
    
    .status-badge {
        padding: 3px 6px;
        font-size: 9px;
    }
    
    .action-btn {
        padding: 4px 6px;
        font-size: 10px;
        min-width: 28px;
        min-height: 28px;
    }
}

@media (max-width: 480px) {
    .dvir-inspection-container {
        padding: 2px;
    }
    
    .header-content {
        padding: 6px;
    }
    
    .page-title h1 {
        font-size: 16px;
        margin-bottom: 1px;
    }
    
    .page-title p {
        font-size: 10px;
    }
    
    .action-bar {
        padding: 4px;
        margin-bottom: 6px;
    }
    
    .action-content {
        gap: 4px;
    }
    
    .action-buttons {
        gap: 4px;
    }
    
    .btn-create,
    .btn-export {
        padding: 16px 12px;
        font-size: 16px;
        font-weight: 700;
        min-height: 52px;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 14px 6px;
        font-size: 10px;
        min-height: 52px;
        gap: 4px;
    }
    
    .btn-create svg,
    .btn-refresh svg,
    .btn-test svg,
    .btn-structure svg,
    .btn-sort svg,
    .btn-export svg {
        width: 20px;
        height: 20px;
    }
    
    .search-container {
        gap: 4px;
        margin-bottom: 6px;
    }
    
    .search-input {
        padding: 16px 12px;
        font-size: 16px;
    }
    
    .search-btn {
        padding: 16px 12px;
        font-size: 16px;
        min-height: 52px;
    }
    
    .table-responsive {
        font-size: 10px;
        margin-bottom: 12px;
        border: 1px solid #d1d5db;
    }
    
    .table {
        min-width: 450px;
    }
    
    .table th,
    .table td {
        padding: 4px 2px;
    }
    
    .table th {
        font-size: 9px;
        padding: 6px 2px;
        background-color: #f3f4f6;
    }
    
    .table td {
        font-size: 9px;
    }
    
    .status-badge {
        padding: 2px 4px;
        font-size: 8px;
    }
    
    .action-btn {
        padding: 3px 4px;
        font-size: 9px;
        min-width: 24px;
        min-height: 24px;
    }
}

@media (max-width: 360px) {
    .dvir-inspection-container {
        padding: 1px;
    }
    
    .header-content {
        padding: 4px;
    }
    
    .page-title h1 {
        font-size: 14px;
        margin-bottom: 0;
    }
    
    .page-title p {
        font-size: 9px;
    }
    
    .action-bar {
        padding: 2px;
        margin-bottom: 4px;
    }
    
    .action-content {
        gap: 2px;
    }
    
    .action-buttons {
        gap: 2px;
    }
    
    .btn-create,
    .btn-export {
        padding: 18px 8px;
        font-size: 14px;
        font-weight: 700;
        min-height: 56px;
    }
    
    .btn-refresh,
    .btn-test,
    .btn-structure,
    .btn-sort {
        padding: 16px 4px;
        font-size: 9px;
        min-height: 56px;
        gap: 2px;
    }
    
    .btn-create svg,
    .btn-refresh svg,
    .btn-test svg,
    .btn-structure svg,
    .btn-sort svg,
    .btn-export svg {
        width: 18px;
        height: 18px;
    }
    
    .search-container {
        gap: 2px;
        margin-bottom: 4px;
    }
    
    .search-input {
        padding: 18px 8px;
        font-size: 14px;
    }
    
    .search-btn {
        padding: 18px 8px;
        font-size: 14px;
        min-height: 56px;
    }
    
    .table-responsive {
        font-size: 9px;
        margin-bottom: 8px;
    }
    
    .table {
        min-width: 400px;
    }
    
    .table th,
    .table td {
        padding: 3px 1px;
    }
    
    .table th {
        font-size: 8px;
        padding: 4px 1px;
    }
    
    .table td {
        font-size: 8px;
    }
    
    .status-badge {
        padding: 1px 2px;
        font-size: 7px;
    }
    
    .action-btn {
        padding: 2px 3px;
        font-size: 8px;
        min-width: 20px;
        min-height: 20px;
    }
}
</style>
@endsection