// Configuration object
const CONFIG = {
    selectors: {
        maintenanceForm: '#maintenanceForm',
        countrySelect: '#country',
        projectSelect: '#project',
        requestNumber: '#requestNumber',
        vehicleModal: '#vehicleModal',
        vehicleList: '#vehicleList',
        vehicleSearch: '#vehicleSearch',
        assignToModal: '#assignToModal',
        assignToListModal: '#assignToListModal',
        assignToSearch: '#assignToSearch',
        requestTypeModal: '#requestTypeModal',
        addTaskModal: '#addTaskModal',
        taskAssignToModal: '#taskAssignToModal',
        faultListModal: '#faultListModal',
        syncModal: '#syncModal',
        successModal: '#successModal'
    },
    messages: {
        loading: 'Loading equipment...',
        error: 'Failed to load equipment data. Please try again.',
        success: 'Maintenance request created successfully!',
        validation: 'Please fill in all required fields.',
        taskTitleRequired: 'Task title is required'
    },
    timing: {
        debounceDelay: 300
    },
    endpoints: {
        equipment: '/api/equipment',
        users: '/api/users',
        storeRequest: '{{ route("maintenance.request.store") }}',
        generatePDF: '/maintenance/request/pdf'
    }
};

// State management
let state = {
    vehicles: [],
    teamMembers: [],
    selectedAssignees: [],
    tasks: [],
    taskAssignees: [],
    selectedFaults: [],
    selectedVehicle: null,
    selectedRequestType: null,
    selectedPriority: null,
    faultListData: [
        { id: '1', name: 'Engine Fault' },
        { id: '2', name: 'Brake Issue' },
        { id: '3', name: 'Electrical Problem' },
        { id: '4', name: 'Hydraulic Failure' },
        { id: '5', name: 'Transmission Issue' }
    ],
    editingTaskIndex: null
};

// DOM element caching
let elements = {};

// Initialize DOM elements
function initElements() {
    elements = {
        maintenanceForm: document.querySelector(CONFIG.selectors.maintenanceForm),
        countrySelect: document.querySelector(CONFIG.selectors.countrySelect),
        projectSelect: document.querySelector(CONFIG.selectors.projectSelect),
        requestNumber: document.querySelector(CONFIG.selectors.requestNumber),
        vehicleModal: document.querySelector(CONFIG.selectors.vehicleModal),
        vehicleList: document.querySelector(CONFIG.selectors.vehicleList),
        vehicleSearch: document.querySelector(CONFIG.selectors.vehicleSearch),
        assignToModal: document.querySelector(CONFIG.selectors.assignToModal),
        assignToListModal: document.querySelector(CONFIG.selectors.assignToListModal),
        assignToSearch: document.querySelector(CONFIG.selectors.assignToSearch),
        requestTypeModal: document.querySelector(CONFIG.selectors.requestTypeModal),
        addTaskModal: document.querySelector(CONFIG.selectors.addTaskModal),
        taskAssignToModal: document.querySelector(CONFIG.selectors.taskAssignToModal),
        faultListModal: document.querySelector(CONFIG.selectors.faultListModal),
        syncModal: document.querySelector(CONFIG.selectors.syncModal),
        successModal: document.querySelector(CONFIG.selectors.successModal)
    };
}

// Main initialization
function init() {
    initElements();
    initEventListeners();
    loadEquipmentFromAPI();
    loadTeamMembersFromAPI();
    updateRequestNumber();
}

// Initialize event listeners
function initEventListeners() {
    if (elements.countrySelect) {
        elements.countrySelect.addEventListener('change', handleCountryChange);
    }
    
    if (elements.projectSelect) {
        elements.projectSelect.addEventListener('change', updateRequestNumber);
    }
    
    if (elements.maintenanceForm) {
        elements.maintenanceForm.addEventListener('submit', handleFormSubmit);
    }

    // Priority selection
    document.querySelectorAll('.priority-button').forEach(btn => {
        btn.addEventListener('click', function() {
            selectPriority(this.dataset.priority);
        });
    });
}

// Generate request number
function generateRequestNumber(country, project, number) {
    const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
    const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
    return `${countryCode}-${projectCode}-RM-${number.toString().padStart(3, '0')}`;
}

function updateRequestNumber() {
    const country = elements.countrySelect?.value;
    const project = elements.projectSelect?.value;
    // Mock last number (replace with API call to fetch last number)
    const lastNumber = 1; // Example, should fetch from backend
    const nextNumber = lastNumber + 1;
    if (elements.requestNumber) {
        elements.requestNumber.textContent = generateRequestNumber(country, project, nextNumber);
    }
}

// Handle country change
function handleCountryChange() {
    const country = this.value;
    if (elements.projectSelect) {
        elements.projectSelect.innerHTML = '<option value="">Select Project</option>';
        elements.projectSelect.disabled = false;
        
        if (country === 'Zambia') {
            const options = ['Kansanshi', 'Kalumbila', 'Kimteto', 'Kobold', 'FQM Exploration', 'Mimosa'];
            options.forEach(opt => {
                elements.projectSelect.innerHTML += `<option value="${opt}">${opt}</option>`;
            });
        } else if (country === 'DRC') {
            const options = ['Lubumbashi', 'Musompo', 'IME', 'Kamoa'];
            options.forEach(opt => {
                elements.projectSelect.innerHTML += `<option value="${opt}">${opt}</option>`;
            });
        } else {
            elements.projectSelect.disabled = true;
        }
    }
    updateRequestNumber();
}

// Priority selection
function selectPriority(priority) {
    state.selectedPriority = priority;
    
    // Update UI
    document.querySelectorAll('.priority-button').forEach(btn => {
        btn.classList.remove('selected');
    });
    
    const selectedBtn = document.querySelector(`[data-priority="${priority}"]`);
    if (selectedBtn) {
        selectedBtn.classList.add('selected');
    }
}

// Vehicle Modal functions
function updateVehicleList(search = '') {
    if (!elements.vehicleList) return;
    
    if (!state.vehicles || state.vehicles.length === 0) {
        elements.vehicleList.innerHTML = '<p class="text-center text-muted p-3">No equipment found</p>';
        return;
    }
    
    const filteredVehicles = search ? state.vehicles.filter(v => 
        (v.equipmentName || v.vehicleNumber || '').toLowerCase().includes(search.toLowerCase()) ||
        (v.model || '').toLowerCase().includes(search.toLowerCase()) ||
        (v.mainCategory || v.equipmentCategory || '').toLowerCase().includes(search.toLowerCase())
    ) : state.vehicles;
    
    elements.vehicleList.innerHTML = filteredVehicles.map(v => `
        <div class="vehicle-item" onclick="selectVehicle('${v.id || v.firebaseKey || 'unknown'}', '${v.equipmentName || v.vehicleNumber || 'Unknown Equipment'}', '${v.equipmentNumber || v.vehicleNumber || 'N/A'}', '${v.model || 'N/A'}', '${v.meterReading || '0'}', '${v.meterUnit || 'KM'}', '${v.status || 'Available'}')">
            <div class="vehicle-name">${v.equipmentName || v.vehicleNumber || 'Unknown Equipment'}</div>
            <div class="vehicle-details">
                <strong>Number:</strong> ${v.equipmentNumber || v.vehicleNumber || 'N/A'}<br>
                <strong>Model:</strong> ${v.model || 'N/A'}<br>
                <strong>Category:</strong> ${v.mainCategory || v.equipmentCategory || 'N/A'}<br>
                <strong>Meter:</strong> ${v.meterReading || '0'} ${v.meterUnit || 'KM'}<br>
                <strong>Status:</strong> ${v.status || 'Available'}
            </div>
        </div>
    `).join('');
}

// Load equipment data from API
async function loadEquipmentFromAPI() {
    try {
        // Show loading state
        if (elements.vehicleList) {
            elements.vehicleList.innerHTML = '<div class="text-center p-3"><div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2">' + CONFIG.messages.loading + '</p></div>';
        }
        
        const response = await fetch(CONFIG.endpoints.equipment);
        const data = await response.json();
        
        if (data.success) {
            state.vehicles = data.equipment || [];
            updateVehicleList();
        } else {
            console.error('Failed to load equipment:', data.message);
            if (elements.vehicleList) {
                elements.vehicleList.innerHTML = '<p class="text-center text-muted p-3">' + CONFIG.messages.error + ': ' + data.message + '</p>';
            }
        }
    } catch (error) {
        console.error('Error loading equipment:', error);
        if (elements.vehicleList) {
            elements.vehicleList.innerHTML = '<p class="text-center text-muted p-3">' + CONFIG.messages.error + '</p>';
        }
    }
}

// Load team members from API
async function loadTeamMembersFromAPI() {
    try {
        const response = await fetch(CONFIG.endpoints.users);
        const data = await response.json();
        
        if (data.success) {
            state.teamMembers = data.users || [];
        } else {
            console.error('Failed to load team members:', data.message);
            state.teamMembers = [];
        }
    } catch (error) {
        console.error('Error loading team members:', error);
        state.teamMembers = [];
    }
}

// Filter vehicles
function filterVehicles() {
    const searchTerm = elements.vehicleSearch?.value.toLowerCase() || '';
    const filtered = state.vehicles.filter(vehicle => 
        (vehicle.equipmentName || vehicle.vehicleNumber || '').toLowerCase().includes(searchTerm) ||
        (vehicle.equipmentNumber || vehicle.vehicleNumber || '').toLowerCase().includes(searchTerm) ||
        (vehicle.model || '').toLowerCase().includes(searchTerm) ||
        (vehicle.mainCategory || vehicle.equipmentCategory || '').toLowerCase().includes(searchTerm)
    );
    updateVehicleList(filtered);
}

// Select vehicle
function selectVehicle(id, name, number, model, meter, unit, status) {
    state.selectedVehicle = { 
        id, 
        equipmentName: name, 
        equipmentNumber: number, 
        model, 
        meterReading: meter, 
        meterUnit: unit, 
        status 
    };
    
    const vehicleDetails = document.getElementById('vehicleDetails');
    const selectedVehicleInput = document.getElementById('selectedVehicle');
    
    if (vehicleDetails) {
        vehicleDetails.innerHTML = `
            <div class="selected-item-details">
                <strong>${name}</strong><br>
                Number: ${number}<br>
                Model: ${model}<br>
                Meter: ${meter} ${unit}<br>
                Status: ${status}
            </div>
        `;
    }
    
    if (selectedVehicleInput) {
        selectedVehicleInput.value = JSON.stringify(state.selectedVehicle);
    }
    
    closeVehicleModal();
}

// Assign To Modal functions
function updateAssignToList(search = '') {
    if (!elements.assignToListModal) return;
    
    const filteredMembers = state.teamMembers.filter(m => 
        m.fullName.toLowerCase().includes(search.toLowerCase()) || 
        m.employeeNumber.toLowerCase().includes(search.toLowerCase())
    );
    
    elements.assignToListModal.innerHTML = filteredMembers.map(m => `
        <div class="team-member-item" onclick="selectTeamMember('${m.id}', '${m.fullName}', '${m.employeeNumber}', '${m.role}')">
            <div class="team-member-name">${m.fullName}</div>
            <div class="team-member-details">
                <strong>Emp#:</strong> ${m.employeeNumber}<br>
                <strong>Role:</strong> ${m.role}
            </div>
        </div>
    `).join('');
}

// Filter team members
function filterTeamMembers() {
    const searchTerm = elements.assignToSearch?.value.toLowerCase() || '';
    const filtered = state.teamMembers.filter(member => 
        (member.fullName || '').toLowerCase().includes(searchTerm) ||
        (member.employeeNumber || '').toLowerCase().includes(searchTerm)
    );
    updateAssignToList(filtered);
}

// Select team member
function selectTeamMember(id, name, employeeNumber, role) {
    if (!state.selectedAssignees.some(a => a.id === id)) {
        state.selectedAssignees.push({ id, fullName: name, employeeNumber, role });
        updateAssignToDisplay();
    }
}

// Update assign to display
function updateAssignToDisplay() {
    const tagsContainer = document.getElementById('assignToList');
    const assignToInput = document.getElementById('assignTo');
    
    if (tagsContainer) {
        tagsContainer.innerHTML = state.selectedAssignees.map(a => `
            <span class="assignee-tag">
                ${a.fullName}
                <button type="button" class="remove-btn" onclick="removeAssignee('${a.id}')">&times;</button>
            </span>
        `).join('');
    }
    
    if (assignToInput) {
        assignToInput.value = JSON.stringify(state.selectedAssignees);
    }
}

// Remove assignee
function removeAssignee(id) {
    state.selectedAssignees = state.selectedAssignees.filter(a => a.id !== id);
    updateAssignToDisplay();
}

// Request Type selection
function selectRequestType(type) {
    state.selectedRequestType = type;
    
    const requestTypeSelected = document.getElementById('requestTypeSelected');
    const requestTypeInput = document.getElementById('requestType');
    
    if (requestTypeSelected) {
        requestTypeSelected.innerHTML = `<div class="selected-item-details">Selected: ${type}</div>`;
    }
    
    if (requestTypeInput) {
        requestTypeInput.value = type;
    }
    
    // Update UI
    document.querySelectorAll('.request-type-label').forEach(label => {
        label.classList.remove('selected');
    });
    
    const selectedLabel = document.querySelector(`#requestType_${type}`).nextElementSibling;
    if (selectedLabel) {
        selectedLabel.classList.add('selected');
    }
    
    closeRequestTypeModal();
}

// Task Management
function updateTaskList() {
    const taskList = document.getElementById('taskList');
    const tasksInput = document.getElementById('tasks');
    
    if (taskList) {
        taskList.innerHTML = state.tasks.map((task, index) => `
            <div class="task-card">
                <div class="task-card-header">
                    <div class="task-card-title">${task.title}</div>
                    <div class="task-card-status-pill">Initiated</div>
                    <button type="button" class="task-card-menu-button" onclick="showTaskMenu(${index})">
                        <i class="fas fa-ellipsis-v"></i>
                    </button>
                </div>
                <div class="task-card-type">${task.description || 'Repair'}</div>
                ${task.assignees && task.assignees.length > 0 ? 
                    `<div class="selected-text">Assigned to: ${task.assignees.map(u => u.fullName).join(', ')}</div>` : 
                    ''
                }
                ${task.note ? `<div class="selected-text">Note: ${task.note}</div>` : ''}
            </div>
        `).join('');
    }
    
    if (tasksInput) {
        tasksInput.value = JSON.stringify(state.tasks);
    }
}

function showTaskMenu(index) {
    // Simple implementation - in a real app you'd have a proper dropdown
    if (confirm('Do you want to edit this task?')) {
        editTask(index);
    } else if (confirm('Do you want to delete this task?')) {
        deleteTask(index);
    }
}

function editTask(index) {
    const task = state.tasks[index];
    state.editingTaskIndex = index;
    
    document.getElementById('taskTitle').value = task.title;
    document.getElementById('taskNote').value = task.note || '';
    document.getElementById('taskType').value = task.description || '';
    
    // Update task type buttons
    document.querySelectorAll('.task-type-btn').forEach(btn => {
        btn.classList.remove('selected');
        if (btn.dataset.type === task.description) {
            btn.classList.add('selected');
        }
    });
    
    state.taskAssignees = task.assignees || [];
    updateTaskAssignToDisplay();
    
    document.getElementById('addTaskModalTitle').textContent = 'Edit Task';
    openAddTaskModal();
}

function deleteTask(index) {
    state.tasks.splice(index, 1);
    updateTaskList();
}

function saveTask() {
    const title = document.getElementById('taskTitle').value.trim();
    if (!title) {
        showError('taskTitleError', 'Task title is required');
        return;
    }
    
    const task = {
        id: state.editingTaskIndex !== null ? state.tasks[state.editingTaskIndex].id : Date.now().toString(),
        title: title,
        description: document.getElementById('taskType').value,
        assignees: state.taskAssignees,
        note: document.getElementById('taskNote').value
    };
    
    if (state.editingTaskIndex !== null) {
        state.tasks[state.editingTaskIndex] = task;
        state.editingTaskIndex = null;
    } else {
        state.tasks.push(task);
    }
    
    // Clear form
    document.getElementById('taskTitle').value = '';
    document.getElementById('taskNote').value = '';
    document.getElementById('taskType').value = '';
    state.taskAssignees = [];
    updateTaskAssignToDisplay();
    updateTaskList();
    closeAddTaskModal();
}

// Task Assign To functions
function updateTaskAssignToDisplay() {
    const taskAssignToList = document.getElementById('taskAssignToList');
    
    if (taskAssignToList) {
        taskAssignToList.innerHTML = state.taskAssignees.map(a => `
            <span class="assignee-tag">
                ${a.fullName}
                <button type="button" class="remove-btn" onclick="removeTaskAssignee('${a.id}')">&times;</button>
            </span>
        `).join('');
    }
}

function selectTaskTeamMember(id, name, employeeNumber, role) {
    if (!state.taskAssignees.some(a => a.id === id)) {
        state.taskAssignees.push({ id, fullName: name, employeeNumber, role });
        updateTaskAssignToDisplay();
    }
}

function removeTaskAssignee(id) {
    state.taskAssignees = state.taskAssignees.filter(a => a.id !== id);
    updateTaskAssignToDisplay();
}

function selectTaskType(type) {
    document.getElementById('taskType').value = type;
    
    // Update UI
    document.querySelectorAll('.task-type-btn').forEach(btn => {
        btn.classList.remove('selected');
    });
    
    const selectedBtn = document.querySelector(`[data-type="${type}"]`);
    if (selectedBtn) {
        selectedBtn.classList.add('selected');
    }
}

// Fault List functions
function updateFaultList(faultList = state.faultListData) {
    const faultListContainer = document.getElementById('faultList');
    if (faultListContainer) {
        faultListContainer.innerHTML = faultList.map(f => `
            <div class="fault-item" onclick="selectFault('${f.id}', '${f.name}')">
                <div class="fault-name">${f.name}</div>
            </div>
        `).join('');
    }
}

function selectFault(id, name) {
    if (!state.selectedFaults.some(f => f.id === id)) {
        state.selectedFaults.push({ id, name });
    }
    closeFaultListModal();
}

function filterFaults() {
    const searchTerm = document.getElementById('faultSearch').value.toLowerCase();
    const filtered = state.faultListData.filter(fault => 
        (fault.name || '').toLowerCase().includes(searchTerm)
    );
    updateFaultList(filtered);
}

// Form validation
function validateForm() {
    let isValid = true;
    const errors = {};
    
    // Required fields
    const requiredFields = ['workOrderTitle', 'country', 'project'];
    requiredFields.forEach(field => {
        const input = document.getElementById(field);
        if (!input || !input.value.trim()) {
            errors[field] = `${field.charAt(0).toUpperCase() + field.slice(1)} is required`;
            isValid = false;
        }
    });
    
    // Show errors
    Object.keys(errors).forEach(field => {
        showError(field + 'Error', errors[field]);
    });
    
    return isValid;
}

function showError(elementId, message) {
    const errorElement = document.getElementById(elementId);
    if (errorElement) {
        errorElement.textContent = message;
        errorElement.classList.add('show');
    }
}

function clearErrors() {
    document.querySelectorAll('.error-text').forEach(el => {
        el.classList.remove('show');
    });
}

// Form submission handler
function handleFormSubmit(e) {
    e.preventDefault();
    
    clearErrors();
    
    if (!validateForm()) {
        return;
    }

    saveMaintenanceRequest();
}

async function saveMaintenanceRequest() {
    try {
        showSyncModal();
        
        const formData = new FormData(elements.maintenanceForm);
        const jsonData = {
            requestNumber: elements.requestNumber?.textContent,
            workOrderTitle: formData.get('workOrderTitle'),
            country: formData.get('country'),
            project: formData.get('project'),
            memo: formData.get('memo'),
            priority: state.selectedPriority,
            vehicle: state.selectedVehicle,
            assignTo: state.selectedAssignees,
            requestType: state.selectedRequestType,
            tasks: state.tasks
        };

        const response = await fetch(CONFIG.endpoints.storeRequest, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('input[name="_token"]').value
            },
            body: JSON.stringify(jsonData)
        });

        const data = await response.json();
        
        hideSyncModal();
        
        if (data.success) {
            showSuccessModal();
            // Store request ID for PDF generation
            window.requestId = data.requestId;
        } else {
            alert('Error: ' + data.message);
        }
    } catch (error) {
        console.error('Error:', error);
        hideSyncModal();
        alert('An error occurred while saving the maintenance request. Please try again.');
    }
}

// Modal functions
function openVehicleModal() {
    if (elements.vehicleModal) {
        elements.vehicleModal.style.display = 'block';
        loadEquipmentFromAPI();
    }
}

function closeVehicleModal() {
    if (elements.vehicleModal) {
        elements.vehicleModal.style.display = 'none';
    }
}

function openAssignToModal() {
    if (elements.assignToModal) {
        elements.assignToModal.style.display = 'block';
        updateAssignToList();
    }
}

function closeAssignToModal() {
    if (elements.assignToModal) {
        elements.assignToModal.style.display = 'none';
    }
}

function openRequestTypeModal() {
    if (elements.requestTypeModal) {
        elements.requestTypeModal.style.display = 'block';
    }
}

function closeRequestTypeModal() {
    if (elements.requestTypeModal) {
        elements.requestTypeModal.style.display = 'none';
    }
}

function openAddTaskModal() {
    if (elements.addTaskModal) {
        elements.addTaskModal.style.display = 'block';
        // Reset form if not editing
        if (state.editingTaskIndex === null) {
            document.getElementById('taskTitle').value = '';
            document.getElementById('taskNote').value = '';
            document.getElementById('taskType').value = '';
            state.taskAssignees = [];
            updateTaskAssignToDisplay();
            document.getElementById('addTaskModalTitle').textContent = 'Add Task';
        }
    }
}

function closeAddTaskModal() {
    if (elements.addTaskModal) {
        elements.addTaskModal.style.display = 'none';
        state.editingTaskIndex = null;
    }
}

function openTaskAssignToModal() {
    if (elements.taskAssignToModal) {
        elements.taskAssignToModal.style.display = 'block';
        updateTaskAssignToList();
    }
}

function closeTaskAssignToModal() {
    if (elements.taskAssignToModal) {
        elements.taskAssignToModal.style.display = 'none';
    }
}

function openFaultListModal() {
    if (elements.faultListModal) {
        elements.faultListModal.style.display = 'block';
        updateFaultList();
    }
}

function closeFaultListModal() {
    if (elements.faultListModal) {
        elements.faultListModal.style.display = 'none';
    }
}

function showSyncModal() {
    if (elements.syncModal) {
        elements.syncModal.style.display = 'block';
    }
}

function hideSyncModal() {
    if (elements.syncModal) {
        elements.syncModal.style.display = 'none';
    }
}

function showSuccessModal() {
    if (elements.successModal) {
        elements.successModal.style.display = 'block';
    }
}

function closeSuccessModal() {
    if (elements.successModal) {
        elements.successModal.style.display = 'none';
    }
    // Redirect to work orders list
    window.location.href = '{{ route("maintenance.workorders") }}';
}

function viewPDF() {
    if (window.requestId) {
        window.open(`${CONFIG.endpoints.generatePDF}/${window.requestId}`, '_blank');
    }
}

// Save functions for modals
function saveAssignTo() {
    closeAssignToModal();
}

function saveTaskAssignTo() {
    closeTaskAssignToModal();
}

function saveFaultList() {
    closeFaultListModal();
}

// Close modals when clicking outside
window.onclick = function(event) {
    const modals = [
        'vehicleModal', 'assignToModal', 'requestTypeModal', 'addTaskModal', 
        'taskAssignToModal', 'faultListModal', 'syncModal', 'successModal'
    ];
    
    modals.forEach(modalId => {
        const modal = document.getElementById(modalId);
        if (event.target === modal) {
            modal.style.display = 'none';
        }
    });
}

// Initialize the maintenance request form
function initializeMaintenanceForm() {
    init();
}

// Make functions globally available
window.selectVehicle = selectVehicle;
window.selectTeamMember = selectTeamMember;
window.selectRequestType = selectRequestType;
window.selectTaskType = selectTaskType;
window.selectFault = selectFault;
window.filterVehicles = filterVehicles;
window.filterTeamMembers = filterTeamMembers;
window.filterTaskTeamMembers = filterTeamMembers;
window.filterFaults = filterFaults;
window.removeAssignee = removeAssignee;
window.removeTaskAssignee = removeTaskAssignee;
window.saveTask = saveTask;
window.saveAssignTo = saveAssignTo;
window.saveTaskAssignTo = saveTaskAssignTo;
window.saveFaultList = saveFaultList;
window.openVehicleModal = openVehicleModal;
window.closeVehicleModal = closeVehicleModal;
window.openAssignToModal = openAssignToModal;
window.closeAssignToModal = closeAssignToModal;
window.openRequestTypeModal = openRequestTypeModal;
window.closeRequestTypeModal = closeRequestTypeModal;
window.openAddTaskModal = openAddTaskModal;
window.closeAddTaskModal = closeAddTaskModal;
window.openTaskAssignToModal = openTaskAssignToModal;
window.closeTaskAssignToModal = closeTaskAssignToModal;
window.openFaultListModal = openFaultListModal;
window.closeFaultListModal = closeFaultListModal;
window.closeSuccessModal = closeSuccessModal;
window.viewPDF = viewPDF;
window.saveMaintenanceRequest = saveMaintenanceRequest;
